/*
    termin.c - Edinburgh term input procedure for IC-Prolog ][
    Written by Frank McCabe, Philip Schwarz and Damian Chu
    Imperial College, Winter 1989

    Modifications:
    5/3/90    dac
	added garbage collection
    2/11/91   sl
	interfaced to QUAM data structures
	commented garbage collection

*/
#include <stdio.h>
#include <stdlib.h>

#include "defs.h"
#include "primitives.h"
#include "bind.h"
#include "build_term.h"
#include "cells.h"
#include "data_area.h"
#include "debug.h"
#include "dereference.h"
#include "errors.h"
#include "icprimitives.h"
#include "io.h"
#include "name_table.h"
#include "ops.h"
#include "parser_errors.h"
#include "persistent.h"
#include "read.h"
#include "sig.h"
#include "string_table.h"
#include "termin.h"
#include "termio.h"
#include "unify.h"
#include "x_registers.h"
#include "var_ren_table.h"
#include "write_read.h"
#include "examine_term.h"
#ifdef	X11
#include "xwidgets.h"
#endif	/* X11 */

enum var_type   
{
	     META_VARIABLEt, 
	     OBJECT_VARIABLEt,
	     LOCAL_VARIABLEt 
};
extern int	atoi(const char *);
extern operator	*is_op(token *tok);	/* test for the presence of an operator */
extern boolean	samesymb();
extern void	printName();
extern CHARTYPE	hedchar(void);
extern toktype	hedtoken(token **tok, int skip);
extern toktype	nextoken(token **tok, int skip);
extern long	int tok_len(token *t);

cell		hash_variable(enum var_type variable_t, offset var, cell val);
extern void	reset_brace(void);
extern void	reset_total_marked();
extern int	debugLevel;
extern jmp_buf	interrupt;
extern	jmp_buf	signaljump;
static	jmp_buf	termjump;

static token	braces = {(toktype)graph, 3, "{}"};
static cell	equals_sign;    /* To become a pointer to the '=' symbol   */

static twoBytes varcount;	/* Used to create names for anonymous vars */
static cellpo	next_eqn;	/* A pointer to the next equation 	   */

/* global variables usd in handling syntax errors */
static long int SP1, SP2, SP3;  /* Stream Pointers for marking positions */

/* globals used for garbage collection during term read 
static boolean	GC;		   have we called the garbage collector?   */
static cellpo	oldH, oldHMAX;  /* shadow registers for H and HMAX	   */
static cell coma_sign;    
static cell old_var;    /* for a Prolog list of variables - OldVars */
static natural read_type=READ_ORD;
void esc_term(cellpo t, twoBytes prior);


/*------------------------------------------------------------*
 *      V A R I A B L E / C O N S T A N T      T A B L E      *
 *------------------------------------------------------------*/

/* maximum number of distinct constant/variable names 
   in a term before the table is expanded */
#define READER_TABLE_SIZE    50

/* structure of reader table */
typedef struct bucket 
{
    enum var_type   variable_type;

    offset	     nme;    /* The (constant/variable)'s name.	     */
    cell	     val;    /* An unbound variable (var) or NULL(const). */
    struct bucket    *lnk;
} *bucketpo;

/* access functions for reader table */
#define VariableType(bucket)	((bucket) -> variable_type)
#define name(bucket)		((bucket) -> nme)
#define value(bucket)		((bucket) -> val)
#define variable(bucket)	(value(bucket))
#define constant(bucket)	(! value(bucket))
#define link(bucket)		((bucket) -> lnk)
#define nonempty(b)		(link(b))
#define NO_LINK			((bucketpo) 1)
#define haslink(b)		(link(b) > NO_LINK)
#define table_full		(next_bucket < first_bucket)

/* This macro assumes that t points to a cellpo in the heap */

static bucketpo	first_bucket = NULL,	/* reader table            */
		next_bucket,		/* The next free slot      */
		tabletop;		/* The end of the table    */

static twoBytes entryno = READER_TABLE_SIZE;	/* size of above   */

/*------------------------------------------------------------------------------
initialise_variable_hash_table()

------------------------------------------------------------------------------*/
void
initialise_variable_hash_table(void)
{
	     /* used to create equations in install_variable() */
    if (!first_bucket) 
    {
	if (!(first_bucket=(bucketpo)alloc(entryno, sizeof(struct bucket))))
	    fatal("unable to allocate variable hash table");
	tabletop = first_bucket + entryno;
    }
    else
    {
        (void)memset((char*)first_bucket, 0, (SIZE_TYPE)(entryno*sizeof(struct bucket)));
    }

    next_bucket = tabletop - 1;
}

/*------------------------------------------------------------------------------
init_reader()

------------------------------------------------------------------------------*/
void
init_reader(void)
{
    reset_brace();	/* initialise the tokenizer */
    if (string_read)
        SP1 = (long int) string_read_base;
    else            
        SP1 = ftell(Qstdin);
    equals_sign = Atom(add_name_string_offset("=", ATOM_W));
    initialise_variable_hash_table();
}

/*------------------------------------------------------------------------------
nextFree(b)
    find next free slot in variable hash table 
------------------------------------------------------------------------------*/
static bucketpo
nextFree(bucketpo b)
{
    while (nonempty(next_bucket)  && !table_full)
	next_bucket--;
    link(b) = next_bucket;
    return(next_bucket);
}
/*------------------------------------------------------------------------------
rehash_table()
    reader table is doubled in size 
------------------------------------------------------------------------------*/
static void
rehash_table(void)
{
    register
    bucketpo	copy1, copy2;
    twoBytes	old_entryno;

    copy1 = copy2 = first_bucket;
    old_entryno   = entryno;
    entryno	  = 2 * old_entryno;
    if (!(first_bucket=(bucketpo)alloc(entryno, sizeof(struct bucket))))
	 fatal("Unable to reallocate variable hash table");
    tabletop    = first_bucket + entryno;
    next_bucket = tabletop - 1;

    while(old_entryno--) 
    {
	if (nonempty(copy1)) 
	{
	    if (variable(copy1))
		(void) hash_variable(VariableType(copy1),
				     name(copy1),
				     value(copy1));
	}
	copy1++;
    }

       /* release a previously allocated block */
   free((char *) copy2);
}



#define	MAXOBJECTVARIABLES	100
static	cell	object_variables[MAXOBJECTVARIABLES];
/*------------------------------------------------------------------------------
extract_object_vars_from_hash_table()

   collect all the dynamic object variables (x) in the array
   object_variables, returning the number of object variables.
   This is used to create distinct_from constraints on the parsed terms.
------------------------------------------------------------------------------*/
natural
extract_object_vars_from_hash_table(void)
{
    register
    bucketpo	b;
    natural	n;
    natural	m;
    
    m = 0;
    n = entryno;
    b = first_bucket;

    while(n--) 
    {
	if (nonempty(b)) 
	{
	    if (VariableType(b) == OBJECT_VARIABLEt) 
	    {
		if (m > MAXOBJECTVARIABLES) 
		{
		    fatal("more than %d object variables in term, extend MAXOBJECTVARIABLES", MAXOBJECTVARIABLES);
		} 
		else 
		{
		    object_variables[m++] = value(b);
		}
	    }
	}
	b++;
    }
    return(m);
}



/*------------------------------------------------------------------------------
             S Y N T A X     E R R O R     H A N D L I N G         
------------------------------------------------------------------------------*/
/*------------------------------------------------------------------------------
syntax_error(errcode, tok)

------------------------------------------------------------------------------*/
void
syntax_error(twoBytes errcode, token **tok)
{
    reset_brace();
    
    if (string_read) 
    {
        SP2 = (long int)string_read_pos - (tok_len(*tok) + 1);
        SP3 = (long int)string_read_pos;
    }
    else
        if (Qstdin != stdin) 
	{
	    SP2 = ftell(Qstdin) - (tok_len(*tok) + 1); 
	    while (nextoken(tok,TRUE) != dot && 
	           (*tok)->tt != (toktype)eof);
	    SP3 = ftell(Qstdin); 
        }
    longjmp(termjump, errcode);
}
/*------------------------------------------------------------------------------
print_syntax_error(errcode)

------------------------------------------------------------------------------*/
void
print_syntax_error(twoBytes errcode)
{
    char	*msg;


    if(string_read){
	/*
        int before = SP2 - (long int)last_string_base,
            after  = SP3 - SP2;
        string_read_pos = last_string_base;
        while(before-- && errorc(charin));
        error(" **HERE**> ");
        while(after-- && errorc(charin));
        error("\n");
        */
    } else
    if(Qstdin == stdin) 
    {
	unsigned char *lb=stdinbuf;
	int posn=stdinposn;
	char ch=0;
	error(" : ");
	while(posn-- > 0)
	    fputc((ch = *lb++),stderr);
	error(" *HERE* ");
	if (ch != '\n') 
	{
	    while((ch = charin) != '\n' && ch != EOF)
	        fputc(ch,stderr);
	}
	fputc('\n',stderr);
	if (ch == EOF)
	    charback(ch);
	stdinposn = 0;
	stdinlast = 0;
    }
    else 
    { 
	int before = SP2 - SP1,
	    after  = SP3 - SP2;
	(void)fseek(Qstdin, SP1, 0);
	error("\n\n");
	while(before-- && fputc(charin, stderr));
	error(" *HERE* ");
	while(after-- && fputc(charin, stderr));
	error("\n");
      }
   

    switch (errcode) {
	case  1: msg = "'}' expected"; break;
	case  2: msg = "']' expected"; break;
	case  3: msg = "',', '|' or ']' expected"; break;
	case  4: msg = "invalid term0"; break;
	case  5: msg = "',' or ')' expected"; break;
	case  6: msg = "prefix operator precedence too high"; break;
	case  7: msg = "invalid term"; break;
	case  8: msg = "'.' expected"; break;
	case  9: msg = "unexpected end of line in string"; break;
	case 10: msg = "unexpected end of file in string"; break;
	case 11: msg = "unexpected end of line in quoted atom"; break;
	case 12: msg = "unexpected end of file in quoted atom"; break;
	case 13: msg = "unexpected end of file in number"; break;
	case 14: msg = "badly formed floating-point number"; break;
	case 15: msg = "free occurrence of local object variable"; break;
	case 16: msg = "quantified position must be an object variable"; break;
        case 17: msg = "the domain of a substitution must contain object variables";break;
        case 18: msg = "object variables in a substitution must be distinct";break;
        case 19: msg = "substitution needs a \"/\"";break;
        case 20: msg = "badly formed substitution ";break;
        case 21: msg = "invalid local object variables";break;
        case 22: msg = "invalid variable type";break;
	case 23: msg = "')' expected"; break;
	otherwise: fatal("unknown error");
    }
    error("** %s **\n", msg);
}



/*------------------------------------------------------------*
 *         H E A P    S P A C E    M A N A G E M E N T        *
 *------------------------------------------------------------*/
/*



   make sure we have enough heap space.
   Call the garbage collector if necessary. 


void
interm_gc_test(Space)
int Space;
{
    if (H + Space >= HMAX) {
	if (GC)		   second time around, we are really out of space 
	    longjmp(interrupt, 505);
	GC = TRUE;
	oldHMAX = H;
	H = oldH;
	HMAX = H + 1;
	if (collect_garbage(4) < Space)		   mark registers A0 to A3 
	    longjmp(interrupt, 505);
	oldH = H;
    }
}


   This procedure is only called if garbage collection was required
   during a read.  This recompacts the heap so that the 'hole' created
   by G/C is removed. = 
recompact_heap(t, eqns)
cellpo    t, eqns;
{
    =  Here is where we use those reserved cells in high memory.
       This ensures that the term and eqn cells are on the top
       of heap after compacting the heap. 
    *oldHMAX = *eqns;	   copy root of eqn list to heap 
    tg(eqns) = var_ref;
    vl(eqns) = oldHMAX++;
    *oldHMAX = *t;	 = copy root of term to heap = 
    tg(t) = var_ref;
    vl(t) = oldHMAX++;

     = the quickest way to unmark a block  =
    while (H < HMAX)
	mknil(H++);

    =  mark and compact the segment used by interm  =
    H = HMAX = oldHMAX;
    reset_total_marked();
    mark_variable(t);
    mark_variable(eqns);
    compact_segment(oldH, H);

    *t = *--H;		=  top of heap is root of term  =
    *eqns = *--H;       =  next top of heap is root of equation list  =
}


*/
/*------------------------------------------------------------*
 *          U T I L I T Y     F U N C T I O N S               *
 *------------------------------------------------------------*/

/*
    Variable names are changed if there is a constant with that
    name in the term.  The name X is changed to X$n where n is
    some integer.  Uniqueness of this name within the term must be
    guaranteed.  Anonymous variables are given a unique name _$n
    in a similar manner.

global int
hash_variable_name(variable_name)
char	*variable_name;
{
    int len = strlen(variable_name);
    int	hash = 0;


    while (*variable_name) {
	hash += *variable_name++;
    }
    return((hash & 0x0FFFF) | len<< 16);
}

*/
/*------------------------------------------------------------------------------
new_variable_name(s)

------------------------------------------------------------------------------*/
void
new_variable_name(char *s)
{
    register
    char     *p;

    if ((s[0] == '_') && (s[1] == '$')) /* An anonymous variable */
	(void) sprintf(s+2,"%hd",varcount++);
    else
	if (p = strrchr(s,'$')) {
	    p++;
	    (void) sprintf(p,"%d",atoi(p) + 1);
	}
	else { (void)strcpy(s + strlen(s),"$0"); }
}

/*
symbpo
make_symbol(pname, len)
char	 *pname;
fourBytes     len;
{
    register
    CHARTYPE	*t;
    cell        temp;
    cellpo	c = &temp;
    int		numcells,       no. of cells needed to represent the symbol 
		hashtotal    = 0,     hash code to be computed 
		i	     = len;

    numcells = symbSize(len, sizeof(cell));
    interm_gc_test(numcells+1);
    alloc_symb(c,t,numcells,len,0);
    while(i--) {
	hashtotal += *pname;
	*t++ = *pname++;
    }
    symbhashval(symbvl(c)) = (hashtotal & 0x0FFFF) | len<< 16;
    return(symbvl(c));
}

symbpo
new_symb(in)
symbpo in;
{
    int		len = symblngth(in);
    char	*ch;	 space for constructing variable names 
    symbpo	result;

    ch = malloc((size_t)(len+5));

    (void) strncpy(ch,symbname(in),len);
    ch[len] = '\0';
    new_variable_name(ch);
    result = make_symbol(ch,(fourBytes)strlen(ch));

    free(ch);
    return(result);
}

*/
/*
symbpo
install_constant(c, b)
symbpo          c;
bucketpo        b;
{
    name(b)  = c;
    value(b) = NULL;
    link(b)  = NO_LINK;
    return(c);
}
*/
/*------------------------------------------------------------------------------
install_variable(variable_type, var, b, val)

------------------------------------------------------------------------------*/
cell
install_variable(enum var_type variable_type, offset var, bucketpo b, cell val)
{
	VALUE	val_h;
	cell	*head;
	char	buf[MAXTOKBUFFER+1];

static	cell	esc_readc_var(enum var_type var_t), esc_readRc_var(enum var_type var_t, offset var_string), esc_readR1c_var(enum var_type var_t, offset var_string);

	if (val != NULL)
	{
		VariableType(b) = variable_type;
		name(b) = var;
		value(b) = val;
		link(b) = NO_LINK;
		return(val);
	}
	/* first time this variable was entered */

	/*
	 * Insert into the table which contains the variables
	 * encountered during this read.
	 */
	VariableType(b) = variable_type;
	name(b)  = var;
	link(b)  = NO_LINK;

	if (IsAnonymous(String2(var)))
	{
		value(b) = esc_readc_var(variable_type);
	}
	else switch (read_type)
	{
	when READ_ORD:
		value(b) = esc_readc_var(variable_type);
	when READ_R:
		value(b) = esc_readRc_var(variable_type, var);
	when READ_R1:
		value(b) = esc_readR1c_var(variable_type, var);
	}
	/*
	 * Build the list that is returned by read.
	 */
	InstantiateTail(next_eqn, head, val_h);
	*head = Apply();
	Argument(*head) = value(b);
	Functor(*head) = Apply();
	if (variable_type == LOCAL_VARIABLEt) 
	{
		buf[0] = '@';
		strcpy(&buf[1], String(var));
		Argument(Functor(*head)) =
			Atom(add_name_string_offset(buf, ATOM_W));
	}
	else
		Argument(Functor(*head)) = Atom(var);
	Functor(Functor(*head)) = equals_sign;
	mknil(next_eqn);

	return(value(b));
}
/*------------------------------------------------------------------------------
hash_variable(variable_type, var, val)
   returns a pointer to an unbound variable 

------------------------------------------------------------------------------*/
cell
hash_variable(enum var_type variable_t, offset var, cell val)
{
register  bucketpo b = first_bucket + 
			hash_variable_name(String2(var)) % entryno;

          
    if (b < first_bucket)
	b += entryno;
    

    if (nonempty(b)) 
    {
	boolean found = FALSE;
        while (!(found=((var == name(b) && variable_t== VariableType(b))))
		&& haslink(b))
	    b = link(b);

	if (found) 
	    return(value(b));
	b = nextFree(b);
	
	if (table_full) 
	{
	    rehash_table();
	    
	    return(hash_variable(variable_t, var, val));
	}
    }
    
    return(install_variable(variable_t, var, b, val));
}

/*            I think there is no need to keep it   
void
make_variable(tt,tok)
cellpo	tt;	 must be pointing to a heap cell 
token	*tok;
{
    register
    fourBytes	len = tok->bufflen - 1;
    char	*ch;	 space for constructing variable names 

    ch = malloc((size_t)(len+5));

    tg(tt) = var_ref;
    if ((len == 1) && (*tok->buff == '_')) {
	(void) sprintf(ch,"_$%hd",varcount++);
	vl(tt)=(cellpo)hash_variable(make_symbol(ch,(fourBytes)strlen(ch)),(cellpo)NULL);
    }
    else vl(tt)=(cellpo)hash_variable(make_symbol(tok->buff,len),(cellpo)NULL);

    free(ch);
}
*/

/*
symbpo
hash_constant(con)
symbpo con;
{
    register
    bucketpo b = first_bucket + symbhashval(con) % entryno;

    if (b < first_bucket)
	b += entryno;

    if (nonempty(b)) {
	boolean found = FALSE;
	while (!(found=samesymb(con,name(b))) && haslink(b))
	    b = link(b);
	if (found) {
	    if (variable(b)) {
		(void) hash_variable(new_symb(name(b)),value(b));
		value(b) = NULL;
	    }
	    return(name(b));
	}
	b = nextFree(b);
	if (table_full) {
	    rehash_table();
	    return(hash_constant(con));
	}
    }
    return(install_constant(con, b));
}
*/

/*------------------------------------------------------------------------------
boolean usepostfix(prior)
    look ahead one token to see if we can commit to a postfix operator 

------------------------------------------------------------------------------*/
boolean usepostfix(twoBytes prior)
{
    register
    operator *nextop;
    token *tok;

    /* look ahead at the next token for the beginning of a term */
    switch(hedtoken(&tok,TRUE)) 
    {
	case quoted:
	case lower:
	case graph:
	case solo:
	case comma:
	case semicolon:
	    return((nextop=is_op(tok)) != NULL
		    && nextop->prefixp == undefprior
		    && nextop->postleft > prior
		    && nextop->infleft > prior);

	case bra:	/* In these cases, the next token */
	case brace:	/* is definitely the beginning of */
	case sqbra:	/* a new term0; therefore the     */
	case string:	/* current operator must be infix.*/
	case number:
	case floating:
	case upper:
	case under:
	    return FALSE;

	default:	    /* this is when the next token signals end  */
	    return TRUE;    /* of term. i.e.. ')', '}', '}', '.' or EOF */
    }
}

/*------------------------------------------------------------------------------
term0(t, is_a_substitution, is_a_list, sub_error_msg)   -    reads in a 
                                                             primitive term
------------------------------------------------------------------------------*/
void 
term0(cellpo t, int *is_a_substitution, int *is_a_list, int *sub_error_msg)
{
       VALUE  val_h;
       token  *tok;

       *is_a_substitution = FALSE;
       *is_a_list = FALSE;
       *sub_error_msg = 20;
    

       switch(nextoken(&tok,TRUE)) 
       {
	   case quoted:
	   case graph:
	   case solo:
	       make_constant(t, tok);
	       break;
	   case lower:
	       if (object_var_prefix_declared(tok->buff)) 
	       {
		   *t = HashVariable(OBJECT_VARIABLEt, tok); 
               }
	       else
	       {
		   make_constant(t, tok);
               }
	       break;
           case bra:			       /* parenthetical term */
	       esc_term(t,minprior);	       /* read in a bracketted term */
	       if (nextoken(&tok,TRUE)!=ket)  /* check for closing bracket */
		    syntax_error(23, &tok);
	       break;

	   case brace:
	
	       /*
		* Change to TRUE after the fix in nxtoken.
	       if (hedtoken(&tok,FALSE)==endbrace) 
		*/
	       if (hedtoken(&tok,TRUE)==endbrace) 
	       {
		   make_constant(t, &braces);  /* the constant '{}' */
		   /*
		    * Change to TRUE after the fix in nxtoken.
		   (void)nextoken(&tok,FALSE);
		    */
		   (void)nextoken(&tok,TRUE);
		   break;
	       }
	       else 			    /* a brace {term} */
	       {
		   register cellpo s;
		   /* interm_gc_test(3);  */
		   *t = Apply();     /* create a new tuple and point it */
		   make_constant(&Functor(*t), &braces); /* fill in with braces symbol */
		   esc_term(&Argument(*t), minprior); /* read in a curly bracketted term */
		   if (nextoken(&tok,TRUE)!=endbrace)
			syntax_error(1, &tok);
		   break;
	       }
	   case sqbra: 	
	   {
	       /* constant [] or
	          an empty substitution [] or
	          a list [t1, ..., tn] or
	          a substitution [t1/x1, ..., tn/xn]
	       
	          The flags is_a_list and is_a_substitution and error message
	          sub_error_msg are used in $term()
	          when reading the next token after this term. If a star * is
	          read then if this term is a list it must also be a
	          substitution. So we need to record the chararcteristics of
	          this term (list if we get here) */
	    
	       *is_a_list = TRUE;
	       /*
		* Change to TRUE after the fix in nxtoken.
	       if (hedtoken(&tok, FALSE) == sqket) 
	       */
	       if (hedtoken(&tok, TRUE) == sqket) 
	       {
		   *is_a_substitution = TRUE;
		   mknil(t);
		   /*
		    * Change to TRUE after the fix in nxtoken.
		   (void)nextoken(&tok, FALSE); /* consume the 
		   */
		   (void)nextoken(&tok, TRUE); /* consume the 
							      closing bracket */
		   break;
	       }
	       else 
	       {
		   reg	cellpo	list=t;	/* point to the list built */
		   while(TRUE) 
		   {
		       cell *head;
		       cell slash_symbol = Atom(add_name_string_offset("/",
		                                                       ATOM_W));

		       InstantiateTail(list, head, val_h);
		    
		       esc_term(head, argprior); /* argprior is the priority of
					         comma */
		       if (IsApply(*head) &&
			   IsApply(Functor(*head)) &&
			   Functor(Functor(*head)) == slash_symbol) 
                       {
			    if (IsObjectReference(Argument(*head))) 
			    {
				*is_a_substitution = TRUE;
			    } 
			    else 
			    {
				*is_a_substitution = FALSE;
				*sub_error_msg = 17;
			    }
		       } 
		       else 
		       {
			   *is_a_substitution = FALSE;
			   *sub_error_msg = 19;
		       }

			
		           /* construct a pair */
		       if (nextoken(&tok, TRUE) == comma) 
		       {
			   /* get rest of list t2, ..., tn] */
			   continue;
                       }
		       /* 
			* The use of '|' as alternative to ';' gets confused
			* when ';' has precedence less than 1000.  So this 
			* alternative use is stopped.
			* else if(tok->tt == semicolon)  [t1|t2] */
		       else if(tok->tt == bar)  /* [t1|t2] */
		       {
			   esc_term(list, argprior);
			   if (nextoken(&tok, TRUE)!=sqket)
			       syntax_error(2, &tok);
			   break;
                       }
		       else if (tok->tt == sqket)   /* end of list */
		       {
			   mknil(list);
			   break;
		        } 
		        else 
			{
			    syntax_error(3, &tok);
		        }
		   }
		   break;
	       }
	   }
	   case string:
	   {
	
               *t = put_string_on_heap(tok->buff);
	       /*  interm_gc_test(2*i);                  */  
	       break;
	   }
	   case number:
	            /* bufflen field of token is overloaded for numbers */
	       mkint(t,tok->bufflen);
	       break;

/*	   case floating:
	   {
	       FLOAT fl_num;
	       fourBytes num;
	       (void)sscanf(tok->buff, "%lg", &fl_num);

	              convert to an integer if possible 
	       num = fl_num;
	       if (num == fl_num) 
	       {
		    mkint(t,num);
	       }
	       else 
	       {
		   interm_gc_test(3);           
		   alloc_float(t,fl_num);
	       }
	       break;
	   }
*/
	   case under:
	       if (IsSameString(tok->buff, "_")) 
	       {
	          /* dont enter anonymous variables in hash table */
	           *t = NewVariable();
               }
               else if (IsAnonymousObjectVariable(tok->buff))
	       {
		   *t = HashVariable(OBJECT_VARIABLEt, tok);
	       }
	       else
	       {
	       	   *t = HashVariable(META_VARIABLEt, tok); 
	       }
	       break;
	   case upper:
	       *t = HashVariable(META_VARIABLEt, tok); 
	       break;
	   default:
	       syntax_error(4, &tok);
       }

       while (hedtoken(&tok,FALSE) == bra) 
       {
	    /* t(t1, ..., tn) */
	   reg     int   maxarg = 0;
	   cell    tmp;
	   cell    functor;

           functor = *t;

	   (void)nextoken(&tok,FALSE);   /* get the '(' */
	   /*
	    * Change to TRUE after the fix in nxtoken.
	   if (hedtoken(&tok, FALSE) != ket)
	    */
	   if (hedtoken(&tok, TRUE) != ket)
	       while(tok->tt!=ket) 
	       {
		   tmp = Apply();
		   Functor(tmp) = functor;
		   esc_term(&Argument(tmp), argprior);
		   functor = tmp;
		   maxarg++;
		   if (nextoken(&tok,FALSE)==ket)
		         /* end of term */
		       break;
		   else if (tok->tt == comma)
		       continue;
		   else
		       syntax_error(5, &tok);
	       }
	   else
	   {
	       /* t() */
	       /*
		* Change to TRUE after the fix in nxtoken.
	       (void)nextoken(&tok, FALSE);
		*/
	       (void)nextoken(&tok, TRUE);
	   }

	   /* determine if we should convert this to a list or a tuple */
	   if (maxarg == 2 &&
	       IsReference(*t) &&
	       IsSameString(String(*t), "."))  
           {
               Functor(Functor(functor)) = Atom(CONS); 
	   }
           *t = functor;
       }
}

/*------------------------------------------------------------------------------
$term(t,prior)

        main term input function 
------------------------------------------------------------------------------*/
void
esc_term(cellpo t, twoBytes prior)
{
       register
       operator   *curr_op;		/* current operator */
       token      *tok;
       cell       lefterm;	        /* buffer to hold the left term */
       int        lprior;			/* left hand priority */
       int        rprior;			/* right hand priority */
       int        quant_precedence;
       boolean    is_a_substitution;
       boolean    is_a_list;
       int        sub_error_msg;
       toktype    hed;

       is_a_substitution = FALSE;
       is_a_list = FALSE;

    /* BEWARE the use of case fall through in the following switch */
    
    
    

    
    switch(hedtoken(&tok,TRUE)) 
    {
	case quoted:
	case lower:
	case graph:
	case solo:
	    
	    if (hedchar()!=bra && (curr_op=is_op(tok))!=NULL
		    && curr_op->prefixp!=undefprior) 
            {
		    /* we have a prefix operator */
		lprior=curr_op->prefixp;       /* priority of left */
		(void)nextoken(&tok,TRUE);     /* commit to the operator */
		make_constant(&lefterm, tok);  /* construct a constant into t */

		if (IsSameString(curr_op->op, "@")) 
		{
		    toktype	tt = hedtoken(&tok,  FALSE);

		    if (tt == (toktype) lower || tt == (toktype) under)
                    {
			/* @x */
			(void)nextoken(&tok, TRUE);
			/* commit to the local */
			if (object_var_prefix_declared(tok->buff)
			    || IsAnonymousObjectVariable(tok->buff))  
			{
			    lefterm = HashVariable(LOCAL_VARIABLEt, tok);
			    lprior=0;	/* this is a term0 after all */
			}
			else
			    syntax_error(21, &tok); 
			break;
		    }
		} 
		else if (IsSameString(curr_op->op, "-")) 
		{
		       /* check for negative number here? */
		    toktype t = hedtoken(&tok,FALSE);
		    if (t == (toktype)number) 
		    {
			(void)nextoken(&tok,TRUE); /* commit to the number */
			/* bufflen field of token is overloaded for numbers */
			mkint(&lefterm,-(tok->bufflen));
			lprior=0;		/* this is a term0 after all */
			break;
		    }
		    /*
		    else if (t == (toktype)floating) {
			FLOAT num;
			(void)nextoken(&tok,TRUE); commit to the number 
			(void)sscanf(tok->buff, "%lg", &num);
			interm_gc_test(3);
			alloc_float(&lefterm,-num);
			lprior=0;		 this is a term0 after all 
			break;
		    }
		    */
		}  
		else if (hedtoken(&tok, TRUE) == ket)   
		{
		     /* (op) overide operator */
                     lprior = 0; /* this is a term0 after all */
		     break;
                }
		if (lprior>prior)		/* a syntax error */
		    syntax_error(6, &tok);

		/* pick up the right precedence of the prefix op */
		rprior=curr_op->preform;

		/* look ahead again for another operator */
		/* Note the use of fall through in the cases ! */
		
		
		switch(hedtoken(&tok,TRUE)) 
		{
		    case quoted:
		    case lower:
		    case graph:
		    case solo:
			if (hedchar() != bra
			    && (curr_op=is_op(tok)) != NULL
			    && curr_op->prefixp > rprior
			    && curr_op->postleft >= lprior
			    && curr_op->infleft >= lprior
			    && quantifier(tok->buff) >= lprior)
			    break;
			else			/* use case fall through */

		    case bra:			/* parenthetical term */
		    case brace:			/* a brace {term} */
		    case sqbra:			/* we have a list to read */
		    case string:
		    case number:
		 /* case floating: */
		    case upper: 
		    case under:
		    {
			cell s;
			/* construct a prefix operator term */
			
			
			s=lefterm;	       /* copy the old cell */
                 /*	interm_gc_test(3);        */
			lefterm = Apply();     /* create a new structure */
			Functor(lefterm) = s;		/* copy old cell in */
			esc_term(&Argument(lefterm),rprior);/*read in another term*/
			break;
		    }

		    default:
			lprior=0;
			break;
		}
		break;  /* top level switch */
	    }
	    else if ((quant_precedence = quantifier(tok->buff)))
	    {
		/* a quantified term q x t */
		cell	q;
		cell	x;
		toktype tt;

		(void) nextoken(&tok, TRUE);
		make_constant(&q, tok);
		tt = hedtoken(&tok,  TRUE);
		
		/* quantified position can contain x or @x */
		if (tt == (toktype) lower || tt == (toktype) under) 
		{
		    /* x */
		    /* added */
		    nextoken(&tok,  TRUE);
		    if (object_var_prefix_declared(tok->buff)
			|| IsAnonymousObjectVariable(tok->buff))
		    {
		        x = HashVariable(OBJECT_VARIABLEt, tok); 
		    }   
		    else  
		    {
			syntax_error(16, &tok); 
		    }
		}  
		else if (IsSameString(tok->buff, "@")) 
		{
		    toktype	tt;

		    /* @x */
		    /* added */
		    nextoken(&tok,  TRUE);
		    tt = nextoken(&tok, FALSE);
		    if (tt == (toktype) lower || tt == (toktype) under)
                    {
			if (object_var_prefix_declared(tok->buff)
			    || IsAnonymousObjectVariable(tok->buff))
			{
			    x = HashVariable(LOCAL_VARIABLEt, tok); 
			} 
			else 
			{
			    syntax_error(16, &tok);
			}
		    }
		    else 
		    {
			syntax_error(16, &tok);
		    }
		} 
		else 
		{
		    lefterm = q;
		    lprior = 0;
		    break;
		    /*
		    syntax_error(16, &tok);
		    */
		}

		esc_term(&lefterm, quant_precedence);
		lefterm = build_quantify(q, x, lefterm);
		lprior = 0;
		break;
	    } else	 /* use case fall through */

	case bra:			/* (t) */
	case brace:			/* {t} */
	case sqbra:			/* [t] */
	case string:
	case number:
	case upper:
	case under:
	{               /* do not delete these braces */
	     term0(&lefterm, &is_a_substitution, &is_a_list, &sub_error_msg);
	     lprior = 0;			/* left hand priority is 0 */
	     break;
	}
	default:
	    (void)nextoken(&tok,  FALSE);
	    /* read past the non-term */
	    syntax_error(7, &tok);
    }

    /* rator branch deals with infix and postfix operators */
    while(TRUE)
	switch(hedtoken(&tok,TRUE)) 
	{
	    case quoted:
	    case lower:
	    case graph:
	    case solo:
	    case comma:
	    case semicolon:
		if((curr_op=is_op(tok))!=NULL) 
		{
		    cell op, s,*p;
		    if (curr_op->infixp<=prior && curr_op->infleft>=lprior) 
		    {
			/* commit to and make the operator */
			(void)nextoken(&tok,TRUE);
			make_constant(&op,tok);

			if(curr_op->postfixp<=prior &&
				curr_op->postleft>=lprior &&
				usepostfix(curr_op->infright)) 
                        { 
			    /* make a postfix term here 
			    interm_gc_test(3); */
			    s = Apply();
			    Functor(s) = op;
			    Argument(s) = lefterm;
			    lefterm = s;
			    lprior = curr_op->postfixp;
			    continue;
			} 
			else if (IsSameString(curr_op->op, "*") &&
				   is_a_list) 
                        { 
			    if (is_a_substitution) 
			    {
				cell	sub = lefterm;
				esc_term(&lefterm, curr_op->infright);
				if ((lefterm = build_substitute(sub,
					    lefterm)) == NULL) 
                                {
					syntax_error(20, &tok);
				}
			    } 
			    else 
			    {
				/* Hack Hack Hack need to realign stdin for
				error messages */
				syntax_error(sub_error_msg, &tok);
			    }
			} 
			else 
			{
			    /* make an infix term here 
			    interm_gc_test(4); */ 
			    s = Apply();
			    Functor(s) = Apply();
			    Functor(Functor(s)) = op;
			    Argument(Functor(s)) = lefterm;
			    esc_term(&Argument(s), curr_op->infright);
			    lefterm = s;
			}
			lprior=curr_op->infixp;
			continue;
		    }
		    else if(curr_op->postfixp<=prior &&
			    curr_op->postleft>=lprior) 
                    {

			/* commit to and make the operator */
			(void)nextoken(&tok,TRUE);
			make_constant(&op,tok);

			/* make a postfix term here */
           /*		interm_gc_test(3);       */ 
			s = Apply();
			Functor(s) = op;
			Argument(s) = lefterm;
			lefterm=s;
			lprior=curr_op->postfixp;
			continue;
		    }
		}

	    default:
		*t = lefterm;
		return;
        }
}

char *string_read_pos, *string_read_base, *last_string_base;
int string_read = 0;
/*------------------------------------------------------------------------------
$no_string_read()
 	turn string read flag off 


------------------------------------------------------------------------------*/
global boolean
esc_no_string_read(void)
{
	if(string_read)
	{
	    string_read = 0;
	    free(string_read_base);
	}
	return(TRUE);
}

/*------------------------------------------------------------------------------
$string_read(atom_or_string_to_be_read_from) 
 	set string read flag 
 	if true currently read from a string 
	if false currently read from a stream 
 	at NULL, an end of file is returned 


------------------------------------------------------------------------------*/
global boolean
esc_string_read(void)
{
	if (IsAtom(Xdref(0)))
	{
	    string_read_base = strdup(String(X(0)));
	    last_string_base = string_read_pos = string_read_base;
	    string_read = 1;
	    return(TRUE);
	}
	else
	    if (IsApply(Xdref(0)))
	    {
		string_read_base = strdup(get_string_from_heap(X(0)));
		last_string_base = string_read_pos = string_read_base;
		string_read = 1;
		return(TRUE);
	    }
	    else 
	        return(FALSE);
}

/*------------------------------------------------------------------------------
$string_read_position(position_within_string_for_read_to_start) 

------------------------------------------------------------------------------*/
global boolean
esc_string_read_position(void)
{
	if( !string_read ) return(FALSE);

	if (IsInteger(Xdref(0))) 
	{
	    last_string_base = string_read_pos = string_read_base + IntOf(X(0));
	    return(TRUE);
	}

	if (IsReference(X(0))) 
	{
	    VALUE val;
	    val.term = Integer((int)(string_read_pos - string_read_base));
	    val.sub = EMPTY_SUB;
	    return(unify(&val, XV(0)));
	}
	return(FALSE);
}

/*
 * Reset the input stream.
 */
static	void
reset_input(void)
{
	if(string_read)
	{
		string_read_pos = last_string_base;
	}
	else if(Qstdin == stdin)
	{
		Fflush(stdin);
	}
	else
	{
		(void)fseek(Qstdin, SP1, 0);
	}
}

/*
 * Set all the object variables read in to be distinct from each other.
 */
static	void
set_distinctness(void)
{
	natural	n, i, j;

	if ((n = extract_object_vars_from_hash_table()) > 1) 
	{
		for (i = 0; i < n; i++) 
		{
			for (j = i+1; j < n; j++) 
			{
				SetDistinct(object_variables[i],
					    object_variables[j]);
			}
		}
	}
}

/*
 * The core of the read.
 */
static	boolean
read(VALUE *readterm, int precedence, VALUE *varslist)
{
	twoBytes	errcode, sigcode;
	VALUE		term, equations;
	token		*tok;

	equations.sub = EMPTY_SUB;
	equations.term = Atom(NIL);
	next_eqn = &(equations.term);
	term.sub = EMPTY_SUB;

	no_delay_signal_handling();
	if(sigcode = setjmp(signaljump))
	{
		delay_signal_handling();
		reset_input();
		return(REDO);
	}
	else if((errcode = setjmp(termjump)))
	{
		delay_signal_handling();
		print_syntax_error(errcode);
		return(FALSE);
	}

	if (hedtoken(&tok, TRUE) == (toktype)eof) 
	{
		term.term = Atom(add_name_string_offset("end_of_file", ATOM_W));
		(void)nextoken(&tok, TRUE);	/* skip the EOF */
	}
	else 
	{
		esc_term(&(term.term), (twoBytes)precedence);
		if(nextoken(&tok,TRUE)!=dot && tok->tt != (toktype)eof)
			syntax_error(8, &tok);

		set_distinctness();
	}

	delay_signal_handling();
	return(unify(&term, readterm) && unify(&equations, varslist));
}

/*-----------------------------------------------------------------------------
$readc(Term, Priority, Equation) -  main entry point for term input 

     Term      = the term being read in (output parametar)
     Priority  = an input integer usually 1200
     Equations =  a list of (name = variable) equations (output parametar)

X[0] and X[2] are assumed to be variables.  
-----------------------------------------------------------------------------*/
global boolean
esc_readc(void)
{
	if (!IsInteger(Xdref(1)))
		return(FALSE);

	init_reader();
	read_type = READ_ORD;
	return(read(XV(0), IntOf(X(1)), XV(2)));
}

/*
 * Given the type of a variable required, a new variable is created.
 * Any variant of the variable existed in any variable table is ignored.
 */
static	cell
esc_readc_var(enum var_type var_t)
{
	cell	var;

	switch(var_t) 
	{
	when META_VARIABLEt:
		var = NewVariable();
	when OBJECT_VARIABLEt:
		var = NewObjectVariable();
	when LOCAL_VARIABLEt:
		var = NewLocalObjectVariable();
	otherwise:
		fatal("Invalid variable type in variable hash table");
	}
	return(var);
}

/*----------------------------------------------------------------------------
$readRc(Term, Priority, OldVars, NewNamePairs)  

    Term         = the term being read in
    OldVars      = a Prolog list of variables that is input to read
    NewNamePairs = a Prolog list of name pairs that is output from read

    Initialise variable name table to contain variable names from the Prolog
    list OldVars
    Initialise rename table to empty

    Variables read in with the same name as variables in OldVars refer to
    these variables.Otherwise variables read in refer to new variables.
    If the name of a variable read in clashes with another variable
    currently in use, then a variant name is generated for the new variable.


----------------------------------------------------------------------------*/
global boolean
esc_readRc(void)
{
	cell		el;
	VALUE		val_head, val_h, val_tail;

	if (!IsInteger(Xdref(1)))
		return(FALSE);
	if (!IsList(Xdref(2)) && !IsNIL(X(2)))
		return(FALSE);

	init_reader();
	for (el = X(2); !IsNIL(el); el = val_tail.term)
	{
		DereferenceTerm(val_head, Head(val_h, el)); 
		if (IsReference(val_head.term) ||
		    IsObjectReference(val_head.term))
		{
			(void)hash_variable((IsReference(val_head.term) ?
						META_VARIABLEt :
						OBJECT_VARIABLEt),
				      NamePtrToOffset(Named(val_head.term)),
				      val_head.term);
		}
		DereferenceTerm(val_tail, Tail(el)); 
	}
	read_type = READ_R; 
	return(read(XV(0), IntOf(X(1)), XV(3)));
}

/*----------------------------------------------------------------------------
$readRc_var(var_t, var_string) 

    if variable name is not in name table
	    add new entry to name table
	    push a new variable into heap
	    set value to variable
	    set ptr_to_name to  name table entry
    else if value of variable name is NULL  
	    push a new variable into heap  
	    set value of name to variable 
	    set ptr_to_name to name table entry
    else
	    generate a variant name based on old name
	    add the string for the name to the string table
	    add new entry to name table
	    push a new variable into heap
	    set value of name to variable
	    set ptr_to_name to name table entry

----------------------------------------------------------------------------*/
static  cell
esc_readRc_var(enum var_type var_t, offset var_string)
{
	cell	variableR; 
	int	hnt_name, hnt;
	char	*root_name, *variant_name;

local	cell	set_var_value_ptr(enum var_type var_t, int i);
		      
	/* hnt_name = hash value of var_string to name_table */  
	if (NameT(name_table[(hnt_name = lookup_name_table(String2(var_string),
							   VARIABLE_W))])
	    == NULL)
	{
		/* var_string is not in name_table */

		/* add new entry to name_table */
		NameT(name_table[hnt_name]) = VARIABLE_W|var_string;
		/* push variable into heap    
		   set value of var_string to variable   
		   set ptr_to_name to  name_table entry */
		variableR = set_var_value_ptr(var_t, hnt_name); 
	}
	else if (ValueOfName(name_table[hnt_name]) == NULL)
	{
		/* var_string is in name_table */
		/* push variable into heap 
		   set value of var_string to variable 
		   set ptr_to_name to name_table */
		variableR = set_var_value_ptr(var_t, hnt_name); 
	}
	else 
	{ 
		/* generate variant_name based on var_string*/
		unsigned int dump_reg = 1; 
		root_name = find_root(String2(var_string)); 
		variant_name = generate_new_name(root_name, &dump_reg);
			       
		  /* add variant_name to string_table
		     add variant_name to name_table  
		     hnt = hash value of variant_name to name_table */ 
		hnt = add_name_string_hash(variant_name, VARIABLE_W);
		  /*  push variable into heap 
		     set value of variant_name to variable     
		     set ptr_to_name to name_table entry */ 
		variableR = set_var_value_ptr(var_t, hnt); 
	} 
	return(variableR);
}
/*-----------------------------------------------------------------------------
$readR1c(Term, Precedence, VarsList)

    Term        = the term being read in
    Precedence	= the starting precedence for Term
    VarsList	= a list of variable name and location pairs

    Same as readR/3 assuming (implicitly) all currently active variables are
    in OldVars. I. E. each variable read in with the same name as a currently
    active variable refer to that variable.

-----------------------------------------------------------------------------*/
global boolean
esc_readR1c(void)
{
	if (!IsInteger(Xdref(1)))
		return(FALSE);

	init_reader();
	read_type = READ_R1;
	return(read(XV(0), IntOf(X(1)), XV(2)));
}

/*----------------------------------------------------------------------------
$readR1c_var(var_t, var_string) 

    For a variable read in do:
    if (name is not in name table)
        add new entry to name table
        push a new variable into heap
        set value of name to variable
        set ptr_to_name to name table entry
    else if (value of name is NULL)
	push a new variable into heap
	set value of name to variable
	set ptr_to_name to name table entry
    else
	variable = value of name 
       
----------------------------------------------------------------------------*/
static   cell
esc_readR1c_var(enum var_type var_t, offset var_string)
{
	cell   	variableR; 
	natural	i;

local	cell	set_var_value_ptr(enum var_type var_t, int i);

	/* i = hash value of var_string to name_table */ 
	i = lookup_name_table(String2(var_string), VARIABLE_W);
	if (NameT(name_table[i]) == NULL)
	{ 
		/* var_string is not in name table */
		/* add var_string to name_table       */ 
		NameT(name_table[i]) = VARIABLE_W|var_string;
		/* push a new variable into heap 
		   set value of var_string to variable    
		   set ptr_to_name to name_table entry of var_string */
		variableR = set_var_value_ptr(var_t, i); 
	}
	else if (ValueOfName(name_table[i]) == NULL)
	{
		/* var_string is in name_table and 
	           value of var_string is  NULL */  
	       
		/* push variable into heap 
		   set value of var_string to variable 
		   set ptr_to_name to name_table entry */
		variableR = set_var_value_ptr(var_t, i); 
	}
	else
	{
		/* var_string is in name_table and 
		   value of var_string is not NULL */  

		variableR = ValueOfName(name_table[i]);
	} 
	return(variableR);
}



/*----------------------------------------------------------------------------
set_var_value_ptr(var_t, i)
    push a new variable into heap 
    set value of name to variable 
    set ptr_to_name to name table 
----------------------------------------------------------------------------*/
local	cell
set_var_value_ptr(enum var_type var_t, int i)
{
	cell	variableR;

	/* push variable into heap */
	variableR = esc_readc_var(var_t);
	/* set value of name to variable */
	set(&ValueOfName(name_table[i]), variableR);
	/* set ptr_to_name to name_table */
	PtrToName(variableR) = (cell)&name_table[i]; 

	return(variableR);
}
