/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

#include <string.h>

#include "bind.h"
#include "data_area.h"
#include "database.h"
#include "defs.h"
#include "dereference.h"
#include "errors.h"
#include "name_table.h"
#include "object_vars.h"
#include "persistent.h"
#include "string_table.h"
#include "trail.h"
#include "unify.h"
#include "x_registers.h"

global	cell	*pvar;
global	cell	*top_of_pvar;
global	natural	pvar_size = DEFAULT_PVAR_SIZE;

global	cell	*pstack;
global	cell	*top_of_pstack;
global	cell	*end_of_pstack;
global	natural	pstack_size = DEFAULT_PSTACK_SIZE;

global	cell	*ptrail;
global	cell	*top_of_ptrail;
global	cell	*end_of_ptrail;
global	natural	ptrail_size = DEFAULT_PTRAIL_SIZE;

local	PNAME	*pname_table = NULL;
local	int	meta_var_num = 0;
local	int	object_var_num = 0;
local	int	top_local_table = 0;
local	LENTRY	local_table[LSIZE];

/*
 * low							high
 * +-------------------------------------------------------+
 * | persistent meta / object variables -> | persistent -> |
 * +-------------------------------------------------------+
 */
global	void
initialise_persistent_area(void)
{
        if ((pvar = (cell *)malloc(Kwords_to_chars(pvar_size + pstack_size)))
					== NULL)
                fatal("not enough memory for the persistent variables and \
							stack %d and %d K",
					pvar_size, pstack_size);
        pstack = (cell *) (((char *)pvar) + Kwords_to_chars(pvar_size));
        end_of_pstack = (cell *)(((char *)pstack) +
					 Kwords_to_chars(pstack_size));
        top_of_pvar = pvar;
        top_of_pstack = pstack;

}

global	void
initialise_persistent_trail(void)
{
        if ((ptrail = (cell *)malloc(Kwords_to_chars(ptrail_size))) == NULL)
                fatal("not enough memory for the persistent trail %d K",
							ptrail_size);
        end_of_ptrail = (cell *)(((char *)ptrail) +
					 Kwords_to_chars(ptrail_size));
        top_of_ptrail = ptrail;
}

global	void
initialise_commit(void)
{
	top_local_table = 0;
}

/*
 * Whenever an unbound meta / object variable is copied from the heap, the
 * existing one in the heap is bound to the newly created persistent meta /
 * object variable.  (ie heap points to persistent stack)
 *
 * Note: commit/0 is intended to be used just before fail/0.
 *	 Otherwise, undesirable behaviour may occur.
 */
global	cell
commit_copy(cell *term)
{
	cell	persistent;
local	cell 	commit_substitution_copy(cell sub);
local	cell 	new_persistent_var(int metavar);
local	void	undo_trailing(cell *term);
local	void	persistent_dereference(cell *t);
local	cell	map_local(cell h);


	persistent_dereference(term);
	if (InPersistentStack(*term))
	{
                if (InPersistentStack(Reference(term)) &&
		    ! Committed(term))
		{
			undo_trailing(term);
			if (term[1] != ALL_DISTINCT)
				Commit(term);
		}
		persistent = *term;
	}
	else
	{
		if (!IsVariable(*term) && InPersistentStack(Reference(term)))
		{
			undo_trailing(term);
			if (term[1] != ALL_DISTINCT)
				Commit(term);
		}
		switch (Tag(*term))
		{
		when CONSTANT:
			persistent = PersistentConstant(*term);
		when APPLY or PAIR or QUANTIFIER:
			persistent = (cell)(Tag(*term)|PersistentNewPair());
			Location(persistent)[0] =
					commit_copy(Location(*term));
			Location(persistent)[1] =
					commit_copy(Location(*term) + 1);
		when REFERENCE:
			persistent = new_persistent_var(TRUE);
			BindVariable(*term, persistent);
		when OBJECT_REFERENCE:
			/*
			 * No need to commit distinction information.
			 * All persistent object variables are distinct
			 * from each other, but not with object variables.
			 */
			if (IsLocalObjectVariable(*term))
				persistent = map_local(*term);
			else
			{
				persistent = new_persistent_var(FALSE);
				BindObjectVariable(*term, persistent);
			}
		when SUBSTITUTION_OPERATOR:
			persistent = PersistentSubstitutionOperator();
			Substitution(persistent) =
				commit_substitution_copy(Substitution(*term));
			Term(persistent) = commit_copy(&(Term(*term)));
		otherwise:
			persistent = *term;
		}
	}
	return(persistent);
}

local	cell
commit_substitution_copy(cell sub)
{
	cell	next, persistent, *table;
reg	int	i, j;

	if (sub == EMPTY_SUB)
		persistent = EMPTY_SUB;
	else
	{
		next = commit_substitution_copy(NextSub(sub));
		table = PersistentAllocate(2 * Size(sub) + 1);
		persistent = NewPersistentSubstitution((cell)table, next,
							Property(sub));
		Size(persistent) = Size(sub);
		for (i = 1, j = Size(sub); i <= j; i++)
		{
			Domain(persistent, i) = commit_copy(&Domain(sub, i));
			Range(persistent, i) = commit_copy(&Range(sub, i));
		}
	}
	return(persistent);
}

global	cell
commit_not_free(cell *term)
{
	cell	persistent;
local	cell	copy_not_free(cell *d);
local	void	undo_trailing(cell *term);

	switch (Tag(*term))
	{
	when VARIABLE:	/* implies to OBJECT_VARIABLE as well */
		undo_trailing(term);
		persistent = VARIABLE|Temperature(*term)|
			    copy_not_free((cell *)RestOf(RestOfTemp(*term)));
	otherwise:
		persistent = *term;
	}
	return(persistent);
}

local	cell
copy_not_free(cell *d)
{
	cell	next;
	cell	*persistent;
	VALUE	val;

	if (d == NULL)
		return(NULL);
	else if (InPersistentStack(Reference(d)))
		return((cell) d);
	else
	{
		next = copy_not_free(Next(d));
		if (InPersistentStack(DerefTerm(val, DelayedVar(d))) &&
		    InPersistentStack(DerefTerm(val, DelayedTerm(d))))
		{
			Solved(d);
			persistent = PersistentAllocate(2);
			/* Delay(persistent) = (delayed *)PersistentAllocate(3);*/
			Delay(persistent)->tag = DelayedType(d);
			DelayedVar(persistent) = commit_copy(&DelayedVar(d));
			DelayedTerm(persistent) = commit_copy(&DelayedTerm(d));
			/* Next(persistent) = (cell *)next; */
			return((cell) persistent);
		}
		else
			return(next);
	}
}

local	void
persistent_dereference(cell *t)
{
	for(;;)
		switch (Tag(*t))
		{
		when REFERENCE:
			if (InPersistentStack(*t) || IsVariable(Value(*t)))
				return;
			else
				*t = Value(*t);
		when OBJECT_REFERENCE:
			if (InPersistentStack(*t) ||
			    IsObjectVariable(ObjectValue(*t)))
				return;
			else
				*t = ObjectValue(*t);
		otherwise:
			return;
		}
}

local	void
undo_trailing(cell *term)
{
	trailer	*t;
	int	temp;
	boolean found;

        found = FALSE;
	temp = THAW;
	for (t = top_of_trail - 1; t >= trail; t--)
	{
		if (t->address == term)
		{
			temp = Temperature(t->previous_value);
			t->address = NULL;
			found = TRUE;
		}
	}
	if (found)
	   PersistentTrail(temp|(int)term);
}

local	cell
map_local(cell h)
{
	int	i;

	for (i = 0; i < top_local_table; i++)
		if (local_table[i].heap == h)
			return(local_table[i].ps);
	if (++top_local_table > LSIZE)
		fatal("commit local table (size = %d) is full", LSIZE);
	else
	{
		local_table[i].heap = h;
		local_table[i].ps = NewPersistentLocalObjectVariable();
		return(local_table[i].ps);
	}
}

global	boolean
esc_persistent_meta_var(void)
{
local	cell	new_persistent_var(int metavar);
	VALUE	val;

	if (InPersistentStack(Xdref(0)))
		return(TRUE);
	else
	{
		val.sub = EMPTY_SUB;
		val.term = new_persistent_var(TRUE);
		return(unify(XV(0), &val));
	}
}

global	boolean
esc_persistent_object_var(void)
{
local	cell	new_persistent_var(int metavar);
	VALUE	val;

	if (InPersistentStack(Xdref(0)))
		return(TRUE);
	else
	{
		val.sub = EMPTY_SUB;
		val.term = new_persistent_var(FALSE);
		return(unify(XV(0), &val));
	}
}

global	boolean
esc_unpersistent_meta_var(void)
{
	cell	new_var;

	if (! InPersistentStack(Xdref(0)))
		return(TRUE);
	else
	{
		new_var = NewVariable();
		set(Location(new_var), Value(X(0)));
		set(Location(X(0)), new_var);
		return(TRUE);
	}
}

global	boolean
esc_unpersistent_object_var(void)
{
	cell	new_var;

	if (! InPersistentStack(Xdref(0)))
		return(TRUE);
	else
	{
		new_var = NewObjectVariable();
		set(Location(new_var), Value(X(0)));
		set(Location(X(0)), new_var);
		return(TRUE);
	}
}

global	boolean
is_persistent_var(void)
{
	return(IsReference(Xdref(0)) && InPersistentStack(X(0)));
}

global	boolean
is_persistent_object_var(void)
{
	return(IsObjectReference(Xdref(0)) && InPersistentStack(X(0)));
}
 
global  boolean
esc_new_persistent_meta_var(void)
{
reg     VALUE   val;

	if (! IsAtom(Xdref(0)))
		return(FALSE);
	else
	{
		val.sub = EMPTY_SUB;
		val.term = new_persistent_meta_var(X(0));
		return(unify(XV(1), &val));
	}
}

global	cell
new_persistent_meta_var(cell name)
{
local	cell	add_persistent_entry(cell name, int metavar);
	char	*s;
	int	i;

	s = String(name);
	while (*s != '_')
		s++;
	i = atoi(++s);
	if (i > meta_var_num)
		meta_var_num = i;
	return(add_persistent_entry(name, TRUE));
}

global  boolean
esc_new_persistent_object_var(void)
{
reg     VALUE   val;

	if (! IsAtom(Xdref(0)))
		return(FALSE);
	else
	{
		val.sub = EMPTY_SUB;
		val.term = new_persistent_object_var(X(0));
		return(unify(XV(1), &val));
	}
}

global	cell
new_persistent_object_var(cell name)
{
local	cell	add_persistent_entry(cell name, int metavar);
	char	*s;
	int	i;

	s = String(name);
	while (*s != '_')
		s++;
	i = atoi(++s);
	if (i > object_var_num)
		object_var_num = i;
	return(add_persistent_entry(name, FALSE));
}

local	cell
new_persistent_var(int metavar)
{
local	cell	add_persistent_entry(cell name, int metavar);
	char	*s;

	if (metavar)
		sprintf(top_of_string_table, "X_%d", ++meta_var_num);
	else
		sprintf(top_of_string_table, "%s_%d", object_var_prefix(),
							++object_var_num);
	s = top_of_string_table;
	top_of_string_table += strlen(top_of_string_table) + 1;
	if (top_of_string_table >
			(string_table + Kwords_to_chars(string_table_size)))
		fatal("Out of space in string table %d K", string_table_size);
	return(add_persistent_entry(Atom(lookup_name_table_offset(
					 s, ATOM_W)), metavar));
}

local	cell
add_persistent_entry(cell name, int metavar)
{
	PNAME	*p = pname_table;

	while (p != NULL)
	{
		if (p->name == name)
			return(p->var);
		else
			p = p->next;
	}
	p = (PNAME *)malloc(sizeof(PNAME));
	p->name = name;
	p->var = metavar ? NewPersistentVariable() :
			   NewPersistentObjectVariable();
	p->next = pname_table;
	pname_table = p;
	return(p->var);
}

global	boolean
esc_persistent_name(void)
{
	cell	get_persistent_name(cell var);

	if (IsReference(Xdref(0)) || IsObjectReference(X(0)))
		return(unify_constant(get_persistent_name(X(0)), XV(1)));
	else
		return(FALSE);
}

global	cell
get_persistent_name(cell var)
{
	PNAME	*p;

	for (p = pname_table; p != NULL && p->var != var; p = p->next)
		;
	if(p != NULL)
	    return(p->name);
	else
	    return((cell)0);
}

global	cell
get_persistent_name2(cell var)
{
	PNAME	*p;

	for (p = pname_table;
	     p != NULL && RestOf(p->var) != RestOf(var);
	     p = p->next)
		;
	if(p != NULL)
	    return(p->name);
	else
	    return((cell)0);
}
