/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

#include <stdio.h>
#include <stdlib.h>

#include "cells.h"
#include "data_area.h"
#include "dereference.h"
#include "errors.h"
#include "examine_term.h"
#include "io.h"
#include "name_table.h"
#include "string_table.h"
#include "unify.h"
#include "x_registers.h"
#ifdef X11
#include "events.h"
#include "xwidgets.h"
#endif /* X11 */

global	natural	qstdin = QSTDIN;
global	natural	qstdout = QSTDOUT;
global	QSTREAM	*streams;
global	natural	num_streams = DEFAULT_NUM_STREAMS;
global	unsigned char linebuf[BUFSIZ];
global	unsigned char stdinbuf[BUFSIZ];
global	int	stdinposn,stdinlast;
extern	cell	*write_count_ptr;

#define inc_write_count(n) (*write_count_ptr 		\
			    = 				\
			    Integer(IntOf(*write_count_ptr)+(n)))

/*----------------------------------------------------------------------------
initialise_streams()
----------------------------------------------------------------------------*/
global	void
initialise_streams(void)
{
        natural j = 0; 
	char    *ch;


	if ((streams = (QSTREAM *) malloc(num_streams * sizeof(QSTREAM))) ==
		NULL)
		fatal("cannot allocate streams");
	InitStream(QSTDIN, stdin);
	InitStream(QSTDOUT, stdout);
	InitStream(QSTDERR, stderr);
	stdinposn = 0;
	stdinlast = 0;
	(void)setvbuf(Qstdin, (char *)linebuf, _IOLBF, BUFSIZ);
			 
	{/* clear the remaining streams */
	int i;
	for(i = 3; i<num_streams; i++)
		ClearStream(i);
	}
}

/*----------------------------------------------------------------------------
expand_streams(size)
    the streams array is expanded by size streams
    the success of this is returned.
----------------------------------------------------------------------------*/
int
expand_streams(int size)
{
	QSTREAM *tmp;
	if(tmp = (QSTREAM *)realloc( streams,
				     sizeof(QSTREAM)
				     *
				     (num_streams + (natural)size) 
				    )
	  )
		{
		streams = tmp;
		{/* clear the new streams */
		int i;
		for(i = num_streams; i < (num_streams+size); i++)
			ClearStream(i);
		}
		num_streams += size;
		return(1);
		}
	else
		return(0);
	}

/*----------------------------------------------------------------------------
put(Char)
    Write the character stored in the first register onto the current output
    stream in ASCII code.
----------------------------------------------------------------------------*/
global	boolean
put(void)
{
	void esc_put(char c);

	if (!IsInteger(Xdref(0)))
	{
		warning("non-integer to put");
		return(FALSE);
	}
	esc_put((char)(IntOf(X(0)) & 0177));
	return(TRUE);
}

/*----------------------------------------------------------------------------
get(Char)
    The first register is unified with the first ASCII code read from the
    current input stream that has a value greater than 32. It is unified
    with -1 on encountering EOF. If EOF was reached on a previous read, a
    fatal error occurs if another read is done. Fails if the character read
    fails to unify with the value in the first register.
----------------------------------------------------------------------------*/
global	boolean
get(void)
{
	int	c;
	VALUE	val;

	if (feof(Qstdin))
		fatal("attempt to read past end of file in get");

	while ((c = fgetc(Qstdin)) <= 32 && c != EOF)
		;
	val.term = Integer(c);
	val.sub = EMPTY_SUB;
	return(unify(XV(0), &val));
}

/*----------------------------------------------------------------------------
put0(Char)
    Write the character stored in the first register onto the current output
    stream in ASCII code.
----------------------------------------------------------------------------*/
global	boolean
put0(void)
{
	if (IsInteger(Xdref(0)))
	{
		warning("non-integer to put0");
		return(FALSE);
	}
	esc_put((char)IntOf(X(0)));
	return(TRUE);
}

/*----------------------------------------------------------------------------
get0(Char)
    The first register is unified with the first ASCII code read from the
    current input stream. It is unified with -1 on encountering EOF. If EOF
    was reached on a previous read, a fatal error occurs if another read is
    done. Fails if the character read fails to unify with the value in the
    first register.
----------------------------------------------------------------------------*/
global	boolean
get0(void)
{
	int	c;
	VALUE	val;

	if (feof(Qstdin))
		fatal("attempt to read past end of file in get0");
	c = fgetc(Qstdin);
	val.term = Integer(c);
	val.sub = EMPTY_SUB;
	return(unify(XV(0), &val));
}

global	boolean
esc_write_atom(void)
{
	void esc_write_string(char *s);

	if (!IsAtom(Xdref(0)))
		return(FALSE);
	else
		esc_write_string(String(X(0)));
	return(TRUE);
}

global	boolean
esc_write_integer(void)
{
	void esc_write_int(int i);

	if (!IsInteger(Xdref(0)))
		return(FALSE);
	else
		esc_write_int(IntOf(X(0)));
	return(TRUE);
}

/*----------------------------------------------------------------------------
    Write out a variable name, use the address - the start of the heap
    (which is below the stack (malloced in a contiguous block)) convert
    this to a base 26 number and use the uppercase letters to represent
    the digits of the number, e.g. AAAG represents 6.
----------------------------------------------------------------------------*/
global	boolean
esc_write_var_name(void)
{
	if (IsReference(Xdref(0)) || IsObjectReference(X(0)))
	{
		WriteVariableName((int)(((cell *)RestOf(X(0))) - heap));
		return(TRUE);
	}
	else
		return(FALSE);
}

/*----------------------------------------------------------------------------
$open(Filename, Mode, Stream) :-
    true if the file open succeeds for Filename for a particular mode.
----------------------------------------------------------------------------*/
global	boolean
esc_open(void)
{
	char	*filename;
	cell	mode;
	char	*new_mode;
	VALUE	val;

	if (((IsAtom(Xdref(0)) && (filename = String(X(0)), TRUE)) ||
	     (IsApply(X(0)) &&
	      (filename = get_string_from_heap(X(0))) != NULL)) &&
	    IsAtom(Xdref(1)))
	{
		mode = RestOfConstant(X(1));
		if (mode == add_name_string_offset("read", ATOM_W))
			new_mode = "r";
		else if (mode == add_name_string_offset("write", ATOM_W))
			new_mode = "w";
		else if (mode == add_name_string_offset("append", ATOM_W))
			new_mode = "a";
		else
			return(FALSE);
		val.sub = EMPTY_SUB;
		return(esc_fopen(filename, new_mode, &(val.term)) &&
		       unify(XV(2), &val));
	}
	return(FALSE);
}


global	boolean
esc_fopen(char *filename, char *mode, cell *fd_term)
{

	natural next_stream = 3;
	/* find first free stream in streams table */
	while(next_stream < num_streams && !IsClearStream(next_stream))
		next_stream++;
	/* expand table if necessary */
	if (next_stream >= num_streams)
		if(!expand_streams(DEFAULT_NUM_STREAMS))
		{
		warning("no more streams available (%d) to open %s",
		        num_streams, filename);
		return(NULL);
	}
	InitStream(next_stream, fopen(filename, mode));
	if (!streams[next_stream].strm)
		return(FALSE);
	*fd_term = NewFD(next_stream, 0);

	return(TRUE);
}

global	boolean
esc_fdopen(int fd, char *mode, cell *fd_term)
{
	natural next_stream = 3;
	/* assign first free stream to next_stream */
	while(next_stream < num_streams && !IsClearStream(next_stream))
		next_stream++;

	/* expand streams table if necessary */
	if (next_stream >= num_streams)
		if(!expand_streams(DEFAULT_NUM_STREAMS))
		{
		warning("no more streams available (%d) to open %d",
		    num_streams, fd);
		return(NULL);
	}
	InitStream(next_stream, fdopen(fd, mode));
	if (!streams[next_stream].strm)
		return(FALSE);
	*fd_term = NewFD(next_stream, 0);
	
	return(TRUE);
}

/*----------------------------------------------------------------------------
$close(Stream) :-
----------------------------------------------------------------------------*/
global	boolean
esc_close(void)
{
	VALUE	val;
	cell	arg_save;
	int	n;
	natural s;

	if (IsApply(Xdref(0)))
	{
		if ((DereferenceTerm(val, Functor(X(0))),
		     IsApply(val.term)) &&
		    (arg_save = Argument(val.term),
		     DereferenceTerm(val, Functor(val.term)),
		     RestOfConstant(val.term) == add_name_string_offset(
		                                 "$stream", ATOM_W)) &&
		    (DereferenceTerm(val, arg_save),
		     IsInteger(val.term)) &&
		    (n = IntOf(val.term)) >= 0 &&
		    n < num_streams &&
		    !IsClearStream(n) &&
		    n > 2)
		{
			if (fclose(streams[n].strm) != 0)
				return(FALSE);
			ClearStream(n);
			return(TRUE);
		}
	}
	else if (IsAtom(X(0)))
	{
	     if (RestOfConstant(X(0)) == add_name_string_offset("user_input",
	                                                        ATOM_W)) 
		  s = QSTDIN;
             else if (RestOfConstant(X(0)) == add_name_string_offset(
					      "user_output", ATOM_W)) 
		  s = QSTDOUT;
             else if (RestOfConstant(X(0)) == add_name_string_offset(
					      "user_error", ATOM_W)) 
		  s = QSTDERR;
             else
		  return(FALSE);
             streams[s].end_of_file = FALSE;
	     clearerr(streams[s].strm);
	     return(TRUE);
         }

}

/*----------------------------------------------------------------------------
$set_input(Stream) :-
    true, iff Stream is a proper stream $stream(A, B), where A is an index
    into the array of io buffers, and B = 0 (?).
----------------------------------------------------------------------------*/
global	boolean
esc_set_input(void)
{
	VALUE	val;
	cell	arg_save;
	int	n;

	if (IsApply(Xdref(0)))
	{
		if ((DereferenceTerm(val, Functor(X(0))),
		     IsApply(val.term)) &&
		    (arg_save = Argument(val.term),
		     DereferenceTerm(val, Functor(val.term)),
		     RestOfConstant(val.term) == add_name_string_offset(
						 "$stream", ATOM_W)) &&
		    (DereferenceTerm(val, arg_save),
		     IsInteger(val.term)) &&
		    (n = IntOf(val.term)) >= 0 &&
		    n < num_streams &&
		    !IsClearStream(n) 
		    )
		{
			qstdin = n;
			return(TRUE);
		}
	}
	else if (IsAtom(X(0)))
	{
		if (RestOfConstant(X(0)) == add_name_string_offset("user",
		                                                   ATOM_W) ||
		    RestOfConstant(X(0)) == add_name_string_offset("user_input",                                                                   ATOM_W))
		{
			qstdin = QSTDIN;
			return(TRUE);
		}
	}

	return(FALSE);
}

/*----------------------------------------------------------------------------
$set_output(Stream) :-
    true, iff Stream is a proper stream $stream(A, B), where A is an index
    into the array of io buffers, and B = 0 (?).
----------------------------------------------------------------------------*/
global	boolean
esc_set_output(void)
{
	VALUE	val;
	cell	arg_save;
	int	n;

	if (IsApply(Xdref(0)))
	{
		if ((DereferenceTerm(val, Functor(X(0))),
		     IsApply(val.term)) &&
		    (arg_save = Argument(val.term),
		     DereferenceTerm(val, Functor(val.term)),
		     RestOfConstant(val.term) == add_name_string_offset(
						 "$stream", ATOM_W)) &&
		    (DereferenceTerm(val, arg_save),
		     IsInteger(val.term)) &&
		    (n = IntOf(val.term)) >= 0 &&
		    n < num_streams &&
		    !IsClearStream(n)
		    )
		{
			qstdout = n;
			return(TRUE);
		}
	}
	else if (IsAtom(X(0)))
	{
		if (RestOfConstant(X(0)) == add_name_string_offset("user",
		                                                   ATOM_W)  ||
		    RestOfConstant(X(0)) == add_name_string_offset(
					    "user_output", ATOM_W))
		{
			qstdout = QSTDOUT;
			return(TRUE);
		}
		else if (RestOfConstant(X(0)) == add_name_string_offset(
						 "user_error", ATOM_W))
		{
			qstdout = QSTDERR;
			return(TRUE);
		}
	}

	return(FALSE);
}

/*----------------------------------------------------------------------------
flush :-
	true always, side-affects flushes current output.
----------------------------------------------------------------------------*/
global	boolean
flush(void)
{
	fflush(Qstdout);
	return(TRUE);
}



global	boolean
esc_read_term_string(void) /*  Read a list of characters until a period followed
			by white space or end-of-file is encountered. */

{
	VALUE	val;
local	cell read_term_string(int quoted);

	val.term = read_term_string(FALSE);
	val.sub = EMPTY_SUB;
	return(unify(XV(0), &val));
}

local	cell
read_term_string(int quoted)
{
	char	c;
	char	c2;

	switch (c = Get0())
	{
	when EOF:
		return(Cons(Integer(c), NIL));
	when '.':
		if (quoted)
			return(Cons(Integer(c),
				read_term_string(quoted)));
		else
		{
			c2 = Get0();
			if (WhiteSpace(c2))
				return(Cons(Integer(c), Cons(Integer(c2),
							     Atom(NIL))));
			else
				return(Cons(Integer(c), Cons(Integer(c2),
					read_term_string(quoted))));
		}
	when '%':
		if (quoted)
			return(Cons(Integer(c), read_term_string(quoted)));
		else
		{
			while ((c = Get0()) != '\n')
				;
			return(read_term_string(quoted));
		}
	when '/':
		if (quoted)
			return(Cons(Integer(c), read_term_string(quoted)));
		else
		{
			c2 = Get0();
			if (c2 == '*')
			{
				c = Get0();
				do
				{
					while (c != '*')
						c = Get0();
				} while ((c = Get0()) != '/');
				return(read_term_string(quoted));
			}
			else
				return(Cons(Integer(c), Cons(Integer(c2),
					read_term_string(quoted))));
		}
	when '\'':
		return(Cons(Integer(c), read_term_string(! quoted)));
	otherwise:
		return(Cons(Integer(c), read_term_string(quoted)));
	}
}

global void
esc_write_string(char *s)
{
#ifdef	X11
		if(widget_putting)
			{
			inc_write_count(strlen(s));
			while(*s) widget_put(*s++);
			}
		else
#endif	/* X11 */
			inc_write_count(fprintf(Qstdout, "%s", s));
	}

global void
esc_write_int(int i)
{
#ifdef	X11
		if(widget_putting) {
			char buf[20];
			char *tmp = buf;
			sprintf(buf,  "%d", i);
			inc_write_count(strlen(tmp));
			while(*tmp)
				widget_put(*tmp++);
		}
		else
#endif	/* X11 */
			inc_write_count(fprintf(Qstdout, "%d", i));
}

global void
esc_put(char c)
{
	inc_write_count(1);
#ifdef	X11
	if(widget_putting)
		widget_put(c);
	else
#endif	/* X11 */
		fputc(c, Qstdout);
}
