/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

#include "bind.h"
#include "cells.h"
#include "code_area.h"
#include "data_area.h"
#include "debug.h"
#include "defs.h"
#include "delayed_problems.h"
#include "dereference.h"
#include "escape.h"
#include "errors.h"
#include "examine_term.h"
#include "execute.h"
#include "indexing.h"
#include "instructions.h"
#include "name_table.h"
#include "opcodes.h"
#include "persistent.h"
#include "pred_table.h"
#include "string_table.h"
#include "substitution.h"
#include "trail.h"
#include "unify.h"
#include "x_registers.h"
#include "write_read.h"
#include "sig.h"
#ifdef X11
#include "events.h"
#include "xwidgets.h"
#endif /* X11 */

/*----------------------------------------------------------------------------
	The QUAM emulator.
	Note:	the body of the main switch statement is processed to generate
		a LaTEX document describing the instruction set.
		Comments in this region are treated as LaTEX input.
		Comment text enclosed in double quotes (") is printed in
		italics, with the quotes removed.  The matching quotes must
		occur on the same line.
		Don't put comments between the 'when' and the end of the 'Get'
		commands.
		The switch statement is also processed by the shell script
		"MkOpCodes" to generate the file "opcodes.h"
----------------------------------------------------------------------------*/
global	int
execute(void)
{
reg	code	*pc;
reg	ENV	*current_env;
	code	*continuation_instr;
	CHOICE	*cut;
	CHOICE	*continuation_cut;
	cell	*substitution_pointer;

reg	int	i;
reg	int	j;
reg	int	k;
reg	natural	n;
reg	cell	value;
	cell	objvar;
	offset	predicate;
	natural	arity;
	offset	predicate_address;
	offset	label;
	offset	variable_label;
	offset	constant_label;
	offset	apply_label;
	offset	pair_label;
	offset	quantifier_label;
	offset	object_variable_label;
	boolean	insert;
	VALUE	temporary;
	VALUE	permanent;
	VALUE	variable;
	VALUE	term;
	ENV	*last_env;
	CHOICE	*last_last_choice_point;
	VALUE	val;
	code	*pc_save;
	cell	*stack_ptr;
	cell    *h;
	int     inserting_predicate = 0;

#ifdef	X11
int 	xcount = 0;
#endif  /* X11 */

	insert = FALSE;

	/* Initialize registers local to execute */
	pc = EntryPoint("$start", 0);

	UnGetConstant();
	UnGetNumber();
	UnGetOffset();
	GetOffset(label);
	GetNumber(arity);
	GetConstant(value);


	current_env = (ENV *) stack;
	cut = (CHOICE *) stack;

    for (;;)
	 {	/* the single space is essential for document generation */

#ifdef 	X11
	{
	if(!(xcount++))
	    xloop();
	xcount = xcount % xperiod;
	}
#endif  /* X11 */

	switch (*pc++) 
	{
	/*
	 * \section{Instruction Set}
	 *
	 * \subsection{Put instructions}
	 *
	 * The purpose of the "put" instructions
	 * is to create the data object specified by the instruction.
	 * They are mainly used in setting up the arguments for the 
	 * goals in a clause.
	 * In most of these instructions, the substitution is ignored,
	 * because any substitution has already been incorporated into
	 * the term.
	 * The substitution are built separately from the data term
	 * because this will avoid an unnecessary burden on other "put"
	 * instructions.
	 */

	when PUT_CONSTANT:
		GetConstant(value);
		GetRegister(i);
#ifdef EBUG_INST
		DebugOp(PUT_CONSTANT);
		DebugConstant(value);
		DebugNumber(i);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Place an atomic data object with value into Xi.
		 */
		X(i) = value;
	when PUT_NIL:
		GetRegister(i);
#ifdef EBUG_INST
		DebugOp(PUT_NIL);
		DebugNumber(i);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Place an empty list into Xi.
		 */
		X(i) = Atom(NIL);
	when PUT_CONS:
		GetRegister(i);
#ifdef EBUG_INST
		DebugOp(PUT_CONS);
		DebugNumber(i);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Place a list operator into Xi.
		 */
		X(i) = Atom(CONS);
	when PUT_APPLY:
		GetRegister(i);
		GetRegister(j);
		GetRegister(k);
#ifdef EBUG_INST
		DebugOp(PUT_APPLY);
		DebugNumber(i);
		DebugNumber(j);
		DebugNumber(k);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Place an application data object into Xk with the functor
		 * and the arguments taken from Xi and Xj respectively.
		 */
		value = Apply();
		Assign(Functor(value), X(i));
		Assign(Argument(value), X(j));
		X(k) = value;
	when PUT_PAIR:
		GetRegister(i);
		GetRegister(j);
		GetRegister(k);
#ifdef EBUG_INST
		DebugOp(PUT_PAIR);
		DebugNumber(i);
		DebugNumber(j);
		DebugNumber(k);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Place an argument list constructor data object into Xk with
		 * the current argument taken from Xi and the rest from Xj.
		 */
		value = Pair();
		Assign(Left(value), X(i));
		Assign(Right(value), X(j));
		X(k) = value;
	when PUT_QUANTIFIER:
		GetRegister(i);
		GetRegister(j);
		GetRegister(k);
#ifdef EBUG_INST
		DebugOp(PUT_QUANTIFIER);
		DebugNumber(i);
		DebugNumber(j);
		DebugNumber(k);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Set Xk pointing to a quantified term with Xi 
		 * containing the object variable and Xj containing 
		 * the body of the quantified term.
		 */
		value = Quantifier();
		BoundVar(value) = X(i);
		Assign(Body(value), X(j));
		X(k) = value;
	when PUT_X_VARIABLE:
		GetRegister(i);
		GetRegister(j);
#ifdef EBUG_INST
		DebugOp(PUT_X_VARIABLE);
		DebugNumber(i);
		DebugNumber(j);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Create an unbound variable on the heap and place 
		 * a reference in Xi and Xj.
		 */
		X(i) = X(j) = NewVariable();
	when PUT_Y_VARIABLE:
		GetRegister(i);
		GetRegister(j);
#ifdef EBUG_INST
		DebugOp(PUT_Y_VARIABLE);
		DebugNumber(i);
		DebugNumber(j);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Create an unbound variable in Yi in the current
		 * environment and place a reference in Xj.
		 */
		AssignYValue(i, NULL_VARIABLE);
		X(j) = Reference(&Y(i));
	when PUT_X_VALUE:
		GetRegister(i);
		GetRegister(j);
#ifdef EBUG_INST
		DebugOp(PUT_X_VALUE);
		DebugNumber(i);
		DebugNumber(j);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The value in Xi is transferred to Xj.
		 */
		X(j) = X(i);
	when PUT_Y_VALUE:
		GetRegister(i);
		GetRegister(j);
#ifdef EBUG_INST
		DebugOp(PUT_Y_VALUE);
		DebugNumber(i);
		DebugNumber(j);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The value in Yi is transferred to Xj.
		 */
		X(j) = Reference(&Y(i));
	when PUT_UNSAFE_VALUE:
		GetRegister(i);
		GetRegister(j);
#ifdef EBUG_INST
		DebugOp(PUT_UNSAFE_VALUE);
		DebugNumber(i);
		DebugNumber(j);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Rather than performing a simple value transfer as 
		 * "put_value Yi Xj",
		 * this instruction ensures the transferred value 
		 * does not refer to the current environment.
		 * This will avoid dangling pointer when the environment 
		 * is deallocated. Permanent variables 
		 * use this instruction
		 * for their last value transfer in the clause.
		 * The value in Yi is dereferenced.
		 * If the result in Xj is a variable in the current 
		 * environment and XSj is empty,
		 * Yi is bound to a new variable created on the heap and
		 * it is referred to by Xj.
		 * Otherwise, Xj is set to the dereferenced value.
		 */
		X(j) = Reference(&Y(i));
		XS(j) = EMPTY_SUB;
		variable_dereference(&X(j));
		if (CurrentEnvironment(X(j)))
		{
			BindVariable(X(j),  NewVariable());
			X(j) = Value(X(j));
		}
	when PUT_X_OBJECT_VARIABLE:
		GetRegister(i);
#ifdef EBUG_INST
		DebugOp(PUT_X_OBJECT_VARIABLE);
		DebugNumber(i);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Create an object variable on the heap and place a
		 * reference in Xi.
		 */
		X(i) = NewObjectVariable();
	when PUT_Y_OBJECT_VARIABLE:
		GetRegister(i);
#ifdef EBUG_INST
		DebugOp(PUT_Y_OBJECT_VARIABLE);
		DebugNumber(i);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Create an object variable on the heap and place a
		 * reference in Yi.
		 */
		AssignYValue(i, NewObjectVariable());
	when PUT_X_OBJECT_VALUE:
		GetRegister(i);
		GetRegister(j);
#ifdef EBUG_INST
		DebugOp(PUT_X_VALUE);
		DebugNumber(i);
		DebugNumber(j);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The value in Xi which contains an object variable
		 * is transferred to Xj.
		 */
		X(j) = X(i);
	when PUT_Y_OBJECT_VALUE:
		GetRegister(i);
		GetRegister(j);
#ifdef EBUG_INST
		DebugOp(PUT_Y_VALUE);
		DebugNumber(i);
		DebugNumber(j);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The value in Yi which has an object variable
		 * is transferred to Xj.
		 */
		X(j) = Y(i);
	when PUT_SUBSTITUTION_OPERATOR:
		GetRegister(i);
		GetRegister(j);
#ifdef EBUG_INST
		DebugOp(PUT_SUBSTITUTION_OPERATOR);
		DebugNumber(i);
		DebugNumber(j);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Join the substitution in XSi and the term in Xi with the
		 * substitution operator and place the result into Xj.
		 */
		X(j) = make_substitution(XV(i));
	when PUT_EMPTY_SUBSTITUTION:
		GetRegister(i);
#ifdef EBUG_INST
		DebugOp(PUT_EMPTY_SUBSTITUTION);
		DebugNumber(i);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Set the substitution register XSi to empty.
		 */
		XS(i) = EMPTY_SUB;
	when PUT_SUBSTITUTION:
		GetRegister(i);
		GetRegister(j);
#ifdef EBUG_INST
		DebugOp(PUT_SUBSTITUTION);
		DebugNumber(i);
		DebugNumber(j);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The substitution in XSi is transferred to XSj.
		 */
		XS(j) = XS(i);
	when PUT_PARALLEL_SUBSTITUTION:
		GetNumber(n);
		GetRegister(i);
#ifdef EBUG_INST
		DebugOp(PUT_PARALLEL_SUBSTITUTION);
		DebugNumber(n);
		DebugNumber(i);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Put a parallel substitution of size "n" into XSi.
		 * Any substitution already existed in XSi is appended
		 * to the end of the new one.
		 * The substitution pointer is set to point to the beginning
		 * so that the substitution pairs will be filled in.
		 */
		substitution_pointer = Allocate(2 * n + 1);
		XS(i) = NewSubstitution((cell)substitution_pointer, XS(i),
					OTHERS);
		*substitution_pointer++ = (cell)n;
	when PUT_PARALLEL_SUBSTITUTION_PAIR:
		GetRegister(i);
		GetRegister(j);
#ifdef EBUG_INST
		DebugOp(PUT_PARALLEL_SUBSTITUTION_PAIR);
		DebugNumber(i);
		DebugNumber(j);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Xi contains the object variable to be substituted for
		 * (i.e. the domain of the substitution).
		 * Xj contains the term to be substituted (i.e. the range
		 * of the substitution).
		 * Both of them are placed into the parallel substitution
		 * pointed by substitution pointer.
		 * After the insertion, the "substitution pointer" is advanced
		 * to the next entry.
		 */
		*substitution_pointer++ = X(i);
		*substitution_pointer++ = X(j);
	when SET_OBJECT_PROPERTY:
		GetRegister(i);
#ifdef EBUG_INST
		DebugOp(SET_OBJECT_PROPERTY);
		DebugNumber(i);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Set the property of the parallel substitution in XSi to
		 * indicate that it consists solely of object variables if
		 * the rest of the substitution is invertible or contains
		 * object variables only.
		 */
		if (!Others(NextSub(XS(i))))
			XS(i) = NewProperty(XS(i), OBJECT_VARIABLES_ONLY);
	when DETERMINE_PROPERTY:
		GetRegister(i);
#ifdef EBUG_INST
		DebugOp(DETERMINE_PROPERTY);
		DebugNumber(i);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Search the parallel substitution in XSi to 
		 * determine its property if
		 * the rest of the substitution is invertible or contains
		 * object variables only.
		 */
		if (!Others(NextSub(XS(i))))
			XS(i) = NewProperty(XS(i), determine_property(XS(i)));

	/*
	 * \subsection{Get instructions}
	 *
	 * The "get" instructions unify the head
	 * arguments with the ones in the query. It is assumed that 
	 * the head arguments do not contain any substitution. 
	 * Otherwise, the head arguments are built and general
	 * unification is called. Generally these instructions have
	 * two modes of operation. If the dereferenced argument in
	 * the query is a variable and the substitution does not yield
	 * the head argument, the instructions are operated in write
	 * mode by binding the variable with the data object in the
	 * head. If the substitution yields the head argument,
	 * the unification is delayed. In read mode, the
	 * data object is read into the clause from the query and
	 * check against the expected data objects.
	 */
	when GET_CONSTANT:
		GetConstant(value);
		GetRegister(i);
#ifdef EBUG_INST
		DebugOp(GET_CONSTANT);
		DebugConstant(value);
		DebugNumber(i);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Unify the dereferenced value in Xi with the
		 * constant in the head argument.
		 */
		if (! unify_constant(value, XV(i)))
			Backtrack();
	when GET_NIL:
		GetRegister(i);
#ifdef EBUG_INST
		DebugOp(GET_NIL);
		DebugNumber(i);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Unify the dereferenced value in Xi with the empty list.
		 */
		if (! unify_constant(Atom(NIL), XV(i)))
			Backtrack();
	when GET_CONS:
		GetRegister(i);
#ifdef EBUG_INST
		DebugOp(GET_CONS);
		DebugNumber(i);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Unify the dereferenced value in Xi with the list operator.
		 */
		if (! unify_constant(Atom(CONS), XV(i)))
			Backtrack();
	when GET_APPLY:
		GetRegister(i);
		GetRegister(j);
		GetRegister(k);
#ifdef EBUG_INST
		DebugOp(GET_APPLY);
		DebugNumber(i);
		DebugNumber(j);
		DebugNumber(k);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Unify the dereferenced value in Xk with an
		 * application in the head.
		 * Xi and Xj are set to the functor and the arguments
		 * of the application respectively.
		 */
		dereference(XV(k));
				 
		switch (Tag(X(k)))
		{
		when APPLY:
						 
			value = X(k);
			X(i) = Reference(&Functor(value));
			X(j) = Reference(&Argument(value));
			XS(i) = XS(j) = XS(k);
		when REFERENCE:
			if (Frozen(X(k)))
				Backtrack();
			else
			{
				value = Apply();
						 
				if (yield_tag(XS(k), APPLY))
				{
					delay(UNIFY, Location(X(k)),
					      make_substitution(XV(k)), value);
					XS(i) = XS(j) = EMPTY_SUB;
				}
				else
				{
					Bind(X(k), value);
					XS(i) = XS(j) = XS(k);
				}
				X(i) = Reference(&Functor(value));
				X(j) = Reference(&Argument(value));
			}
		when OBJECT_REFERENCE:
			if (yield_tag(XS(k), APPLY))
			{
				value = Apply();
				delay(UNIFY, Location(X(k)),
				      make_substitution(XV(k)), value);
				XS(i) = XS(j) = EMPTY_SUB;
				X(i) = Reference(&Functor(value));
				X(j) = Reference(&Argument(value));
			}
			else
				Backtrack();
		otherwise:
			Backtrack();
		}
	when GET_PAIR:
		GetRegister(i);
		GetRegister(j);
		GetRegister(k);
#ifdef EBUG_INST
		DebugOp(GET_PAIR);
		DebugNumber(i);
		DebugNumber(j);
		DebugNumber(k);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Unify the dereferenced value in Xk with an argument
		 * list in the head. Xi and Xj are set to the first
		 * and the rest of the arguments respectively.
		 */
		dereference(XV(k));
		switch (Tag(X(k)))
		{
		when PAIR:
			value = X(k);
			X(i) = Reference(&Left(value));
			X(j) = Reference(&Right(value));
			XS(i) = XS(j) = XS(k);
		when REFERENCE:
			if (Frozen(X(k)))
				Backtrack();
			else
			{
				value = Pair();
				if (yield_tag(XS(k), PAIR))
				{
					delay(UNIFY, Location(X(k)),
					      make_substitution(XV(k)), value);
					XS(i) = XS(j) = EMPTY_SUB;
				}
				else
				{
					Bind(X(k), value);
					XS(i) = XS(j) = XS(k);
				}
				X(i) = Reference(&Left(value));
				X(j) = Reference(&Right(value));
			}
		when OBJECT_REFERENCE:
			if (yield_tag(XS(k), PAIR))
			{
				value = Pair();
				delay(UNIFY, Location(X(k)),
				      make_substitution(XV(k)), value);
				XS(i) = XS(j) = EMPTY_SUB;
				X(i) = Reference(&Left(value));
				X(j) = Reference(&Right(value));
			}
			else
				Backtrack();
		otherwise:
			Backtrack();
		}
	when GET_QUANTIFIER:
		GetRegister(i);
		GetRegister(j);
		GetRegister(k);
#ifdef EBUG_INST
		DebugOp(GET_QUANTIFIER);
		DebugNumber(i);
		DebugNumber(j);
		DebugNumber(k);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The instruction unifies a quantified term in Xk.
		 * Xi must be initialized to an object variable before
		 * the instruction is executed.
		 * Xj contains the body of the quantified term after
		 * the unification.
		 * The "not_free_in" check for the object variable Xi in
		 * the quantified term Xk is performed by the "not_free_in"
		 * instruction separately.
		 */
		dereference(XV(k));
		switch (Tag(X(k)))
		{
		when QUANTIFIER:
			value = NewLocalObjectVariable();
			XS(j) = add_substitution(XS(k), RenameSub(value, X(i),
								  EMPTY_SUB));
			XS(j) = RenameSub(BoundVar(X(k)), value, XS(j));
			X(j) = Reference(&Body(X(k)));
		when REFERENCE:
			if (Frozen(X(k)))
				Backtrack();
			else
			{
				value = Quantifier();
				BoundVar(value) = X(i);
				if (XS(k) == EMPTY_SUB)
					Bind(X(k), value);
				else if (yield_tag(XS(k), QUANTIFIER))
				{
					delay(UNIFY, Location(X(k)),
					      make_substitution(XV(k)), value);
					XS(j) = EMPTY_SUB;
				}
				else
				{
					BoundVar(value) = NewObjectVariable();
					Bind(X(k), value);
					objvar = NewLocalObjectVariable();
					XS(j) = add_substitution(XS(k),
							RenameSub(objvar, X(i),
								  EMPTY_SUB));
					XS(j) = RenameSub(BoundVar(value),
							  objvar, XS(j));
				}
				X(j) = Reference(&Body(value));
			}
		when OBJECT_REFERENCE:
			if (yield_tag(XS(k), QUANTIFIER))
			{
				value = Quantifier();
				BoundVar(value) = X(i);
				delay(UNIFY, Location(X(k)),
				      make_substitution(XV(k)), value);
				X(j) = Reference(&Body(value));
				XS(j) = EMPTY_SUB;
			}
			else
				Backtrack();
		otherwise:
			Backtrack();
		}
	when GET_X_VARIABLE:
		GetRegister(i);
		GetRegister(j);
#ifdef EBUG_INST
		DebugOp(GET_X_VARIABLE);
		DebugNumber(i);
		DebugNumber(j);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The instruction is used for unifying an unbound
		 * variable in the head with an argument in the query.
		 * Because a variable can unify with any data object,
		 * the values in Xj and XSj are simply transferred to Xi
		 * and XSi.
		 */
		X(i) = X(j);
		XS(i) = XS(j);
	when GET_Y_VARIABLE:
		GetRegister(i);
		GetRegister(j);
#ifdef EBUG_INST
		DebugOp(GET_Y_VARIABLE);
		DebugNumber(i);
		DebugNumber(j);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The instruction is used for unifying an unbound
		 * variable in the head with an argument in the query.
		 * Because a variable can unify with any data object,
		 * a substitution operator is built from the values in
		 * Xj and XSj and transferred to Yi.
		 */
		AssignYValue(i, make_substitution(XV(j)));
	when GET_X_VALUE:
		GetRegister(i);
		GetRegister(j);
#ifdef EBUG_INST
		DebugOp(GET_X_VALUE);
		DebugNumber(i);
		DebugNumber(j);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The register Xi is bound to some data object by the
		 * previous "get_variable" instruction.
		 * This instruction unifies the data objects in Xi and Xj.
		 */
		if (! unify(XV(i), XV(j)))
			Backtrack();
	when GET_Y_VALUE:
		GetRegister(i);
		GetRegister(j);
#ifdef EBUG_INST
		DebugOp(GET_Y_VALUE);
		DebugNumber(i);
		DebugNumber(j);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The register Yi is bound to some data object by the
		 * previous "get_variable" instruction.
		 * This instruction unifies the data objects in Yi and Xj.
		 */
		permanent.term = Reference(&Y(i));
		permanent.sub = EMPTY_SUB;
		if (! unify(&permanent, XV(j)))
			Backtrack();
	when GET_X_OBJECT_VARIABLE:
		GetRegister(i);
		GetRegister(j);
#ifdef EBUG_INST
		DebugOp(GET_X_OBJECT_VARIABLE);
		DebugNumber(i);
		DebugNumber(j);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The instruction unifies an unbound object variable
		 * in the head with the argument in the query.
		 * If the query has an unbound variable, it is bound to a
		 * newly created object variable.
		 * If it is an object variable, Xi is set pointing to it.
		 * If it is anything else, the instruction fails because an
		 * object variable cannot unified with any other data object.
		 */
		dereference(XV(j));
		switch (Tag(X(j)))
		{
		when REFERENCE:
			if (Frozen(X(j)))
				Backtrack();
			else
			{
				value = NewObjectVariable();
				if (Invertible(XS(j)))
				{
					temporary.sub = EMPTY_SUB;
					temporary.term = value;
					if (invert(XS(j), &temporary))
						Bind(X(j), make_substitution(
								&temporary));
					else
						Backtrack();
				}
				else if (yield_object_variable(XS(j), value))
				{
					delay(UNIFY, Location(X(j)),
					      make_substitution(XV(j)), value);
                                }
				else
				{
					Bind(X(j), value);
					generate_distinction_information(value,
									XS(j));
				}
				X(i) = value;
				XS(i) = EMPTY_SUB;
			}
		when OBJECT_REFERENCE:
			if (XS(j) == EMPTY_SUB)
				X(i) = X(j);
			else if (yield_any_object_variable(XS(j)))
			{
				value = NewObjectVariable();
				delay(UNIFY, Location(value),
				      value, make_substitution(XV(j)));
				X(i) = value;
			}
			else
			{
				X(i) = X(j);
				generate_distinction_information(X(i), XS(j));
			}
			XS(i) = EMPTY_SUB;
		otherwise:
			Backtrack();
		}
	when GET_Y_OBJECT_VARIABLE:
		GetRegister(i);
		GetRegister(j);
#ifdef EBUG_INST
		DebugOp(GET_Y_OBJECT_VARIABLE);
		DebugNumber(i);
		DebugNumber(j);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The instruction unifies an unbound object variable
		 * in the head with the argument in the query.
		 * If the query has an unbound variable, it is bound to a
		 * newly created object variable.
		 * If it is an object variable, Yi is set pointing to it.
		 * If it is anything else, the instruction fails because an
		 * object variable cannot unified with any other data object.
		 */
		dereference(XV(j));
		switch (Tag(X(j)))
		{
		when REFERENCE:
			if (Frozen(X(j)))
				Backtrack();
			else
			{
				AssignYValue(i, NewObjectVariable());
				if (Invertible(XS(j)))
				{
					temporary.sub = EMPTY_SUB;
					temporary.term = Y(i);
					if (invert(XS(j), &temporary))
						Bind(X(j), make_substitution(
								&temporary));
					else
						Backtrack();
				}
				else if (yield_object_variable(XS(j), Y(i)))
				{
					delay(UNIFY, Location(X(j)),
					      make_substitution(XV(j)), Y(i));
                                }
				else
				{
					Bind(X(j), Y(i));
					generate_distinction_information(Y(i),
									XS(j));
				}
				XS(i) = EMPTY_SUB;
			}
		when OBJECT_REFERENCE:
			if (XS(j) == EMPTY_SUB)
				AssignYValue(i, X(j));
			else if (yield_any_object_variable(XS(j)))
			{
				AssignYValue(i, NewObjectVariable());
				delay(UNIFY, Location(Y(i)),
				      Y(i), make_substitution(XV(j)));
			}
			else
			{
				AssignYValue(i, X(j));
				generate_distinction_information(Y(i), XS(j));
			}
			XS(i) = EMPTY_SUB;
		otherwise:
			Backtrack();
		}
	when GET_X_OBJECT_VALUE:
		GetRegister(i);
		GetRegister(j);
#ifdef EBUG_INST
		DebugOp(GET_X_OBJECT_VALUE);
		DebugNumber(i);
		DebugNumber(j);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The register Xi is bound to an object variable by
		 * the previous "get_object_variable" instruction.
		 * This instruction unifies the object variable in Xi
		 * and data object in Xj.
		 */
		dereference(XV(i));
		if (! unify_object_variable(XV(i), XV(j)))
			Backtrack();
	when GET_Y_OBJECT_VALUE:
		GetRegister(i);
		GetRegister(j);
#ifdef EBUG_INST
		DebugOp(GET_Y_OBJECT_VALUE);
		DebugNumber(i);
		DebugNumber(j);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The register Yi is bound to an object variable by
		 * the previous "get_object_variable" instruction.
		 * This instruction unifies the object variable in Yi
		 * and data object in Xj.
		 */
		permanent.term = Y(i);
		permanent.sub = EMPTY_SUB;
		dereference(&permanent);
		if (! unify_object_variable(&permanent, XV(j)))
			Backtrack();

	/*
	 * \subsection{Procedural instructions}
	 *
	 * The procedural instructions are responsible for the control
	 * flow in an abstract machine program.
	 */
	when CALL_PREDICATE:
		GetPredAtom(predicate);
		GetNumber(arity);
		GetNumber(n);
		if(inserting_predicate = is_signal()){
		    /* create choice point */
		    X(1) = build_goal(predicate, arity);
		    XS(1) = EMPTY_SUB;
		    predicate = get_signal_handler();
		    arity = 2;
		}
#ifdef EBUG_INST
		DebugOp(CALL_PREDICATE);
		DebugString(predicate);
		DebugNumber(arity);
		DebugNumber(n);
		Debugln;
#endif /* EBUG_INST */
#ifdef EBUG
		spy("call", string_table+predicate, arity);
#endif  /* EBUG */
		/*
		 * The address of the "Predicate" is looked up from the
		 * predicate table before the "Predicate" is called.
		 * The return address to the current clause is recorded as
		 * the continuation instruction.
		 * "n" indicates the number of Y registers
		 * remaining in the environment.
		 * This is known as environment trimming and it is performed
		 * only when the current environment is on the top of the
		 * local stack.
		 * The cut choice point is also recorded.
		 * The instruction is used for all the
		 * predicate calls in the clause except the last one.
		 */
	call_predicate:
		continuation_cut = cut;
		cut = last_choice_point;
		continuation_instr = pc;
		if (!inserting_predicate
		    &&
                    DefinedCompiledPredicate(Address(Predicate(predicate,
								arity))))
			pc = UpdateCallInstruction(Address(
					Predicate(predicate, arity)));
		else
		{
			X(0) = build_goal(predicate, arity);
			pc = EntryPoint("call", 1);
		}
		if (TopOfLocalStack(current_env, last_choice_point))
		{
			current_env->num_permanents = n;
			top_of_stack = (cell *)((char *)current_env +
						EnvSize(n));
		}
	when CALL_ADDRESS:
		GetAddress(predicate_address);
		GetNumber(n);
		if(inserting_predicate = is_signal()){
                    pc_save = pc;
                    pc = code_area + predicate_address;
                    UnGetConstant();
                    UnGetNumber();
                    UnGetOffset();
                    GetOffset(predicate);
                    GetNumber(arity);
                    pc = pc_save;

                    X(1) = build_goal(predicate, arity);
                    XS(1) = EMPTY_SUB;
                    predicate = get_signal_handler();
                    arity = 2;
                    goto call_predicate;
		}
#ifdef EBUG_INST
		pc_save = pc;
		pc = code_area + predicate_address;
		UnGetConstant();
		UnGetNumber();
		UnGetOffset();
		GetOffset(predicate);
		GetNumber(arity);
		GetConstant(value);
		pc = pc_save;
		DebugOp(CALL_ADDRESS);
		DebugString(predicate);
		DebugNumber(arity);
		DebugNumber(n);
		Debugln;
#endif /* EBUG_INST */
#ifdef EBUG
		pc_save = pc;
		pc = code_area + predicate_address;
		UnGetConstant();
		UnGetNumber();
		UnGetOffset();
		GetOffset(predicate);
		GetNumber(arity);
		GetConstant(value);
		pc = pc_save;
		spy("call", string_table+predicate, arity);
#endif  /* EBUG */
		/*
		 * The address for the "Predicate" is known and the
		 * "Predicate" is called without looking up via any table.
		 * The return address to the current clause is recorded as
		 * the continuation instruction.
		 * "n" indicates the number of Y registers
		 * remaining in the environment.
		 * As with "call_predicate" environment trimming is performed.
		 * The cut choice point is also recorded.
		 * The instruction is used for all the
		 * predicate calls in the clause except the last one.
		 */
		continuation_cut = cut;
		cut = last_choice_point;
		continuation_instr = pc;
		pc = Address(predicate_address);
		if (TopOfLocalStack(current_env, last_choice_point))
		{
			current_env->num_permanents = n;
			top_of_stack = (cell *)(((char *)current_env) +
						EnvSize(n));
		}
	when EXECUTE_PREDICATE:
		GetPredAtom(predicate);
		GetNumber(arity);
		if(inserting_predicate = is_signal()){
                    X(1) = build_goal(predicate, arity);
                    XS(1) = EMPTY_SUB;
                    predicate = get_signal_handler();
                    arity = 2;
                }
#ifdef EBUG_INST
		DebugOp(EXECUTE_PREDICATE);
		DebugString(predicate);
		DebugNumber(arity);
		Debugln;
#endif /* EBUG_INST */
#ifdef EBUG
		spy("call", string_table+predicate, arity);
#endif  /* EBUG */
		/*
		 * Similar to "call_predicate" instruction, the address of
		 * the "Predicate" is obtained via the predicate table
		 * before the call is made.
		 * This instruction is used in the final predicate
		 * call in a clause and it indicates the termination
		 * of the current clause.
		 * The cut choice point is also recorded.
		 */
	execute_predicate:
		cut = last_choice_point;
                if (!inserting_predicate
                    &&
                    DefinedCompiledPredicate(Address(Predicate(predicate,
								arity))))
			pc = UpdateExecuteInstruction(Address(
					Predicate(predicate, arity)));
		else
		{
			X(0) = build_goal(predicate, arity);
			pc = EntryPoint("call", 1);
		}
	when EXECUTE_ADDRESS:
		GetAddress(predicate_address);
		if(inserting_predicate = is_signal()){
                    pc_save = pc;
                    pc = code_area + predicate_address;
                    UnGetConstant();
                    UnGetNumber();
                    UnGetOffset();
                    GetOffset(predicate);
                    GetNumber(arity);
                    pc = pc_save;

                    X(1) = build_goal(predicate, arity);
                    XS(1) = EMPTY_SUB;
                    predicate = get_signal_handler();
                    arity = 2;
                    goto execute_predicate;
                }
#ifdef EBUG_INST
		pc_save = pc;
		pc = code_area + predicate_address;
		UnGetConstant();
		UnGetNumber();
		UnGetOffset();
		GetOffset(predicate);
		GetNumber(arity);
		GetConstant(value);
		pc = pc_save;
		DebugOp(EXECUTE_ADDRESS);
		DebugString(predicate);
		DebugNumber(arity);
		Debugln;
#endif /* EBUG_INST */
#ifdef EBUG
		pc_save = pc;
		pc = code_area + predicate_address;
		UnGetConstant();
		UnGetNumber();
		UnGetOffset();
		GetOffset(predicate);
		GetNumber(arity);
		GetConstant(value);
		pc = pc_save;
		spy("call", string_table+predicate, arity);
#endif  /* EBUG */
		/*
		 * The address of the "Predicate" is known and the
		 * "Predicate" is called.
		 * This instruction is used in the final predicate
		 * call in a clause and it indicates the termination
		 * of the current clause.
		 * The cut choice point is also recorded.
		 */
		cut = last_choice_point;
		pc = Address(predicate_address);
	when PROCEED:
#ifdef EBUG_INST
		DebugOp(PROCEED);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * This instruction indicates the end of a unit clause.
		 * The computation continues from after the 
		 * most recent "call" instruction
		 * (this location is called the "continuation instruction").
		 */
		cut = continuation_cut;
		pc = continuation_instr;
	when GOTO:
#ifdef EBUG_INST
		pc_save = pc;
		pc = code_area + IntOf(Xdref(0));
		UnGetConstant();
		UnGetNumber();
		UnGetOffset();
		GetOffset(predicate);
		GetNumber(arity);
		GetConstant(value);
		pc = pc_save;
		DebugOp(GOTO);
		DebugString(predicate);
		DebugNumber(arity);
		Debugln;
#endif /* EBUG_INST */
#ifdef EBUG
		pc_save = pc;
		variable.sub = XS(0);
		variable.term = X(0);
		term.sub = XS(1);
		term.term = X(1);
		pc = code_area + IntOf(Xdref(0));
		UnGetConstant();
		UnGetNumber();
		UnGetOffset();
		GetOffset(predicate);
		GetNumber(arity);
		GetConstant(value);
		setup_x_registers(XV(1));
		spy("call", string_table+predicate, arity);
		XS(1) = term.sub;
		X(1) = term.term;
		XS(0) = variable.sub;
		X(0) = variable.term;
		pc = pc_save;
#endif  /* EBUG */
		/*
		 * Used to implement the call to compiled predicate
		 * from the interpreter.
		 */
		dereference(XV(0));
		if (IsInteger(X(0)))
		{
			pc = code_area + IntOf(X(0));
			setup_x_registers(XV(1));
		}
		else
			Backtrack();
	when ESCAPE:
		GetNumber(n);
		pc_save = pc - SizeOfNumber - 1; /* for redo */
                if(inserting_predicate = is_signal()){
                    /* CALL_PREDICATE */
                    X(1) = build_goal(add_new_string(EscapePredicateName(n)),
                                      EscapePredicateArity(n));
                    XS(1) = EMPTY_SUB;
                    predicate = get_signal_handler();
                    arity = 2;
		    continuation_cut = cut;
                    cut = last_choice_point;
                    continuation_instr = pc;
                    X(0) = build_goal(predicate, arity);
                    XS(0) = EMPTY_SUB;
                    pc = EntryPoint("call", 1);
                    break;
                }
#ifdef EBUG_INST
		DebugOp(ESCAPE);
		DebugString2(EscapePredicateName(n));
		DebugNumber(EscapePredicateArity(n));
		Debugln;
#endif /* EBUG_INST */
#ifdef EBUG
		spy("call", EscapePredicateName(n), EscapePredicateArity(n));
#endif  /* EBUG */
		/*
		 * This instruction calls a QuAM internal C function.
		 */
                switch (EscapePredicate(n))
		{
		when FALSE:
#ifdef EBUG
			spy("fail", EscapePredicateName(n),
				EscapePredicateArity(n));
#endif  /* EBUG */
			Backtrack();
		when REDO:
                        /* restart instruction processing */
                        pc = pc_save;
                when TRUE:

#ifdef EBUG
			spy("exit", EscapePredicateName(n), 
                                    EscapePredicateArity(n));
#endif  /* EBUG */
			;
		}
	when FAIL:
#ifdef EBUG_INST
		DebugOp(FAIL);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The instruction invokes backtracking.
		 */
	backtrack:

		top_of_stack = (cell *)((char *)last_choice_point +
			       ChoiceSize(last_choice_point->num_registers));
                meta_var_extention = last_choice_point->meta_var_extention;
                object_var_extention = last_choice_point->object_var_extention;
		current_env = last_choice_point->current_env;
		continuation_instr = last_choice_point->continuation_instr;
		continuation_cut = last_choice_point->continuation_cut;
		cut = last_choice_point->cut;
		top_of_heap = last_choice_point->top_of_heap;
		top_delayed_stack = last_choice_point->top_delayed_stack;
		delayed_problems_pointer = NULL;
		while (top_of_trail > last_choice_point->top_of_trail)
		{
			top_of_trail--;
			if (top_of_trail->address != NULL)
				* (top_of_trail->address) =
					top_of_trail->previous_value;
		}
		for (i = 0; i < last_choice_point->num_registers; i++)
		{
			XS(i) = last_choice_point->x[i].sub;
			X(i) = last_choice_point->x[i].term;
		}
		pc = last_choice_point->next_clause;
	when CUT:
#ifdef EBUG_INST
		DebugOp(CUT);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * This instruction represents the cut operator in a clause.
		 * All the choice points created after the current predicate,
		 * and the one for the current predicate, are removed.
		 */
		if (TopOfLocalStack(last_choice_point, current_env))
			if (TopOfLocalStack(cut, current_env))
				top_of_stack = ((cell *)cut) +
					ChoiceSize(cut->num_registers);
			else
			{
				while (TopOfLocalStack(
					last_choice_point->last_choice_point,
					current_env))
				{
					last_choice_point = 
					 last_choice_point->last_choice_point;
				}
				top_of_stack = (cell *)last_choice_point;
			}
		last_choice_point = cut;
	when DO_DELAYED_PROBLEMS:
#ifdef EBUG_INST
		DebugOp(DO_DELAYED_PROBLEMS);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * This instruction executes all the delay problems that are
		 * woken up by the head unification in the current clause.
		 */

                for (;
                     delayed_problems_pointer != NULL;
                     delayed_problems_pointer = Next(delayed_problems_pointer))
                {
                        if (DelayedSolved(delayed_problems_pointer) == UNSOLVED)
                        {
                                Solved(delayed_problems_pointer);
                                if (!do_delayed_problem(
                                        delayed_problems_pointer))
                                {
                                        Backtrack();
                                }
                                if (delayed_problems_pointer == NULL)  
                                {
                                        break;
                                }
                        }
                }

	when NOT_FREE_IN:
		GetRegister(i);
		GetRegister(j);
#ifdef EBUG_INST
		DebugOp(NOT_FREE_IN);
		DebugNumber(i);
		DebugNumber(j);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The object variable in Xi does not occur free in
		 * the term held in Xj.
		 */
		dereference(XV(i));
		if (freeness(X(i), XV(j)))
			Backtrack();

	/*
	 * \subsection{Memory management instructions}
	 * 
	 * The instructions are used to allocate and deallocate
	 * environments for clauses during the computation.
	 */
	when ALLOCATE:
		GetNumber(n);
#ifdef EBUG_INST
		DebugOp(ALLOCATE);
		DebugNumber(n);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * A new environment is allocated on the "local stack" to hold 
		 * "n" permanent variables from the current clause and
		 * other information.
		 */
		last_env = current_env;
		current_env = (ENV *)top_of_stack;
		StackAlloc(EnvSize(n));
		current_env->continuation_instr = continuation_instr;
		current_env->current_env = last_env;
		current_env->continuation_cut = continuation_cut;
		current_env->num_permanents = n;
	when DEALLOCATE:
#ifdef EBUG_INST
		DebugOp(DEALLOCATE);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * This instruction appears before the end of a clause.
		 * The program counter and current environment are restored
		 * to their previous values so that computation can continue
		 * with the rest of the goals in the previous clause.
		 * Although the environment appears to be deallocated,
		 * it is not necessary so.
		 * If the environment is shared by more than one search path,
		 * the environment must remain in the stack until all
		 * search paths are exhausted.
		 * The shared enironment is protected against deallocation
		 * by having a choice point after its creation.
		 */
		if (TopOfLocalStack(current_env, last_choice_point))
			top_of_stack = (cell *)current_env;
		continuation_instr = current_env->continuation_instr;
		continuation_cut = current_env->continuation_cut;
		current_env = current_env->current_env;
		top_of_stack =
			TopOfLocalStack(current_env, last_choice_point) ?
			((cell *)current_env) +
				EnvSize(current_env->num_permanents) :
			((cell *)last_choice_point) +
				ChoiceSize(last_choice_point->num_registers);

	/*
	 * \subsection{Choice point manipulation instructions}
	 *
	 * The choice points are manipulated by this set of instructions.
	 * Choice points are created by "try_me_else" or "try" and removed by
	 * "trust_me_else" or "trust".
	 * The updating on choice points are done by the instructions
	 * "retry_me_else" or "retry".
	 * This group of instructions can be sub-divided into
	 * "try_me_else" or "try" which are complement to each other.
	 * The jump provided by the instructions is
	 * relative to the program counter.
	 */
	when TRY_ME_ELSE:
		GetNumber(n);
		GetOffset(label);
#ifdef EBUG_INST
		DebugOp(TRY_ME_ELSE);
		DebugNumber(n);
		DebugNumber(label);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * This instruction is the first of a sequence of
		 * instructions which link groups of clauses with
		 * different types/values from the first argument
		 * in a predicate together.
		 * A choice point is created by this instruction.
		 */
		last_last_choice_point = last_choice_point;
		last_choice_point = (CHOICE *)top_of_stack;
		StackAlloc(ChoiceSize(n));
		last_choice_point->meta_var_extention = meta_var_extention;
		last_choice_point->object_var_extention = object_var_extention;
		last_choice_point->current_env = current_env;
		last_choice_point->continuation_instr = continuation_instr;
		last_choice_point->continuation_cut = continuation_cut;
		last_choice_point->cut = cut;
		last_choice_point->last_choice_point = last_last_choice_point;
		last_choice_point->next_clause = pc + label;
		last_choice_point->top_of_trail = top_of_trail;
		last_choice_point->top_of_heap = top_of_heap;
		last_choice_point->top_delayed_stack = top_delayed_stack;
		last_choice_point->num_registers = n;
		for (i = 0; i < n; i++)
		{
			last_choice_point->x[i].sub = XS(i);
			last_choice_point->x[i].term = X(i);
		}
	when RETRY_ME_ELSE:
		GetOffset(label);
#ifdef EBUG_INST
		DebugOp(RETRY_ME_ELSE);
		DebugNumber(label);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The instruction occurs in the middle of the link
		 * between different groups of clauses with different
		 * types/values from the first argument in a predicate.
		 */
		last_choice_point->next_clause = pc + label;
	when TRUST_ME_ELSE_FAIL:
#ifdef EBUG_INST
		DebugOp(TRUST_ME_ELSE_FAIL);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The instruction appears as the last of the link
		 * for the groups of clauses with different types/values
		 * from the first argument in a predicate.
		 * The current choice point is deleted.
		 */
		top_of_stack = (cell *)last_choice_point;
		last_choice_point = last_choice_point->last_choice_point;
	when TRY:
		GetNumber(n);
		GetOffset(label);
#ifdef EBUG_INST
		DebugOp(TRY);
		DebugNumber(n);
		DebugNumber(label);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * This instruction is similar to the instruction
		 * "try_me_else".
		 * It is the first of a sequence of instructions which link
		 * groups of clauses with the same types/values from the
		 * first argument in a predicate together.
		 * A choice point is created by this instruction.
		 */
		last_last_choice_point = last_choice_point;
		last_choice_point = (CHOICE *)top_of_stack;
		StackAlloc(ChoiceSize(n));
		last_choice_point->meta_var_extention = meta_var_extention;
		last_choice_point->object_var_extention = object_var_extention;
		last_choice_point->current_env = current_env;
		last_choice_point->continuation_instr = continuation_instr;
		last_choice_point->continuation_cut = continuation_cut;
		last_choice_point->cut = cut;
		last_choice_point->last_choice_point = last_last_choice_point;
		last_choice_point->next_clause = pc;
		last_choice_point->top_of_trail = top_of_trail;
		last_choice_point->top_of_heap = top_of_heap;
		last_choice_point->top_delayed_stack = top_delayed_stack;
		last_choice_point->num_registers = n;
		for (i = 0; i < n; i++)
		{
			last_choice_point->x[i].sub = XS(i);
			last_choice_point->x[i].term = X(i);
		}
		pc += label;
	when RETRY:
		GetOffset(label);
#ifdef EBUG_INST
		DebugOp(RETRY);
		DebugNumber(label);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The instruction is similar to the "retry_me_else"
		 * instruction.
		 * It occurs in the middle of the link between different
		 * groups of clauses with the same types/values from the first
		 * argument in a predicate.
		 */
		last_choice_point->next_clause = pc;
		pc += label;
	when TRUST:
		GetOffset(label);
#ifdef EBUG_INST
		DebugOp(TRUST);
		DebugNumber(label);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The instruction is similar to the "trust_me_else_fail"
		 * instruction.
		 * It appears as the last of the link for the groups of
		 * clauses with the same types/values from the first
		 * argument in a predicate.
		 * The current choice point is deleted.
		 */
		top_of_stack = (cell *)last_choice_point;
		last_choice_point = last_choice_point->last_choice_point;
		pc += label;

	/*
	 * \subsection{Indexing instructions}
	 *
	 * The purpose of the indexing instructions are to filter out
	 * those clauses which cannot unify with a goal, and link
	 * the candidate clauses together.
	 * The jump provided by the instructions is relative to the
	 * program counter.
	 */
	when SWITCH_ON_TERM:
		GetOffset(variable_label);
		GetOffset(constant_label);
		GetOffset(apply_label);
		GetOffset(pair_label);
		GetOffset(quantifier_label);
		GetOffset(object_variable_label);
#ifdef EBUG_INST
		DebugOp(SWITCH_ON_TERM);
		DebugNumber(variable_label);
		DebugNumber(constant_label);
		DebugNumber(apply_label);
		DebugNumber(pair_label);
		DebugNumber(quantifier_label);
		DebugNumber(object_variable_label);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * Depending on the type of argument after the register "X0"
		 * is dereferenced, control is transferred to
		 * "variable_label", "constant_label", "apply_label",
		 * "pair_label",
		 * "quantifier_label" or "object_variable_label" for a
		 * variable, constant, application, argument list constructor,
		 * quantifier or object variable respectively.
		 * If there is a substitution, "variable_label" is used because
		 * every clause in the predicate must be tried,
		 * as in the case of variable.
		 */
		dereference(XV(0));
		if (XS(0) != EMPTY_SUB)
			label = variable_label;
		else
			switch (Tag(X(0)))
			{
			when CONSTANT:
				label = constant_label;
			when APPLY:
				value = fetch_functor(X(0));
				if (IsVariable(value) || IsReference(value))
					label = variable_label;
				else
				{
					term.sub = EMPTY_SUB;
					term.term =
						Reference(&Argument(X(0)));
					dereference(&term);
					if (Tag(term.term) == QUANTIFIER)
						label = quantifier_label;
					else
						label = apply_label;
				}
			when PAIR:
				label = pair_label;
			when REFERENCE:
				label = variable_label;
			when OBJECT_REFERENCE:
				label = object_variable_label;
			}
		if (label == NO_MATCH)
			Backtrack();
		else
			pc += label;
	when SWITCH_ON_CONSTANT:
		GetNumber(n);
#ifdef EBUG_INST
		DebugOp(SWITCH_ON_CONSTANT);
		DebugNumber(n);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * A group of clauses with constants in the first argument
		 * are hashed into a hash table "Table" of sized $2^n$.
		 * Each entry in the table is a
		 * (value of constant, address of clause) pair.
		 * Control is transferred to the clause depending on the
		 * constant in the register "X0".
		 * The "default" branch is chosen if the constant is not in
		 * the table.
		 */
		dereference(XV(0));
		label = lookup_constant_table(X(0), pc, n);
		if (label == NO_MATCH)
			Backtrack();
		else
			pc += label;
	when SWITCH_ON_STRUCTURE:
		GetNumber(n);
#ifdef EBUG_INST
		DebugOp(SWITCH_ON_STRUCTURE);
		DebugNumber(n);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * This instruction is similar to the instruction
		 * "switch_on_constant".
		 * The functor and arity of the structures from the first
		 * argument are hashed into the "Table" instead of the
		 * constant.
		 */
		dereference(XV(0));
		label = lookup_structure_table(fetch_functor(X(0)),
					       fetch_arity(X(0)), pc, n);
		if (label == NO_MATCH)
			Backtrack();
		else
			pc += label;
	when SWITCH_ON_QUANTIFIER:
		GetNumber(n);
#ifdef EBUG_INST
		DebugOp(SWITCH_ON_QUANTIFIER);
		DebugNumber(n);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * This instruction is similar to the instruction
		 * "switch_on_structure".
		 * The quantifier and arity from the first
		 * argument are hashed into the "Table" instead of the
		 * functor and arity.
		 */
		dereference(XV(0));
		label = lookup_structure_table(fetch_functor(X(0)),
					       fetch_arity(X(0)), pc, n);
		if (label == NO_MATCH)
			Backtrack();
		else
			pc += label;

	/*
	 * \subsection{Persistent data manipulation instructions}
	 *
	 * This section discusses all instructions concerning the
	 * persistent data objects.
	 * The details of the instructions are not included in this
	 * report.
	 */
	when COMMIT:
#ifdef EBUG_INST
		DebugOp(COMMIT);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The instruction commits the bindings of the persistent
		 * variables by copying them into the persistent stack.
		 */
		initialise_commit();
		if (! retry_delay() || test_delayed_problem(UNIFY))
			Backtrack();
		else
		{

			cell * old_top = top_of_pstack;

			stack_ptr = top_of_ptrail;
			for (i = MaxPersistentVariables; i >= 0; i--)
				if (! Committed(P(i)))
					*P(i) = commit_copy(P(i));
			for (; delayed_problems_pointer != NULL;
			     delayed_problems_pointer =
					Next(delayed_problems_pointer))
			{
				if (!do_delayed_problem(delayed_problems_pointer))
					Backtrack();
			}
			for (i = MaxPersistentVariables; i >= 0; i--)
				*P(i) = commit_not_free(P(i));
			*top_of_ptrail++ = (cell)old_top;
			*top_of_ptrail++ = (cell)stack_ptr;
			*top_of_ptrail++ = CommitStage(stack_ptr) + 1;
		}
	when BACK_TO:
#ifdef EBUG_INST
		DebugOp(BACK_TO);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The instruction returns to the commit stage in the
		 * persistent stack specified in "X0".
		 * All the committed bindings made between current stage
		 * and the designated stage are revoked.
		 */
                dereference(XV(0));
                if (! IsInteger(X(0)))
                {
                        warning("back_to argument not an integer");
                        Backtrack();
                }
                else if (IntOf(X(0)) >= 0)
			while (X(0) < Integer(CommitStage(top_of_ptrail)))
			{
				stack_ptr = top_of_ptrail - 4;
				top_of_pstack = (cell *)*(top_of_ptrail - 3);
				top_of_ptrail = (cell *)*(top_of_ptrail - 2);
				while (stack_ptr >= top_of_ptrail)
				{
					*(cell *)(RestOfTemp(*stack_ptr)) =
						Temperature(*stack_ptr) |
						NULL_VARIABLE;
					Uncommit((cell *)RestOfTemp(
								*stack_ptr));
					stack_ptr--;
				}
			}
	when COMMIT_STAGE:
#ifdef EBUG_INST
		DebugOp(COMMIT_STAGE);
		Debugln;
#endif /* EBUG_INST */
		/*
		 * The instruction binds the variable in "X0" with the
		 * commit level in the persistent stack.
		 */
                if (! unify_constant(Integer(CommitStage(top_of_ptrail)),
				     XV(0)))
                        Backtrack();
	otherwise:
		fatal("Internal error: unknown op-code %d\n", pc[-1]);
	}
	}
}
