/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

#include <stdio.h>

#include "bind.h"
#include "cells.h"
#include "data_area.h"
#include "debug.h"
#include "defs.h"
#include "dereference.h"
#include "errors.h"
#include "examine_term.h"
#include "name_table.h"
#include "string_table.h"
#include "substitution.h"
#include "unify.h"
#include "x_registers.h"

/*
 * True if Term is an atom, fails otherwise.
 */

int equal1 (VALUE *term1, VALUE *term2);
int equal_constant (cell constant, VALUE *t2);
int equal_apply (VALUE *t1, VALUE *t2);
int equal_pair (VALUE *t1, VALUE *t2);
int equal_quantifier (VALUE *t1, VALUE *t2);
int equal_var (VALUE *t1, VALUE *t2);
int equal_obvar (VALUE *t1, VALUE *t2);

global	boolean
atom(void)
{
	return(IsAtom(Xdref(0)));
}

/*
 * True if Term is an integer, fails otherwise.
 */ 
global	boolean
integer(void)
{
	return(IsInteger(Xdref(0)));
}

/*
 * True if Term is a variable, fails otherwise.
 */
global	boolean
var(void)
{
	return(IsReference(Xdref(0)));
}

/*
 * True if Term is a object variable, fails otherwise.
 */
global	boolean
is_object_var(void)
{
	return(IsObjectReference(Xdref(0)));
}

/*
 * True if Term is a local object variable, fails otherwise.
 */
global	boolean
esc_is_local_object_var(void)
{
	return(IsObjectReference(Xdref(0)) && IsLocalObjectVariable(X(0)));
}
/*
 * List is the list of ASCII codes representing the name Name.
 */
global	boolean
atom_chars(void)
{
	char	*string;
	VALUE	val;

	if (IsAtom(Xdref(0)))
	{
		val.term = put_string_on_heap(String(RestOfConstant(X(0))));
		val.sub = EMPTY_SUB;
		return(unify(&val, XV(1)));
	}
	else if (IsReference(X(0)) &&
		 (string = get_string_from_heap(X(1))) != NULL)
	{
		val.term = Atom(lookup_name_table_offset(string, ATOM_W));
		if (RestOfConstant(val.term) == 
		    (top_of_string_table - string_table))
                {
			top_of_string_table +=
			        strlen(top_of_string_table) + 1;
                }
		val.sub = EMPTY_SUB;
		return(unify(XV(0), &val));
	}
	else
        {
		return(FALSE);
        }
}
/*
 * e.g. Atoms = [a, b, c], Atom = 'a/b/c'
 */
global	boolean
esc_atoms_to_atom(void)
{
	VALUE	val;

	if (IsList(Xdref(0)))
	{
		val.term = get_atom_from_heap(X(0));
		val.sub = EMPTY_SUB;
		return(unify(&val, XV(1)));
	}
	else
        {
		return(FALSE);
	}

}

/*
 * note: no delaying - only failure
 */

global boolean
number_chars(void)
{
	VALUE val;

	if( IsList(Xdref(1)) )
	{
		val.term = Integer(atoi(get_string_from_heap(X(1))));
		val.sub = EMPTY_SUB;
		return(unify(&val, XV(0)));
	}
	else if( IsInteger(Xdref(0)) )
	{
		char s[20];
		sprintf(s, "%d", IntOf(X(0)) );
		val.term = put_string_on_heap(s);
		val.sub = EMPTY_SUB;
		return(unify(&val, XV(1)));
	}
	else 
        {
                return(FALSE);
        }
}
/*
 * [a, b, c] -> 'a/b/c' 
 */

global	cell
get_atom_from_heap(cell value)
{
	char	*s;
	VALUE	val;
	VALUE	val1;
	VALUE	val_h;
	offset	tmp;

	s = top_of_string_table;

	while  ((DereferenceTerm(val, value), TRUE) &&
		IsList(val.term) &&
		IsConstant(DerefTerm(val1, Head(val_h, val.term))))
	{
		if (IsAtom(val1.term))
                {
			sprintf(s, "%s/", String(val1.term));
                }
		else
                {
			sprintf(s, "%d/", IntOf(val1.term));
                }
		s += strlen(s);
		value = Tail(val.term);
	}

	if (s > (string_table + Kwords_to_chars(string_table_size)))
        {
		fatal("Out of space in string table %d K", string_table_size);
        }

	if (IsNIL(val.term))
	{
		*--s = '\0';
		if ((tmp = add_name_string_offset(top_of_string_table, ATOM_W))
		         == (top_of_string_table - string_table))
                {
			top_of_string_table +=
				strlen(top_of_string_table) + 1;
                }
		return(Atom(tmp));
	}
	else
        {
		return(NULL);
        }
}

/*
 *   Put the string s onto the heap in WAM format
 *   e.g. "abc" is @(@(., 97), @(@(., 98), @(@(., 99), [])))
 *
 * string_list([], []).
 * string_list([H|T], @(@(., H), Rest)) :-
 *         string_list(T, Rest).
 */

global	cell
put_string_on_heap(char *s)
{
        VALUE   val_h;
	cell	head_of_list;
reg	cell	*end_of_list;
reg	cell	value;


	end_of_list = &head_of_list;
	while (*s)
	{
		*end_of_list = value = Apply();
		Functor(value) = Apply();
		Functor(Functor(value)) = Atom(CONS);
		Head(val_h, value) = Integer(*s++);
		end_of_list = &Tail(value);
	}
	*end_of_list = Atom(NIL);
	return(head_of_list);
}

/*
 *   get the string s from the heap
 *   e.g. "abc" is @(@(., 97), @(@(., 98), @(@(., 99), [])))
 *
 * string_list([], []).
 * string_list([H|T], @(@(., H), Rest)) :-
 *         string_list(T, Rest).
 */

global	char *
get_string_from_heap(cell value)
{
	char	*s;
	VALUE	val;
	VALUE	val1;
	VALUE	val_h;

	s = top_of_string_table;

	while  ((DereferenceTerm(val, value), TRUE) &&
		IsList(val.term) &&
		IsInteger(DerefTerm(val1, Head(val_h, val.term))))
	{
		*s++ = (char) IntOf(val1.term);
		value = Tail(val.term);
	}
	if (s > (string_table + Kwords_to_chars(string_table_size)))
        {
		fatal("Out of space in string table %d K", string_table_size);
        }
	if (IsNIL(val.term))
	{
		*s = '\0';
		return(top_of_string_table);
	}
	else
        {
		return(NULL);
        }

}

/*
 *   Term =.. Termlist
 *       f(t1, ..., tn) =.. [f, t1, ..., tn].
 *       X =.. [f, t1, ..., tn], (X = f(t1, ..., tn)).
 */

global	boolean
univ(void)
{
local	cell	apply_to_list_on_heap(cell term, cell list);
local	boolean list_to_apply_on_heap(VALUE *list, cell fn, cell *term);
	VALUE	val;
	VALUE	fn;
	VALUE	tail;
	VALUE	val_h;
	cell	term;

	if ((IsApply(Xdref(0)) &&
	     !IsQuantifier(DerefTerm(val, Reference(&Argument(X(0)))))) ||
	    IsConstant(X(0)))
	{
		val.term = apply_to_list_on_heap(X(0), Atom(NIL));
		val.sub = XS(0);
		return(unify(&val, XV(1)));
	}
	else if (IsReference(X(0)) &&
		 IsList(Xdref(1)) &&
		 (tail.sub = EMPTY_SUB, tail.term = Tail(X(1)),
		  list_to_apply_on_heap(&tail,
				       DerefTerm(fn, Reference(&Head(val_h,
								     X(1)))),
				       &term)))
	{
	        val.term = term;
		val.sub = XS(1);
		return(unify(XV(0), &val));
	}
	else
        {
		return(FALSE);
        }
}
/*
 *  @(@(... @(f, t1), ..., tn-1), tn)
 *  =..
 *  @(@(., f), @(@(., t1), @(... @(@(., tn), []))))
 *
 *  apply_to_list(F(Arg), Args, List) :-
 *	apply_to_list(F, [Arg|Args], List).
 *   apply_to_list(F, Args, [F|Args]).
 *
 *   ?- apply_to_list(f(t1, ..., tn), [], X).
 *       X = [f, t1, ..., tn] ?
 */

local	cell
apply_to_list_on_heap(cell term, cell list)
{
	cell	apply(cell fn, cell argument);
	VALUE	val;
	cell	argu;

	DereferenceTerm(val, term);

	if (IsApply(val.term))
	{
		argu = Reference(&Argument(val.term));
		variable_dereference(&argu);
		return(apply_to_list_on_heap(Functor(val.term),
			Cons(argu, list)));
	}
	else
        {
		return(Cons(val.term, list));
        }
}

/*
 *  @(@(., f), @(@(., t1), @(... @(@(., tn), []))))
 *  --->  
 *  @(@(... @(f, t1), ..., tn-1), tn)
 */

local	boolean
list_to_apply_on_heap(VALUE *list, cell fn, cell *term)
{
	cell	apply(cell fn, cell argument);
	VALUE	argu;
	VALUE	val_h;

	dereference(list);
	if (IsList(list->term))
	{
		argu.sub = list->sub;
		argu.term = Reference(&Head(val_h, list->term));
		variable_dereference(&(argu.term));
		list->term = Tail(list->term);
		return(list_to_apply_on_heap(list,
			apply(fn, make_substitution(&argu)), term));
	}
	else
	{
		*term = fn;
		return(IsNIL(list->term));
	}
}
/*
 *  an example of use to create f(a, b) on the heap
 *  call apply(apply(f, a), b) 
 */

global	cell
apply(cell fn, cell argument)
{
	cell	value;

	value = Apply();
	Functor(value) = fn;
	Argument(value) = argument;
	return(value);
}

/*
 * Given functor and arity, arguments are taken from Xs and the goal is built
 * on the heap.
 */

global	cell
build_goal(offset fn, unsigned int arity)
{
	int	i;
	cell	value;

	value = Atom(fn);
	for (i=0; i < arity; i++)
        {
		value = apply(value, make_substitution(XV(i)));
        }
	return(value);
}

/*
 * Term has Functor and Arity.
 */

global	boolean
functor(void)
{
	VALUE	val;
	VALUE	fn;
	VALUE	arity;
	VALUE	term;
	cell	*f;
	int	n;
	natural	i;

	if (IsApply(Xdref(0)) &&
	    !IsQuantifier(DerefTerm(val, Reference(&Argument(X(0))))))
	{
		/* What is going to happen when we have variables in
		   functor positions? I will tell you. I will track down a
		   bug to here. See you then, bye */
		fn.term = fetch_functor(X(0));
		fn.sub = EMPTY_SUB;
		arity.term = Integer(fetch_arity(X(0)));
		arity.sub = EMPTY_SUB;
		return(unify(&fn, XV(1)) && unify(&arity, XV(2)));
	}
	else if (IsAtom(X(0)) || IsInteger(X(0)))
	{
		term.term = Integer(0);
		term.sub = EMPTY_SUB;
		return(unify(XV(0), XV(1)) && unify(&term, XV(2)));
	}
	else if (IsAtom(Xdref(1)) && IsInteger(Xdref(2)))
	{
		n = IntOf(X(2));
		if (n > 0)
		{
			term.term = Apply();
			term.sub = EMPTY_SUB;
			f = &(term.term);
			for (i = 0; i < n; i++)
			{
				Functor(*f) = Apply();
				f = &(Functor(*f));
			}
			*f = X(1);
			return(unify(XV(0), &term));
		}
		else if (n < 0)
                {
			return(FALSE);
                }
		else
                {
			return(unify(XV(0), XV(1)));
                }
	}
	else if (IsInteger(X(1)) && IsInteger(Xdref(2)))
	{
		n = IntOf(X(2));
		if (n == 0)
		{
			return(unify(XV(0), XV(1)));
		}
	}
	return(FALSE);
}

/*
 *  The goal is a term of the form f(t1, ..., tn) which has to be broken down
 *  into x registers allocating the terms ti to the x register X(i-1)
 *  The term is stored in the form @(@( ... @(f, t1) ..., tn-1), tn).
 *
 *  Returns the number of arguments in the term.
 *
 *
 *  goal(F(Arg), Args, Args2) :-
 *     !,
 *      goal(F, [Arg|Args], Args2).
 *   goal(_Term, Args, Args).
 *
 *  with the query
 *
 *    ?- goal(Term, [], Args).
 *
 *   Args = list of arguments [t1, ..., tn]
 */

global	natural
setup_x_registers(VALUE *goal)
{
	natural	n;
	VALUE	new_goal;
	VALUE	val;

	DereferenceTerm(val, goal->term);
	if (IsApply(val.term))
	{
		new_goal.term = Functor(val.term);
		new_goal.sub = val.sub;
		n = setup_x_registers(&new_goal);
		X(n) = Reference(&Argument(val.term));
		XS(n) = val.sub;
		return(n+1);
	}
	return(0);
}

/*
 * The Nth argument of Term is Arg.
 */

global	boolean
arg(void)
{
local	boolean get_arg(unsigned int arity, unsigned int n, VALUE *term, VALUE *argument);
	natural	n;
	VALUE	val;

	if (IsInteger(Xdref(0)))
	{
		n = fetch_arity(X(1));
		val.sub = XS(1);
		val.term = X(1);
		return((natural)IntOf(X(0)) <= n &&
		       get_arg(n, (natural)IntOf(X(0)), &val, XV(2)));
	}
	else
        {
		return(FALSE);
        }
}

/*
 * Get the nth argument of term.
 */

local	boolean
get_arg(unsigned int arity, unsigned int n, VALUE *term, VALUE *argument)
{
	VALUE	tmp;
	VALUE	val1;

	dereference(term);
	if (IsApply(term->term) &&
	    !IsQuantifier(DerefTerm(val1, Reference(&Argument(term->term)))))
        { 
		if ((arity - n) == 0)
		{
			tmp.term = Reference(&Argument(term->term));
			tmp.sub = term->sub;
			return(unify(&tmp, argument));
		}
		else
		{
			tmp.term = Reference(&Functor(term->term));
			tmp.sub = term->sub;
			return(get_arg(arity - 1, n, &tmp, argument));
		}
        }
	else
        {
		return(FALSE);
        }
}
/*
 * $equal(X, Y) :-
 *         True if the terms X and Y are equal.
 */

global	boolean
esc_equal(void)
{
	return(equal(XV(0), XV(1)));
}

/*
 *  Dereference the term - term1.
 *  Split equal probem according to the data object type of term->term1.
 */

global	boolean	
equal(VALUE *term1, VALUE *term2)
{

        if (try_delay_problems() == FALSE)
        {
                return(FALSE);
        }
        else
        {
                return(equal1(term1, term2));
        }

}

/*
 * Dereference the term - term1.
 * Split equal probem according to the data object type of term->term1.
 */

global	boolean	
equal1(VALUE *term1, VALUE *term2)
{
	VALUE	     t1;
	VALUE	     t2;

	t1.sub = term1->sub;
	t1.term = term1->term;
	t2.sub = term2->sub;
	t2.term = term2->term;

	dereference(&t1);
	switch (Tag(t1.term))
	{
	when CONSTANT:
		return(equal_constant(t1.term, &t2));
	when APPLY:
		return(equal_apply(&t1, &t2));
	when PAIR:
		return(equal_pair(&t1, &t2));
	when QUANTIFIER:
		return(equal_quantifier(&t1, &t2));
	when REFERENCE:
		dereference(&t2);
		return(t1.term == t2.term &&
		       equal_var(&t1, &t2));
	when OBJECT_REFERENCE:
		dereference(&t2);
		return(t1.term == t2.term &&
		       equal_obvar(&t1, &t2));
	}
	return(FALSE);
}

/*
 * Dereference term - t2.
 *
 * If constant and just term of t2 are the same then
 *         return TRUE
 * else
 *         return FALSE
 */

local	boolean
equal_constant(cell constant, VALUE *t2)
{
	dereference(t2);
	return(constant == t2->term);
}

/*
 * t1 is a data object of an APPLY type.
 *
 *  Do dereference for t2.
 *  If t2 is of APPLY type then
 *       .knowing that S * f(a1, a2, ...,an) = f(S * a1, S * a2, ... , S * an).
 *        the problem is split to two: 
 *            equal1(S1 * f1(a1,...,a(n-1)), S2 * f2(b1,...,b(n-1)))
 *            equal1(f1(S1 * an, S2 * bn)
 *  else 
 *            return FALSE
 */

local	boolean
equal_apply(VALUE *t1, VALUE *t2)
{
	VALUE	     t1fn;
	VALUE	     t2fn;
	VALUE	     t1arg;
	VALUE	     t2arg;

	dereference(t2);
	if  (IsApply(t2->term))
	{
                t1fn.sub = t1arg.sub = t1->sub;
                t2fn.sub = t2arg.sub = t2->sub;
                t1fn.term = Reference(&Functor(t1->term));
                t1arg.term = Reference(&Argument(t1->term));
                t2fn.term = Reference(&Functor(t2->term));
                t2arg.term = Reference(&Argument(t2->term));
                return(equal1(&t1fn, &t2fn) && equal1(&t1arg, &t2arg));
	}
	return(FALSE);
}

/*
 * equal_pair(t1, t2)
 */

local	boolean
equal_pair(VALUE *t1, VALUE *t2)
{
        VALUE        t1left;
        VALUE        t2left;
        VALUE        t1right;
        VALUE        t2right;

	dereference(t2);
	if (IsPair(t2->term))
	{
                t1left.sub = t1right.sub = t1->sub;
                t2left.sub = t2right.sub = t2->sub;
                t1left.term = Reference(&Left(t1->term));
                t1right.term = Reference(&Right(t1->term));
                t2left.term = Reference(&Left(t2->term));
                t2right.term = Reference(&Right(t2->term));
                return(equal1(&t1left, &t2left) && equal1(&t1right, &t2right));
	}
	return(FALSE);
}

/*
 * equal_quantifier(t1, t2)
 */

local   boolean
equal_quantifier(VALUE *t1, VALUE *t2)
{
        cell    objvar;
        VALUE   t1body;
        VALUE   t2body;
 
        dereference(t2);
	if (IsQuantifier(t2->term))
	{
                objvar = NewLocalObjectVariable();
                t1body.sub = RenameSub(BoundVar(t1->term), objvar, t1->sub);
                t1body.term = Body(t1->term);
                t2body.sub = RenameSub(BoundVar(t2->term), objvar, t2->sub);
                t2body.term = Body(t2->term);
                return(equal1(&t1body, &t2body));
	}
	return(FALSE);
}

/*
 * Before this procedure is called, full dereference of both 
 * terms t1 and t2 were done, so we have substitutions and 
 * terms separated.
 *
 * It is called only if t1->term and t2->term are the same   
 * meta variables. 
 *
 * Descriptions:
 *  If substitutions are the same then
 *      .return TRUE
 *  else
 *      .set choice point
 *           procedure pass_sub is described in this file
 *      .if (pass_sub(t1, t2) && pass_sub(t2, t1)) then
 *           .set last choice point to the previous value 
 *           .return TRUE
 *       else 
 *           .set last choice point to the previous value
 *           .return FALSE
 */

local	boolean
equal_var(VALUE *t1, VALUE *t2)
{
        cell        newvar;
        VALUE       t1new;
        VALUE       t2new;

local   void        create_incomplete_choice_point(void);
local   void        do_backtrack(void);
local   boolean     create_del_problem(cell var, cell objvar);
	   
        if (t1->sub == t2->sub)
        {
                return(TRUE);
	}
        else
	{
	       	create_incomplete_choice_point();
                newvar = NewObjectVariable();

	        if (create_del_problem(t1->term, newvar) != FALSE)  
	        {
	                t1new.sub = t1->sub;
	                t1new.term = newvar;
	                t2new.sub = t2->sub;
	                t2new.term = newvar;
			   
                        if (equal1(&t1new, &t2new))
                        {
		                do_backtrack();
                                last_choice_point =
                                        last_choice_point->last_choice_point;
                                return(TRUE);
                        }
                        else
		        {
		                do_backtrack();
                                last_choice_point =
                                        last_choice_point->last_choice_point;
                                return(FALSE);
                        }
                }
		else
                {
	                do_backtrack();
                        last_choice_point =
                                last_choice_point->last_choice_point;
                        return(TRUE);
                }
        }  
}

/*
 * 
 */

boolean 
create_del_problem(cell var, cell objvar)
{
        VALUE       val;  
        cell        *k;
        boolean     b;


        b = TRUE;
        for (k = (cell *)RestOfVariable(Value(var)); 
			   b && k != NULL; k = Next(k))
        {
                Solved(k);
	        /* take care of NOT_FREE problems only */
		if  (DelayedType(k) == NOT_FREE)
	        {
                        DereferenceTerm(val, DelayedVar(k));
                        val.term = objvar;

                        /* check whether new NOT_FREE
			       problem fails                 */
                        if (freeness(DelayedTerm(k), &val))
                        {
			        b = FALSE;
                        }
                }
        }
        return(b);
}

/*
 * Create incomplete choice point.
 */

void
create_incomplete_choice_point(void)
{
        CHOICE *cp;

	cp = last_choice_point;
	last_choice_point = (CHOICE *)top_of_stack;
	StackAlloc(ChoiceSize(0));
	last_choice_point->last_choice_point = cp;
	last_choice_point->top_of_trail = top_of_trail;
	last_choice_point->top_of_heap = top_of_heap;
	last_choice_point->top_delayed_stack = top_delayed_stack;
}

/*
 * Do incomplete backtrack.
 */

local	void
do_backtrack(void)
{

        top_of_heap = last_choice_point->top_of_heap;
        top_delayed_stack = last_choice_point->top_delayed_stack;
	while (top_of_trail > last_choice_point->top_of_trail)
	{ 
                top_of_trail--;
                if (top_of_trail->address != NULL)
                {
		        * (top_of_trail->address) = 
			        top_of_trail->previous_value;
                }
        }
}

/*
 * Before this procedure is called, full dereference of both 
 * terms t1 and t2 were done, so we have substitutions and 
 * terms separated.
 *
 * It is called only if t1->term and t2->term are the same   
 * object variables. 
 *
 * Description:
 *         If substitutions are the same then
 *                 return TRUE
 *         else
 *	   .if substitutions of the term t1 is EMPTY_SUB then
 *                  .return equal_obvar(t2, t1)
 *          else
 *	            .find the most right domain in t1->sub  
 *	                    which is not apart from t1->term
 *                  .set choice point 
 *
 *	            .bind t1->term and the most right domain
 *	            .replace t1->term in woken up NOT_FREE delayed 
 *      	             problems, by the most right domain
 *                  .if no NOT_FREE problem fails and retry_delay succeeds then
 * 	                     .if equal1(t1, t2) then 
 *		                     .backtrack 
 *                            else 
 *		                     .backtrack
 *	                             .return FALSE
 *                   else 
 *		             .backtrack
 *
 *	            .set the most right domain apart from t1->term 
 *	            .replace t1->term in woken up NOT_FREE delayed 
 *	             problems, by the most right domain
 *                  .if no NOT_FREE problem fails and retry_delay succeeds then
 *		             .if equal1(t1, t2) then 
 *		                      .backtrack 
 *                            else 
 *		                      .backtrack
 *		                      .return FALSE
 *                   else 
 *		             .backtrack
 *         .return TRUE
 */

local boolean
equal_obvar(VALUE *t1, VALUE *t2)
{
        VALUE   val;
	cell    z;
	cell    *k;
	boolean b;
local   cell    get_last_dom(cell sub, cell objvar);
local   void    do_backtrack(void);
local   void    create_incomplete_choice_point(void);

        if (t1->sub == t2->sub)
        {
	        return(TRUE);
        }
        else  
	{
	        if (t1->sub == EMPTY_SUB)
                {
	 	        return(equal_obvar(t2, t1));
                }
                else
	        {
		    /* set z to the most right domain in substitution t1->sub */
	                z = get_last_dom(t1->sub, t1->term);
		        object_dereference(&z);
		        /* set choice point */ 
	                create_incomplete_choice_point();
                        /* bind the most right domain - z and  
		           the object variable - t1->term     */
            
			b = TRUE;
                        for (k = (cell *)RestOfVariable(Value(t1->term));
       	                     b && k != NULL; k = Next(k)) 
                        {
                                Solved(k);
			        /* take care of NOT_FREE problems only */
               	                if (DelayedType(k) == NOT_FREE)
	       	                {
                                        val.sub = EMPTY_SUB;
                                        val.term = DelayedVar(k);  
                                        /* check whether new NOT_FREE
                                           problem fails              */
                                        if (freeness(DelayedTerm(k), &val))
                                        {
				                b = FALSE;
                                        }
                                }
  	                }
               	        BindObjectVariable(z, t1->term);
	                if (b && try_delay_problems() && retry_nfi_delay())
	                {
	                        if (equal1(t1, t2))
                                {
		                        do_backtrack();
                                }
                                else
		                {
		                        do_backtrack();
		                        last_choice_point =
		                           last_choice_point->last_choice_point;
                                        return(FALSE);
                                }
                        }
                        else
                        {
	                        do_backtrack();
                        }
                     
                        /* set the most right domain z apart 
		           from the object variable t1->term */   
                        SetDistinct(z, t1->term);

                        b = TRUE;
		   
		        /* passing a list of delayed problems associated 
		           with the object variable t1->term             */
                        for (k = (cell *)RestOfVariable(Value(t1->term));
       	                     b && k != NULL; k = Next(k)) 
                        {
                                Solved(k);
			        /* take care of NOT_FREE problems only */ 
		                if (DelayedType(k) == NOT_FREE)
	       	                {
                                        val.sub = EMPTY_SUB;
                	                val.term = DelayedVar(k);
                                        if (freeness(DelayedTerm(k), &val))
                                        {
			                        b = FALSE;
                                        }
		                }
	                }
		                 /* passing a list of delayed problems
		                    associated with z                 */
                        for (k = (cell *)RestOfVariable(Value(z));
        	             b && k != NULL; k = Next(k)) 
                        {
                                Solved(k);
			        /* take care of NOT_FREE problems only */ 
               	                if (DelayedType(k) == NOT_FREE)
	       	                {
                                        val.sub = EMPTY_SUB;
                	                val.term = DelayedVar(k);
			                /* check whether new NOT_FREE	
				           problem fails                 */
                                        if (freeness(DelayedTerm(k), &val))
                                        {
			                        b = FALSE;
                                        }
		                }
                        }
                        if (b && retry_nfi_delay())
                        {
	                        if (equal1(t1, t2))
                                {
		                        do_backtrack();
                                }
                                else
                                {
		                        do_backtrack();
			                last_choice_point =
		                           last_choice_point->last_choice_point;
		                        return(FALSE);
                                }
                        }
                        else
                        {
                                do_backtrack();
                        } 
	                last_choice_point = 
                                last_choice_point->last_choice_point;
                        return(TRUE);
                }  /* if one EMPTY_SUB */
	}  /* if substitutions are not equal */
}

/*
 * Return the most right domain from the substitution - sub
 * which is not apart from the object variable - objvar.
 */

local cell
get_last_dom(cell sub, cell objvar)
{
        int     i, j;

        for (i = 1, j = Size(sub); i <= j; i++)
        {
	        if (!distinct_from(objvar, Domain(sub, i)))
                {
		        return(Domain(sub, i));
                }
        }
        fatal("ERROR in get_last_dom   ");
}
 
/*
 * Boolean function which returns:
 *         TRUE  - if Term1 and Term2 are (object) variables and 
 *                 Term1 physically is less than Term2. Object vars 
 *                 are less than variables.
 *         FALSE - otherwise.
 */

global	boolean
esc_var_less_than(void)
{

	if (IsObjectReference(Xdref(0)))
        {
	        return(IsReference(Xdref(1)) ||  
		      (IsObjectReference(X(1)) && RestOf(X(0)) < RestOf(X(1))));
        }
	else
        {
	        return(IsReference(Xdref(1)) && IsReference(X(0)) && 
	               RestOf(X(0)) < RestOf(X(1)));
        }
}

/*
 *   Find the functor of an APPLY data object type.
 */

global  cell
fetch_functor(cell term)
{
        VALUE   val;

        DereferenceTerm(val, term);
        while (IsApply(val.term))
        { 
                DereferenceTerm(val, Reference(&Functor(val.term)));
        }
        return(val.term);
}

/*
 * Find the arity of an APPLY data object type. 
 */

global  natural
fetch_arity(cell term)
{
reg     natural i;
        VALUE   val;
 
        DereferenceTerm(val, term);
        i = 0;
        while (IsApply(val.term))
        {
                i++;
                DereferenceTerm(val, Reference(&Functor(val.term)));
        }
        return(i);
}

/*
 * Dereference value in X(0) register.
 * If data object type in X(0) register is variable or object variable then
 *         .set temperature (28th) bit to FREEZE.
 *         .return TRUE
 * else
 *         .return FALSE
 */

global	boolean
esc_freeze(void)
{

	if (IsReference(Xdref(0)) || IsObjectReference(X(0)))
	{
		set(Location(X(0)), RestOfTemp(Value(X(0)))|FREEZE);
		return(TRUE);
	}
	else
        {
		return(FALSE);
        }
}

/*
 * Dereference value in X(0) register.
 * If data object type in X(0) register is variable or object variable then
 *         .set temperature (28th) bit to THAW.
 *         .return TRUE
 * else
 *         .return FALSE
 */

global	boolean
esc_thaw(void)
{

	if (IsReference(Xdref(0)) || IsObjectReference(X(0)))
	{
		set(Location(X(0)), RestOfTemp(Value(X(0)))|THAW);
		return(TRUE);
	}
	else
        {
		return(FALSE);
        }
}

/*
 * Dereference value in X(0) register.
 * If data object type in X(0) register is an (object) 
 *      variable and temperature bit is set to FREEZE then 
 *	   .return TRUE
 * else
 *	   .return FALSE
 */

global	boolean
esc_frozen(void)
{

	if (IsReference(Xdref(0)) || IsObjectReference(X(0)))
        {
		return(Frozen(X(0)));
        }
	else
        {
		return(FALSE);
        }
}
