/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

#ifdef EBUG

#include <fcntl.h>
#include "cells.h"
#include "code_area.h"
#include "data_area.h"
#include "database.h"
#include "debug.h"
#include "defs.h"
#include "delayed_problems.h"
#include "dereference.h"
#include "examine_term.h"
#include "io.h"
#include "name_table.h"
#include "persistent.h"
#include "pred_table.h"
#include "primitives.h"
#include "spy.h"
#include "string_table.h"
#include "system.h"
#include "x_registers.h"
#include "unify.h"
#define ErrorVariableName(p)     {fprintf(stderr, "_%c%c%c%c",      \
                                      UpperCaseChar((p / 17576) % 26),  \
                                      UpperCaseChar((p / 676) % 26),    \
                                      UpperCaseChar((p / 26) % 26),     \
                                      UpperCaseChar(p % 26));           \
                                     fflush(stderr);                   \
				 }
#define ErrorObjectVariableName(p) {fprintf(stderr, "_x%c%c%c%c",      \
                                      UpperCaseChar((p / 17576) % 26),  \
                                      UpperCaseChar((p / 676) % 26),    \
                                      UpperCaseChar((p / 26) % 26),     \
                                      UpperCaseChar(p % 26));           \
                                     fflush(stderr);                   \
                                   }


/*----------------------------------------------------------------------------
----------------------------------------------------------------------------*/

extern int fprintf (FILE *, const char *, ...);
extern int fflush (FILE *);

global void
spy(char *type, char *predicate_name, unsigned int arity)
{
local	boolean	spied(char *fn, unsigned int arity);
local	void	spy_on_port(char *type, char *predicate_name, unsigned int arity);

	if (spied("$$all", 0)){
		spy_on_port(type, predicate_name, arity);
	}
	else if (spied("$all", 0)) {
		/* only spy on preds not starting with '$' */
		if((*predicate_name) != '$')
		    spy_on_port(type, predicate_name, arity);
	}
	else if (spied(predicate_name, arity)) {
		spy_on_port(type, predicate_name, arity);
	}
}

/*----------------------------------------------------------------------------
    True, iff the predicate functor/arity is a member of the spied predicates
    table.
----------------------------------------------------------------------------*/
local	boolean
spied(char *fn, unsigned int arity)
{
	natural	i;

	for (i = 0; i < spied_predicate_table_size; i++)
	{
		if ((!strcmp(fn, spied_predicates[i].functor)) &&
		    (arity == spied_predicates[i].arity))
			return(TRUE);
	}
	return(FALSE);
}

/*----------------------------------------------------------------------------
----------------------------------------------------------------------------*/
local void
spy_on_port(char *type, char *predicate_name, unsigned int arity)
{
	natural	i;

	fprintf(stderr, "\n");
	fprintf(stderr, "%s : %s(", type, predicate_name);
	for (i = 0; i < arity; i++)
	{
		write_termv(XV(i));
		if (i<(arity-1))
			fprintf(stderr, ", ");
	}
	fprintf(stderr, ")\n");
}

/*----------------------------------------------------------------------------
----------------------------------------------------------------------------*/
global void
writeln_termv(VALUE *term)
{
	*top_of_heap++ = term->sub;
	*top_of_heap++ = term->term;
	*top_of_heap = (cell)(SUBSTITUTION_OPERATOR|(int)(top_of_heap - 2));
	writeln_term(top_of_heap);
	top_of_heap -= 2;
}

/*----------------------------------------------------------------------------
----------------------------------------------------------------------------*/
global void
write_termv(VALUE *term)
{
	*top_of_heap++ = term->sub;
	*top_of_heap++ = term->term;
	*top_of_heap = (cell)(SUBSTITUTION_OPERATOR|(int)(top_of_heap - 2));
	write_term(top_of_heap);
	top_of_heap -= 2;
}

/*----------------------------------------------------------------------------
----------------------------------------------------------------------------*/
global void
writeln_term(cell *term)
{
	write_term(term);
	fprintf(stderr, "\n");
}

/*----------------------------------------------------------------------------
----------------------------------------------------------------------------*/
global void
write_term(cell *term)
{
local	void	write_tail(cell *term);
local	void	write_functor(cell *term);
local	void	write_sub(cell sub);
local	boolean	safe_string(char *s);
	VALUE	val;
	VALUE	val_h;
	FILE	*fp;


	DereferenceTerm(val, *term);
	if (val.sub != EMPTY_SUB)
	{
		write_sub(val.sub);
		fprintf(stderr, " * ");
	}
	switch (Tag(val.term)) 
	{ 
	when CONSTANT:
		if (IsAtom(val.term))
		{
			if (safe_string(String(val.term)))
				fprintf(stderr, "%s", String(val.term));
			else
				fprintf(stderr, "\'%s\'", String(val.term));
		}
		else
		{
			fprintf(stderr, "%d", IntOf(val.term));
		}
	when CONS:
		fprintf(stderr, ".");
	when APPLY:
		if (IsList(val.term))
		{
			fprintf(stderr, "[");
			write_term(&Head(val_h, val.term));
			write_tail(&Tail(val.term));
			fprintf(stderr, "]");
		}
		else
		{
			write_functor(&Functor(val.term)),
			write_term(&Argument(val.term)),
			fprintf(stderr, ")");
		}
	when PAIR:
		fprintf(stderr, "PAIR");
	when QUANTIFIER:
		write_term(&BoundVar(val.term));
		fprintf(stderr, " ^ ");
		write_term(&Body(val.term));
	when VARIABLE:		/* OBJECT_VARIABLE case as well */
		if (InPersistentStack(Reference(term)))
			fprintf(stderr, "#%s",
				String(get_persistent_name2(Reference(term))));
		else
		{
			ErrorVariableName((int)(term - heap));
			if (PtrToName(Reference(term)) != NULL)
			{
			    fprintf(stderr, " [aka ");
			    fprintf(stderr, "%s",
			            String3(Reference(term)));
			    fprintf(stderr, "]");
			}

		}
	when REFERENCE:
		if (InPersistentStack(val.term))
			fprintf(stderr, "#%s",
				String(get_persistent_name(val.term)));
		else
		{
			ErrorVariableName((int)(Location(val.term) - heap));
			if (PtrToName(val.term) != NULL)
			{
			    fprintf(stderr, " [aka ");
			    fprintf(stderr, "%s", String3(val.term));
			    fprintf(stderr, "]");
                        }

		}
	when OBJECT_REFERENCE:
		if (IsLocalObjectVariable(val.term))
		{
			fprintf(stderr, "@");
			ErrorObjectVariableName((int)(Location(val.term) -
						heap));
		}
		else if (InPersistentStack(val.term))
			fprintf(stderr, "#%s",
				String(get_persistent_name(val.term)));
		else
		{
			ErrorObjectVariableName((int)(Location(val.term) -
						heap));
			if (PtrToName(val.term) != NULL)
			{
			    fprintf(stderr, " [aka ");
			    fprintf(stderr, "%s", String3(val.term));
			    fprintf(stderr, "]");
                        }

		}
	otherwise:
		fprintf(stderr, "Internal Error: OTHER");
	} 
}

/*----------------------------------------------------------------------------
----------------------------------------------------------------------------*/
global void
writeln_termv_deb(VALUE *term)
{
	*top_of_heap++ = term->sub;
	*top_of_heap++ = term->term;
	*top_of_heap = (cell)(SUBSTITUTION_OPERATOR|(int)(top_of_heap - 2));
	writeln_term_deb(top_of_heap);
	top_of_heap -= 2;
}

/*----------------------------------------------------------------------------
----------------------------------------------------------------------------*/
global void
write_termv_deb(VALUE *term)
{
	*top_of_heap++ = term->sub;
	*top_of_heap++ = term->term;
	*top_of_heap = (cell)(SUBSTITUTION_OPERATOR|(int)(top_of_heap - 2));
	write_term_deb(top_of_heap);
	top_of_heap -= 2;
}

/*----------------------------------------------------------------------------
----------------------------------------------------------------------------*/
global void
writeln_term_deb(cell *term)
{
	write_term_deb(term);
	fprintf(stderr, "\n");
}
/*----------------------------------------------------------------------------
----------------------------------------------------------------------------*/
global void
write_term_deb(cell *term)
{
local	void	write_tail(cell *term);
local	void	write_functor(cell *term);
local	void	write_sub(cell sub);
local	boolean	safe_string(char *s);
	VALUE	val;
	VALUE	val_h;
	FILE	*fp;



	variable_dereference(term);
	switch (Tag(val.term)) 
	{ 
	when CONSTANT:
		if (IsAtom(val.term))
		{
			if (safe_string(String(val.term)))
				fprintf(stderr, "%s", String(val.term));
			else
				fprintf(stderr, "\'%s\'", String(val.term));
		}
		else
		{
			fprintf(stderr, "%d", IntOf(val.term));
		}
	when CONS:
		fprintf(stderr, ".");
	when APPLY:
		if (IsList(val.term))
		{
			fprintf(stderr, "[");
			write_term_deb(&Head(val_h, val.term));
			write_tail(&Tail(val.term));
			fprintf(stderr, "]");
		}
		else
		{
			write_functor(&Functor(val.term)),
			write_term_deb(&Argument(val.term)),
			fprintf(stderr, ")");
		}
	when PAIR:
		fprintf(stderr, "PAIR");
	when QUANTIFIER:
		write_term_deb(&BoundVar(val.term));
		fprintf(stderr, " ^ ");
		write_term_deb(&Body(val.term));
	when VARIABLE:		/* OBJECT_VARIABLE case as well */
		if (InPersistentStack(Reference(term)))
			fprintf(stderr, "#%s",
				String(get_persistent_name2(Reference(term))));
		else
		{
			ErrorVariableName((int)(term - heap));
			if (PtrToName(Reference(&(val.term))) != NULL)
			{
			    fprintf(stderr, " [aka ");
			    fprintf(stderr, "%s",
			            String3(Reference(&(val.term))));
			    fprintf(stderr, "]");
			}

		}
	when REFERENCE:
		if (InPersistentStack(val.term))
			fprintf(stderr, "#%s",
				String(get_persistent_name(val.term)));
		else
		{
			ErrorVariableName((int)(Location(val.term) - heap));
			if (PtrToName(val.term) != NULL)
			{
			    fprintf(stderr, " [aka ");
			    fprintf(stderr, "%s", String3(val.term));
			    fprintf(stderr, "]");
                        }

		}
	when SUBSTITUTION_OPERATOR:
		write_sub(Substitution(*term));
		fprintf(stderr, " * ");
		write_term_deb(&Term(*term));   
	when OBJECT_REFERENCE:
		if (IsLocalObjectVariable(val.term))
		{
			fprintf(stderr, "@");
			ErrorObjectVariableName((int)(Location(val.term) -
						heap));
		}
		else if (InPersistentStack(val.term))
			fprintf(stderr, "#%s",
				String(get_persistent_name(val.term)));
		else
		{
			ErrorObjectVariableName((int)(Location(val.term) -
						heap));
			if (PtrToName(val.term) != NULL)
			{
			    fprintf(stderr, " [aka ");
			    fprintf(stderr, "%s", String3(val.term));
			    fprintf(stderr, "]");
                        }

		}
	otherwise:
		fprintf(stderr, "Internal Error: OTHER");
	} 
}

/*----------------------------------------------------------------------------
----------------------------------------------------------------------------*/
global	void
write_sub(cell sub)
{
        natural j;

	if (sub == EMPTY_SUB)
	{
		fprintf(stderr, " ES ");
		return;
	}
        if (NextSub(sub) != EMPTY_SUB)
	{
		write_sub(NextSub(sub));
		fprintf(stderr, " * ");
	}
	fprintf(stderr, "[");
	for (j = Size(sub); j > 1; j--)
	{
		write_term(&Range(sub, j));
		fprintf(stderr, "/");
		write_term(&Domain(sub, j));
		fprintf(stderr, ", ");
	}
	write_term(&Range(sub, j));
	fprintf(stderr, "/");
	write_term(&Domain(sub, j));
	fprintf(stderr, "]");
}

/*----------------------------------------------------------------------------
----------------------------------------------------------------------------*/
local	void
write_functor(cell *term)
{
	VALUE	val;

	DereferenceTerm(val, *term);

	if (IsApply(val.term))
	{
		write_functor(&Functor(val.term));
		write_term(&Argument(val.term));
		fprintf(stderr, ", ");
	}
	else
	{
		write_term(term);
		fprintf(stderr, "(");
	}

}

/*----------------------------------------------------------------------------
----------------------------------------------------------------------------*/
local	void
write_tail(cell *term)
{
	VALUE	val;
	VALUE	val_h;
	cell    *h;

	DereferenceTerm(val, *term);

	if (val.sub == EMPTY_SUB && IsList(val.term))
	{
		fprintf(stderr, ", ");
		write_term(&Head(val_h, val.term));
		write_tail(&Tail(val.term));
	}
	else if (!IsNIL(val.term))
	{
		fprintf(stderr, "|");
		write_term(term);
	}
}

global	void
write_delay(cell variable)
{
	cell	*d;
	cell	var;

	VariableDereference(var, variable);
	if (IsReference(var) || IsObjectReference(var))
	{
		for (d = (cell *)RestOfVariable(Value(var));
		     d != NULL;
		     d = Next(d))
		{
			write_term(&DelayedTerm(d));
			switch (DelayedType(d))
			{
			when UNIFY:
				fprintf(stderr, " = ");
			when NOT_FREE:
				fprintf(stderr, " not_free_in ");
			}
			writeln_term(&DelayedVar(d));
		}
	}
}

global	void
write_delayed_stack(void)
{
	delayed	*d;

	for (d = delayed_stack; d < top_delayed_stack; d++)
	{

		if ((d->tag & SOLVEDMASK) == UNSOLVED)
		{
			switch (d->tag & TYPEMASK)
			{
			when UNIFY:
				write_term(&(d->term));
				fprintf(stderr, " = ");
				writeln_term(&(d->var));
			when NOT_FREE:
				write_term(&(d->term));
				fprintf(stderr, " not_free_in ");
				writeln_term(&(d->var));
			}
		}
	 }
}

global	void
write_distinctness_info(cell objectvar)
{
	cell	*d;
	cell	objvar;

	ObjectDereference(objvar, objectvar);
	if (! IsObjectReference(objvar))
		return;
	else
	{
		write_term(&objvar);
		fprintf(stderr, " distinct_from ");
		for (d = Location(Distinction(objvar));
		     d != NULL;
		     d = NextDistinction(d))
		{
			write_term(&(DistinctObjectVar(d)));
			fprintf(stderr, " ");
		}
		fprintf(stderr, "\n");
	}
}

/*----------------------------------------------------------------------------
----------------------------------------------------------------------------*/
local	boolean
safe_string(char *s)
{
local	boolean	safe_char(char c);

	while (*s)
		if (!safe_char(*s++))
			return(FALSE);
	
	return(TRUE);
}

/*----------------------------------------------------------------------------
    True, if the character c is an alphanumeric char or an underscore or
    dollar sign.
----------------------------------------------------------------------------*/
local boolean
safe_char(char c)
{
local	boolean	member(char c, char *s);

	return(member(c, "$abcdefghijklmnopqrstuvwxyz_0123456789"));
}

/*----------------------------------------------------------------------------
    True, iff the character c is a member of the string s.
----------------------------------------------------------------------------*/
local	boolean
member(char c, char *s)
{
	while (*s)
		if (c == *s++)
			return(TRUE);

	return(FALSE);
}

/*----------------------------------------------------------------------------
    Dump the positions of all the data structures used by the QuAM (abstract
    machine) in memory.
----------------------------------------------------------------------------*/
global	void
dump_positions(void)
{
	fprintf(stderr, "\nBlock\t\tStart\tEnd\tTop\tSize\n\n");
	fprintf(stderr, "x\t\t%x\t%x\t\t%d\n", x, (((char *)x) +
		num_x_registers * sizeof(VALUE)), BlockSize(x));
	fprintf(stderr, "code_area\t%x\t%x\t%x\t%d\n", code_area,
		(code_area + Kwords_to_chars(code_area_size)),
		top_of_code_area,
		BlockSize(code_area));
	fprintf(stderr, "string_table\t%x\t%x\t%x\t%d\n", string_table,
		(string_table + Kwords_to_chars(string_table_size)),
		top_of_string_table,
		BlockSize(string_table));
	fprintf(stderr, "name_table\t%x\t%x\t\t%d\n", name_table,
		(((char *)name_table) + name_table_size * sizeof(offset)),
		BlockSize(name_table));
	fprintf(stderr, "predicate_table\t%x\t%x\t\t%d\n", predicate_table,
		(((char *)predicate_table) + predicate_table_size *
		sizeof(offset)),
		BlockSize(predicate_table));
	fprintf(stderr, "delayed_stack\t%x\t%x\t%x\t%d\n", delayed_stack, 
		end_delayed_stack, top_delayed_stack,
		BlockSize(delayed_stack));
	fprintf(stderr, "heap\t\t%x\t%x\t%x\n", heap, (((char *)heap) +
		Kwords_to_chars(heap_size)), top_of_heap);
	fprintf(stderr, "stack\t\t%x\t%x\t%x\n", stack, end_of_stack,
		top_of_stack);
	fprintf(stderr, "trail\t\t%x\t%x\t%x\t%d\n", trail, end_of_trail,
		top_of_trail, BlockSize(trail));
	fprintf(stderr, "database_table\t%x\t%x\t\t%d\n\n", database_table, 
		(((char *)database_table) + database_table_size *
		sizeof(DYNAMIC *)), BlockSize(database_table));
	fprintf(stderr, "pers. vars.\t%x\t%x\t%x\t%d\n", pvar, 
		(((char *)pvar) + Kwords_to_chars(pvar_size)), top_of_pvar,
		BlockSize(pvar));
	fprintf(stderr, "pers. stack\t%x\t%x\t%x\n", pstack, end_of_pstack,
		top_of_pstack);
	fprintf(stderr, "pers. trail\t%x\t%x\t%x\t%d\n\n", ptrail, 
		end_of_ptrail, top_of_ptrail, BlockSize(ptrail));
}


#endif /* EBUG */
