/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

#include <memory.h>

#include "bind.h"
#include "cells.h"
#include "data_area.h"
#include "database.h"
#include "defs.h"
#include "dereference.h"
#include "errors.h"
#include "examine_term.h"
#include "main.h"
#include "name_table.h"
#include "pred_table.h"
#include "string_table.h"
#include "substitution.h"
#include "unify.h"
#include "x_registers.h"

global	DYNAMIC	**database_table;
global	natural	database_table_size = DEFAULT_DATABASE_TABLE_SIZE;

local	cell	term_template[TEMPLATE];
local	cell	*term_ptr;

local	variable_map	variables[K];	
local	natural		variable_index;
local	variable_map	ov_variables[K];	
local	natural		ov_variable_index;

void dump_database(void);
void write_dynamic(struct _DYNAMIC *r);

/*----------------------    a few utility functions    ---------------------*/
/*----------------------------------------------------------------------------
poke_dynamic(ref)
----------------------------------------------------------------------------*/

extern int fprintf (FILE *, const char *, ...);
extern void writeln_term (cell *term);

static void 
poke_dynamic(struct _DYNAMIC *ref)
{
	/*
	fprintf(stderr, "poking %x", ref);
	*/
        if(ref) {
		++(ref->conceptual_ref_count);
		/*
		fprintf(stderr, "(%d)\n", (ref->conceptual_ref_count));
		*/
	}
	/*
	else
	*/
		/*
		fprintf(stderr, "()\n");
		*/
}
/*----------------------------------------------------------------------------
ignore_dynamic(ref)
----------------------------------------------------------------------------*/
static void
ignore_dynamic(struct _DYNAMIC *ref)
{
	/*
	fprintf(stderr, "ignoring %x", ref);
	*/
        if(ref) {
	    /*
	    fprintf(stderr, "(%d)\n", ((ref)->conceptual_ref_count)-1);
	    */
	    if(!(--((ref)->conceptual_ref_count))) {
		/*
		free((char *)ref);
		*/
		/*
		fprintf(stderr, "free %x\n", ref);
		*/
	    }
	}
	/*
	else fprintf(stderr, "\n");
	*/
}
/*----------------------------------------------------------------------------
is_retracted(ref)
----------------------------------------------------------------------------*/
static boolean
is_retracted(struct _DYNAMIC *ref)
{
        if(!ref) return(FALSE);
        return(ref->is_retracted);
}
/*----------------------------------------------------------------------------
assign_dynamicp(pl, r)
----------------------------------------------------------------------------*/

static void
assign_dynamicp(struct _DYNAMIC **pl, struct _DYNAMIC *r)
{
	/*
	fprintf(stderr, "assigning %x = %x\n", *pl, r);
	*/
	ignore_dynamic(*pl);
	*pl = r;
	poke_dynamic(r);
	/*
	fprintf(stderr, "\n");
	*/
}
/*----------------------------------------------------------------------------
initialise_database_table()
    allocate the memory for the database table, note the size should be a
    large prime number. Zeroise each element in the array.
----------------------------------------------------------------------------*/
global	void
initialise_database_table(void)
{
	natural	i;

	if ((database_table =
		(DYNAMIC **) malloc((natural) (database_table_size *
					       sizeof(DYNAMIC *)))
	    ) == NULL)
		fatal("not enough memory for the database table %d",
			database_table_size);
	for (i = 0; i < database_table_size; i++)
		database_table[i] = (DYNAMIC *)NULL;
}
/*----------------------------------------------------------------------------
insert_record(new, front)
    insert a record at the front of the list of records for that functor and
    arity (hashing may hash more than one predicate (functor/arity) to the
    same list).
----------------------------------------------------------------------------*/
local	void
insert_record(struct _DYNAMIC *new, struct _DYNAMIC **front)
{
	if (*front != NULL)
		(*front)->back_ptr = &(new->next);
	assign_dynamicp(&(new->next), *front); 	
	new->back_ptr = front;
	assign_dynamicp(front, new); 		

}
/*----------------------------------------------------------------------------
append_record(new, back)
    append the record (clause) to the end of the list for the predicate.
    Note: Hashing may hash different predicates to the same list.

    chain down the list until the end is reached and add the new element.
----------------------------------------------------------------------------*/
local	void
append_record(struct _DYNAMIC *new, struct _DYNAMIC **back)
{
	while (*back != NULL)
		back = &((*back)->next);
	new->back_ptr = back;
	assign_dynamicp(&(new->next), NULL);
	assign_dynamicp(back, new);

}

/*----------------------------------------------------------------------------
assert(add_list)
    assert a record given in X0 in the front of the procedure (list of
    clauses) for the predicate or at the back depending on the function
    passed in add_list (either insert_record or append_record).

    f(a). is stored as f(a) :- true.
----------------------------------------------------------------------------*/
local	boolean
assert(void (*add_list) (/* ??? */))
{
	offset	fn;
	natural	arity;
	DYNAMIC	*new;
	local	DYNAMIC	*copy_term_to_database(VALUE *term);
	VALUE	val;

	if (IsApply(Xdref(0)) || IsAtom(X(0)))
	{
		if (IsAtom(X(0)))
		{
			fn = RestOfConstant(X(0));
			arity = 0;
		}
		else
		{
			fn = RestOfConstant(fetch_functor(X(0)));
			arity = fetch_arity(X(0));
		}

		if (!IsAtom(X(0)) && fn == add_name_string_offset(":-", ATOM_W)
			&& arity == 2)
		{ /* Head :- Body */
			fn = RestOfConstant(
				  fetch_functor(Argument(Functor(X(0)))));
			arity = fetch_arity(Argument(Functor(X(0))));
			val.term = X(0);
			val.sub = XS(0);
		}
		else
		{
			val.term = Apply();
			val.sub = XS(0);
			Functor(val.term) = Apply();
			Functor(Functor(val.term)) =
				Atom(add_name_string_offset(":-", ATOM_W));
			Argument(Functor(val.term)) = X(0);
			Argument(val.term) = Atom(add_name_string_offset("true",
			                                               ATOM_W));
		}

		new = copy_term_to_database(&val);
		new->functor = fn;
		new->arity = arity;
		new->conceptual_ref_count = (unsigned)0;
		new->is_retracted = FALSE;
		new->next = (DYNAMIC *)NULL;
		new->back_ptr = (DYNAMIC **)NULL;

		(*add_list)(new,
			&database_table[HashDynamic(fn, arity)]);
		return(TRUE);
	}
	else
		return(FALSE);
}
/*----------------------------------------------------------------------------
$asserta(Term) :-
    Insert the term into the front its appropriate predicate definition.
----------------------------------------------------------------------------*/
global	boolean
esc_asserta(void)
{
	local	void	insert_record(struct _DYNAMIC *new, struct _DYNAMIC **front);

	return(assert(insert_record));
}
/*----------------------------------------------------------------------------
$assertz(Term) :-
    Append the term to the back of its appropriate predicate definition.
----------------------------------------------------------------------------*/
global	boolean
esc_assertz(void)
{
	local	void	append_record(struct _DYNAMIC *new, struct _DYNAMIC **back);

	return(assert(append_record));
}
/*----------------------------------------------------------------------------
$first_clause(Functor, Arity, Ref) :-
    Ref is the pointer to the first clause of the predicate Functor/Arity.

    Chain down the list of records in the database_table Hash entry for
    Functor/Arity, until a record is found for that predicate.

    Note: several predicates may hash to the same entry in the database table.
----------------------------------------------------------------------------*/
global	boolean
esc_first_clause(void)
{
	offset	fn;
	natural	arity;
	DYNAMIC	*ref;
	VALUE	val;

	if (IsAtom(Xdref(0)) && IsInteger(Xdref(1)))
	{
		fn = RestOfConstant(X(0));
		arity = IntOf(X(1));

		for (ref = database_table[HashDynamic(fn, arity)];
		     ref != NULL;
		     ref = ref->next)
			if (ref->functor == fn && ref->arity == arity) {
				val.term = PtrToInt(ref);
				val.sub = EMPTY_SUB;
				if(unify(&val, XV(2))) {
				    /*
				    fprintf(stderr, "$first_clause\n");
				    */
				    poke_dynamic(ref);
				    return(TRUE);
				}
				else
				    return(FALSE);
			}
		return(FALSE);
	}
	else
		return(FALSE);
}
/*----------------------------------------------------------------------------
$next_clause(Ref, NewRef) :-
    NewRef is the pointer to the next record of the predicate pointed to be
    Ref after Ref.
----------------------------------------------------------------------------*/
global	boolean
esc_next_clause(void)
{
	DYNAMIC	*ref;
	offset	fn;
	natural	arity;
	VALUE	val;

	if (IsInteger(Xdref(0)) && (ref = (DYNAMIC *)IntToPtr(X(0))) != NULL)
	{
		fn = ref->functor;
                arity = ref->arity;

                for (assign_dynamicp(&ref, ref->next); 
                     ref != NULL; 
                     assign_dynamicp(&ref, ref->next))
			if(!is_retracted(ref))
                        if (ref->functor == fn && ref->arity == arity)
                        {
                                val.term = PtrToInt(ref);
                                val.sub = EMPTY_SUB;
                                return(unify(&val, XV(1)));
                        }
                return(FALSE);
	}
	else
		return(FALSE);
}
/*----------------------------------------------------------------------------
$last_clause(Ref) :-
    Check whether this is the last alternative clause.
----------------------------------------------------------------------------*/
global	boolean
esc_last_clause(void)
{
	DYNAMIC	*ref;
	offset	fn;
	natural	arity;

	if (IsInteger(Xdref(0)) && (ref = (DYNAMIC *)IntToPtr(X(0))) != NULL)
	{
		fn = ref->functor;
		arity = ref->arity;
		
		for (ref = ref->next; ref != NULL; ref = ref->next)
			if (ref->functor == fn && ref->arity == arity)
				return(FALSE);
		return(TRUE);
	}
	else
		return(FALSE);
}
/*----------------------------------------------------------------------------
$erase(Ref) :-
    Remove the record pointed to be Ref from the list of records in the
    database table entry.
----------------------------------------------------------------------------*/
global	boolean
esc_erase(void)
{
	DYNAMIC	*ref;

	if (IsInteger(Xdref(0)))
	{
		ref = (DYNAMIC *)IntToPtr(X(0));
		ref->is_retracted = TRUE;
                if (ref->next)
                        ref->next->back_ptr = ref->back_ptr;
                assign_dynamicp(ref->back_ptr, ref->next);
                return(TRUE);
	}
	else
		return(FALSE);
}
/*----------------------------------------------------------------------------
$instance(Ref, Term) :-
    True, if Ref points to a record that will unify with Term.
----------------------------------------------------------------------------*/
global	boolean
esc_instance(void)
{
	DYNAMIC	*ref;
	local	void	copy_term_to_heap(cell *to, cell from);
	VALUE	val;

	if (IsInteger(Xdref(0)))
	{
		ref = (DYNAMIC *)IntToPtr(X(0));
		copy_term_to_heap(&(val.term), *(ref->term_space));
		val.sub = EMPTY_SUB;
		return(unify(&val, XV(1)));
	}
	else
		return(FALSE);
}
/*----------------------------------------------------------------------------
$instance_no_copy(Ref, Term) :-
    True, if Ref points to a record that will unify with Term.
----------------------------------------------------------------------------*/
global	boolean
esc_instance_no_copy(void)
{
	DYNAMIC	*ref;
	VALUE	val;

	if (IsInteger(Xdref(0)))
	{
		ref = (DYNAMIC *)IntToPtr(X(0));
		val.term = *ref->term_space;
		val.sub = EMPTY_SUB;
		return(unify(&val, XV(1)));
	}
	else
		return(FALSE);
}
/*----------------------------------------------------------------------------
copy_term_to_database(term)
    return a pointer to the copy of term.
    translate structure pointers to be relative to the allocated record,
    also allocate in contiguous block.
----------------------------------------------------------------------------*/
local	DYNAMIC *
copy_term_to_database(VALUE *term)
{
	DYNAMIC	*ref;
	local	cell	copy_term_to_db(VALUE *term);
	local void	translate_cells(cell *to, cell *from, cell *start);
	natural	i;

	term_ptr = term_template + 1;
	term_template[0] = copy_term_to_db(term);
	i = (natural)SizeDynamic(term_ptr - term_template);
	if ((ref = (DYNAMIC *) malloc(i)) == NULL)
		fatal("not enough memory for assert");
	translate_cells(ref->term_space, term_template, ref->term_space);
	return(ref);
}
/*----------------------------------------------------------------------------
copy_term_to_db(term)
    copy a term to the database template, allocating structures in the
    contiguous template.
----------------------------------------------------------------------------*/
local	cell
copy_term_to_db(VALUE *term)
{
	cell	value;
	VALUE	new;
	VALUE	val;
local	cell	copy_sub_to_db(cell sub);

	DereferenceTerm(val, term->term);

	if (val.sub != EMPTY_SUB)
	{
		value = SubstitutionOperatorDB();
		Substitution(value) = copy_sub_to_db(val.sub);
		new.sub = EMPTY_SUB;
		new.term = val.term;
		Term(value) = copy_term_to_db(&new);
		return(value);
	}
	else switch (Tag(val.term))
	{
	when CONSTANT:
		return(val.term);
	when REFERENCE:
		if (InPersistentStack(val.term))
			return(val.term);
		else
			return(RestOfTemp(val.term)|
					Temperature(Value(val.term)));
	when OBJECT_REFERENCE:
		if (InPersistentStack(val.term))
			return(val.term);
		else if (IsLocalObjectVariable(val.term))
			return(RestOfTemp(val.term) |
				Temperature(Value(val.term)) |
				LocalFlag);
		else
			return(RestOfTemp(val.term) |
				Temperature(Value(val.term)));
	when APPLY:
		value = ApplyDB();
		FunctorV(new, val);
		Functor(value) = copy_term_to_db(&new);
		ArgumentV(new, val);
		Argument(value) = copy_term_to_db(&new);
		return(value);
	when PAIR:
		value = PairDB();
		LeftV(new, val);
		Left(value) = copy_term_to_db(&new);
		RightV(new, val);
		Right(value) = copy_term_to_db(&new);
		return(value);
	when QUANTIFIER:
		value = QuantifierDB();
		BoundVarV(new, val);
		BoundVar(value) = copy_term_to_db(&new);
		BodyV(new, val);
		Body(value) = copy_term_to_db(&new);
		return(value);
	}
	return(val.term);
}
/*------------------------------------------------------------------------------
copy_sub_to_db(sub)
------------------------------------------------------------------------------*/
local	cell
copy_sub_to_db(cell sub)
{
	natural	i, j;
	cell	*table;
	cell	newsub;
	cell	tail;
	VALUE	val;

	if (sub == EMPTY_SUB)
		return(sub);
	else
	{
		tail = copy_sub_to_db(NextSub(sub));
		j = Size(sub);
		table = AllocateDB(2 * j + 1);
		newsub = NewSubstitutionDB((cell)table, tail, Property(sub));
		Size(newsub) = j;
		for (i = 1; i <= j; i++)
		{
			val.sub = EMPTY_SUB;
			val.term = Domain(sub, i);
			Domain(newsub, i) = copy_term_to_db(&val);
			val.sub = EMPTY_SUB;
			val.term = (IsLocalObjectVariable(Range(sub, i)) &&
				    ! in_all_domain(Range(sub, i),
						    NextSub(sub))) ?
					Atom(DOLLAR) :
					Reference(&Range(sub, i));
			Range(newsub, i) = copy_term_to_db(&val);
		}
		return(newsub);
	}
}
/*----------------------------------------------------------------------------
translate_cells(to, from, start)
    copy cells from ->  to, translating ptrs to be relative to to, instead of
    the term template.
----------------------------------------------------------------------------*/
local void
translate_cells(cell *to, cell *from, cell *start)
{
local	void	translate_sub_cells(cell *to, cell *from, cell *start);

	switch (Tag(*from))
	{
	when APPLY:
		*to = (cell) (APPLY|(cell)(start + 
				(((cell *)RestOf(*from)) - term_template)));
		translate_cells(&Functor(*to), &Functor(*from), start);
		translate_cells(&Argument(*to), &Argument(*from), start);
	when PAIR:
		*to = (cell) (PAIR|(cell)(start + 
				(((cell *)RestOf(*from)) - term_template)));
		translate_cells(&Left(*to), &Left(*from), start);
		translate_cells(&Right(*to), &Right(*from), start);
	when QUANTIFIER:
		*to = (cell) (QUANTIFIER|(cell)(start + 
				(((cell *)RestOf(*from)) - term_template)));
		translate_cells(&BoundVar(*to), &BoundVar(*from), start);
		translate_cells(&Body(*to), &Body(*from), start);
	when SUBSTITUTION_OPERATOR:
		*to = (cell) (SUBSTITUTION_OPERATOR|(cell)(start + 
				(((cell *)RestOf(*from)) - term_template)));
		translate_sub_cells(&Substitution(*to), &Substitution(*from),
					start);
		translate_cells(&Term(*to), &Term(*from), start);
	otherwise:
		*to = *from;
	}
}
/*-----------------------------------------------------------------------------
translate_sub_cells(to, from, start)
-----------------------------------------------------------------------------*/
local	void
translate_sub_cells(cell *to, cell *from, cell *start)
{
	natural	i, j;

	if (*from == EMPTY_SUB)
		*to = *from;
	else
	{
		*to = NewProperty((cell)(start + (SubPointer(*from) -
							term_template)),
				  Property(*from));
		Table(*to) = (cell)(start + ((cell *)Table(*from) -
				term_template));
		Size(*to) = Size(*from);
		for (i = 1, j = Size(*from); i <= j; i++)
		{
			translate_cells(&Domain(*to, i), &Domain(*from, i),
					start);
			translate_cells(&Range(*to, i), &Range(*from, i),
					start);
		}
		translate_sub_cells(&NextSub(*to), &NextSub(*from), start);
	}
}
/*----------------------------------------------------------------------------
copy_term_to_heap(to, from)
    copy the term, from, in a database record to the heap pointed to be to.
----------------------------------------------------------------------------*/
local	void
copy_term_to_heap(cell *to, cell from)
{
	local	void	 copy_term_to_hp(cell *to, cell from);
reg	int i, j;

	variable_index = 0;
	ov_variable_index = 0;
	copy_term_to_hp(to, from);
	for (i = 0; i < ((int)ov_variable_index - 1); i++)
	    if (! (ov_variables[i].variable & LocalFlag))
		for (j = i + 1; j < ov_variable_index; j++)
		    if (! (ov_variables[j].variable & LocalFlag))
			SetDistinct(ov_variables[i].reference,
				    ov_variables[j].reference);
}
/*----------------------------------------------------------------------------
copy_term_to_hp(to, from)
    copy the term from (in database record) to to (in heap) replacing
    references to variables with references to new variables in the heap.
----------------------------------------------------------------------------*/
local	void
copy_term_to_hp(cell *to, cell from)
{
	natural	i;
	local	void	 copy_sub_to_hp(cell *to, cell from);

        switch (Tag(from))
	{
	when CONSTANT:
		*to = from;
	when REFERENCE:
		if (InPersistentStack(from))
			*to = from;
		else
		{
			for (i = 0; i < variable_index; i++)
				if (variables[i].variable == from)
				{
					*to = variables[i].reference;
					return;
				}
			*to =  NewVariable(); 
			Value(*to) = Value(*to)|Temperature(from);
			variables[variable_index].variable = from;
			variables[variable_index++].reference = *to;
		}
	when OBJECT_REFERENCE:
		if (InPersistentStack(from))
			*to = from;
		else
		{
			for (i = 0; i < ov_variable_index; i++)
				if (ov_variables[i].variable == from)
				{
					*to = ov_variables[i].reference;
					return;
				}
			*to = (from & LocalFlag ? NewLocalObjectVariable() :
						  NewObjectVariable());
			Value(*to) = Value(*to)|Temperature(from);
			ov_variables[ov_variable_index].variable = from;
			ov_variables[ov_variable_index++].reference = *to;
		}
	when APPLY:
		*to = Apply();
		copy_term_to_hp(&Functor(*to), Functor(from));
		copy_term_to_hp(&Argument(*to), Argument(from));
	when PAIR:
		*to = Pair();
		copy_term_to_hp(&Left(*to), Left(from));
		copy_term_to_hp(&Right(*to), Right(from));
	when QUANTIFIER:
		*to = Quantifier();
		copy_term_to_hp(&BoundVar(*to), BoundVar(from));
		copy_term_to_hp(&Body(*to), Body(from));
	when SUBSTITUTION_OPERATOR:
		*to = SubstitutionOperator();
		copy_sub_to_hp(&Substitution(*to), Substitution(from));
		copy_term_to_hp(&Term(*to), Term(from));

	}

}
/*-----------------------------------------------------------------------------
copy_sub_to_hp(to, from)
-----------------------------------------------------------------------------*/
local	void
copy_sub_to_hp(cell *to, cell from)
{
	natural	i, j;
	cell	*table;

	if (from == EMPTY_SUB)
		*to = from;
	else
	{
		j = Size(from);
		table = Allocate(2 * j + 1);
		*to = NewSubstitution((cell)table, EMPTY_SUB, Property(from));
		Size(*to) = j;
		for (i = 1; i <= j; i++)
		{
			copy_term_to_hp(&Domain(*to, i), Domain(from, i));
			copy_term_to_hp(&Range(*to, i), Range(from, i));
		}
		copy_sub_to_hp(&NextSub(*to), NextSub(from));
	}
}
/*----------------------------------------------------------------------------
$static_predicate(F, N, Address)
    true if the predicate F/N is a compiled predicate at Address.
----------------------------------------------------------------------------*/
global	boolean
esc_static_predicate(void)
{
	boolean	insert = FALSE;
	offset	code_offset;
	offset	fn;
	natural	arity;
	VALUE	val;

	if (IsAtom(Xdref(0)) && IsInteger(Xdref(1)))
	{
		fn = RestOfConstant(X(0));
		arity = IntOf(X(1));
		if ((code_offset = lookup_predicate_table(fn, arity,
			NULL_OFFSET, insert)) != NULL_OFFSET)
		{
			val.term = Integer(code_offset);
			val.sub = EMPTY_SUB;
			return(unify(&val, XV(2)));
		}
	}
	return(FALSE);
		
}
/*----------------------------------------------------------------------------
$first_not_free()
----------------------------------------------------------------------------*/
global	boolean
esc_first_not_free(void)
{
	VALUE	val;
	cell	*ref;

	if (IsObjectReference(Xdref(0)) || IsReference(X(0)))
	{
		for (ref = (cell *)RestOfVariable(Value(X(0)));
		     ref != NULL;
		     ref = Next(ref))
			if (DelayedSolved(ref) == UNSOLVED &&
			    DelayedType(ref) == NOT_FREE)
			{
				val.term = PtrToInt(ref);
				val.sub = EMPTY_SUB;
				return(unify(&val, XV(1)));
			}
		return(FALSE);
	}
	else
		return(FALSE);
}
/*----------------------------------------------------------------------------
$get_not_free()
----------------------------------------------------------------------------*/
global	boolean
esc_get_not_free(void)
{
	cell	*p;
	cell	sub;
	VALUE	term;
	VALUE	call;

	if (! IsInteger(Xdref(0)))
	    return(FALSE);
	else
	{
	    p = (cell *)IntToPtr(X(0));
	    term.sub = EMPTY_SUB;
	    term.term = DelayedTerm(p);
	    call.sub = EMPTY_SUB;
	    call.term = apply(apply(Atom(add_name_string_offset("not_free_in", 
		                    ATOM_W)), DelayedTerm(p)), DelayedVar(p));
	    return(unify(XV(1), &term) && unify(XV(2), &call));
	}
}
/*----------------------------------------------------------------------------
$next_not_free()
----------------------------------------------------------------------------*/
global	boolean
esc_next_not_free(void)
{
	VALUE	val;
	cell	*ref;

	if (! IsInteger(Xdref(0)))
		return(FALSE);
	else
	{
		for (ref = Next((cell *)IntToPtr(X(0)));
		     ref != NULL;
		     ref = Next(ref))
			if (DelayedSolved(ref) == UNSOLVED &&
			    DelayedType(ref) == NOT_FREE)
			{
				val.term = PtrToInt(ref);
				val.sub = EMPTY_SUB;
				return(unify(&val, XV(1)));
			}
		return(FALSE);
	}
}
/*----------------------------------------------------------------------------
dump_database()
----------------------------------------------------------------------------*/
void
dump_database(void)
{
	DYNAMIC **cursor=database_table;
	int i;
	fprintf(stderr, "database_table ... %x\n\n", database_table);
	fprintf(stderr, "tadd\taddr\tfstr\t\tarity\tnext\tback");
	for(i = 0; i < database_table_size; cursor=database_table+(i++))
		{
		DYNAMIC * c2=*cursor;
		if(c2)
		fprintf(stderr, "\n****************************************************");
		while(c2)
		    {
#ifdef	EBUG
		    write_dynamic(c2);
#endif  /* EBUG */
		    c2=(c2)->next;
		    }
		}
}

#ifdef	EBUG
/*----------------------------------------------------------------------------
write_dynamic(r)
----------------------------------------------------------------------------*/
void
write_dynamic(struct _DYNAMIC *r)
{
    if(r) {
	fprintf(stderr, "%x\t", r);
	fprintf(stderr, "%s\t", string_table+((r)->functor));
	fprintf(stderr, "%d\t", (r)->arity);
	fprintf(stderr, "%x\t", (r)->next);
	fprintf(stderr, "%x\t", (r)->conceptual_ref_count);
	fprintf(stderr, "%x\n", (r)->back_ptr);
	writeln_term((r)->term_space);
    }
}

#endif  /* EBUG */
