/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 *
 * write - Procedures to write out QuAM code to the .qs assembly file.
 */

/*----------------------------------------------------------------------------
write_clauses(Clauses, Predicate, IndexingInstructions, ClausesInstructions) :-

    Clauses	The actual clauses being compiled.
    Predicate	F/N
    IndexingInstructions 

			switch_on_term(Var, Const, Apply, Pair, Quant, Obvar)
            Var:        try(1)
                        retry(2)
                        ...
                        retry(n-1)
                        trust(n)
            Const:      switch_on_constant(m, [default(Default),
                                               cluster(Constant1, Constant1L),
						 ...])
            Default:    try(i)
                        retry(j)
                        ...
                        trust(k)
                           
            Constant1L: try(o)
                        retry(p)
                        ...
                        trust(r)
	    Apply:      switch_on_structure(m, [default(Default),
                                                cluster(F/N, StruL), ...])
            StruL:      ....

    ClausesInstructions
 
            1:          <clause 1's instructions>
            2:          <clause 2's instructions>
 
                        ....
 
            n:          <clause n's instructions>


    Add the labels
	F/N :
	    <IndexingInstructions>
	    <ClausesInstructions>
	end(F/N):
 

----------------------------------------------------------------------------*/
write_clauses(Clauses, Predicate, IndexingInstructions, ClausesInstructions):-
    allocate_vars(Clauses),
    allocate_vars(ClausesInstructions),
    write_instruction(label(Predicate)),
    write_instructions(IndexingInstructions),
    write_clauses2(Clauses, ClausesInstructions),
    write_instruction(label(end(Predicate))),
    fail.


/*----------------------------------------------------------------------------
write_clauses2(Clauses, Instructions) :-
----------------------------------------------------------------------------*/
write_clauses2([], []).
write_clauses2([Clause|Clauses], [InstructionSet|Instructions]) :-
    nl,
    write_clause_comment(Clause),
    write_instructions(InstructionSet),
    write_clauses2(Clauses, Instructions).


/*----------------------------------------------------------------------------
write_clause_comment(Clause) :-
----------------------------------------------------------------------------*/
write_clause_comment((Head :- Body)) :-
    !,
    write('/*'),
    nl,
    write(' * '),
    write(Head),
    write(' :- '),
    nl,
    rhs_list(Body, Bodies),
    write_instructions_comment(Bodies),
    write(' */'),
    nl.

write_clause_comment(Head) :-
    write('/*'),
    nl,
    write(' * '),
    write(Head),
    write('.'),
    nl,
    write(' */'),
    nl.


/*----------------------------------------------------------------------------
write_instructions_comment(Bodies) :-
----------------------------------------------------------------------------*/
write_instructions_comment([F]) :-
    write_formula_comment(F),
    write('.'),
    nl.
write_instructions_comment([F, F2|Bodies]) :-
    write_formula_comment(F), 
    write(','),
    nl,
    write_instructions_comment([F2|Bodies]).

/*----------------------------------------------------------------------------
write_formula_comment(F) :-
----------------------------------------------------------------------------*/
write_formula_comment(F) :-
    write(' *	'),
    write(F).

/*----------------------------------------------------------------------------
write_instructions(Body) :-
    Writes out the list of QuAM instructions in Body to the ".qs" file.
----------------------------------------------------------------------------*/
write_instructions(Is) :-
    member(I, Is),
    write_instruction(I),
    fail.
write_instructions(_).

write_instruction(label(Label)) :-
    !,
    nl,
    write_label(Label),
    write(:),
    nl.
write_instruction(Instruction) :-
    Instruction =.. [F|Args],
    write('	'),
    write(F),
    (Args \== [] ->
	write('('),
	write_args(Args),
	write(')')
    ;
	true
    ),
    nl.

write_label(F/N) :-
    !,
    write_arg(F),
    write(/),
    write(N).
write_label(end(F/N)) :-
    !,
    write('end('),
    write_arg(F),
    write(/),
    write(N), 
    write(')').
write_label(Label) :-
    write(Label).

/*----------------------------------------------------------------------------
write_args(Args) :-
    Writes out a list of QuAM instruction arguments, separated by commas.
----------------------------------------------------------------------------*/
write_args([Arg]) :-
    !,
    write_arg(Arg).
write_args([Arg|Args]) :-
    write_arg(Arg),
    write(', '),
    write_args(Args).


write_arg(Arg) :-
    Arg = [_Head|_Tail],
    !,
    write('['),
    write_args(Arg),
    write(']').
write_arg(Term:Label) :-
    !,
    write_arg(Term),
    write(':'),
    write_arg(Label).
write_arg(F/N) :-
    !,
    write_arg(F),
    write('/'),
    write_arg(N).
write_arg(default(Label)) :-
    !,
    put(0''),
    write(default),
    put(0''),
    write('('),
    write_arg(Label),
    write(')').
write_arg(Arg) :-
    integer(Arg),
    !,
    write(Arg).
write_arg(Arg) :-
    unquoted_atom(Arg),
    !,
    put(0''),
    write(Arg),
    put(0'').
write_arg(Arg) :-
    writeq(Arg).

% write_arg(Arg) :-
%     name(Arg, CharList),
%     add_char(0'', CharList, CharList1),
%     name(Arg1, CharList1),
%     put(0''),
%     write(Arg1),
%     put(0'').

unquoted_atom('!') :- !.
unquoted_atom(';') :- !.
unquoted_atom('[]') :- !.
unquoted_atom('{}') :- !.
unquoted_atom(A) :-
	name(A, AL),
	memberall(AL, "+-*/\^<>=`~:.?@#$&"),
	!.
unquoted_atom(A) :-
	name(A, [L|AL]),
	member(L, "abcdefghijklmnopqrstuvwxyz"),
	memberall(AL, "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_1234567890").

memberall([], _):- !.
memberall([X|AL], L) :-
	member(X, L),
	memberall(AL, L).

add_char(_, [], []):- !.
add_char(Char, [Char|CharList], [Char, Char|CharList1]):-
	!,
	add_char(Char, CharList, CharList1).
add_char(Char, [C|CharList], [C|CharList1]):-
	add_char(Char, CharList, CharList1).

/*----------------------------------------------------------------------------
write_allocations(Instructions, Registers) :-
	Debugging only
----------------------------------------------------------------------------*/
write_allocations([Head|Instructions], [HeadRegs|Registers]) :-
    write(Head),
    write((:-)),
    tab(8),
    write(HeadRegs),
    nl,
    write_rest(Instructions, Registers).


/*----------------------------------------------------------------------------
write_rest(Instructions, Registers) :-
	Debugging only
----------------------------------------------------------------------------*/
write_rest([], []).
write_rest([Instruction|Instructions], [Register|Registers]) :-
    tab(8),
    write(Instruction),
    tab(8),
    write(Register),
    nl,
    write_rest(Instructions, Registers).
