/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 *
 * unravel - Unravel clauses into QuAM assembly code instructions.
 */

/*----------------------------------------------------------------------------
unravel(Sentence, UnravelledInstructions) :-

    The unravel for arguments in QuAM in the head is


	head [| p(a1, ..., an) |] ->
	    head(p(A1, ..., An)),
	    get [| a1, A1 |],
	    ...
	    get [| an, An |].


	get [| f(t1, ..., tn), A |] ->
	    get_apply(A1, A2, A),
	    get [| f(t1, ..., tn-1), A1 |],
	    get [| tn, A2 |].


	get [| X, A |] ->
	    get(_, X, A).

	get [| c, A |] ->
	    get_constant(c, A).

    for the Qu-Prolog extensions (Note: Persistents {X#, x#} are not allowed
					in compiled code)
	get [| x, A |] ->
	    get_object_variable(_, x, A).
	
	get [| x@n, A |] -> ?
	
	get [| x ^ t, A |] ->
	    if (A1 is not initialised by put_object_variable or
			get_object_variable)
		put_object_variable(_, A1, A1)
	    free_in(A1, A)		} done by peephole
	    get_quantifier(A1, A2, A)
	    get [| t, A2 |]

	% Before the first call/escape/proceed, add 'do_delayed_problems'
	% instruction.  To be done in peephole.

	get [| [t1/x1, ..., tn/xn |] * t, A |] ->
	    put [| x1, A1 |]
	    put [| t1, A2 |]
	    ...
	    put [| xn, A2n-1 |]
	    put [| tn, A2n |]
	    put_parallel_substitution(n, B)
	    put_sub_pair(A1, A2),
	    ...
	    put_sub_pair(A2n-1, A2n)
	    if (for all ti is object variables)
		set_object_property(B)
	    else
		determine_property(B)
	    put [| t, B |]
	    get_value(B, A)		% i.e. build substituion and term
					% then call general unify

    and for arguments in the body


	goal [| ! |] ->
	    cut.
	goal [| fail |] ->
	    fail.
	goal [| t1 = t2 |] ->
	    put [| t1, A |]
	    get [| t2, A |].
	goal [| p(a1, ..., an) |] ->
	    put [| a1, A1 |],
	    ...
	    put [| an, An |],
	    call_predicate(p, n, EnvSize). (escape(p, n) if built-in function)



	construct the branch of the apply
		f(tn, ..., tn) = @(f(t1, ..., tn-1), tn)
	which has the greatest depth to minimize the number of registers
	used.

	l = f(t1, ..., tn-1), r = tn
	depth:
	    ld := depth(l)
	    rd := depth(r)
	    if ld > rd
	    then l, r, put_apply
		 d := rd + 1
	    else r, l, put_apply
		 d := ld + 1

	for l, r, put_apply

	    put [| f(t1, ..., tn), A |] ->
		put [| f(t1, ..., tn-1), A1 |],
		put [| tn, A2 |],
		put_apply(A1, A2, A).

	for r, l, put_apply

	    put [| f(t1, ..., tn), A |] ->
		put [| tn, A2 |],
		put [| f(t1, ..., tn-1), A1 |],
		put_apply(A1, A2, A).


	put [| X, A |] ->
	    put(_, X, A).

	put [| c, A |] ->
	    put_constant(c, A).

	for Qu-Prolog

	put [| x, A |] ->
	    put_object_variable(_, x, A)
	
	put [| x@n, A |] -> ?

	put [| x ^ t, A |] ->
	    put [| x, A1 |]
	    put [| t, A2 |]
	    put_quantifier(A1, A2, A)
	
	put [| [t1/x1, ..., tn/xn |] * t, A |] ->
	    put [| x1, A1 |]
	    put [| t1, A2 |]
	    ...
	    put [| xn, A2n-1 |]
	    put [| tn, A2n |]
	    put_parallel_substitution(n, A)
	    put_sub_pair(A1, A2),
	    ...
	    put_sub_pair(A2n-1, A2n)
	    if (for all ti is object variables)
		set_object_property(A)
	    else
		determine_property(A)
	    put [| t, A |]
	
----------------------------------------------------------------------------*/
unravel(Sentence, OVsRegs, UnravelledInstructions) :-
    sentence(Sentence, OVsRegs, UnravelledHead - [], UnravelledBody - []),
    append(UnravelledHead, UnravelledBody, UnravelledInstructions).

/*----------------------------------------------------------------------------
sentence(Sentence, OVsRegs, UnravelledHead, UnravelledBody) :-
    UnravelledHead and UnravelledBody are difference list of instructions
    coded from the Sentence. If the Sentence UnravelledBody is the empty
    difference list.
----------------------------------------------------------------------------*/
sentence((Head :- Body), OVsRegs, NewHead, NewBodies) :-
    !,
    unravel_head(Head, OVsRegs, NewHead), 
    rhs_list(Body, Bodies),
    unravel_bodies(Bodies, OVsRegs, NewBodies).
sentence(F, OVsRegs, Gs, Tail - Tail) :-
    unravel_head(F, OVsRegs, Gs).

/*----------------------------------------------------------------------------
unravel_head(Formula, OVsRegs, UnravelledFormula) :-
    UnravelledFormula is the instructions used to implement the Formula in
    the head of the clause.

	head [| p(a1, ..., an) |] ->
	    head(p(A1, ..., An)),
	    get [| a1, A1 |],
	    ...
	    get [| an, An |].

    the instruction head(F/N) is used to indicate which registers are alive 
    that is, the first N registers. These are allocated by instantiation,
    Hence, they are sprinkled through the instructions.
----------------------------------------------------------------------------*/
unravel_head(Predicate, OVsRegs, [head(F/N)|ArgsUnifications] - Tail) :-
    Predicate =.. [F|Args],
    length(Args, N),
    length(NewArgs, N),
    unravel_head_args(Args, OVsRegs, NewArgs, ArgsUnifications - Tail),
    allocate_x_registers(NewArgs).


/*----------------------------------------------------------------------------
unravel_head_args(Args, OVsRegs, NewArgs, UnravelledArgs) :-

	get [| a1, A1 |],
	...
	get [| an, An |].

----------------------------------------------------------------------------*/
unravel_head_args([], _OVsRegs, [], Unifications - Unifications).
unravel_head_args([Arg|Args], OVsRegs, [NewArg|NewArgs],
	ArgUnifications - Tail) :-
    unravel_get_arg(Arg, OVsRegs, NewArg, ArgUnifications - ArgsUnifications),
    unravel_head_args(Args, OVsRegs, NewArgs, ArgsUnifications - Tail).


/*----------------------------------------------------------------------------
unravel_get_arg(Arg, OVsRegs, NewArg, UnravelledArg) :-
    unravel_get_arg(f(t1, ..., tn))
	unravel_get_term([tn, ..., t1, f])
----------------------------------------------------------------------------*/
unravel_get_arg(Arg, OVsRegs, NewArg, UnravelledArg) :-
    ('$isa_unpersistent_var'(Arg); atomic(Arg)), !,
    unravel_get_constant_or_var(Arg, OVsRegs, NewArg, UnravelledArg).
unravel_get_arg(Arg, OVsRegs, NewArg, UnravelledArg) :-
    Arg =.. [F|Args],
    unravel_get_arg2(F, Args, OVsRegs, NewArg, UnravelledArg).

unravel_get_arg2((^), [X, T], OVsRegs, A, [put_object(_Type, A1, A1),
	put_substitution(empty, A1), get_quantifier(A1, A2, A)|UnravelledArg]-Tail) :-
    '$isa_object_var'(X),
    !,
    '$dereference_var2'(X, OVsRegs, A1),
    unravel_get_arg(T, OVsRegs, A2, UnravelledArg-Tail).
unravel_get_arg2((*), [Ss, T], OVsRegs, A, UnravelledArg-Tail) :-
    '$isa_substitution'(Ss),
    !,
    unravel_put_arg(T, OVsRegs, B, 
		    UnravelledArg-[put_substitution(eMPTY, B)|Tail2], _),
    unravel_subs(Ss, OVsRegs, B, Tail2-[get(value, B, A)|Tail]).
unravel_get_arg2(F, Args, OVsRegs, NewArg, UnravelledArg) :-
    reverse(Args, ArgsReversed),
    unravel_get_term(ArgsReversed, OVsRegs, F, NewArg, UnravelledArg).

% unravel_subs(Ss, OVsRegs, Instructions, Instructions) :-
%	Ss = [t1/x1, ..., tn/xn] 
%	generate 
%		put [| xn, A2n-1 |]
%		put [| tn, A2n |]
%		...
%		put [| x1, A1 |]
%		put [| t1, A2 |]
%	    	put_parallel_substitution(n, B)
%	    	put_sub_pair(A2n-1, A2n)
%	    	...
%	    	put_sub_pair(A1, A2)

unravel_subs(Ss1*Ss2, OVsRegs, B, UnravelledArg-Tail2) :-
    unravel_subs(Ss1, OVsRegs, B, UnravelledArg-Tail1),
    unravel_subs(Ss2, OVsRegs, B, Tail1-Tail2).
unravel_subs(Ss, OVsRegs, B, UnravelledArg-Tail2) :-
    unravel_subs2(Ss, OVsRegs,
		 UnravelledArg-[put_parallel_substitution(N, B)|Tail1],
		 Tail1-[Instruction|Tail2]),
    length(Ss, N),
    '$property_range'(Ss, B, Instruction).

unravel_subs2([], _OVsRegs, Tail-Tail, Tail1-Tail1).
unravel_subs2([T/X|Ss], OVsRegs, UnravelledArg-Tail3, Tail4-Tail5) :-
    unravel_put_arg(X, OVsRegs, A1, Tail6-Tail, _),
    unravel_put_arg(T, OVsRegs, A2, Tail-Tail2, _),
    (nonvar(T), T =.. [(*), Ss1, _], '$isa_substitution'(Ss1) ->
	Tail2 = [put_substitution_operator(A2, A3)|Tail3],
	A4 = A3
    ;
	Tail2 = Tail3,
	A4 = A2
    ),
    unravel_subs2(Ss, OVsRegs, UnravelledArg-Tail6,
		  Tail4-[put_parallel_substitution_pair(A1, A4)|Tail5]).

%'$property_range'(Ss, B, Instruction) :-
%	if all range are object variables then
%		Instruction = set_object_property(B)
%	else
%		Instruction = determine_property(B)
%
'$property_range'(Ss, B, set_object_property(B)) :-
    \+ (member(T/_X, Ss), \+ '$isa_object_var'(T)), !.
'$property_range'(_Ss, B, determine_property(B)).
    

/*----------------------------------------------------------------------------
unravel_get_term(Term, OVsRegs, F, NewTerm, UnravelledTerm) :-

    Term = [tn, .., t1],  F = f.

	get [| X, A |] ->
	    get(_, X, A).

	get [| c, A |] ->
	    get_constant(c, A).

	get [| f(t1, ..., tn), A |] ->
	    get_apply(A1, A2, A),
	    get [| f(t1, ..., tn-1), A1 |],
	    get [| tn, A2 |].

----------------------------------------------------------------------------*/
unravel_get_term([], OVsRegs, F, A, UnravelledTerm) :-
    unravel_get_constant_or_var(F, OVsRegs, A, UnravelledTerm).
unravel_get_term([Tn|Rest], OVsRegs, F, A, [get_apply(A1, A2, A)|L] - Tail):-
    unravel_get_arg(Tn, OVsRegs, A2, R - Tail),
    unravel_get_term(Rest, OVsRegs, F, A1, L - R).

/*----------------------------------------------------------------------------
unravel_get_constant_or_var(Term, OVsRegs, NewTerm, UnravelledTerm) :-

	get [| X, A |] ->
	    get(_, X, A).

	get [| c, A |] ->
	    get_constant(c, A).
----------------------------------------------------------------------------*/
unravel_get_constant_or_var(T, OVsRegs, A,
	[get_object(_Type, TRegister, A)|Tail] - Tail) :-
    '$isa_object_var'(T), !,
    '$dereference_var2'(T, OVsRegs, TRegister).
unravel_get_constant_or_var(T, _OVsRegs, A,
	[get_constant(T, A)|Tail] - Tail) :-
    atomic(T), !.
unravel_get_constant_or_var(T, _OVsRegs, A, [get(_Type, T, A)|Tail] - Tail).

/*----------------------------------------------------------------------------
unravel_bodies(Body, OVsRegs, UnravelledBody) :-
    UnravelledBody is the difference list of instructions made up from the
    list of goals Body.
----------------------------------------------------------------------------*/
unravel_bodies([], _OVsRegs, Tail - Tail).
unravel_bodies([F|Fs], OVsRegs, F2 - Tail) :-
    !,
    unravel_goal(F, OVsRegs, F2 - Fs2),
    unravel_bodies(Fs, OVsRegs, Fs2 - Tail).

/*----------------------------------------------------------------------------
unravel_goal(Formula, OVsRegs, UnravelledFormula) :-

	goal [| ! |] ->
	    cut.
	goal [| fail |] ->
	    fail.
	goal [| t1 = t2 |] ->
	    put [| t1, A |]
	    get [| t2, A |].
	goal [| p(a1, ..., an) |] ->
	    put [| a1, A1 |],
	    ...
	    put [| an, An |],
	    call_predicate(p, n, EnvSize).	(escape(p, n) if built-in function)

----------------------------------------------------------------------------*/
unravel_goal(!, _OVsRegs, [cut|Tail] - Tail) :-
    !.
unravel_goal(fail, _OVsRegs, [fail|Tail] - Tail) :-
    !.
unravel_goal(object_var(X), _OVsRegs, [put_constant(X, rX(0)),
				       put_substitution(empty, rX(0)),
				       call_predicate(object_var, 1, _)|Tail] -
								      Tail) :-
    atom(X), !.
/*
 * Call the = predicate directly.  No optimisation.
unravel_goal(T1 = T2, OVsRegs, L1 - Tail) :-
    !,
    unravel_put_arg(T1, OVsRegs, A, L1 - L2, _Depth),
    unravel_get_arg(T2, OVsRegs, A, L2 - Tail).
 */
unravel_goal(Predicate, OVsRegs, ArgsUnifications - Tail) :-
    Predicate =.. [F|Args],
    length(Args, N),
    length(NewArgs, N),
    (built_in(F, N) ->
	Call = escape(F, N)
    ;
	Call = call_predicate(F, N, _EnvSize)
    ),
    unravel_put_args(Args, OVsRegs, NewArgs, ArgsUnifications - [Call|Tail]),
    allocate_x_registers(NewArgs).


/*----------------------------------------------------------------------------
unravel_put_args(Args, OVsRegs, NewArgs, UnravelledArgs) :-
----------------------------------------------------------------------------*/
unravel_put_args([], _OVsRegs, [], Unifications - Unifications).
unravel_put_args([Arg|Args], OVsRegs, [NewArg|NewArgs],
	ArgUnifications - Tail) :-
    unravel_put_arg(Arg, OVsRegs, NewArg, ArgUnifications - ArgsUnifications,
	_Depth),
    (nonvar(Arg), Arg =.. [(*), Ss, _], '$isa_substitution'(Ss) ->
	ArgsUnifications = ArgsUnifications1
    ;
	ArgsUnifications = [put_substitution(empty, NewArg)|ArgsUnifications1]
    ),
    unravel_put_args(Args, OVsRegs, NewArgs, ArgsUnifications1 - Tail).


/*----------------------------------------------------------------------------
unravel_put_arg(Arg, OVsRegs, NewArg, UnravelledArg, Depth) :-
    unravel_put_arg(f(t1, ..., tn))
	unravel_put_term([tn, ..., t1, f]).
----------------------------------------------------------------------------*/
unravel_put_arg(Arg, OVsRegs, NewArg, UnravelledArg, 0) :-
    ('$isa_unpersistent_var'(Arg); atomic(Arg)), !,
    unravel_put_constant_or_var(Arg, OVsRegs, NewArg, UnravelledArg).

unravel_put_arg(Arg, OVsRegs, NewArg, UnravelledArg, Depth) :-
    Arg =.. [F|Args],
    unravel_put_arg2(F, Args, OVsRegs, NewArg, UnravelledArg, Depth).

unravel_put_arg2((^), [X, T], OVsRegs, A, UnravelledArg-Tail, Depth) :-
    '$isa_object_var'(X),
    !,
    unravel_put_arg(T, OVsRegs, A2, THead-TTail, D),
    unravel_put_constant_or_var(X, OVsRegs, A1, ObjHead-ObjTail),
    (D == 0 ->
	UnravelledArg = ObjHead,
	ObjTail = THead,
	TTail = [put_quantifier(A1, A2, A)|Tail]
    ;
	UnravelledArg = THead,
	TTail = ObjHead,
	ObjTail = [put_quantifier(A1, A2, A)|Tail]
    ),
    Depth is D + 1.
unravel_put_arg2((*), [Ss, T], OVsRegs, A, UnravelledArg-Tail, Depth) :-
    '$isa_substitution'(Ss),
    !,
    unravel_put_arg(T, OVsRegs, A,
		    UnravelledArg-[put_substitution(eMPTY, A)|Tail2], D),
    unravel_subs(Ss, OVsRegs, A, Tail2-Tail),
    Depth is D + 1.
unravel_put_arg2(F, Args, OVsRegs, NewArg, UnravelledArg, Depth) :-
    reverse(Args, ArgsReversed),
    unravel_put_term(ArgsReversed, OVsRegs, F, NewArg, UnravelledArg, Depth).


/*----------------------------------------------------------------------------
unravel_put_term(Args, OVsRegs, F, NewArg, UnravelledArg, Depth) :-

    Args = [tn, .., t1], F = f

	put [| X, A |] ->
	    put(_, X, A).

	put [| c, A |] ->
	    put_constant(c, A).

	depth:
	    ld := depth(l)
	    rd := depth(r)
	    if ld > rd
	    then l, r, put_apply
		 d := rd + 1
	    else r, l, put_apply
		 d := ld + 1

	for l, r, put_apply

	    put [| f(t1, ..., tn), A |] ->
		put [| f(t1, ..., tn-1), A1 |],
		put [| tn, A2 |],
		put_apply(A1, A2, A).

	for r, l, put_apply

	    put [| f(t1, ..., tn), A |] ->
		put [| tn, A2 |],
		put [| f(t1, ..., tn-1), A1 |],
		put_apply(A1, A2, A).
----------------------------------------------------------------------------*/
unravel_put_term([], OVsRegs, F, A, UnravelledTerm, 0) :-
    unravel_put_constant_or_var(F, OVsRegs, A, UnravelledTerm).
unravel_put_term([Tn|Rest], OVsRegs, F, A, U1 - U2, D) :-
    unravel_put_arg(Tn, OVsRegs, A2, R1 - R3, RD),
    unravel_put_term(Rest, OVsRegs, F, A1, L1 - L2, LD),
    ((nonvar(Tn), Tn =.. [TF|TA], '$isa_substitution'(TF, TA, _, _)) ->
	R3 = [put_substitution_operator(A2, A2)|R2]
    ;
	R3 = R2
    ),
    (LD > RD ->
	U1 = L1,
	L2 = R1,
	R2 = [put_apply(A1, A2, A)|U2],
	D is RD + 1
    ;
	U1 = R1,
	R2 = L1,
	L2 = [put_apply(A1, A2, A)|U2],
	D is LD + 1
    ).

/*----------------------------------------------------------------------------
unravel_put_constant_or_var(Term, OVsRegs, NewTerm, UnravelledTerm) :-

	put [| X, A |] ->
	    put(_, X, A).

	put [| c, A |] ->
	    put_constant(c, A).
----------------------------------------------------------------------------*/
unravel_put_constant_or_var(T, OVsRegs, A,
	[put_object(_Type, TRegister, A)|Tail] - Tail) :-
    '$isa_object_var'(T), !,
    '$dereference_var2'(T, OVsRegs, TRegister).
unravel_put_constant_or_var(T, _OVsRegs, A, [put_constant(T, A)|Tail] - Tail):-
    atomic(T), !.
unravel_put_constant_or_var(T, _OVsRegs, A, [put(_Type, T, A)|Tail]-Tail).
