/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 *
 * tempalloc - Temporary register allocation routines.
 */

/*----------------------------------------------------------------------------
tempalloc(Instrs, RA) :-
    Allocate X registers to all of the temporary registers in RA.

    Instrs 	instructions used to implement a clause.
    RA 		registers alive after each instruction in Instrs. RA stands 
		for Registers Alive.

    This module allocates registers to the registers (variables) alive before
    each instruction (see the module alive to determine which registers were
    allocated and which were by default). This example shows the code for
    append:

    append([], X, X).
    append([X|Y], Z, [X|U]):- append(Y, Z, U).

	Instructions 			List of temporary variables alive
					after each instruction

    head(append / 3):-			[]
	get_constant([], rX(0))		[rX(0), rX(1), rX(2)]
	get(variable, rX(1), rX(1))	[rX(1), rX(2)]
	get(value, rX(1), rX(2))	[rX(2), rX(1)]
    head(append / 3):-			[]
	get_apply(rX(0), rX(3), rX(0))	[rX(0), rX(1), rX(2)]
	get_apply(rX(0), rX(4), rX(0))	[rX(1), rX(2), rX(0), rX(3)]
	get_constant(., rX(0))		[rX(1), rX(2), rX(3), rX(0), rX(4)]
	get(variable, rX(4), rX(4))	[rX(1), rX(2), rX(3), rX(4)]
	get(variable, rX(3), rX(3))	[rX(1), rX(2), rX(3), rX(4)]
	get(variable, rX(5), rX(1))	[rX(1), rX(2), rX(4), rX(3)]
	get_apply(rX(0), rX(1), rX(2))	[rX(2), rX(4), rX(3), rX(5)]
	get_apply(rX(0), rX(2), rX(0))	[rX(4), rX(3), rX(5), rX(0), rX(1)]
	get_constant(., rX(0))		[rX(4), rX(3), rX(5), rX(1), rX(0),
					rX(2)]
	get(value, rX(4), rX(2))	[rX(4), rX(3), rX(5), rX(1), rX(2)]
	get(variable, rX(6), rX(1))	[rX(4), rX(3), rX(5), rX(1)]
	put(value, rX(3), rX(0))	[rX(4), rX(3), rX(5), rX(6)]
	put(value, rX(5), rX(1))	[rX(4), rX(3), rX(5), rX(6), rX(0)]
	put(value, rX(6), rX(2))	[rX(4), rX(3), rX(5), rX(6), rX(0),
					rX(1)]
	call_predicate(append, 3, _WPHI)[rX(4), rX(3), rX(5), rX(6), rX(0),
					rX(1), rX(2)]

    Using the register allocations that happen by default (args to
    predicates in head and call_predicate) find all of the possible
    allocations that
    would make some instructions  noops (optimise them out). Allocate these
    if there are no clashes with registers in each alive list. Continue this
    process until the suggested register allocations are the same as ones
    that that failed previously or there are no further optimisations.

    Then for each instruction see if they can be optimised(further
    optimisations will become evident as more registers are allocated)
    otherwise try allocating any register from rX(0) that does not clash
    with registers alive in any of the instructions.

----------------------------------------------------------------------------*/
tempalloc(Instrs, RA) :-
    try_all_optimisations(Instrs, RA, []), 
    allocate_registers_in_instrs(Instrs, RA),
    !.
    %write_allocations(Instrs, RA).


/*----------------------------------------------------------------------------
try_all_optimisations(Instrs, RA, RegisterAllocations) :-
    See if there are any register allocations that make instructions in Instrs
    no-ops, that is, source and destination of get_variable and put_value
    instructions are the same.

    allocate the registers if there are no clashes
    then continue doing this process until either there are no more 
    optimisable instructions or the suggested register allocations are the
    same as the previous which also failed.
----------------------------------------------------------------------------*/
try_all_optimisations(Instrs, RA, RegisterAllocations) :-
    optimise_instrs(Instrs, RegisterAllocations2),
    (RegisterAllocations2 \== RegisterAllocations,
     RegisterAllocations2 \== [] ->
	allocate_registers(RegisterAllocations2, RA),
	try_all_optimisations(Instrs, RA, RegisterAllocations2)
    ;
	true
    ).


/*----------------------------------------------------------------------------
optimise_instrs(Instrs, RegisterAllocations) :-
    Collect all of the RegisterAllocations for optimising instructions, 
    ones that have resulted in no register movement by selecting register
    allocations that do not move any data (i.e. instantiating source to dest
    or vice versa).

    Instrs	instructions we are examining for optimisable instructions.
	    
    RegisterAllocations
		Register assigments (Var = rX(i)).
----------------------------------------------------------------------------*/
optimise_instrs([], []).
optimise_instrs([Instr|Instrs], RegisterAllocations2) :-
    optimise_instr(Instr, RegisterAllocation), !,
    optimise_instrs(Instrs, RegisterAllocations),
    insert2(RegisterAllocation, RegisterAllocations, RegisterAllocations2).
optimise_instrs([_Instr|Instrs], RegisterAllocations) :-
    optimise_instrs(Instrs, RegisterAllocations).


/*----------------------------------------------------------------------------
optimise_instr(Instr, RegisterAllocation) :-
    True, if Instr is an instruction which can be optimised (get_variable,
    put_value).
    
    If there is one variable and one temporary register in the instruction.
    then the instruction can be optimised by making the registers the same,
    so the instruction can be optimised out later (by peephole), since it is
    a no-op.

    Instr	The instruction.
    RegisterAllocation
		The optimising allocation Var = rX(i).
----------------------------------------------------------------------------*/
optimise_instr(get(variable, A, X), A = X) :-
    suitable_instantiation(A, X).
optimise_instr(get_object(variable, A, X), A = X) :-
    suitable_instantiation(A, X).
optimise_instr(put(value, A, X), A = X) :-
    suitable_instantiation(A, X).
optimise_instr(put_object(value, A, X), A = X) :-
    suitable_instantiation(A, X).


/*----------------------------------------------------------------------------
suitable_instantiation(A, X) :-
    A suitable instantiation is one of the following
	A		X

	Var 	<- 	rX(i)
	rX(i) 	-> 	Var

    i.e. there is one variable and one temporary register in the instruction.
----------------------------------------------------------------------------*/
suitable_instantiation(A, X) :-
    var(A),
    var(X), !, fail.
suitable_instantiation(A, X) :-
    var(A), !,
    x_register(X).
suitable_instantiation(A, X) :-
    x_register(A),
    var(X).

/*----------------------------------------------------------------------------
allocate_registers_in_instrs(Instrs, RA) :-
    Allocate registers to all variables in Instrs so that no clash occurs in
    RA.

    Instrs	instructions  implementing the clause.
    RA		registers alive after each instruction.

    See if the instruction can be optimised (into a no-op) by a register
    allocation, otherwise use any available registers for the variables
    in the instruction.
----------------------------------------------------------------------------*/
allocate_registers_in_instrs([], []).
allocate_registers_in_instrs([Instr|Instrs], [Register|RA]) :-
    (
	optimise_instr(Instr, RegisterAllocation),
	RegisterAllocation,
	\+ register_clashes([Register|RA])
    ;
	allocate_in_instruction(Instr, [Register|RA])
    ),
    allocate_registers_in_instrs(Instrs, RA).


/*----------------------------------------------------------------------------
allocate_in_instruction(Instr, RA) :-
    Allocate all of the variables in Instr with registers that do not
    clash with registers in RA.

    Instr 	The current instruction we are allocating registers to.
    
    RA		Registers alive after each instruction, with
		the list for the current instruction as the head.
----------------------------------------------------------------------------*/
allocate_in_instruction(get_constant(_C, X), RA) :-
    !,
    allocate_register2(X, RA).
allocate_in_instruction(get(_Type, A, X), RA) :-
    !,
    allocate_registers2([A, X], RA).
allocate_in_instruction(get_object(_Type, A, X), RA) :-
    !,
    allocate_registers2([A, X], RA).
allocate_in_instruction(get_apply(A1, A2, A), RA) :-
    !,
    allocate_registers2([A1, A2, A], RA).
allocate_in_instruction(get_quantifier(A1, A2, A), RA) :-
    !,
    allocate_registers2([A2, A1, A], RA).
allocate_in_instruction(put_constant(_C, X), RA) :-
    !,
    allocate_register2(X, RA).
allocate_in_instruction(put(_Type, A, X), RA) :-
    !,
    allocate_registers2([A, X], RA).
allocate_in_instruction(put_object(_Type, A, X), RA) :-
    !,
    allocate_registers2([A, X], RA).
allocate_in_instruction(put_apply(A1, A2, A), RA) :-
    !,
    allocate_registers2([A1, A2, A], RA).
allocate_in_instruction(put_quantifier(A1, A2, A), RA) :-
    !,
    allocate_registers2([A1, A2, A], RA).
allocate_in_instruction(put_substitution_operator(A, X), RA) :-
    !,
    allocate_registers2([A, X], RA).
allocate_in_instruction(put_parallel_substitution(_, A), RA) :-
    !,
    allocate_registers2([A], RA).
allocate_in_instruction(_Instr, _RA).


/*----------------------------------------------------------------------------
allocate_registers2(Variables, RA) :-
    Allocate all of the variables in the list Variables with a register 
    that does not clash with allocations already done in RA.

    Variables	variables/registers affected by the current instruction.
    
    RA		Registers alive after each instruction, with
		the list for the current instruction as the head.
----------------------------------------------------------------------------*/
allocate_registers2([], _RA).
allocate_registers2([Variable|Variables], RA) :-
    allocate_register2(Variable, RA),
    allocate_registers2(Variables, RA).

/*----------------------------------------------------------------------------
allocate_register2(Variable, RA) :-
    If the Variable has not already been allocated a register (is a variable)
	then allocate it a register starting from rX(0) until one is found
	that does not clash with other allocations.
----------------------------------------------------------------------------*/
allocate_register2(Variable, RA) :-
    var(Variable), !,
    try_allocate(Variable, 0, RA).
allocate_register2(_Variable, _RA).


/*----------------------------------------------------------------------------
try_allocate(Variable, RegNum, RA) :-
    try instantiating Variable = rX(RegNum) that is try a register allocation
    for a particular variable and see if it clashes with any of the 
    allocations done so far, in the rest of clause (RA).
    We only need to look at the rest of the clause since all allocations are
    done for all alive registers before this instruction.

    Variable	The variable we are trying to allocate the register to.
    RegNum	The number identifying the register we are trying to allocate.
    RA		Registers alive after each instruction.

    Try the allocation rX(RegNum) if it clashes continue to try
    RegNum +1, .... until a register is found that does not clash.
----------------------------------------------------------------------------*/
try_allocate(Variable, RegNum, RA) :-
    x_register(RegNum, Variable),
    \+ register_clashes(RA).
try_allocate(Variable, RegNum, RA) :-
    RegNum2 is RegNum + 1,
    try_allocate(Variable, RegNum2, RA).


/*----------------------------------------------------------------------------
allocate_registers(RegisterAllocations, RA).
    Try all of the register allocations suggested from optimising instructions
    any that clash with existing allocations, do not use.

    RegisterAllocations
		Unifications (Var = rX(i)), that allocate temporary
		registers to variables in the instructions. They were 
		suggested by looking for optimising instructions
		(no movement).
    
    RA		The variables (and hence their registers) that
		are alive after each instruction.
----------------------------------------------------------------------------*/
allocate_registers([], _RA).
allocate_registers([RegisterAllocation|RegisterAllocations], RA) :-
    RegisterAllocation,
    \+ register_clashes(RA),
    allocate_registers(RegisterAllocations, RA).
allocate_registers([_RegisterAllocation|RegisterAllocations], RA) :-
    allocate_registers(RegisterAllocations, RA).


/*----------------------------------------------------------------------------
register_clashes(RA) :-
    True, if there is any list in the list of registers with duplicate 
    elements. This corresponds to two variables having the same register
    allocated.

    RA		The registers alive after each instruction in the clause.
----------------------------------------------------------------------------*/
register_clashes([]) :- !, fail.
register_clashes([First|_Rest]) :-
    register_clashes2(First),!.
register_clashes([_First|Rest]) :-
    register_clashes(Rest).

register_clashes2([]) :- !,fail.
register_clashes2([First|Rest]) :-
    member2(First,Rest),!.
register_clashes2([_First|Rest]) :-
    register_clashes2(Rest).
