/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 *
 * permvars - Determine which variables are permanent in a procedure.
 */

/*----------------------------------------------------------------------------
permvars(Sentence, OVRegs, Permanents) :-
    Constructs a list (Permanents) of the permanent variables used in the
    clause Sentence.  OVRegs is a list of object variables in the clause,
    as determined by the call to "'$select_sub_class2'/3" in "compile_clause"
    (in the comp_procs module).

    In WAM 
    Temporary Variables are defined as:
	1st occurence in (head(where the first goal is included);
	structure; last goal) and not occur in more than one goal.
    Permanent Variables are
	non temporary variables.
    Unsafe Variables are
	permanent variables where the 1st occurence is not in the 
	(head; structure).

    In QuAM 

    Temporary Variables are
	variables that occur only in one goal (head and first are one goal).
    Permanent Variables are
	non temporary variables
    Unsafe Variables are
	occurences of permanent variables where there are no occurences
	after the call they occur in.


    Algorithm
    ---------

    Transform the clause:

	f(...) :- g1(...), g2(...), ..., gn(...).

    into a list of formulas, treating the head and the first goal as a 
    single formula:

	[(f(...) :- g1(...)), g2(...), ..., gn(...)].

    for each 'goal' (formula) collect the variables in it,
	if a variable in the goal has occured in a previous goal then
	    the add the variable to the permanent variables
	else
	    add the variable to the list of variables that occur in the 
	    current goal that have not occured in other previous goals.
    
	add the list of variables that have not been seen to the list of
	variables seen so far in the list of goals

    The list of permanent variables has now been collected. Now order 
    the list based on the occurence of the permanents starting from the
    end of the goals. This is so that permanent variables are allocated
    Y1, ..., Yn from the end of the clause, so that when the environment
    is trimmed to r permanents, the values from Yr+1, ..., Yn can be 
    discarded.
    
----------------------------------------------------------------------------*/
permvars((Head :- Body), OVsRegs, Permanents) :-
    !,
    permvars_formula(Head, Body, OVsRegs, Permanents).

permvars(_Head, _OVsRegs, []).

/*----------------------------------------------------------------------------
permvars_formula(Head, Body, Permanents) :-
----------------------------------------------------------------------------*/
permvars_formula(Head, Body, OVsRegs, OrderedPermanents) :-
    rhs_list(Body, Bodies),
    first_goal(Bodies, FirstFormula, RestBodies),
    vars_in_body([(Head :- FirstFormula)|RestBodies], _Vars, Permanents),
    permanents_in_body(RestBodies, OVsRegs, Permanents, OrderedPermanents),
    tail_ol(OrderedPermanents, []).


/*----------------------------------------------------------------------------
first_goal(Bodies, FirstFormula, RestBodies) :-
    Sets FirstFormula to the first goal in Bodies that will destroy all of
    the temporary registers, and RestBodies to the tail of Bodies after
    FirstFormula.  It skips built-in predicates, and calls to "!" and "=",
    since they do not destroy all of the temporary registers.
----------------------------------------------------------------------------*/
first_goal(Goals, FirstFormula, RestGoals) :-
    first_goal2(Goals, FirstFormulas, RestGoals),
    rhs_list(FirstFormula, FirstFormulas).
    

/*----------------------------------------------------------------------------
first_goal2(Goals, FirstFormulas, RestGoals) :-
----------------------------------------------------------------------------*/
first_goal2([], [], []).
first_goal2([Goal|Goals], [Goal|FirstFormulas], RestGoals) :-
    no_call_goal(Goal), !,
    first_goal2(Goals, FirstFormulas, RestGoals).
first_goal2([Goal|Goals], [Goal], Goals).

/*----------------------------------------------------------------------------
no_call_goal(Goal) :-
    True, if  Goal does not affect the temporary registers, that is,
    in the destructive way in a call.
----------------------------------------------------------------------------*/
no_call_goal(!).
/* no_call_goal(_A = _B). */
no_call_goal(Goal) :-
    Goal =.. [F|Args],
    length(Args, N),
    built_in(F, N).

/*----------------------------------------------------------------------------
vars_in_body(Formulas, Vars, Permanents) :-
    Extracts the permanent variables from a list of forumulas.

    Formulas	list of formulas
    Vars 	open list of variables that occur in the list of Formulas.
    Permanents	list of permanent variables in the list of Formulas,
		where a permanent variable is one that occurs in more 
		than one goal.
----------------------------------------------------------------------------*/
vars_in_body([], _Vars, _Permanents).
vars_in_body([Formula|Formulas], Vars, Permanents) :-
    Formula =.. [_F|Args],
    vars_in_args(Args, Vars, VarsInFormula, Permanents),
    append_ol(Vars, VarsInFormula),
    vars_in_body(Formulas, Vars, Permanents).

/*----------------------------------------------------------------------------
vars_in_term(Term, Vars, VarsInFormula, Permanents) :-

    Find the permanent variables in the term, Term.

    Vars	list of variables seen in previous terms in the clause
    VarsInFormula
		list of variables seen in the current formula (goal) that
		have not been seen previously
    Permanents	list of variables seen sofar that occur in more than one
		formula

    if the term, Term, is a variable then
	if the variable has been seen in a previous formula then
	    add the variable to the list of permanent variables(no duplicates)
	else
	    add the variable to the list of variables in the current formula
	    that have not occured previously (no duplicates)
    else
	find the permanent variables in the term
----------------------------------------------------------------------------*/
vars_in_term(Term, Vars, VarsInFormula, Permanents) :-
    '$isa_var'(Term), !,
    (member_ol2(Term, Vars) ->
	insert_ol2(Term, Permanents)
    ;
	insert_ol2(Term, VarsInFormula)
    ).
vars_in_term(Term, Vars, VarsInFormula, Permanents) :-
    Term =.. [_F|Args],
    vars_in_args(Args, Vars, VarsInFormula, Permanents).
    

/*----------------------------------------------------------------------------
vars_in_args(Args, Vars, VarsInFormula, Permanents) :-
    Collect all of the Permanent variables in the list of arguments, Args.
----------------------------------------------------------------------------*/
vars_in_args([], _Vars, _VarsInFormula, _Permanents).
vars_in_args([Arg|Args], Vars, VarsInFormula, Permanents) :-
    vars_in_term(Arg, Vars, VarsInFormula, Permanents),
    vars_in_args(Args, Vars, VarsInFormula, Permanents).
    
/*----------------------------------------------------------------------------
permanents_in_body(Formulas, Permanents, OrderedPermanents) :-
    OrderedPermanents is the list of permanent variables occuring in the 
    clause, represented by the list of formulas,  Formulas, recorded by their
    occurence in the from the end of the clause (list).

    Formulas	the list of formulas representing the clause
		[(f(...) :- g1(...)), g2(...), ..., gn(...)]
    
    Permanents	the list of permanent variables in the clause, Formulas.
    OrderedPermanents
		list of permanent variables in the clause, ordered on
		their first occurence from the end of the clause, that is,
		their last occurence in the clause. Where ordering within
		a goal does not matter, so we take the method chosen by 
		Warren, of ordering from left to right in the goal.
----------------------------------------------------------------------------*/
permanents_in_body([], _OVsRegs, _Permanents, _OrderedPermanents).
permanents_in_body([Formula|Formulas], OVsRegs, Permanents, OrderedPermanents) :-
    permanents_in_body(Formulas, OVsRegs, Permanents, OrderedPermanents),
    Formula =.. [_F|Args],
    permanents_in_args(Args, OVsRegs, Permanents, OrderedPermanents).

/*----------------------------------------------------------------------------
permanents_in_term(Term, Permanents, OrderedPermanents) :-
    If the term, Term, is a variable then
	if the variable is a permanent then
	    insert it at the end of the list if it hashas not been seen before
    else
	order the permanents in the term
----------------------------------------------------------------------------*/
permanents_in_term(Term, _OVsRegs, Permanents, OrderedPermanents) :-
    '$isa_meta_var'(Term), !,
    (member_ol2(Term, Permanents) -> insert_ol2(Term, OrderedPermanents)
    ;
				     true).
permanents_in_term(Term, OVsRegs, Permanents, OrderedPermanents) :-
    '$isa_object_var'(Term), !,
    (member_ol2(Term, Permanents) -> '$dereference_var2'(Term, OVsRegs, Register),
				     insert_ol2(Register, OrderedPermanents)
    ;
				     true).
permanents_in_term(Term, OVsRegs, Permanents, OrderedPermanents) :-
    Term =.. [_F|Args],
    permanents_in_args(Args, OVsRegs, Permanents, OrderedPermanents).

/*----------------------------------------------------------------------------
permanents_in_args(Args, Permanents, OrderedPermanents) :-
    order the permanent variables in the arguments.
----------------------------------------------------------------------------*/
permanents_in_args([], _OVsRegs, _Permanents, _OrderedPermanents).
permanents_in_args([Arg|Args], OVsRegs, Permanents, OrderedPermanents) :-
    permanents_in_term(Arg, OVsRegs, Permanents, OrderedPermanents),
    permanents_in_args(Args, OVsRegs, Permanents, OrderedPermanents).
