/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 *
 * ol - Library routines for manipulating open lists.
 */

/*----------------------------------------------------------------------------
insert_ol(X, Xs) :-
    append X to the end of the open list, Xs, iff Xs does not already 
    contain X.

    or true, iff the open list Xs contains exactly one X.

    True, always.
    Fail on redo.

    Gives the first element of a closed list.
----------------------------------------------------------------------------*/
insert_ol(X, [X|_Xs]) :- !.
insert_ol(X, [_|Xs]) :-
    insert_ol(X, Xs).

/*----------------------------------------------------------------------------
lookup_ol(X, Xs, N) :-
    append X to the end of the open list, Xs, iff Xs does not already 
    contain X. N is X's position in Xs. Where N is a C index i.e. starts
    from 0.

    or true, iff the open list Xs contains exactly one X.

    True, always.
    Fail on redo.

    Gives the first element of a closed list.
----------------------------------------------------------------------------*/
lookup_ol(X, Xs, N) :-
    lookup_ol(X, Xs, 0, N).
lookup_ol(X, [X|_Xs], N, N) :- !.
lookup_ol(X, [_|Xs], N, N2) :-
    M is N + 1,
    lookup_ol(X, Xs, M, N2).

/*----------------------------------------------------------------------------
insert_ol2(X, Xs) :-
    append X to the end of the open list, Xs, iff Xs does not already 
    contain X (==).

    or true, iff the open list Xs contains exactly one X.

    True, always.
    Fail on redo.

    Gives the first element of a closed list.
----------------------------------------------------------------------------*/
insert_ol2(X, Xs) :-  var(Xs), !, Xs = [X|_].
insert_ol2(X, [Y|_Xs]) :-  X == Y, !.
insert_ol2(X, [_Y|Xs]) :-
    insert_ol2(X, Xs).


/*----------------------------------------------------------------------------
postpend_ol(X, Xs) :- 
    Append X to the end of the open list Xs.

    or True, iff the open list Xs contains at least one X.

    True, always.
    Fail on redo.
----------------------------------------------------------------------------*/
postpend_ol(X, Xs) :- 
    var(Xs), !,
    Xs = [X|_].
postpend_ol(X, [_|Xs]) :-
    postpend_ol(X, Xs).
    

/*----------------------------------------------------------------------------
append_ol(Xs, Ys) :- 
    Instantiate the end of the openlist Xs, to the openlist Ys.

    Note: This is destructive, the end of the openlist Xs is now the end of 
    the openlist Ys.

    Fail on redo.
----------------------------------------------------------------------------*/
append_ol(Xs, Ys) :- 
    var(Xs), !,
    Xs = Ys.
append_ol([_X|Xs], Ys) :-
    append_ol(Xs, Ys).
    

/*----------------------------------------------------------------------------
tail_ol(Ys, Tail) :-
    Tail is the tail (last element in the open list Ys).
----------------------------------------------------------------------------*/
tail_ol(Y, Tail) :-
    var(Y), !,
    Tail = Y.
tail_ol([_Y|Ys], Tail) :-
    tail_ol(Ys, Tail).

/*----------------------------------------------------------------------------
member_ol(X, Xs) :-
    True, iff X is an element of Xs.

    Generates elements of the open list Xs from left to right until reach end
    of list, then fails.

    Works for closed lists as well.
----------------------------------------------------------------------------*/
member_ol(_X, Xs) :- 
    var(Xs), !, fail.
member_ol(X, [X|_Xs]).
member_ol(X, [_|Xs]) :-
    member_ol(X, Xs).

/*----------------------------------------------------------------------------
member_ol2(X, Xs) :-
    True, iff X is an element of Xs.

    Finds the members that are == to X.
    Works for closed lists as well.
----------------------------------------------------------------------------*/
member_ol2(_X, Xs) :- 
    var(Xs), !, fail.
member_ol2(X, [Y|_Xs]) :- X == Y.
member_ol2(X, [_|Xs]) :-
    member_ol2(X, Xs).
    


/*----------------------------------------------------------------------------
split_ol(L, H, T) :- 
    True, if H and T are the head and tail of the openlist L.

    Fail on redo.

    Works for closed lists as well.
----------------------------------------------------------------------------*/
split_ol(L, H, T) :- 
    nonvar(L),
    L = [H|T].

/*----------------------------------------------------------------------------
empty_ol(L) :-
    True, iff L is an empty open list.

    Fail on redo.
----------------------------------------------------------------------------*/
empty_ol(L) :-
    var(L).
