/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 *
 * alive - Process QuAM assembly to determine the "live" temporary registers.
 */

/*----------------------------------------------------------------------------
alive(Instrs, RA) :-
    RA is the list of lists of temporary registers alive before each 
    instruction in the list of instructions Instrs.  For the purposes of
    this module, a temporary register may either be an X register, or a
    Prolog variable.
    e.g. for the append predicate

    append([], X, X).
    append([X|Y], Z, [X|U]):- append(Y, Z, U).

    alive would determine which temporary registers are alive before each
    instruction. (see the tempalloc module for allocation of actual registers
    to the variables in RA)

	Instructions			Registers Alive

    head(append / 3):-			[]
	get_constant([], rX(0))		[rX(0), rX(1), rX(2)]
	get(variable, _WMTW, rX(1))	[rX(1), rX(2)]
	get(value, _WMTW, rX(2))	[rX(2), _WMTW]

    head(append / 3):-			[]
	get_apply(_WOYJ, _WOYK, rX(0))	[rX(0), rX(1), rX(2)]
	get_apply(_WOYU, _WOYV, _WOYJ)	[rX(1), rX(2), _WOYJ, _WOYK]
	get_constant(., _WOYU)		[rX(1), rX(2), _WOYK, _WOYU, _WOYV]
	get(variable, _WNFY, _WOYV)	[rX(1), rX(2), _WOYK, _WOYV]
	get(variable, _WNGT, _WOYK)	[rX(1), rX(2), _WOYK, _WNFY]
	get(variable, _WNHT, rX(1))	[rX(1), rX(2), _WNFY, _WNGT]
	get_apply(_WPBN, _WPBO, rX(2))	[rX(2), _WNFY, _WNGT, _WNHT]
	get_apply(_WPBY, _WPBZ, _WPBN)	[_WNFY, _WNGT, _WNHT, _WPBN, _WPBO]
	get_constant(., _WPBY)		[_WNFY, _WNGT, _WNHT, _WPBO, _WPBY,
					_WPBZ]
	get(value, _WNFY, _WPBZ)	[_WNFY, _WNGT, _WNHT, _WPBO, _WPBZ]
	get(variable, _WNJG, _WPBO)	[_WNFY, _WNGT, _WNHT, _WPBO]
	put(value, _WNGT, rX(0))	[_WNFY, _WNGT, _WNHT, _WNJG]
	put(value, _WNHT, rX(1))	[_WNFY, _WNGT, _WNHT, _WNJG, rX(0)]
	put(value, _WNJG, rX(2))	[_WNFY, _WNGT, _WNHT, _WNJG, rX(0),
					rX(1)]
	call_predicate(append, 3, _WPHI)[_WNFY, _WNGT, _WNHT, _WNJG, rX(0),
					rX(1), rX(2)]
----------------------------------------------------------------------------*/
alive(Instrs, RA) :-
    alive_instructions(Instrs, [], RA).
    %write_allocations(Instrs, RA).


/*----------------------------------------------------------------------------
alive_instructions(Instrs, RAInstr, RA) :-
    Find out which temporary registers are alive at any time in the 
    instructions.

    RAInstr 	list of registers alive before the instruction before the
		head of the list Instrs.
    RA 		list of lists of temporary registers alive before each 
   		instruction.

    Firstly remove any registers that are dead from the list of registers that
    were alive before the previous instruction. Dead registers are ones where
    we no longer care about the contents of the register (it is available for
    reuse). These are where the variable does not occur again in the rest
    of the clause.
	
    Remove/Add temporary registers from/to the list of alive regs, depending 
    on the instruction.

----------------------------------------------------------------------------*/
alive_instructions([], _RAInstr, []).
alive_instructions([Instr|Instrs], RAInstr, [RAInstr|RA]) :-
    Instr =.. [_F|_],
    remove_the_dead(RAInstr, [Instr|Instrs], RAInstr2),
    alive_instruction(Instr, Instrs, RAInstr2, RAInstr3),
    alive_instructions(Instrs, RAInstr3, RA).


/*----------------------------------------------------------------------------
alive_instruction(Instr, Instrs, RAInstr, RAInstr2) :-
    Remove/Add temporary registers from/to the list of alive regs, depending 
    on the instruction.

    Instr 	is the instruction we are currently examining to see
		its affect on the alive registers.
    Instrs 	are the instructions that come after Instr.
    RAInstr 	list of registers alive before execution of Instr.
    RAInstr2 	list of registers alive after execution of Instr.

    Instr

    fail	No registers are alive (it does not matter anyway (backtrack))
    cut		No affect on temporary registers
    call_predicate f/n
		All temporary registers dead
    escape f/n
		No affect on temporary registers
    determine_property
		No affect on temporary registers
    set_object_property
		No affect on temporary registers
    head(f/n)
		Allocate n X registers for the procedure arguments.  These
		will be the only ones alive before the instruction.
    get_constant c Source
		Source is dead
    get _ Dest Source
		Source dead and Dest alive
    get_object _ Dest Source
		Source dead and Dest alive
    get_apply l r Source
		Source dead and l and r alive
    get_quantifier l r Source
		Source dead and l and r alive
    put_constant c Dest
		Dest is alive
    put _ Source Dest
		Unify Source and Dest if Source is not used in any following
		instruction, otherwise Source dead and Dest alive
    put_object _ Source Dest
		Unify Source and Dest if Source is not used in any following
		instruction, otherwise Source dead and Dest alive
    put_apply l r Dest
		l r dead and Dest alive
    put_quantifier l r Dest
		l r dead and Dest alive
    put_substitution_operator Source Dest
		Source is dead if this is the last occurrence and Dest is
		alive always
    put_substitution Source Dest
		Source is dead if this is the last occurrence and Dest is
		alive always
    put_parallel_substitution _ Dest
		Dest alive
    put_parallel_substitution_pair l r
		l and r are dead if they are no longer required
----------------------------------------------------------------------------*/
alive_instruction(fail, _Instrs, _RAInstr, []).
alive_instruction(cut, _Instrs, RAInstr, RAInstr).
alive_instruction(escape(_F, _N), _Instrs, RAInstr, RAInstr).
alive_instruction(determine_property(_), _Instrs, RAInstr, RAInstr).
alive_instruction(set_object_property(_), _Instrs, RAInstr, RAInstr).
alive_instruction(head(_F/N), _Instrs, [], RA) :-
    length(RA, N),
    allocate_x_registers(RA).
alive_instruction(call_predicate(_F, _N, _EnvSize), _Instrs, _RAInstr, []).
alive_instruction(get_constant(_C, A), _Instrs, RAInstr, RAInstr2) :-
    delete_register(A, RAInstr, RAInstr2).
alive_instruction(get(_Type, A1, A), _Instrs, RAInstr, RAInstr3) :-
    delete_register(A, RAInstr, RAInstr2),
    insert_register(A1, RAInstr2, RAInstr3).
alive_instruction(get_object(_Type, A1, A), _Instrs, RAInstr, RAInstr3) :-
    delete_register(A, RAInstr, RAInstr2),
    insert_register(A1, RAInstr2, RAInstr3).
alive_instruction(get_apply(A1, A2, A), _Instrs, RAInstr, RAInstr4) :-
    delete_register(A, RAInstr, RAInstr2),
    insert_register(A1, RAInstr2, RAInstr3),
    insert_register(A2, RAInstr3, RAInstr4).
alive_instruction(get_quantifier(A1, A2, A), _Instrs, RAInstr, RAInstr4) :-
    delete_register(A, RAInstr, RAInstr2),
    insert_register(A1, RAInstr2, RAInstr3),
    insert_register(A2, RAInstr3, RAInstr4).
alive_instruction(put_constant(_C, A), _Instrs, RAInstr, RAInstr2) :-
    insert_register(A, RAInstr, RAInstr2).
alive_instruction(put(variable, A1, A), Instrs, RAInstr, RAInstr3) :-
    !,
    assign_if_last_occur(A1, A, Instrs),
    insert_register(A1, RAInstr, RAInstr2),
    insert_register(A, RAInstr2, RAInstr3).
alive_instruction(put_object(variable, A1, A), Instrs, RAInstr, RAInstr3) :-
    !,
    assign_if_last_occur(A1, A, Instrs),
    insert_register(A1, RAInstr, RAInstr2),
    insert_register(A, RAInstr2, RAInstr3).
alive_instruction(put(_Type, A1, A), Instrs, RAInstr, RAInstr3) :-
    delete_register_last_occur(A1, Instrs, RAInstr, RAInstr2),
    insert_register(A, RAInstr2, RAInstr3).
alive_instruction(put_object(_Type, A1, A), Instrs, RAInstr, RAInstr3) :-
    delete_register_last_occur(A1, Instrs, RAInstr, RAInstr2),
    insert_register(A, RAInstr2, RAInstr3).
alive_instruction(put_apply(A1, A2, A), Instrs, RAInstr, RAInstr4) :-
    delete_register_last_occur(A1, Instrs, RAInstr, RAInstr2),
    delete_register_last_occur(A2, Instrs, RAInstr2, RAInstr3),
    insert_register(A, RAInstr3, RAInstr4).
alive_instruction(put_quantifier(A1, A2, A), Instrs, RAInstr, RAInstr4) :-
    delete_register_last_occur(A1, Instrs, RAInstr, RAInstr2),
    delete_register_last_occur(A2, Instrs, RAInstr2, RAInstr3),
    insert_register(A, RAInstr3, RAInstr4).
alive_instruction(put_substitution_operator(A1, A), Instrs, RAInstr, RAInstr3):-
    delete_register_last_occur(A1, Instrs, RAInstr, RAInstr2),
    insert_register(A, RAInstr2, RAInstr3).
alive_instruction(put_substitution(A1, A), _, RAInstr, RAInstr1) :-
    delete_register_last_occur(A1, _Instrs, RAInstr, _RAInstr2),
    insert_register(A, RAInstr, RAInstr1).
alive_instruction(put_parallel_substitution(_, A), _, RAInstr, RAInstr1) :-
    insert_register(A, RAInstr, RAInstr1).
alive_instruction(put_parallel_substitution_pair(A1, A2), Instrs, RAInstr,
		  RAInstr3) :-
    delete_register_last_occur(A1, Instrs, RAInstr, RAInstr2),
    delete_register_last_occur(A2, Instrs, RAInstr2, RAInstr3).

/*----------------------------------------------------------------------------
assign_if_last_occur(Source, Destination, Instrs) :-
    Assign Source to register Destination if Source does not occur in any
    further instruction in the list of instructions Instrs.
----------------------------------------------------------------------------*/
assign_if_last_occur(A1, A, Instrs) :-
    (\+ register_occurs_in_instructions(A1, Instrs) ->
	A1 = A
    ;
	true
    ).

/*----------------------------------------------------------------------------
insert_register(A, Registers, Register2) :-
    If A is a register, and is not already in the list Registers, then
    insert it to create the list Registers2, otherwise unify Registers
    and Registers2.
----------------------------------------------------------------------------*/
insert_register(A, Registers, Registers2) :-
    (once((var(A) ; x_register(A))) ->
	insert2(A, Registers, Registers2) 
    ;
	Registers2 = Registers
    ).

/*----------------------------------------------------------------------------
insert2(A, Bs, Cs) :-
    Cs is the Union of Bs and the element A.
----------------------------------------------------------------------------*/
insert2(A, [], [A]).
insert2(A, [B|Bs], [B|Bs]) :-
    A == B, !.
insert2(A, [B|Bs], [B|Cs]) :-
    insert2(A, Bs, Cs).

/*----------------------------------------------------------------------------
delete_register(A, Registers, Registers2) :-
    If A is a register that occurs in the list Registers, then delete it
    to produce the list Registers2, otherwise unify Registers and Registers2.
----------------------------------------------------------------------------*/
delete_register(A, Registers, Registers2) :-
    (once((var(A) ; x_register(A))) ->
	delete2(A, Registers, Registers2)
    ;
	Registers2 = Registers
    ).



/*----------------------------------------------------------------------------
delete_register_last_occur(A, Instrs, Registers, Registers2) :-
    If A is a register and it doesn't occur anywhere in the list Instrs
    up to and including the first call_predicate instruction (if present),
    then delete it from Registers to produce Registers2, otherwise unify
    Registers and Registers2.
----------------------------------------------------------------------------*/
delete_register_last_occur(A, Instrs, Registers, Registers2) :-
    (once((var(A) ; x_register(A))) ->
	(append(Front, [call_predicate(F, N, EnvSize)|_Back], Instrs) ->
	    append(Front, [call_predicate(F, N, EnvSize)], Instrs2)
	;
	    Instrs2 = Instrs
	),
	(\+ register_occurs_in_instructions(A, Instrs2) ->
	    delete2(A, Registers, Registers2)
	;
	    Registers2 = Registers
	)
    ;
	Registers2 = Registers
    ).


/*----------------------------------------------------------------------------
delete2(A, Bs, Cs) :-
    Cs is the result of removing A from Bs.
----------------------------------------------------------------------------*/
delete2(A, [B|Bs], Bs) :-
    A == B, !.
delete2(A, [B|Bs], [B|Cs]) :-
    delete2(A, Bs, Cs).


/*----------------------------------------------------------------------------
remove_the_dead(RAInstr, Instrs, RAInstr2) :-
    Alive registers are dead if they are not used in any of the 
    instructions up to the next call.

    RAInstr2 is the result of removing any of these registers that
    are no longer used up till the next call, from RAInstr.
----------------------------------------------------------------------------*/
remove_the_dead(RAInstr, Instrs, RAInstr2) :-
    (append(Front, [call_predicate(F, N, EnvSize)|_Back], Instrs) ->
	append(Front, [call_predicate(F, N, EnvSize)], Instrs2)
    ;
	Instrs2 = Instrs
    ),
    remove_the_dead2(RAInstr, Instrs2, RAInstr2).

/*----------------------------------------------------------------------------
remove_the_dead2(RAInstr, Instrs, RAInstr2) :-
    RAInstr2 is the set of registers with the dead registers removed
    from RAInstr. Where dead registers are ones whose value
    no longer interests us, that is, a value that is no longer referenced 
    (needed).

    if the register does not occur in some instruction in Instrs then
	it is dead, do remove it
    else
	it is still alive for the  current instruction.
----------------------------------------------------------------------------*/
remove_the_dead2([], _Instrs, []).
remove_the_dead2([A|RAInstr], Instrs, RAInstr2) :-
    \+ register_occurs_in_instructions(A, Instrs), !,
    remove_the_dead2(RAInstr, Instrs, RAInstr2).
remove_the_dead2([A|RAInstr], Instrs, [A|RAInstr2]) :-
    remove_the_dead2(RAInstr, Instrs, RAInstr2).
	

/*----------------------------------------------------------------------------
register_occurs_in_instructions(A, Instrs) :-
    True, if A occurs in some instruction in the list Instrs (where n 
    registers are input to a call instruction).
----------------------------------------------------------------------------*/
register_occurs_in_instructions(A, [Instr|_Instrs]) :-
    register_occurs_in_instruction(A, Instr).
register_occurs_in_instructions(A, [_Instr|Instrs]) :-
    register_occurs_in_instructions(A, Instrs).


/*----------------------------------------------------------------------------
register_occurs_in_instruction(A, Instr) :-
    True if the register A occurs in Instr.
----------------------------------------------------------------------------*/
register_occurs_in_instruction(A, call_predicate(_F, N, _EnvSize)) :-
    !,
    nonvar(A),
    x_register(M, A),
    M < N.
register_occurs_in_instruction(A, escape(_F, N)) :-
    !,
    nonvar(A),
    x_register(M, A),
    M < N.
register_occurs_in_instruction(A, Instr) :-
    Instr =.. [_InstrClass|Args],
    member2(A, Args).
