/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

#include "errors.h"
#include "instructions.h"
#include "opcodes.h"
#include "pc.h"
#include "query_code.h"

global	code	*query_code;
global	natural	query_code_size = DEFAULT_QUERY_CODE_SIZE;
global	code	*top_of_query_code;
global	natural	number_of_file_query_clauses;


/*----------------------------------------------------------------------------
    Leave space for these three parameters to be added later.
    query_code
	+-------+
	|string#|	-> '$query'
	+-------+
	| arity |	0
	+-------+
	| size  |
	+-------+	<- set top_of_query_code to here.
	| query |
	| code  |
	  ...
----------------------------------------------------------------------------*/

extern size_t fread (void *, size_t, size_t, FILE *);

global 	void
reinitialise_query_code(void)
{
	top_of_query_code = query_code;

	/* setup query procedure */
	pc = top_of_query_code;
	SkipOffset();
	SkipNumber();
	SkipConstant();
	top_of_query_code = pc;
	number_of_file_query_clauses = 0;
}

/*----------------------------------------------------------------------------
    add the query code of an object file (in one cluster) to the query 
    code of the whole executable. This simply involves appending to the
    end.

    object_file	The file ptr to the object file from which to read the
		query_code.
    filename	The name of the object file.

    each query of the form
	?- Query.
    is translated into 
	$query :- once(Query), fail.

    if there are no queries in a file then the clause for $query is added
	$query :- fail.

    The linker then adds further choice points by wrapping 
	try_me_else, retry_me_else ..., trust_me_else_fail to the combined
	definitions for $query/0.

    $query/0:
		try_me_else(0, 1)
		<code for file 1's queries>
    1:		retry_me_else(2)
		<code for file 2's queries>
    2:		...

    n:		retry_me_else(n+1)
		<code for file n's queries>
    n+1:	trust_me_else_fail
		proceed

----------------------------------------------------------------------------*/
global	void
add_query_code(FILE *object_file, char *filename)
{
	natural	obj_query_code_size;

	if (fread((char *)&obj_query_code_size, SizeOfConstant, 1,
		object_file) != 1)
		fatal("invalid file format for object file %s", filename);
	if (obj_query_code_size > 0)
	{
		if (number_of_file_query_clauses == 0)
		{
			PutInstruction(TRY_ME_ELSE);
			PutNumber(0);
			PutOffset(obj_query_code_size);
		}
		else
		{
			PutInstruction(RETRY_ME_ELSE);
			PutOffset(obj_query_code_size);
		}

		if ((pc + obj_query_code_size) > (query_code +
		    Kwords_to_chars(query_code_size)))
			fatal("Out of space in query code area %d K",
			      query_code_size);

		if (fread((char *)pc, (int)obj_query_code_size, 1,
		    object_file) != 1)
			fatal("invalid file format for object file %s",
				filename);
		pc += obj_query_code_size;
		number_of_file_query_clauses++;
	}
}
