
/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

#include <string.h>

#include "cells.h"
#include "name_table.h"
#include "errors.h"
#include "string_table.h"

global  NAME	*name_table;
global	natural	name_table_size = DEFAULT_NAME_TABLE_SIZE;


/*----------------------------------------------------------------------------
initialise_name_table()
    Allocate memory for the name table and initialise its contents to NULL.
----------------------------------------------------------------------------*/
global	void
initialise_name_table(void)
{
	int	i;

	if ((name_table = (NAME *) malloc(name_table_size * 
					    sizeof(NAME))) == NULL)
		fatal("not enough memory for name table %d",
		    name_table_size);
	for (i = 0; i < name_table_size; i++)
	{
	     NameT(name_table[i]) = NULL;
	     ValueOfName(name_table[i]) = NULL;
        }
}

/*----------------------------------------------------------------------------
lookup_name_table(s, name_table_tag) 

    If name (s) is in name table with its corresponding tag
        return its hash value from name table
    else 
	return a hash value to name table where name (s) can be put 
----------------------------------------------------------------------------*/
global natural
lookup_name_table(char *s, int name_table_tag)
{
reg	natural	incr;
reg     natural i;
	

	incr = 1;

	for (i = hash_string(s, name_table_size); NameT(name_table[i]) != NULL;
	     i = (i+incr)%name_table_size, incr += 2)
	{
		if ((NameTableTag(name_table[i]) == name_table_tag) && 
		    (strcmp(s, String2(NameOffset(name_table[i]))) == 0))
                { 
			return(i);
                }
		if (incr%name_table_size == 0)
			fatal("Out of space in name hash table %d",
			    name_table_size);
	}
	return(i);
}
/*----------------------------------------------------------------------------
lookup_name_table_offset(s, name_table_tag) 

     While the position the string is hashed into is full do
         if the string in the hash position is the same as the string s then
             return that offset into the string table
         else
	     if all of the possible hash positions have been tried then
                 abort
     end-while
     place the string in that hash position in name table 

----------------------------------------------------------------------------*/
global offset
lookup_name_table_offset(char *s, int name_table_tag)
{
reg	natural	incr;
reg     natural i;
	

	incr = 1;


	for (i = hash_string(s, name_table_size); NameT(name_table[i]) != NULL;
	     i = (i+incr)%name_table_size, incr += 2)
	{
		if ((NameTableTag(name_table[i]) == name_table_tag) && 
		    (strcmp(s, String2(NameOffset(name_table[i]))) == 0))
		    return(NameOffset(name_table[i]));
		    
		if (incr%name_table_size == 0)
			fatal("Out of space in name hash table %d",
			    name_table_size);
	}
	NameT(name_table[i]) = name_table_tag|(s - string_table);
	return(s - string_table);
}
/*----------------------------------------------------------------------------
hash_string(s, table_size)
    returns a hash value for the string s.
----------------------------------------------------------------------------*/
global natural
hash_string(char *s, unsigned int table_size)
{
reg	natural	n;

	n = 0;
	while (*s) {
		n *= 167;
		n ^= *s++;
	}
	return (n%table_size);
}
