/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

#include "code_area.h"
#include "errors.h"
#include "executable.h"
#include "instructions.h"
#include "name_table.h"
#include "pred_table.h"
#include "query_code.h"
#include "string_table.h"

/*----------------------------------------------------------------------------
Format of an executable file
----------------------------

	+-----------------------+
	| code_area_size	|
	+-----------------------+
	| query_code_size	|
	+-----------------------+
	| string_table_size	|
	+-----------------------+
	| name_table_size	|
	+-----------------------+
	| predicate_table_size	|
	+-----------------------+
	| code_area		|
	+-----------------------+
	| query_code		|
	+-----------------------+
	| string_table		| list of strings
	+-----------------------+
	| name_table		| hash table of offsets into string_table
	+-----------------------+
	| predicate_table	| hash table of offsets into the code_area
	+-----------------------+

	code_area
	+-------+
	|string#|			}
	+-------+			}
	| arity	|			}
	+-------+			}
	| size	|			} repeat for each predicate
	+-------+			}
	|  c	|	}		}
	|  o	|	} size bytes	}
	|  d	|	}		}
	|  e	|	}		}
	+-------+

	query_code
	+-------+
	|string#| for '$query'	 (entry in the hash table is the size of
	+-------+		  the code table, not including the
	| 0	|		  query_code)
	+-------+
	| size	|
	+-------+
	| Q1	|
	| ...	|
	| Qn	|
	|proceed|
	+-------+
	
----------------------------------------------------------------------------*/
global char *executable_filename = DEFAULT_EXECUTABLE_FILENAME;


extern size_t fwrite (const void *, size_t, size_t, FILE *);
extern int fclose (FILE *);
extern size_t fread (void *, size_t, size_t, FILE *);

global	void
dump_executable(void)
{
	FILE	*executable_file;
	natural	exe_code_area_size;
	natural	exe_query_code_size;
	natural	exe_string_table_size;

	if ((executable_file = fopen(executable_filename, "w")) == NULL)
		fatal("Could not open %s", executable_filename);
	exe_query_code_size = top_of_query_code - query_code;
	exe_code_area_size = top_of_code_area - code_area;
	exe_string_table_size = top_of_string_table - string_table;

	if (fwrite((char *)&exe_code_area_size, SizeOfConstant, 1,
		executable_file) != 1 ||
	    fwrite((char *)&exe_query_code_size, SizeOfConstant, 1,
		executable_file) != 1 ||
	    fwrite((char *)&exe_string_table_size, SizeOfConstant, 1,
		executable_file) != 1 ||
	    fwrite((char *)&name_table_size, SizeOfConstant, 1,
		executable_file) != 1 ||
	    fwrite((char *)&predicate_table_size, SizeOfConstant, 1,
		executable_file) != 1 ||
	    fwrite((char *)code_area, (int)exe_code_area_size, 1,
		executable_file) != 1 ||
	    fwrite((char *)query_code, (int)exe_query_code_size, 1,
		executable_file) != 1 ||
	    fwrite((char *)string_table, (int)exe_string_table_size, 1,
		executable_file) != 1 ||
	    fwrite((char *)name_table, (int)name_table_size*sizeof(NAME), 1,
		executable_file) != 1 ||
	    fwrite((char *)predicate_table,
		(int)(predicate_table_size*sizeof(offset)), 1,
		executable_file) != 1 )
	    	fatal("writing to file failed %s", executable_filename);

	fclose(executable_file);
}

/*----------------------------------------------------------------------------
    read in the sizes of each of the areas in the executable (see
    dump_executable for format). Take the maximum between default values and
    values in the executable. The code and string table size are flexible,
    however, name and predicate table size is constant (relative to setting)
    in linker.
----------------------------------------------------------------------------*/
global	void
load_executable(char *filename)
{
	int	exe_code_area_size;
	int	exe_query_code_size;
	int	exe_string_table_size;
	FILE	*executable;

	if ((executable = fopen(filename, "r")) == NULL)
		fatal("Could not open %s", filename);

	if (fread((char *)&exe_code_area_size, SizeOfConstant, 1, executable)
		!= 1 )  
		fatal(" 1.1 invalid format of executable image %s", filename);
	if (fread((char *)&exe_query_code_size, SizeOfConstant, 1, executable)
		!= 1 )  
                fatal(" 1.2 invalid format of executable image %s", filename);
	if (fread((char *)&exe_string_table_size, SizeOfConstant, 1,
		executable) != 1)   
		fatal(" 1.3 invalid format of executable image %s", filename);
	if (fread((char *)&name_table_size, SizeOfConstant, 1, executable)
		!= 1 )  
		fatal(" 1.4 invalid format of executable image %s", filename);
	if (fread((char *)&predicate_table_size, SizeOfConstant, 1,
		executable) != 1)
		fatal(" 1.5 invalid format of executable image %s", filename);

	code_area_size = Max(code_area_size,
			     Chars_to_Kwords(exe_code_area_size));
	query_code_size = Max(query_code_size,
			     Chars_to_Kwords(exe_query_code_size));
	string_table_size = Max(string_table_size,
			        Chars_to_Kwords(exe_string_table_size));

	initialise_code_area();
	initialise_string_table();
	initialise_name_table();
	initialise_predicate_table();

	/* Read in binary */
	if (fread((char *)code_area, exe_code_area_size, 1, executable)
		!= 1 ) 
            fatal(" 2.1 invalid format of executable image %s", filename);
	if (fread((char *)query_code, exe_query_code_size, 1, executable)
		!= 1 )
            fatal(" 2.2 invalid format of executable image %s", filename);
	if (fread((char *)string_table, exe_string_table_size, 1,
		executable) != 1 ) 
            fatal(" 2.3 invalid format of executable image %s", filename);
	if (fread((char *)name_table, (int)(name_table_size*sizeof(NAME)),
		  1, executable) != 1 )
            fatal(" 2.4 invalid format of executable image %s", filename);
	if (fread((char *)predicate_table,
		(int)(predicate_table_size*sizeof(offset)), 1, executable) !=
			1)
	    fatal(" 2.5 invalid format of executable image %s", filename);

	top_of_code_area = code_area + exe_code_area_size;
	top_of_query_code = query_code + exe_query_code_size;
	top_of_string_table = string_table + exe_string_table_size;

	fclose(executable);
}
