/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

#ifndef	CELLS_H
#define	CELLS_H

#include "name_table.h"
#include "string_table.h"
#include "errors.h"
#include "primitives.h"

/* Constants */

                       
#define	NIL     1                  /* the first string in the string_table */ 
                        
#define	CONS    4                  /* second string (add 3 for []\0 (NIL)) */	
                       	       
#define	DOLLAR  6                    /* third string (add 2 for .\0 (CONS)) */ 

#define	EMPTY_SUB	0

/* Tags */

#define	CONSTANT		0x00000000
#	define	ATOM		(CONSTANT|0x00000000)
#	define	INTEGER		(CONSTANT|0x10000000)
#define	APPLY			0x20000000
#define	PAIR			0x40000000
#define	VARIABLE		0x60000000
#define	OBJECT_VARIABLE		VARIABLE
#define REFERENCE		0x80000000
#define	OBJECT_REFERENCE 	0xa0000000
#define QUANTIFIER		0xc0000000
#define SUBSTITUTION_OPERATOR	0xe0000000

/* freeze and thaw */
#define	THAW			0x00000000
#define	FREEZE			0x10000000

/* Substitution Property Type */
/* the properties must arrange in the increasing order with the variety of
 * objects appeared in the range of a substitution */

#define	INVERTIBLE		0x00000000
#define OBJECT_VARIABLES_ONLY	0x80000000
#define OTHERS			0xc0000000

/* Masks */

#define	TAGMASK			0xe0000000
#define	CONSTANT_MASK		0xf0000000
#define ALL_DISTINCT		0x80000000
#define PROPERTYMASK		0xc0000000
#define	TEMPMASK		0x10000000

/* Field Access */

#define	Tag(c)			((c)&TAGMASK)
#define	ConstantTag(c)		((c)&CONSTANT_MASK)
#define	RestOf(c)		((c)&~TAGMASK)		/* remove tag */
#define	RestOfConstant(c) 	((c)&~CONSTANT_MASK)	/* remove tag */
#define	RestOfVariable(c) 	(RestOfConstant(c))	/* remove tag & temp */
#define	Location(c)		((cell *)RestOf(c))	/* turn into pointer */
#define Property(s)		((s)&PROPERTYMASK)
#define SubPointer(s)		((cell *)((s)&~PROPERTYMASK))
#define	Temperature(s)		((s)&TEMPMASK)
#define	RestOfTemp(s)		((s)&~TEMPMASK)
#define	Frozen(s)		(Temperature(Value(s)) == FREEZE)

/* Testing */

#define	IsVariable(x)		(Tag(x) == VARIABLE)
#define	IsReference(x)		(Tag(x) == REFERENCE)
#define	IsConstant(x)		(Tag(x) == CONSTANT)
#define	IsAtom(x)		(ConstantTag(x) == ATOM)
#define	IsInteger(x)		(ConstantTag(x) == INTEGER)
#define	IsApply(x)		(Tag(x) == APPLY)
#define	IsPair(x)		(Tag(x) == PAIR)
#define	IsQuantifier(x)		(Tag(x) == QUANTIFIER)
#define IsList(x)		(IsCONS(fetch_functor(x)) && \
				 fetch_arity(x) == 2)
#define	IsObjectVariable(x)	(Tag(x) == OBJECT_VARIABLE)
#define	IsLocalObjectVariable(x)    (IsObjectReference(x) && AllDistinct(x))
#define	IsObjectReference(x)	(Tag(x) == OBJECT_REFERENCE)
#define	IsSubstitution(x)	(Tag(x) == SUBSTITUTION_OPERATOR)
#define IsCONS(x)		(IsAtom(x) && RestOfConstant(x) == CONS)
#define IsNIL(x)		(IsAtom(x) && RestOfConstant(x) == NIL)

/* Cells */

#define	Integer(n)		((cell)(INTEGER|((n)&0x03ffffff)))
#define	IntOf(c)		(((int)(c))<<6>>6)
/* Hacked at the moment to be compatible with Nu-Prolog 26 bit
#define	Integer(n)		((cell)(INTEGER|((n)&0x0fffffff)))
#define	IntOf(c)		(((int)(c))<<4>>4)
*/

#define	Atom(offset)		((cell)(CONSTANT|(offset)))
#define	String(offset)		(String2(RestOfConstant(offset)))
#define	String2(offset)		(string_table + (offset))

#define Apply()			((cell)(APPLY|NewPair()))

#define	NewPair()        	(HeapCheck(6),				\
				 *top_of_heap   = Reference(top_of_heap+2),\
				 top_of_heap++ ,                           \
				 *top_of_heap = Reference(top_of_heap+3),\
				 top_of_heap++ ,                           \
				 *top_of_heap++ = NULL_VARIABLE,       	\
				 *top_of_heap++ = NULL,         	\
				 *top_of_heap++ = NULL_VARIABLE,       	\
				 *top_of_heap++ = NULL,           	\
				 (cell)(top_of_heap-6))

#define	Argument(c)		((Location(c))[0])
#define	Functor(c)		((Location(c))[1])
#define ArgumentV(to,from)	((to).term = Reference(&Argument((from).term)), \
				 (to).sub = EMPTY_SUB)
#define FunctorV(to,from)	((to).term = Reference(&Functor((from).term)), \
				 (to).sub = EMPTY_SUB)

#define Cons(h,t)		(apply(apply(Atom(CONS), h), t))
/*
#define Head(c)			(Argument(Functor(c)))
*/
#define Head(v, c)		(Argument(DerefTerm(v, Functor(c))))
#define Tail(c)			(Argument(c))
#define Pair()			((cell)(PAIR|NewPair()))
#define	Left(c)			((Location(c))[0])
#define	Right(c)		((Location(c))[1])
#define	LeftV(to,from)		((to).term = Reference(&Left((from).term)), \
				 (to).sub = EMPTY_SUB)
#define	RightV(to,from)		((to).term = Reference(&Right((from).term)), \
				 (to).sub = EMPTY_SUB)

#define Quantifier()		((cell)(QUANTIFIER|NewPair()))
#define	BoundVar(c)		((Location(c))[0])
#define	Body(c)			((Location(c))[1])
#define	BoundVarV(to,from)	((to).term = BoundVar((from).term), \
				 (to).sub = EMPTY_SUB)
#define	BodyV(to,from)		((to).term = Reference(&Body((from).term)), \
				 (to).sub = EMPTY_SUB)

#define	NULL_VARIABLE		(VARIABLE|(cell)(NULL))

#define	Reference(v)		(REFERENCE|(cell)(v))
#define	Value(c)		(*Location(c))

#define	NULL_OBJECT_VARIABLE	(OBJECT_VARIABLE|(cell)(NULL))

#define	ObjectReference(v)	(OBJECT_REFERENCE|(cell)(v))
#define	ObjectValue(c)		(*Location(c))
#define Distinction(c)		(Location(c)[2])
#define AllDistinct(d)		(Distinction(d) == ALL_DISTINCT)

#define	SubstitutionOperator()	((cell)(SUBSTITUTION_OPERATOR|NewPair()))
#define Substitution(c)		((Location(c))[0])
#define Term(c)			((Location(c))[1])

#define NewProperty(s,p)	((cell)(((int)(s)&~PROPERTYMASK)|(p)))
#define	Table(s)		(SubPointer(s)[0])
#define NextSub(s)		(SubPointer(s)[1])
#define	Size(s)			(((cell *)Table(s))[0])
#define	Domain(s,n)		(((cell *)Table(s))[2 * n - 1])
#define Range(s,n)		(((cell *)Table(s))[2 * n])

/* Support Macros */

#define	HeapCheck(n)		((top_of_heap+(n)) > stack ?		\
					fatal("Out of heap %d K", heap_size) :\
					(void)TRUE)

/*
#define	NewPair()		(HeapCheck(2),				\
				 *top_of_heap++ = NULL_VARIABLE,	\
				 *top_of_heap++ = NULL_VARIABLE,       	\
				 (cell)(top_of_heap-2))
*/

#define	NewVariable()		(HeapCheck(2),				\
				 *top_of_heap++ = NULL_VARIABLE,	\
				 *top_of_heap++ = NULL,            	\
				 Reference(top_of_heap-2))

#define	NewObjectVariable()	(HeapCheck(3),				\
				 *top_of_heap++ = NULL_OBJECT_VARIABLE,	\
				 *top_of_heap++ = NULL,	                \
				 *top_of_heap++ = NULL,			\
				 ObjectReference(top_of_heap-3))

#define	NewLocalObjectVariable()	(HeapCheck(3),			\
					 *top_of_heap++ = NULL_OBJECT_VARIABLE,	\
				         *top_of_heap++ = NULL,	        \
					 *top_of_heap++ = ALL_DISTINCT,		\
					 ObjectReference(top_of_heap-3))

#define NewDistinction(o,d)	(HeapCheck(2),		\
				 *top_of_heap++ = (o),	\
			 	 *top_of_heap++ = (d),	\
			 	 (cell)(top_of_heap-2))

#define Allocate(n)		(HeapCheck(n),		\
				 top_of_heap += (n),	\
				 (top_of_heap-(n)))

#define NewSubstitution(t,n,p)	(HeapCheck(2),		\
				 *top_of_heap++ = (t),	\
				 *top_of_heap++ = (n),	\
				 NewProperty(top_of_heap-2, p))

#define RenameSub(d, r, n)	(HeapCheck(5),		\
				 *top_of_heap++ = 1,	\
				 *top_of_heap++ = (d),	\
				 *top_of_heap++ = (r),	\
				 *top_of_heap = (cell)(top_of_heap-3), \
				 top_of_heap++, \
				 *top_of_heap++ = (n),	\
				 NewProperty(top_of_heap-2,	\
					(Property(n) > INVERTIBLE ?	\
							Property(n) :	\
							INVERTIBLE)))

#define DelayLink(d, n)		(HeapCheck(2),		\
				 *top_of_heap++ = (d),	\
				 *top_of_heap++ = (n),	\
				 (int)(top_of_heap-2))

#endif	/* CELLS_H */
