;;; -*- Package: Nuprl; Syntax: Common-lisp; Base: 10. -*-


;;;************************************************************************
;;;                                                                       *
;;;                Nuprl Proof Development System                         *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the Nuprl group, Department of Computer Science,       *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify Nuprl provided this notice  *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************



(in-package "NUPRL")


;===============================;
;                               ;
;           REF.L               ;
;                               ;
;   Routines to Refine Goals    ;
;                               ;
;===============================;



(global ref-assums)   ;-- the assumptions of pt
(global ref-concl)    ;-- the conclusion of pt
(global ref-rule-body);-- the rule body of pt
(global ref-rule)     ;-- the rule parsed from the rule-body
(global ref-children) ;-- the children generated by applying ref-rule
                      ;-- to the goal
(global ref-hidden)   ;-- the hidden assumptions of pt

(proclaim '(special ted-parsed-rule))
                         ;-- when ^D is entered in a ted window while editing
                         ;-- a rule, ted parses the rule to see whether or not
                         ;-- to kill the window.  This variable is used to 
                         ;-- store the result of that parse so parse-rule-and
                         ;-- refine won't have to parse the rule again.
(global AUTO-TACTIC)

(defun ref-init ()
    (<- ref-assums nil)
    (<- ref-concl nil)
    (<- ref-rule-body nil)
    (<- ref-rule nil)
    (<- ref-children nil)
    (<- ref-hidden nil)
    (<- ted-parsed-rule nil)
    (for (x in '((APPLY-COMP refine-apply-comp)             
		 (DECIDE-COMP refine-decide-comp)
		 (IND-COMP-DOWN refine-ind-comp-down)
		 (IND-COMP-BASE refine-ind-comp-base)
		 (IND-COMP-UP refine-ind-comp-up)
		 (LIST-IND-COMP refine-list-ind-comp)
		 (LIST-IND-COMP-BASE refine-list-ind-comp-base)
		 (LIST-IND-COMP-UP refine-list-ind-comp-up)
		 (SPREAD-COMP refine-spread-comp)
		 (ATOMEQ-COMP-TRUE refine-atomeq-comp)
		 (ATOMEQ-COMP-FALSE refine-atomeq-comp)
		 (INTEQ-COMP-TRUE refine-inteq-comp)
		 (INTEQ-COMP-FALSE refine-inteq-comp)
		 (INTLESS-COMP-TRUE refine-intless-comp)
		 (INTLESS-COMP-FALSE refine-intless-comp)
		 (ATOM-INTRO refine-atom-intro)
		 (DOM-COMP refine-dom-comp)
		 (FIX-COMP refine-fix-comp)
		 (REC-IND-COMP refine-rec-ind-comp)
		 (SIMPLE-REC-IND-COMP refine-simple-rec-ind-comp)
		 (DIRECT-COMP refine-direct-comp)
		 (DIRECT-COMP-HYP refine-direct-comp-hyp)
		 (EQUAL-INTRO-UNIVERSE refine-eq-intro-universe)
		 (EQUAL-INTRO-AXIOM-EQUAL refine-eq-intro-axiom-equal)
		 (EQUAL-INTRO-EQUAL refine-eq-intro-equal)
		 (EQUAL-INTRO-LESS refine-eq-intro-less)  
		 (EQUAL-INTRO-AXIOM-LESS refine-eq-intro-axiom-less)
		 (EQUAL-INTRO-ADD refine-eq-intro-bin-op)
		 (EQUAL-INTRO-SUB refine-eq-intro-bin-op)
		 (EQUAL-INTRO-MUL refine-eq-intro-bin-op)
		 (EQUAL-INTRO-DIV refine-eq-intro-bin-op)
		 (EQUAL-INTRO-MOD refine-eq-intro-bin-op)
		 (EQUAL-INTRO-POS-NUMBER refine-eq-intro-pos-number)
		 (EQUAL-INTRO-MINUS refine-eq-intro-minus)
		 (EQUAL-INTRO-VOID refine-eq-intro-void)
		 (EQUAL-INTRO-ANY refine-eq-intro-any)  
		 (EQUAL-INTRO-TOKEN refine-eq-intro-token)
		 (EQUAL-INTRO-ATOM refine-eq-intro-atom)
		 (EQUAL-INTRO-INT refine-eq-intro-int)
		 (EQUAL-INTRO-OBJECT refine-eq-intro-object)
		 (EQUAL-INTRO-IND refine-eq-intro-ind)
		 (EQUAL-INTRO-LIST refine-eq-intro-list)
		 (EQUAL-INTRO-NIL refine-eq-intro-nil)
		 (EQUAL-INTRO-CONS refine-eq-intro-cons)
		 (EQUAL-INTRO-LIST-IND refine-eq-intro-list-ind)
		 (EQUAL-INTRO-UNION refine-eq-intro-union)
		 (EQUAL-INTRO-INL refine-eq-intro-inl)
		 (EQUAL-INTRO-INR refine-eq-intro-inr)
		 (EQUAL-INTRO-DECIDE refine-eq-intro-decide)
		 (EQUAL-INTRO-PRODUCT refine-eq-intro-product)
		 (EQUAL-INTRO-PAIR refine-eq-intro-pair)
		 (EQUAL-INTRO-SPREAD refine-eq-intro-spread)
		 (EQUAL-INTRO-FUNCTION refine-eq-intro-function)
		 (EQUAL-INTRO-LAMBDA refine-eq-intro-lambda)
		 (EQUAL-INTRO-APPLY refine-eq-intro-apply)
		 (EQUAL-INTRO-QUOTIENT refine-eq-intro-quotient)
		 (EQUAL-INTRO-QUOTIENT-ELEM refine-eq-intro-quotient-elem)
		 (EQUAL-INTRO-SET refine-eq-intro-set)
		 (EQUAL-INTRO-SET-ELEM refine-eq-intro-set-elem)
		 (EQUAL-INTRO-VAR refine-eq-intro-var)
		 (EQUAL-INTRO-ATOMEQ refine-eq-intro-atomeq)
		 (EQUAL-INTRO-INTEQ refine-eq-intro-inteq)
		 (EQUAL-INTRO-INTLESS refine-eq-intro-intless)
		 (EQUAL-INTRO-PARFUNCTION refine-eq-intro-parfunction)
		 (EQUAL-INTRO-FIX refine-eq-intro-fix)
		 (EQUAL-INTRO-APPLY-PARTIAL refine-eq-intro-apply-partial)
		 (EQUAL-INTRO-DOM refine-eq-intro-dom)
		 (EQUAL-INTRO-RECURSIVE refine-eq-intro-recursive)
		 (EQUAL-INTRO-SIMPLE-REC refine-eq-intro-simple-rec)
		 (EQUAL-INTRO-REC-MEMBER refine-eq-intro-rec-member)
		 (EQUAL-INTRO-SIMPLE-REC-MEMBER refine-eq-intro-simple-rec-member)
		 (EQUAL-INTRO-REC-IND refine-eq-intro-rec-ind)
		 (EQUAL-INTRO-SIMPLE-REC-IND refine-eq-intro-simple-rec-ind)
		 (EQUAL-INTRO-OBJECT-MEMBER refine-eq-intro-object-member)
		 (FUNCTION-INTRO refine-function-intro)
		 (INT-INTRO-OP refine-int-intro-op)
		 (INT-INTRO-NUMBER refine-int-intro-number)
		 (LESS-INTRO refine-less-intro)
		 (LIST-INTRO-NIL refine-list-intro-nil)
		 (LIST-INTRO-CONS refine-list-intro-cons)
		 (PRODUCT-INTRO refine-product-intro)
		 (QUOTIENT-INTRO refine-quotient-intro)
		 (SET-INTRO refine-set-intro)
		 (UNION-INTRO refine-union-intro)
		 (RECURSIVE-INTRO refine-recursive-intro)
		 (SIMPLE-REC-INTRO refine-simple-rec-intro)
		 (UNIVERSE-INTRO-ATOM refine-ui-atom)
		 (UNIVERSE-INTRO-EQUAL refine-ui-equal)
		 (UNIVERSE-INTRO-FUNCTION refine-ui-function)
		 (UNIVERSE-INTRO-INT refine-ui-int)
		 (UNIVERSE-INTRO-LESS refine-ui-less)
		 (UNIVERSE-INTRO-LIST refine-ui-list)
		 (UNIVERSE-INTRO-UNION refine-ui-union)
		 (UNIVERSE-INTRO-PRODUCT refine-ui-product)     
		 (UNIVERSE-INTRO-QUOTIENT refine-ui-quotient)
		 (UNIVERSE-INTRO-SET refine-ui-set)
		 (UNIVERSE-INTRO-UNIVERSE refine-ui-universe)
		 (UNIVERSE-INTRO-VOID refine-ui-void)
		 (FUNCTION-ELIM refine-function-elim)
		 (INT-ELIM refine-int-elim)     
		 (LIST-ELIM refine-list-elim)
		 (PRODUCT-ELIM refine-product-elim)
		 (QUOTIENT-ELIM refine-quotient-elim)
		 (SET-ELIM refine-set-elim)
		 (PARFUNCTION-ELIM refine-parfunction-elim)
		 (RECURSIVE-ELIM refine-recursive-elim)
		 (SIMPLE-REC-ELIM refine-simple-rec-elim)
		 (RECURSIVE-UNROLL-ELIM refine-recursive-unroll-elim)
		 (SIMPLE-REC-UNROLL-ELIM refine-simple-rec-unroll-elim)
		 (UNION-ELIM refine-union-elim)
		 (VOID-ELIM refine-void-elim)
		 (HYP refine-hyp)
		 (LEMMA refine-lemma)
		 (DEF refine-def)
		 (SEQUENCE refine-sequence)
		 (EXPLICIT-INTRO refine-explicit-intro)
		 (CUMULATIVITY refine-cumulativity)
		 (EQUALITY refine-equality)
		 (SUBSTITUTE refine-substitute)
		 (BECAUSE refine-because)
		 (EXPERIMENTAL refine-experimental)
		 (TACTIC refine-tactic)
		 (EXTENSIONALITY refine-extensionality)
		 (ARITH refine-arith)
		 (THINNING refine-thinning)
		 (MONOT refine-monot)
		 (DIVISION refine-division)
		 )    

         )
         (do 
            (setf (get (car x) 'ref-function) (cadr x))
         )
    )

)


;--
;-- parse-rule-and-refine (pt:proof-tree)
;--
;--     Parse the rule-body in pt, place the resulting rule in that
;--     field of pt, and refine the goal using the rule to produce
;--     children for pt.  Print messages for any errors that occur.
;--     Returns nil if no errors occur.  If there is an error, the 
;--     rule field of pt will be set to nil.

(defun parse-rule-and-refine (pt)
    (parse-rule-and-refine$ pt t)
)

(defun parse-rule-and-refine$ (pt use-auto-tactic)
    (Plet (rule  nil
          error nil
         )         
        (Pif ted-parsed-rule -->
            (<- rule ted-parsed-rule)
            (<- ted-parsed-rule nil)
         || t -->
            (<- rule (parse-rule pt))
         fi)

        (Pif (null rule) -->
            (<- (rule-of-proof-tree pt) nil)
            (<- (children-of-proof-tree pt) nil)

         || (eql (kind-of-rule rule) 'TACTIC) -->
            (<- error (refine pt rule nil))
            (Pif error -->
                (display-message (append (istring '|ERROR IN REFINEMENT: |)
                                         (istring (cadr error))
                                 )
                )
             fi)

         || (eql (kind-of-rule rule) 'HELP) -->
            (<- (rule-of-proof-tree pt) nil)
            (<- (children-of-proof-tree pt) nil)
            (display-message (append (istring '|HELP: |)
                                     (istring (cadr rule))
                             )
            )  
 
         || (eql (kind-of-rule rule) 'ERR) -->                           
            (<- (rule-of-proof-tree pt) nil)
            (<- (children-of-proof-tree pt) nil)
            (display-message (append (istring '|ERROR IN REFINEMENT: |)
                                     (istring (cadr rule))
                             )
            )
    
         || t -->
            (<- error (refine pt rule use-auto-tactic))
            (Pif error -->
                (display-message (append (istring '|Error in Refinement: |)
                                         (istring (cadr error))
                                 )     
                )
             fi)
    
         fi)
    )
)

;--
(defun parse-rule (pt)
    (Plet (rule       nil
          refd-defs  nil
         )
        (<- ref-assums (assumptions-of-proof-tree pt))
        (<- ref-concl (conclusion-of-proof-tree pt))
        (<- ref-rule-body (rule-body-of-proof-tree pt))
        (<- ref-hidden (hidden-of-proof-tree pt))
        (<- refd-defs (scan-init (rule-body-of-proof-tree pt)))
        (<- rule  (catch 'process-err (parse-rule$)))
        (Pif (and (not (null rule))
                 (not (member (kind-of-rule rule) '(TACTIC EXPERIMENTAL HELP ERR)))
                 (not (= token-type TEndInput))
            ) -->
            (<- rule 
                (list 'ERR
                    (concat '|text "|
                        (delim-descr-for token-type)
                        '|" following the end of the rule.|
                    )
                )
            )
         fi)
        rule
    )
)



;-- deduce-children(pt:proof-tree, rule:rule)
;--
;-- The result is of one of the following forms:
;-- (SUCCESS.(rule.children))    or
;-- (FAILURE.message).  This is necessary since refining by a tactic
;-- rule potentially changes the rule (i.e., it includes the proof-top
;-- in the rule.


(defun deduce-children (pt rule)

    (Plet (ref-assums-save    ref-assums
          ref-concl-save     ref-concl
          ref-rule-body-save ref-rule-body
          ref-rule-save      ref-rule
          ref-children-save  ref-children
          ref-hidden-save    ref-hidden
          old-pt-rule-body   (rule-body-of-proof-tree pt)
          old-pt-rule        (rule-of-proof-tree pt)
          old-pt-children    (children-of-proof-tree pt)
         )

	 (unwind-protect
	    (Plet (ref-result (refine pt rule nil))
		(Pif (null ref-result) -->
		    (cons 'SUCCESS (cons ref-rule ref-children))
    
		 || t -->
		    (cons 'FAILURE (cadr ref-result))
		 fi)
	    )
    
	    (<- ref-assums    ref-assums-save)
	    (<- ref-concl     ref-concl-save)
	    (<- ref-rule-body ref-rule-body-save)
	    (<- ref-rule      ref-rule-save)
	    (<- ref-children  ref-children-save)
	    (<- ref-hidden    ref-hidden-save)
    
	    ;-- These are  necessary because of destructive changes 
		(<- (rule-body-of-proof-tree pt) old-pt-rule-body)
		(<- (rule-of-proof-tree pt) old-pt-rule)
		(<- (children-of-proof-tree pt) old-pt-children)
	)
    )
)


;--
;-- refine (pt:proof-tree rule:ref-rule use-auto-tactic:boolean)
;--
;-- Refines the proof-tree node, using the parsed rule.  Returns t iff the 
;-- refinement is successful.
;-- 

(defun refine (pt rule use-auto-tactic)
    (Pif (ok-rule-kind$ rule) -->                                  
        (<- ref-assums (assumptions-of-proof-tree pt))
        (<- ref-concl  (conclusion-of-proof-tree pt))
        (<- ref-rule-body (rule-body-of-proof-tree pt))
        (<- ref-rule rule)
        (<- ref-children nil)
        (<- ref-hidden (hidden-of-proof-tree pt))
        (Plet (error
                 (catch
                     'process-err
                     (progn
                         (funcall (get (kind-of-rule rule) 'ref-function))
                         nil
                     )
                 )
             )
            (Pif error -->
                (<- (rule-of-proof-tree pt) nil)
                (<- (children-of-proof-tree pt) nil)
                error   ;-- Error indicator 

             || t -->
                (<- (rule-of-proof-tree pt) ref-rule)

                (Pif use-auto-tactic -->
                    (mapc                           ;-- destructive.
                        #'apply-auto-tactic$
                        ref-children
                    )
                fi)

                (<- (children-of-proof-tree pt) ref-children)
         
                nil     ;-- success indicator

             fi)
        )
        
     || t -->
        nil
     fi)
)



;-- 
;-- Apply the auto tactic to each unproved leaf.  If the result is not an
;-- error, then splice this result in just as in transformation-tactics.
;-- The error function passed to do-tactic is an ignore-error instruction.
;--

(defun apply-auto-tactic$ (proof)
    (Plet (
             result-proof                       
             (do-tactic
                 proof AUTO-TACTIC
		 #'(lambda (x) x nil)
             )
         )
        (Pif (not (null result-proof)) -->
            (<- (rule-of-proof-tree proof)
                (rule-of-proof-tree result-proof)
            )
            (<- (rule-body-of-proof-tree proof)
                (rule-body-of-proof-tree result-proof)
            )
            (<- (children-of-proof-tree proof)
                (children-of-proof-tree result-proof)
            )
         fi)
    )
)



(defun ok-rule-kind$ (rule)
  (declare (ignore rule))
    t
)


;--
;-- check-proof-tree (goal-body:Ttree pt:proof-tree)
;--
;--     Parse goal-body as the main goal of a proof-tree.  Check it
;--     against the goal of pt.  If it matches, then refine pt and
;--     check if the rule and children match those present.  If so,
;--     continue checking the children of pt.  If anything fails to
;--     match, note this in the status of pt. ????
;--

(defun check-proof-tree (goal-body pt)
  (declare (ignore goal-body pt))

    nil

)


;--
;--
;--
;--
;--
;--

(defun check-pt-levels$ (pt)
  (declare (ignore pt))

    nil

)


;--
;-- build-proof-tree (goal-body:Ttree, rule-body-tree:crunched-proof-tree)
;--
;--     Parse goal-body as the main goal of a proof-tree. Then use the
;--     Ttrees in rule-body-tree to refine the goal into a proof-tree.
;--     A rule-body-tree is either nil, or a list whose first element
;--     is a rule-body and whose second element is a list of rule-body-trees
;--     for the children that result from refining the goal with the rule-body.
;--

(defun build-proof-tree (goal-body rule-body-tree)

    (Plet (pt  nil       ;-- the proof tree to be returned

         )

        ;-- build the root of the proof tree, pt
            (<- pt (catch 'process-err (parse-goal goal-body)))

        (Pif (eql (car pt) 'ERR) -->
            ;-- the main goal was not well formed, return nil
                nil

         || t -->
            ;-- the main goal was OK, now build the biggest tree possible
                (build-pt-levels$ pt rule-body-tree)
                pt

         fi)

    )
)


;--
;-- build-pt-levels$ (pt:proof-tree,
;--                   rule-body-tree:crunched-proof-tree)
;--
;--     Given a proof tree, pt, and a rule-body-tree (as above), put the
;--     first rule-body in the tree into pt, then refine pt, and continue
;--     with the children of pt and rule-body-tree.  When the  rule-body-tree
;--     doesn't match the children generated, leave things "nil".
;--

(defun build-pt-levels$ (pt rule-body-tree)
  (when (not (null rule-body-tree))
    (<- (rule-body-of-proof-tree pt) (car rule-body-tree))
    (when (not (null-Ttree (rule-body-of-proof-tree pt)))
      (parse-rule-and-refine$ pt nil)
      ;; pt will have nil as rule field if the refinement
      ;; was unsuccessful.
      (cond ((null (rule-of-proof-tree pt))
	     (<- (status-of-proof-tree pt) 'BAD))
	    (t
	     (<- (rule-of-proof-tree pt) ref-rule)
	     (<- (children-of-proof-tree pt) ref-children)
	     (let ((rules (cadr rule-body-tree)))
	       (for
		 (child in ref-children)
		 (do
		   (when (not (null rules))
		     (build-pt-levels$ child (car rules))
		     (<- rules (cdr rules))))))
	     (<- (status-of-proof-tree pt)
		 (calculate-status-from-proof pt)))))))



;--
;-- ref-help (text:atom)
;--a process-err from parse-rule with the result:
;--     (HELP  text)
;--

(defun ref-help (text)

    (throw 'process-err (list 'HELP text))

)

;-- 
;-- ref-error (text:atom)
;-- 
;--    Throw a process-err with (ERR text).
;-- 
(defun ref-error (text)
    (throw 'process-err `(ERR ,text))
)






