(* Copyright (c) 1991 by Carnegie Mellon University *)
(* Author: Frank Pfenning <fp@cs.cmu.edu>           *)

(* Constraint management *)
(* Printing should be fixed to use formatter !!! *)

functor Constraints (structure Term : TERM
		     structure ConstraintsDataTypes : CONSTRAINTS_DATATYPES
			sharing ConstraintsDataTypes.Term = Term
		     structure Print : PRINT
		        sharing Print.Term = Term
		     structure Sb : SB
		        sharing Sb.Term = Term
		     structure Reduce : REDUCE
		        sharing Reduce.Term = Term) : CONSTRAINTS =
struct

open ConstraintsDataTypes

structure Term = Term
structure F = Print.F
structure S = Print.S

local open Term
in

  exception Nonunifiable of unit -> string

  fun bare_term (Gvar(M,_)) = M
    | bare_term (Flex(M,_)) = M
    | bare_term (Abstraction(M)) = M
    | bare_term (Rigid(M,_)) = M
    | bare_term (Quant(M)) = M
    | bare_term (Any(M)) = M

  fun makestring_eqterm (Gvar(M,_)) = "[G]" ^ Print.makestring_term M
    | makestring_eqterm (Flex(M,_)) = "[F]" ^ Print.makestring_term M
    | makestring_eqterm (Abstraction(M)) = "[L]" ^ Print.makestring_term M
    | makestring_eqterm (Rigid(M,_)) = "[R]" ^ Print.makestring_term M
    | makestring_eqterm (Quant(M)) = "[Q]" ^ Print.makestring_term M
    | makestring_eqterm (Any(M)) = "[A]" ^ Print.makestring_term M

  val mkDpair = Dpair

  fun makestring_dpair (Dpair(eqM,eqN)) =
      "(" ^ makestring_eqterm eqM ^ ", "
      ^ makestring_eqterm eqN ^ ")"

  fun makestring_dset ds =
      " [[ "
      ^ (let fun ms nil = ""
		  | ms (dpair :: rest) =
		       makestring_dpair dpair
		       ^ (case rest of nil => "" | _ => ",\n    ") ^ ms rest
	      in ms ds end)
      ^ " ]] \n"

  (* This might still miss opportunities (head-normalization, eta conv.) *)

  fun eq_uvar_or_non_uvar stamp1 (Uvar(_,stamp2)) = (stamp1 = stamp2)
    | eq_uvar_or_non_uvar stamp1 (Evar(_,_,_,ref(SOME(M0)))) =
	 eq_uvar_or_non_uvar stamp1 M0
    | eq_uvar_or_non_uvar stamp1 _ = true

  fun dominates_all depends_on nil = true
    | dominates_all depends_on (Uvar(_,stamp)::rest) =
	 if exists (eq_uvar_or_non_uvar stamp) rest
	      orelse exists (Sb.eq_uvar stamp) depends_on
	    then false
	    else dominates_all depends_on rest
    | dominates_all depends_on _ = false  (* first arg is not a Uvar *)


  fun anno_norm (M as (Abst _)) = Abstraction(M)
    | anno_norm (Evar(_,_,_,ref(SOME(M0)))) = anno_norm M0
    | anno_norm (M as Type) = Rigid(M,(M,nil))
    | anno_norm (M as Pi _ ) = Quant(M)
    | anno_norm M = 
	let fun anorm (head as Const _) args = Rigid(M,(head,args))
	      | anorm (head as Uvar _) args = Rigid(M,(head,args))
	      | anorm (Evar(_,_,_,ref(SOME(M0)))) args = anorm M0 args
	      | anorm (head as Evar(_,_,uvars,_)) args =
		   if dominates_all uvars args
		      then Gvar(M,(head,args))
		      else Flex(M,head)
	      | anorm (Appl(M1,N1)) args = anorm M1 (N1::args)
	      | anorm (Abst _) args = anno_norm (Reduce.head_norm M)
	      | anorm (head as Fvar _) args = Rigid(M,(head,args))
	      | anorm M _ = raise Print.subtype("anorm",M,"illegal argument.")
	 in anorm M nil end

  (* fun anno M = anno_norm (Reduce.head_norm M) *)
  val anno = anno_norm

  fun reclassify M = anno_norm (Reduce.head_args_norm M)

  val empty_constraint = Con(nil)
  fun is_empty_constraint (Con(nil)) = true
    | is_empty_constraint _ = false

  fun makeformat_dpair (Dpair(eqM,eqN)) =
      F.HOVbox [Print.makeformat_term (bare_term eqM), F.Break,
		F.String S.equal, F.Space,
		Print.makeformat_term (bare_term eqN)]

  fun makeformat_dset nil = nil
    | makeformat_dset (dpair :: rest) =
	 (makeformat_dpair dpair
	  :: (case rest of nil => [] | _ => [F.String S.comma, F.Break]))
	 @ makeformat_dset rest

  fun makeformat_constraint (Con(dset)) =
    F.Hbox [F.String S.lparen, F.String S.lparen, F.Space,
	    F.Vbox0 0 1 (makeformat_dset dset),
	    F.Space, F.String S.rparen, F.String S.rparen,
	    F.Newline ()]
  val makestring_constraint = F.makestring_fmt o makeformat_constraint

end  (* local ... *)
end  (* functor Constraints *)
