(* Copyright (c) 1991 by Carnegie Mellon University *)
(* Author: Spiro Michaylov <spiro@cs.cmu.edu>       *)
(* Modified: Frank Pfenning <fp@cs.cmu.edu>, Feb  4 1992 *)

(* Program table *)

functor Progtab
   (structure Basic : BASIC
    structure Term : TERM
    structure Sb : SB sharing Sb.Term = Term 
    structure Skeleton : SKELETON sharing Skeleton.Term = Term
    structure Reduce : REDUCE  sharing Reduce.Term = Term)
   : PROGTAB =
struct

structure Term = Term
structure Skeleton = Skeleton

local open Term
in

  exception Progtab

  datatype mobility = Dynamic of bool | Static | Unknown of bool

  datatype progentry =
      Progentry of 
	  {
	  Faml: Term.term,                (* family *) 
	  Name: Term.term,                (* name of rule *)
	  Vars: Term.varbind list,        (* variables *)
	  Head: Term.term,                (* head of rule *)
	  Subg: mobility list,            (* subgoals *)
	  Indx: Term.term option,	  (* principal functor of 1st arg *)
	  Skln: Skeleton.skeleton         (* skeleton for unification *)
	  }

  val rules : progentry list Array.array = Array.array (1000,nil)

  val free_prog : int ref = ref 0

  (* reset should only be called if the symbol table is cleared out at the *)
  (* same time.  Normally, store_prog will reinitialize the program table *)
  fun reset () = ( free_prog := 0 )

  fun clean_prog_table () = 
	  let fun cpt (0) = ()
		| cpt (n) = (Array.update (rules,n-1,nil) ; cpt (n-1))
	   in cpt (!free_prog) end

  fun end_of_rel pos rule = 
	  Array.update (rules, pos, ( Array.sub (rules,pos) @ (rule::nil) ))

  fun store_rule (rule as (Progentry {Faml = (Const(E(ref {Prog = index_r, ...}))), ...})) = 
	  (case !index_r
	     of NONE =>          (index_r := SOME (!free_prog) ;
				  Array.update (rules,(!free_prog),(rule::nil)) ;
				  free_prog := !free_prog + 1 ;
				  () )
	      | SOME ri =>       (end_of_rel ri rule;
				  () ))
    | store_rule (rule) =
         raise Basic.Illegal("store_rule: head is not a constant")

  fun store_prog progs = 
	  let val _ = clean_prog_table ()
	      fun ss nil = ()
		| ss (rule::rules) = (store_rule rule ; ss rules)
	      fun sp nil = ()
		| sp (prog::progs) = (ss prog ; sp progs)
	   in sp progs end

  (* The notion of dynamic below presumes that polymorphism is static *)

  fun mark_dynamic (Const(E(ref {Prog = index_r, ...}))) =
         (case !index_r
	    of NONE => (index_r := SOME (!free_prog) ;
			Array.update (rules,!free_prog,nil) ;
			free_prog := !free_prog + 1)
             | _ => ())
    | mark_dynamic (Uvar _) = ()  (* Uvar's must be static right now *)
    | mark_dynamic _ =
         raise Basic.Illegal("mark_dynamic: family is not constant")

  fun is_dynamic (Const(E(ref {Prog = ref(SOME _), ...}))) = true
    | is_dynamic _ = false

  fun occurs_to_bool Maybe = true
    | occurs_to_bool Vacuous = false

  fun mobility_occ A occurs =
    let fun mobility (A as Const _) =
	       if is_dynamic A
	          then Dynamic (occurs_to_bool occurs)
		  else Static
	  | mobility (Appl(A,M)) = mobility A
	  | mobility (Pi((_,B),_)) = mobility B
	  | mobility (Type) = Static
	  | mobility (Abst(_,A)) = mobility A
	  | mobility (Evar(_,_,_,ref(SOME(A)))) = mobility A
	  | mobility (Evar(_,_,_,ref NONE)) = Unknown (occurs_to_bool occurs)
	  | mobility (Uvar _) = Unknown (occurs_to_bool occurs)
	  | mobility _ = raise Basic.Illegal ("mobility: unexprected argument")
    in mobility A end

  fun sbgs A sg =
       let val A' = Reduce.head_norm A
	in case A'
	     of Pi((yofB as Varbind(y,B),C),occurs) =>
		 let val sg' = mobility_occ B occurs :: sg
		  in if (occurs = Maybe)
		        then let val b = Sb.new_uvar yofB
			      in sbgs (Sb.apply_sb (Sb.term_sb yofB b) C)
				          sg'
			     end
			else sbgs C sg'
		 end
	      | _ => sg
       end

  fun subgoals A = sbgs A nil

  fun get_rules (Uvar _) = nil  
    | get_rules (Const(entry as E(ref {Prog = index_r, ...}))) = 
	  (case !index_r
	     of NONE =>           nil
	      | SOME ri =>        Array.sub (rules,ri) )
    | get_rules _ =
         raise Basic.Illegal("get_rules: head is not a Uvar or constant")

  fun first_arg _ nil = NONE
    | first_arg (Appl(M,_)) (h::t) = first_arg M t
    | first_arg _ (h::t) = SOME(h)

  fun term_index (Const e) = SOME(Const e)
    | term_index (Uvar v) = SOME(Uvar v)
    | term_index (Appl(M,_)) = term_index(M)
    | term_index (Evar(_,_,_,ref(SOME M0))) = term_index M0
    | term_index (Abst(_,M)) = term_index(M)
    | term_index _ = NONE

  fun get_index (Const(E(ref {Full = f, ...}))) args = 
	  let val a = first_arg f args
	   in
		  (case a 
		     of NONE => NONE
		      | SOME(t) => term_index t)
	  end
    | get_index (Uvar v) args = 
		  (case args 
		     of h::t => SOME(h)
		      | _ => NONE)
    | get_index _ _ =
         raise Basic.Illegal("get_index: head is not a Uvar or constant")

  fun indexes_match NONE _ = true
    | indexes_match _ NONE = true
    | indexes_match (SOME i1) (SOME i2) = Reduce.eq_head (i1,i2)

end (* local ... *)
end (* functor Progtab *)
