(* Copyright (c) 1991 by Carnegie Mellon University *)
(* Author: Frank Pfenning <fp@cs.cmu.edu>           *)

structure Tokens = Tokens
structure Interface = Interface
open Interface

type pos = Interface.pos
type svalue = Tokens.svalue
type ('a,'b) token = ('a,'b) Tokens.token
type lexresult= (svalue,pos) token

val comLevel = ref 0
fun eof () = (* last position will be inaccurate! *)
	     let val lastpos = Interface.last_newline () + 1
	      in
	        ( if !comLevel > 0
		  then ( comLevel := 0 ; 
			 error ("unclosed comment %{", (lastpos, lastpos)) )
		  else  () ;
		  Tokens.EOF(lastpos,lastpos) )
             end

fun makeInt (s : string) =
    (revfold (fn (c,a) => a*10 + (ord c - ord("0"))) (explode s) 0)
fun stripquotes (s : string) = substring(s,1,size(s) - 2)

%%
%header (functor ElfLexFun(structure Tokens: Elf_TOKENS
			   structure Interface: INTERFACE
			      sharing type Interface.pos = int) : LEXER);
%s COMMENT DCOMMENT PRAGMA;
lcstart=[a-z!&$^+/<=>?@~|#*`;,]|\-|\\;
ucstart=[A-Z_];
idchars={lcstart}|{ucstart}|[0-9'];
lcid={lcstart}{idchars}*;
ucid={ucstart}{idchars}*;
quid='{lcid}'|'{ucid}';
ws=[\t\ ]*;
num=[0-9]+;
%%
<INITIAL>{ws}	=> (continue());
<INITIAL>\n	=> (next_line(yypos); continue());
<INITIAL>"%"	=> (YYBEGIN COMMENT; continue());
<INITIAL>"<-"	=> (Tokens.BACKARROW(yypos,yypos+2));
<INITIAL>":"	=> (Tokens.COLON(yypos,yypos+1));
<INITIAL>"."    => (Tokens.DOT(yypos,yypos+1));
<INITIAL>"("	=> (Tokens.LPAREN(yypos,yypos+1));
<INITIAL>")"	=> (Tokens.RPAREN(yypos,yypos+1));
<INITIAL>"["	=> (Tokens.LBRACKET(yypos,yypos+1));
<INITIAL>"]"	=> (Tokens.RBRACKET(yypos,yypos+1));
<INITIAL>"{"	=> (Tokens.LBRACE(yypos,yypos+1));
<INITIAL>"}"	=> (Tokens.RBRACE(yypos,yypos+1));
<INITIAL>"->"	=> (Tokens.ARROW(yypos,yypos+2));
<INITIAL>"_"	=> (Tokens.UNDERSCORE(yypos,yypos+1));
<INITIAL>"sigma"=> (Tokens.SIGMA(yypos,yypos+5));
<INITIAL>"type"	=> (Tokens.TYPE(yypos,yypos+4));
<INITIAL>"%infix" => (YYBEGIN PRAGMA ; Tokens.INFIX(yypos,yypos+6));
<INITIAL>"%prefix" => (YYBEGIN PRAGMA ; Tokens.PREFIX(yypos,yypos+7));
<INITIAL>"%postfix" => (YYBEGIN PRAGMA ; Tokens.POSTFIX(yypos,yypos+8));
<INITIAL>"%name" => (YYBEGIN PRAGMA; Tokens.NAME(yypos,yypos+5));
<INITIAL>"%{"   => (YYBEGIN DCOMMENT; comLevel := 1; continue());
<INITIAL>"}%"	=> (error ("unmatched close comment }%", (yypos, yypos+2));
		    continue());
<INITIAL>{lcid} => (Tokens.LCID (yytext,yypos,yypos+size yytext));
<INITIAL>{ucid} => (Tokens.UCID (yytext,yypos,yypos+size yytext));
<INITIAL>{num}	=> (Tokens.LCID (yytext,yypos,yypos+size yytext));
<INITIAL>{quid} => (Tokens.QUID (stripquotes(yytext),yypos,yypos+size yytext));
<INITIAL>.	=> (error ("ignoring illegal character" ^ yytext,
			   (yypos,yypos+1)); continue());

<COMMENT>\n     => (next_line(yypos); YYBEGIN INITIAL; continue());
<COMMENT>.	=> (continue());

<DCOMMENT>"%{"  => (comLevel := !comLevel+1; continue());
<DCOMMENT>"}%"  => (comLevel := !comLevel-1;
		    if !comLevel=0 then YYBEGIN INITIAL else ();
		    continue());
<DCOMMENT>\n    => (next_line(yypos); continue());
<DCOMMENT>.	=> (continue());

<PRAGMA>{ws}	=> (continue());
<PRAGMA>\n	=> (next_line(yypos); YYBEGIN INITIAL;
		    Tokens.EOFPRAGMA(yypos,yypos+1));
<PRAGMA>"%"	=> (YYBEGIN COMMENT; Tokens.EOFPRAGMA(yypos,yypos+1));
<PRAGMA>"sigma" => (Tokens.SIGMA(yypos,yypos+5));
<PRAGMA>"left"  => (Tokens.LEFT(yypos,yypos+4));
<PRAGMA>"right"	=> (Tokens.RIGHT(yypos,yypos+5));
<PRAGMA>"none"	=> (Tokens.NONE_(yypos,yypos+4));
<PRAGMA>{num}	=> (Tokens.NUM(makeInt(yytext),yypos,yypos+size yytext));
<PRAGMA>{lcid}	=> (Tokens.LCID(yytext,yypos,yypos+size yytext));
<PRAGMA>{ucid}	=> (Tokens.UCID(yytext,yypos,yypos+size yytext));
<PRAGMA>{quid}  => (Tokens.QUID(stripquotes(yytext),yypos,yypos+size yytext));
<PRAGMA>.	=> (error ("ignoring illegal character " ^ yytext
			   ^ " in pragma", (yypos, yypos+1)); continue());
