(*****************************************************************************)
(*          Projet Formel - Calculus of Inductive Constructions V5.8         *)
(*****************************************************************************)
(*									     *)
(*      	Development of Warshall's Algorithm (program)                *)
(*									     *)
(*****************************************************************************)

Require SetTheory.

(******************************************)
(* V is finite : each set on V is finite  *)
(******************************************)
Section Warshall.

Axiom finit_V : (enumerate allV).
Hint finit_V.

(*****************************************)
(* A decidable relation on V : the edges *)
(*****************************************)

Variable E : V->V->Prop.
Variable dec_E : (x,y:V){(E x y)}+{~(E x y)}.
Hint dec_E.

(***********************************************************************)
(* Two vertices are connected in W if there exists a path from x to y  *)
(* with inner vertex in W.                                             *)
(***********************************************************************)

(********************************************************)
(*               Q                                      *)
(*   We write x ---> y   for (connected Q x y)          *)
(********************************************************)

Inductive Definition connected [W:set] : V->V->Prop
        = direct : (x,y:V)(E x y)->(connected W x y)
        | one_in : (x,y,z:V)(E x y)->(In W y)->(connected W y z)
                    ->(connected W x z).

Hint direct.

(************************************************************************)
(*    First Part, general lemmas and naive proof                        *)
(************************************************************************)

(********************************************************)
(*          Lemmas about connected                      *)
(********************************************************)

(****************************************************)
(*        W                                  W'     *)
(*  If x ---> y and W included in W' then x ---> y  *)
(****************************************************)

Lemma connect_incl : (W,W':set)(x,y:V)(incl W W')->(connected W x y)->(connected W' x y).
Goal. 
Intros W W' x y H H0 ; Elim H0 ; Auto ; Intros x0 y0 z H1 H2 H3 H4.
Apply one_in with y0 ; Auto.
(*Apply H ; Auto.*)
Save.

(****************************************************************)
(*  A trivial application to be used as an automatic tactic     *)
(****************************************************************)

Lemma connect_add : (W:set)(x,y,z:V)(connected W y z)->(connected (add W x) y z).
Goal. 
Intros W x y z H ; Apply (connect_incl W) ; Auto.
Save.
Hint connect_add.

(*************************************************************)
(*         W            W                        W           *)
(*   If x ---> y and y ---> z and y in W then x ---> z       *)
(*************************************************************)

Lemma connect_trans : (W:set)(x,y,z:V)
        (connected W x y)->(connected W y z)->(In W y)->(connected W x z).
Goal. 
Intros W x y z H ; Elim H.
Intros x0 y0 H0 H1 H2.
Apply one_in with y0 ; Auto.
Intros x0 y0 z0 H0 H1 H2 H3 H4 H5 ; Apply one_in with y0 ; Auto.
Save.

(*********************************************)
(*     {}              E                     *)
(*   x--->y     then x -> y                  *)
(*********************************************)

Lemma connect_empty : (x,y:V)(connected empty x y)->(E x y).
Goal. 
Intros x y H ; Elim H ; Auto ; Intros x0 y0 z H0 H1 H2 H3.
Elim H1.
Save.
Hint connect_empty.

(********************************************************************)
(*         Q+y              Q             Q            Q            *)
(* if    x ---> z   then x ---> z  or (x ---> y and y --->z)        *)
(********************************************************************)

Lemma connect_lem : (Q:set)(x,y,z:V)(connected (add Q y) x z)
    ->((connected Q x z)\/((connected Q x y)/\(connected Q y z))).
Goal. 
Intros Q x y z H ; Elim H ; Auto ; Intros x0 y0 z0 H0 H1 H2 H3.
Elim H3 ; Intro H4.
Elim H1 ; Intro H5.
Left; Apply one_in with y0 ; Auto.
Rewrite -> H5; Auto.
Elim H4 ; Intros H5 H6.
Cut (connected Q x0 y) ; Auto.
Elim H1 ; Intro H7.
Apply one_in with y0 ; Auto.
Rewrite -> H7; Auto.
Save.


Lemma connect_cut : (Q:set)(x,y,z:V)(~(connected Q x z))->(connected (add Q y) x z)
	->((connected Q x y)/\(connected Q y z)).
Goal. 
Intros Q x y z H H0 ; Elim (connect_lem Q x y z) ; Auto ; Intro H1.
Absurd (connected Q x z) ; Auto.
Save.
Hint connect_cut.

(**********************************************************************)
(*          Specification                                             *)
(**********************************************************************)

(*******************)
(* Case W is empty *)
(*******************)

Lemma warshall_empty : (x,y:V){(connected empty x y)}+{~(connected empty x y)}.
Goal. 
Realizer [x:V][y:V](dec_E x y).
Program_all.
Red ; Intro.
Absurd (E x y) ; Auto.
Save.
Hint warshall_empty.

(*********************************************)
(*	Induction step			     *)
(*********************************************)

Lemma warshall_ind : (Q:set)(z:V)
     ((x,y:V){(connected Q x y)}+{~(connected Q x y)})
     ->(x,y:V){(connected (add Q z) x y)}+{~(connected (add Q z) x y)}.
Goal. 
Realizer [z:V][H:V->V->sumbool][x:V][y:V]
                     (<sumbool>Match (H x y) with
			left
                        (<sumbool>Match (H x z) with
                           (H z y)
                           right)).
Program_all.
Apply connect_trans with z ; Auto.
Red ; Intro H0.
Elim (connect_cut Q x z y b) ; Auto.
Red ; Intro H0.
Elim (connect_cut Q x z y b) ; Auto ; Intros H1 H2.
(*Absurd (connected Q x z) ; Auto.*)
Save.
Hint warshall_ind.

(*********************************************************)
(*  First proof by induction                             *)
(*********************************************************)

Lemma warshall1 : (x,y:V){(connected allV x y)}+{~(connected allV x y)}.
Goal.
Realizer  (<V->V->sumbool>Match finit_V with
                 [x:V][y:V](warshall_empty x y)
                 [y:V][y0:enumerate][H:V->V->sumbool][x:V][y1:V]
			(warshall_ind y H x y1)).
Program_all.
Save.

(************************************************************************)
(*    Second Part, keeping results in a matrice                         *)
(************************************************************************)

(************************************)
(*      Arrays on V		    *)
(************************************)


Section Array.
Variable A: V->Set.

Inductive Definition array : set->Set =
    empty_array : (array empty)
   | add_array  : (P:set)(y:V)(A y)->(array P)->(array (add P y)).

End Array.
Hint empty_array add_array.

(***************)
(* Access      *)
(***************)

Lemma acces : (A:V->Set)(P:set)(array A P)->(x:V)(In P x)->(A x).
Goal.
Realizer  [A:Data]
           [H:(array A)]
            (<V->A>Match H with
	       [x:V](False_rec A)
               [{Q:set}][y:V][H0:A][H1:(array A)][H2:V->A][x:V]
                    (<A>Match (add_null x y) (: (In (add Q y) x)->({(<V>y=x)}+{(In Q x)}) :) with
			 (eq_rec V y A H0 x) 
			 (H2 x))).
Program_all.
Save.

Lemma acces_allV : (A:V->Set)(array A allV)->(x:V)(A x).
Goal. 
Realizer [A:Data][H:(array A)][x:V](acces A H x).
Program_all.
Save.

(*******************)
(* Initialisation  *)
(*******************)

Lemma array_init : (A:V->Set)((y:V)(A y))->(array A allV).
Goal. 
Realizer [A:Data][H:V->A]
                 (<(array A)>Match finit_V with
		   (empty_array A)
                    [y:V][y0:enumerate][H0:(array A)]
			(add_array A y (H y) H0)).
Program_all.
Save.
Hint array_init.

(********************************************************************)
(* Representation of a relation on a finite set by a matrix         *)
(********************************************************************)

Definition repr_matrix 
    [P:V->V->Prop](array [x:V](array [y:V]{(P x y)}+{~(P x y)} allV) allV).

Lemma build_matrix : (P:V->V->Prop)((x,y:V){(P x y)}+{~(P x y)})->(repr_matrix P).
Goal. 
Realizer [H:V->V->sumbool]
             (array_init (array sumbool)
                     [y:V](array_init sumbool [y0:V](H y y0))).
Program_all.
Save.

Lemma acces_matrix : (P:V->V->Prop)(repr_matrix P)->(x,y:V){(P x y)}+{~(P x y)}.
Goal. 
Realizer [H:(array (array sumbool))]
           [x:V]
             [y:V](acces_allV sumbool (acces_allV (array sumbool) H  x) y).

Program_all.  
Save.

Lemma warshall_rep : (repr_matrix (connected allV)).
Goal. 
Realizer (enumerate_rec repr_matrix
                    (build_matrix [x:V][y:V](warshall_empty x y))
                    [y:V]
                     [y0:enumerate]
                      [H:repr_matrix]
                       (build_matrix
                          [x:V]
                           [y1:V]
                            (warshall_ind y
                               [x0:V][y2:V](acces_matrix H x0 y2) x y1))
                    finit_V).
Program_all.
Save.

Theorem warshall2 : (x,y:V){(connected allV x y)}+{~(connected allV x y)}.
Goal. 
Realizer [x:V][y:V](acces_matrix warshall_rep x y).
Program_all.
Save.

End Warshall.

Provide Warshall_prog.