(*****************************************************************************)
(*          Projet Formel - Calculus of Inductive Constructions V5.8         *)
(*****************************************************************************)
(*									     *)
(*      	Development of Warshall's Algorithm (definitions)            *)
(*									     *)
(*      	Christine Paulin-Mohring                                     *)
(*      	after F. Pfenning                                            *)
(*      	Program Development through Proof Transformation             *)
(*****************************************************************************)

Require SetTheory.

(******************************************)
(* V is finite : each set on V is finite  *)
(******************************************)
(*Section Warshall.*)

Axiom finit_V : (enumerate allV).
Hint finit_V.

(*****************************************)
(* A decidable relation on V : the edges *)
(*****************************************)

Variable E : V->V->Prop.
Variable dec_E : (x,y:V){(E x y)}+{~(E x y)}.
Hint dec_E.

(***********************************************************************)
(* Two vertices are connected in W if there exists a path from x to y  *)
(* with inner vertex in W.                                             *)
(***********************************************************************)

(********************************************************)
(*               Q                                      *)
(*   We write x ---> y   for (connected Q x y)          *)
(********************************************************)

Inductive Definition connected [W:set] : V->V->Prop
        = direct : (x,y:V)(E x y)->(connected W x y)
        | one_in : (x,y,z:V)(E x y)->(In W y)->(connected W y z)
                    ->(connected W x z).

Hint direct.

(************************************************************************)
(*    First Part, general lemmas and naive proof                        *)
(************************************************************************)

(********************************************************)
(*          Lemmas about connected                      *)
(********************************************************)

(****************************************************)
(*        W                                  W'     *)
(*  If x ---> y and W included in W' then x ---> y  *)
(****************************************************)

Lemma connect_incl : (W,W':set)(x,y:V)(incl W W')->(connected W x y)->(connected W' x y).
Goal. 
Intros W W' x y H H0 ; Elim H0 ; Auto ; Intros x0 y0 z H1 H2 H3 H4.
Apply one_in with y0 ; Auto.
(*Apply H ; Auto.*)
Save.

(****************************************************************)
(*  A trivial application to be used as an automatic tactic     *)
(****************************************************************)

Lemma connect_add : (W:set)(x,y,z:V)(connected W y z)->(connected (add W x) y z).
Goal. 
Intros W x y z H ; Apply (connect_incl W) ; Auto.
Save.
Hint connect_add.

(*************************************************************)
(*         W            W                        W           *)
(*   If x ---> y and y ---> z and y in W then x ---> z       *)
(*************************************************************)

Lemma connect_trans : (W:set)(x,y,z:V)
        (connected W x y)->(connected W y z)->(In W y)->(connected W x z).
Goal. 
Intros W x y z H ; Elim H.
Intros x0 y0 H0 H1 H2.
Apply one_in with y0 ; Auto.
Intros x0 y0 z0 H0 H1 H2 H3 H4 H5 ; Apply one_in with y0 ; Auto.
Save.

(*********************************************)
(*     {}              E                     *)
(*   x--->y     then x -> y                  *)
(*********************************************)

Lemma connect_empty : (x,y:V)(connected empty x y)->(E x y).
Goal. 
Intros x y H ; Elim H ; Auto ; Intros x0 y0 z H0 H1 H2 H3.
Elim H1.
Save.
Hint connect_empty.

(********************************************************************)
(*         Q+y              Q             Q            Q            *)
(* if    x ---> z   then x ---> z  or (x ---> y and y --->z)        *)
(********************************************************************)

Lemma connect_lem : (Q:set)(x,y,z:V)(connected (add Q y) x z)
    ->((connected Q x z)\/((connected Q x y)/\(connected Q y z))).
Goal. 
Intros Q x y z H ; Elim H ; Auto ; Intros x0 y0 z0 H0 H1 H2 H3.
Elim H3 ; Intro H4.
Elim H1 ; Intro H5.
Left; Apply one_in with y0 ; Auto.
Rewrite -> H5; Auto.
Elim H4 ; Intros H5 H6.
Cut (connected Q x0 y) ; Auto.
Elim H1 ; Intro H7.
Apply one_in with y0 ; Auto.
Rewrite -> H7; Auto.
Save.


Lemma connect_cut : (Q:set)(x,y,z:V)(~(connected Q x z))->(connected (add Q y) x z)
	->((connected Q x y)/\(connected Q y z)).
Goal. 
Intros Q x y z H H0 ; Elim (connect_lem Q x y z) ; Auto ; Intro H1.
Absurd (connected Q x z) ; Auto.
Save.
Hint connect_cut.

(**********************************************************************)
(*          Specification                                             *)
(**********************************************************************)

(*******************)
(* Case W is empty *)
(*******************)

Lemma warshall_empty : (x,y:V){(connected empty x y)}+{~(connected empty x y)}.
Goal. 
Intros x y ; Elim (dec_E x y) ; Auto ; Intro b.
Right ; Red ; Intro.
Absurd (E x y) ; Auto.
Save.
Hint warshall_empty.

(*********************************************)
(*	Induction step			     *)
(*********************************************)

Lemma warshall_ind : (Q:set)(z:V)
     ((x,y:V){(connected Q x y)}+{~(connected Q x y)})
     ->(x,y:V){(connected (add Q z) x y)}+{~(connected (add Q z) x y)}.
Goal. 

Intros Q z H x y ; Elim (H x y) ; Auto ; Intro b.
 (**********************************)
 (* Case ~(connected Q x y)        *)
 (**********************************)
Elim (H x z).
    (**********************************)
    (* Subcase (connected Q x z)      *)
    (**********************************)
    Intro a ; Elim (H z y).
        (**********************************)
        (* Subsubcase (connected Q z y)  *)
        (**********************************)
        Intro a0 ; Left ; Apply connect_trans with z ; Auto.
        (**********************************)
        (* Subsubcase ~(connected Q z y) *)
        (**********************************)
        Intro b0 ; Right ; Red ; Intro H0.
        Elim (connect_cut Q x z y b) ; Auto.
    (**********************************)
    (* Subcase ~(connected Q x y)     *)
    (**********************************)
    Intro b0 ; Right ; Red ; Intro H0.
    Elim (connect_cut Q x z y b) ; Auto ; Intros H1 H2.
(*    Absurd (connected Q x z) ; Auto.*)
Save.
Hint warshall_ind.

(*********************************************************)
(*  First proof by induction                             *)
(*********************************************************)

Lemma warshall1 : (x,y:V){(connected allV x y)}+{~(connected allV x y)}.
Goal. 
Elim finit_V ; Auto.
Save.

(************************************************************************)
(*    Second Part, keeping results in a matrice                         *)
(************************************************************************)

(************************************)
(*      Arrays on V		    *)
(************************************)


Section Array.
Variable A: V->Set.

Inductive Definition array : set->Set =
    empty_array : (array empty)
   | add_array  : (P:set)(y:V)(A y)->(array P)->(array (add P y)).

End Array.
Hint empty_array add_array.

(***************)
(* Access      *)
(***************)

Lemma acces : (A:V->Set)(P:set)(array A P)->(x:V)(In P x)->(A x).
Goal. 
Intros A P H ; Elim H.
Intros x H0; Elim H0 ; Auto.
Intros Q y H0 H1 H2 x H3.
Elim (add_null Q x y); Auto.
Intro a0; Elim a0 ; Auto.
Save.

Lemma acces_allV : (A:V->Set)(array A allV)->(x:V)(A x).
Goal. 
Intros; Apply acces with allV; Auto.
Save.

(*******************)
(* Initialisation  *)
(*******************)

Lemma array_init : (A:V->Set)((y:V)(A y))->(array A allV).
Goal. 
Intros ; Elim finit_V ; Auto.
Save.
Hint array_init.

(********************************************************************)
(* Representation of a relation on a finite set by a matrix         *)
(********************************************************************)

Definition repr_matrix 
    [P:V->V->Prop](array [x:V](array [y:V]{(P x y)}+{~(P x y)} allV) allV).

(* End Warshall. *)

Provide Warshall_def.

