(*****************************************************************************)
(*          Projet Formel - Calculus of Inductive Constructions V5.8         *)
(*****************************************************************************)
(* 	                 						     *)
(*  A tautology checker, following Boyer and Moore : a Computational Logic   *)
(*		               (program)   				     *)
(* 	                 						     *)
(*****************************************************************************)

Require Bool.
Hint diff_true_false.

Require Peano_dec.

(*************************************************************)
(* Formula of propositional logic : definition and semantics *)
(*************************************************************)

Inductive Set PropForm = 
          Fvar : nat -> PropForm
        | Or   : PropForm -> PropForm -> PropForm
        | And  : PropForm -> PropForm -> PropForm
        | Impl : PropForm -> PropForm -> PropForm
        | Neg  : PropForm -> PropForm
        | Bot  : PropForm.

(* Assignment of truth values to propositional variables *)

Definition Assign = nat -> bool.

(* The semantics of a propositional formula under an assignment *)

Definition Prop_sem : Assign -> PropForm -> bool
  = [A:Assign][F:PropForm]
  (<bool> Match F with 
        (* Fvar n *) [n:nat](A n)
        (* Or F G *) [F:PropForm][SF:bool][G:PropForm][SG:bool](orb SF SG)
       (* And F G *) [F:PropForm][SF:bool][G:PropForm][SG:bool](andb SF SG)
      (* Impl F G *) [F:PropForm][SF:bool][G:PropForm][SG:bool](implb SF SG)
         (* Neg F *) [F:PropForm][SF:bool](negb SF)
           (* Bot *) false).

(***********************************************************)
(*  IF expressions : definition and semantics              *)
(***********************************************************)

Inductive Set IFExpr = 
          Var  : nat -> IFExpr
        | Tr   : IFExpr
        | Fa   : IFExpr
        | IfE   : IFExpr -> IFExpr -> IFExpr -> IFExpr.

(* The semantics of an IF formula under an assignment *)

Definition IF_sem = [A:Assign][F:IFExpr]
  (<bool> Match F with
         (* Var n *) [n:nat](A n)
            (* Tr *) true
            (* Fa *) false
      (* IfE F G H *) [F:IFExpr][SF:bool][G:IFExpr][SG:bool][H:IFExpr][SH:bool]
                     (ifb SF SG SH)).

(*****************************************************************)
(* 1- Transformation of a propositional formula into an          *)
(*    equivalent IF-expression.				         *)
(*****************************************************************)

Definition Prop_IF_Equiv : PropForm -> IFExpr -> Prop
  = [F:PropForm][I:IFExpr](A:Assign)<bool>(Prop_sem A F)=(IF_sem A I).

Lemma PropForm_IFExpr : (F:PropForm){I:IFExpr|(Prop_IF_Equiv F I)}.
Goal. 
Realizer [F:PropForm]
	    (<IFExpr>Match F with
                  [n:nat](Var n)
                  [G:PropForm][HG:IFExpr][H:PropForm][HH:IFExpr]
			(IfE HG Tr HH)
                  [G:PropForm][HG:IFExpr][H:PropForm][HH:IFExpr]
                        (IfE HG HH Fa)
                  [G:PropForm][HG:IFExpr][H:PropForm][HH:IFExpr]
                        (IfE HG HH Tr)
                  [G:PropForm][HG:IFExpr]
                        (IfE HG Fa Tr)
                  Fa).
Program_all.
Unfold Prop_IF_Equiv; Simpl; Auto.
Unfold Prop_IF_Equiv; Simpl.
Intro A; Elim p; Elim p0; Trivial.
Unfold Prop_IF_Equiv; Simpl.
Intro A; Elim p; Elim p0; Trivial.
Unfold Prop_IF_Equiv; Simpl.
Intro A; Elim p; Elim p0; Trivial.
Unfold Prop_IF_Equiv; Simpl.
Intro A; Elim p; Trivial.
Unfold Prop_IF_Equiv; Simpl; Trivial.
Save.

(****************************************************************************)
(* Definition of Normal IF expressions.                                     *)
(* The conditional test is restricted to be a propositional variable and    *)
(* not an arbitrary IF expression                                           *)
(****************************************************************************)

Inductive Definition Normal : IFExpr -> Prop
  = NVar : (n:nat)(Normal (Var n))
  | NTr  : (Normal Tr)
  | NFa  : (Normal Fa)
  | NIf  : (n:nat)(F,G:IFExpr)
           (Normal F)->(Normal G)->(Normal (IfE (Var n) F G)).

Hint NVar NTr NFa NIf.
(*******************************************)
(* Properties of Normal                    *)
(*******************************************)

(* The inverse properties for normal *)

Definition Norminv = [F:IFExpr]
    (<Prop> Match F with 
         (* Var n *) [n:nat]True
            (* Tr *) True
            (* Fa *) True
      (* IfE G H I *) [G:IFExpr][PG:Prop][H:IFExpr][PH:Prop][I:IFExpr][PI:Prop]
                     (<Prop> Match G with 
                        (* Var m *) [m:nat]((Normal H)/\(Normal I))
                           (* Tr *) False
			   (* Fa *) False
                  (* IfE G' H' I' *) [G':IFExpr][PG':Prop][H':IFExpr][PH':Prop]
                                    [I':IFExpr][PI':Prop]False)).

Lemma Normal_inv : (F:IFExpr)(Normal F) -> (Norminv F).
Goal. 
Induction 1; Simpl; Auto.
Save.

Lemma not_Norm_If_Tr : (F,G:IFExpr)~(Normal (IfE Tr F G)).
Goal. 
Intros F G; Exact (Normal_inv (IfE Tr F G)).
Save.

Lemma not_Norm_If_Fa : (F,G:IFExpr)~(Normal (IfE Fa F G)).
Goal. 
Intros F G; Exact (Normal_inv (IfE Fa F G)).
Save.

Lemma not_Norm_If_If : (F,G,H,I,J:IFExpr)~(Normal (IfE (IfE F G H) I J)).
Goal. 
Intros F G H I J;  Exact (Normal_inv (IfE (IfE F G H) I J)).
Save.

Lemma Norm_If_Var : (n:nat)(F,G:IFExpr)(Normal (IfE (Var n) F G))->((Normal F)/\(Normal G)).
Goal. 
Intros n F G; Exact (Normal_inv (IfE (Var n) F G)).
Save.

Hint not_Norm_If_Tr not_Norm_If_Fa not_Norm_If_If.

(******************************************************)
(* The proof of Normal corresponds to a partial match *)
(******************************************************)

Lemma Normal_rec : (P:IFExpr->Set)
        ((n:nat)(P (Var n)))->(P Tr)->(P Fa)->
        ((n:nat)(F:IFExpr)(G:IFExpr)(Normal F)->(P F)->
                (Normal G)->(P G)->(P (IfE (Var n) F G)))
        ->(i:IFExpr)(Normal i)->(P i).
Goal. 
Realizer [P:Data][H:nat->P][H0:P][H1:P][H2:nat->IFExpr->IFExpr->P->P->P]
           [i:IFExpr](<P>Match i with
			[n:nat](H n) 
			H0 
			H1
                        [F:IFExpr][PF:P][G:IFExpr][PG:P][I:IFExpr][PI:P]
                              (<P>Match F with
                                 [n:nat](and_rec P (H2 n G I PG PI))
                                 (False_rec P) 
				 (False_rec P)
                                 [y:IFExpr][H3:P][y0:IFExpr][H4:P]
					[y1:IFExpr][H5:P](False_rec P))).
Program_all.
Elim (Norm_If_Var n y0 y1); Auto.
Elim (Norm_If_Var n y0 y1); Auto.
Elim (Norm_If_Var n y0 y1); Auto.
Elim (Norm_If_Var n y0 y1); Auto.
Absurd (Normal (IfE Tr y0 y1)); Auto.
Absurd (Normal (IfE Fa y0 y1)); Auto.
Absurd (Normal (IfE (IfE y2 y3 y4) y0 y1)); Auto.
Save.

(************************************************************************)
(* Representation of partial assignment with two lists of propositional *)
(* variables  whose values will be set either to true or false          *)
(************************************************************************)

Inductive Set Constraints = 
  Emp :  Constraints | Cons : nat -> Constraints -> Constraints.

(* Assigned n P means that n occurs in P *)

Inductive Definition Assigned [n:nat] : Constraints -> Prop 
  = Assign_hd : (PA:Constraints)(Assigned n (Cons n PA))
  | Assign_tl : (m:nat)(PA:Constraints)(Assigned n PA)->
                (Assigned n (Cons m PA)).

Hint Assign_hd Assign_tl.

Lemma Assign_eq : (n,m:nat)(PA:Constraints)(<nat>m=n)->(Assigned n (Cons m PA)).
Goal. 
Induction 1; Auto.
Save.
Hint Assign_eq.

(****************************************)
(* Properties of Assigned               *)
(****************************************)

Definition Assign_inv = [n:nat][PA:Constraints]
 (<Prop> Match PA with 
         (* Emp *) False
   (* Cons m PA *) [m:nat][PA:Constraints][PPA:Prop]
                   (Assigned n PA)\/<nat>m=n).

Lemma Assigned_inv : (n:nat)(PA:Constraints)(Assigned n PA)->(Assign_inv n PA).
Goal. 
Induction 1; Simpl; Auto.
Save.

Lemma not_Ass_emp : (n:nat)~(Assigned n Emp).
Goal. 
Intro n; Exact (Assigned_inv n Emp).
Save.
Hint not_Ass_emp.

Lemma Assign_cons : (n,m:nat)(PA:Constraints)
     (Assigned n (Cons m PA))->((Assigned n PA)\/<nat>m=n).
Goal. 
Intros n m PA; Exact (Assigned_inv n (Cons m PA)).
Save.

Lemma not_Ass_cons_intro : (n,m:nat)(PA:Constraints)
     (~<nat>m=n)->(~(Assigned n PA))->~(Assigned n (Cons m PA)).
Goal. 
Red; Intros n m PA MN NPA AC.
Elim (Assign_cons n m PA); Trivial.
Save.
Hint not_Ass_cons_intro.

Lemma not_Ass_cons_elim : (n,m:nat)(PA:Constraints)
     (~(Assigned n (Cons m PA)))->((~<nat>m=n)/\~(Assigned n PA)).
Goal. 
Unfold not; Auto.
Save.
Hint not_Ass_cons_elim.

(**************************************)
(* The relation Assigned is decidable *)
(**************************************)

Lemma Assign_dec : (n:nat)(PA:Constraints){(Assigned n PA)}+{~(Assigned n PA)}.
Goal. 
Realizer [n:nat][PA:Constraints]
                 (<sumbool>Match PA with
		    right
                    [m:nat][PA':Constraints][HPA:sumbool]
                       (<sumbool>Match HPA with
			  left
                          (<sumbool>Match (eq_nat_dec m n) with
                             [{H:(<nat>m=n)}](eq_rec nat m sumbool left n)
			     right))).
Program_all.
Save.

(**********************************************************************)
(* An assignment A is compatible with a constraint P for the boolean  *)
(* value b if A is b for each variable in P                           *)
(**********************************************************************)

Inductive Definition Compatible [A:Assign;b:bool] : Constraints -> Prop 
 = Compat_emp  : (Compatible A b Emp)
 | Compat_cons : (n:nat)(lb:Constraints)(<bool>b=(A n))
                  ->(Compatible A b lb)->(Compatible A b (Cons n lb)).

Hint Compat_emp Compat_cons.

(**********************************************)
(* Properties of the compatibility predicate  *)
(**********************************************)

Definition Compat_inv = [A:Assign][b:bool][lb:Constraints]
( <Prop> Match lb  with
       (* Emp *)  True
(* Cons n lb' *)  [n:nat][lb':Constraints][P:Prop]
                  ((Compatible A b lb')/\<bool>b=(A n))).

Lemma Compatible_inv : (A:Assign)(b:bool)(PA:Constraints)
     (Compatible A b PA)->(Compat_inv A b PA).
Goal. 
Induction 1; Simpl; Auto.
Save.

Lemma Compat_cons_elim : (A:Assign)(b:bool)(n:nat)(lb:Constraints)
     (Compatible A b (Cons n lb))
   ->((Compatible A b lb)/\(<bool>b=(A n))).
Goal. 
Intros A b n lb; Exact (Compatible_inv A b (Cons n lb)).
Save.

Lemma Compat_assign : (lb:Constraints)(b:bool)(A:Assign)(n:nat)
     (Compatible A b lb)->(Assigned n lb)-><bool>b=(A n).
Goal. 
Induction 1.
Intro; Absurd (Assigned n Emp); Auto.
Intros; Elim (Assign_cons n n0 lb0); Auto.
Induction 1; Auto.
Save.

(* Two partial assignments are coherent if they do not contains 
   the same variables *)
Definition Coherent = 
  [ltrue,lfalse:Constraints]
  (n:nat)(Assigned n ltrue)->~(Assigned n lfalse).

Lemma Coherent_sym : (l,m:Constraints)(Coherent l m)->(Coherent m l).
Goal. 
Red; Intros.
Red; Intros.
Exact (H n H1 H0).
Save.
Immediate Coherent_sym.

Lemma Coh_emp_r : (l:Constraints)(Coherent l Emp).
Goal. 
Red; Red; Intros.
Absurd (Assigned n Emp); Auto.
Save.
Hint Coh_emp_r.

Lemma Coherent_not_Ass_true : (ltrue,lfalse:Constraints)
     (n:nat)(Coherent ltrue lfalse)->(~(Assigned n ltrue))
     ->(Coherent ltrue (Cons n lfalse)).
Goal. 
Unfold Coherent; Intros.
Apply not_Ass_cons_intro; Auto.
Red; Intro; Absurd (Assigned n0 ltrue); Auto.
Elim H2; Auto.
Save.
Hint Coherent_not_Ass_true.

Lemma Coherent_not_Ass_false : (ltrue,lfalse:Constraints)
     (n:nat)(Coherent ltrue lfalse)->(~(Assigned n lfalse))
     ->(Coherent (Cons n ltrue) lfalse).
Goal. 
Intros; Apply Coherent_sym; Auto.
Save.
Hint Coherent_not_Ass_false.

Lemma Coherent_cons_false_elim : (ltrue,lfalse:Constraints)(n:nat)
	(Coherent ltrue (Cons n lfalse))
        ->((Coherent ltrue lfalse)/\~(Assigned n ltrue)).
Goal. 
Unfold Coherent; Intros; Split.
Intros m Am; Elim (not_Ass_cons_elim m n lfalse); Auto.
Red; Intros; Absurd (Assigned n (Cons n lfalse)); Auto.
Save.

(* Proof of if we have two coherent partial assignments
   ltrue and lfalse, we can find an assignment which is compatible
   with ltrue for the value true and lfalse for the value false *)

(* a constant assignment *)

Lemma Compat_Ass_const : (lb:Constraints)(b:bool)(Compatible [n:nat]b b lb).
Goal. 
Induction lb; Auto.
Save.
Hint Compat_Ass_const.

(* Updating one value in an assignment *)
Definition upd_assign : Assign -> nat -> bool -> Assign =
    [A:Assign][n:nat][b:bool][m:nat]
    (<bool> Match (eq_nat_dec n m) with 
	(* Inl H *) [H:<nat>n=m]b
        (* Inr H *) [H:~<nat>n=m](A m)).

Lemma upd_assign_val : (A:Assign)(n:nat)(b:bool)(m:nat)
     (<nat>n=m)-><bool>b=(upd_assign A n b m).
Goal. 
Unfold upd_assign.
Intros; Elim (eq_nat_dec n m); Simpl; Auto.
Intro; Absurd <nat>n=m; Auto.
Save.
Hint upd_assign_val.

Lemma upd_assign_rest : (A:Assign)(n:nat)(b:bool)(m:nat)
     (~<nat>n=m)-><bool>(A m)=(upd_assign A n b m).
Goal. 
Unfold upd_assign.
Intros; Elim (eq_nat_dec n m); Simpl; Auto.
Intro; Absurd <nat>n=m; Auto.
Save.
Hint upd_assign_rest.

Lemma upd_assign_same : (A:Assign)(n:nat)(b:bool)(m:nat)
     (<bool>b=(A m))-><bool>b=(upd_assign A n b m).
Goal. 
Intros; Elim (eq_nat_dec n m); Simpl; Auto.
Intro; Elim upd_assign_rest; Auto.
Save.
Hint upd_assign_same.

Lemma Compat_upd_assign_neg : (lb:Constraints)(A:Assign)(n:nat)(b,b':bool)
     (~<bool>b=b')->(Compatible A b lb)
     ->(~(Assigned n lb))->(Compatible (upd_assign A n b') b lb).
Goal. 
Induction 2; Intros; Trivial.
Elim (not_Ass_cons_elim n n0 lb0); Trivial.
Intros; Apply Compat_cons; Auto.
Elim upd_assign_rest; Auto.
Save.
Hint Compat_upd_assign_neg.

Lemma Compat_upd_assign_Cons : (lb:Constraints)(A:Assign)(n:nat)(b:bool)
     (Compatible A b lb) -> (Compatible (upd_assign A n b) b (Cons n lb)).
Goal. 
Induction 1; Intros.
Auto.
Elim (Compat_cons_elim (upd_assign A n b) b n lb0); Auto.
Save.
Hint Compat_upd_assign_Cons.

Definition Extends : Constraints -> Constraints -> Assign -> Prop
  = [ltrue,lfalse:Constraints][A:Assign]
    (Compatible A true ltrue) /\ (Compatible A false lfalse).

Lemma Extends_intro : (ltrue,lfalse:Constraints)(A:Assign)
     (Compatible A true ltrue)->(Compatible A false lfalse)
     ->(Extends ltrue lfalse A).
Goal. 
Red; Auto.
Save.
Hint Extends_intro.

Lemma Extends_cons_false : (ltrue,lfalse:Constraints)(A:Assign)(n:nat)
     (Extends ltrue lfalse A)->(<bool>false=(A n))
     ->(Extends ltrue (Cons n lfalse) A).
Goal. 
Induction 1; Auto.
Save.
Hint Extends_cons_false.

Lemma Extends_cons_true : (ltrue,lfalse:Constraints)(A:Assign)(n:nat)
     (Extends ltrue lfalse A)->(<bool>true=(A n))
     ->(Extends (Cons n ltrue) lfalse A).
Goal. 
Induction 1; Auto.
Save.
Hint Extends_cons_true.

Lemma Extends_cons_elim_false : (ltrue,lfalse:Constraints)(A:Assign)(n:nat)
     (Extends ltrue (Cons n lfalse) A) ->
     ((Extends ltrue lfalse A) /\ <bool>false=(A n)).
Goal. 
Induction 1; Intros.
Elim (Compat_cons_elim A false n lfalse); Auto.
Save.

Lemma Extends_cons_elim_true : (ltrue,lfalse:Constraints)(A:Assign)(n:nat)
     (Extends (Cons n ltrue) lfalse A) ->
     ((Extends ltrue lfalse A) /\ <bool>true=(A n)).
Goal. 
Induction 1; Intros.
Elim (Compat_cons_elim A true n ltrue); Auto.
Save.

Lemma Extends_assign_false : (ltrue,lfalse:Constraints)(A:Assign)(n:nat)
     (Extends ltrue lfalse A) -> (~(Assigned n ltrue)) 
     -> (Extends ltrue (Cons n lfalse) (upd_assign A n false)).
Goal. 
Red; Induction 1; Auto.
Save.
Hint Extends_assign_false.

Lemma Coherent_Compat : (ltrue,lfalse:Constraints)(Coherent ltrue lfalse)
      ->{A:Assign|(Extends ltrue lfalse A)}.
Goal. 
Realizer [ltrue:Constraints][lfalse:Constraints]
                      (<(sig Assign)>Match lfalse with
                         (exist Assign ([n:nat]true))
                         [n:nat][lfalse':Constraints][Hf:(sig Assign)]
                            (and_rec (sig Assign)
				(<(sig Assign)>let (A:Assign) = Hf in
				     (exist Assign (upd_assign A n false))))).
Program_all.
Elim (Coherent_cons_false_elim ltrue y n); Trivial.
Intros; Auto.
Elim (Coherent_cons_false_elim ltrue y n); Trivial.
Save.
Hint Coherent_Compat.

Definition Sem : Assign -> IFExpr -> bool -> Prop
               = [A:Assign][F:IFExpr][b:bool]<bool>b=(IF_sem A F).

Lemma Sem_Tr : (A:Assign)(Sem A Tr true).
Goal. 
Red; Auto.
Save.
Hint Sem_Tr.

Lemma Sem_Fa : (A:Assign)(Sem A Fa false).
Goal. 
Red; Auto.
Save.
Hint Sem_Fa.

Lemma Sem_If : (A:Assign)(F,G:IFExpr)(n:nat)(b:bool)
        (((<bool>true=(A n)) -> (Sem A F b)) 
          /\ ((<bool>false=(A n)) -> (Sem A G b)))
     -> (Sem A (IfE (Var n) F G) b).
Goal. 
Red; Simpl; Induction 1.
Elim (A n); Simpl; Intros.
Red in H0; Auto.
Red in H1; Auto.
Save.
Hint Sem_If.


Definition Tautology : IFExpr -> Prop
           = [F:IFExpr](A:Assign)(Sem A F true).

Definition Refutable : IFExpr -> Prop
           = [F:IFExpr]<Assign>Ex([A:Assign](Sem A F false)).

Definition Part_Tauto : IFExpr -> Constraints -> Constraints -> Prop
           = [F:IFExpr][ltrue,lfalse:Constraints]
             (A:Assign)(Extends ltrue lfalse A)->(Sem A F true).

Definition Part_Refut : IFExpr -> Constraints -> Constraints -> Prop
           = [F:IFExpr][ltrue,lfalse:Constraints]
            <Assign>Ex([A:Assign]((Sem A F false)/\(Extends ltrue lfalse A))).

Lemma Part_Refut_intro : (F:IFExpr)(ltrue,lfalse,ltrue',lfalse':Constraints)
     (Coherent ltrue lfalse)->
     ((A:Assign)(Extends ltrue lfalse A)->
                ((Sem A F false)/\(Extends ltrue' lfalse' A)))
     -> (Part_Refut F ltrue' lfalse').
Goal. 
Intros; Elim (Coherent_Compat ltrue lfalse); Trivial.
Intros A EA; Red; Exists A; Auto.
Save.

(* Useful lemmas *)

(* (Part_Equiv F G ltrue lfalse) means that the two formulas F and G
   are equivalent under all assignemnts which extend ltrue,lfalse *)

Definition Part_Equiv = 
     [F,G:IFExpr][ltrue,lfalse:Constraints]
     (A:Assign)(Extends ltrue lfalse A)-><bool>(IF_sem A F)=(IF_sem A G).

Lemma Part_Equiv_Sem : (F,G:IFExpr)(ltrue,lfalse:Constraints)
     (Part_Equiv F G ltrue lfalse)->(A:Assign)(Extends ltrue lfalse A)
     ->(b:bool)(Sem A F b)->(Sem A G b).
Goal. 
Red; Intros.
Elim H; Auto.
Save.

(* If F and G are equivalent under all assignment which extends
   ltrue, lfalse then the partial tautology problem is the same
   for both formulas *)

Lemma Part_Equiv_Tauto : (F,G:IFExpr)(ltrue,lfalse:Constraints)
     (Part_Equiv F G ltrue lfalse)
     ->({Part_Tauto F ltrue lfalse}+{Part_Refut F ltrue lfalse})
     ->({Part_Tauto G ltrue lfalse}+{Part_Refut G ltrue lfalse}).
Goal. 
Realizer [F:IFExpr][G:IFExpr][ltrue:Constraints][lfalse:Constraints]
		[H0:sumbool]H0.
Program_all.
Red; Intros.
Apply Part_Equiv_Sem with F ltrue lfalse; Auto.
(*Apply a; Auto.*)
Elim b; Intros.
Red; Exists x.
Elim H1; Split; Trivial.
Apply Part_Equiv_Sem with F ltrue lfalse; Auto.
Save.

(* Some equivalent formulas *)

Lemma Part_Equiv_Var_true : (n:nat)(ltrue,lfalse:Constraints)
     (Assigned n ltrue)->(Part_Equiv Tr (Var n) ltrue lfalse).
Goal. 
Red; Intros; Simpl.
Apply Compat_assign with ltrue; Auto.
Elim H0; Auto.
Save.
Hint Part_Equiv_Var_true.

Lemma Part_Equiv_If_true : (n:nat)(F,G:IFExpr)(ltrue,lfalse:Constraints)
     (Assigned n ltrue)->(Part_Equiv F (IfE (Var n) F G) ltrue lfalse).
Goal. 
Red; Intros; Simpl.
Replace (A n) with true; Auto.
Apply Compat_assign with ltrue; Auto.
Elim H0; Auto.
Save.
Hint Part_Equiv_If_true.

Lemma Part_Equiv_If_false : (n:nat)(F,G:IFExpr)(ltrue,lfalse:Constraints)
     (Assigned n lfalse)->(Part_Equiv G (IfE (Var n) F G) ltrue lfalse).
Goal. 
Red; Intros; Simpl.
Replace (A n) with false; Auto.
Apply Compat_assign with lfalse; Auto.
Elim H0; Auto.
Save.
Hint Part_Equiv_If_false.

(* We solve the problem for all normal formulas and 
   coherent partial assignments *)

Lemma Partial_Tauto : (F:IFExpr)(Normal F)->
     (ltrue,lfalse:Constraints)
     (Coherent ltrue lfalse)
     ->({Part_Tauto F ltrue lfalse}+{Part_Refut F ltrue lfalse}).
Goal. 
Realizer [F:IFExpr][{H:(Normal F)}]
                   (Normal_rec Constraints->Constraints->sumbool
                      [n:nat][ltrue:Constraints][lfalse:Constraints]
                         (<sumbool>Match (Assign_dec n ltrue) with
                            (Part_Equiv_Tauto Tr (Var n) ltrue lfalse left)
                            right)
                      [ltrue:Constraints][lfalse:Constraints]left
                      [ltrue:Constraints][lfalse:Constraints]right
                      [n:nat][F':IFExpr][G':IFExpr]
                         [HF:Constraints->Constraints->sumbool]
                          [HG:Constraints->Constraints->sumbool]
                           [ltrue:Constraints][lfalse:Constraints]
                             (<sumbool>Match (Assign_dec n ltrue) with
                                (Part_Equiv_Tauto F'
                                   (IfE (Var n) F' G') ltrue lfalse
                                   (HF ltrue lfalse))
                                (<sumbool>Match (Assign_dec n lfalse) with
                                   (Part_Equiv_Tauto G'
                                      (IfE (Var n) F' G') ltrue lfalse
                                      (HG ltrue lfalse))
                                (<sumbool>Match (HF (Cons n ltrue) lfalse) with
                                         (HG ltrue (Cons n lfalse))
                                         right)))
                      F).
Program_all.
Red ; Auto.
Apply Part_Refut_intro with ltrue (Cons n lfalse); Auto.
Intros; Elim (Extends_cons_elim_false ltrue lfalse A n); Auto.
Red; Auto.
Apply Part_Refut_intro with ltrue lfalse; Auto.
Red.  
Intros; Apply Sem_If.
Split; Intro.
Apply a; Auto.
Apply a0; Auto.
Elim b1; Intros A SA.
Red; Exists A.
Elim SA; Intros.
Elim (Extends_cons_elim_false ltrue lfalse A n); Trivial.
Split; Trivial.
Apply Sem_If.
Split; Trivial.
Elim H8; Intro; Absurd <bool>true=false; Auto.
Elim b1; Intros A SA.
Red; Exists A.
Elim SA; Intros.
Elim (Extends_cons_elim_true ltrue lfalse A n); Trivial.
Split; Trivial.
Apply Sem_If.
Split; Trivial.
Elim H8; Intro; Absurd <bool>true=false; Auto.
Save.

(* The final program corresponds to the case ltrue=lfalse=empty *)
Lemma Norm_Tautology : (F:IFExpr)(Normal F)->{(Tautology F)}+{(Refutable F)}.
Goal. 
Realizer [F:IFExpr]
                    (sumbool_rec sumbool left right
                       (Partial_Tauto F Emp Emp)).
Program_all.
Red.
Intro; Apply a; Auto.
Red.
Elim b; Intros A EA.
Exists A; Elim EA; Auto.
Save.

(****************************************************)
(* Each IF formula admits an equivalent Normal form *)
(****************************************************)

Definition Equiv = [F,G:IFExpr](A:Assign)<bool>(IF_sem A F)=(IF_sem A G).

Lemma Equiv_If_Tr : (F,G:IFExpr)(Equiv (IfE Tr F G) F).
Goal. 
Unfold Equiv; Auto.
Save.

Lemma Equiv_If_Fa : (F,G:IFExpr)(Equiv (IfE Fa F G) G).
Goal. 
Unfold Equiv; Auto.
Save.

Lemma Equiv_If_If : (F,G,H,F',G':IFExpr)(Equiv (IfE (IfE F G H) F' G')
                                (IfE F (IfE G F' G') (IfE H F' G'))).
Goal. 
Unfold Equiv; Simpl; Intros.
Elim (IF_sem A F); Unfold 2 3 ifb; Auto.
Save.
Hint Equiv_If_If.

Lemma Equiv_refl : (F:IFExpr)(Equiv F F).
Goal. 
Unfold Equiv;Auto.
Save.
Hint Equiv_refl.

Lemma Equiv_trans : (F,G,H:IFExpr)(Equiv F G)->(Equiv G H)->(Equiv F H).
Goal. 
Unfold Equiv; Intros.
Transitivity (IF_sem A G); Auto.
Save.

Lemma Equiv_congr : (F,G,H,F',G',H':IFExpr)
     (Equiv F F')->(Equiv G G')->(Equiv H H')->
     (Equiv (IfE F G H) (IfE F' G' H')).
Goal. 
Unfold Equiv; Intros; Simpl.
Elim H0; Elim H1; Elim H2; Auto.
Save.
Hint Equiv_congr.

(* The well-founded relation should satisfy :
                                G < IfE Tr G H
                                H < IfE Fa G H
                                G,H < (IfE Var G H)
   (IfE F (IfE G F' G') (IfE H F' G')) < (IfE (IfE F G H) F' G')
The measure is m : IFExpr -> nat
   such that m(Tr)=m(Fa)=m(Var(n))=1
        and  m(IfE F G H)=m(F)*(S(m(G)+m(H)))

*)

Lemma IFExpr_norm_rec1 : (P:IFExpr->Set)
     (P Tr) -> (P Fa) -> ((n:nat)(P (Var n)))
     -> ((Y,Z:IFExpr)(P Y)->(P (IfE Tr Y Z)))
     -> ((Y,Z:IFExpr)(P Z)->(P (IfE Fa Y Z)))
     -> ((n:nat)(Y,Z:IFExpr)(P Y)->(P Z)->(P (IfE (Var n) Y Z)))
     -> ((X,Y,Z,T,U:IFExpr)(P (IfE X (IfE T Y Z) (IfE U Y Z)))
                           ->(P (IfE (IfE X T U) Y Z)))
     -> (X:IFExpr)(P X).
Goal. 
Realizer [P:Data][H:P][H0:P][H1:nat->P][H2:IFExpr->IFExpr->P->P]
           [H3:IFExpr->IFExpr->P->P][H4:nat->IFExpr->IFExpr->P->P->P]
             [H5:IFExpr->IFExpr->IFExpr->IFExpr->IFExpr->P->P]
               [X:IFExpr](<P>Match X with
			      H1	
			      H
			      H0
                              [X':IFExpr][PX':P]
                                (<IFExpr->P->IFExpr->P->P>Match X' with
                                      [n:nat][y0:IFExpr][H6:P][y1:IFExpr][H7:P]
					(H4 n y0 y1 H6 H7)
                                      [y0:IFExpr][H6:P][y1:IFExpr][H7:P]
					(H2 y0 y1 H6)
                                      [y0:IFExpr][H6:P][y1:IFExpr][H7:P]
					(H3 y0 y1 H7)
                                      [y:IFExpr][H6:IFExpr->P->IFExpr->P->P]
                                       [y0:IFExpr][H7:IFExpr->P->IFExpr->P->P]
                                        [y1:IFExpr][H8:IFExpr->P->IFExpr->P->P]
                                         [y2:IFExpr][H9:P][y3:IFExpr][H10:P]
                                                (H5 y y2 y3 y0 y1
                                                   (H6 (IfE y0 y2 y3)
                                                      (H7 y2 H9 y3 H10)
                                                      (IfE y1 y2 y3)
                                                      (H8 y2 H9 y3 H10))))).
Program_all.
Save.

Lemma IFExpr_norm_ind : (P:IFExpr->Prop)
     (P Tr) -> (P Fa) -> ((n:nat)(P (Var n)))
     -> ((Y,Z:IFExpr)(P Y)->(P (IfE Tr Y Z)))
     -> ((Y,Z:IFExpr)(P Z)->(P (IfE Fa Y Z)))
     -> ((n:nat)(Y,Z:IFExpr)(P Y)->(P Z)->(P (IfE (Var n) Y Z)))
     -> ((X,Y,Z,T,U:IFExpr)(P (IfE X (IfE T Y Z) (IfE U Y Z)))
                           ->(P (IfE (IfE X T U) Y Z)))
     -> (X:IFExpr)(P X).
Goal. 
Induction X; Trivial.
Intros X' PX'; Elim X'; Auto.
Save.

Definition norm_order : IFExpr->IFExpr->Prop =
     [X,Y:IFExpr]
    (<Prop>Match Y with 
	(* Var n *) [n:nat]False
           (* Tr *) False
           (* Fa *) False
     (* IfE T U V *) [T:IFExpr][PT:Prop][U:IFExpr][PU:Prop][V:IFExpr][PV:Prop]
                    (<Prop>Match T with 
                     (* Var n *) [n:nat](<IFExpr>U=X) \/ (<IFExpr>V=X)
                        (* Tr *) <IFExpr>U=X
                        (* Fa *) <IFExpr>V=X
                  (* IfE P Q R *) [P:IFExpr][PP:Prop][Q:IFExpr][PQ:Prop]
                                 [R:IFExpr][PR:Prop]
                                 <IFExpr>(IfE P (IfE Q U V) (IfE R U V))=X)).

Require Wf.

Lemma Wf_norm_order : (well_founded IFExpr norm_order).
Goal. 
Red; Intro.
Apply IFExpr_norm_ind; Intros; Apply Acc_intro; Simpl.
Induction 1.
Induction 1.
Induction 1.
Induction 1; Trivial.
Induction 1; Trivial.
Induction 1; Induction 1; Trivial.
Induction 1; Trivial.
Save.
Hint Wf_norm_order.

Lemma IFExpr_match : (P:IFExpr->Set)
        ((n:nat)(P (Var n)))->(P Tr)->(P Fa)->((X,Y,Z:IFExpr)(P (IfE X Y Z)))
      ->(I:IFExpr)(P I).
Goal. 
Realizer [P:Data][H:nat->P][H0:P][H1:P][H2:IFExpr->IFExpr->IFExpr->P][I:IFExpr]
            (<P>Match I with
	       H 
	       H0
	       H1
               [y:IFExpr][H3:P][y0:IFExpr][H4:P][y1:IFExpr][H5:P](H2 y y0 y1)).
Program_all.
Save.

(* A VOIR - RECURSIVITE *)
Lemma IFExpr_norm_rec2 : (P:IFExpr->Set)
     (P Tr) -> (P Fa) -> ((n:nat)(P (Var n)))
     -> ((Y,Z:IFExpr)(P Y)->(P (IfE Tr Y Z)))
     -> ((Y,Z:IFExpr)(P Z)->(P (IfE Fa Y Z)))
     -> ((n:nat)(Y,Z:IFExpr)(P Y)->(P Z)->(P (IfE (Var n) Y Z)))
     -> ((X,Y,Z,T,U:IFExpr)(P (IfE X (IfE T Y Z) (IfE U Y Z)))
                           ->(P (IfE (IfE X T U) Y Z)))
     -> (X:IFExpr)(P X).
Goal.
(* 
Realizer [P:Data][H:P][H0:P][H1:nat->P][H2:IFExpr->IFExpr->P->P]
           [H3:IFExpr->IFExpr->P->P][H4:nat->IFExpr->IFExpr->P->P->P]
              [H5:IFExpr->IFExpr->IFExpr->IFExpr->IFExpr->P->P][X:IFExpr]
                (well_founded_induction IFExpr P
                      [F':IFExpr]
                        (IFExpr_match (IFExpr->P)->P
                              [n:nat][H6:IFExpr->P](H1 n)
                              [H6:IFExpr->P]H [H6:IFExpr->P]H0
                              [Y:IFExpr][Z:IFExpr][T:IFExpr]
                                 (IFExpr_match (IFExpr->P)->P
                                      [n:nat][H6:IFExpr->P]
                                             (H4 n Z T (H6 Z) (H6 T))
                                      [H6:IFExpr->P](H2 Z T (H6 Z))
                                      [H6:IFExpr->P](H3 Z T (H6 T))
                                      [X0:IFExpr][Y0:IFExpr][Z0:IFExpr]
                                         [H6:IFExpr->P]
                                            (H5 X0 Z T Y0 Z0
                                                  (H6 (IfE X0
                                                        (IfE Y0 Z T)
                                                        (IfE Z0 Z T))))
                                           Y)
                                     F')
                                 X).
*)
Intros; Apply well_founded_induction with IFExpr norm_order; Trivial.
Intros F'; Pattern F'; Apply IFExpr_match; Trivial.
Intros Y Z T; Pattern Y; Apply IFExpr_match; Simpl; Auto.
Save.

Variable IFExpr_norm_rec :
(P:IFExpr->Set)
     (P Tr) -> (P Fa) -> ((n:nat)(P (Var n)))
     -> ((Y,Z:IFExpr)(P Y)->(P (IfE Tr Y Z)))
     -> ((Y,Z:IFExpr)(P Z)->(P (IfE Fa Y Z)))
     -> ((n:nat)(Y,Z:IFExpr)(P Y)->(P Z)->(P (IfE (Var n) Y Z)))
     -> ((X,Y,Z,T,U:IFExpr)(P (IfE X (IfE T Y Z) (IfE U Y Z)))
                           ->(P (IfE (IfE X T U) Y Z)))
     -> (X:IFExpr)(P X).

Theorem Norm_prog : (F:IFExpr){G:IFExpr | (Normal G) & (Equiv F G)}.
Goal. 

Realizer [F:IFExpr]
            (IFExpr_norm_rec IFExpr
                Tr
                Fa
                [n:nat](Var n)
                [Y:IFExpr][Z:IFExpr][H:IFExpr] H
                [Y:IFExpr][Z:IFExpr][H:IFExpr] H
                [n:nat][Y:IFExpr][Z:IFExpr][H:IFExpr][H0:IFExpr]
                        (IfE (Var n) H H0)
                [X:IFExpr][Y:IFExpr][Z:IFExpr][T:IFExpr][U:IFExpr][H:IFExpr] H
             F).
Program_all.
Apply Equiv_trans with (IfE X (IfE T Y Z) (IfE U Y Z)); Auto.
Save.

Provide Tauto_prog.
