(*****************************************************************************)
(*      Coq V5.8                                                             *)
(*****************************************************************************)
(*                                                                           *)
(*      Ramsey Theory                                                        *)
(*                                                                           *)
(*      Marc Bezem                                                           *)
(*      Utrecht University                                                   *)
(*                                                                           *)
(*      June 1993                                                            *)
(*                                                                           *)
(*****************************************************************************)

(* For dimension one, the Infinite Ramsey Theorem states that, 
   for any subset A of the natural numbers nat, either A or nat\A 
   is infinite. This special case of the Pigeon Hole Principle
   is classically equivalent to:
   if A and B are both co-finite, then so is their intersection.
   None of these principles is constructively valid. In [VB] the notion
   of an almost full set is introduced, classically equivalent
   to co-finiteness, for which closure under finite intersection can
   be proved constructively. A is almost full if for every (strictly) 
   increasing sequence f: nat -> nat there exists an x in nat such
   that f(x) in A. The notion of almost full and its closure under
   finite intersection are generalized to all finite dimensions,
   yielding constructive Ramsey Theorems. The proofs for dimension
   two and higher essentially use Brouwer's Bar Theorem.

   In the proof development below we strengthen the notion of almost full
   for dimension one in the following sense. A: nat -> Prop is called 
   Y-full if for every (strictly) increasing sequence f: nat -> nat 
   we have (A (f (Y f)). Here of course Y : (nat -> nat) -> nat.
   Given YA-full A and YB-full B we construct X from YA and YB
   such that the intersection of A and B is X-full. 
   This is essentially [VB, Th. 5.4], but now it 
   can be done without using axioms, using only inductive types. 
   The generalization to higher dimensions will be much more
   difficult and is not pursued here.

   [VB] Wim Veldman and Marc Bezem, Ramsey's Theorem and the Pigeon Hole 
        Principle in intuitionistic mathematics, Journal of the London 
        Mathematical Society (2), Vol. 47, April 1993, pp. 193-211.
*)

Require Lt.
Require Plus.

(* To add to package Plus *)
Lemma lt_reg_r (x,y,z:nat)(lt x y) -> (lt (plus x z) (plus y z)).
Goal.
Intros; Rewrite plus_sym.
Replace (plus x z) with (plus z x); Auto.
Save.

Definition increasing = 
    [f: nat -> nat](x,y:nat)(lt x y) -> (lt (f x) (f y)).

Lemma compose_increasing : 
    (f,g:nat->nat)(increasing f) -> (increasing g) ->
	          (increasing [x:nat](f (g x))).
Goal.
Unfold increasing; Auto.
Save.
Hint compose_increasing.

Lemma increasingbystep :
    (f:nat->nat)((n:nat)(lt (f n) (f (S n)))) -> (increasing f).
Goal.
Unfold increasing; Intros f i x y ltxy.
Elim ltxy; Trivial.
Intros; Apply lt_trans with (f m); Auto.
Save.

(* A is Y-full : (full A Y) *)

Definition full = 
  [A:nat->Prop][Y:(nat->nat)->nat](f:nat->nat)(increasing f)->(A (f (Y f))).

Definition enumerate = 
    [Y : (nat -> nat) -> nat][x:nat](<nat>Match x with 
        (Y [n:nat]n)
	[y,z:nat](plus (Y ([n:nat](plus n (S z)))) (S z))).

Lemma increasing_enumerate : (Y:(nat->nat)->nat)(increasing (enumerate Y)).
Goal.
Intro; Apply increasingbystep; Unfold enumerate; Auto.
Save.

Section dimension_one.

Variable A: nat -> Prop.
Variable YA: (nat -> nat) -> nat.

Definition FYA = [x,n:nat](plus n (S (enumerate YA x))).

Lemma increasing_FYA : (x:nat)(increasing (FYA x)).
Goal.
Unfold increasing FYA.
Intros; Apply lt_reg_r; Trivial.
Save.
Hint increasing_FYA.

Lemma enumerate_YA : (full A YA) -> (x:nat)(A (enumerate YA x)).
Goal.
Intro YAfull; Unfold enumerate; Induction x.
Apply (YAfull [n:nat]n).
Unfold increasing; Trivial.
Intros y H.
Change (A (FYA y (YA (FYA y)))).
Apply YAfull; Auto.
Save.

Variable B: nat -> Prop.
Variable YB: (nat -> nat) -> nat.

Lemma YB_enumerate_YA : (full B YB) -> (B (enumerate YA (YB (enumerate YA)))).
Goal.
Intro YBfull.
Apply YBfull.
Apply increasing_enumerate.
Save.

Lemma pre_Ramsey1 : (full A YA) -> (full B YB) ->
        (A (enumerate YA (YB (enumerate YA)))) 
     /\ (B (enumerate YA (YB (enumerate YA)))).
Goal.
Intros YAfull YBfull; Split.
Apply enumerate_YA; Trivial.
Apply YB_enumerate_YA; Trivial.
Save.

End dimension_one.

Definition inter = [A,B:nat->Prop][n:nat](A n) /\ (B n).

Definition combine = [YA,YB:(nat -> nat) -> nat]
       [f:nat->nat](enumerate [g:nat -> nat](YA ([x:nat](f (g x))))
                              ([g:nat -> nat](YB ([x:nat](f (g x))))
                   (enumerate ([g:nat -> nat](YA ([x:nat](f (g x)))))))).

Theorem Ramsey1 :	
      (A,B : nat -> Prop)(YA,YB : (nat -> nat) -> nat)
      (full A YA) -> (full B YB) -> (full (inter A B) (combine YA YB)).
Goal.
Unfold full inter combine; Intros A B YA YB FAYA FBYB f If.
Apply (pre_Ramsey1 [x:nat](A (f x))
                   [g:nat->nat](YA [x:nat](f (g x)))
                   [x:nat](B (f x))
                   [g:nat->nat](YB [x:nat](f (g x))));
  Unfold full; Intros g Ig.
Apply (FAYA [x:nat](f (g x))); Auto.
Apply (FBYB [x:nat](f (g x))); Auto.
Save.




