(*****************************************************************************)
(*          Projet Formel - Calculus of Inductive Constructions V5.8         *)
(*****************************************************************************)
(*                                                                           *)
(*                        The Gilbreath Trick                                *)
(*                                                                           *)
(*****************************************************************************)
(*                                                                           *)
(*                         Gerard Huet - May 1991                            *)
(*                                                                           *)
(*****************************************************************************)

Require Bool.

(*****************)
(* Boolean words *)
(*****************)

Inductive Set word = empty : word 
                    | bit : bool -> word -> word.

(* Remark : word ~ bool list *)

Definition Nonempty = [w:word](<Prop>Match w with
    (* empty *)   False
    (* bit b w *) [b:bool][w:word][P:Prop]True).


(* word concatenation : logical definition *)
Inductive Definition conc : word -> word -> word -> Prop =
  conc_empty : (v:word)(conc empty v v)
| conc_bit   : (u,v,w:word)(b:bool)(conc u v w)->(conc (bit b u) v (bit b w)).

(* word concatenation : functional definition *)
Definition append = [u,v:word](<word>Match u with
    (* empty *)   v
    (* bit b w *) [b:bool][w:word][conc_w_v:word](bit b conc_w_v)).

(* Relating the two definitions; unused below *)
Lemma conc_append : (u,v,w:word)(conc u v w)-><word>w=(append u v).
Goal.
Induction 1; Simpl; Trivial.
(* (u,v,w:word)(b:bool)
   (conc u v w)->(<word>w=(append u v))->(<word>(bit b w)=(bit b (append u v))) *)
Induction 2; Trivial.
Save.

(* Associativity of append; unused below *)
Lemma assoc_append : (u,v,w:word)<word>(append u (append v w))=(append (append u v) w).
Goal.
Induction u; Simpl; Intros; Auto.
(*  <word>(bit b (append y (append v w)))=(bit b (append (append y v) w))
  ============================
    w : word
    v : word
    H : (v:word)(w:word)(<word>(append y (append v w))=(append (append y v) w)) *)
Rewrite H; Trivial.
Save.


(**************)
(* Singletons *)
(**************)

Definition single = [b:bool](bit b empty).


(*********************)
(* Alternating words *)
(*********************)

(* (alt b w) == w = [b ~b b ~b ...]  *)

Inductive Definition alt : bool -> word -> Prop =
  alt_empty   : (b:bool)(alt b empty)
| alt_bit     : (b:bool)(w:word)(alt (neg b) w)->(alt b (bit b w)).

Hint alt_empty alt_bit.

Definition Alt = [b:bool][w:word](<Prop>Match w with
    (* empty *)   True
    (* bit b w *) [b':bool][w:word][P:Prop](alt (neg b) w)).
(* (alt b w) implies (Alt b w) *)

Lemma alt_neg_intro : (b,b':bool)(w:word)(alt b (bit b' w))->(alt (neg b) w).
Goal.
Intros b b' w al.
Change (Alt b (bit b' w)).
Elim al; Simpl; Trivial.
Save.

Lemma alt_neg_elim : (b,b':bool)(w:word)(alt (neg b) (bit b' w))->(alt b w).
Goal. 
Intros; Elim (neg_intro b); Apply alt_neg_intro with b'; Trivial.
Save.

Definition Alt' = [b:bool][w:word](<Prop>Match w with
    (* empty *)   True
    (* bit b w *) [b':bool][w:word][P:Prop]<bool>b=b').

(* (alt b w) implies (Alt' b w) *)

Lemma alt_eq : (b,b':bool)(w:word)(alt b (bit b' w))-><bool>b=b'.
Goal.
Intros b b' w al.
Change (Alt' b (bit b' w)).
Elim al; Simpl; Trivial.
Save.

Lemma alt_back : 
(b,b':bool)(w:word)(alt b (bit b' w))->((<bool>b=b') /\ (alt (neg b) w)).
Goal.
Intros; Split.
Apply alt_eq with w; Trivial.
Apply alt_neg_intro with b'; Trivial.
Save.

Inductive Definition alternate [w:word] : Prop =
  alter : (b:bool)(alt b w)->(alternate w).

(* (alternate w) iff Exists b  (alt b w)  *)


(*********************)
(* Parities of words *)
(*********************)

Inductive Definition odd : word -> Prop =
  odd_single : (b:bool)(odd (single b))
| odd_bit    : (w:word)(odd w)->(b,b':bool)(odd (bit b (bit b' w))).

Definition Odd = [w:word](<Prop>Match w with
    (* empty *)   False
    (* bit b w *) [b:bool][w:word][P:Prop](
      <Prop>Match w with
         (* empty *)   True
         (* bit b w *) [b:bool][w:word][P:Prop](odd w))).

(* (Odd w) iff (odd w) *)

Lemma not_odd_empty : ~(odd empty).
Goal.
Unfold not; Intro od.
Change (Nonempty empty).
Elim od; Simpl; Trivial.
Save.

Hint not_odd_empty.

Lemma odd_down : (w:word)(b,b':bool)(odd (bit b (bit b' w)))->(odd w).
Goal.
Intros w b b' od.
Change (Odd (bit b (bit b' w))).
Elim od; Simpl; Trivial.
Save.

Inductive Definition even : word -> Prop =
  even_empty : (even empty)
| even_bit : (w:word)(even w)->(b,b':bool)(even (bit b (bit b' w))).

Hint even_empty.

Definition Even = [w:word](<Prop>Match w with
    (* empty *)   True
    (* bit b w *) [b:bool][w:word][P:Prop](<Prop>Match w with
         (* empty *)   False
         (* bit b w *) [b:bool][w:word][P:Prop](even w))).

(* (Even w) iff (even w) *)

Definition Not_single = [w:word](<Prop>Match w with
    (* empty *)   True
    (* bit b w *) [b:bool][w:word][P:Prop](
      <Prop>Match w with
         (* empty *)   False
         (* bit b w *) [b:bool][w:word][P:Prop]True)).

Lemma not_even_single : (b:bool)~(even (single b)).
Goal.
Intro b; Unfold not; Intro ev.
Change (Not_single (single b)).
Elim ev; Simpl; Trivial.
Save.

Lemma even_down : (w:word)(b,b':bool)(even (bit b (bit b' w)))->(even w).
Goal.
Intros w b b' ev.
Change (Even (bit b (bit b' w))).
Elim ev; Simpl; Trivial.
Save.

Lemma odd_even : (w:word)(odd w)->(b:bool)(even (bit b w)).
Goal.
Induction 1.
Intros b b'; Unfold single; Apply even_bit; Apply even_empty.
Intros; Apply even_bit; Auto.
Save.

Lemma even_odd : (w:word)(even w)->(b:bool)(odd (bit b w)).
Goal.
Induction 1.
Intro b; Change (odd (single b)); Apply odd_single.
Intros; Apply odd_bit; Auto.
Save.

Hint odd_even even_odd.

Lemma inv_odd : (w:word)(b:bool)(odd (bit b w))->(even w).
Goal.
Induction w; Auto.
Intros; Apply odd_even; Apply odd_down with b0 b; Auto.
Save.

Lemma inv_even : (w:word)(b:bool)(even (bit b w))->(odd w).
Goal.
Induction w; Intros.
Absurd (even (single b)); Trivial.
Apply not_even_single.
Apply even_odd.
Apply even_down with b0 b; Trivial.
Save.

(**********************)
(* (odd w) + (even w) *)
(**********************)

Lemma odd_or_even : (w:word)((odd w) \/ (even w)).
Goal.
Induction w; Auto.
Induction 1; Intros.
Right; Auto.
Left; Auto.
Save.

Lemma not_odd_and_even : (w:word)(odd w)->(even w)->False.
Goal.
Induction w; Intros.
Elim not_odd_empty; Trivial.
(*   False
  ============================
    H1 : (even (bit b y))
    H0 : (odd (bit b y))
    H : (odd y)->(even y)->False *)
Apply H.
Apply inv_even with b; Trivial.
Apply inv_odd with b; Trivial.
Save.


(************************)
(* Parities of subwords *)
(************************)

Lemma odd_even_conc : (u,v,w:word)(conc u v w) ->
           (odd w) /\
                   ( (odd u) /\ (even v)
                  \/ (even u) /\ (odd v))
        \/ (even w) /\
                   ( (odd u) /\ (odd v)
                  \/ (even u) /\ (even v)).
Goal.
Induction 1; Intros.
Elim (odd_or_even v0); Intro.
Left; Split; Auto.
Right; Split; Auto.
Elim H1; Intros.
(* 1 (odd w0) *)
Right; Elim H2; Intros.
Split; Auto.
Elim H4; Intros.
Right; Split; Elim H5; Auto.
Left; Split; Elim H5; Auto.
(* 2 (even w0) *)
Left; Elim H2; Intros.
Split; Auto.
Elim H4; Intros.
Right; Split; Elim H5; Auto.
Left; Split; Elim H5; Auto.
Save.

Lemma even_conc : (u,v,w:word)(conc u v w) -> (even w) ->
                    (odd u) /\ (odd v)
                 \/ (even u) /\ (even v).
Goal.
Intros u v w c e; Elim odd_even_conc with u v w; Intros.
Elim H; Intro o; Elim not_odd_and_even with w; Auto.
Elim H; Intros; Trivial.
Trivial.
Save.


(*********************************************)
(* Subwords of alternate words are alternate *)
(*********************************************)

Lemma alt_conc_l : (u,v,w:word)(conc u v w)->(b:bool)(alt b w)->(alt b u).
Goal.
Induction 1; Auto; Intros.
(*  (alt b0 (bit b u0)) *)
Elim alt_back with b0 b w0.
Intros eq A.
(*   (alt b0 (bit b u0))
  ============================
    A : (alt (neg b0) w0)
    eq : <bool>b0=b *)
Elim eq; Auto.
Trivial.
Save.

Lemma alt_conc_r : (u,v,w:word)(conc u v w) -> (b:bool)(alt b w) ->
     (odd u) /\ (alt (neg b) v)
  \/ (even u) /\ (alt b v).
Goal.
Induction 1; Intros.
Right; Split; Intros; Auto.
(*  ((odd (bit b u0)) /\ (alt (neg b0) v0)) \/ (even (bit b u0)) /\ (alt b0 v0)
  ============================
    H2 : (alt b0 (bit b w0))
    b0 : bool
    H1 : (b:bool)(alt b w0)->
           (((odd u0) /\ (alt (neg b) v0)) \/ (even u0) /\ (alt b v0)) *)
Elim H1 with (neg b0).
Elim neg_elim with b0; Intro.
Right; Split; Elim H3; Auto.
Intro; Left; Split; Elim H3; Auto.
Apply alt_neg_intro with b; Trivial.
Save.

Lemma alt_conc : (u,v,w:word)(conc u v w) -> (alternate w) ->
                 (alternate u) /\ (alternate v).
Goal.
Induction 2; Intros b ab; Split.
Apply alter with b; Apply alt_conc_l with v w; Trivial.
Elim alt_conc_r with u v w b; Intros; Trivial.
Elim H1; Intros; Apply alter with (neg b); Trivial.
Elim H1; Intros; Apply alter with b; Trivial.
Save. (* unused below *)


(************)
(* Opposite *)
(************)

Inductive Definition opposite : word -> word -> Prop =
  opp : (u,v:word)(b:bool)(opposite (bit b u) (bit (neg b) v)).

Hint opp.

Definition Opp = [u:word][v:word](<Prop>Match u with
    (* empty *)   False
    (* bit b w *) [b:bool][w:word][P:Prop](
      <Prop>Match v with
         (* empty *)   False
         (* bit b w *) [b':bool][w':word][P':Prop]<bool>(neg b)=b')).

(* (Opp u v) iff (opposite u v) *)

Lemma not_opp_empty_r : (u:word)~(opposite u empty).
Goal.
Unfold not; Intros u op.
Change (Nonempty empty).
Elim op; Simpl; Trivial.
Save.

Lemma not_opp_empty_l : (u:word)~(opposite empty u).
Goal.
Unfold not; Intros u op.
Change (Nonempty empty).
Elim op; Simpl; Trivial.
Save.

Lemma not_opp_same : (u,v:word)(b:bool)~(opposite (bit b u) (bit b v)).
Goal.
Unfold not; Intros u v b op.
Apply (no_fixpoint_neg b).
Change (Opp (bit b u) (bit b v)).
Elim op; Simpl; Trivial.
Save.

Lemma opposite1 : (u,v:word)(b:bool)(odd u)->(alt b u)->
               (odd v)->(alt (neg b) v)->(opposite u v).
Goal.
Induction u.
Intros v b odd_empty;
Absurd (odd empty); Trivial.
Intros b u' H v; Elim v.
Intros b' H1 H2 odd_empty.
Absurd (odd empty); Trivial.
Intros b' v' H' b'' H1 H2 H3 H4.
(*  (opposite (bit b u) (bit b' v'))
  ============================
    H4 : (alt (neg b'') (bit b' v'))
    H3 : (odd (bit b' v'))
    H2 : (alt b'' (bit b u')) *)
Elim (alt_eq (neg b'') b' v'); Trivial.
Elim (alt_eq b'' b u'); Trivial.
Save.

Lemma opposite2 : (u,v:word)(b:bool)(alt b u)->(alt b v)->~(opposite u v).
Goal.
Induction u.
Intros; Apply not_opp_empty_l.
Intros b u' H v; Elim v.
Intros; Apply not_opp_empty_r.
Intros b' v' H1 b'' H2 H3.
(*   ~(opposite (bit b u') (bit b' v'))
  ============================
    H3 : (alt b'' (bit b' v'))
    H2 : (alt b'' (bit b u')) *)
Elim (alt_eq b'' b' v'); Trivial.
Elim (alt_eq b'' b u'); Trivial.
Apply not_opp_same.
Save.


(****************)
(* Paired words *)
(****************)

(* (paired w) ==  w = [b1 ~b1 b2 ~b2 ... bn ~bn] *)
Inductive Definition paired : word -> Prop =
  paired_empty : (paired empty)
| paired_bit : (w:word)(paired w)->(b:bool)(paired (bit (neg b) (bit b w))).

(* paired_odd_l b w == w = [b b1 ~b1 b2 ~b2 ... bn ~bn] *)
Definition paired_odd_l = [b:bool][w:word](paired (bit (neg b) w)).

Lemma paired_odd_l_intro : (b:bool)(w:word)(paired w)->(paired_odd_l b (bit b w)).
Goal.
Unfold paired_odd_l; Intros.
Apply paired_bit; Trivial.
Save.

Lemma paired_odd_l_elim : (b:bool)(w:word)(paired_odd_l (neg b) w)->(paired (bit b w)).
Goal.
Unfold paired_odd_l; Intros.
Elim (neg_intro b); Trivial.
Save.

(* paired_odd_r b w == w = [b1 ~b1 b2 ~b2 ... bn ~bn ~b] *)
Definition paired_odd_r = [b:bool][w:word](paired (append w (single b))).

(* paired_rot b w == w = [b b2 ~b2 ... bn ~bn ~b] *)
Inductive Definition paired_rot : bool -> word -> Prop =
  paired_rot_empty : (b:bool)(paired_rot b empty)
| paired_rot_bit : (b:bool)(w:word)(paired_odd_r b w)
                                 ->(paired_rot b (bit b w)).

Lemma paired_odd_r_from_rot : (w:word)(b:bool)(paired_rot b w)->(paired_odd_r b (bit (neg b) w)).
Goal.
Induction 1.
Intro; Unfold paired_odd_r; Simpl.
Unfold single; Apply paired_bit.
Apply paired_empty.
Intros b0 b' w'; Unfold paired_odd_r; Intros.
Simpl; Apply paired_bit; Auto.
Save.

(* paired_bet b w == w = [b b2 ~b2 ... bn ~bn b] *)
Inductive Definition paired_bet [b:bool] :  word -> Prop =
  paired_bet_bit : (w:word)(paired_odd_r (neg b) w)->(paired_bet b (bit b w)).

Lemma paired_odd_r_from_bet : (b:bool)(w:word)(paired_bet (neg b) w)->(paired_odd_r b (bit b w)).
Goal.
Intros b w pb.
Elim (neg_intro b).
Elim pb.
Unfold paired_odd_r. (* Unfolds twice *)
Intros; Simpl.
Apply paired_bit; Trivial.
Save.


(***********************)
(* Shuffling two words *)
(***********************)

Inductive Definition shuffle : word -> word -> word -> Prop =
  shuffle_empty :    (shuffle empty empty empty)
| shuffle_bit_left : (u,v,w:word)(shuffle u v w) ->
                            (b:bool)(shuffle (bit b u) v (bit b w))
| shuffle_bit_right : (u,v,w:word)(shuffle u v w) ->
                            (b:bool)(shuffle u (bit b v) (bit b w)).


(***********************)
(* The shuffling lemma *)
(***********************)

Lemma Shuffling : (u,v,w:word)(shuffle u v w)->(b:bool)(alt b u)->
  ( ((odd u) /\ ( ((odd v)  /\ ((alt (neg b) v) -> (paired w))
                            /\ ((alt b v)       -> (paired_bet b w)))
               \/ ((even v) /\ ((alt b v)       -> (paired_odd_l b w))
                            /\ ((alt (neg b) v) -> (paired_odd_r (neg b) w)))))
 \/ ((even u) /\ ( ((odd v) /\ ((alt (neg b) v) -> (paired_odd_r b w))
                            /\ ((alt b v)       -> (paired_odd_l b w)))
               \/ ((even v) /\ ((alt b v)       -> (paired_rot b w))
                            /\ ((alt (neg b) v) -> (paired w)))))).
Goal.
Induction 1; Intros.
(* 0. empty case *)
Right.
Split; Auto.
Right.
Split; Auto.
Split; Intro.
Apply paired_rot_empty.
Apply paired_empty.
(* 1. u before v *)
Elim (alt_eq b0 b u0); Trivial.
Elim (H1 (neg b0)); Intros.
(* 1.1. *) Right.
Elim H3; Intros.
Split; Auto.
Elim H5; Intros.
Elim H6; Intros.
(* 1.1.1. *) Left.
Elim H8; Intros. 
Split; Auto.
Split; Intro.
Apply paired_odd_r_from_bet; Auto.
Apply paired_odd_l_intro; Apply H9; Elim (neg_elim b0); Auto.
Elim H6; Intros.
Elim H8; Intros.
(* 1.1.2. *) Right.
Split; Auto.
Split; Intro. 
Apply paired_rot_bit; Elim (neg_intro b0); Apply H10; Elim (neg_elim b0); Auto.
Apply paired_odd_l_elim; Auto.
(* 1.2. *) Left.
Elim H3; Intros.
Split; Auto.
Elim H5; Intros.
(* 1.2.1. *) Left. 
Elim H6; Intros. 
Elim H8; Intros.
Split; Auto. 
Split; Intro. 
Apply paired_odd_l_elim; Auto.
Apply paired_bet_bit; Apply H9; Elim (neg_elim b0); Auto.
(* 1.2.2. *) Right.
Elim H6; Intros. 
Elim H8; Intros.
Split; Auto.
Split; Intro.
Apply paired_odd_l_intro; Apply H10; Elim (neg_elim b0); Auto.
Pattern 2 b0; Elim (neg_intro b0).
Apply paired_odd_r_from_rot; Auto.
Apply alt_neg_intro with b; Trivial.
(* 2. v before u *)
Elim (H1 b0); Intros.
(* 2.1. *) Left.
Elim H3; Intros.
Split; Auto.
Elim H5; Intros.
(* 2.1.1. *) Right.
Elim H6; Intros.
Elim H8; Intros.
Split; Auto.
Split; Intro.
Elim (alt_eq b0 b v0); Trivial.
Apply paired_odd_l_intro; Apply H9; Apply alt_neg_intro with b; Auto.
Elim (alt_eq (neg b0) b v0); Trivial.
Apply paired_odd_r_from_bet; Elim (neg_elim b0); Apply H10; Apply alt_neg_elim with b; Auto.
(* 2.1.2. *) Left.
Elim H6; Intros.
Elim H8; Intros.
Split; Auto.
Split; Intro.
Apply paired_odd_l_elim.
Elim (alt_eq (neg b0) b v0); Trivial.
Elim (neg_elim b0).
Apply H9.
Elim (neg_intro b0).
Apply alt_neg_intro with b; Auto.
Elim (alt_eq b0 b v0); Trivial.
Apply paired_bet_bit; Apply H10; Apply alt_neg_intro with b; Auto.
(* 2.2. *) Right.
Elim H3; Intros.
Split; Auto.
Elim H5; Intros.
(* 2.2.1. *) Right.
Elim H6; Intros.
Elim H8; Intros.
Split; Auto.
Split; Intro.
Elim (alt_eq b0 b v0); Trivial.
Apply paired_rot_bit; Apply H9; Apply alt_neg_intro with b; Auto.
Elim (alt_eq (neg b0) b v0); Trivial.
Apply paired_odd_l_elim.
Elim (neg_elim b0); Apply H10; Elim (neg_intro b0); Apply alt_neg_intro with b; Auto.
(* 2.2.2. *) Left.
Elim H6; Intros.
Elim H8; Intros.
Split; Auto.
Split; Intro.
Elim (alt_eq (neg b0) b v0); Trivial.
Apply paired_odd_r_from_rot; Apply H9; Elim (neg_intro b0); Apply alt_neg_intro with b; Auto.
Elim (alt_eq b0 b v0); Trivial.
Apply paired_odd_l_intro; Apply H10; Apply alt_neg_intro with b; Auto.
Trivial.
Save.

(************)
(* Rotation *)
(************)

Definition rotate [w:word](<word>Match w with
   (* empty *)    empty 
   (* bit b u *)  [b:bool][u,v:word](append u (single b))).

Lemma paired_rotate : (w:word)(b:bool)(paired_rot b w)->(paired (rotate w)).
Goal.
Induction 1.
Intro; Simpl; Apply paired_empty.
Intros b' w'; Simpl.
Unfold paired_odd_r; Trivial.
Save.


(********************)
(* The main theorem *)
(********************)

Section Main.

Variable x:word.
Hypothesis Even_x : (even x).
Variable b:bool. (* witness for (alternate x) *)
Hypothesis A : (alt b x).
Variables u,v:word.
Hypothesis C : (conc u v x).
Variable w:word.
Hypothesis S : (shuffle u v w).

Lemma Alt_u : (alt b u).
Goal.
Apply alt_conc_l with v x.
Apply C.
Apply A.
Save.

Section Case1.
Hypothesis Odd_u : (odd u).

Lemma Not_even_u : ~(even u).
Goal.
Red; Intro.
Elim not_odd_and_even with u; Trivial.
Apply Odd_u. 
Save.

Lemma Odd_v : (odd v).
Goal.
Elim even_conc with u v x.
Intro H; Elim H; Trivial.
Intro H; Elim H; Intro; Elim Not_even_u; Trivial.
Apply C.
Apply Even_x.
Save.

Lemma Alt_v : (alt (neg b) v).
Goal.
Elim alt_conc_r with u v x b.
Intro H; Elim H; Trivial.
Intro H; Elim H; Intro; Elim Not_even_u; Trivial.
Apply C.
Apply A.
Save.

Lemma Opp_uv : (opposite u v).
Goal.
Apply opposite1 with b.
Apply Odd_u.
Apply Alt_u.
Apply Odd_v.
Apply Alt_v.
Save.

Lemma Case1 : (paired w).
Goal.
Elim Shuffling with u v w b.
Induction 1; Induction 2; Induction 1; Induction 2; Intros P1 P2.
Apply P1.
Apply Alt_v.
Elim not_odd_and_even with v; Trivial.
Apply Odd_v.
Induction 1; Intro; Elim Not_even_u; Trivial.
Apply S.
Apply Alt_u.
Save.
End Case1.

Section Case2.
Hypothesis Even_u : (even u).

Lemma Not_odd_u : ~(odd u).
Goal.
Red; Intro; Elim not_odd_and_even with u; Trivial.
Apply Even_u.
Save.

Lemma Even_v : (even v).
Goal.
Elim even_conc with u v x.
Intro H; Elim H; Intro; Elim Not_odd_u; Trivial.
Intro H; Elim H; Trivial.
Apply C.
Apply Even_x.
Save.

Lemma Alt_v : (alt b v).
Goal.
Elim alt_conc_r with u v x b.
Intro H; Elim H; Intro; Elim Not_odd_u; Trivial.
Intro H; Elim H; Trivial.
Apply C.
Apply A.
Save.

Lemma Not_opp_uv : ~(opposite u v).
Goal.
Apply opposite2 with b.
Apply Alt_u.
Apply Alt_v.
Save.

Lemma Case2 : (paired (rotate w)).
Goal.
Apply paired_rotate with b.
Elim Shuffling with u v w b.
Induction 1; Intro; Elim Not_odd_u; Trivial.
Induction 1; Induction 2.
Induction 1; Intros; Elim not_odd_and_even with v; Trivial.
Apply Even_v.
Induction 1; Induction 2; Intros P1 P2.
Apply P1.
Apply Alt_v.
Apply S.
Apply Alt_u.
Save.

End Case2.

(* We recall from the prelude the definition of the conditional :
Definition IF = [P,Q,R:Prop](P /\ Q) \/ ((~P) /\ R)
Syntax IF "if _ then _ else _" *)

Lemma Main : if (opposite u v) then (paired w) else (paired (rotate w)).
Goal.
Unfold IF; Elim odd_or_even with u; Intros.
(* (odd u) : Case 1 *)
Left; Split.
Apply Opp_uv; Trivial.
Apply Case1; Trivial.
(* (even u) : Case 2 *)
Right; Split.
Apply Not_opp_uv; Trivial.
Apply Case2; Trivial.
Save.

End Main.


(*********************)
(* Gilbreath's trick *)
(*********************)


Theorem Gilbreath_trick : (x:word)(even x)
  -> (alternate x)
  -> (u,v:word)(conc u v x)
  -> (w:word)(shuffle u v w)
  -> if (opposite u v) then (paired w) else (paired (rotate w)).

Goal.
Induction 2; Intros. (* Existential intro *)
Apply Main with x b; Trivial.
Save.

Provide Shuffle.
