(*****************************************************************************)
(*          Projet Formel - Calculus of Inductive Constructions V5.8         *)
(*****************************************************************************)
(*                                                                           *)
(*    Higman.v                                                               *)
(*                                                                           *)
(*    A version of Higman's lemma on a 2-letter alphabet - H. Herbelin       *)
(*    Obtained by A-translation from the classical proof by Nash-Williams    *)
(*                                                                           *)
(*****************************************************************************)
(* Uses Prelude.v , Specif.v and Nat.v                                       *)
 
(*****************************************************************************)
(*    Recall of what is needed in the initial state of Coq                   *)

Require Le.

(*  What is needed in Prelude.v **

Inductive Definition True : Prop = I : True.

Inductive Definition False : Prop = .
Theorem False_ind (P:Prop)False->P.

Axiom except : (P:Set)False->P.

Inductive Definition eq [A:Set;x:A] : A -> Prop =
   refl_equal : (eq A x x).
Theorem eq_ind (A:Set)(x:A)(P:A->Prop)(P x)->(a:A)(<A>x=a)->(P a).
Axiom eq_rec : (A:Set)(x:A)(P:A->Set)(P x)->(y:A)(eq A x y)->(P y).

Theorem sym_equal (A:Set)(x:A)(y:A)(<A>x=y)->(<A>y=x).
Theorem trans_equal (A:Set)(x:A)(y:A)(z:A)(<A>x=y)->(<A>y=z)->(<A>x=z).
Theorem f_equal (A:Set)(B:Set)(f:A->B)(x:A)(y:A)(<A>x=y)->(<B>(f x)=(f y)).

**  What is needed in Specif.v **

Inductive Definition sumbool [A,B:Prop] : Set =
   left : A ->({A}+{B})
 | right : B->({A}+{B}).

**  What is needed in Nat.v **

  ** natural numbers **

Inductive Set nat =
   O : nat
 | S : nat -> nat.
Theorem nat_ind (P:nat->Prop)(P O)->((y:nat)(P y)->(P (S y)))->(n:nat)(P n).
Theorem nat_rec (P:nat->Set)(P O)->((y:nat)(P y)->(P (S y)))->(n:nat)(P n).

  ** less or equal on natural numbers **

Inductive Definition le [n:nat] : nat -> Prop =
   le_n : (le n n)
 | le_S : (m:nat)(le n m)->(le n (S m)).

Theorem le_O_n (n:nat)(le O n).
Theorem le_trans (n,m,p:nat)(le n m)->(le m p)->(le n p).
Theorem le_n_S (n,m:nat)(le n m)->(le (S n) (S m)).
Theorem le_n_O_eq (n:nat)(le n O)->(<nat>O=n).

Definition pred : nat -> nat
            = [n:nat](<nat>Match n with O [u,v:nat]u).

Theorem n_predSn (m:nat)<nat>(pred (S m))=m.
Theorem le_S_n (n,m:nat)(le (S n) (S m))->(le n m).
Theorem le_trans_S (n,m:nat)(le (S n) m)->(le n m).
Theorem le_Sn_O (n:nat)~(le (S n) O).
Theorem le_Sn_n (n:nat)~(le (S n) n).
Theorem eq_add_S (n,m:nat)(<nat>(S n)=(S m))-><nat>n=m.
Theorem le_antisym (n,m:nat)(le n m)->(le m n)-><nat>n=m.

*)

(*****************************************************************************)
(*                          Higman's lemma                                   *)
(*****************************************************************************)

Section Higman.


(*****************************************************************************)
(*                           Preliminaries                                   *)
(*****************************************************************************)

(* Non dependent version of sumbool and sumor *)

Goal (A:Prop)(B:Prop)(P:Set)(A->P)->(B->P)->({A}+{B})->P.
  Intros.
  Apply (sumbool_rec A B [h:{A}+{B}]P);Assumption.
Save sumbool_rec.

(* To be a total order *)

Inductive Definition connected [A:Set;R:A->A->Prop;x,y:A] : Set =
    R_connected : (R x y)->(connected A R x y)
  | Eq_connected : (eq A x y)->(connected A R x y)
  | L_connected : (R y x)->(connected A R x y).


(****************************************************************************)
(*         Some results about natural numbers                               *)

Definition ksi (nat_rect [n:nat]Prop True [n:nat][P:Prop]False).

Theorem eq_O_Sn (n:nat)(eq nat O (S n))->False
 Proof [n:nat][h:(eq nat O (S n))]
(eq_ind nat O [p:nat](ksi p) I (S n) h).

Definition lne [p,q:nat] (le (S p) q).

Theorem lne_trans (n,p,q:nat)(le n p)->(lne p q)->(lne n q)
 Proof [n,p,q:nat][H:(le n p)][H0:(lne p q)]
       (le_trans (S n) (S p) q (le_n_S n p H) H0).

Definition lne_give_le : (n,p:nat)(lne n p)->(le n p) = le_trans_S.

Theorem not_le_and_lne (n,p:nat)(le n p)->(lne p n)->False
 Proof [n,p:nat][H:(le n p)]
   (le_ind n [n0:nat]~(lne n0 n) (le_Sn_n n)
             [m:nat][H0:(le n m)][H1:~(lne m n)][H2:(lne (S m) n)]
                    (H1 (le_trans_S (S m) n H2)) p H).

Goal (n,p:nat)(le n (S p))->(sumbool (le n p) (<nat>n=(S p))).
  Induction n.
    Left;Apply le_O_n.
  Induction p.
    Right;Elim (le_n_O_eq y).
      Apply refl_equal.
    Apply le_S_n;Assumption.
  Intros;Elim (H y0).
  3:Apply le_S_n;Assumption.
    Left;Apply le_n_S;Assumption.
  Right;Elim H2;Apply refl_equal.
Save le_dec.

Goal (n,p:nat)(lne n (S p))->(sumbool (lne n p) (eq nat n p)).
  Intros.
  Elim (le_dec (S n) p H);Intros.
    Apply left;Assumption.
  Apply right.
  Apply eq_add_S;Assumption.
Save lne_dec.

Goal (n,p:nat)(sumbool (le n p) (lne p n)).
  Induction n.
    Left.
    Apply le_O_n.
  Induction p.
  Right.
  Red.
  Apply le_n_S.
  Apply le_O_n.
  Intros.
  Elim (H y0).
  Left.
  Apply le_n_S;Assumption.
  Right.
  Unfold lne.
  Apply le_n_S;Assumption.
Save le_or_lne.

Goal (n,p:nat)(connected nat lne n p).
  Intros;Elim (le_or_lne n p);Intro.
    Elim (le_or_lne p n);Intro.
      Apply Eq_connected;Apply le_antisym;Assumption.
    Apply R_connected;Assumption.
  Apply L_connected;Assumption.
Save lne_connected.


(* Other function on integers *)

Definition diff [n:nat] 
 (<nat->nat> Match n with [m:nat]O 
        [p:nat][f:nat->nat][m:nat](<nat> Match m with (S p) [r,s:nat](f r))).

Goal (n:nat)(eq nat O (diff n n)).
  Induction n.
    Apply refl_equal.
  Intros;Assumption.
Save diff_n_n.

Axiom lne_diff : (i,j,n:nat)(le n i)->(lne i j)->(lne (diff i n) (diff j n)).

(* maximum of two elements *)

Definition max [n:nat]
 (<nat->nat> Match n with [m:nat]m 
       [p:nat][f:nat->nat][m:nat](S (<nat> Match m with p [r,s:nat](f r)))).

Goal (n,p:nat)(le n (max n p)).
  Induction n.
    Intro;Apply le_O_n.
  Induction p;Intros;Simpl.
    Apply le_n.
  Apply le_n_S.
  Apply H.
Save is1_max.

Goal (n,p:nat)(le p (max n p)).
  Induction n.
    Intro;Apply le_n.
  Induction p;Intros;Simpl.
    Apply le_O_n. 
  Apply le_n_S.
  Apply H.
Save is2_max.

(* well-founded induction on natural numbers *)

Section wfi_lne.

Goal (P:nat->Set)(n:nat)
 ((m:nat)((p:nat)(lne p m)->(P p))->(P m))->(i:nat)(le i n)->(P i).
  Do 3 Intro.
  Elim n;Intros.
    Apply H;Intros.
    Absurd (lne p i);Try Assumption.
    Pattern i;Elim le_n_O_eq.
      Unfold lne;Apply le_Sn_O.
    Assumption.
  Elim (le_dec i y H1);Intros.
    Apply H0;Assumption.
  Apply H;Intros.
  Apply H0.
  Apply le_S_n;Intros;Elim H2;Assumption.
Save Remark wfi_lne1.

Goal (P:nat->Set)(n:nat)((m:nat)((p:nat)(lne p m)->(P p))->(P m))->(P n).
  Intros.
  Apply wfi_lne1 with n.
    Assumption.
  Apply le_n.
Save wfi_lne.

End wfi_lne.


(*****************************************************************************)
(*          Basic notions about words                                        *)

(* The two letters alphabet *)

Inductive Set M =
  A0 : M | A1 : M.

Inductive Definition lneM : M->M->Prop =
 lneM_intro : (lneM A0 A1).

Axiom lneM_rec : (P:M->M->Set)
(P A0 A1)->(y:M)(y0:M)(lneM y y0)->(P y y0).

(* 
Realize lneM_rec [p,y,y0]p.
*)

Theorem A0_A1 (eq M A0 A1)->False
 Proof [h:(eq M A0 A1)](eq_ind M A0 (M_rect [a:M]Prop True False) I A1 h).

Goal (a,b:M)(connected M lneM a b).
  Induction a;Induction b.
        Apply Eq_connected;Apply refl_equal.
      Apply R_connected;Apply lneM_intro.
    Apply L_connected;Apply lneM_intro.
  Apply Eq_connected;Apply refl_equal.
Save lneM_connected.


(* The -trivial- well founded induction on letters *)

Goal (P:M->Set)(a:M)((b:M)((c:M)(lneM c b)->(P c))->(P b))->(P a).
  Intros;Apply H.
  Elim a;Intros.  (* on disjoncte a=A0 ou a=A1 *)
    Apply except;Apply A0_A1.
    Elim H0;Apply refl_equal.  (* H0 : (c < A0 *)
  Elim H0;Apply H;Intros.
    Apply except;Apply A0_A1.
    Elim H1;Apply refl_equal.  (* H0 : (c0 < A0 *)
Save wfi_lneM.


(* words *)

Inductive Set word =
   Emptyword : word
 | Cons : M -> word -> word.


(* tail and head of a word *)

Definition tail [l:word] (<word> Match l with Emptyword [a:M][x,y:word]x).

Definition head [l:word] (<M> Match l with A0 [a:M][x:word][b:M]a).


(* Embedding of a word in an other one *)

Inductive Definition IN : word->word->Prop =
   INMV : (IN Emptyword Emptyword)
 | INcons : (x,y:word)(a:M)(IN x y)->(IN x (Cons a y))
 | INconscons : (x,y:word)(a:M)(IN x y)->(IN (Cons a x) (Cons a y)).

Goal (a:word)(IN Emptyword a).
  Induction a.
    Exact INMV.
  Intros.
  Apply INcons;Assumption.
Save MV_IN_all.


(* Definition of a total order on the words *)

Definition len [x:word] (<nat> Match x with O [a:M][y:word]S).
Definition eql [x,y:word] (eq nat (len x) (len y)).
Definition lel [x,y:word] (le (len x) (len y)).
Definition lnel [x,y:word] (lne (len x) (len y)).

(* The strict lexicographic order on words of same length *)

Inductive Definition slexn : word -> word -> Prop =
   slexn_lneM : (x,y:word)(a,b:M)(lneM a b)->(eql x y)
             ->(slexn (Cons a x) (Cons b y))
 | slexn_Cons : (x,y:word)(a:M)(slexn x y)
             ->(slexn (Cons a x) (Cons a y)).

(* Recursive version of slexn *)

Definition slexn' [x:word]
  (<word->Prop> Match x with 
                          [y:word]False
 [a:M][z:word][f:word->Prop][y:word](<Prop> Match y with 
                                     False
                 [b:M][t:word][H:Prop](<Prop> Match (lneM_connected a b) with
                         [Hab:(lneM a b)](eql z t)
                         [Heq:(eq M a b)](slexn z t)
                         [Hba:(lneM b a)]False))).

(* The strict total order on words *)

Inductive Definition lnew [x,y:word] : Prop =
   lnew_eql : (slexn x y)->(lnew x y)
 | lnew_lnel : (lnel x y)->(lnew x y).

Axiom slexn_MV : (x:word)(slexn x Emptyword)->False.
Axiom lnew_lnel_false : (x,y:word)(lnew x y)->(lnel y x)->False.
Axiom lnew_eq_false : (x,y:word)(lnew x y)->(eq word x y)->False.
Axiom lnew_slexn_false : (x,y:word)(lnew x y)->(slexn y x)->False.

Goal (x,y:word)(slexn x y)->(slexn' x y).
  Induction 1.
    Induction 1;Simpl;Intros;Assumption.
  Induction a;Simpl;Intros;Assumption.
Save slexn_slexn'.

Goal (P:word->word->Set)
 ((x,y:word)(a,b:M)(lneM a b)->(eql x y)->(P (Cons a x) (Cons b y)))
   ->((x,y:word)(a:M)(slexn x y)->(P x y)->(P (Cons a x) (Cons a y)))
         ->(w,w0:word)(slexn' w w0)->(P w w0).
  Induction w.
    Simpl;Intros;Apply except;Assumption.
  Induction w0.
    Simpl;Intros;Apply except;Assumption.
  Do 3 Intro;Simpl;Elim (lneM_connected m m0);Intros.
      Apply H;Assumption.
    Elim e;Apply H0.
      Assumption.
    Apply H1;Apply slexn_slexn';Assumption.
  Apply except;Assumption.
Save slexn'_rec.

Goal (x,y:word)(eql x y)->(connected word slexn x y).
  Induction x.
    Induction y;Intros.
        Apply Eq_connected;Apply refl_equal.
      Absurd (eql Emptyword (Cons m y0));Try Assumption.
      Unfold eql;Simpl;Apply O_S.
  Do 2 Intro;Induction y0;Intros.
    Absurd (eql Emptyword (Cons m y)).
      Unfold eql;Simpl;Apply O_S.
    Unfold eql;Apply sym_equal;Assumption.
  Elim (lneM_connected m m0);Intros.
      Apply R_connected;Apply slexn_lneM;Try Assumption.
        Unfold eql;Apply eq_add_S;Assumption.
    Elim e;Elim (H y1);Intros.
          Apply R_connected;Apply slexn_Cons;Assumption.
        Apply Eq_connected;Elim e0;Apply refl_equal.
      Apply L_connected;Apply slexn_Cons;Assumption.
    Unfold eql;Apply eq_add_S;Assumption.
  Apply L_connected;Apply slexn_lneM;Try Assumption.
        Unfold eql;Apply eq_add_S;Apply sym_equal;Assumption.
Save slexn_connected.

Goal (x,y:word)(P:Set)((slexn x y)->P)->((lnel x y)->P)->(lnew x y)->P.
  Intros x y.
  Elim (lne_connected (len x) (len y));Intros.
      Apply H0;Assumption.
  2:Apply except;Apply (lnew_lnel_false x y);Assumption.
  Elim (slexn_connected x y).
  4:Assumption.
      Assumption.
    Intro;Apply except;Apply (lnew_eq_false x y);Assumption.
  Intro;Apply except;Apply (lnew_slexn_false x y);Assumption.
Save lnew_rec.

Goal (x,y:word)(connected word lnew x y).
  Intros;Elim (lne_connected (len x) (len y));Intro.
      Apply R_connected;Apply lnew_lnel;Assumption.
  2:Apply L_connected;Apply lnew_lnel;Assumption.
  Elim (slexn_connected x y);Intros.
  4:Assumption.
      Apply R_connected;Apply lnew_eql;Assumption.
    Apply Eq_connected;Assumption.
  Apply L_connected;Apply lnew_eql;Assumption.
Save lnew_connected.


(*         well founded induction on words           *)

(* well founded induction on words ordered by length *)

Section wfi_lnel.

Variable P:word->Set.

Variable x:word.

Goal ((y:word)((z:word)(lnel z y)->(P z))->(P y))->(t:word)(lel t x)->(P t).
  Intro H.
  Elim x;Intros.
    Apply H;Intros.
    Absurd (lnel z t);Try Assumption.
    Unfold lnel;Elim (le_n_O_eq (len t)).
      Unfold lne;Apply le_Sn_O.
    Assumption.
  Elim (le_dec (len t) (len y) H1);Intros.
    Apply H0;Assumption.
  Apply H;Intros.
  Apply H0.
  Unfold lel;Apply le_S_n;Intros;Elim H2;Assumption.
Save Remark wfi_lnel1.

Goal ((y:word)((z:word)(lnel z y)->(P z))->(P y))->(P x).
  Intros.
  Apply wfi_lnel1.
    Assumption.
  Unfold lel;Apply le_n.
Save wfi_lnel.

End wfi_lnel.


(* well founded induction on words of fixed length ordered lexicographically *)

Goal 
(n:nat)
 (P:word->Set)
     ((y:word)
       ((z:word)(slexn z y)->(P z))
          ->(eq nat (len y) n)->(P y))
    ->(x:word)(eq nat (len x) n)->(P x).
  Induction n;Induction x;Intros.
        Apply H;Intros.
          Apply except;Apply slexn_MV with z;Assumption.
        Assumption.
      Apply except;Apply eq_O_Sn with (len y);Apply sym_equal;Assumption.
    Apply except;Apply eq_O_Sn with y;Assumption.
  Apply (wfi_lneM [a:M](x:word)(eq nat (len x) y)->(P (Cons a x)));
                       Intros.
  2:Apply eq_add_S;Assumption.
  Apply (H [x:word](P (Cons b x)));Intros.
  2:Assumption.
  Apply H0;Intros.
  2:Elim H6;Apply refl_equal.
  Cut (eq M (head (Cons b y1)) b).
  2:Apply refl_equal.
  Cut (eq word (tail (Cons b y1)) y1).
  2:Apply refl_equal.
  Pattern z (Cons b y1).
  Apply slexn'_rec;Intros.
  3:Apply slexn_slexn';Assumption.
    Apply H3.
      Elim H11;Assumption.
    Elim H6;Elim H10;Assumption.
  Elim (sym_equal M a b H11).
  Apply H5.
  Elim H10;Assumption.
Save wfi_slexn.


(* well founded induction on words totally ordered by lnew *)

Goal (P:word->Set)(x:word)((y:word)((z:word)(lnew z y)->(P z))->(P y))
  ->(P x).
  Intros.
  Apply wfi_lnel;Intros.
  Apply wfi_slexn with (len y);Intros.
  2:Apply refl_equal.
  Apply H.
  Induction 1;Intros.
    Apply H1;Assumption.
  Apply H0.
  Unfold lnel;Elim H2;Assumption.
Save wfi_lnew.



(*****************************************************************************)
(*          The formal A-translated proof of Higman's lemma                  *)
(*****************************************************************************)

Section Formal_Higman.


(* The variable of A-translation *)

Variable A :Set.


(* sequences of words are defined as a predicat and relatively to A *)
(* bad sequences are those which don't verify the lemma             *)

Section Bad_sequence.

Variable f:nat->word->Set.

Definition exi_im (n:nat)((x:word)(f n x)->A)->A.

Definition uniq_im (n:nat)(x,y:word)(f n x)->(f n y)->((eq word x y)->A)->A.

Definition cex (i,j:nat)(x,y:word)(f i x)->(f j y)->(lne i j)->(IN x y)->A.

Inductive Definition bad : Set =
  bad_intro : exi_im -> uniq_im -> cex -> bad.

End Bad_sequence.


(* To be equal on the n-1 first terms *)

Definition eqgn [n:nat][h,h':nat->word->Set]
  (i:nat)(lne i n)->(s,t:word)(h i s)->(h' i t)->((eq word s t)->A)->A.


(* To be minimal on the nth term *)

Definition Minbad [n:nat][h:nat->word->Set][y:word]
   (h':nat->word->Set)(bad h')->(eqgn n h h')
                    ->(z:word)(h' n z)->(lnew z y)->A.


(* To be minimal on the n-1 first terms *)

Definition Minbadns [n:nat][h:nat->word->Set] 
  (p:nat)(lne p n)->((y:word)(h p y)->(Minbad p h y)->A)->A.


(* The minimal (bad) counter-example *)

Definition Mincex [n:nat][x:word]
 (C:Set)
  ((h:nat->word->Set)(bad h)->(Minbadns n h)->(h n x)->(Minbad n h x)->C)->C.

Theorem Mincex_intro (n:nat)(x:word)
(h:nat->word->Set)(bad h)->(Minbadns n h)
    ->(h n x)->(Minbad n h x)->(Mincex n x)
Proof [n:nat][x:word][h:nat->word->Set][p:(bad h)][p1:(Minbadns n h)]
[p2:(h n x)][p3:(Minbad n h x)]
 [C:Set]
  [p4:((h:nat->word->Set)(bad h)->(Minbadns n h)->(h n x)->(Minbad n h x)->C)]
      (p4 h p p1 p2 p3).

Goal (n:nat)(x:word)(P:Set)
  ((h:nat->word->Set)(bad h)->(Minbadns n h)->(h n x)->(Minbad n h x)->P)
      ->(Mincex n x)->P.
  Intros.
  Apply H0.
  Assumption.
Save Mincex_rec.

(* A thick sequence has no empty term. It is the case of counter-examples *)

Section thick.

Definition thick [f:nat->word->Set] (n:nat)(f n Emptyword)->A.

Variable f:nat->word->Set.

Goal (exi_im f)->(cex f)->(thick f).
  Red;Intros.
  Apply (H (S n));Intros.
  Apply (H0 n (S n) Emptyword x).
        Assumption.
      Assumption.
    Exact (le_n (S n)).
  Apply MV_IN_all.
Save bad_thick.

End thick.


(* There is an infinity of terms with the same first letter *)

Section infinitely_many.

Variable h:nat->word->Set.

Hypothesis exi_imh : (exi_im h).

Hypothesis thick_h : (thick h).

Definition the_next_after [a:M][k:nat]
	(n:nat)(x:word)(le k n)->(h n (Cons a x))->A.

Definition infinitely_many [a:M]
	(k:nat)(the_next_after a k)->A.


Goal (k,k0:nat)(the_next_after A0 k)->(the_next_after A1 k0)
    ->(x:word)(h (max k k0) x)->A.
  Induction x.
  (* x=Emptyword *)
      Intros;Apply (thick_h (max k k0)).
    Assumption.
  (* x=m::y *)
  Induction m;Intros.
      (* m=A0 *)
      Apply (H (max k k0) y).
        Apply is1_max.
      Assumption.
    (* m=A1 *)
    Apply (H0 (max k k0) y).
      Apply is2_max.
    Assumption.
Save switcher.

Goal (k,k0:nat)(the_next_after A0 k)->(the_next_after A1 k0)->A.
  Intros.
  Apply (exi_imh (max k k0));Intros.
  Apply switcher with k k0 x.
      Assumption.
    Assumption.
  Assumption.
Save reader.

Goal (k:nat)(the_next_after A0 k)->((infinitely_many A1)->A)->A.
  Intros.
  Apply H0.
  Red;Intros.
  Apply (reader k k0).
    Assumption.
  Assumption.
Save little_inf_A0_or_inf_A1.

Goal ((infinitely_many A0)->A)->((infinitely_many A1)->A)->A.
  Intros.
  Apply H.
  Red;Intros.
  Apply (little_inf_A0_or_inf_A1 k).
    Assumption.
  Assumption.
Save inf_A0_or_inf_A1.

End infinitely_many.


(****************************************************************************)
(*      We show that the minimal counter-example is effectively bad         *)

Section BadMin.

Variable f : nat->word->Set.
Hypothesis Badf : (bad f).
Variable f0 : word.
Hypothesis f_0_f0 : (f O f0).


Section ExiMin.

Goal 
(n:nat)(h,h',h'':nat->word->Set)(exi_im h')->(eqgn n h h')->(eqgn n h' h'')->(eqgn n h h'').
  Intros.
  Red;Intros.
  Apply (H i);Intros.  (* H = (exi_im h') *)
  Unfold eqgn in H0.
  Unfold eqgn in H1.
  Apply H0 with i s x.
        Assumption.
      Assumption.
    Assumption.
  Intros.
  Apply H1 with i x t.
        Assumption.
      Assumption.
    Assumption.
  Intros.
  Apply H5.
  Apply trans_equal with x.
    Assumption.
  Assumption.
Save eqgn_trans.

Goal (h,h':nat->word->Set)(bad h')
     ->(n:nat)(eqgn n h h')->(Minbadns n h)->(Minbadns n h').
  Unfold 2 Minbadns.
  Intros.
  Unfold Minbadns in H1 ; Apply H1 with p.
    Assumption.
  Intros.
  Elim H;Intros;Unfold exi_im in e;Apply e with p;Intros.
  Unfold eqgn in H0 ; Apply H0 with p y x;Try Assumption.
  Intro.
  Apply H3 with x.
    Assumption.
  Unfold Minbad.
  Intros.
  Unfold Minbad in H5.
  Apply H5 with h'0 z; Try  Assumption.
    Apply eqgn_trans with h';Try Assumption.
    Unfold eqgn; Do 2 Intro.
    Intros.
    Apply H0 with i s t;Try Assumption.
    Apply lne_trans with p.
      Apply lne_give_le .
      Assumption.
    Assumption.
  Cut (eq word x y).
    Intros H_eq; Elim H_eq ; Assumption.
  Apply sym_equal; Assumption.
Save eqgn_Minbadns.

Section exi_cmin_S.

Variable n:nat.

Goal 
((x:word)(h:nat->word->Set)(bad h)->(Minbadns n h)->(Minbad n h x)->(h n x)->A)
 ->(x:word)(h:nat->word->Set)(bad h)->(Minbadns n h)->(h n x)->A.
  Intros H x;Pattern x;Apply wfi_lnew.
  Intros;Apply H with y h;Try Assumption.  
  Unfold Minbad;Intros.
  Apply H0 with z h';Try Assumption.
  Apply eqgn_Minbadns with h;Try Assumption.
Save exi_cmin_S.

Goal
((x:word)(h:nat->word->Set)(bad h)->(Minbadns (S n) h)->(h (S n) x)->A)
->(x:word)(h:nat->word->Set)(bad h)->(Minbadns n h)->(Minbad n h x)->(h n x)->A.
  Induction 2;Intros.
  Unfold exi_im in e;Apply e with (S n);Intros.
  Apply H with x0 h;Try Assumption.
  Unfold Minbadns;Intros.
  Elim (lne_dec p n);Intros.
  3:Assumption.
     Unfold Minbadns in H1;Apply H1 with p;Try Assumption.
  Apply H6 with x.  
    Elim (sym_equal nat p n H7); Assumption.
  Elim (sym_equal nat p n H7); Assumption.
Save exi_cmin_exi.

End exi_cmin_S.

Goal (x:word)(f:nat->word->Set)(bad f)->(f O x)->(exi_im Mincex).
  Red.
  Unfold Mincex.
  Induction n;Intros.
    Apply exi_cmin_S with O x f;Try Assumption.
      Intros.
      Apply H1 with x0;Intros.
      Apply H6 with h;Try Assumption.
    Unfold Minbadns;Intros.
    Absurd (le (S p) O);Apply le_Sn_O Orelse Assumption.
  Apply H1;Intros.
  Apply H3;Intros.
  Apply exi_cmin_exi with y x0 h;Try Assumption.
  Intros.
  Apply exi_cmin_S with (S y) x1 h0;Try Assumption.
  Intros.
  Apply H2 with x2;Intros.
  Apply H15 with h1;Try Assumption.
Save ExiMin1.

Goal (exi_im Mincex).
  Apply ExiMin1 with f0 f.
  Exact Badf.
  Exact f_0_f0.
Save ExiMin.

End ExiMin.


Section UniMin.

Goal (h,h':nat->word->Set)(n:nat)(i:nat)(lne i (S n))
   ->(bad h)->(bad h')
   ->(Minbadns i h)->(Minbadns i h')
   ->(eqgn n Mincex Mincex)
   ->(eqgn i h h').
  Unfold eqgn;Intros.
  Apply H4 with i0 s t;Try Assumption.
      Unfold lne;Apply le_trans with i.
        Assumption.
      Apply le_S_n;Assumption.
    Unfold Minbadns in H2.
    Apply Mincex_intro with h;Try Assumption.
      Unfold Minbadns;Intros.
      Apply H2 with p.
        Unfold lne;Apply le_trans with i0;Try Assumption.
        Apply (lne_give_le);Assumption.
      Assumption.
    Unfold Minbad;Intros.
    Apply H2 with i0.
      Assumption.
    Unfold Minbad;Intros.
    Elim H0;Intros.
    Unfold uniq_im in u;Apply u with i0 s y;Try Assumption.
    Intro;Apply H14 with h'0 z;Try Assumption.
    Elim H15;Assumption.
  Unfold Minbadns in H3.
  Apply Mincex_intro with h';Try Assumption.
    Unfold Minbadns;Intros.
    Apply H3 with p.
      Unfold lne;Apply le_trans with i0;Try Assumption.
      Apply (lne_give_le);Assumption.
    Assumption.
  Unfold Minbad;Intros.
  Apply H3 with i0.
    Assumption.
  Unfold Minbad;Intros.
  Elim H1;Intros.
  Unfold uniq_im in u;Apply u with i0 t y;Try Assumption.
  Intro;Apply H14 with h'0 z;Try Assumption.
  Elim H15;Assumption.
Save UniMin_eqgn.

Goal (n:nat)(eqgn n Mincex Mincex).
  Induction n;Intros;Unfold eqgn;Intros.
    Absurd (le (S i) O).
      Apply le_Sn_O.
    Assumption.
  Elim (lne_dec i y H0);Intros.
    (* i < y *)
    Unfold eqgn in H;Apply H with i s t;Assumption.
  (* i = y *)
  Apply H1;Intros.
  Apply H2;Intros.
  (* 3 cas : x<y, y<x, ou x=y *) 
  Elim (lnew_connected s t).
  (* Le cas x=y se resoud ds les hypotheses *)
  2:Assumption.
      (* x < y *)
    Intro;Unfold Minbad in H12.
    Apply H12 with h s;Try Assumption.
    Apply UniMin_eqgn with y;Try Assumption.
  Intro;Unfold Minbad in H8.
  Apply H8 with h0 t;Try Assumption.
  Apply UniMin_eqgn with y;Try Assumption.
Save UniMin1.

Goal (uniq_im Mincex).
  Unfold uniq_im;Intros.
  Cut (eqgn (S n) Mincex Mincex);Apply UniMin1 Orelse Intro H_eqgn.
  Unfold eqgn in H_eqgn;Apply H_eqgn with n x y.
        Exact (le_n (S n)).
      Assumption.
    Assumption.
  Assumption.
Save UniMin.

End UniMin.


Section CexMin.

Goal (cex Mincex).
  Red;Intros.
  (* (Mincex j y) donne un h min sur 1..j valant y en j *)
  Apply H0.
  Intros.
  Elim H3;Intros.
  Unfold Minbadns in H4.
  (* On exhibe la valeur y0 de h en i *)
  Apply H4 with i;Assumption Orelse Intros.
  Cut (Mincex i y0).
    Intros.
    Apply (UniMin i x y0).
        Assumption.
      Assumption.
    Intros.
    Apply (c i j y0 y).  (* on utilise le fait que h est bad *)
          Assumption.
        Assumption.
      Assumption.
    Elim H10.                  (* c = (cex h)  *)
    Assumption.
  Apply (Mincex_intro i y0 h).
        Assumption.
      Unfold Minbadns;Intros.
      Apply H4 with p.
        Apply lne_trans with i.
          Apply lne_give_le.
          Assumption.
        Assumption.
      Assumption.
    Assumption.
  Assumption.
Save CexMin.

End CexMin.


Theorem BadMin (bad Mincex)
  Proof (bad_intro Mincex ExiMin UniMin CexMin).

End BadMin. 



(****************************************************************************)
(* We show in this section that the extracted sequence of a bad sequence    *)
(* is also a bad sequence                                                   *)

Section Badfx.

Variable f:nat->word->Set.

Hypothesis Badf : (bad f).

Variable a0:M.

Hypothesis inf_many : (infinitely_many f a0).


(*   Xa is the set of word beginning with a0   *)
(* (Xn n) is Xa with the first n words removed *)

Inductive Definition Xn : nat->nat->Set =
   Xa_intro : (n:nat)(y:word)(f n (Cons a0 y))->(Xn O n)
 | X'_intro : (p:nat)(n:nat)(Xn p n)->(n':nat)(Xn p n')->(lne n' n)
        ->(Xn (S p) n).

Inductive Definition Leastp [X:nat->Set;p:nat] : Set =
 Leastp_intro : (X p)->((q:nat)(lne q p)->(X q)->A)->(Leastp X p).


(* The subsequence of word beginning with the letter a0 *)

Definition g [n,p:nat] (Leastp (Xn n) p).

Axiom eq_O_Sn : (n:nat)~(eq nat O (S n)).

Goal (P:nat->nat->Set)(y:nat)(y0:nat)(Xn y y0)
->((n:nat)(y:word)(f n (Cons a0 y))->(P O n))
->((p:nat)(n:nat)(P p n)->(n':nat)(P p n')->(lne n' n)->(P (S p) n))
->(P y y0).
  Intros.
  Elim H.
    Assumption.
  Intros.
  Apply H1 with n'.
      Assumption.
    Assumption.
  Assumption.
Save Xn_elim.


Goal (i,q:nat)(Xn (S i) q)->((q':nat)(Xn i q')->(lne q' q)->A)->A.
  Intros i q H_Xn.
  Cut (eq nat (pred (S i)) i);Try Apply refl_equal.
  Intro H_eq;Elim H_eq.
  Cut (eq nat (S i) (S i));Try Apply refl_equal.
  Pattern 1 2 4 (S i) q H_Xn.
  Elim H_Xn;Intros.
    Absurd <nat>O=(S i);Assumption Orelse Apply O_S.
  Apply H2 with n';Assumption.
Save X'_elim.

Goal  (q:nat)(Xn O q)->((y:word)(f q (Cons a0 y))->A)->A.
  Intros.
  Apply (Xn_elim [p,q:nat]((y:word)(f q (Cons a0 y))->A)->A O q).
        Assumption.
      Intros.
      Apply (H2 y).
      Assumption.
    Do 6 Intro.
    Assumption.
  Assumption.
Save Xa_elim.

Section Xn_decreases.

Variable n,q:nat.

Local P [p:nat](le n p)->(Xn p q)->(Xn n q).

Goal (p:nat)(Xn (S p) q)->(Xn p q).
  Intros.
  Cut (Xn (pred (S p)) q);Intros.
    Assumption.
  Elim H;Intros.
    Apply Xa_intro with y.
    Assumption.
  Assumption.
Save Xn_decreases1.

Goal (p:nat)(P p).
  Intro.
  Apply (nat_rec).
    Red;Intros.
    Elim (le_n_O_eq n H).
    Assumption.
  Red;Intros.
  Elim (le_dec n y H0);Intros.
    Apply (H H2).
    Apply Xn_decreases1.
    Assumption.
  Elim (sym_equal nat n (S y) H2).
  Assumption.
Save Xn_decreases.

End Xn_decreases.


Goal (n:nat)(p:nat)(Xn n p)->(Xn O p).
  Intros.
  Apply Xn_decreases with n.
    Apply (le_O_n n).
  Assumption.
Save Xn_in_Xa.

Section Xn_non_empty.

Section X1.

Local P [n:nat](p,q:nat)(lne p q)->(Xn n p)->(Xn O q)->(Xn (S n) q).

Goal (n:nat)(P n).
  Intros.
  Apply nat_rec.
    Red;Intros.
    Apply X'_intro with p.
        Assumption.
      Assumption.
    Assumption.
  Unfold P.
  (*Red;*)Intros.
  Apply X'_intro with p.
      Apply H with p.
            Assumption.
          Cut (Xn (pred (S y)) p);Intros.
            Assumption.
          Elim H1;Intros.
          Apply Xa_intro with y0.
          Assumption.
        Assumption.
      Assumption.
    Assumption.
  Assumption.
Save X1.

End X1.

Local P [n:nat]((p:nat)(Xn n p)->A)->A.

Goal (n:nat)(P n).
  Intros.
  Apply nat_rec.
    Red;Intros.
    Apply (inf_many O).
    Red;Intros.
    Apply (H n0).
    Apply Xa_intro with x.
    Assumption.
  Unfold P.
  Intros.
  Apply H.
  Intros.
  Apply (inf_many (S p)).
  Red;Intros.
  Apply (H0 n0).
  Apply X1 with p.
      Assumption.
    Assumption.
  Apply Xa_intro with x.
  Assumption.
Save Xn_non_empty.

End Xn_non_empty.

(* g is effectively a sequence *)

Goal (n:nat)((p:nat)(g n p)->A)->A.
  Intros.
  Apply (Xn_non_empty n).
  Intro;Pattern p.
  Apply wfi_lne.
  Intros.
  Apply (H m).
  Red;Intros.
  Apply Leastp_intro.
    Assumption.
  Assumption.
Save Exig.

Goal (n:nat)(p,q:nat)(g n p)->(g n q)->((eq nat p q)->A)->A.
  Unfold g.
  Unfold g.
  Intros.
  Elim H.
  Elim H0.
  Intros.
  Elim (lne_connected p q).
      Intro;Apply (a p);Assumption. (* a = (q0:nat)(lne q0 p)->(Xn n q0)->A *)
    Assumption.
  Intro;Apply (a0 q);Assumption.    (* a0 = (q:nat)(lne q p)->(Xn n q)->A   *)
Save Unig.


(* Definition of the extracted sequence and proof of its badness *)

Section Badfx.


Variable n0:nat.

Hypothesis n0_min : (g O n0).

Inductive Definition fx [i:nat;x:word] : Set =
   fxhd : (lne i n0)->(f i x)->(fx i x)
 | fxtl : (p:nat)(le n0 i)->(g (diff i n0) p)
                       ->(f p (Cons a0 x))->(fx i x).


Theorem Badfx.

Statement (bad fx).


Goal (n:nat)(x:word)(y:word)(fx n x)->(fx n y)->((eq word x y)->A)->A.
  Intros.
  Elim Badf;Intros.
  Elim H;Intros.
    Elim H0;Intros.
      Apply (u n x y).    (* u = (uniq_im f) *)
          Assumption.
        Assumption.
      Assumption.
    Apply except.
    Apply (not_le_and_lne n0 n).
      Assumption.
    Assumption.
  Elim H0.
    Intros.
    Apply except.
    Apply (not_le_and_lne n0 n).
      Assumption.
    Assumption.
  Intros p0 Hl Hg Hf.     (* si on ne force pas, clash anormal avec Unig *)
  Apply (Unig (diff n n0) p0 p).
      Assumption.
    Assumption.
  Intro.
  Apply (u p (Cons a0 x) (Cons a0 y)).
      Assumption.
    Elim H2.
    Assumption.
  Intro.
  Apply H1.
  Cut (eq word (tail (Cons a0 x)) (tail (Cons a0 y)));Intros.
    Assumption.
  Elim H3;Apply refl_equal.
Save Unifx.


Goal (n:nat)((x:word)(fx n x)->A)->A.
  Intros.
  Elim Badf;Intros.
  Elim (le_or_lne n0 n);Intros.
    Apply (Exig (diff n n0)).
    Intros.
    Cut (Leastp  (Xn (diff n n0)) p);Intros.
    Elim H2.
    Intros.
    Apply (Xn_elim [q,q':nat](g (diff n n0) q')->A (diff n n0) p).
            Assumption.
          Intros.
          Apply (H y).
          Apply fxtl with n1.
              Assumption.
            Assumption.
          Assumption.
        Do 6 Intro.
        Assumption.
      Assumption.
    Assumption.
  Apply (e n).
  Intros.
  Apply (H x).
  Apply fxhd.
    Assumption.
  Assumption.
Save Exifx.

Goal (i,j:nat)(p,q:nat)(lne i j)->(g i p)->(g j q)->(le q p)->A.
  Intros.
  Unfold g in H0.
  Elim H0.
  Intros.
  Apply (X'_elim i q).
    Unfold g in H1.
    Apply Xn_decreases with j.
      Assumption.
    Elim H1.
  Intros.
  Assumption.
  Intros.
  Apply (a q').
  2:Assumption.
  Unfold lne;Apply le_trans with q.
    Assumption.
  Elim (le_or_lne q p);Intros.
    Assumption.
  Assumption.
Save g_grows.

Goal (i,p:nat)(g i p)->(lne p n0)->A.
  Intros.
  Unfold g in H.
  Elim H.
  Intros.
  Cut (Leastp (Xn O) n0);Intros.
    Elim H1.
    Intros.
    Apply (a0 p).  (* a0 = (q:nat)(Xn O q)->((le n0 q)->A)->A  *)
      Assumption.
    Apply (Xn_in_Xa i).
    Assumption.
  Exact n0_min.
Save g_grows_0.

Goal (i:nat)(j:nat)(x:word)(y:word)(fx i x)->(fx j y)->(lne i j)->(IN x y)->A.
  Intros.
  Elim Badf;Intros.
  Elim H;Intros.
    Elim H0;Intros.
      Apply (c i j x y);Intros.      (* c = (cex f) *)
            Assumption.
          Assumption.
        Assumption.
      Assumption.
    Elim (le_or_lne n0 p);Intros.
      Apply (c i p x (Cons a0 y)).
            Assumption.
          Assumption.
        Unfold lne;Apply le_trans with n0.
          Assumption.
        Assumption.
      Apply INcons.
      Assumption.
    Apply (g_grows_0 (diff j n0) p);Intros.
      Assumption.
    Assumption.
  Elim H0;Intros.
    Apply except.
    Apply (not_le_and_lne j i).
      Apply le_trans with n0.
        Apply lne_give_le.
        Assumption.
      Assumption.
    Assumption.
  Elim (le_or_lne p0 p);Intro.
    Apply (g_grows (diff i n0) (diff j n0) p p0).
    Apply lne_diff.
            Assumption.
          Assumption.
        Assumption.
      Assumption.
    Assumption.
  Apply (c p p0 (Cons a0 x) (Cons a0 y)).
        Assumption.
      Assumption.
    Assumption.
  Apply INconscons.
  Assumption.
Save Cexfx.

Proof  (bad_intro fx Exifx Unifx Cexfx).

(* fx and f are identical on the n0-1 first terms *)

Goal (eqgn n0 f fx).
  Red;Intros.
  Elim H1;Intros.
    Elim Badf;Intros.
    Apply (u i s t).
        Assumption.
      Assumption.
    Assumption.
  Apply except.
  Apply (not_le_and_lne n0 i).
    Assumption.
  Assumption.
Save eqgn_f_fx.

End Badfx.

End Badfx.


(*****************************************************************************)
(*                    The contradiction                                      *)

Section contradiction.

Variable f:nat->word->Set.
Hypothesis Badf : (bad f).
Variable f0:word.
Hypothesis f_0_f0: (f O f0).

Goal (a:M)(n:nat)(h:nat->word->Set)(bad h)->(Minbadns n h)
         ->(eqgn n h (fx Mincex a n)).
  Unfold eqgn;Intros.
  Cut (bad Mincex).
    Intro H_BadMin.
    Unfold Minbadns in H0;Apply H0 with i;Try Assumption.
    Intro;Elim H;Intros;Apply (u i s y);Assumption Orelse Intro.
    Intros.
    Apply (eqgn_f_fx Mincex H_BadMin a n i H1 y t);Try Assumption.
      Apply Mincex_intro with h;Try Assumption.
        Red;Intros.
        Apply H0 with p.
        Unfold lne;Apply le_trans with i.
          Assumption.
        Apply lne_give_le.
        Assumption.
      Assumption.
    Elim H7;Assumption.
  Exact (BadMin f Badf f0 f_0_f0).
Save eqgn_fxMin_h.

Goal (a:M)(infinitely_many Mincex a)->
(n0:nat)((q:nat)(lne q n0)->(Xn Mincex a O q)->A)->(Xn Mincex a O n0)->A.
  Intros.
  Cut (bad Mincex);Intros.
    Apply (Xa_elim Mincex a n0);Intros.
      Assumption.
    Cut (bad (fx Mincex a n0));Intros.
      Apply H3;Intros.
      Unfold Minbad in H8.
      Apply H8 with (fx Mincex a n0) y.
            Assumption.
          Apply eqgn_fxMin_h;Assumption.
        Apply fxtl with n0.
            Apply le_n.
          Elim (diff_n_n n0).
          Red;Intros.
          Apply Leastp_intro.
            Assumption.
          Assumption.
        Assumption.
      Apply lnew_lnel.
      Unfold lnel lne.
      Apply le_n.
    Apply Badfx.
        Assumption.
      Assumption.
    Red;Intros.
    Apply Leastp_intro.
      Assumption.
    Assumption.
  Exact (BadMin f Badf f0 f_0_f0).
Save snake.

Goal (a:M)(infinitely_many Mincex a)
           ->(n:nat)(x:word)(le O n)->(Mincex n (Cons a x))->A.
  Intros.
  Apply (wfi_lne [n:nat](Xn Mincex a O n)->A n).
    Exact (snake a H).
  Apply Xa_intro with x.
  Assumption.
Save holf.

Goal (a:M)(infinitely_many Mincex a)->A.
  Intros.
  Apply (H O).
  Exact (holf a H).
Save one_letter_final.


Goal A.
  Elim (BadMin f Badf f0 f_0_f0);Intros.
  Apply (inf_A0_or_inf_A1 Mincex);Intros.
        Assumption.
      Apply bad_thick;Assumption.
    Apply (one_letter_final A0);Assumption.
  Apply (one_letter_final A1);Assumption.
Save final.

End contradiction.

End Formal_Higman.


(*****************************************************************************)
(*             "A" is now Higman's lemma statement                           *)

Section Higman_end.

Variable f0 : nat->word.

Inductive Definition f [n:nat] : word->Set =
   f_intro : (f n (f0 n)).

Inductive Definition A : Set =
 A_intro : (i,j:nat)(lne i j)->(IN (f0 i) (f0 j))->A.


Goal (exi_im A f).
  Red;Intros.
  Apply (H (f0 n)).
  Apply f_intro.
Save Exif.

Goal (uniq_im A f).
  Unfold uniq_im;Intros n x y H_f_x H_f_y H_eq_x_y.
  Apply H_eq_x_y.
  Pattern x.
  Elim H_f_x.
  Pattern y.
  Elim H_f_y.
  Apply refl_equal.
Save Unif.

Goal (cex A f).
  Red;Do 7 Intro.
  Pattern x.
  Elim H.
  Pattern y.
  Elim H0.
  Intro.
  Apply (A_intro i j).
    Assumption.
  Assumption.
Save Cexf.

Theorem Badf (bad A f)
  Proof (bad_intro A f Exif Unif Cexf).


Theorem Higman A
 Proof (final A f Badf (f0 O) (f_intro O)).


End Higman_end.

End Higman.
Provide Higman.