(*****************************************************************************)
(*                CORRECTNESS PROOF OF A MINI_ML COMPILER                    *)
(*                             Samuel Boutin                                 *)
(*                               June 1993                                   *)
(*                               Coql  V5.8                                  *)
(*****************************************************************************)

(*****************************************************************************)
(* First we define a Set "OP" of variable which is a representation of the   *)
(* arithmetic functions *,+,-... that we will not describe explicitely       *)
(* We take the natural numbers to code the variables (but of course it has no*)
(* relation with the De bruijn indexes)                                      *)
(*****************************************************************************)

Variable OP:Set.

Variable eval_op:OP->nat->nat->nat.

Definition  Pat=nat.

(*****************************************************************************)

Inductive Set MLexp=

  Bool:bool->MLexp
 |Num:nat->MLexp
 |op:OP->MLexp
 |id:Pat->MLexp
 |appl:MLexp->MLexp->MLexp
 |mlpair:MLexp->MLexp->MLexp
 |lambda:Pat->MLexp->MLexp
 |let':Pat->MLexp->MLexp->MLexp
 |letrec:Pat->Pat->MLexp->MLexp->MLexp
 |ite:MLexp->MLexp->MLexp->MLexp.

(*****************************************************************************)
(*           Here we define as an inductive Set mini-ML values               *)
(* with the help of dependant types we define in the same time mini-ML       *)
(* environments.                                                             *)
(*****************************************************************************)

Inductive Definition V_E:bool->Set=

  boolean:bool->(V_E true)
 |num:nat->(V_E true)
 |valpair:(V_E true)->(V_E true)->(V_E true)
 |OP_clos:OP->(V_E true)
 |Clos:Pat->MLexp->(V_E false)->(V_E true)
 |Clos_rec:Pat->MLexp->Pat->(V_E false)->(V_E true)
 |Enil:(V_E false)
 |Econs:Pat->(V_E true)->(V_E false)->(V_E false).

(*****************************************************************************)

Definition MLval=(V_E true).
                
Definition MLenv=(V_E false).

(*****************************************************************************)
(*We have nevertheless to define "good" induction schemes for MLval and MLenv*)
(*****************************************************************************)

Definition to_V_Etrue=[Q:MLval->Prop][b:bool]
     (<[b0:bool](V_E b0)->Prop>Match b with
   (*true*)Q
   (*false*)[H:(V_E false)]True).  
       


Lemma to_V_EQ_Q.
Statement(Q:MLval->Prop)(v:MLval)(to_V_Etrue Q true v)->(Q v).

Goal.
 Auto.
Save.


Lemma MLval_ind.
Statement (Q:MLval->Prop)(ml:MLval)
                  (((b:bool)(Q (boolean b)))->
                  ((n:nat)(Q (num n)))->
                  ((c:OP)(Q (OP_clos c)))->
                  ((a,b:MLval)(Q a)->(Q b)->(Q (valpair a b)))->
                  ((P:Pat)(E:MLexp)(e:MLenv)(Q (Clos P E e)))->
                  ((P:Pat)(E:MLexp)(x:Pat)(e:MLenv)(Q (Clos_rec P E x e)))->
                                  (Q ml)).


Goal.
 Intros Q ml cas_bool cas_n cas_op cas_pair cas_clos cas_rec.
 Apply to_V_EQ_Q.
 Elim ml;Auto.
 Intros;Apply cas_pair;Auto.
 Intros;Apply cas_clos;Auto.
 Intros;Apply cas_rec;Auto.
 Simpl;Trivial.
Save.
           
(*****************************************************************************)

Definition to_V_Efalse=[E:MLenv->Prop][b:bool]
         (<[b0:bool](V_E b0)->Prop>Match b with
       (*true*)[H:(V_E true)]True
       (*false*)E).

Lemma to_V_EE_E.
Statement (E:MLenv->Prop)(e:MLenv)(to_V_Efalse E false e)->(E e).

Goal.
 Auto.
Save.

Lemma  MLenv_ind.
Statement (E:MLenv->Prop)(e:MLenv)((E Enil)->
                             ((P:Pat)(ml:MLval)(e1:MLenv)
                                (E e1)->(E (Econs P ml e1)))->
                                    (E e)).

Goal.
 Intros.
 Apply  to_V_EE_E.
 Elim e;Auto;Simpl;Intros;Trivial.
 Exact (H0 p y y0 H2).
Save.

(*****************************************************************************)
(* Definition of VAL_OF which decide if a variable belongs to an environment *)
(*****************************************************************************)

Inductive Definition VAL_OF:MLenv->Pat->MLval->Prop=

 ELT:(e:MLenv)(I:Pat)(a:MLval)
         (VAL_OF (Econs I a e) I a)

 |CHG:(e:MLenv)(X,I:Pat)(a,b:MLval)
         (VAL_OF e I a)->
                    ( ~(<Pat> X=I))->
             (VAL_OF (Econs X b e) I a).


(*****************************************************************************)
(*                              Inversion of VAL_OF                          *)
(*****************************************************************************)

Definition Val_inv=[e:MLenv][I:Pat][V:MLval]
         (<[b:bool][p:(V_E b)]Prop>Match e with
    (*e is a boolean*)
                [b:bool]False

    (*e est un num*)
                [n:nat]False

    (*e est une paire de valeurs ML*)
                [V1:MLval][p1:Prop][V2:MLval][p2:Prop]False
 
   (* e is a predefinite operator*)
                [c:OP]False

    (*e est une fermeture*)
                [x:Pat][E:MLexp][e':MLenv][pe:Prop]False

    (*e est une fermeture recursive*)
                [x:Pat][E:MLexp][P:Pat][e:MLenv][pe:Prop]False

    (*e est un environnement Enil*)
                False
                
    (*e est un environnement Econs*) 
                [P:Pat][V':MLval][pv:Prop][f:MLenv][pf:Prop]
                   ((<Pat> P=I)/\ (<MLval>V'=V))
                \/ ((~(<Pat>P=I))/\ (VAL_OF f I V))).
                
(*****************************************************************************)
(*                  We show now that VAL_OF->Val_inv                         *)
(*****************************************************************************)


Lemma of_inv.
Statement (e:MLenv)(I:Pat)(V:MLval)(VAL_OF e I V)->(Val_inv e I V).


Goal.
Induction 1;Intros;Unfold Val_inv.
Left;Split;Trivial.
Right;Split;Trivial.
Save.

Hint of_inv.


(*****************************************************************************)
(*         The following lemma shows us that VAL_OF is deterministic         *)
(*****************************************************************************)


Lemma determ_VAL_OF.
Statement (e:MLenv)(i:Pat)(V,V':MLval)
                (VAL_OF e i V')->
                    (VAL_OF e i V)->
                       (<MLval> V=V').


Goal.
Induction 1.
Intros.
Cut (Val_inv (Econs I a e0) I V).
Unfold Val_inv.
Induction 1;Induction 1;Auto.
Induction 1;Auto.
Auto.
Intros e0 X I a b VAL_a  Eq_V_a neq VAL_Econs.
Cut (Val_inv (Econs X b e0) I V).
Unfold Val_inv.
Induction 1;Induction 1;Auto.
Intro HH;Elim (neq HH).
Auto.
Save.


(*****************************************************************************)
(*                   We define now the Mini-ML semantics                     *)
(*In this semantics we want to distinguish the places where application is   *)
(*used in a recursive scheme from the other places.                          *)
(*That's why we have a special rule for application in the case where the    *)
(*function applied has as value a recursive closure (this correspond in fact *)
(*to a test during application                                               *)
(*****************************************************************************)



Inductive Definition ML_DS :MLenv->MLexp->MLval->Prop=

 BOOL:(b:bool)(e:MLenv)(ML_DS e (Bool b) (boolean b))

 |NUM:(n:nat)(e:MLenv)(ML_DS e (Num n) (num n))

 |Sem_OP:(c:OP)(e:MLenv)(ML_DS e (op c) (OP_clos c))

 |LAMBDA:(e:MLenv)(P:Pat)(E:MLexp)
               (ML_DS e (lambda P E) (Clos P E e))

 |IDENT:(e:MLenv)(v:MLval)(I:Pat)
               (VAL_OF e I v)->
                  (ML_DS e (id I) v)

 |ITE1:(e:MLenv)(E1,E2,E3:MLexp)(v:MLval)
               (ML_DS e E1 (boolean true))->
                  (ML_DS e E2 v)->
                     (ML_DS e (ite E1 E2 E3) v)

 |ITE2:(e:MLenv)(E1,E2,E3:MLexp)(v:MLval)
               (ML_DS e E1 (boolean false))->
                  (ML_DS e E3 v)->
                     (ML_DS e (ite E1 E2 E3) v)
 
 |MLPAIR:(e:MLenv)(E1,E2:MLexp)(u,v:MLval)
               (ML_DS e E1 u)->
                  (ML_DS e E2 v)->
                     (ML_DS e (mlpair E1 E2) (valpair u v))

 |APPml1:(e,e1:MLenv)(P:Pat)(E,E1,E2:MLexp)(u,v:MLval)
               (ML_DS e E1 (Clos P E e1))->
                 (ML_DS e E2 u)->
                   (ML_DS (Econs P u e1) E v)->
                      (ML_DS e (appl E1 E2) v)

 |APPml2:(e,e1:MLenv)(x,P:Pat)(E,E1,E2:MLexp)(u,v:MLval)
               (ML_DS e E1 (Clos_rec x E P e1))-> 
                 (ML_DS e E2 u)->
                   (ML_DS (Econs x u (Econs P (Clos_rec x E P e1) e1)) E v)->
                     (ML_DS e (appl E1 E2) v)

 |APPml_op:(e:MLenv)(E1,E2:MLexp)(n,m:nat)(c:OP)
           (ML_DS e E1 (OP_clos c))->
               (ML_DS e E2 (valpair (num n) (num m)))->
                     (ML_DS e (appl E1 E2) (num (eval_op c n m)))

 |Sem_let:(e:MLenv)(P:Pat)(E1,E2:MLexp)(u,v:MLval)
               (ML_DS e E2 u)->
                  (ML_DS (Econs P u e) E1 v)->
                     (ML_DS e (let' P E2 E1) v)

 |Sem_letrec:(e:MLenv)(P:Pat)(x:Pat)(E,E2:MLexp)(u:MLval)
               (ML_DS (Econs P (Clos_rec x E P e) e) E2 u)->
                  (ML_DS e (letrec P x E E2) u).   


(*****************************************************************************)
(*                 We now invert the preceding definition                    *)
(*****************************************************************************)

Definition Precise_closure=[e:MLenv][E1:MLexp][E2:MLexp][V:MLval]
 [u:MLval][v:MLval]
    (<[b:bool][p:(V_E b)]Prop>Match v with

    (*v is a boolean*)[b:bool]False

    (*v is an integer*)[n:nat]False

    (*v is a pair of  ML values*)
                [V1:MLval][p1:Prop][V2:MLval][p2:Prop]False

    (*v is an operator *)[c:OP]
             (<nat>Ex ([n:nat]
                (<nat>Ex ([m:nat]
                    (ML_DS e E2 (valpair (num n) (num m)))
                 /\ (<MLval> V=(num (eval_op c n m)))))))

    (*v is a closure*)
                [Q:Pat][E:MLexp][e1:MLenv][pe:Prop]
                (ML_DS e E2 u)
            /\  (ML_DS (Econs Q u e1) E V)

    (*v is a recursive closure*)
                [Q:Pat][E:MLexp][P:Pat][e1:MLenv][pe:Prop]
                (ML_DS e E2 u)
            /\  (ML_DS (Econs Q u (Econs P (Clos_rec Q E P e1) e1)) E V) 

    (*v is an environment Enil*)False
                
    (*v is an environment Econs*) 
                [P:Pat][V':MLval][pv:Prop][f:MLenv][pf:Prop]False).
                

(*****************************************************************************)


Definition App_inv=[e:MLenv][E1:MLexp][E2:MLexp][V:MLval]
    (<MLval>Ex ([u:MLval]
      (<MLval>Ex ([v:MLval]
         (ML_DS e E1 v)
      /\ (Precise_closure e E1 E2 V u v))))).


(*****************************************************************************)
(*                        ML_DS is now inverted                              *)
(*****************************************************************************)

Definition ML_inv=[e:MLenv][E:MLexp][V:MLval]
                   (<Prop>Match E with
   (*E is a boolean*)
                    [b:bool](<MLval> V=(boolean b))
   (*E is an integer*)   
                    [n:nat](<MLval> V=(num n))

   (*E is a predefinite operator*)
                    [c:OP](<MLval> V=(OP_clos c))

   (*E=ident*)       
                     [X:Pat](VAL_OF e X V)
                                  
         
   (*E=app*)         [E1:MLexp][raf:Prop][E2:MLexp][raf:Prop]
                     (App_inv e E1 E2 V)
  
   (*E=mlpair*)
                     [E1:MLexp][raf:Prop][E2:MLexp][raf:Prop]
                            <MLval>Ex([u:MLval]((ML_DS e E1 u )
                       /\   <MLval>Ex([v:MLval](ML_DS e E2 v )
                       /\   (<MLval> V=(valpair u v)))))

   (*E=lambda*)     
                     [P:Pat][E':MLexp][raf:Prop]
                     
                    (<MLval> V=(Clos P E' e))
   
   (*E=let*)
                     [P:Pat][E1:MLexp][raf:Prop][E2:MLexp][raf:Prop]
                            <MLval>Ex([u:MLval]
                            (ML_DS e E1 u)
                       /\   (ML_DS (Econs P u e) E2 V))
                          

   (*E=letrec*)     
                     [P:Pat][x:Pat][E1:MLexp][raf:Prop][E2:MLexp][raf:Prop]
                            (ML_DS (Econs P (Clos_rec x E1 P e) e) E2 V)


   (*E= if_then_else*)
                     [E1:MLexp][raf:Prop][E2:MLexp][raf:Prop]
                                     [E3:MLexp][raf:Prop]
                   ((ML_DS e E1 (boolean true))/\(ML_DS e E2 V))
               \/  ((ML_DS e E1 (boolean false))/\(ML_DS e E3 V))).


 
(*****************************************************************************)
(*             We get now the lemma ML_DS -> ML_inv                          *)
(*****************************************************************************)

Lemma ds_inv.
Statement (e:MLenv)(V:MLval)(E:MLexp)(ML_DS e E V)->(ML_inv e E V).

Goal.
Intros e V.
Pattern V.
Apply MLval_ind.


(*****************************************************************************)
(*     we show (E:MLexp) (ML_DS e E bool)->(ML_inv e E bool)                 *)
(*****************************************************************************)


Induction 1;Intros;Unfold ML_inv;Auto.
Exists u;Split;Auto;Exists v;Split;Auto.
Unfold App_inv;Exists u;Exists (Clos P E0 e1);Unfold Precise_closure;
Split;Auto. 
Unfold App_inv;Exists u;Exists (Clos_rec x E0 P e1);Unfold Precise_closure;
Split;Auto.
Unfold App_inv;Exists (valpair (num n) (num m));Exists (OP_clos c);
Unfold Precise_closure;Split;Auto.
Exists n;Exists m;Auto.
Exists u;Split;Auto.


(*****************************************************************************)
(*   (E:MLexp) (ML_DS e E n)->(ML_inv e E n)  for n an integer               *)
(*****************************************************************************)


Induction 1;Intros;Unfold ML_inv;Auto.
Exists u;Split;Auto;Exists v;Split;Auto.
Unfold App_inv;Exists u;Exists (Clos P E0 e1);Unfold Precise_closure;
Split;Auto. 
Unfold App_inv;Exists u;Exists (Clos_rec x E0 P e1);Unfold Precise_closure;
Split;Auto.
Unfold App_inv;Exists (valpair (num n) (num m));Exists (OP_clos c);
Unfold Precise_closure;Split;Auto.
Exists n0;Exists m;Auto.
Exists u;Split;Auto.


(*****************************************************************************)
(*                          We must now prove                                *)
(* (op:OP)(E:MLexp)(ML_DS e E (OP_clos op))->(ML_inv e E (OP_clos op))       *)
(*****************************************************************************)


Induction 1;Intros;Unfold ML_inv;Auto.
Exists u;Split;Auto;Exists v;Split;Auto.
Unfold App_inv;Exists u;Exists (Clos P E0 e1);Unfold Precise_closure;
Split;Auto. 
Unfold App_inv;Exists u;Exists (Clos_rec x E0 P e1);Unfold Precise_closure;
Split;Auto.
Unfold App_inv;Exists (valpair (num n) (num m));Exists (OP_clos c0);
Split;Auto;Unfold Precise_closure.
Exists n;Exists m;Auto.
Exists u;Split;Auto.


(*****************************************************************************)
(*The next step is                                                           *)
(* (a:MLval)                                                                 *)
(*  (b:MLval)                                                                *)
(*    ((E:MLexp) (ML_DS e E a)->(ML_inv1 e E a))->                           *)
(*     ((E:MLexp) (ML_DS e E b)->(ML_inv1 e E b))->                          *)
(*        (E:MLexp) (ML_DS e E (valpair a b))->(ML_inv e E (valpair a b))   *)
(*****************************************************************************)


Intros;Elim H1;Intros;Simpl;Auto.
Exists u;Split;Auto;Exists v;Split;Auto.
Unfold App_inv;Exists u;Exists (Clos P E0 e1);Unfold Precise_closure;
Split;Auto. 
Unfold App_inv;Exists u;Exists (Clos_rec x E0 P e1);Unfold Precise_closure;
Split;Auto.
Unfold App_inv;Exists (valpair (num n) (num m));Exists (OP_clos c);
Split;Auto;Unfold Precise_closure.
Exists n;Exists m;Auto.
Exists u;Split;Auto.


(*****************************************************************************)
(*Ensuite ,il s'agit de prouver                                              *)
(*    (E:MLexp)                                                              *)
(*     (e0:MLenv)                                                            *)
(*      (E0:MLexp) (ML_DS e E0 (Clos E e0))->(ML_inv e E0 (Clos E e0))      *)
(*****************************************************************************)

Intros;Elim H;Intros;Simpl;Auto.
Exists u;Split;Auto;Exists v;Split;Auto.
Unfold App_inv;Exists u;Exists (Clos P0 E1 e2);Unfold Precise_closure;
Split;Auto. 
Unfold App_inv;Exists u;Exists (Clos_rec x E1 P0 e2);Unfold Precise_closure;
Split;Auto.
Unfold App_inv;Exists (valpair (num n) (num m));Exists (OP_clos c);
Split;Auto;Unfold Precise_closure.
Exists n;Exists m;Auto.
Exists u;Split;Auto.


(*****************************************************************************)
(* at the end                                                                *)
(*(P:Pat)                                                                    *)
(*   (E:MLexp)                                                               *)
(*    (x:Pat)                                                                *)
(*     (e0:MLenv)                                                            *)
(*      (E0:MLexp)                                                           *)
(*      (ML_DS e E0 (Clos_rec P E x e0))->                                   *)
(*        (ML_inv e E0 (Clos_rec P E x e0))                                 *)
(*****************************************************************************)

Intros;Elim H;Intros;Simpl;Auto.
Exists u;Split;Auto;Exists v;Split;Auto.
Unfold App_inv;Exists u;Exists (Clos P0 E1 e2);Unfold Precise_closure;
Split;Auto. 
Unfold App_inv;Exists u;Exists (Clos_rec x0 E1 P0 e2);Unfold Precise_closure;
Split;Auto.
Unfold App_inv;Exists (valpair (num n) (num m));Exists (OP_clos c);
Split;Auto;Unfold Precise_closure.
Exists n;Exists m;Auto.
Exists u;Split;Auto.

Save.

Hint ds_inv.


(*****************************************************************************)
(*The following lemma shows that each ML term has exactly one ML value.      *)
(*To obtain the result we make great use of inversion                        *)
(*             Nevertheless we need the two following lemmae                 *)
(*****************************************************************************)


Definition Dif_true_false:MLval->Prop=[v:MLval]
  (<[b:bool][p:(V_E b)]Prop> Match v with
               (*v is a boolean*)
                [b:bool](<Prop>Match b with 
               (*b is true*) True
               (*b is false*)False)
                 [n:nat]False
                 [V1:MLval][p1:Prop][V2:MLval][p2:Prop]False
                 [c:OP]False
                 [x:Pat][E:MLexp][e':MLenv][pe:Prop]False
                 [x:Pat][E:MLexp][P:Pat][e:MLenv][pe:Prop]False
                 False
                 [P:Pat][V':MLval][pv:Prop][f:MLenv][pf:Prop]False).
                


Lemma notEq_Tr_Fa.
Statement ~<MLval>(boolean true)=(boolean false).

Goal.
Cut (<MLval>(boolean true)=(boolean false))->False.
Intro H.
Auto.
Cut (<MLval>(boolean true)=(boolean false))->
   (Dif_true_false (boolean true))->(Dif_true_false (boolean false)).
Unfold Dif_true_false.
Intros;Elim H.
2:Auto.
Exact H0.
Intro Hyp.
Elim Hyp.
Intro;Assumption.

Save.


Definition Dif_pair_left:MLval->MLval->Prop=
[u:MLval][Val:MLval](<[b:bool][p:(V_E b)]Prop>Match Val with 
      [b:bool]False
      [n:nat]False
       (*Case of the pair*)
      [V1:MLval][p1:Prop][V2:MLval][p2:Prop](<MLval> u=V1)
      [c:OP]False
      [x:Pat][E:MLexp][e':MLenv][pe:Prop]False
      [x:Pat][E:MLexp][P:Pat][e:MLenv][pe:Prop]False
      False
      [P:Pat][V':MLval][pv:Prop][f:MLenv][pf:Prop]False).


Definition Dif_pair_right:MLval->MLval->Prop=
[u:MLval][Val:MLval](<[b:bool][p:(V_E b)]Prop>Match Val with 
      [b:bool]False
      [n:nat]False
       (*Case of the pair*)
      [V1:MLval][p1:Prop][V2:MLval][p2:Prop](<MLval> u=V2)
      [c:OP]False
      [x:Pat][E:MLexp][e':MLenv][pe:Prop]False
      [x:Pat][E:MLexp][P:Pat][e:MLenv][pe:Prop]False
      False
      [P:Pat][V':MLval][pv:Prop][f:MLenv][pf:Prop]False).

                

Lemma pair1:(a,b,c,d:MLval)(<MLval>(valpair a b)=(valpair c d))->
          ((<MLval> a=c)/\(<MLval> b=d)).


Goal.
Intros.
Split.
Cut (<MLval>(valpair a b)=(valpair c d))->(Dif_pair_left a (valpair a b))->
                               (Dif_pair_left a (valpair c d)).
Unfold Dif_pair_left.
Intro Hyp.
Elim Hyp;Auto.
Induction 1;Intro Hyp;Assumption.
Cut (<MLval>(valpair a b)=(valpair c d))->(Dif_pair_right b (valpair a b))->
                    (Dif_pair_right b (valpair c d)).
Unfold Dif_pair_right.
Intro Hyp.
Elim Hyp;Auto.
Induction 1;Intro Hyp;Assumption.

Save.

Definition Dif_num:nat->MLval->Prop=
[u:nat][Val:MLval](<[b:bool][p:(V_E b)]Prop>Match Val with 
      [b:bool]False
      [n:nat](<nat>n=u)
       (*Case of the pair*)
      [V1:MLval][p1:Prop][V2:MLval][p2:Prop]False
      [c:OP]False
      [x:Pat][E:MLexp][e':MLenv][pe:Prop]False
      [x:Pat][E:MLexp][P:Pat][e:MLenv][pe:Prop]False
      False
      [P:Pat][V':MLval][pv:Prop][f:MLenv][pf:Prop]False).


Lemma pair2: (n,m:nat)(<MLval> (num n)=(num m))->(<nat>n=m).

Goal.
Intros.
Cut (<MLval> (num n)=(num m))->(Dif_num n (num n))->(Dif_num n (num m)).
Unfold Dif_num.
Intro Hyp.
Elim Hyp;Auto.
Induction 1;Intro;Assumption.

Save.

Lemma Pairing_inject: (n,m,p,q:nat)
        (<MLval> (valpair (num n) (num m))=(valpair (num p) (num q)))->
                 ((<nat> n=p)/\ (<nat> m=q)). 

Goal.
Intros.
Cut ((<MLval> (num n)=(num p)) /\ (<MLval> (num m)=(num q))).
2:Exact (pair1 (num n) (num m) (num p) (num q) H). 
Intro Hyp;Elim Hyp;Intros Hyp1 Hyp2.
Split.
Exact (pair2 n p Hyp1).
Exact (pair2 m q Hyp2).

Save.

          (***********************************************)

Lemma ML_DS_determ.
Statement  (e:MLenv)(E:MLexp)(V:MLval)(ML_DS e E V)->
              (V':MLval)
                 (ML_DS e E V')->
                    (<MLval> V=V').


Goal.
Induction 1;Intros.

Cut (ML_inv e0 (Bool b) V').
Unfold ML_inv;Auto.
Auto.

Cut (ML_inv e0 (Num n) V').
Unfold ML_inv;Auto.
Auto.

Cut (ML_inv e0 (op c) V').
Unfold ML_inv;Auto.
Auto.

Cut (ML_inv e0 (lambda P E0) V').
Unfold ML_inv;Auto.
Auto.

Cut (ML_inv e0 (id I) V').
Unfold ML_inv.
Intro.
Apply determ_VAL_OF with e0 I;Auto.
Auto.

Cut (ML_inv e0 (ite E1 E2 E3) V'). 
Unfold ML_inv.
Induction 1;Induction 1;Auto.
Intros.
Cut <MLval> (boolean true)=(boolean false).
Intro.
Elim (notEq_Tr_Fa H9).
Elim (H1 (boolean false)  H7).
Trivial.
Auto.

Cut (ML_inv e0 (ite E1 E2 E3) V').
Unfold ML_inv.
Induction 1;Induction 1;Auto.
Intros.
Cut <MLval> (boolean true)=(boolean false).
Intro.
Elim (notEq_Tr_Fa H9).
Elim (H1 (boolean true) H7).
Trivial.
Auto.

Cut (ML_inv e0 (mlpair E1 E2) V').
Unfold ML_inv.
Intro.
Elim H5;Intros.
Elim H6;Intros.
Elim H8;Intros.
Elim H9.
Intro.
Cut <MLval> v=x0.
Intro;Elim H11.
Cut <MLval> u=x.
Intro;Elim H12.
Auto.
Elim (H1 x H7);Auto.
Elim (H3 x0 H10);Auto.
Auto.

Cut (ML_inv e0 (appl E1 E2) V').
Unfold ML_inv.
Unfold App_inv.
Intro.
Elim H7;Intros.
Elim H8;Intros.
Elim H9;Intros.
Cut (<MLval>(Clos P E0 e1)=x0);Intros.
2:Exact (H1 x0 H10).
Cut  (Precise_closure e0 E1 E2 V' x (Clos P E0 e1)).
2:Rewrite ->H12;Assumption.
Unfold Precise_closure;Intro.
Elim H13;Intros.
Cut (<MLval>u=x);Intros.
2:Exact (H3 x H14).
Cut (ML_DS (Econs P x e1) E0 V').
Rewrite <- H16;Intros.
Exact (H5 V' H17).
Auto.
Auto.

Cut (ML_inv e0 (appl E1 E2) V').
Unfold ML_inv.
Unfold App_inv.
Intro.
Elim H7;Intros.
Elim H8;Intros.
Elim H9;Intros.
Cut (<MLval>(Clos_rec x E0 P e1)=x1);Intros.
2:Exact (H1 x1 H10).
Cut  (Precise_closure e0 E1 E2 V' x0 (Clos_rec x E0 P e1)).
2:Rewrite ->H12;Assumption.
Unfold Precise_closure;Intro.
Elim H13;Intros.
Cut (<MLval>u=x0);Intros.
2:Exact (H3 x0 H14).
Cut (ML_DS (Econs x x0 (Econs P (Clos_rec x E0 P e1) e1)) E0 V').
Rewrite <- H16;Intros.
Exact (H5 V' H17).
Auto.
Auto.

Cut (ML_inv e0 (appl E1 E2) V').
Unfold ML_inv.
Unfold App_inv.
Intro.
Elim H5;Intros.
Elim H6;Intros.
Elim H7;Intros.
Elim (H1 x0 H8);Intros.
Cut (<MLval> (OP_clos c)=x0).
2:Exact (H1 x0 H8).
Intro.
Cut (Precise_closure e0 E1 E2 V' x x0).
2:Auto.
Rewrite <- H10.
Unfold Precise_closure.
Intro.
Elim H11;Intros.
Elim H12;Intros.
Elim H13;Intros.
Cut (<MLval> (valpair (num n) (num m))=(valpair (num x1) (num x2))).
2:Exact (H3 (valpair (num x1) (num x2)) H14).
Intro.
Cut ((<nat> n=x1)/\ (<nat>m=x2)).
Intro.
2:Exact (Pairing_inject n m x1 x2 H16).
Elim H17;Intros.
Cut <MLval>V'=(num (eval_op c x1 x2)).
Rewrite H18;Rewrite H19.
Induction 1;Auto.
Assumption.
Auto.



Cut (ML_inv e0 (let' P E2 E1) V').
2:Auto.
Unfold ML_inv.
Intro.
Elim H5;Intros.
Elim H6;Intros.
Cut <MLval> u=x.
Intro.
Cut (ML_DS (Econs P u e0) E1 V').
Intro.
Elim  (H3 V' H10).
Auto.
Cut (ML_DS (Econs P x e0) E1 V').
Elim H9;Auto.
Auto.
Elim (H1 x H7);Auto.

Cut  (ML_inv e0 (letrec P x E0 E2) V').
2:Auto.
Unfold ML_inv.
Intro.
Elim (H1 V' H3).
Auto.

Save.


(*****************************************************************************)
(*We define now a new inversion of ML_DS which is used to resolve the case   *)
(*of the compilation of an application                                       *)
(* This shows us that there is often many way of inverting a definition  and *)
(*that the user must choose the most suitable one in view of its use         *)
(*****************************************************************************)



Definition ML_invAppProof=[e:MLenv][E:MLexp][V:MLval]
                   (<Prop>Match E with
   (*E is a boolean*)[b:bool](<MLval> V=(boolean b))

   (*E is an integer*)[n:nat](<MLval> V=(num n))

   (*E is an operator*)[c:OP](<MLval> V=(OP_clos c))
                   
   (*E=ident*)       
                     [X:Pat](VAL_OF e X V)
                                  
         
   (*E=app*)         [E1:MLexp][raf:Prop][E2:MLexp][raf:Prop]
((<MLval>Ex ([u:MLval]
    (<Pat>Ex([P:Pat](<MLexp>Ex([E':MLexp](<MLenv>Ex([e1:MLenv]   
                (ML_DS e E1 (Clos P E' e1))                
            /\  (ML_DS e E2 u)
            /\  (ML_DS (Econs P u e1) E' V)))))))))
\/
(<MLval>Ex ([u:MLval]
  (<Pat>Ex([Q:Pat](<MLexp>Ex([E':MLexp](<Pat>Ex ([P:Pat](<MLenv>Ex([e1:MLenv]  
        (ML_DS e E1 (Clos_rec Q E' P e1))
    /\  (ML_DS e E2 u)
    /\  (ML_DS (Econs Q u (Econs P (Clos_rec Q E' P e1) e1)) E' V)))))))))))) 
\/
(<OP> Ex([c:OP]
    (ML_DS e E1 (OP_clos c))
 /\ (<nat> Ex([n:nat] (<nat> Ex([m:nat]
           (ML_DS e E2 (valpair (num n) (num m)))
        /\ (<MLval> V=(num (eval_op c n m)))))))))
 
   (*E=mlpair*)
                     [E1:MLexp][raf:Prop][E2:MLexp][raf:Prop]
                            <MLval>Ex([u:MLval]((ML_DS e E1 u )
                       /\   <MLval>Ex([v:MLval](ML_DS e E2 v )
                       /\   (<MLval> V=(valpair u v)))))

   (*E=lambda*)     
                     [P:Pat][E':MLexp][raf:Prop]
                     
                    (<MLval> V=(Clos P E' e))
   
   (*E=let*)
                     [P:Pat][E1:MLexp][raf:Prop][E2:MLexp][raf:Prop]
                            <MLval>Ex([u:MLval]
                            (ML_DS e E1 u)
                       /\   (ML_DS (Econs P u e) E2 V))
                          

   (*E=letrec*)     
                     [P:Pat][x:Pat][E1:MLexp][raf:Prop][E2:MLexp][raf:Prop]
                            (ML_DS (Econs P (Clos_rec x E1 P e) e) E2 V)


   (*E= if_then_else*)
                     [E1:MLexp][raf:Prop][E2:MLexp][raf:Prop]
                                     [E3:MLexp][raf:Prop]
                   ((ML_DS e E1 (boolean true))/\(ML_DS e E2 V))
               \/  ((ML_DS e E1 (boolean false))/\(ML_DS e E3 V))).



(*****************************************************************************)
(* We prove now that the preceding definition is an inversion                *)
(*****************************************************************************)

Lemma DS_inv_App.
Statement (e:MLenv)(E:MLexp)(V:MLval)
         (ML_DS e E V)->(ML_invAppProof e E V).


Goal.
Intros e E V.
Pattern V.
Apply MLval_ind.

Induction 1;Intros;Unfold ML_invAppProof;Auto.
Exists u;Split;Auto;Exists v;Split;Auto.
Left;Left;Exists u;Exists P;Exists E0;Exists e1;Split;Auto.
Left;Right;Exists u;Exists x;Exists E0;Exists P;Exists e1;Split;Auto.
Right;Exists c;Split;Auto;Exists n;Exists m;Split;Auto.
Exists u;Split;Auto.

Induction 1;Intros;Unfold ML_invAppProof;Auto.
Exists u;Split;Auto;Exists v;Split;Auto.
Left;Left;Exists u;Exists P;Exists E0;Exists e1;Split;Auto.
Left;Right;Exists u;Exists x;Exists E0;Exists P;Exists e1;Split;Auto.
Right;Exists c;Split;Auto;Exists n0;Exists m;Split;Auto.
Exists u;Split;Auto.

Induction 1;Intros;Unfold ML_invAppProof;Auto.
Exists u;Split;Auto;Exists v;Split;Auto.
Left;Left;Exists u;Exists P;Exists E0;Exists e1;Split;Auto.
Left;Right;Exists u;Exists x;Exists E0;Exists P;Exists e1;Split;Auto.
Right;Exists c0;Split;Auto;Exists n;Exists m;Split;Auto.
Exists u;Split;Auto.

Intros;Elim H1;Intros;Simpl;Auto.
Exists u;Split;Auto;Exists v;Split;Auto.
Left;Left;Exists u;Exists P;Exists E0;Exists e1;Split;Auto.
Left;Right;Exists u;Exists x;Exists E0;Exists P;Exists e1;Split;Auto.
Right;Exists c;Split;Auto;Exists n;Exists m;Split;Auto.
Exists u;Split;Auto.

Intros;Elim H;Intros;Simpl;Auto.
Exists u;Split;Auto;Exists v;Split;Auto.
Left;Left;Exists u;Exists P0;Exists E1;Exists e2;Split;Auto.
Left;Right;Exists u;Exists x;Exists E1;Exists P0;Exists e2;Split;Auto.
Right;Exists c;Split;Auto;Exists n;Exists m;Split;Auto.
Exists u;Split;Auto.

Intros;Elim H;Intros;Simpl;Auto.
Exists u;Split;Auto;Exists v;Split;Auto.
Left;Left;Exists u;Exists P0;Exists E1;Exists e2;Split;Auto.
Left;Right;Exists u;Exists x0;Exists E1;Exists P0;Exists e2;Split;Auto.
Right;Exists c;Split;Auto;Exists n;Exists m;Split;Auto.
Exists u;Split;Auto.

Save.

(*****************************************************************************)
(*We present now the definitions that concerns the Categorical Abstract      *)
(*Machine. The Set Commande defines its syntax                               *)
(*****************************************************************************)

Inductive Set Value=
null:Value
|elem:bool->Value
|int:nat->Value
|def_op:OP->Value.

(*****************************************************************************)

Inductive Set Commande=
quote:Value->Commande
 |car:Commande
 |cdr:Commande
 |cons:Commande
 |push:Commande
 |swap:Commande
 |branch:Commande->Commande->Commande
 |cur:Commande->Commande
 |cur_rec:Commande->Commande
 |app:Commande
 |o:Commande->Commande->Commande.

(*****************************************************************************)

Inductive Set CSem_val=
 val:Value->CSem_val
 |Cam_pair:CSem_val->CSem_val->CSem_val
 |Cam_clos:Commande->CSem_val->CSem_val 
 |Cam_clos_rec:Commande->CSem_val->CSem_val  
 |Cam_nil:CSem_val.

(*****************************************************************************)

Inductive Set Etat=
  nil:Etat
  |ETcons:CSem_val->Etat->Etat.
           
(*****************************************************************************)
(*                   We get now the Semantics of the CAM                     *)
(*****************************************************************************)


Inductive Definition CAM_DS:Etat->Commande->Etat->Prop= 


  QUO:(s:Etat)(a:CSem_val)(b:Value)
        (CAM_DS (ETcons a s) (quote b) (ETcons (val b) s))

 |CAR:(s:Etat)(a,b:CSem_val)
        (CAM_DS (ETcons (Cam_pair a b) s) car (ETcons a s))

 |CDR:(s:Etat)(a,b:CSem_val)
        (CAM_DS (ETcons (Cam_pair a b) s) cdr (ETcons b s))

 |CONS:(s:Etat)(a,b:CSem_val)
        (CAM_DS (ETcons b (ETcons a s)) cons (ETcons (Cam_pair a b) s))

 |PUSH:(s:Etat)(a:CSem_val)
        (CAM_DS (ETcons a s) push (ETcons a (ETcons a s)))
 
 |SWAP:(s:Etat)(a,b:CSem_val)
        (CAM_DS (ETcons a (ETcons b s)) swap (ETcons b (ETcons a s)))
          
 |BRANCHT:(s,s1:Etat)(c1,c2:Commande)
        (CAM_DS s c1 s1)->
            (CAM_DS (ETcons (val (elem true)) s) (branch c1 c2) s1)
 
 |BRANCHF:(s,s2:Etat)(c1,c2:Commande)
        (CAM_DS s c2 s2)->
            (CAM_DS (ETcons (val (elem false)) s) (branch c1 c2) s2)

 |CUR:(s:Etat)(a:CSem_val)(c:Commande)
        (CAM_DS (ETcons a s) (cur c) (ETcons (Cam_clos c a) s))

 |APPcam1:(s,s1:Etat)(a,b:CSem_val)(c:Commande)
        (CAM_DS (ETcons (Cam_pair a b) s) c s1)->
            (CAM_DS (ETcons (Cam_pair (Cam_clos c a) b) s) app s1)
 
 |APPcam2:(s,s1:Etat)(a,b:CSem_val)(c:Commande)
    (CAM_DS (ETcons (Cam_pair (Cam_pair b (Cam_clos_rec c b)) a) s) c s1)->
            (CAM_DS (ETcons (Cam_pair (Cam_clos_rec c b) a) s) app s1)

 |APPcam_op:(s:Etat)(n,m:nat)(oper:OP)
     (CAM_DS (ETcons (Cam_pair (val (def_op oper))
       (Cam_pair (val (int n)) (val (int m)))) s) app 
           (ETcons (val (int (eval_op oper n m))) s)) 

 |CUR_REC:(s:Etat)(a:CSem_val)(c:Commande)
      (CAM_DS (ETcons a s) (cur_rec c) (ETcons (Cam_clos_rec c a) s))

 |o_DS:(s,s1,s2:Etat)(c1,c2:Commande)
       (CAM_DS s c1 s1)->
           (CAM_DS s1 c2 s2)->
               (CAM_DS s (o c1 c2) s2).
             

(*****************************************************************************)
(*We now define the "Squelette"of an environment which is the intuitive      *)
(*corresponding list of Debruijn indexes of an environment                   *)
(*****************************************************************************)


Inductive Set Squelette=
nil_squelette:Squelette
|cons_squelette:Pat->Squelette->Squelette.

(*****************************************************************************)

Inductive Definition Habite :MLenv->Squelette->Prop=

triv_habite:(Habite Enil nil_squelette)

|cons_habite:(x:Pat)(u:MLval)(e:MLenv)(s:Squelette)
              (Habite e s)->
               (Habite (Econs x u e) (cons_squelette x s)).


(*****************************************************************************)

Definition Habite_inv=[e:MLenv][s:Squelette]
       (<Prop>Match s with
(*cas ou s est nil_squelette*)
          (<MLenv> e =Enil)
(*cas ou s est (cons_squelette p s0)*)
    [p:Pat][s0:Squelette][Ps0:Prop]
  (<MLval>Ex ([V:MLval]
    (<MLenv>Ex ([e0:MLenv]
       (<MLenv> e=(Econs p V e0))
    /\ (Habite e0 s0)))))).

(*****************************************************************************)

Lemma Habite_Habite_inv.
Statement  (s:Squelette)(e:MLenv)
             (Habite e s)->(Habite_inv e s).


Goal.
Induction 1;Intros;Unfold Habite_inv.
Auto.
Exists u;Exists e0;Split;Auto.
Save.


(*****************************************************************************)
(*  The Acces predicate is defined using the notion of Squelette             *)
(* It describes how to reach an identifier in an environment                 *)
(*****************************************************************************)



Inductive Definition Access:Pat->Squelette->Commande->Prop=


  Rule1:(P:Pat)(s:Squelette)
                (Access P (cons_squelette P s) cdr)

  |Rule2:(P,T:Pat)(s:Squelette)(C:Commande)
               (~<Pat> P=T)->
                (Access P s C)->
                 (Access P (cons_squelette T s) (o car C)).


(*****************************************************************************)


Definition Access_inv=[P:Pat][S:Squelette][C:Commande]
   (<Prop>Match S with
     (*nil_squelette*) False
     (*(cons_squelette x s0)*)
     [x:Pat][s0:Squelette][Ps:Prop]
              ((<Pat> x=P)/\ (<Commande> C=cdr))
           \/ ((~<Pat> P=x)/\ 
                (<Commande>Ex ([C0:Commande]
                    (Access P s0 C0)
                 /\ (<Commande> C=(o car C0)))))).
          
 
(*****************************************************************************)
(*             We prove now  Access->Access_inv                              *)
(*****************************************************************************)

Lemma Access_Access_inv.
Statement (p:Pat)(S:Squelette)(C:Commande)
           (Access p S C)->(Access_inv p S C).

Goal.
Intros p S C.
Induction 1;Intros.
Simpl.
Left.
Split;Auto.
Simpl.
Right.
Split.
Auto.
Exists C0;Auto.

Save. 

(*****************************************************************************)
(*   The following small lemma shows that Access is injective                *)
(*****************************************************************************)

Lemma Access_inject.
Statement (x:Pat)(s:Squelette)(C,C':Commande)
            (Access x s C')->
              (Access x s C)->
                (<Commande> C=C').


Goal.
Intro.
Induction s;Intros.
Cut (Access_inv x nil_squelette C).
Simpl;Intros.
Elim H1.
Exact (Access_Access_inv x nil_squelette C H0).
Cut (Access_inv x (cons_squelette p y) C).
2:Exact (Access_Access_inv x (cons_squelette p y) C H1).
Simpl;Intros.
Elim H2;Intros.
Elim H3;Intros.
Rewrite -> H5.
Cut (Access_inv x (cons_squelette p y ) C').
2:Exact (Access_Access_inv x (cons_squelette p y) C' H0).
Rewrite -> H4.
Simpl;Intros.
Elim H6;Intros.
Elim H7;Intros.
Rewrite -> H9.
Auto.
Elim H7;Intros.
Elim H8.
Trivial.
Elim H3;Intros.
Cut (Access_inv x (cons_squelette p y ) C').
2:Exact (Access_Access_inv x (cons_squelette p y) C' H0).
Simpl;Intros.
Elim H6;Intros.
Elim H7;Intros.
Cut False.
Intro;Elim H10.
Cut (<Pat>x=p)->(~<Pat>x=p).
Intro.
Elim H10.
Rewrite -> H8;Auto.
Rewrite -> H8;Auto.
Intro.
Auto.
Elim H7;Intros.
Elim H9;Intro C0';Intros.
Elim H5;Intro C0;Intros.
Elim H10;Intros.
Elim H11;Intros.
Rewrite -> H13;Rewrite -> H15.
Elim (H C0 C0' H12 H14);Intros.
Auto.

Save.


(*****************************************************************************)

Lemma Squelet.
Statement  (e:MLenv)(<Squelette>Ex ([s:Squelette](Habite e s))).

Goal.
Intro.
Pattern e.
Apply MLenv_ind.
Exists nil_squelette.
Exact triv_habite.
Intros.
Elim H;Intro s;Intro.
Exists (cons_squelette P s).
Exact (cons_habite P ml e1 s H0).
Save.


(*****************************************************************************)
(*It's now time to define the translation of ML code onto CAM code           *)
(*****************************************************************************)


Inductive Definition Traduction:Squelette->MLexp->Commande->Prop=
            

Bool_Trad:(b:bool)(S:Squelette)
              (Traduction S (Bool b) (quote (elem b)))

|Trad_num:(n:nat)(S:Squelette)
              (Traduction S (Num n) (quote (int n)))

|Trad_clos:(c:OP)(S:Squelette)
              (Traduction S (op c) (quote (def_op c)))

|Trad_var:(p:Pat)(S:Squelette)(C:Commande)
              (Access p S C)->(Traduction S (id p) C)


|Trad_ite:(S:Squelette)(E1,E2,E3:MLexp)(C1,C2,C3:Commande)
              (Traduction S E1 C1)->
                (Traduction S E2 C2)->
                  (Traduction S E3 C3)->
  (Traduction S (ite E1 E2 E3) (o push (o C1 (branch C2 C3))))


|Trad_pair:(S:Squelette)(E1,E2:MLexp)(C1,C2:Commande)
               (Traduction S E1 C1)->
                 (Traduction S E2 C2)->
  (Traduction S (mlpair E1 E2) (o push (o C1 (o swap (o C2 cons)))))


|Trad_app:(S:Squelette)(E1,E2:MLexp)(C1,C2:Commande)
               (Traduction S E1 C1)->
                 (Traduction S E2 C2)->
  (Traduction S (appl E1 E2) (o push (o C1 (o swap (o C2 (o cons app))))))


|Trad_let:(p:Pat)(S:Squelette)(E1,E2:MLexp)(C1,C2:Commande)
              (Traduction S E1 C1)->
                 (Traduction (cons_squelette p S) E2 C2)->
  (Traduction S (let' p E1 E2) (o push (o C1 (o cons C2))))


|Trad_let_rec:(p,x:Pat)(S:Squelette)(E,E2:MLexp)(C,C2:Commande)
 (Traduction (cons_squelette x (cons_squelette p S)) E C)->
                 (Traduction (cons_squelette p S) E2 C2)->
  (Traduction S (letrec p x E E2) (o push (o (cur_rec C) (o cons C2))))


|Trad_lambda:(S:Squelette)(p:Pat)(E:MLexp)(C:Commande)
               (Traduction (cons_squelette p S) E C)->
                 (Traduction S (lambda p E) (cur C)). 
              
(*****************************************************************************)
(*             As usual, we invert this important definition                 *)
(*****************************************************************************)


Definition Trad_inv=[s:Squelette][E:MLexp][C:Commande]
    (<Prop>Match E with

(*case where E is a boolean*) [b:bool](<Commande> C=(quote (elem b)))

(*case where E is an integer*) [n:nat](<Commande> C=(quote (int n)))

(*case where E is a predefinite operator*) [c:OP]
                                       (<Commande> C=(quote (def_op c)))

(*case where E is an identifier*)
          [x:Pat](Access x s C)

(*case where E=appl*)
        [E1:MLexp][P1:Prop]
                   [E2:MLexp][P2:Prop]
             (<Commande>Ex ([C1:Commande]
               (Traduction s E1 C1)
             /\ (<Commande>Ex ([C2:Commande]
                  (Traduction s E2 C2)
 /\  (<Commande> C=(o push (o C1 (o swap (o C2 
               (o cons app))))))
                 ))))
              

(*case where E is a  pair*)
              [E1:MLexp][P1:Prop][E2:MLexp][P2:Prop]
        (<Commande>Ex ([C1:Commande]
               (Traduction s E1 C1)
             /\ (<Commande>Ex ([C2:Commande]
                  (Traduction s E2 C2)
  /\  (<Commande> C=(o push (o C1 (o swap (o C2 cons)))))))))


(*case where E is an abstraction*)
               [p:Pat][E1:MLexp][P1:Prop]
       (<Commande>Ex ([C1:Commande]
          (Traduction (cons_squelette p s) E1 C1)
       /\ (<Commande> C=(cur C1))))

(*case where E is a let*)
               [p:Pat][E1:MLexp][P1:Prop][E2:MLexp][P2:Prop]
        (<Commande>Ex ([C1:Commande]
            (Traduction s E1 C1)
         /\ (<Commande>Ex ([C2:Commande] 
                (Traduction (cons_squelette p s) E2 C2)
      /\ (<Commande> C=(o push (o C1 (o cons C2))))))))

(*case where E is a let_rec*)
         [p,x:Pat][E1:MLexp][P1:Prop][E2:MLexp][P2:Prop]
        (<Commande>Ex ([C1:Commande]
            (Traduction (cons_squelette x (cons_squelette p s)) E1 C1)
         /\ (<Commande>Ex ([C2:Commande]
             (Traduction (cons_squelette p s) E2 C2)
    /\ (<Commande> C=(o push (o (cur_rec C1) (o cons C2))))))))
            
(*case where E is of the form if_then_else*)
     [E1:MLexp][P1:Prop][E2:MLexp][P2:Prop][E3:MLexp][P3:Prop]
         (<Commande>Ex ([C1:Commande]
                (Traduction s E1 C1)
          /\  (<Commande>Ex  ([C2:Commande]
                   (Traduction s E2 C2)
             /\  (<Commande>Ex ([C3:Commande]
                      (Traduction s E3 C3)
  /\ (<Commande> C=(o push (o C1 (branch C2 C3))))))))))).


(*****************************************************************************)
(*                We prove now that Traduction->Trad_inv                     *)
(*****************************************************************************)

Lemma Trad_Trad_inv.
Statement (s:Squelette)(E:MLexp)(C:Commande)
              (Traduction s E C)->(Trad_inv s E C).

Goal.
Intros s E C.
Induction 1;Intros;Simpl;Intros;Auto.

Exists C1;Split.
Exact H0.
Exists C2;Split.
Exact H2.
Exists C3;Split.
Exact H4.
Trivial.

Exists C1;Split.
Exact H0.
Exists C2;Split.
Exact H2.
Trivial.

Exists C1;Split.
Exact H0.
Exists C2;Split.
Exact H2.
Trivial.

Exists C1;Split.
Exact H0.
Exists C2;Split.
Exact H2.
Trivial.

Exists C0;Split.
Exact H0.
Exists C2;Split.
Exact H2.
Trivial.

Exists C0;Split.
Exact H0.
Trivial.

Save.

Hint Trad_Trad_inv.


(*****************************************************************************)
(* We use the preceding lemma to show that the predicate traduction is       *)
(* injective                                                                 *)
(*****************************************************************************)

Lemma Traduction_inject.
Statement  (E:MLexp)(C,C':Commande)(s:Squelette)
              (Traduction s E C)->
                 (Traduction s E C')->
                    (<Commande> C=C').



Goal.

Induction E;Intros.


Cut (Trad_inv s (Bool b) C).
Simpl;Intros.
2:Apply Trad_Trad_inv;Trivial.
Cut (Trad_inv s (Bool b) C').
Simpl;Intros.
2:Apply Trad_Trad_inv;Trivial.
Rewrite -> H1.
Rewrite -> H2.
Trivial.


Cut (Trad_inv s (Num n) C).
Simpl;Intros.
2:Apply Trad_Trad_inv;Trivial.
Cut (Trad_inv s (Num n) C').
Simpl;Intros.
2:Apply Trad_Trad_inv;Trivial.
Rewrite -> H1.
Rewrite -> H2.
Trivial.


Cut (Trad_inv s (op o) C).
Simpl;Intros.
2:Apply Trad_Trad_inv;Trivial.
Cut (Trad_inv s (op o) C').
Simpl;Intros.
2:Apply Trad_Trad_inv;Trivial.
Rewrite -> H1.
Rewrite -> H2.
Trivial.


Cut (Trad_inv s (id p) C).
2:Apply Trad_Trad_inv;Trivial.
Simpl;Intro.
Cut (Trad_inv s (id p) C').
2:Apply Trad_Trad_inv;Trivial.
Simpl;Intro.
Exact (Access_inject p s C C' H2 H1).


Cut (Trad_inv s (appl y y0) C').
2:Apply Trad_Trad_inv;Trivial.
Simpl;Intros.
Cut (Trad_inv s (appl y y0) C).
2:Apply Trad_Trad_inv;Trivial.
Simpl;Intros.
Elim H4;Intro C1;Intros.
Elim H5;Intros.
Elim H7;Intro C2;Intros.
Elim H3;Intro C'1;Intros.
Elim H9;Intros.
Elim H11;Intro C'2;Intros.
Elim H12;Intros.
Elim H8;Intros.
Rewrite -> H14.
Rewrite -> H16.
Elim (H0 C2 C'2 s H15 H13);Intros.
Elim (H C1 C'1 s H6 H10);Intros.
Trivial.


Cut (Trad_inv s (mlpair y y0) C').
2:Apply Trad_Trad_inv;Trivial.
Simpl;Intros.
Cut (Trad_inv s (mlpair y y0) C).
2:Apply Trad_Trad_inv;Trivial.
Simpl;Intros.
Elim H4;Intro C1;Intros.
Elim H5;Intros.
Elim H7;Intro C2;Intros.
Elim H3;Intro C'1;Intros.
Elim H9;Intros.
Elim H11;Intro C'2;Intros.
Elim H12;Intros.
Elim H8;Intros.
Rewrite -> H14.
Rewrite -> H16.
Elim (H0 C2 C'2 s H15 H13);Intros.
Elim (H C1 C'1 s H6 H10);Intros.
Trivial.


Cut (Trad_inv s (lambda p y) C').
2:Apply Trad_Trad_inv;Trivial.
Simpl;Intros.
Cut (Trad_inv s (lambda p y) C).
2:Apply Trad_Trad_inv;Trivial.
Simpl;Intros.
Elim H3;Intro C1;Intros.
Elim H4;Intros.
Elim H2;Intro C'1;Intros.
Elim H7;Intros.
Rewrite -> H9.
Rewrite -> H6.
Elim (H C1 C'1 (cons_squelette p s) H5 H8).
Auto.


Cut (Trad_inv s (let' p y y0) C').
2:Apply Trad_Trad_inv;Trivial.
Simpl;Intros.
Cut (Trad_inv s (let' p y y0) C).
2:Apply Trad_Trad_inv;Trivial.
Simpl;Intros.
Elim H4;Intro C1;Intros.
Elim H5;Intros.
Elim H7;Intro C2;Intros.
Elim H8;Intros.
Elim H3;Intro C'1;Intros.
Elim H11;Intros.
Elim H13;Intro C'2;Intros.
Elim H14;Intros.
Rewrite -> H16.
Rewrite -> H10.
Elim (H C1 C'1 s H6 H12).
Elim (H0 C2 C'2 (cons_squelette p s) H9 H15).
Auto.


Cut (Trad_inv s (letrec p p0 y y0) C').
2:Apply Trad_Trad_inv;Trivial.
Simpl;Intros.
Cut (Trad_inv s (letrec p p0 y y0) C).
2:Apply Trad_Trad_inv;Trivial.
Simpl;Intros.
Elim H4;Intro C1;Intros.
Elim H5;Intros.
Elim H7;Intro C2;Intros.
Elim H8;Intros.
Elim H3;Intro C'1;Intros.
Elim H11;Intros.
Elim H13;Intro C'2;Intros.
Elim H14;Intros.
Rewrite -> H16.
Rewrite -> H10.
Elim (H C1 C'1 (cons_squelette p0 (cons_squelette p s)) H6 H12).
Elim (H0 C2 C'2 (cons_squelette p s) H9 H15).
Auto.


Cut (Trad_inv s (ite y y0 y1) C').
2:Apply Trad_Trad_inv;Trivial.
Simpl;Intros.
Cut (Trad_inv s (ite y y0 y1) C).
2:Apply Trad_Trad_inv;Trivial.
Simpl;Intros.
Elim H5;Intro C1;Intros.
Elim H6;Intros.
Elim H8;Intro C2;Intros.
Elim H9;Intros.
Elim H11;Intro C3;Intros.
Elim H12;Intros.
Elim H4;Intro C'1;Intros.
Elim H15;Intros.
Elim H17;Intro C'2;Intros.
Elim H18;Intros.
Elim H20;Intro C'3;Intros.
Elim H21;Intros.
Rewrite -> H23.
Rewrite -> H14.
Elim (H C1 C'1 s H7 H16).
Elim (H0 C2 C'2 s H10 H19).
Elim (H1 C3 C'3 s H13 H22).
Auto.

Save.


(*****************************************************************************)
(*We can now define an equivalence between ML values and CAM values with the *)
(*the help of the predicate Traduction                                       *)
(*****************************************************************************)


Inductive Definition Equiv:(b:bool)(V_E b)->CSem_val->Prop=


 Eqbool:(b:bool)(Equiv true (boolean b) (val (elem b)))


|Eqnum:(n:nat)(Equiv true (num n) (val (int n)))

|Eq_op:(c:OP)(Equiv true (OP_clos c) (val (def_op c)))

|Eqpair:(V1,V2:MLval)(Cval1,Cval2:CSem_val)
         (Equiv true V1 Cval1)->
           (Equiv true V2 Cval2)->
              (Equiv true (valpair V1 V2) (Cam_pair Cval1 Cval2))


|Eqenv1:(Equiv false Enil Cam_nil)


|Eqenv2:(p:Pat)(E:MLenv)(CV0:CSem_val)
            (Equiv false E CV0)->
              (V:MLval)(CV:CSem_val)(Equiv true V CV)->
                (Equiv false (Econs p V E) (Cam_pair CV0 CV))


|Eqclos:(p:Pat)(E:MLexp)(C:Commande)(e:MLenv)(CV:CSem_val)
 (s:Squelette)
             (Equiv false e CV)->
                (Habite e s)->  
                (Traduction (cons_squelette p s) E C)->
                 (Equiv true (Clos p E e) (Cam_clos C CV))


|Eqclos_rec:(p,x:Pat)(E:MLexp)(e:MLenv)(C:Commande)(CV:CSem_val)
       (s:Squelette)
          (Equiv false e CV)->
              (Habite e s)->
                (Traduction (cons_squelette x (cons_squelette p s)) E C)->
   (Equiv true (Clos_rec x E p e) (Cam_clos_rec C CV)).


(*****************************************************************************)
(*              This definition is inverted immediately!!                    *)
(*****************************************************************************)

Definition Equiv_inv=
[b:bool][var_env:(V_E b)][Cval:CSem_val]
  (<Prop>Match b with

 (*b evaluates in true and then var_env is a value*)
       (<[b':bool][v_e:(V_E b')]Prop>Match var_env with 

     (*var_env is a boolean*)[b:bool](<CSem_val> Cval=(val (elem b)))

     (*var_env is an integer*)[n:nat](<CSem_val> Cval=(val (int n)))

     (*var_env is a pair of values*)
        [V1:MLval][P1:Prop][V2:MLval][P2:Prop]
          (<CSem_val>Ex ([Cval1:CSem_val]
             (<CSem_val>Ex ([Cval2:CSem_val]
                (Equiv true V1 Cval1)
            /\  (Equiv true V2 Cval2)
             /\  (<CSem_val> Cval=(Cam_pair Cval1 Cval2))))))

     (*var_env is a predefinite operator*)[c:OP]
                  (<CSem_val> Cval=(val (def_op c)))

     (*var_env is a closure: (Clos p E e)*)
        [p:Pat][E:MLexp][e:MLenv][Penv:Prop]
           (<Commande>Ex ([C:Commande]
              (<CSem_val>Ex ([CV:CSem_val]
                 (<Squelette>Ex ([s:Squelette]
                    (Equiv false e CV)
                 /\ (Habite e s)
                 /\ (Traduction (cons_squelette p s) E C)
                 /\ (<CSem_val> Cval=(Cam_clos C CV))))))))


     (*var_env is a recursive closure*)

                [Q:Pat][E:MLexp][P:Pat][e:MLenv][Pe:Prop]
       (<Squelette>Ex ([s:Squelette]
         (<Commande>Ex ([C:Commande]
           (<CSem_val>Ex ([Cval_e:CSem_val]
                  (Equiv false e Cval_e)
               /\ (Habite e s)
               /\ (Traduction (cons_squelette Q (cons_squelette P s)) E C)
               /\ (<CSem_val>  Cval=(Cam_clos_rec C Cval_e))))))))


     (*var_env is Enil*)
        False

  
     (*var_env is a non trivial environment *)
   [p:Pat][V:MLval][PV:Prop][e:MLenv][Pe:Prop]
         False)


 (*b evaluates in false and then var_env represents an environment*)
        (<[b':bool][v_e:(V_E b')]Prop>Match  var_env with

      (*var_env is a boolean*)[b:bool]False

      (*var_env is an integer*)[n:nat]False

      (*var_env is a pair of values*)
         [V1:MLval][P1:Prop][V2:MLval][P2:Prop] False

      (*var_env is a predefinite operator*)[c:OP]False

      (*var_env is a closure*)
         [p:Pat][E:MLexp][e:MLenv][Penv:Prop] False

      (*var_env is a recursive closure*)
         [Q:Pat][E:MLexp][P:Pat][e:MLenv][Pe:Prop]False

      (*var_env is the empty environment*)  (<CSem_val> Cval=Cam_nil)
 
  
      (*var_env is a non empty environment*)
    [p:Pat][V:MLval][PV:Prop][e:MLenv][Pe:Prop]
       (<CSem_val>Ex ([Cval_e:CSem_val]
            (Equiv false e Cval_e)
          /\ (<CSem_val>Ex ([Cval_V:CSem_val]
                (Equiv true V Cval_V)
             /\ (<CSem_val> Cval=(Cam_pair Cval_e Cval_V)))))))).




(*****************************************************************************)
(*             We are now going to prove  Equiv->Equiv_inv                   *)
(*****************************************************************************)

Lemma Equiv_Equiv_inv.
Statement (b:bool)(var_env:(V_E b))(Cval:CSem_val)
          (Equiv b var_env Cval)->(Equiv_inv b var_env Cval).


Goal.

Induction b.
Intro;Intro.
Induction 1;Intros;Simpl;Auto.
Exists Cval1;Exists Cval2;Split;Auto.
Exists CV0;Split;Auto;Exists CV;Split;Auto.
Exists C;Exists CV;Exists s;Split;Auto.
Exists s;Exists C;Exists CV;Split;Auto.
Intro;Intro.
Induction 1;Intros;Simpl;Auto.
Exists Cval1;Exists Cval2;Split;Auto.
Exists CV0;Split;Auto;Exists CV;Split;Auto.
Exists C;Exists CV;Exists s;Split;Auto.
Exists s;Exists C;Exists CV;Split;Auto.

Save.

Hint Equiv_Equiv_inv.




(*****************************************************************************)
(*                 We can now give a formulation of the proof                *)
(*****************************************************************************)

Inductive Definition compilation[E:MLexp]:Prop=
preuve_compilation:
  ((e:MLenv)(V:MLval)(ML_DS e E V)->     
         (s:Squelette)(C:Commande)(Traduction s E C)->
           (Habite e s)->
               (CV:CSem_val)(Equiv false e CV)->
                 (<CSem_val>Ex ([CV1:CSem_val]
                   ((Equiv true V CV1)
                  /\ (s:Etat)(CAM_DS (ETcons CV s) C (ETcons CV1 s))))))->
                          (compilation E).
                 



(*****************************************************************************)
(*This formulation permits us to make the proof for the different terms      *)
(*that can match E except the application                                    *)
(*We use it for the simple cases.                                            *)
(*****************************************************************************)



(*****************************************************************************)
(*                  Case where E is a boolean                                *)
(*****************************************************************************)

Lemma Proof_bool.
Statement(b:bool)(compilation (Bool b)).


Goal.
Intro b.
Apply preuve_compilation.
Intros e V ML_b s C Trad_b hab CV Eq.
Cut (ML_inv e (Bool b) V).
Simpl;Intro Eq_V_b.
Rewrite -> Eq_V_b.
Cut (Trad_inv s (Bool b) C).
Simpl;Intro Com.
Rewrite -> Com.
Exists (val (elem b)).
Split.
Exact (Eqbool b).
Intro;Exact (QUO s0 CV (elem b)).
Exact (Trad_Trad_inv s (Bool b) C Trad_b).
Exact (ds_inv e V (Bool b) ML_b).
Save.



(*****************************************************************************)
(*                  Case where E is an integer                               *)
(*****************************************************************************)

Lemma  Proof_int.
Statement (n:nat)(compilation (Num n)).


Goal.
Intro n.
Apply preuve_compilation.
Intros e V ML_n s C Trad_n hab CV Eq.
Cut (ML_inv e (Num n) V).
Simpl;Intro Eq_V_n.
Rewrite -> Eq_V_n.
Cut (Trad_inv s (Num n) C).
Simpl;Intro Com.
Rewrite -> Com.
Exists (val (int n)).
Split.
Exact (Eqnum n).
Intro;Exact (QUO s0 CV (int n)).
Exact (Trad_Trad_inv s (Num n) C Trad_n).
Exact (ds_inv e V (Num n) ML_n).
Save.

(*****************************************************************************)
(*               Case where E is a predefinite operator                      *)
(*****************************************************************************)


Lemma  Proof_op.
Statement (c:OP)(compilation (op c)).


Goal.
Intro c.
Apply preuve_compilation.
Intros e V ML_c s C Trad_c hab CV Eq.
Cut (ML_inv e (op c) V).
Simpl;Intro Eq_V_c.
Rewrite -> Eq_V_c.
Cut (Trad_inv s (op c) C).
Simpl;Intro Com.
Rewrite -> Com.
Exists (val (def_op c)).
Split.
Exact (Eq_op c).
Intro;Exact (QUO s0 CV (def_op c)).
Exact (Trad_Trad_inv s (op c) C Trad_c).
Exact (ds_inv e V (op c) ML_c).
Save.

(*****************************************************************************)
(*                   Case where E is an identifier                          *)
(* This case is special in the sens that we cannot use the predicate         *)
(*compilation because the variables of the formulation must be introduced    *)
(* in a very precise order.                                                  *)
(*****************************************************************************)




Inductive Definition compilation_id[E:MLexp]:Prop=

preuve_compilation_id: ((s:Squelette)      
          (C:Commande)(Traduction s E C)->
            (e:MLenv)(Habite e s)->
             (V:MLval)(ML_DS e E V)->
               (CV:CSem_val)(Equiv false e CV)->
                  (<CSem_val>Ex ([CV1:CSem_val]
                   ( (Equiv true V CV1)
                  /\ (s:Etat)(CAM_DS (ETcons CV s) C (ETcons CV1 s))))))->
                          (compilation_id E).



Lemma Proof_ident.
Statement (x:Pat)(compilation_id (id x)).


Goal.
Intro.
Apply preuve_compilation_id.
Intro;Intro;Intro.
Cut (Trad_inv s (id x) C).
2:Auto.
Simpl.
Induction 1;Intros.
Cut (Habite_inv e (cons_squelette P s0)).
2:Exact (Habite_Habite_inv (cons_squelette P s0) e  H1).
Unfold Habite_inv.
Intros.
Elim H4;Intro val_P;Intros.
Elim H5;Intro e0;Intros.
Elim H6;Intros.
Cut (Equiv_inv false e CV).
2:Auto.
Rewrite -> H7.
Simpl.
Intros.
Elim H9;Intro Cval_e;Intros.
Elim H10;Intros.
Elim H12;Intro Cval_V;Intros.
Elim H13;Intros.
Rewrite -> H15;Exists Cval_V.
Split.
2:Intro;Exact (CDR s1 Cval_e Cval_V).
Cut (ML_DS e (id P) val_P).
Intro.
Cut (<MLval> val_P=V).
Intro.
Elim H17;Exact H14.
Exact (ML_DS_determ e (id P) val_P H16 V H2).
Rewrite -> H7.
Cut (VAL_OF (Econs P val_P e0) P val_P).
Intro.
Exact (IDENT (Econs P val_P e0) val_P P H16).
Exact (ELT e0 P val_P).
Cut (Habite_inv e (cons_squelette T s0)).
2:Exact (Habite_Habite_inv (cons_squelette T s0) e H4).
Simpl;Intros.
Elim H7;Intro V_T;Intros.
Elim H8;Intro e0;Intros.
Elim H9;Intros.
Cut (ML_DS e (id P) V).
2:Auto.
Rewrite -> H10;Intros.
Cut (ML_inv (Econs T V_T e0) (id P) V).
2:Auto.
Simpl;Intros.
Cut (Val_inv (Econs T V_T e0) P V).
2:Auto.
Unfold Val_inv;Intros.
Elim H14;Intros.
Elim H15;Intros.
Cut <Pat>P=T.
Intro.
Elim (H1 H18).
Auto.
Elim H15;Intros.
Cut (ML_DS e0 (id P) V).
2:Exact (IDENT e0 V P H17).
Intro.
Cut (Equiv_inv false e CV).
2:Auto.
Rewrite -> H10.
Simpl;Intros.
Elim H19;Intro Cval_e0;Intros.
Elim H20;Intros.
Elim H22;Intro Cval_T;Intros.
Elim H23;Intros.
Rewrite -> H25.
Elim (H3 e0 H11 V H18 Cval_e0 H21).
Intro Cval_V;Intros.
Elim H26;Intros.
Exists Cval_V.
Split.
Auto.
Intro.
Cut (CAM_DS (ETcons (Cam_pair Cval_e0 Cval_T) s1) car
              (ETcons Cval_e0 s1)).
2:Exact (CAR s1 Cval_e0 Cval_T).
Intro.
Exact (o_DS (ETcons (Cam_pair Cval_e0 Cval_T) s1)
            (ETcons Cval_e0 s1)
            (ETcons Cval_V s1)
            car 
            C0
            H29
            (H28 s1)).
Save.




(*****************************************************************************)
(*                     Case where E is an abstraction                        *)
(*****************************************************************************)


Lemma Proof_abstraction.
Statement  (E:MLexp)(p:Pat)(compilation (lambda p E)).



Goal.
Intros E p.
Apply preuve_compilation.
Intros e V ML_lambda s C Trad_lambda hab CV Eq.
Cut (ML_inv e (lambda p E) V).
2:Exact (ds_inv e V (lambda p E) ML_lambda).
Simpl;Intro Eq_V_Clos.
Rewrite -> Eq_V_Clos.
Cut (Trad_inv s (lambda p E) C).
2:Exact (Trad_Trad_inv s (lambda p E) C Trad_lambda).
Simpl;Intro HH.
Elim HH;Intros C1 HH1;Clear HH.
Elim HH1;Intros Trad_E Com;Clear HH1.
Rewrite -> Com.
Exists (Cam_clos C1 CV).
Split.
Exact (Eqclos p E C1 e CV s Eq hab Trad_E).
Intro.
Exact (CUR s0 CV C1).
Save.



(*****************************************************************************)
(*          We have now all the tools we need to make the proof              *)
(* We want to prove that the following diagram commutes:                     *)
(*                                                                           *)
(*                                                                           *)
(*                                                                           *)
(*                           "Traduction"                                    *)
(*         Mini_ML terms ---------------------> CAM terms                    *)
(*             |                                  |                          *)
(*             |                                  |                          *)
(*             |                                  |                          *)
(*             |                                  |                          *)
(*      "ML_DS"|                                  |"CAM_DS"                  *)
(*             |                                  |                          *)
(*             |                                  |                          *)
(*             |                                  |                          *)
(*             V              "Equiv"             V                          *)
(*      Mini_ML values ---------------------> CAM values                     *)
(*                                                                           *)
(*                                                                           *)
(*                                                                           *)
(* This means that having an MLexp E its ML value and its translation onto   *)
(*CAM code we can find a  CAM value such that the diagram commutes           *)
(*                                                                           *)
(*      The proof is made by induction on the Predicate ML_DS.               *)
(*****************************************************************************)
    
Lemma  differT_F: (E:MLexp)(e:MLenv)
         (ML_DS e E (boolean true))->(ML_DS e E (boolean false))->False.


Goal.
Intros E e ML_true ML_false.
Cut <MLval>(boolean true)=(boolean false).
2:Exact (ML_DS_determ e E (boolean true) ML_true (boolean false) ML_false).
Exact notEq_Tr_Fa.

Save.


Lemma final_proof.
Statement (E:MLexp)(e:MLenv) 
       (V:MLval)(ML_DS e E V)->     
         (s:Squelette)(C:Commande)(Traduction s E C)->
           (Habite e s)->
             (V:MLval)(ML_DS e E V)->
               (CV:CSem_val)(Equiv false e CV)->
                 (<CSem_val>Ex ([CV1:CSem_val]
                   ((Equiv true V CV1)
                  /\ (s:Etat)(CAM_DS (ETcons CV s) C (ETcons CV1 s))))).


Goal.
Intros E e V.
Induction 1.

(*****************************************************************************)
(*                Simple cases have already been treated                     *)
(*****************************************************************************)

Intros b e0 s C Trad_b hab V0 ML_b CV Eq.
Cut (compilation (Bool b)).
2:Exact (Proof_bool b).
Intro comp.
Elim comp;Intro hyp.
Exact (hyp e0 V0 ML_b s C Trad_b hab CV Eq).

Intros n e0 s C Trad_int hab V0 ML_int CV Eq.
Cut (compilation (Num n)).
2:Exact (Proof_int n).
Intro comp.
Elim comp;Intro hyp.
Exact (hyp e0 V0 ML_int s C Trad_int hab CV Eq).

Intros c e0 s C Trad_op hab V0 ML_op CV Eq.
Cut (compilation (op c)).
2:Exact (Proof_op c).
Intro comp.
Elim comp;Intro hyp.
Exact (hyp e0 V0 ML_op s C Trad_op hab CV Eq).

Intros e0 P E0 s C Trad_lambda hab V0 ML_lambda CV Eq.
Cut  (compilation (lambda P E0)).
2:Exact (Proof_abstraction  E0 P).
Intro comp.
Elim comp;Intro hyp.
Exact (hyp e0 V0 ML_lambda s C Trad_lambda hab CV Eq).

Intros e0 v I VAL_I s C Trad_pat hab V0 ML_pat CV Eq.
Cut (compilation_id (id I)).
2:Exact (Proof_ident I).
Intro comp.
Elim comp;Intro hyp.
Exact (hyp s C Trad_pat e0 hab V0 ML_pat CV Eq).


(*****************************************************************************)
(*We have now to treat the cases where "compilation" cannot be used          *)
(*****************************************************************************)


(*****************************************************************************)
(*       Case of the " if then else"  with E1 -> true                        *)
(*****************************************************************************)

Intros e0 E1 E2 E3 v ML_E1 hyp_E1 ML_E2 hyp_E2 
s C Trad_ite hab V0 ML_ite CV Eq.
Cut (Trad_inv s (ite E1 E2 E3) C).
2:Exact (Trad_Trad_inv s (ite E1 E2 E3) C Trad_ite).
Simpl;Intro Hyp0.
Elim Hyp0;Intro C1;Intro Hyp1;Clear Hyp0.
Elim Hyp1;Intros Trad_E1 Hyp0;Clear Hyp1.
Elim Hyp0;Intro C2;Intro Hyp1;Clear Hyp0.
Elim Hyp1;Intros Trad_E2 Hyp0;Clear Hyp1.
Elim Hyp0;Intro C3;Intro Hyp1;Clear Hyp0.
Elim Hyp1;Intros Trad_E3 Com;Clear Hyp1.
Rewrite -> Com.
Cut (ML_inv e0 (ite E1 E2 E3) V0).
2:Exact (ds_inv e0 V0 (ite E1 E2 E3) ML_ite).
Simpl;Intro Hyp0.
Elim Hyp0;Intro Cas_E1;Clear Hyp0.
Elim Cas_E1;Intros ML_E1_Tr ML_E2_V0;Clear Cas_E1.
(*Ici on differe du cas E1 -> false*)
Elim (hyp_E2 s C2 Trad_E2 hab V0 ML_E2_V0 CV Eq).
Intro Cval_V0;Intro Hyp0.
Elim Hyp0;Intros Eq_V0 CAM_C2;Clear Hyp0.
Exists Cval_V0;Split.
Exact Eq_V0.
Intro s0.
Cut (CAM_DS (ETcons CV (ETcons CV s0))
               (o C1 (branch C2 C3))
                 (ETcons Cval_V0 s0)).
Cut (CAM_DS (ETcons CV s0)  push (ETcons CV (ETcons CV s0))).
Exact (o_DS  (ETcons CV s0)
             (ETcons CV (ETcons CV s0))
             (ETcons Cval_V0 s0)
                  push 
                  (o C1 (branch C2 C3))).
Exact (PUSH s0 CV).
Elim (hyp_E1 s C1 Trad_E1 hab (boolean true) ML_E1 CV Eq).
Intro Cval_E1;Clear hyp_E1;Intro Hyp0.
Elim Hyp0;Intros Eq_Tr CAM_C1;Clear Hyp0.
Cut (Equiv_inv true (boolean true) Cval_E1).
2:Exact (Equiv_Equiv_inv true (boolean true) Cval_E1 Eq_Tr).
Simpl;Intro CSem_E1.
Cut (CAM_DS (ETcons Cval_E1 (ETcons CV s0))
                (branch C2 C3)
                   (ETcons Cval_V0 s0)).
Cut (CAM_DS (ETcons CV (ETcons CV s0))
                C1
                    (ETcons Cval_E1 (ETcons CV s0))).
Exact (o_DS (ETcons CV (ETcons CV s0))
            (ETcons Cval_E1 (ETcons CV s0))
            (ETcons Cval_V0 s0)
                C1 
                (branch C2 C3)).
Exact (CAM_C1 (ETcons CV s0)).
Rewrite -> CSem_E1.
Exact (BRANCHT (ETcons CV s0) (ETcons Cval_V0 s0)
         C2 C3 (CAM_C2 s0)).
Elim Cas_E1;Intros ML_E1_Fa ML_E3_V0;Clear Cas_E1.
Elim (differT_F E1 e0 ML_E1 ML_E1_Fa).


(*****************************************************************************)
(*     Case of the "if the else" where  E1-> False dans le if_then_else      *)
(*****************************************************************************)


Intros e0 E1 E2 E3 v ML_E1 hyp_E1 ML_E3 hyp_E3
s C Trad_ite hab V0 ML_ite CV Eq.
Cut (Trad_inv s (ite E1 E2 E3) C).
2:Exact (Trad_Trad_inv s (ite E1 E2 E3) C Trad_ite).
Simpl;Intro Hyp0.
Elim Hyp0;Intro C1;Intro Hyp1;Clear Hyp0.
Elim Hyp1;Intros Trad_E1 Hyp0;Clear Hyp1.
Elim Hyp0;Intro C2;Intro Hyp1;Clear Hyp0.
Elim Hyp1;Intros Trad_E2 Hyp0;Clear Hyp1.
Elim Hyp0;Intro C3;Intro Hyp1;Clear Hyp0.
Elim Hyp1;Intros Trad_E3 Com;Clear Hyp1.
Rewrite -> Com.
Cut (ML_inv e0 (ite E1 E2 E3) V0).
2:Exact (ds_inv e0 V0 (ite E1 E2 E3) ML_ite).
Simpl;Intro Hyp0.
Elim Hyp0;Intro Cas_E1;Clear Hyp0.
Elim Cas_E1;Intros ML_E1_Tr ML_E2_V0;Clear Cas_E1.
Elim (differT_F E1 e0 ML_E1_Tr ML_E1).
Elim Cas_E1;Intros ML_E1_Fa ML_E3_V0;Clear Cas_E1.
Elim (hyp_E3 s C3 Trad_E3 hab V0 ML_E3_V0 CV Eq).
Intro Cval_V0;Intro Hyp0.
Elim Hyp0;Intros Eq_V0 CAM_C3;Clear Hyp0.
Exists Cval_V0;Split.
Exact Eq_V0.
Intro s0.
Cut (CAM_DS (ETcons CV (ETcons CV s0))
               (o C1 (branch C2 C3))
                 (ETcons Cval_V0 s0)).
Cut (CAM_DS (ETcons CV s0)  push (ETcons CV (ETcons CV s0))).
Exact (o_DS  (ETcons CV s0)
             (ETcons CV (ETcons CV s0))
             (ETcons Cval_V0 s0)
                  push 
                  (o C1 (branch C2 C3))).
Exact (PUSH s0 CV).
Elim (hyp_E1 s C1 Trad_E1 hab (boolean false) ML_E1 CV Eq).
Intro Cval_E1;Clear hyp_E1;Intro Hyp0.
Elim Hyp0;Intros Eq_Fa CAM_C1;Clear Hyp0.
Cut (Equiv_inv true (boolean false) Cval_E1).
2:Exact (Equiv_Equiv_inv true (boolean false) Cval_E1 Eq_Fa).
Simpl;Intro CSem_E1.
Cut (CAM_DS (ETcons Cval_E1 (ETcons CV s0))
                (branch C2 C3)
                   (ETcons Cval_V0 s0)).
Cut (CAM_DS (ETcons CV (ETcons CV s0))
                C1
                    (ETcons Cval_E1 (ETcons CV s0))).
Exact (o_DS (ETcons CV (ETcons CV s0))
            (ETcons Cval_E1 (ETcons CV s0))
            (ETcons Cval_V0 s0)
                C1 
                (branch C2 C3)).

Exact (CAM_C1  (ETcons CV s0)).
Rewrite -> CSem_E1.
Exact (BRANCHF (ETcons CV s0) (ETcons Cval_V0 s0)
         C2 C3 (CAM_C3 s0)).


(*****************************************************************************)
(*                            the pair case                                  *)
(*****************************************************************************)


Intros e0 E1 E2 u v ML_E1 hyp_E1 ML_E2 hyp_E2 
s C Trad_pair hab V0 ML_pair CV Eq.
Cut (ML_inv e0 (mlpair E1 E2) V0).
2:Exact (ds_inv e0 V0 (mlpair E1 E2) ML_pair).
Simpl;Intro Hyp0.
Elim Hyp0;Intros Val_E1 Hyp1;Clear Hyp0.
Elim Hyp1;Intros ML_E1_Val Hyp0;Clear Hyp1.
Elim Hyp0;Intros Val_E2 Hyp1;Clear Hyp0.
Elim Hyp1;Intros ML_E2_Val VAL_V0;Clear Hyp1.
Cut (Trad_inv s (mlpair E1 E2) C).
2:Exact (Trad_Trad_inv s (mlpair E1 E2) C Trad_pair).
Simpl;Intro Hyp0.
Elim Hyp0;Intros C1 Hyp1;Clear Hyp0.
Elim Hyp1;Intros Trad_E1 Hyp0;Clear Hyp1.
Elim Hyp0;Intros C2 Hyp1;Clear Hyp0.
Elim Hyp1;Intros Trad_E2 Com;Clear Hyp1.
Rewrite ->Com.
Rewrite -> VAL_V0.
Elim (hyp_E1 s C1 Trad_E1 hab Val_E1 ML_E1_Val CV Eq).
Intro Cam_val_E1;Intro Hyp0.
Elim Hyp0;Intros Eq_Val_E1 CAM_C1;Clear Hyp0.
Elim (hyp_E2 s C2 Trad_E2 hab Val_E2 ML_E2_Val CV Eq).
Intro Cam_val_E2 ;Intro Hyp0.
Elim Hyp0;Intros Eq_Val_E2 CAM_C2;Clear Hyp0.
Exists (Cam_pair Cam_val_E1 Cam_val_E2).
Split.
Exact (Eqpair Val_E1 Val_E2 Cam_val_E1 Cam_val_E2 Eq_Val_E1 Eq_Val_E2).
Intro s0.
Cut (CAM_DS (ETcons CV (ETcons CV s0)) 
            (o C1 (o swap  (o C2 cons)))
                (ETcons (Cam_pair Cam_val_E1 Cam_val_E2) s0)).
Cut (CAM_DS (ETcons CV s0) push (ETcons CV (ETcons CV s0))).
Exact (o_DS  (ETcons CV s0)
             (ETcons CV (ETcons CV s0))
             (ETcons (Cam_pair Cam_val_E1 Cam_val_E2) s0)
               push
               (o C1 (o swap  (o C2 cons)))).
Exact (PUSH s0 CV).
Cut (CAM_DS (ETcons Cam_val_E1 (ETcons CV s0)) (o swap (o C2 cons))
              (ETcons (Cam_pair Cam_val_E1 Cam_val_E2) s0)).
Cut (CAM_DS (ETcons CV (ETcons CV s0))  C1
              (ETcons Cam_val_E1 (ETcons CV s0))).
Exact (o_DS (ETcons CV (ETcons CV s0))
            (ETcons Cam_val_E1 (ETcons CV s0))
            (ETcons (Cam_pair Cam_val_E1 Cam_val_E2) s0)
               C1
               (o swap (o C2 cons))).
Exact (CAM_C1 (ETcons CV s0)).
Cut (CAM_DS (ETcons CV (ETcons Cam_val_E1 s0)) (o C2 cons )
               (ETcons  (Cam_pair Cam_val_E1 Cam_val_E2) s0)).
Cut (CAM_DS (ETcons Cam_val_E1 (ETcons CV s0)) swap
               (ETcons CV (ETcons Cam_val_E1 s0))).
Exact (o_DS  (ETcons Cam_val_E1 (ETcons CV s0))
             (ETcons CV (ETcons Cam_val_E1 s0))
             (ETcons (Cam_pair Cam_val_E1 Cam_val_E2) s0)
                 swap
                 (o C2 cons)).
Exact (SWAP s0 Cam_val_E1 CV).
Cut (CAM_DS (ETcons Cam_val_E2 (ETcons Cam_val_E1 s0)) cons
               (ETcons (Cam_pair Cam_val_E1 Cam_val_E2) s0)).
Cut (CAM_DS (ETcons CV (ETcons Cam_val_E1 s0)) C2
                 (ETcons Cam_val_E2 (ETcons Cam_val_E1 s0))).
Exact (o_DS (ETcons CV (ETcons Cam_val_E1 s0))
            (ETcons Cam_val_E2 (ETcons Cam_val_E1 s0))
            (ETcons (Cam_pair Cam_val_E1 Cam_val_E2) s0)
               C2 
               cons ).
Exact (CAM_C2 (ETcons Cam_val_E1 s0)).
Exact (CONS s0 Cam_val_E1 Cam_val_E2).


(*****************************************************************************)
(*We have now to solve the case of the application (It is important to notice*)
(*that we were not able to make the proof on induction on E because of the   *)
(*application case                                                           *)
(*****************************************************************************)


Intros e0 e1 P E0 E1 E2 u v ML_E1 hyp_E1 ML_E2 hyp_E2 ML_E0 hyp_E0
s C Trad_appl hab V0 ML_appl CV Eq.
Cut (Trad_inv s (appl E1 E2) C).
2:Exact (Trad_Trad_inv s (appl E1 E2) C Trad_appl).
Simpl;Intro Hyp0.
Elim Hyp0;Intros C1 Hyp1;Clear Hyp0.
Elim Hyp1;Intros Trad_E1 Hyp0;Clear Hyp1.
Elim Hyp0;Intros C2 Hyp1;Clear Hyp0.
Elim Hyp1;Intros Trad_E2 Com_C;Clear Hyp1.
Elim (hyp_E2 s C2 Trad_E2 hab u ML_E2 CV Eq);Intro Cval_u;Intro HH.
Elim HH;Intro Eq_u;Intro CAM_u.
Clear HH.Clear hyp_E2.
Elim (hyp_E1 s C1 Trad_E1 hab (Clos P E0 e1) ML_E1 CV Eq);
Intro Cval_E1;Intro HH.
Elim HH;Intro Eq_valE1;Intro CAM_val1.
Clear HH.Clear hyp_E1.
Cut (Equiv_inv true (Clos P E0 e1) Cval_E1).
2:Exact (Equiv_Equiv_inv true (Clos P E0 e1) Cval_E1 Eq_valE1).
Simpl;Intro HH.
Elim HH;Intro C_E0;Intro HH1;Clear HH.
Elim HH1;Intro Cval_e1;Intro HH;Clear HH1.
Elim HH;Intro s1;Intro HH1;Clear HH.
Elim HH1;Intro Eq_e1;Intro HH;Clear HH1.
Elim HH;Intro hab_e1;Intro HH1;Clear HH.
Elim HH1;Intro Trad_E0;Intro val_E1;Clear HH1.
Cut (Habite (Econs P u e1) (cons_squelette P s1)).
2:Exact (cons_habite P u e1 s1 hab_e1).
Intro hab_Pe1.
Cut (Equiv false (Econs P u e1) (Cam_pair Cval_e1 Cval_u)).
2:Exact (Eqenv2 P e1 Cval_e1 Eq_e1 u Cval_u Eq_u).
Intro Eq_Pe1.
Elim (hyp_E0 (cons_squelette P s1) C_E0 Trad_E0 hab_Pe1 v ML_E0
             (Cam_pair Cval_e1 Cval_u) Eq_Pe1);Intro Cval_v;Intro HH.
Clear hyp_E0.
Elim HH;Intro Eq_v;Intro CAM_E0;Clear HH.
Exists Cval_v;Split.
Cut (ML_DS e0 (appl E1 E2) v).
2:Exact (APPml1 e0 e1 P E0 E1 E2 u v ML_E1 ML_E2 ML_E0).
Intro ML_v.
Cut <MLval>v=V0.
2:Exact (ML_DS_determ e0 (appl E1 E2) v ML_v V0 ML_appl).
Intro HH.
Rewrite <-HH.
Exact Eq_v.
Intro s0.
Rewrite ->Com_C.
Cut (CAM_DS (ETcons CV (ETcons CV s0)) (o C1 (o swap (o C2 (o cons app))))
             (ETcons Cval_v s0)).
Cut (CAM_DS (ETcons CV s0) push (ETcons CV (ETcons CV s0))).
Exact (o_DS (ETcons CV s0)
            (ETcons CV (ETcons CV s0))
            (ETcons Cval_v s0)
            push
            (o C1 (o swap (o C2 (o cons app))))).
Exact (PUSH s0 CV).
Cut (CAM_DS (ETcons Cval_E1 (ETcons CV s0)) (o swap (o C2 (o cons app)))
            (ETcons Cval_v s0)).
Cut (CAM_DS (ETcons CV (ETcons CV s0)) C1 (ETcons Cval_E1 (ETcons CV s0))).
Exact (o_DS (ETcons CV (ETcons CV s0))
            (ETcons Cval_E1 (ETcons CV s0))
            (ETcons Cval_v s0)
            C1 
            (o swap (o C2 (o cons app)))).
Exact (CAM_val1 (ETcons CV s0)).
Cut (CAM_DS (ETcons CV (ETcons Cval_E1 s0)) (o C2 (o cons app))
            (ETcons Cval_v s0)).     
Cut (CAM_DS (ETcons Cval_E1 (ETcons CV s0)) swap
                (ETcons CV (ETcons Cval_E1 s0))).
Exact (o_DS (ETcons Cval_E1 (ETcons CV s0))
            (ETcons CV (ETcons Cval_E1 s0))
            (ETcons Cval_v s0)
            swap
            (o C2 (o cons app))).
Exact (SWAP s0 Cval_E1 CV).
Cut (CAM_DS (ETcons Cval_u (ETcons Cval_E1 s0)) (o cons app)
               (ETcons Cval_v s0)).
Cut (CAM_DS (ETcons CV (ETcons Cval_E1 s0)) C2 
                  (ETcons Cval_u (ETcons Cval_E1 s0))).
Exact (o_DS (ETcons CV (ETcons Cval_E1 s0))
            (ETcons Cval_u (ETcons Cval_E1 s0))
            (ETcons Cval_v s0)
            C2
            (o cons app)).
Exact (CAM_u (ETcons Cval_E1 s0)).
Cut (CAM_DS (ETcons (Cam_pair Cval_E1 Cval_u) s0) app
               (ETcons Cval_v s0)).
Cut (CAM_DS (ETcons Cval_u (ETcons Cval_E1 s0))
    cons (ETcons (Cam_pair Cval_E1 Cval_u) s0)).
Exact (o_DS (ETcons Cval_u (ETcons Cval_E1 s0))
            (ETcons (Cam_pair Cval_E1 Cval_u) s0)
            (ETcons Cval_v s0)
            cons 
            app).
Exact (CONS s0 Cval_E1 Cval_u).
Rewrite -> val_E1.
Exact (APPcam1 s0 (ETcons Cval_v s0) Cval_e1 Cval_u C_E0 (CAM_E0 s0)).


(*****************************************************************************)
(*                         Case of a recursive closure                       *)
(*****************************************************************************)


Intros e0 e1 x P E0 E1 E2 u v ML_E1 hyp_E1 ML_E2 hyp_E2 ML_E0 hyp_E0
s C Trad_appl hab V0 ML_appl CV Eq.
Cut (Trad_inv s (appl E1 E2) C).
2:Exact (Trad_Trad_inv s (appl E1 E2) C Trad_appl).
Simpl;Intro HH.
Elim HH;Intros C1 HH1;Clear HH.
Elim HH1;Intros Trad_E1 HH;Clear HH1.
Elim HH;Intros C2 HH1;Clear HH.
Elim HH1;Intros Trad_E2 Com_C;Clear HH1.
Elim (hyp_E2 s C2 Trad_E2 hab u ML_E2 CV Eq);Intro Cval_u;Intro HH.
Elim HH;Intros Eq_u CAM_u;Clear HH;Clear hyp_E2.
Elim (hyp_E1 s C1 Trad_E1 hab (Clos_rec x E0 P e1) ML_E1 CV Eq);
Intro Cval_E1;Intro HH.
Elim HH;Intros Eq_valE1 CAM_val1;Clear HH;Clear hyp_E1.
Cut (Equiv_inv true (Clos_rec x E0 P e1) Cval_E1).
2:Exact (Equiv_Equiv_inv true (Clos_rec x E0 P e1) Cval_E1 Eq_valE1).
Simpl;Intro HH.
Elim HH;Clear HH;Intro s1;Intro HH.
Elim HH;Intro C_E0;Intro HH1;Clear HH.
Elim HH1;Intro Cval_e1;Intro HH;Clear HH1.
Elim HH;Intro Eq_e1;Intro HH1;Clear HH.
Elim HH1;Intro hab_e1;Intro HH;Clear HH1.
Elim HH;Intro Trad_E0;Intro val_E1;Clear HH.
Cut (Habite (Econs x u (Econs P (Clos_rec x E0 P e1) e1)) 
                    (cons_squelette x (cons_squelette P s1))).
Intro Hab.
Cut (Equiv false (Econs x u (Econs P (Clos_rec x E0 P e1) e1)) 
                    (Cam_pair (Cam_pair Cval_e1 Cval_E1) Cval_u)).
Intro Eqn.
Elim (hyp_E0 (cons_squelette x (cons_squelette P s1)) C_E0 Trad_E0 
  Hab v ML_E0 (Cam_pair (Cam_pair Cval_e1 Cval_E1) Cval_u) Eqn);
Intro Cval_v;Intro HH;Clear hyp_E0.    
Elim HH;Intro Eq_v;Rewrite ->val_E1;Intro CAM_E0;Clear HH.
Exists Cval_v;Split.
Cut (ML_DS e0 (appl E1 E2) v).
2:Exact(APPml2 e0 e1 x P E0 E1 E2 u v ML_E1 ML_E2 ML_E0).
Intro ML_v.
Cut <MLval>v=V0.
2:Exact (ML_DS_determ e0 (appl E1 E2) v ML_v V0 ML_appl).
Intro HH.
Rewrite <-HH.
Exact Eq_v.
Intro s0.
Rewrite ->Com_C.
Cut (CAM_DS (ETcons CV (ETcons CV s0)) (o C1 (o swap (o C2 (o cons app))))
             (ETcons Cval_v s0)).
Cut (CAM_DS (ETcons CV s0) push (ETcons CV (ETcons CV s0))).
Exact (o_DS (ETcons CV s0)
            (ETcons CV (ETcons CV s0))
            (ETcons Cval_v s0)
            push
            (o C1 (o swap (o C2 (o cons app))))).
Exact (PUSH s0 CV).
Cut (CAM_DS (ETcons Cval_E1 (ETcons CV s0)) (o swap (o C2 (o cons app)))
            (ETcons Cval_v s0)).
Cut (CAM_DS (ETcons CV (ETcons CV s0)) C1 (ETcons Cval_E1 (ETcons CV s0))).
Exact (o_DS (ETcons CV (ETcons CV s0))
            (ETcons Cval_E1 (ETcons CV s0))
            (ETcons Cval_v s0)
            C1 
            (o swap (o C2 (o cons app)))).
Exact (CAM_val1 (ETcons CV s0)).
Cut (CAM_DS (ETcons CV (ETcons Cval_E1 s0)) (o C2 (o cons app))
            (ETcons Cval_v s0)).     
Cut (CAM_DS (ETcons Cval_E1 (ETcons CV s0)) swap
                (ETcons CV (ETcons Cval_E1 s0))).
Exact (o_DS (ETcons Cval_E1 (ETcons CV s0))
            (ETcons CV (ETcons Cval_E1 s0))
            (ETcons Cval_v s0)
            swap
            (o C2 (o cons app))).
Exact (SWAP s0 Cval_E1 CV).
Cut (CAM_DS (ETcons Cval_u (ETcons Cval_E1 s0)) (o cons app)
               (ETcons Cval_v s0)).
Cut (CAM_DS (ETcons CV (ETcons Cval_E1 s0)) C2 
                  (ETcons Cval_u (ETcons Cval_E1 s0))).
Exact (o_DS (ETcons CV (ETcons Cval_E1 s0))
            (ETcons Cval_u (ETcons Cval_E1 s0))
            (ETcons Cval_v s0)
            C2
            (o cons app)).
Exact (CAM_u (ETcons Cval_E1 s0)).
Cut (CAM_DS (ETcons (Cam_pair Cval_E1 Cval_u) s0) app
               (ETcons Cval_v s0)).
Cut (CAM_DS (ETcons Cval_u (ETcons Cval_E1 s0))
    cons (ETcons (Cam_pair Cval_E1 Cval_u) s0)).
Exact (o_DS (ETcons Cval_u (ETcons Cval_E1 s0))
            (ETcons (Cam_pair Cval_E1 Cval_u) s0)
            (ETcons Cval_v s0)
            cons 
            app).
Exact (CONS s0 Cval_E1 Cval_u).
Rewrite -> val_E1.
Exact (APPcam2 s0 (ETcons Cval_v s0) Cval_u Cval_e1 C_E0 (CAM_E0 s0)).
Cut (Equiv false (Econs P (Clos_rec x E0 P e1) e1) (Cam_pair Cval_e1 Cval_E1)).
2:Exact (Eqenv2 P e1 Cval_e1 Eq_e1 (Clos_rec x E0 P e1) Cval_E1 Eq_valE1).
Intro hyp.
Exact (Eqenv2 x  (Econs P (Clos_rec x E0 P e1) e1) (Cam_pair Cval_e1 Cval_E1)
  hyp u Cval_u Eq_u).
Cut (Habite (Econs P (Clos_rec x E0 P e1) e1) (cons_squelette P s1)).
Intro hyp.
2:Exact (cons_habite P (Clos_rec x E0 P e1) e1 s1 hab_e1).
Exact (cons_habite x u (Econs P (Clos_rec x E0 P e1) e1) (cons_squelette P s1)
   hyp).



(*****************************************************************************)
(*            Case where we apply a predefined operator                      *)
(*****************************************************************************)

Intros e0 E1 E2 n m c ML_E1 hyp_E1 ML_E2 hyp_E2.
Intros s C Trad_appl hab V0 ML_appl CV Eq.
Cut (Trad_inv s (appl E1 E2) C).
2:Exact (Trad_Trad_inv s (appl E1 E2) C Trad_appl).
Simpl;Intro HH.
Elim HH;Intros C1 HH1;Clear HH.
Elim HH1;Intros Trad_E1 HH;Clear HH1.
Elim HH;Intros C2 HH1;Clear HH.
Elim HH1;Intros Trad_E2 Com_C;Clear HH1.
Elim (hyp_E2 s C2 Trad_E2 hab (valpair (num n) (num m)) 
     ML_E2 CV Eq);Intro Cval_u;Intro HH.
Elim HH;Intros Eq_u CAM_u;Clear HH;Clear hyp_E2.
Elim (hyp_E1 s C1 Trad_E1 hab (OP_clos c) ML_E1 CV Eq);
Intro Cval_E1;Intro HH.
Elim HH;Intros Eq_valE1 CAM_val1;Clear HH;Clear hyp_E1.
Cut (Equiv_inv true (OP_clos c) Cval_E1).
2:Exact (Equiv_Equiv_inv true (OP_clos c) Cval_E1 Eq_valE1).
Simpl;Intro HH.
Cut (Equiv_inv true (valpair (num n) (num m)) Cval_u).
2:Exact (Equiv_Equiv_inv true (valpair (num n) (num m)) Cval_u Eq_u).
Simpl;Intro Hu.
Elim Hu;Intro Cval_n;Intro HHu;Clear Hu.
Elim HHu;Intro Cval_m;Intro Hu;Clear HHu.
Elim Hu;Intro Eq_n;Intro HHu;Clear Hu.
Elim HHu;Intro Eq_m;Intro Hu;Clear HHu.
Cut (Equiv_inv true (num m) Cval_m).
2:Exact  (Equiv_Equiv_inv true (num m) Cval_m Eq_m).
Simpl;Intro Hm.
Cut (Equiv_inv true (num n) Cval_n).
2:Exact  (Equiv_Equiv_inv true (num n) Cval_n Eq_n).
Simpl;Intro Hn.
Exists (val (int (eval_op c n m))).
Split.
Cut (ML_DS e0 (appl E1 E2) (num (eval_op c n m))).
2:Exact (APPml_op e0 E1 E2 n m c ML_E1 ML_E2).
Intro ML_app.
Cut (<MLval> V0=(num (eval_op c n m))).
2:Exact (ML_DS_determ e0 (appl E1 E2) V0 ML_appl 
       (num (eval_op c n m)) ML_app).
Intro HYP;Rewrite -> HYP;Clear HYP.
Exact (Eqnum (eval_op c n m)).

Intro s0.
Rewrite -> Com_C.
Cut (CAM_DS (ETcons CV (ETcons CV s0)) (o C1 (o swap (o C2 (o cons app))))
             (ETcons (val (int (eval_op c n m))) s0)).
Cut (CAM_DS (ETcons CV s0) push (ETcons CV (ETcons CV s0))).
Exact (o_DS (ETcons CV s0)
            (ETcons CV (ETcons CV s0))
            (ETcons (val (int (eval_op c n m))) s0)
            push
            (o C1 (o swap (o C2 (o cons app))))).
Exact (PUSH s0 CV).
Cut (CAM_DS (ETcons Cval_E1 (ETcons CV s0)) (o swap (o C2 (o cons app)))
            (ETcons (val (int (eval_op c n m))) s0)).
Cut (CAM_DS (ETcons CV (ETcons CV s0)) C1 (ETcons Cval_E1 (ETcons CV s0))).
Exact (o_DS (ETcons CV (ETcons CV s0))
            (ETcons Cval_E1 (ETcons CV s0))
            (ETcons (val (int (eval_op c n m))) s0)
            C1 
            (o swap (o C2 (o cons app)))).
Exact (CAM_val1 (ETcons CV s0)).
Cut (CAM_DS (ETcons CV (ETcons Cval_E1 s0)) (o C2 (o cons app))
            (ETcons (val (int (eval_op c n m))) s0)).     
Cut (CAM_DS (ETcons Cval_E1 (ETcons CV s0)) swap
                (ETcons CV (ETcons Cval_E1 s0))).
Exact (o_DS (ETcons Cval_E1 (ETcons CV s0))
            (ETcons CV (ETcons Cval_E1 s0))
            (ETcons (val (int (eval_op c n m))) s0)
            swap
            (o C2 (o cons app))).
Exact (SWAP s0 Cval_E1 CV).
Cut (CAM_DS (ETcons Cval_u (ETcons Cval_E1 s0)) (o cons app)
               (ETcons (val (int (eval_op c n m))) s0)).
Cut (CAM_DS (ETcons CV (ETcons Cval_E1 s0)) C2 
                  (ETcons Cval_u (ETcons Cval_E1 s0))).
Exact (o_DS (ETcons CV (ETcons Cval_E1 s0))
            (ETcons Cval_u (ETcons Cval_E1 s0))
            (ETcons (val (int (eval_op c n m))) s0)
            C2
            (o cons app)).
Exact (CAM_u (ETcons Cval_E1 s0)).
Cut (CAM_DS (ETcons (Cam_pair Cval_E1 Cval_u) s0) app
               (ETcons (val (int (eval_op c n m))) s0)).
Cut (CAM_DS (ETcons Cval_u (ETcons Cval_E1 s0))
    cons (ETcons (Cam_pair Cval_E1 Cval_u) s0)).
Exact (o_DS (ETcons Cval_u (ETcons Cval_E1 s0))
            (ETcons (Cam_pair Cval_E1 Cval_u) s0)
            (ETcons (val (int (eval_op c n m))) s0)
            cons 
            app).
Exact (CONS s0 Cval_E1 Cval_u).
Rewrite -> HH.
Rewrite -> Hu.
Rewrite -> Hn.
Rewrite -> Hm.
Exact (APPcam_op s0 n m c).


(*****************************************************************************)
(*                        Case where E is a let'                             *)
(*****************************************************************************)



Intros e0 P E1 E2 u v ML_E2 hyp_E2 ML_E1 hyp_E1 
s C Trad_let hab V0 ML_let CV Eq.
Cut (ML_DS e0 (let' P E2 E1) v).
2:Exact (Sem_let e0 P E1 E2 u v ML_E2 ML_E1).
Intro ML_letv.
Cut (<MLval> v=V0).
2:Exact (ML_DS_determ e0 (let' P E2 E1) v ML_letv V0 ML_let).
Intro Eq_v_V0.
Cut (Trad_inv s (let' P E2 E1) C).
2:Exact (Trad_Trad_inv s (let' P E2 E1) C Trad_let).
Simpl;Intro HH.
Elim HH;Intros C2 HH1;Clear HH.
Elim HH1;Intros Trad_E2 HH;Clear HH1.
Elim HH;Intros C1 HH1;Clear HH.
Elim HH1;Intros Trad_E1 Com;Clear HH1.
Rewrite -> Com.
Elim (hyp_E2 s C2 Trad_E2 hab u ML_E2 CV Eq);Clear hyp_E2.
Intro Cam_val_E2;Intro HH.
Elim HH;Intros Eq_u CAM_C2;Clear HH.
Cut (Habite (Econs P u e0) (cons_squelette P s)).
2:Exact (cons_habite  P u e0 s hab).
Intro Hab.
Cut (Equiv false (Econs P u e0) (Cam_pair CV Cam_val_E2)).
2:Exact (Eqenv2 P e0 CV Eq u Cam_val_E2 Eq_u).
Intro Eq_u_e0.
Elim (hyp_E1 
(cons_squelette P s) C1 Trad_E1 Hab v ML_E1 (Cam_pair CV Cam_val_E2) Eq_u_e0).
Intro Cam_val_E1;Intro HH.
Elim HH;Intros Eq_v CAM_C1;Clear HH.
Exists Cam_val_E1;Split;Intros.
Rewrite <- Eq_v_V0.
Assumption.
Cut  (CAM_DS (ETcons CV (ETcons CV s0)) (o C2 (o cons C1))
                 (ETcons Cam_val_E1 s0)).
Cut (CAM_DS (ETcons CV s0) push (ETcons CV (ETcons CV s0))).
Exact (o_DS  (ETcons CV s0) 
             (ETcons CV (ETcons CV s0))
             (ETcons Cam_val_E1 s0)
                  push
                  (o C2 (o cons C1))).
Exact (PUSH s0 CV).   
Cut (CAM_DS (ETcons Cam_val_E2 (ETcons CV s0))  (o cons C1)
                   (ETcons Cam_val_E1 s0)). 
Cut (CAM_DS (ETcons CV (ETcons CV s0)) C2 
               (ETcons Cam_val_E2 (ETcons CV s0))).
Exact (o_DS (ETcons CV (ETcons CV s0))
            (ETcons Cam_val_E2 (ETcons CV s0))
            (ETcons Cam_val_E1  s0)
                  C2
                  (o cons C1)).
Exact (CAM_C2 (ETcons CV s0)).
Cut (CAM_DS (ETcons (Cam_pair CV Cam_val_E2) s0) C1
             (ETcons Cam_val_E1 s0)).
Cut (CAM_DS (ETcons Cam_val_E2 (ETcons CV s0)) cons 
               (ETcons (Cam_pair CV Cam_val_E2) s0)). 
Exact (o_DS  (ETcons Cam_val_E2 (ETcons CV s0))
             (ETcons (Cam_pair CV Cam_val_E2) s0)
             (ETcons Cam_val_E1 s0)
                   cons 
                   C1).
Exact (CONS s0 CV Cam_val_E2).
Exact (CAM_C1 s0).


(*****************************************************************************)
(*                     Case where E is a let_rec                             *)
(*****************************************************************************)



Intros e0 P x E0 E2 u ML_E2 hyp_E2 
s C Trad_letrec hab V0 ML_letrec CV Eq.
Cut (ML_DS e0 (letrec P x E0 E2) u).
2:Exact (Sem_letrec e0 P x E0 E2 u ML_E2).
Intro ML_letrec_u.
Cut (<MLval> u=V0).
2:Exact (ML_DS_determ e0 (letrec P x E0 E2) u ML_letrec_u V0 ML_letrec).
Intro Eq_u_V0.
Cut (Trad_inv s (letrec P x E0 E2) C).
2:Exact (Trad_Trad_inv s (letrec P x E0 E2) C Trad_letrec).
Simpl;Intro HH.
Elim HH;Intros C_E0 HH1;Clear HH.
Elim HH1;Intros Trad_E0 HH;Clear HH1.
Elim HH;Intros C2 HH1;Clear HH.
Elim HH1;Intros Trad_E2 Com;Clear HH1.
Rewrite -> Com.
Cut (Habite (Econs P (Clos_rec x E0 P e0) e0) (cons_squelette P s)).
Intro Hab.
2:Exact (cons_habite P (Clos_rec x E0 P e0) e0 s hab).
Cut (Equiv false (Econs P (Clos_rec x E0 P e0) e0)
        (Cam_pair CV (Cam_clos_rec C_E0 CV))).
Intro Eq_P_e0.
2:Cut (Equiv true (Clos_rec x E0 P e0) (Cam_clos_rec C_E0 CV)). 
2:Intro HH.
2:Exact (Eqenv2 P e0 CV Eq (Clos_rec x E0 P e0) (Cam_clos_rec C_E0 CV) HH).
2:Exact (Eqclos_rec P x E0 e0 C_E0 CV s Eq hab Trad_E0).
Elim (hyp_E2 (cons_squelette P s) C2 Trad_E2 Hab u ML_E2  
        (Cam_pair CV (Cam_clos_rec C_E0 CV)) Eq_P_e0).
Intro Cval_u;Intro HH.
Exists Cval_u;Split.
Elim HH;Intros Eq_u CAM_C2.
Rewrite <- Eq_u_V0.
Assumption.
Intro s0.
Cut (CAM_DS (ETcons CV (ETcons CV s0)) (o (cur_rec C_E0) (o cons C2))
                 (ETcons Cval_u s0)).
Cut (CAM_DS (ETcons CV s0) push (ETcons CV (ETcons CV s0))).
2:Exact (PUSH s0 CV).
Exact (o_DS (ETcons CV s0)
            (ETcons CV (ETcons CV s0))
            (ETcons Cval_u s0)
            push 
            (o (cur_rec C_E0) (o cons C2))).
Cut (CAM_DS (ETcons (Cam_clos_rec C_E0 CV) (ETcons CV s0))
                (o cons C2) (ETcons Cval_u s0)).
Cut (CAM_DS (ETcons CV (ETcons CV s0)) (cur_rec C_E0)
           (ETcons (Cam_clos_rec C_E0 CV) (ETcons CV s0))).
2:Exact (CUR_REC (ETcons CV s0) CV C_E0).
Exact (o_DS (ETcons CV (ETcons CV s0))
            (ETcons (Cam_clos_rec C_E0 CV) (ETcons CV s0))
            (ETcons Cval_u s0)
            (cur_rec C_E0)
            (o cons C2)).
Cut (CAM_DS (ETcons (Cam_pair CV (Cam_clos_rec C_E0 CV)) s0)
   C2 (ETcons Cval_u s0)).
Cut (CAM_DS (ETcons (Cam_clos_rec C_E0 CV) (ETcons CV s0))
            cons (ETcons (Cam_pair CV (Cam_clos_rec C_E0 CV)) s0)).
2:Exact  (CONS s0 CV (Cam_clos_rec C_E0 CV)).
Exact (o_DS (ETcons (Cam_clos_rec C_E0 CV) (ETcons CV s0))
            (ETcons (Cam_pair CV (Cam_clos_rec C_E0 CV)) s0)
            (ETcons Cval_u s0)
            cons 
            C2).
Elim HH;Intros Eq_u CAM_C2.
Exact (CAM_C2 s0).



Save.




(*****************************************************************************)





