(*****************************************************************************)
(*          Projet Formel - Calculus of Inductive Constructions V5.8         *)
(*****************************************************************************)
(*                                                                           *)
(*          General algebra : setoids, categories, monoids, groups           *)
(*                                                                           *)
(*          G. Huet, following P. Aczel's LEGO axiomatization. Aug. 1993     *)
(*                                                                           *)
(*****************************************************************************)

(* 1. The category of Setoids *)

(* 1.1 The type of Setoids *)

Definition U = Type.

Definition Rel = [S:U]S->S->Prop.

Section Equivalence.

Variable S: U.
Variable R: (Rel S).

Definition Refl = (x:S)(R x x).
Definition Sym = (x,y:S)(R x y) -> (R y x).
Definition Trans = (x,y,z:S)(R x y) -> (R y z) -> (R x z).

Definition PER = Sym /\ Trans.

Definition Equiv = Refl /\ Sym /\ Trans.

End Equivalence.

(* Setoids are triples composed of a Type S, a relation R over S,
   and a proof that R is an equivalence relation over S *)

Inductive Definition Setoid : U 
  = Setoid_intro : (S:U)(R:(Rel S))(Equiv S R) -> Setoid.

Definition elem = [A:Setoid]
                (<U>Match A with [S:U][R:(Rel S)][e:(Equiv S R)]S).

Lemma equal : (A:Setoid)(Rel (elem A)).
Goal.
Induction A; Intros; Simpl; Assumption.
Save. 

(* or equivalently with more redexes
Definition equal = [A:Setoid]
 (<[s:Setoid](Rel (elem s))>Match A with [S:U][R:(Rel S)][e:(Equiv S R)]R).
*)

Lemma pr_equiv : (A:Setoid)(Equiv (elem A) (equal A)).
Goal.
Induction A; Intros; Simpl; Assumption.
Save.

Lemma pr_refl : (A:Setoid)(Refl (elem A) (equal A)).
Goal.
Induction A; Simpl; Induction e; Trivial.
Save.

Lemma pr_sym : (A:Setoid)(Sym (elem A) (equal A)).
Goal.
Induction A; Simpl; Induction e; Induction 2; Trivial.
Save.

Lemma pr_trans : (A:Setoid)(Trans (elem A) (equal A)).
Goal.
Induction A; Simpl; Induction e; Induction 2; Trivial.
Save.


(* An example *)

Inductive Definition Nat : U
        = N : (n:nat)Nat.

Definition Eq_Nat = [N1,N2:Nat](<Prop>Match N1 with [n1:nat]
                               (<Prop>Match N2 with [n2:nat]<nat>n1=n2)).

Lemma set_of_nat : Setoid.
Goal.
Apply Setoid_intro with Nat Eq_Nat.
Unfold Equiv; Unfold Eq_Nat; Split.
Unfold Refl; Induction x; Simpl; Auto.
Split.
Unfold Sym; Induction x; Induction y; Simpl; Auto.
Unfold Trans; Induction x; Induction y; Induction z; Simpl; Induction 1; Auto.
Save.

(* Alternative : PERS instead of Equivalences *)

Inductive Definition Partial_Setoid : U
  = Partial_Setoid_intro : (S:U)(R:(Rel S))(PER S R)->Partial_Setoid.

Definition partial_elem = [A:Partial_Setoid]
                (<U>Match A with [S:U][R:(Rel S)][e:(PER S R)]S).

Lemma coherence : (A:Partial_Setoid)(Rel (partial_elem A)).
Goal.
Induction A; Intros; Simpl; Assumption.
Save.

Definition total = [A:Partial_Setoid][x:(partial_elem A)](coherence A x x).


(* 1.2  The set of maps between two Setoid *)

Section Maps.
Variables A,B: Setoid.

Definition Map_law = [f:(elem A)->(elem B)]
    (x,y:(elem A))(equal A x y) -> (equal B (f x) (f y)).

Inductive Definition Map : U =
    Map_intro : (f:(elem A)->(elem B))(p:(Map_law f))Map.

Definition ap = [m:Map](<(elem A)->(elem B)>Match m with 
         [f:(elem A)->(elem B)][p:(Map_law f)]f).

Lemma pres : (m:Map)(Map_law (ap m)).
Goal.
Induction m; Simpl; Trivial.
Save.

Definition ext = [f,g:Map]
         (x:(elem A))(equal B (ap f x) (ap g x)).

Lemma Equiv_map_eq : (Equiv Map ext).
Goal.
Intros; Unfold Equiv; Split.
Unfold Refl; Unfold ext; Intros f x.
Apply (pr_refl B).
Split.
Unfold Sym; Unfold ext; Intros f g e x.
Apply (pr_sym B); Trivial.
Unfold Trans; Unfold ext; Intros f g h e1 e2 x.
Apply (pr_trans B); Trivial.
Exact (ap g x).
Save.

Definition Map_setoid = (Setoid_intro Map ext Equiv_map_eq).

End Maps.

Definition ap2 = 
  [A,B,C:Setoid][f:(elem (Map_setoid A (Map_setoid B C)))][a:(elem A)]
  (ap B C (ap A (Map_setoid B C) f a)).


(* 1.3  The type of categories *)

Section cat.

Variable Ob:U. (* Objects *)

Variable H:Ob->Ob->Setoid. (* Hom Setoid *)

Definition Comp_Type = 
  (a,b,c:Ob)(elem (Map_setoid (H a b) (Map_setoid (H b c) (H a c)))).

Variable o:Comp_Type.

Definition comp = [a,b,c:Ob](ap2 (H a b) (H b c) (H a c) (o a b c)).

Definition Assoc_law = (a,b,c,d:Ob)
      (f:(elem (H a b)))(g:(elem (H b c)))(h:(elem (H c d)))
      (equal (H a d) (comp a b d f (comp b c d g h))
                     (comp a c d (comp a b c f g) h)).

Definition Id_Type = (a:Ob)(elem (H a a)).

Variable id:Id_Type.

Definition Idl_law = (a,b:Ob)
   (f:(elem (H a b))) (equal (H a b) (comp a a b (id a) f) f).

Definition Idr_law = (a,b:Ob)
   (f:(elem (H b a))) (equal (H b a) f (comp b a a f (id a))).

End cat.

Inductive Definition Category : U =
 Cat_intro : (Ob:U)(Hom:Ob->Ob->Setoid)
             (o:(Comp_Type Ob Hom))(i:(Id_Type Ob Hom))
             (Assoc_law Ob Hom o) -> (Idl_law Ob Hom o i)
                                  -> (Idr_law Ob Hom o i) -> Category.

(* 1.4  The category of Setoid *)

(* Here we must have intermediate lemmas for comp *)

Lemma Map_comp : (a,b,c:Setoid)(Map a b)->(Map b c)->(Map a c).
Goal.
Intros a b c phi psi.
Apply Map_intro with [x:(elem a)](ap b c psi (ap a b phi x)).
Unfold Map_law.
Elim psi; Intros f p x y e; Apply p.
Elim phi; Intros g q; Apply q; Trivial.
Save.

Lemma Map_comp1 : 
      (a,b,c:Setoid)(Map a b)->(Map (Map_setoid b c) (Map_setoid a c)).
Goal.
Intros a b c phi; Apply Map_intro with (Map_comp a b c phi).
Unfold Map_law; Simpl; Intros f g e.
Unfold ext; Intro z.
Unfold Map_comp; Simpl; Apply e.
Save.

Lemma comp_set : (Comp_Type Setoid Map_setoid).
Goal.
Unfold Comp_Type; Intros; Simpl. 
Apply Map_intro with (Map_comp1 a b c).
Unfold Map_comp1; Simpl.
Unfold Map_law; Simpl; Intros f g e.
Unfold ext; Simpl; Intro z.
Elim z; Intros F p.
Unfold ext; Intro x; Simpl.
Apply p; Apply e.
Save.

Lemma assoc_set : (Assoc_law Setoid Map_setoid comp_set).
Goal.
Unfold Assoc_law; Simpl; Intros.
Unfold ext; Intro x; Simpl.
Elim h; Intros H p; Apply p.
Elim g; Intros G q; Apply q.
Elim f; Intros F r; Apply r.
Apply (pr_refl a).
Save.

Lemma id_set : (A:Setoid)(Map A A).
Goal.
Intro A.
Apply Map_intro with [x:(elem A)]x.
Unfold Map_law; Trivial.
Save.

Lemma idl_set : (Idl_law Setoid Map_setoid comp_set id_set).
Goal.
Unfold Idl_law; Intros.
Unfold comp id O.
Unfold ap2; Simpl.
Unfold ext; Simpl.
Intro x; Apply (pr_refl b).
Save.

Lemma idr_set : (Idr_law Setoid Map_setoid comp_set id_set).
Goal.
Unfold Idr_law; Intros.
Unfold comp id O.
Unfold ap2; Simpl.
Unfold ext; Simpl.
Intro x; Apply (pr_refl a).
Save.

Definition SET : Category = 
    (Cat_intro Setoid Map_setoid comp_set id_set assoc_set idl_set idr_set).


(* 2. The Types of Monoids and Groups *)

(* 2.1 The Type of Monoids *)

Definition Bin_op = [A:Setoid](Map_setoid A (Map_setoid A A)).

Section Monoid_laws.

Variable A:Setoid.
Variable op:(elem (Bin_op A)).
Variable i:(elem A).

Definition c = [x:(elem A)](ap A A (ap A (Map_setoid A A) op x)).
Definition E = (equal A).

Definition Monoid_ass = (x,y,z:(elem A))(E (c (c x y) z) (c x (c y z))).
Definition Monoid_idl = (x:(elem A))(E (c i x) x).
Definition Monoid_idr = (x:(elem A))(E (c x i) x).

End Monoid_laws.

Inductive Definition Monoid : U = 
   Monoid_intro : (A:Setoid)(op:(elem (Bin_op A)))(i:(elem A))
                  (Monoid_ass A op) -> 
                  (Monoid_idl A op i) -> (Monoid_idr A op i) -> Monoid.


Section Congruence.

Variable M:Monoid.

Definition carrier = (<Setoid>Match M with 
    [A:Setoid][op:(elem (Bin_op A))][i:(elem A)]
    [p:(Monoid_ass A op)][q:(Monoid_idl A op i)][r:(Monoid_idr A op i)]A).

Lemma op : (elem (Bin_op carrier)).
Goal.
Unfold carrier; Elim M; Simpl; Trivial.
Save.

Lemma unit : (elem carrier).
Goal.
Unfold carrier; Elim M; Simpl; Trivial.
Save.

Lemma monoid_ass : (Monoid_ass carrier op).
Goal.
Unfold carrier; Unfold op; Elim M; Simpl; Trivial.
Save.

Lemma monoid_idl : (Monoid_idl carrier op unit).
Goal.
Unfold carrier; Unfold op; Unfold unit; Elim M; Simpl; Trivial.
Save.

Lemma monoid_idr : (Monoid_idr carrier op unit).
Goal.
Unfold carrier; Unfold op; Unfold unit; Elim M; Simpl; Trivial.
Save.

Definition cong_l = [x,y,u:(elem carrier)][z:(equal carrier x y)]
   (pres carrier (Map_setoid carrier carrier) op x y z u).

Definition cong_r = [x,y,u:(elem carrier)][z:(equal carrier x y)]
   (pres carrier carrier (ap carrier (Map_setoid carrier carrier) op u) x y z).
End Congruence.


(* 2.2 The Type of Groups *)

Section Group_laws.

Variable A:Monoid.

Definition A_ = (carrier A).

Definition E = (equal A_).

Definition c = (ap2 A_ A_ A_ (op A)).

Definition e = (unit A).

Definition Inverses_rel = [x,y:(elem A_)](E (c x y) e).

Definition Group_invl = 
  [inv:(Map A_ A_)](x:(elem A_))
     (Inverses_rel (ap A_ A_ inv x) x).

Definition Group_invr = 
  [inv:(Map A_ A_)](x:(elem A_))
     (Inverses_rel x (ap A_ A_ inv x)).

End Group_laws.

Inductive Definition Group : U = 
   Group_intro : (A:Monoid)(inv:(Map (carrier A) (carrier A))) 
                 (Group_invl A inv) -> (Group_invr A inv) -> Group.


(* 3. The Group of Permutations of a Set *)

(* 3.1 The Monoid of Endomorphisms on a Set *)

Lemma Endo_Monoid : Setoid->Monoid.
Goal.
Intro X; Apply Monoid_intro with (Map_setoid X X) (comp_set X X X) (id_set X).
Exact (assoc_set X X X X).
Exact (idl_set X X).
Exact (idr_set X X).
Save.

(* 3.2 The Monoid of inverse pairs on a Monoid *)

Section Inv.

Variable A:Monoid.

Definition A_ = (carrier A).
Definition AX = (op A).

(* The set Inverses_set of inverse pairs on A *)

Inductive Definition Inverses : U = 
   Inverses_intro : (a,a':(elem A_))
                    (Inverses_rel A a a') ->
                    (Inverses_rel A a' a) -> Inverses.

Definition Inverses_eq = [A1,A2:Inverses](<Prop>Match A1 with 
  [a1,a1':(elem A_)][i1:(Inverses_rel A a1 a1')][i'1:(Inverses_rel A a1' a1)]
            (<Prop>Match A2 with 
  [a2,a2':(elem A_)][i2:(Inverses_rel A a2 a2')][i'2:(Inverses_rel A a2' a2)]
            (equal A_ a1 a2))).

Lemma Inverses_equiv : (Equiv Inverses Inverses_eq).
Goal.
Unfold Equiv.
Split.
Unfold Refl.
Induction x; Simpl.
Intros; Apply (pr_refl A_ a).
Split.
Unfold Sym.
Induction x; Induction y; Simpl.
Intros; Apply (pr_sym A_); Trivial.
Unfold Trans.
Induction x; Induction y; Induction z; Simpl.
Intros; Apply (pr_trans A_ a a0 a1); Trivial.
Save.

Definition Inverses_set = (Setoid_intro Inverses Inverses_eq Inverses_equiv).

(* Composition operation on Inverses_set *)

Definition E = (equal A_).

Definition Tr = (pr_trans A_).

Definition c = (ap2 A_ A_ A_ (op A)).

Definition e = (unit A).

Definition inv = [x,x':(elem A_)](E (c x x') e).

Definition opposite = [x:Inverses](<Inverses>Match x with
    [a,a':(elem A_)][p:(Inverses_rel A a a')][p':(Inverses_rel A a' a)]
    (Inverses_intro a' a p' p)).

(* General group theory equational reasoning *)
Lemma abb'a' : (a,b,a',b' : (elem A_))(inv a a')->
               (inv b b')->(inv (c a b) (c b' a')).
Goal.
Intros a b a' b' i j.
Red.
(* (a.b).b'.a' = e *)
Apply Tr. Instantiate (c a (c b (c b' a'))).
Apply (monoid_ass A).
(* a.b.b'.a' = e *)
Apply Tr. Instantiate (c a a'). 2: Trivial. (* i *)
(* a.b.b'.a' = a.a' *)
Apply (cong_r A (c b (c b' a')) a' a). 
(* b.b'.a' = a' *)
Apply Tr. Instantiate (c (c b b') a').
Apply (pr_sym A_).
Apply (monoid_ass A).
(* (b.b').a' = a' *)
Apply Tr. Instantiate (c e a'). 
Apply (cong_l A (c b b') e a'). Trivial. (* j *)
(* e.a' = a' *)
Apply (monoid_idl A).
Save.

Lemma Inverses_comp : 
  (elem Inverses_set)->(elem Inverses_set)->(elem Inverses_set).
Goal.
Induction 1; Intros a a' i i'.
Induction 1; Intros b b' j j'.
Apply Inverses_intro with (c a b) (c b' a').
Apply abb'a'; Trivial.
Apply abb'a'; Trivial.
Save.

Lemma Inverses_fun : 
   (elem Inverses_set)->(elem (Map_setoid Inverses_set Inverses_set)).
Goal.
Intro z; Apply (Map_intro Inverses_set Inverses_set (Inverses_comp z)).
Red; Intros x y.
Elim x; Elim y; Elim z; Simpl; Intros.
Apply (cong_r A a1 a0 a); Trivial.
Save.

(* Horrible *)
Lemma Inverses_op : (elem (Bin_op Inverses_set)).
Goal.
Red; Apply Map_intro with Inverses_fun.
Red; Induction x; Induction y; Simpl; Intros.
Red. 
Intro z; Elim z; Simpl; Intros.
Apply (cong_l A a a0 a1); Trivial.
Save.

(* The Monoid of inverse pairs *)

Lemma Inverses_ass : (Monoid_ass Inverses_set Inverses_op).
Goal.
Red.
Induction x; Induction y; Induction z; Simpl; Intros.
Apply (monoid_ass A).
Save.

Lemma Inverses_unit : Inverses.
Goal.
Apply (Inverses_intro e e); Apply (monoid_idl A e).
Save.

Lemma Inverses_idl : (Monoid_idl Inverses_set Inverses_op Inverses_unit).
Goal.
Red.
Induction x; Simpl; Intros.
Apply (monoid_idl A a).
Save.

Lemma Inverses_idr : (Monoid_idr Inverses_set Inverses_op Inverses_unit).
Goal.
Red.
Induction x; Simpl; Intros.
Apply (monoid_idr A a).
Save.

Definition Inverses_Monoid = 
    (Monoid_intro Inverses_set Inverses_op Inverses_unit 
                  Inverses_ass Inverses_idl Inverses_idr).


Lemma opposite_map : (Map_law Inverses_set Inverses_set opposite).
Goal.
Red; Intros x y. Elim x; Elim y; Simpl.
Intros a a' aa' a'a b b' bb' b'b ba.
Apply Tr. Instantiate (c b' e).
Apply (pr_sym A_).
Apply (monoid_idr A b').
Apply Tr. Instantiate (c b' (c a a')).
Apply (cong_r A e (c a a') b').
Apply (pr_sym A_); Trivial.
Apply Tr. Instantiate (c (c b' a) a').
Apply (pr_sym A_).
Apply (monoid_ass A).
Apply Tr. Instantiate (c e a').
2: Apply (monoid_idl A a').
Apply (cong_l A (c b' a) e a').
Apply Tr. Instantiate (c b' b).
2:Assumption.
Apply (cong_r A a b b').
Apply (pr_sym A_); Trivial.
Save.

Definition Inverses_inv_map = 
      (Map_intro Inverses_set Inverses_set opposite opposite_map).

Lemma Inverses_invl : (Group_invl Inverses_Monoid Inverses_inv_map).
Goal.
Red; Induction x; Trivial.
Save.

Lemma Inverses_invr : (Group_invr Inverses_Monoid Inverses_inv_map).
Goal.
Red; Induction x; Trivial.
Save.

Definition Inverses_group = 
  (Group_intro Inverses_Monoid Inverses_inv_map Inverses_invl Inverses_invr).

End Inv.

(* 3.4 The Group of permutations on a set *)

Definition Perm_Group = [X:Setoid](Inverses_group (Endo_Monoid X)).

Provide Algebra.


(*************************************************************************)


(* Note: comparison with P. Aczel's LEGO development, Fall 92.

- All Sigma types replaced by appropriate inductive constructions

In favor of LEGO:
- Lack of argument synthesis (notation |) painful
- Lack of local definitions [x=e]M prevents useful abbreviations
- Lack of infix notation x.f for (f x) not really a problem
- Better treatment of dependent goals - proof of comp spectacular
- Better treatment of Unfold. Automatic Unfold by Intro and Apply.

In favor of Coq:
- Sections look better than horrible Discharge
- No need of sigma types [but projections painful]

Problem: We want to change Lemma x:T into Definition x=X:T when
T:S with S # Prop, in order to keep X in the development - 
in particular for equal, set_of_nat, comp_set, id_set.
I.e. proof irrelevance only for sort S=Prop.

Minor renamings 
 Equiv_rel -> Equiv
 Set -> Setoid (necessary since Set is a Coq sort)
 pr_symm -> pr_sym
 pr_tran -> pr_trans
 Map_eq -> ext 
 set -> carrier
 el -> elem
 eq -> equal (in order not no hide =)
 i -> id
 O -> comp  (in order not to hide O:nat)
 id -> id_set
 comp -> comp_set
 Monoid_law1 -> Monoid_ass
 Monoid_law2 -> Monoid_idl et Monoid_idr
 Inverses_rel is just the half x*x'=1
 Inverses_law -> Group_invl et Group_invr
 f -> opposite
 mlf -> opposite_map

NB. We could request only Group_invl, and prove Group_invr by equational
reasoning. Actually, we could also prove Monoid_idr from the three other
axioms. Thus it is debatable whether a Group should be defined as a
Monoid with further properties.

*)

