(*****************************************************************************)
(*      Coq V5.8                                                             *)
(*****************************************************************************)
(*                                                                           *)
(*      First-order Unification                                              *)
(*                                                                           *)
(*      Joseph Rouyer                                                        *)
(*                                                                           *)
(*      November 1992                                                        *)
(*                                                                           *)
(*****************************************************************************)
(************************* term_unif.v ***************************************)
(*****************************************************************************)

Require Prelude.
Require Specif.
Require Le.
Require nat_complements.
Require nat_term_eq_quasiterm.
Require is_in_quasiterm_term_subst.
Require listv_is_in_lv.

(***********************************************************************)
(** Predicates that count the different elements of a list: ********************)
(**** diffelnb is an inductive definition  ***********************************)
(**** The definition of DIFFELNB is recursive. ****************************)
(***********************************************************************)

Inductive Definition diffelnb:listv->nat->Prop=
diffelnb_init:(diffelnb Nilv O)
|diffelnb_Consv_in:(l:listv)(n:nat)(x:var)
     (diffelnb l n)->(IS_IN_LV x l)->(diffelnb (Consv x l) n)
|diffelnb_Consv_n_in:(l:listv)(n:nat)(x:var)
     (diffelnb l n)->(~(IS_IN_LV x l))->(diffelnb (Consv x l) (S n)).

Definition DIFFELNB:listv->nat->Prop=[l:listv](<nat->Prop>Match l
with [x:nat]<nat>O=x
     [y:var][l1:listv][f:nat->Prop]
        [n:nat](<Prop>Match n
         with False
          [p:nat][q:Prop]
            ((IS_IN_LV y l1)/\(f (S p)))\/((~(IS_IN_LV y l1))/\(f p)))).

(***********************************************************************)
(**************** DIFFELNB is equivalent to diffelnb *********************)
(***********************************************************************)

Goal (l:listv)(n:nat)(diffelnb l n)->(DIFFELNB l n).
Intros;Elim H;Simpl;Auto.
Induction n0;Simpl;Auto.
Simpl;Auto.
Elim l0;Simpl;Auto.
Save diffelnb_DIFFELNB.

Goal (l:listv)(n:nat)(DIFFELNB l n)->(diffelnb l n).
Induction l.
Induction n;Simpl;Intros.
Exact diffelnb_init.
Absurd <nat>O=(S y);Auto.                     (*Apply O_S*)
Induction n;Intros.
Elim H0.                            (*(DIFFELNB (Consv v y) O):False*)
Elim H1;Intros h;Elim h;Intros.
Apply diffelnb_Consv_in;Auto.
Apply diffelnb_Consv_n_in;Auto.
Save DIFFELNB_diffelnb.

(***********************************************************************)
(****************** Existence of an integer that counts ******************)
(***********************************************************************)

Goal (l:listv){count:nat|(DIFFELNB l count)}.
Induction l.
Exists O;Simpl;Auto.  (*case l=Nilv*)
Intros.               (*case l=(Consv v y)*)
Elim H;Intros.
Elim (IS_IN_LV_decS v y);Intros.
Exists x.             (*case v in y:(IS_IN_LV v y)*)
Cut (DIFFELNB y x);Auto.
Elim x;Simpl;Auto.    (*case x=(S p)*)
Cut (IS_IN_LV v y);Auto. (*case x=O*)
Elim y;Simpl;Auto.    (*case v not in y:~(IS_IN_LV v y)*)
Exists (S x);Simpl;Auto.
Save DIFFELNBor.

(***********************************************************************)
(******** The empty list is the only list that contains 0 element. ************)
(***********************************************************************)

Goal (l:listv)(DIFFELNB l O)-><listv>Nilv=l.
Induction l;Simpl;Auto.
Intros;Absurd False;Auto.
Save DIFFELNB_O.

Goal (l:listv)(n:nat)(x:var)(DIFFELNB (Consv x l) n)->(~<nat>O=n).
Induction n.
Simpl;Intros;Absurd False;Auto.
Intros;Unfold not;Intros h;Absurd <nat>O=(S y);Auto.(*Apply O_S*)
Save DIFFELNB_Consv_n_O.

(*******************************************************************)
(********* Inclusion and strict inclusion for the lists as sets. ***********)
(*******************************************************************)

Inductive Definition inclv[l1,l2:listv]:Prop=
inclv_init:((y:var)(IS_IN_LV y l1)->(IS_IN_LV y l2))->(inclv l1 l2).

Inductive Definition st_inclv[l1,l2:listv]:Prop=
st_inclv_init:(x:var)(~(IS_IN_LV x l1))->(IS_IN_LV x l2)
->((y:var)(IS_IN_LV y l1)->(IS_IN_LV y l2))->(st_inclv l1 l2).

Goal (l:listv)~(st_inclv l Nilv).
Intros l;Unfold not;Intros h;Elim h;Auto.
Save n_st_inclv_l_Nilv.

(*******************************************************************)
(*************************** Remove : ******************************)
(**************** remove is inductively defined, ***********************)
(**************** REMOVE is recursively defined. *********************)
(*******************************************************************)

Inductive Definition remove[x:var]:listv->listv->Prop=
remove_init:(remove x Nilv Nilv)
|remove_eq:(l,l0:listv)(remove x l l0)->(remove x (Consv x l) l0)
|remove_n_eq:(l,l0:listv)(x0:var)(remove x l l0)->(~<var>x=x0)
->(remove x (Consv x0 l) (Consv x0 l0)).

Definition REMOVE:var->listv->listv->Prop=
[x:var][l:listv](<listv->Prop>Match l
with [l1:listv]<listv>Nilv=l1
     [v:var][l1:listv][f:listv->Prop]
        [l2:listv]((<var>x=v)/\(f l2))
         \/((~<var>x=v)/\(<Prop>Match l2 
             with False
             [w:var][l3:listv][p:Prop](<var>v=w)/\(f l3)))).
     

(*******************************************************************)
(************** REMOVE is equivalent to remove *********************)
(*******************************************************************)

Goal (l,l0:listv)(x:var)(remove x l l0)->(REMOVE x l l0).
Intros;Elim H;Intros;Simpl;Auto.
Save remove_REMOVE.

Goal (l,l0:listv)(x:var)(REMOVE x l l0)->(remove x l l0).
Induction l.
Intros;Elim H;Apply remove_init.
Induction l0.
Simpl;Intros.
Elim H0;Intros h1;Elim h1;Intros.
Elim H1;Apply remove_eq;Auto.
Absurd False;Auto.
Intros.
Elim H1;Intros.
Elim H2;Intros h h0;Elim h;
Apply remove_eq;Auto.
Elim H2;Intros h h0;Elim h0;Intros h1 h2;Elim h1;
Apply remove_n_eq;Auto.
Save REMOVE_remove.

(*******************************************************************)
(******************** REMOVE and not IS_IN_LV ********************)
(*******************************************************************)

Goal (l,l0:listv)(x:var)(REMOVE x l l0)->(~(IS_IN_LV x l))
-><listv>l=l0.
Intros l l0 x h;Cut (remove x l l0).
2:Apply REMOVE_remove;Auto.
Intros h0;Elim h0;Auto.
Intros;Elim H1;Simpl;Auto.
Simpl;Intros.
Elim H0;Auto;Unfold not;Intros;Elim H2;Auto.
Save REMOVE_n_IS_IN_LV_eq.


(*******************************************************************)
(********************** REMOVE and Consv ************************)
(*******************************************************************)

Goal (l,l0:listv)(x:var)(REMOVE x (Consv x l) l0)->(REMOVE x l l0).
Intros until x;Simpl;Intros.
Elim H;Intros h;Elim h;Intros;Auto.
Absurd <var>x=x;Auto.
Save REMOVE_Consv_eq.

Goal (l,l0:listv)(x,x0:var)(REMOVE x (Consv x0 l) (Consv x0 l0))
->(~<var>x0=x)->(REMOVE x l l0).
Intros l l0 x x0;Simpl;Intros;Auto.
Elim H;Intros h;Elim h;Intros.
Absurd <var>x0=x;Auto.
Elim H2;Auto.
Save REMOVE_Consv_n_eq.

(*******************************************************************)
(************** Existence of an l0 such that (REMOVE l l0) ***********)
(*******************************************************************)

Goal (l:listv)(x:var){l0:listv|(REMOVE x l l0)}.
Induction l;Intros.
Exists Nilv;Simpl;Auto.
Elim (H x);Intros l0 h.
Elim (var_eq_decS x v);Intros h0.
Elim h0;Exists l0;Simpl.
Left;Split;Auto.
Exists (Consv v l0);Simpl.
Right;Split;Auto.
Save sig_REMOVE.

Goal (l:listv)(x:var)<listv>Ex([l0:listv](REMOVE x l l0)).
Intros;Elim (sig_REMOVE l x);Intros l0 h;Exists l0;Auto.
Save exi_REMOVE.

(*******************************************************************)
(************************ Properties of REMOVE ********************)
(*******************************************************************)

Goal (l,l0:listv)(x,x0:var)(REMOVE x (Consv x0 l) l0)
     ->(~<var>x=x0)->(~<listv>Nilv=l0).
Induction l0;Intros.
Elim H;Intros h;Elim h;Intros.
Absurd <var>x=x0;Auto.
Absurd False;Auto.
Apply Diff with BNilv;Auto;Simpl;Auto.
Save Headv_REMOVE_Consv_Nilv.

Goal (l,l0:listv)(x,x0:var)(REMOVE x (Consv x0 l) l0)->(~<var>x=x0)
     ->((~<listv>Nilv=l0)/\<var>x0=(Headv x l0)).
Intros;Split.
Apply Headv_REMOVE_Consv_Nilv with l x x0;Auto.
Cut (REMOVE x (Consv x0 l) l0).
2:Auto.
Elim l0.
Simpl;Intros h;Elim h.
Intros h0;Elim h0;Auto.
Intros h0;Elim h0;Intros;Absurd False;Auto.
Intros;Simpl.
Elim H2;Intros.
Elim H3;Intros.
Absurd <var>x=x0;Auto.
Elim H3;Intros h1 h2;Elim h2;Auto.
Save REMOVE_Consv_n_eq_Headv.

(*******************************************************************)
(*********** Properties linked to REMOVE and IS_IN_LV ***************)
(*******************************************************************)

Goal (l,l0:listv)(x,x0:var)(REMOVE x l l0)->(IS_IN_LV x0 l0)->(IS_IN_LV x0 l).
Intros l l0 x x0 h;Cut (remove x l l0).
2:Apply REMOVE_remove;Auto.
Intro;Elim H;Simpl;Auto.
Intros.
Elim H3;Intros;Auto.
Save REMOVE_IS_IN_LV_IS_IN_LV.

Goal (l,l0:listv)(x:var)(REMOVE x l l0)->(x0:var)(~<var>x=x0)
     ->(IS_IN_LV x0 l)->(IS_IN_LV x0 l0).
Intros l l0 x H;Cut (remove x l l0).
2:Apply REMOVE_remove;Auto.
Intro;Elim H0;Simpl;Auto.
Intros.
Elim H4;Auto.
Intros;Absurd <var>x=x0;Auto.
Intros.
Elim H5;Auto.
Save REMOVE_n_eq_IS_IN_LV_IS_IN_LV.

(**********************************************************************)
(************************* REMOVE and st_inclv : **********************) 
(*********************** if x does not belong to l ************************)
(******* and x belongs to l0 and l st_inclv l0, and (REMOVE x l0 l1) ******)
(************************** then l st_inclv l1. ***************************)
(**********************************************************************)

Goal (l,l0,l1:listv)(x:var)(st_inclv (Consv x l) l0)
     ->(REMOVE x l0 l1)->(~(IS_IN_LV x l))->(st_inclv l l1).
Intros l l0 l1 x h;Elim h.
Simpl;Intros.
Apply st_inclv_init with x0.
Unfold not;Intros;Apply H;Auto.
Cut ~<var>x0=x.
Intros;Apply (REMOVE_n_eq_IS_IN_LV_IS_IN_LV l0 l1 x);Auto.
Unfold not;Intros;Elim H4;Auto.
(*Unfold not;Intros;Apply H;Auto.*)
Intros;Cut ~<var>x=y.
Intros;Apply (REMOVE_n_eq_IS_IN_LV_IS_IN_LV l0 l1 x);Auto.
Unfold not;Intros;Apply H3;Replace x with y;Auto.
Save st_inclv_Consv_REMOVE_n_IS_IN_LV_st_inclv.

(**********************************************************************)
(*********************** REMOVE and DIFFELNB **********************)
(**********************************************************************)

Goal (l:listv)(n:nat)(DIFFELNB l n)
     ->(l0:listv)(x:var)(REMOVE x l l0)->(IS_IN_LV x l)->(DIFFELNB l0 (pred n)).
Intros l n h;Cut (diffelnb l n).
2:Apply DIFFELNB_diffelnb;Auto.
Intros H;Elim H.
(* Case1 :  l=Nilv *)
Simpl;Intros;Absurd False;Auto.
(* Case2 l=(Consv x l0) and (IS_IN_LV x l0) *)
Intros until x0;Elim (var_eq_decP x x0);Intro.
(* Case2 ... H3:<var>x=x0*)
Elim H3;Intros.
Apply H1 with x;Auto.
Apply REMOVE_Consv_eq;Auto.
(* Case2 ... H3:~<var>x=x0*)
Elim l1.
(* Case2 ... H3:~<var>x=x0;l1=Nilv**)
Simpl;Intros.
Elim H4;Intros h0;Elim h0;Intros.
Elim H3;Auto.
Elim H7.
(* Case2 ... H3:~<var>x=x0;l1=(Consv v y)*)
Intros v y h0 h1;Elim h1;Intros.
Elim H4;Intros;Absurd <var>x=x0;Auto.
Elim H4;Intros h2 h3;Elim h3;Intros h4 h5;Elim h4.
Apply diffelnb_DIFFELNB;Apply diffelnb_Consv_in.
Apply DIFFELNB_diffelnb;Apply (H1 y x0);Auto.
Elim H5;Intros;Auto;Absurd <var>x=x0;Auto.
Apply REMOVE_n_eq_IS_IN_LV_IS_IN_LV with l0 x0;Auto.
(* Case3 l=(Consv x l0) and ~(IS_IN_LV x l0) *) 
Induction l1.
(* Case3 ...l1=Nilv*)
Intros.
Elim H3;Intros h0;Elim h0;Intros.
Cut ~(IS_IN_LV x l0).
2:Auto.
Elim H5;Intros;Cut <listv>l0=Nilv.
2:Apply REMOVE_n_IS_IN_LV_eq with x0;Auto.
Intros;Cut (DIFFELNB l0 n0).
2:Apply diffelnb_DIFFELNB;Auto;
Replace l0 with Nilv;Simpl;Auto.
Elim H8;Simpl;Auto.
Elim H6.
(*Case3 ...l1=(Consv v y)*)
Intros.
Change (DIFFELNB (Consv v y) n0).
Elim H4;Intros.
(*Case3 ...l1=(Consv v y))and (<var>x0=x) *)
(*           and (REMOVE x0 l0 (Consv v y))*)
Elim H6;Intros.
Cut (DIFFELNB l0 n0).
2:Apply diffelnb_DIFFELNB;Auto.
Cut ~(IS_IN_LV x l0).
2:Auto.
Elim H7;Intros;Replace (Consv v y) with l0;
Try Apply REMOVE_n_IS_IN_LV_eq with x0;Auto.
(*Case3 ...l1=(Consv v y))and ~<var>x0=x*)
(*                         and <var>x=v) and (REMOVE x0 l0 y) *)
Elim H6;Intros h0 h1;Elim h1;Intros.
Elim H5;Intros.
Cut (DIFFELNB y (pred n0)).
2:Apply (H1 y x0);Auto.
Cut (DIFFELNB l0 n0).
2:Apply diffelnb_DIFFELNB;Auto.
Pattern n0;Apply (nat_case n0).
Simpl;Intros;Absurd (IS_IN_LV x0 l0);Auto.
Replace l0 with Nilv;Try Apply DIFFELNB_O;Auto.
Simpl;Intros.
Elim H7;Right;Split;Auto.
Unfold not;Intros;Elim H2;Apply (REMOVE_IS_IN_LV_IS_IN_LV l0 y x0);Auto.
Absurd <var>x0=x;Auto.
Save REMOVE_n_IS_IN_LV_DIFFELNB_pred.

(**********************************************************************)
(************************* st_inclv and DIFFELNB **********************)
(**********************************************************************)

Goal (l:listv)(n:nat)(DIFFELNB l n)->(l0:listv)(n0:nat)(DIFFELNB l0 n0)
     ->(st_inclv l l0)->(le (S n) n0).
Intros l n hyp;Cut (diffelnb l n).
2:Apply DIFFELNB_diffelnb;Auto.
Intros h;Elim h.
(* Case1 :  l=Nilv *)
Intros l0 n0;Pattern n0;Apply (nat_case n0).
Simpl;Intros.
Elim H0;Intros;Cut (DIFFELNB l0 O);Auto.
Intros;Absurd (IS_IN_LV x l0);Auto.
Cut <listv>Nilv=l0.
Intros H5;Elim H5;Simpl;Auto.
Apply DIFFELNB_O;Auto.
Intros;Auto.(* Apply le_n_S ; Apply le_O_n *)
(* Case2 l=(Consv x l0) and (IS_IN_LV x l0) *)
Intros until n1;Pattern n1;Apply (nat_case n1).
Intros h0 h1;Elim h1;Intros.
Absurd (IS_IN_LV x0 l1);Auto.
Cut <listv>Nilv=l1.
Intros H5;Elim H5;Simpl;Auto.
Apply DIFFELNB_O;Auto.
Intros;Apply (H0 l1 (S m));Auto.
Elim H3;Intros.
Apply st_inclv_init with x0;Auto.
Unfold not;Intros;Elim H4;Simpl;Auto.
Intros;Apply H6;Simpl;Auto.
(* Case3 l=(Consv x l0) and ~(IS_IN_LV x l0) *) 
Intros until n1;Pattern n1;Apply (nat_case n1).
Intros h0 h1;Elim h1;Intros.
Absurd (IS_IN_LV x0 l1);Auto.
Cut <listv>Nilv=l1.
Intros H5;Elim H5;Simpl;Auto.
Apply DIFFELNB_O;Auto.
Intros;Apply le_n_S.
Elim (exi_REMOVE l1 x);Intros l2 H5.
Apply (H0 l2).
Replace m with (pred (S m));
Try Apply REMOVE_n_IS_IN_LV_DIFFELNB_pred with l1 x;Auto.
Elim H3;Intros.
Apply (H7 x);Simpl;Auto.
Apply st_inclv_Consv_REMOVE_n_IS_IN_LV_st_inclv with l1 x;Auto.
Save DIFFELNB_st_inclv_le_S.

(**********************************************************************)
(************************* inclv and DIFFELNB ************************)
(**********************************************************************)

Goal (l:listv)(n:nat)(DIFFELNB l n)->(l0:listv)(n0:nat)(DIFFELNB l0 n0)
     ->(inclv l l0)->(le n n0).
Intros l n hyp;Cut (diffelnb l n).
2:Apply DIFFELNB_diffelnb;Auto.
Intros h;Elim h.
(* Case1 :  l=Nilv *)
Auto.(*Apply le_O_n*)
(* Case2 l=(Consv x l0) and (IS_IN_LV x l0) *)
Intros.
Apply H0 with l1;Auto.
Apply inclv_init;Intros;Elim H3;Simpl;Auto.
(* Case3 l=(Consv x l0) and ~(IS_IN_LV x l0) *) 
Intros until n1;Pattern n1;Apply (nat_case n1).
Intros;Absurd (IS_IN_LV x l1).
Cut <listv>Nilv=l1.
2:Apply DIFFELNB_O;Auto.
Intros H4;Elim H4;Simpl;Auto.
Elim H3;Simpl;Auto.
Intros;Elim (exi_REMOVE l1 x);Intros.
Apply le_n_S;Apply H0 with x0.
Replace m with (pred (S m));Auto.
Apply REMOVE_n_IS_IN_LV_DIFFELNB_pred with l1 x;Auto.
Elim H3;Simpl;Auto.
Apply inclv_init;Intros.
Apply REMOVE_n_eq_IS_IN_LV_IS_IN_LV with l1 x;Auto.
Unfold not;Intros h0;Absurd (IS_IN_LV y l0);Auto;Elim h0;Auto.
Elim H3;Simpl;Auto.
Save inclv_le.

(*******************************************************************)
(********************* (infv t u) is true  ****************************)
(********* if every variable of t is a variable of u and *****************)
(***** if there exists a variable of u which is not a variable of t. ********)
(************ One reads t is less than u by its variables. ***************)
(*******************************************************************)

Inductive Definition infv[t1,t2:quasiterm]:Prop=
infv_init:(x:var)(~(IS_IN x t1))->(IS_IN x t2)
     ->((y:var)(IS_IN y t1)->(IS_IN y t2))->(infv t1 t2).

(*******************************************************************)
(********** A closed quasiterm has no variable ***********************)
(*******************************************************************)

Inductive Definition clos[t:quasiterm]:Prop=
clos_init:((x:var)(~(IS_IN x t)))->(clos t).

(*******************************************************************)
(*********** The list of the variables that occur in a quasiterm *********)
(*******************************************************************)

Definition list_var:quasiterm->listv=[t:quasiterm](<listv>Match t
with [x:var](Consv x Nilv)
     [l:fun]Nilv
     [l:fun][t0:quasiterm][l:listv]l
     [t1:quasiterm][l1:listv][t2:quasiterm][l2:listv](Appv l1 l2)).

(*******************************************************************)
(********** (IS_IN x t) and (IS_IN_LV x (list_var t)) are equivalent. *****)
(*******************************************************************)

Goal (t:quasiterm)(x:var)(IS_IN x t)->(IS_IN_LV x (list_var t)).
Induction t;Simpl;Intros;Auto.
(*Case t=(V v) : Apply or_is_inror ; Assumption
Case t=(C c) : Assumption
Case t=(Root f y) : Apply H ; Assumption *)
(*Case t=(ConsArg y y0) *) 
Elim H1;Intros.
Apply IS_IN_LV_Appv1;Auto.
Apply IS_IN_LV_Appv2;Auto.
Save IS_IN_IS_IN_LV.

Goal (t:quasiterm)(x:var)(IS_IN_LV x (list_var t))->(IS_IN x t).
Induction t;Simpl;Intros;Auto.
(*Case t=(C c) : Assumption
Case t=(Root f y) : Apply H ; Assumption *)
(*Case t=(V v)*)
Elim H;Intros;Auto;Absurd False;Auto.
(*Exact n_False; Assumption; Assumption *)
(*Case t=(ConsArg y y0) *)
Elim (IS_IN_LV_Appv_IS_IN_LV (list_var y) (list_var y0) x H1);Intros;Auto.
Save IS_IN_LV_IS_IN.

(**********************************************************************)
(********** To count the different variables of a quasiterm  *****************)
(**********************************************************************)
Goal (t:quasiterm)(DIFFELNB (list_var t) O)->(clos t).
Intros;Apply clos_init.
Unfold not;Intros x h;Absurd (IS_IN_LV x (list_var t)).
Replace (list_var t) with Nilv;
Try Apply (DIFFELNB_O (list_var t));Auto.
Apply IS_IN_IS_IN_LV;Auto.
Save DIFFELNB_O_clos.

(**********************************************************************)
(********************** clos and ConsArg. ******************************)
(**********************************************************************)

Goal (t1,t2:quasiterm)(clos (ConsArg t1 t2))->(clos t1).
Intros;Elim H;Intros;Apply clos_init.
Intros x;Unfold not;Intros;Elim (H0 x);Simpl;Auto.
Save closConsArg1.

Goal (t1,t2:quasiterm)(clos (ConsArg t1 t2))->(clos t2).
Intros;Elim H;Intros;Apply clos_init.
Intros x;Unfold not;Intros;Elim (H0 x);Simpl;Auto.
Save closConsArg2.

Goal (t1,t2:quasiterm)(clos (ConsArg t1 t2))->((clos t1)/\(clos t2)).
Intros;Elim H;Intros;Split;Apply clos_init.
Intros x;Unfold not;Intros;Elim (H0 x);Simpl;Auto.
Intros x;Unfold not;Intros;Elim (H0 x);Simpl;Auto.
Save closConsArg.

Goal (t:quasiterm)(clos t)->(DIFFELNB (list_var t) O).
Induction t.
(*Case t=(V v) *)
Intros;Elim H.
Simpl;Intros;Absurd <var>v=v;Auto.
(*Case t=(C c) *)
Intros;Simpl;Auto.
(*Case t=(Root f y) *)
Intros;Simpl;Auto.
(*Case t=(ConsArg y y0) *)
Simpl;Intros.
Elim (closConsArg y y0 H1);Intros.
Replace (list_var y) with Nilv;Try Apply DIFFELNB_O;Auto.
Save clos_DIFFELNBO.

Goal (t0,t1:quasiterm)(infv t0 t1)->
(st_inclv (list_var t0) (list_var t1)).
Intros;Elim H;Intros.
Apply st_inclv_init with x.
Unfold not;Intros;Elim H0;Apply IS_IN_LV_IS_IN;Auto.
Apply IS_IN_IS_IN_LV;Auto.
Intros;Apply IS_IN_IS_IN_LV;Apply H2;Apply IS_IN_LV_IS_IN;Auto. 
Save infv_st_inclv.

Goal (t0,t1:quasiterm)(st_inclv (list_var t0) (list_var t1))
     ->(infv t0 t1).
Intros;Elim H;Intros.
Apply infv_init with x.
Unfold not;Intros;Elim H0;Apply IS_IN_IS_IN_LV;Auto.
Apply IS_IN_LV_IS_IN;Auto.
Intros;Apply IS_IN_LV_IS_IN;Apply H2;Apply IS_IN_IS_IN_LV;Auto. 
Save st_inclv_infv.

Goal (t:quasiterm)(clos t)->(t0:quasiterm)~(infv t0 t).
Unfold not;Intros.
Elim H;Intros.
Elim H0;Intros.
Elim (H1 x);Auto.
Save n_infv_t_clos.

(**********************************************************************)
(*************** Quasisubstitution extensively equal to V ******************)
(**********************************************************************)

Goal (f:quasisubst)(t:quasiterm)((x:var)<quasiterm>(V x)=(f x))
     -><quasiterm>t=(Subst f t).
Induction t;Simpl;Auto;Intros.
Elim H;Auto.
Elim H;Elim H0;Auto.
Save eq_V_stab.

Goal (t:quasiterm)<quasiterm>t=(Subst V t).
Intros;Apply eq_V_stab;Auto.
Save V_stab.

Goal (t:quasiterm)(f:quasisubst)(clos t)->(<quasiterm>t=(Subst f t)).
Intros;Elim H;Intros.
Apply trans_equal with (Subst V t);
Try Apply V_stab.
Apply eq_restriction_s_t;Intros;Absurd (IS_IN x t);Auto.
Save clossubst.

(**********************************************************************)
(***************** Predicate dom : (dom f x) is true *********************)
(******************** if f(x) is not equal to V(x). ***********************)
(**********************************************************************)

Definition dom:quasisubst->var->Prop=
[f:quasisubst][x:var]~<quasiterm>(V x)=(f x).

Goal (f:quasisubst)(x:var)(~(dom f x))-><quasiterm>(V x)=(f x).
Intros f x;Elim (quasiterm_eq_decP (V x) (f x));Intros;Auto.
Elim H0;Auto.
Save n_dom.

Goal (f:quasisubst)(x:var)(dom f x)\/(~(dom f x)).
Intros f x;Elim (quasiterm_eq_decP (V x) (f x));Intros;Auto.
Right;Unfold not;Intros;Absurd <quasiterm>(V x)=(f x);Auto.
Save dom_decP.

(**********************************************************************)
(***************** Predicate range : (range f x) is true *******************)
(***************** if the variable x is in the image **********************)
(***************** of an other variable under f. *************************)
(**********************************************************************)

Inductive Definition range[s:quasisubst;x:var]:Prop=
range_init:(y:var)(dom s y)->(IS_IN x (s y))->(range s x).

Goal (f:quasisubst)(x:var)(~(range f x))
     ->(y:var)(dom f y)->~(IS_IN x (f y)).
Unfold not;Intros.
Elim H;Apply range_init with y;Auto.
Save n_range_n_IS_IN.

Goal (f:quasisubst)(x,y:var)(~(range f x))->(IS_IN x (f y))-><var>x=y.
Intros.
Elim (dom_decP f y);Intros h.
(*Case (dom f y) *)
Absurd (IS_IN x (f y));Auto.
Unfold not;Intros;Elim H;Intros.
Apply range_init with y;Auto.
(*Case ~(dom f y) *)
Change (IS_IN x (V y)).
Replace (V y) with (f y);Auto.
Elim (n_dom f y);Auto.
Save n_range_IS_IN_eq.

Goal (f:quasisubst)(t:quasiterm)(x:var)(~(range f x))
     ->(IS_IN x (Subst f t))->(IS_IN x t).
Induction t;Simpl;Intros;Auto.
Simpl;Intros;Apply n_range_IS_IN_eq with f;Auto.
Elim H2;Intros;Auto.
Save n_range_IS_IN_Subst.

(**********************************************************************)
(***************** Predicate over : (over f t) is true *********************)
(***************** if f does not modify the variables *********************)
(***************** which are not in t. **********************************)
(**********************************************************************)

Definition over : quasisubst->quasiterm->Prop=
[f:quasisubst][t:quasiterm](x:var)(~(IS_IN x t))
-><quasiterm>(V x)=(f x).

(**********************************************************************)
(***************** Predicate under : (under f t) is true ******************)
(***************** if every variable which is in the image ****************)
(***************** of an other variable under f is in t. ******************)
(**********************************************************************)

Definition under : quasisubst->quasiterm->Prop=
[s:quasisubst][t:quasiterm](x,y:var)(dom s y)
->(IS_IN x (s y))->(IS_IN x t).

(**********************************************************************)
(***************** If (under f t) is true then ****************************)
(***************** every variable which is in the *************************)
(***************** image of t under the quasisubstitution f ****************)
(***************** was before in t. *************************************)
(**********************************************************************)

Goal (f:quasisubst)(t:quasiterm)(x:var)(IS_IN x (Subst f t))
     -><var>Ex([y:var](IS_IN y t)/\(IS_IN x (f y))).
Induction t;Simpl;Intros;Auto.(*Case t=(Root f u) *)
(*Case t=(V v) *)
Exists v;Auto.
(*Case t=(C c) *)
Elim H.
(*Case t=(ConsArg y y0) *)
Elim H1;Intros.
Elim H with x;Intros;Auto.
Elim H3;Intros;Exists x0;Auto.
Elim H0 with x;Intros;Auto.
Elim H3;Intros;Exists x0;Auto.
Save IS_IN_Subst_exi_IS_IN.

Goal (f:quasisubst)(t:quasiterm)(under f t)
     ->(x:var)(IS_IN x (Subst f t))
     ->(IS_IN x t).
Unfold under;Intros.
Elim (IS_IN_Subst_exi_IS_IN f t x H0);Intros x0 h;Elim h;Intros.
Elim (var_eq_decP x0 x);Intros h0.
Elim h0;Auto.
Apply (H x x0);Auto.
Unfold dom;Unfold not;Intros.
Elim h0;Symmetry;Change (IS_IN x (V x0)).
Replace (V x0) with (f x0);Auto.
Save under_IS_IN_Subst_IS_IN.

(**********************************************************************)
(***************** Strict order sub_term *********************************)
(***************** (usual sense). ***************************************)
(**********************************************************************)

Inductive Definition sub[u:quasiterm]:quasiterm->Prop=
     subConsArgl:(t:quasiterm)(sub u (ConsArg u t))
|subConsArgr:(t:quasiterm)(sub u (ConsArg t u))
|subRoot:(l:fun)(sub u (Root l u))
|subConsArgl2:(t,v:quasiterm)(sub u t)->(sub u (ConsArg t v))
|subConsArgr2:(t,v:quasiterm)(sub u t)->(sub u (ConsArg v t))
|subRoot2:(t:quasiterm)(l:fun)(sub u t)->(sub u (Root l t)).

(**********************************************************************)
(***************** Recursive definition : *********************************)
(**********************************************************************)

Definition SUP:quasiterm->quasiterm->Prop=
[t:quasiterm](<quasiterm->Prop>Match t
with [x:var][u:quasiterm]False
     [l:fun][u:quasiterm]False
     [l:fun][u1:quasiterm][sup1:quasiterm->Prop][u:quasiterm]
           ((<quasiterm>u1=u)\/(sup1 u))
     [u1:quasiterm][sup1:quasiterm->Prop]
          [u2:quasiterm][sup2:quasiterm->Prop]
          [u:quasiterm]((<quasiterm>u1=u)\/(<quasiterm>u2=u)
                          \/(sup1 u)\/(sup2 u))).

Definition SUB:quasiterm->quasiterm->Prop=
[t1,t2:quasiterm](SUP t2 t1).

(**********************************************************************)
(***************** SUB is equivalent to sub *****************************)
(**********************************************************************)

Goal (t1,t2:quasiterm)(sub t1 t2)->(SUB t1 t2).
Intros;Elim H;Unfold SUB;Simpl;Auto.
Save sub_SUB.

Goal (t1,t2:quasiterm)(SUB t1 t2)->(sub t1 t2).
Induction t2;Intros.
(*Case t=(V v) *)
Elim H.
(*Case t=(C c) *)
Elim H.
(*Case t=(Root f y) *)
Elim H0;Intros.
Elim H1;Apply subRoot.
Apply subRoot2;Auto.
(*Case t=(ConsArg y y0) *)
Elim H1;Intros.
Elim H2;Apply subConsArgl.
Elim H2;Intros.
Elim H3;Apply subConsArgr.
Elim H3;Intros.
Apply subConsArgl2;Auto.
Apply subConsArgr2;Auto.
Save SUB_sub.

(**********************************************************************)
(***************** Transitivity of SUP and SUB : ***********************)
(**********************************************************************)

Goal (v,u,w:quasiterm)(SUP v u)->(SUP w v)->(SUP w u).
Induction v.
Intros;Absurd False;Auto.(*Case v=(V v0) *)
Intros;Absurd False;Auto.(*Case v=(C f) *)
(*Case v=(Root f y)*)
Induction w.
Intros;Absurd False;Auto.(*Case w=(V v0) *)
Intros;Absurd False;Auto.(*Case w=(C f0) *)
(*Case v=(Root f y);w=(Root f0 y0)*)
Intros.
Elim H2;Intros;Elim H1;Intros;Simpl;Auto.
Elim H4;Replace y0 with (Root f y);Simpl;Auto.
Replace y0 with (Root f y);Simpl;Auto.
(*Case v=(Root f y);w=(ConsArg y0 y1)*)
Intros.
Elim H3;Intros.
Replace y0 with (Root f y);Simpl;Auto.
Elim H4;Intros.
Replace y1 with (Root f y);Simpl;Auto.
Elim H5;Intros;Simpl;Auto.
(*Case v=(ConsArg y y0)*)
Induction w.
Intros;Absurd False;Auto.(*Case w=(V v0) *)
Intros;Absurd False;Auto.(*Case w=(C f0) *)
(*Case v=(ConsArg y y0);w=(Root f0 y0)*)
Intros.
Elim H3;Intros.
Replace y1 with (ConsArg y y0);Simpl;Auto.
Simpl;Auto.
(*Case v=(ConsArg y y0);w=(ConsArg y0 y1)*)
Intros.
Elim H4;Intros.
Elim H3;Intros.
Elim H6;Intros;Replace y1 with (ConsArg y y0);Simpl;Auto.
Replace y1 with (ConsArg y y0);Simpl;Auto.
Elim H5;Intros.
Replace y2 with (ConsArg y y0);Simpl;Auto.
Elim H6;Intros;Simpl;Auto.
Save trans_SUP.

Goal (v,u,w:quasiterm)(SUB u v)->(SUB v w)->(SUB u w).
Unfold SUB;Intros;Apply trans_SUP with v;Auto.
Save trans_SUB.

Goal (v,u:quasiterm)(SUB u v)->~<quasiterm>u=v.
Intros.
Unfold not;Intros h;Cut (SUB u v);Auto;Elim h.
Elim u;Simpl;Auto.(*Case u=(V v) or u=(C f)*)
(*Case u=(Root f y)*)
Intros.
Elim H1;Intros.
Apply H0;Pattern y;Replace y with (Root f y);Auto.
Apply H0;Apply trans_SUB with (Root f y);Unfold SUB;Simpl;Auto.
(*Case u=(ConsArg y y0)*)
Intros.
Elim H2;Intros.
Apply H0;Pattern y;Replace y with (ConsArg y y0);Auto.
Elim H3;Intros.
Apply H1;Pattern y0;Replace y0 with (ConsArg y y0);Auto.
Elim H4;Intros.
Apply H0;Apply trans_SUB with (ConsArg y y0);Unfold SUB;Simpl;Auto.
Apply H1;Apply trans_SUB with (ConsArg y y0);Unfold SUB;Simpl;Auto.
Save SUB_diff.

Goal (t1,t2:quasiterm)(f:quasisubst)(SUB t1 t2)
     ->(SUB (Subst f t1) (Subst f t2)).
Unfold SUB;Induction t2;Simpl;Intros.
Elim H.
Elim H.
Elim H0;Intros h0.
Elim h0;Auto.
Auto.
Elim H1;Intros h1.
Elim h1;Auto.
Elim h1;Intros h2;Elim h2;Auto.
Save SUB_subst.

Goal (x:var)(t:quasiterm)(f:quasisubst)(IS_IN x t)
     ->(~<quasiterm>(V x)=t)->(SUB (f x) (Subst f t)).
Intros;Change (SUB (Subst f (V x)) (Subst f t));Apply SUB_subst.
Unfold SUB;Cut ~<quasiterm>(V x)=t;Auto;
Cut (IS_IN x t);Try Apply IS_IN_is_in;Auto.
Elim t.
(*Case t=(V v)*)
Simpl;Intros;Absurd <quasiterm>(V x)=(V v);Auto.
Elim H1;Auto.
(*Case t=(C f)*)
Simpl;Intros;Absurd False;Auto.
(*Case t=(Root f0 y)*)
Intros;Elim (quasiterm_eq_decP (V x) y);Intros.
Elim H4;Simpl;Auto.
Cut (IS_IN x (Root f0 y));Simpl;Auto.
(*Case t=(ConsArg y y0)*)
Simpl;Intros.
Elim (quasiterm_eq_decP (V x) y);Intros;Auto.
Elim (quasiterm_eq_decP (V x) y0);Intros;Auto.
Elim H3;Intros;Auto.
Save IS_IN_SUB.

(*****************************************************************)
(******************* Idempotence of a quasisubstitution **************)
(*****************************************************************)

Definition idempotent:quasisubst->Prop=
[s:quasisubst](x:var)<quasiterm>(s x)=(Subst s (s x)).

(*****************************************************************)
(************ If for each variable of the range ***********************)
(******************** of a quasisubstitution, ************************)
(************** this is transformed to itself *************************)
(************ the quasisubstitution is idempotent. ********************)
(*****************************************************************)

Goal (f:quasisubst)((x:var)(range f x)-><quasiterm>(V x)=(f x))
     ->(idempotent f).
Unfold idempotent;Intros.
Elim (quasiterm_eq_decP (V x) (f x));Intros h.
(*Case <quasiterm>(V x)=(f x)*)
Elim h;Auto.
(*Case ~<quasiterm>(V x)=(f x)*)
Apply trans_equal with (Subst V (f x)).
Apply V_stab.
Apply eq_restriction_s_t.
Intros;Apply H;Apply range_init with x;Auto.
Save range_n_dom_idempotent.

(*****************************************************************)
(***** Conversely if a quasisubstitution is idempotent, then ***********)
(***** the variables of the domain are not in the range. **************)
(*****************************************************************)

Goal (f:quasisubst)(idempotent f)->(x:var)(dom f x)->~(range f x).
Unfold not;Intros.
Elim H0;Elim H1;Intros.
Apply (eq_restriction_t_s (f y) V f);Auto.
Elim H;Elim V_stab with (f y);Auto.
Save idempotent_n_range.

(*****************************************************************)
(************* Version without range of the same result. *************)
(*****************************************************************)

Goal (f:quasisubst)(idempotent f)
     ->(x:var)(dom f x)->(y:var)~(IS_IN x (f y)).
Intros.
Elim (dom_decP f y);Intros.
(*Case (dom f y)*)
Apply n_range_n_IS_IN;Auto.
Apply idempotent_n_range;Auto.
(*Case ~(dom f y)*)
Elim (n_dom f y H1);Simpl.
Unfold not;Intros h;Absurd (dom f y);Auto;Elim h;Auto.
Save idempotent_dom_n_IS_IN.

(*****************************************************************)
(**** If the quasisubstitutions f et g are such that: *******************)
(***** 1)the quasisubstitutions f et g are idempotent, *****************)
(***** 2)the variables images under g are in f(t), ********************)
(***** 3)the variables of the domain of g are in f(t), *****************)
(**** then g o f est idempotent. ************************************)
(*****************************************************************)

Goal (t:quasiterm)(f,g:quasisubst)(idempotent f)->(idempotent g)
     ->(under g (Subst f t))->(over g (Subst f t))
     ->(idempotent [x:var](Subst g (f x))).
Unfold under;Unfold over;Intros;Apply range_n_dom_idempotent;Intros.
Elim H3;Intros.
Elim (dom_decP g x);Intros.
(*Case (dom g x)*)
Absurd (IS_IN x (Subst g (f y)));Auto.
Apply n_IS_IN_Subst.
Intros;Apply idempotent_dom_n_IS_IN;Auto.
(*Case ~(dom g x)*)
Elim (dom_decP f x);Intros.
(*Case ~(dom g x);(dom f x)*)    
Absurd (IS_IN x (f y)).
Apply idempotent_dom_n_IS_IN;Auto.
Apply (n_range_IS_IN_Subst g (f y));Auto.
Unfold not;Intros H8;Elim H8;Intros.
Absurd (IS_IN x (Subst f t)).
Apply n_IS_IN_Subst.
Intros;Apply idempotent_dom_n_IS_IN;Auto.
Apply (H1 x y0);Auto.
(*Case ~(dom g x);~(dom f x)*)
Elim (n_dom f x H7);Simpl;Elim (n_dom g x H6);Simpl;Auto.
Save idempotent_Fondamental.

(*****************************************************************)
(************** Two evaluations of g(f(t)) : *************************)
(*****************************************************************)

Goal (f,g:quasisubst)(t:quasiterm)
  <quasiterm>(Subst g (Subst f t))=(Subst [x:var](Subst g (f x)) t).
Induction t;Simpl;Intros;Auto.
Elim H;Auto.
Elim H;Elim H0;Auto.
Save comp_subst.

Goal (f,g:quasisubst)(t:quasiterm)
<quasiterm>(Subst [x:var](Subst g (f x)) t)=(Subst g (Subst f t)).
Intros;Symmetry;Apply comp_subst.
Save exp_comp_subst.

(******************************************************************)
(******** First lemma of induction **********************************)
(****** if the variables of the domain of f and the range of f **********)
(*** are in (ConsArg p1 p2) and if the variables of the domain of g ****)
(****** are in f(ConsArg p3 p4) then the variables of the domain ******)
(***** of g o f are in (ConsArg (ConsArg p1 p3) (ConsArg p2 p4)). ***)
(******************************************************************)

Goal (p1,p2,p3,p4:quasiterm)(f,g:quasisubst)(over f (ConsArg p1 p2))
     ->(over g (ConsArg (Subst f p3) (Subst f p4)))
     ->(under f (ConsArg p1 p2))
     ->(over [x:var](Subst g (f x))
(ConsArg (ConsArg p1 p3) (ConsArg p2 p4))).
Unfold over under;Intros.
Cut ~(IS_IN x (ConsArg p1 p2)).
Intros h;Elim (H x h).
Simpl;Apply H0;Unfold not;Intros;Elim H2.

(*n_range_IS_IN_Subst:(f:quasisubst)(t:quasiterm)(x:var)
(~(range f x))->(IS_IN x (Subst f t))->(IS_IN x t)*)

Apply n_range_IS_IN_Subst with f.
Unfold not;Intros h0;Elim h0;Intros.
Elim h;Apply H1 with y;Auto.
Simpl;Elim H3;Auto.
Simpl;Unfold not;Intros h;Elim H2;Simpl;Elim h;Intros;Auto.
Save over_comp.

(******************************************************************)
(******** Second lemma of induction ********************************)
(******* If the variables of the domain of f and the images under f ****)
(******* are in (ConsArg p1 p2) and if ******************************)
(******* the variables of the domain of g and the images under g*******) 
(******* are in f(ConsArg p3 p4) then the images under gof ***********)
(******* are in (ConsArg (ConsArg p1 p3) (ConsArg p2 p4)). **********)
(******************************************************************)

Goal (p1,p2,p3,p4:quasiterm)(f,g:quasisubst)
     (under f (ConsArg p1 p2))
     ->(under g (ConsArg (Subst f p3) (Subst f p4)))
     ->(over f (ConsArg p1 p2))
     ->(over g (ConsArg (Subst f p3) (Subst f p4)))
     ->(under [x:var](Subst g (f x))
          (ConsArg (ConsArg p1 p3) (ConsArg p2 p4))).
Unfold over dom under;Intros.

(*under_IS_IN_Subst_IS_IN:(f:quasisubst)(t:quasiterm)
(under f t)->(x:var)(IS_IN x (Subst f t))->(IS_IN x t)*)

Apply under_IS_IN_Subst_IS_IN with g.
(*Goal (under...)*) 
Unfold under over dom;Intros.
Cut (IS_IN x0 (Subst f (ConsArg p3 p4))).
Intros H7;Elim (IS_IN_Subst_exi_IS_IN f (ConsArg p3 p4) x0 H7);Intros.
Elim H8;Intros h1 h2.
Apply under_IS_IN_Subst_IS_IN with f.
Unfold under;Intros.
Elim H with x2 y1;Simpl;Auto.
Elim (H0 x0 y0);Simpl;Auto.
Elim (H0 x0 y0);Simpl;Auto.
(*Goal (IS_IN x...)*) 
Elim (IS_IN_Subst_exi_IS_IN g (f y) x H4);Intros.
Elim H5;Intros.
Apply IS_IN_Subst_IS_IN with x0;Auto.
Elim (dom_decP f y);Intros.
(**)
Elim (H x0 y);Simpl;Auto.
Cut (IS_IN x0 (f y));Auto;Elim (n_dom f y H8);Intros.
Elim (dom_decP g x0);Intros.
(**)
Elim (IS_IN_decP x0 (ConsArg (Subst f p3) (Subst f p4)));Intros h.
Elim (IS_IN_Subst_exi_IS_IN f (ConsArg p3 p4) x0 h);Intros x1 h1;
Elim h1;Intros.
Elim (dom_decP f x1);Intros.
(**)
Elim H with x0 x1;Simpl;Auto.
(**)
Cut (IS_IN x0 (f x1));Auto;Elim (n_dom f x1 H13);Intros.
Cut (IS_IN x1 (ConsArg p3 p4));Auto;Elim H14.
Simpl;Intros h2;Elim h2;Intros;Auto.
Absurd <quasiterm>(V x0)=(g x0);Auto.
Absurd <quasiterm>(V y)=(Subst g (f y));Auto.
Elim (n_dom f y H8);Simpl;Elim H9;Elim (n_dom g x0 H10);Auto.
Save under_comp.

(*****************************************************************)
(*** If the domain of the quasisubstitution f is finite, *****************)
(*** then either f is the injection V from var is_ino quasiterm, ********)
(******** or there exists x in var such that f(x) is not V(x). **********)
(*** In the second case f decreases the number of variables. ***********)
(*****************************************************************)

Goal (t:quasiterm)(f:quasisubst)(over f t)
     ->((<var>Ex([x:var]
     (~<quasiterm>(V x)=(f x))))\/((v:var)<quasiterm>(V v)=(f v))).
(*cut*)
Cut (l:listv)(f:quasisubst)
((x:var)(~<quasiterm>(V x)=(f x))->(IS_IN_LV x l))
->((<var>Ex([x:var](~<quasiterm>(V x)=(f x))))\/
((v:var)<quasiterm>(V v)=(f v))).
Unfold over;Intros.
Apply H with (list_var t);Intros.
Apply IS_IN_IS_IN_LV.
Elim (IS_IN_decP x t);Intros;Auto.
Absurd <quasiterm>(V x)=(f x);Auto.
(*proof of the cut*)
Induction l;Simpl;Intros.
Right;Intros.
Elim (quasiterm_eq_decP (V v) (f v));Intros;Auto.
Elim (H v H0).
Elim (quasiterm_eq_decP (V v) (f v));Intros.
Apply (H f);Intros.
Elim (H0 x H2);Intros;Auto.
Absurd <quasiterm>(V v)=(f v);Auto;Elim H3;Auto.
Left;Exists v;Auto.
Save ident_or_not.

Goal (t:quasiterm)(f:quasisubst)(over f t)
     ->{x:var|~<quasiterm>(V x)=(f x)}+{(x:var)<quasiterm>(V x)=(f x)}.
Cut (l:listv)(f:quasisubst)
((x:var)(~<quasiterm>(V x)=(f x))->(IS_IN_LV x l))
->{x:var|~<quasiterm>(V x)=(f x)}+{(x:var)<quasiterm>(V x)=(f x)}.
Unfold over;Intros.
Apply H with (list_var t);Intros.
Apply IS_IN_IS_IN_LV.
Elim (IS_IN_decP x t);Intros;Auto.
Absurd <quasiterm>(V x)=(f x);Auto.
(**)
Induction l;Simpl;Intros.
Right;Intros.
Elim (quasiterm_eq_decP (V x) (f x));Intros;Auto.
Elim (H x H0).
Elim (quasiterm_eq_decS (V v) (f v));Intros.
Apply (H f);Intros.
Elim (H0 x H1);Intros;Auto.
Absurd <quasiterm>(V v)=(f v);Auto;Elim H2;Auto.
Left;Exists v;Auto.
Save ident_or_notS.

(*****************************************************************)
(******* If f is idempotent and if the variables of the domain of f ****)
(******* and the images under f are in (ConsArg t1 t3) *************) 
(******* and if the domain of f is not empty ***********************) 
(***** then the number of different variables in ConsArg(f(t2),f(t4)) ***)
(***** is strictly less than in ConsArg(ConsArg(t1,t2),ConsArg(t3,t4)).**)
(*****************************************************************)

Goal (t1,t2,t3,t4:quasiterm)(n:nat)
     (DIFFELNB (list_var (ConsArg (ConsArg t1 t2)(ConsArg t3 t4))) n)
     ->(f:quasisubst)(idempotent f)
     ->(over f (ConsArg t1 t3))
     ->(under f (ConsArg t1 t3))
     -><var>Ex([x:var](~<quasiterm>(V x)=(f x)))
     ->(n0:nat)
         (DIFFELNB (list_var (ConsArg (Subst f t2) (Subst f t4))) n0)
     ->(le (S n0) n).
Unfold over under;Intros.
Elim H3;Intros.
Apply DIFFELNB_st_inclv_le_S with 
(list_var (ConsArg (Subst f t2) (Subst f t4)))
(list_var (ConsArg (ConsArg t1 t2)(ConsArg t3 t4)));Auto.
Apply st_inclv_init with x.
Change ~(IS_IN_LV x (list_var (Subst f (ConsArg t2 t4)))).
Unfold not;Intros;Cut (IS_IN x (Subst f (ConsArg t2 t4)));Intros.
2:Apply IS_IN_LV_IS_IN;Auto.
Elim (IS_IN_Subst_exi_IS_IN f (ConsArg t2 t4) x H7);Intros.
Elim H8;Intros.
Absurd (IS_IN x (f x0));Auto;Apply idempotent_dom_n_IS_IN;Auto.
Apply IS_IN_IS_IN_LV;Elim (IS_IN_decP x (ConsArg t1 t3));Intros.
Cut (IS_IN x (ConsArg t1 t3)).
2:Auto.
Simpl;Intros h;Elim h;Auto.
Absurd <quasiterm>(V x)=(f x);Auto.
Intros;Cut (IS_IN y (Subst f (ConsArg t2 t4))).
2:Apply IS_IN_LV_IS_IN;Auto.
Intros;Apply IS_IN_IS_IN_LV;
Elim (IS_IN_Subst_exi_IS_IN f (ConsArg t2 t4) y H7);Intros.
Elim H8;Intros.
Elim (dom_decP f x0);Intros.
Cut (IS_IN y (ConsArg t1 t3)).
2:Apply (H2 y x0);Auto.
Simpl;Intros h;Elim h;Auto.
Cut <quasiterm>(V x0)=(f x0).
2:Apply n_dom;Auto.
Intros;Cut <var>y=x0.
Intros h;Cut (IS_IN x0 (ConsArg t2 t4)).
2:Auto.
Elim h;Simpl;Intros h1;Elim h1;Auto.
Cut (IS_IN y (f x0));Auto;Elim H12;Simpl;Auto.
Save f_n_id_minus.

(**********************************************************************)
(*************** The iterative construction of the unifiers, ****************)
(****** Here is a method which does not requires the axiom of choice *******)
(***************** but uses an induction over the integers. ****************)
(**********************************************************************)

Inductive Definition elem_subst[x:var;t:quasiterm;f:quasisubst]:Prop=
elem_subst_init:((y:var)(~<var>x=y)-><quasiterm>(V y)=(f y))
     ->((y:var)(<var>x=y)-><quasiterm>t=(f y))->(elem_subst x t f). 

Goal (t:quasiterm)(n:var)(f:quasisubst)
     {g:quasisubst|(<quasiterm>t=(g n))
     /\((p:nat)(~<nat>n=p)-><quasiterm>(f p)=(g p))}.
Induction n.
Intros;Exists [m:nat](<quasiterm>Match m
with t [p:nat][b:quasiterm](f m));Split;Auto.
Induction p;Auto.
Intros;Absurd <nat>O=O;Auto.
Intros.
Elim (H [x:nat](f (S x)));Intros g0 h0;Elim h0;Intros.
Exists [m:nat](<quasiterm>Match m
with (f O) [p:nat][b:quasiterm](g0 p));Split;Auto.
Induction p;Auto.
Intros y0 h1 h2;Apply H1;Unfold not;Intros;Elim h2;Auto.
Save sig_elem_subst0.

Goal (x:var)(t:quasiterm){f:quasisubst|(elem_subst x t f)}.
Intros;Elim (sig_elem_subst0 t x V);Intros.
Exists x0;Elim p;Intros;Apply elem_subst_init;Auto.
Intros y0 h;Elim h;Auto.
Save sig_elem_subst.

Goal (t:quasiterm)(x:var)(f:quasisubst)((y:var)
     ((~(<var>x=y))-><quasiterm>(V y)=(f y)))
     ->(~(IS_IN x t))-><quasiterm>t=(Subst f t).
Intros.
Replace (Subst f t) with (Subst V t);Try Apply V_stab.
Apply eq_restriction_s_t.
Intros;Elim (var_eq_decP x x0);Intros;Auto.
Absurd (IS_IN x0 t);Auto;Elim H2;Auto.
Save elem_subst_conserve.

(*****************************************************************)
(************** Order in quasisubst ********************************)
(*****************************************************************)

Inductive Definition less_subst[f,g:quasisubst]:Prop=
less_subst_init:(h:quasisubst)((x:var)(<quasiterm>(Subst h (f x))=(g x)))
     ->(less_subst f g).

(*****************************************************************)
(********************** Definition of an unifier: ********************)
(*****************************************************************)

Definition unif:quasisubst->quasiterm->quasiterm->Prop
=[f:quasisubst][t,u:quasiterm]<quasiterm>(Subst f t)=(Subst f u).

(*****************************************************************)
(***************** Definition of a minimal unifier: ******************)
(*****************************************************************)

Definition min_unif:quasisubst->quasiterm->quasiterm->Prop
=[f:quasisubst][t,u:quasiterm](g:quasisubst)(unif g t u)
     ->(less_subst f g).

(*****************************************************************)
(***************** Specification of the problem of the unification ******)
(*****************************************************************)
(********************* Unif defines the minimal unifiers.*************)
(*****************************************************************)

Inductive Definition Unification[t1,t2:quasiterm]:Set=
Unif_succeed:(f:quasisubst)
     (unif f t1 t2)
     ->(idempotent f)
     ->(over f (ConsArg t1 t2))
     ->(under f (ConsArg t1 t2))
     ->(min_unif f t1 t2)
(*------------ Verifications on the terms:begin -------------*)
(**) ->((l:list_nat)(L_TERM l t1)->(L_TERM l t2)
(**)     ->(<nat>(Length t1)=(Length t2))->(term_subst l f))
(*------------ Verifications on the terms:end --------------*)
     ->(Unification t1 t2)
|Unif_fail:((f:quasisubst)~<quasiterm>(Subst f t1)=(Subst f t2))
     ->(Unification t1 t2).

(*****************************************************************)
(************** Failure in the unification with the head symbol: *******)
(*****************************************************************)

Inductive Definition head_diff:quasiterm->quasiterm->Prop=
Fail_hd1:(l:fun)(t1,t2:quasiterm)(head_diff (C l) (ConsArg t1 t2))
|Fail_hd2:(l:fun)(t1,t2:quasiterm)(head_diff (ConsArg t1 t2) (C l))
|Fail_hd3:(l,l0:fun)(t:quasiterm)(head_diff (C l) (Root l0 t))
|Fail_hd4:(l,l0:fun)(t:quasiterm)(head_diff (Root l0 t) (C l))
|Fail_hd5:(t1,t2,t3:quasiterm)(l:fun)
            (head_diff (ConsArg t1 t2)(Root l t3))
|Fail_hd6:(t1,t2,t3:quasiterm)(l:fun)
            (head_diff (Root l t3)(ConsArg t1 t2)).

Goal (t1,t2:quasiterm)(head_diff t1 t2)->(Unification t1 t2).
Intros;Apply Unif_fail;Elim H;Intros.
Apply Diff with BC;Simpl;Auto.
Apply Diff with BConsArg;Simpl;Auto.
Apply Diff with BC;Simpl;Auto.
Apply Diff with BRoot;Simpl;Auto.
Apply Diff with BConsArg;Simpl;Auto.
Apply Diff with BRoot;Simpl;Auto.
Save Decomp_fail.

Hint Fail_hd1 Fail_hd2 Fail_hd3 Fail_hd4 Fail_hd5 Fail_hd6 Decomp_fail.

(*****************************************************************)
(*************** Symetry of the problem of the unification : **********)
(*****************************************************************)

Goal (t1,t2:quasiterm)(Unification t1 t2)->(Unification t2 t1).
Intros;Elim H;Unfold idempotent over under min_unif unif;Intros.
Apply Unif_succeed with f;
Unfold idempotent over under min_unif unif;Auto;Intros.
Apply o;Unfold not;Intros h;Elim H0;Simpl;Elim h;Auto.
Simpl;Elim (u0 x y);Auto.
Apply Unif_fail;Intros.
Unfold not;Intros;Elim (n f);Auto.
Save sym_Unification.

(*****************************************************************)
(*********** Unification when one of the terms is a variable : ********)
(*****************************************************************)

Goal (n:var)(t:quasiterm)(Unification (V n) t).
Intros;Elim (IS_IN_decS n t);Intros.
Elim (quasiterm_eq_decS (V n) t);Intros.
(*Case (IS_IN n t) and <quasiterm>(V n)=t*)
Elim a0;Apply Unif_succeed with [x:var](V x);
Unfold unif idempotent over under;Simpl;Auto;Intros.
Absurd <quasiterm>(V y)=(V y);Auto.
Unfold min_unif unif;Intros.
Apply less_subst_init with g;Auto.
(*------------ Verifications on the terms:begin -------------*)
Apply term_subst_init;Intros;Apply term_init;Simpl;Auto.
(*------------ Verifications on the terms:end --------------*)
(*Case (IS_IN n t) and ~<quasiterm>(V n)=t*)
Apply Unif_fail;Intros;Apply SUB_diff.
Simpl;Apply IS_IN_SUB;Auto.
(*Case ~(IS_IN n t)*)
Elim (sig_elem_subst n t);Intros.
Apply Unif_succeed with x;
Unfold unif idempotent over under;Simpl;Elim p;Intros.
Replace (x n) with t;Auto.
Apply elem_subst_conserve with n;Auto.
Elim (var_eq_decP n x0);Intros.
Elim (H0 x0);Auto.
Apply elem_subst_conserve with n;Auto.
Elim (H x0);Simpl;Auto.
Elim (var_eq_decP x0 n);Intros.
Elim H1;Auto.
Elim (H x0);Auto.
(*BOGUS: Unfold not;Intros;Elim H2;Auto.*)
Elim (var_eq_decP n y);Intros.
Replace t with (x y);Auto.
Symmetry;Auto.
Absurd <quasiterm>(V y)=(x y);Auto.
Unfold min_unif unif;Intros.
Apply less_subst_init with g.
Intros;Elim (var_eq_decP n x0);Intros.
Elim H2;Elim (H0 n);Auto.
Elim (H x0);Auto.
(*------------ Verifications on the terms:begin -------------*)
Apply term_subst_init.
Intros;Elim (var_eq_decP x0 n);Intros.
Elim (H0 x0);Auto.
Apply Length_SO_term;Auto.
Elim (H x0).
Apply term_init;Simpl;Auto.
Unfold not;Intros;Elim H4;Auto.
(*------------ Verifications on the terms:end --------------*)
Save UnifV1.

Goal (t:quasiterm)(x:var)(Unification t (V x)).
Intros;Apply sym_Unification;Apply UnifV1.
Save UnifV2.

Hint UnifV1 UnifV2.

(*****************************************************************)
(********** Unification when one of the terms is a constant : ********)
(*****************************************************************)

Goal (t:quasiterm)(l:fun)(Unification (C l) t).
Induction t;Auto.
Intros;Elim (fun_eq_decS l f);Intros.
Elim a;Apply Unif_succeed with V;
Unfold unif idempotent over under;Simpl;Auto.
(*Intros;Absurd <quasiterm>(V y)=(V y);Auto.*)
Unfold min_unif unif;Intros.
Apply less_subst_init with g ;Auto.
(*------------ Verifications on the terms:begin -------------*)
Intros;Apply term_subst_init;Intros;Apply term_init;Simpl;Auto.
(*------------ Verifications on the terms:end --------------*)
Apply Unif_fail;Intros;Simpl.
Apply C_diff_C;Auto.
Save UnifC1.
Goal (t:quasiterm)(l:fun)(Unification t (C l)).
Intros;Apply sym_Unification;Apply UnifC1.
Save UnifC2.

Hint UnifC1 UnifC2.

(*****************************************************************)
(********** Link between the unification of  t1 and t2 ****************)
(****** and the unification of (Root l1 t1) and (Root l2 t2) : *********)
(*****************************************************************)

Goal (t1,t2:quasiterm)(l1,l2:fun)
     (Unification t1 t2)
     ->(Unification (Root l1 t1) (Root l2 t2)).
Intros;Elim (fun_eq_decS l1 l2);Intros.
Elim a;Elim H;Unfold unif idempotent over under;Intros.
Apply Unif_succeed with f;
Unfold unif idempotent over under;Simpl;Auto.
Elim u;Auto.
Unfold min_unif unif;Simpl;Intros.
Apply m;Unfold unif.
Apply proj_Root2 with l1 l1;Auto.
Intros.
Elim H0;Elim H1;Intros;Apply t;Auto.
Elim H4;Auto.
Apply Unif_fail;Intros.
Simpl;Apply Root_diff_Root;Auto.
Apply Unif_fail;Intros.
Simpl;Apply Root_diff_Root;Auto.
Save UnifRoot.

Hint UnifRoot.

(********************************************************************)
(********** The failure of the unification of t1 and t3 implies ************)
(* the failure of the unification of (ConsArg t1 t2) and (ConsArg t3 t4) : **)
(********************************************************************)

Goal (t1,t2,t3,t4:quasiterm)
     ((f:quasisubst)(~<quasiterm>(Subst f t1)=(Subst f t3)))
     ->(Unification (ConsArg t1 t2) (ConsArg t3 t4)).
Intros;Apply Unif_fail;Intros.
Simpl;Apply ConsArg_diff_ConsArg;Auto.
Save UnifConsArgfail1.

Hint UnifConsArgfail1.

(*****************************************************************)
(********** The unification of the ground terms : ********************)
(*****************************************************************)

Goal (t1,t2:quasiterm)
     (DIFFELNB (list_var (ConsArg t1 t2)) O)
     ->(Unification t1 t2).
Intros;Elim (quasiterm_eq_decS t1 t2);Intros.
Elim a.
Apply Unif_succeed with V;
Unfold unif idempotent over under min_unif;Auto.
Intros.
Absurd <quasiterm>(V y)=(V y);Auto.
Intros;Apply less_subst_init with g;Auto.
Intros;Apply term_subst_init;Intros;Apply term_init;Simpl;Auto.
Apply Unif_fail;Intros.
Elim (clossubst t1 f).
Elim (clossubst t2 f);Auto.
Apply closConsArg2 with t1;Apply DIFFELNB_O_clos;Auto.
Apply closConsArg1 with t2;Apply DIFFELNB_O_clos;Auto.
Save Unif_DIFFELNB_O.

(*****************************************************************)
(********* The number of variables of (ConsArg t1 t3)  **************)
(************ and the one of the variables of (ConsArg t2 t4) ********)
(********* are less or equal to the number of the variables of *********)
(********** (ConsArg (ConsArg t1 t2)(ConsArg t3 t4)). **************) 
(*****************************************************************)

Goal (t1,t2,t3,t4:quasiterm)(n:nat)
     (DIFFELNB (list_var (ConsArg (ConsArg t1 t2)(ConsArg t3 t4))) n)
     ->(n0:nat)(DIFFELNB (list_var (ConsArg t2 t4)) n0)
     ->({(le (S n0) n)}+{<nat>n0=n}).
Intros;Elim (le_decS n0 n);Intros.
Apply (le_S_eqS n0 n a).
Absurd (le n0 n);Auto.
Apply inclv_le with (list_var (ConsArg t2 t4))
(list_var (ConsArg (ConsArg t1 t2) (ConsArg t3 t4)));Auto.
Apply inclv_init;Intros;Apply IS_IN_IS_IN_LV;Simpl.
Elim (IS_IN_LV_IS_IN (ConsArg t2 t4) y H1);Auto.
Save DIFFELNB_ConsArg_ConsArg24_le.
 
Goal (t1,t2,t3,t4:quasiterm)(n:nat)
     (DIFFELNB (list_var (ConsArg (ConsArg t1 t2) (ConsArg t3 t4))) n)
     ->(n0:nat)(DIFFELNB (list_var (ConsArg t1 t3)) n0)
     ->({(le (S n0) n)}+{<nat>n0=n}).
Intros;Elim (le_decS n0 n);Intros h.
Apply (le_S_eqS n0 n h).
Absurd (le n0 n);Auto.
Apply inclv_le with (list_var (ConsArg t1 t3))
(list_var (ConsArg (ConsArg t1 t2) (ConsArg t3 t4)));Auto.
Apply inclv_init;Intros;Apply IS_IN_IS_IN_LV;Simpl.
Elim (IS_IN_LV_IS_IN (ConsArg t1 t3) y H1);Auto.
Save DIFFELNB_ConsArg_ConsArg13_le.

Hint eq_V_stab.

(*****************************************************************)
(******************** If f unifies t1 and u1 ************************)
(********** and if g unifies (Subst f t2) (Subst f u2), ****************)
(************** then [x:var](Subst g (f x)) unifies ********************)
(************** (ConsArg t1 t2) et (ConsArg u1 u2). ****************)
(*****************************************************************)

Goal (t1,t2,u1,u2:quasiterm)(f,g:quasisubst)
     (unif f t1 u1)
     ->(unif g (Subst f t2) (Subst f u2))
     ->(unif [x:var](Subst g (f x)) (ConsArg t1 t2) (ConsArg u1 u2)).
Unfold unif;Simpl;Intros.
Elim (comp_subst f g t2);Elim (comp_subst f g u2);
Elim (comp_subst f g t1);Elim (comp_subst f g u1).
Elim H;Elim H0;Auto.
Save unif_comp.

Hint unif_comp.

(*****************************************************************)
(**************** If f is a minimal unifier of t1 and u1 *************)
(********** and if g is a one of (Subst f t2) (Subst f u2), then *******)
(********************** [x:var](Subst g (f x)) ***********************) 
(************ is a one of (ConsArg t1 t2) and (ConsArg u1 u2) : ****)
(*****************************************************************)

Goal (t1,t2,u1,u2:quasiterm)(f,g:quasisubst)
     (min_unif f t1 u1)
     ->(min_unif g (Subst f t2) (Subst f u2))
     ->(min_unif [x:var](Subst g (f x)) (ConsArg t1 t2) (ConsArg u1 u2)).
Unfold min_unif unif;Simpl;Intros.
Elim (H g0);Intros.
Elim (H0 h);Intros.
Apply less_subst_init with h0.
Intros;Elim (H2 x).
Elim (exp_comp_subst g h0 (f x)).
Apply eq_restriction_s_t;Auto.
Elim (exp_comp_subst f h t2);Elim (exp_comp_subst f h u2).
Transitivity (Subst g0 t2).
Apply eq_restriction_s_t;Auto.
Transitivity (Subst g0 u2).
Apply proj_ConsArg2 with (Subst g0 t1) (Subst g0 u1);Auto;Intros.
Apply eq_restriction_s_t;Auto.
Apply proj_ConsArg1 with (Subst g0 t2) (Subst g0 u2);Auto;Intros.
Save min_unif_comp.

Hint min_unif_comp.

(*------------ Verifications on the terms:begin -------------*)


Goal (t1,t2,t3,t4:quasiterm)(f,g:quasisubst)(l:list_nat)
     (L_TERM l (ConsArg t1 t2))
     ->(L_TERM l (ConsArg t3 t4))
     ->(<nat>(Length (ConsArg t1 t2))=(Length (ConsArg t3 t4)))
     ->((l:list_nat)(L_TERM l t1)->(L_TERM l t3)
         ->(<nat>(Length t1)=(Length t3))->(term_subst l f))
     ->((l:list_nat)(L_TERM l (Subst f t2))->(L_TERM l (Subst f t4))
         ->(<nat>(Length (Subst f t2))=(Length (Subst f t4)))
         ->(term_subst l g))
     ->(term_subst l [x:var](Subst g (f x))).
Simpl;Intros.
Elim H;Elim H0;Intros.
Elim H5;Elim H7;Intros.
Cut (term_subst l f).
Intros hyp;Apply term_subst_init.
Intros;Simpl;Apply term_term_subst.
Elim hyp;Auto.
Apply H3;Try Apply L_TERM_term_subst;Auto.
Elim (term_subst_eq_Length l f t2);Elim (term_subst_eq_Length l f t4);
Auto.
Cut <nat >(<nat>Match (Length t1) with (Length t2) [x:nat]S)
=(<nat>Match (Length t3) with (Length t4) [x:nat]S);Auto.
Replace  (Length t1) with (S O);Try Apply SIMPLE_SO;Auto.
Replace  (Length t3) with (S O);Try Apply SIMPLE_SO;Auto.
Apply (H2 l H8 H10).
Replace  (Length t1) with (S O);Try Apply SIMPLE_SO;Auto.
Save term_subst_comp.
(*------------ Verifications on the terms:end --------------*)

(*****************************************************************)
(********** If f is a minimal unifier of t1 and t3 *******************)
(********** whose domain and image are in (ConsArg t1 t3) *********)
(**************** and if g is a minimal unifier *********************)
(**************** of (Subst f t2) and (Subst f  t4) ******************)
(******** then [x:var](Subst g (f x)) is also a minimal unifier *********)
(*********** of (ConsArg t1 t2) and (ConsArg t3 t4) : ***************)
(*****************************************************************)

Hint over_comp under_comp.

Goal (t1,t2,t3,t4:quasiterm)(f,g:quasisubst)
     (unif f t1 t3)
     ->(idempotent f)
     ->(over f (ConsArg t1 t3))
     ->(under f (ConsArg t1 t3))
     ->(min_unif f t1 t3)
     ->((l:list_nat)(L_TERM l t1)->(L_TERM l t3)
         ->(<nat>(Length t1)=(Length t3))->(term_subst l f))
     ->(unif g (Subst f t2) (Subst f t4))
     ->(idempotent g)
     ->(over g (ConsArg (Subst f t2) (Subst f t4)))
     ->(under g (ConsArg (Subst f t2) (Subst f t4)))
     ->(min_unif g (Subst f t2) (Subst f t4))
     ->((l:list_nat)(L_TERM l (Subst f t2))->(L_TERM l (Subst f t4))
         ->(<nat>(Length (Subst f t2))=(Length (Subst f t4)))
         ->(term_subst l g))
     ->(Unification (ConsArg t1 t2) (ConsArg t3 t4)).
Intros;Apply Unif_succeed with [x:var](Subst g (f x));Intros;Auto.
(*Apply unif_comp;over_comp;under_comp;min_unif_comp*)
Apply (idempotent_Fondamental (ConsArg t2 t4));Auto.
(*------------ Verifications on the terms:begin -------------*)
Apply term_subst_comp with t1 t2 t3 t4;Auto.
(*------------ Verifications on the terms:end --------------*)
Save two_succes.

(*****************************************************************)
(************** If f is a minimal unifier of t1 and t3 ***************)
(*********** whose domain and image are in (ConsArg t1 t3) ********)
(*********************** and if the unification  *********************)
(***************** of (Subst f t2) and (Subst f  t4) fails *************)
(**** then the unification of (ConsArg t1 t2) and (ConsArg t3 t4) *****)
(**************************** fails also. ***************************)
(*****************************************************************)

Goal (t1,t2,t3,t4:quasiterm)(f:quasisubst)
     (unif f t1 t3)
     ->(idempotent f)
     ->(min_unif f t1 t3)
     ->((g:quasisubst)~<quasiterm>(Subst g (Subst f t2))
                                =(Subst g (Subst f t4)))(*1*)
     ->(Unification (ConsArg t1 t2) (ConsArg t3 t4)).
Unfold unif idempotent over under min_unif;Intros;Apply Unif_fail;Intros.
Unfold not;Simpl;Intros.
Elim (H2 f0).
Elim (H1 f0);Unfold unif;Intros.
(**)
Elim (exp_comp_subst f f0 t2);Elim (exp_comp_subst f f0 t4).
Replace (Subst [x:var](Subst f0 (f x)) t2) with (Subst f0 t2).
Replace (Subst [x:var](Subst f0 (f x)) t4) with (Subst f0 t4).
Apply proj_ConsArg2 with (Subst f0 t1) (Subst f0 t3);Auto.
(**)
Apply eq_restriction_s_t;Intros.
Elim (H4 x).
Replace (Subst h (f x)) with (Subst h (Subst f (f x))).
Elim (exp_comp_subst f h (f x));Apply eq_restriction_s_t;Auto. 
Elim H0;Auto.
(**)
Apply eq_restriction_s_t;Intros.
Elim (H4 x).
Replace (Subst h (f x)) with (Subst h (Subst f (f x))).
Elim (exp_comp_subst f h (f x));Apply eq_restriction_s_t;Auto. 
Elim H0;Auto.
(**)
Apply proj_ConsArg1 with (Subst f0 t2) (Subst f0 t4);Auto.
Save one_only_succes.

(*****************************************************************)
(*********** If t1 and t3 can be unified by the identity, **************)
(******** then (ConsArg t1 t2) and (ConsArg t3 t4) can this also *****) 
(**************** if and only if t2 and t4 can this . ****************)
(*****************************************************************)

Goal (t1,t2,t3,t4:quasiterm)(f:quasisubst)
     ((x:var)<quasiterm>(V x)=(f x))
     ->(unif f t1 t3)
     ->(Unification t2 t4)
     ->(Unification (ConsArg t1 t2) (ConsArg t3 t4)).
Unfold unif;Intros.
Elim H1;Unfold unif over under idempotent min_unif;Intros.
(*Success*)
Apply two_succes with f f0;
Replace (Subst f t2) with t2;
Replace (Subst f t4) with t4;
Unfold unif over under idempotent dom;Simpl;Auto.
(*Apply eq_V_stab*)
Intros;Absurd <quasiterm>(V y)=(f y);Auto.
Unfold min_unif;Intros.
Apply less_subst_init with g.
Intros;Elim (H x);Auto.
(*------------ Verifications on the terms:begin -------------*)
(**)Intros;Apply term_subst_init;Intros;Apply term_init;
(**)Elim H;Simpl;Auto.
(*------------ Verifications on the terms:end --------------*)
(*Fail*)
Apply Unif_fail.
Simpl;Intros;Unfold not;Intros.
Elim (n f0);
Apply proj_ConsArg2 with (Subst f0 t1) (Subst f0 t3);Auto.
Save eq_V_stab3.

(*****************************************************************)
(******************** For any terms u1 and u2, *******************)
(********** either there exists a minimal unifier of u1 and u2 ********)
(*********** whose domain and image are in (ConsArg u1 u2) *******)
(************** or u1 and u2 do not admit any unifier. *************)
(*****************************************************************)

Goal (n:nat)(u1,u2:quasiterm)
     (DIFFELNB (list_var (ConsArg u1 u2)) n)
     ->(Unification u1 u2).
Intros n;Pattern n;Apply (ind_leS n).


   (***************************************************************)
   (********************* Case without variable : ********************)
   (***************************************************************)
Intros;Apply Unif_DIFFELNB_O;Auto.
   (***************************************************************)
   (************** Case with p+1 different variables, *******************)
   (**************** induction on u1, then on u2 : *********************)
   (***************************************************************)
Induction u1;Auto.(*Apply UnifV1;UnifC1*)
(*Case u1 = (Root ...) : *)
Induction u2;Auto.(*Apply UnifV2;UnifC2;UnifRoot;Decomp_fail;Fail_hd6*)
(*Case u1 = (ConsArg ...) : *)
Induction u2;Auto.(*Apply UnifV2;UnifC2;Decomp_fail;Fail_hd5*)
Intros.
   (**      1 subgoal                                                *)
   (**      (Unification (ConsArg y y0) (ConsArg y1 y2))              *)
   (**       ============================            *)
   (**      number of different variables in (ConsArg y y1) :           *)
Elim (DIFFELNBor (list_var (ConsArg y y1)));Intros p0 H5.

   (***************************************************************)
   (******** Comparison of this number named p0 with p+1, **********)
   (*********** one knows that p0 < p+1 or p0 = p+1 : *************)
   (***************************************************************)
Elim (DIFFELNB_ConsArg_ConsArg13_le y y0 y1 y2 (S p) H4 p0 H5);Intros.
   (******************************************************************)
   (*  DIFFELNB_ConsArg_ConsArg13_le:(t1,t2,t3,t4:quasiterm)(n:nat)      *)
   (*(DIFFELNB (list_var (ConsArg (ConsArg t1 t2) (ConsArg t3 t4))) n) *)
   (*      ->(n0:nat)(DIFFELNB (list_var (ConsArg t1 t3)) n0          *)
   (*               ->({(le (S n0) n)}+{<nat>n0=n})                 *)
   (******************************************************************)

   (***************************************************************)
   (***************************************************************)
   (************************ Case p0 < p+1. **********************)
   (***************************************************************)
   (***************************************************************)

   (***************************************************************)
   (*********** one applies the recurrence hypothesis H : *************)
   (* (q:nat)(le q p)->(u1,u2:quasiterm)                              *)
   (* (DIFFELNB (list_var (ConsArg u1 u2)) q)->(Unification u1 u2)  *)
   (***************************************************************)
Elim (H p0 (le_S_n p0 p a) y y1 H5);Intros.
2 : Auto.
(*Apply UnifConsArgfail1*)

   (***************************************************************)
   (******** the unifier f of f and y1 is either the identity, ***********) 
   (***************** or different of the identity : *******************)
   (***************************************************************)
Elim (ident_or_notS (ConsArg y y1) f o);Intros.

   (***************************************************************)
   (************************ Case f non identity ********************)
   (***************************************************************)
Elim a0;Intros x H6.

   (***************************************************************)
   (********* one supposes (Unification (Subst f y0) (Subst f y2)) ******)
   (***************************************************************)
Cut (Unification (Subst f y0) (Subst f y2)).
Intros H7;Elim H7;Intros.
Apply two_succes with f f0;Auto.
Apply one_only_succes with f;Auto.

   (***************************************************************)
   (********* proof of (Unification (Subst f y0) (Subst f y2)) : *********)
   (********* 1) (list_var (ConsArg (Subst f y0) (Subst f y2))) *********)
   (******************** counts x1 different variables, ****************)
   (********** 2) one applies the recurrence hypothesis H *************)
   (********** 3) for this, one uses the fact that the unifier f *********)
   (********** makes the number of different variables decreasing. ******)
   (***************************************************************)
Elim (DIFFELNBor (list_var (ConsArg (Subst f y0) (Subst f y2))));
Intros p1 H7.
Apply (H p1);Auto.
Apply le_S_n;
Apply (f_n_id_minus y y0 y1 y2 (S p)) with f;Auto.
   (****************************************************************)
   (***** f_n_id_minus:(t1,t2,t3,t4:quasiterm)(n:nat) *********************)
   (* (DIFFELNB **************************************************)
   (*** (list_var (ConsArg (ConsArg t1 t2) (ConsArg t3 t4))) n) ********)
   (* ->(f:quasisubst)(idempotent f)->(over f (ConsArg t1 t3)) ********)
   (* ->(under f (ConsArg t1 t3)) **********************************)
   (* ->(<var>Ex([x:var](~<quasiterm>(V x)=(f x)))) ****************)
   (* ->(n0:nat)(DIFFELNB  **************************************)
   (*********** (list_var (ConsArg (Subst f t2) (Subst f t4))) n0) *******)
   (* ->(le (S n0) n) **********************************************)
   (****************************************************************)
Exists x;Auto.

   (****************************************************************)
   (**************** Case f = identity extentionnelly ******************)
   (** 1) eq_V_stab3 reduces the problem to (Unification y0 y2), **********)
   (** 2) (list_var (ConsArg y0 y2)) counts p0 different variables, ********)
   (***** 3) si p0 < p+1, one applies the recurrence hypothesis H *******)
   (***** 4) si p0 = p+1, one applies the recurrence hypothesis H1. *****)
   (****************************************************************)
Apply eq_V_stab3 with f;Auto.
Elim (DIFFELNBor (list_var (ConsArg y0 y2)));Intros p1 H6.
Elim (DIFFELNB_ConsArg_ConsArg24_le y y0 y1 y2 (S p) H4 p1 H6);
Intros.
   (***************************************************************)
   (* DIFFELNB_ConsArg_ConsArg24_le:(t1,t2,t3,t4:quasiterm)(n:nat) ****)
   (* (DIFFELNB *************************************************)
   (* ***(list_var (ConsArg (ConsArg t1 t2) (ConsArg t3 t4))) n) ******)
   (* ->(n0:nat)(DIFFELNB (list_var (ConsArg t2 t4)) n0) ***********)
   (* ->({(le (S n0) n)}+{<nat>n0=n}) ****************************)
   (***************************************************************)
Apply (H p1 (le_S_n p1 p a0) y0 y2);Auto.
Apply (H1 y2 );Elim b0;Auto.

   (***************************************************************)
   (***************************************************************)
   (********************* Case p0 = p+1 **************************)
   (***************************************************************)
   (***************************************************************)

   (***************************************************************)
   (******** 1) one applies the recurrence hypothesis H0 **************)
   (********** which proves (Unification y y1), **********************)
   (********2) one proceeds after exactly like above. ******************)
   (***************************************************************)
Elim (H0 y1);Elim b;Auto;Intros.(*Apply UnifConsArgfail1*)
Elim (ident_or_notS (ConsArg y y1) f o);Intros.
Elim a;Intros x H6.

   (***************************************************************)
   (********* one supposes (Unification (Subst f y0) (Subst f y2))******)
   (***************************************************************)
Cut (Unification (Subst f y0) (Subst f y2)).
Intros H7;Elim H7;Intros.
Apply two_succes with f f0;Auto.
Apply one_only_succes with f;Auto.

   (***************************************************************)
Elim (DIFFELNBor (list_var (ConsArg (Subst f y0) (Subst f y2))));
Intros p1 H7.
Apply (H p1);Auto.
Apply le_S_n;Apply (f_n_id_minus y y0 y1 y2 (S p)) with f;Auto.
Exists x;Auto.

   (***************************************************************)
Apply eq_V_stab3 with f;Auto.
Elim (DIFFELNBor (list_var (ConsArg y0 y2)));Intros p1 H6.
Elim (DIFFELNB_ConsArg_ConsArg24_le y y0 y1 y2 (S p) H4 p1 H6);
Intros.
Apply (H p1);Auto.(*Apply Apply le_S_n*)
Apply (H1 y2 );Elim b1;Auto.
Save proof_unif.

Goal (t,u:quasiterm)(Unification t u).
Intros;Elim (DIFFELNBor (list_var (ConsArg t u)));Intros n0 p;
Apply proof_unif with n0;Auto.
Save unif_proof.

Provide term_unif.
