(*****************************************************************************)
(*      Coq V5.8                                                             *)
(*****************************************************************************)
(*                                                                           *)
(*      First-order Unification                                              *)
(*                                                                           *)
(*      Joseph Rouyer                                                        *)
(*                                                                           *)
(*      November 1992                                                        *)
(*                                                                           *)
(*****************************************************************************)
(*********************** nat_term_eq_quasiterm.v *****************************)
(*****************************************************************************)

Require Prelude.
Require Specif.
Require Le.
Require nat_complements.

(*---------------------------------------------*)
(*------------ Verifications on the terms:begin -------------*)
(*---------------------------------------------*)

Section nat_sequence.

Inductive Definition list_nat:Set=
nil_nat:list_nat
|cons_nat:nat->list_nat->list_nat.

Definition At_last:nat->list_nat->list_nat
=[x:nat][l:list_nat]((<nat->list_nat>Match l 
   with [x:nat](cons_nat x nil_nat)
        [c:nat][q:list_nat][f:nat->list_nat]
          [x:nat](cons_nat c (f x))) x).

Definition List_queue_nat:list_nat->list_nat
=[l:list_nat](<list_nat>Match l 
              with nil_nat
                   [x:nat][tail:list_nat]
                     [tail_file:list_nat](At_last x tail_file)).

Definition Constr_f:list_nat->nat->nat=
[l:list_nat](<nat->nat>Match l with 
    [x:nat]O
    [m:nat][tail:list_nat][f_tail:nat->nat]
       [n:nat](<nat>Match n with m [p:nat][q:nat](f_tail p))).

(***********************************************************************)
(********************** Properties of  Constr_f ***************************)
(***********************************************************************)

Inductive Definition OK_Constr_f[f:list_nat->nat->nat]:Prop=
OK_Constr_f_init:((x:nat)<nat>O=(f nil_nat x))
->((val:nat)(l:list_nat)<nat>val=(f (cons_nat val l) O))
->((x,val:nat)(l:list_nat)<nat>(f l x)=(f (cons_nat val l) (S x)))
->(OK_Constr_f f).

Goal (OK_Constr_f Constr_f).
Apply OK_Constr_f_init;Simpl;Auto;Induction x;Simpl;Auto.
Save Constr_f_OK.

Inductive Definition
OK_At_last[f:nat->list_nat->list_nat]:Prop=
OK_At_last_init:
((x:nat)<list_nat>(cons_nat x nil_nat)=(f x nil_nat))
->((x,y:nat)<list_nat>(cons_nat y (cons_nat x nil_nat))
     =(f x (f y nil_nat)))
->(OK_At_last f).

Goal (OK_At_last At_last).
Apply OK_At_last_init;Simpl;Auto.
Save At_last_OK.

Inductive Definition
OK_List_queue_nat[f:list_nat->list_nat]:Prop=
OK_List_queue_nat_init:(<list_nat>nil_nat=(f nil_nat))
->((x:nat)(l:list_nat)<list_nat>(At_last x (f l))=(f (cons_nat x l)))
->(OK_List_queue_nat f).

Goal (OK_List_queue_nat List_queue_nat).
Apply OK_List_queue_nat_init;Simpl;Auto.
Save List_queue_nat_OK.

End nat_sequence.

(**********************************************************************)
(******************* Specifications of terms ******************************)
(**********************************************************************)

Section terms.

Definition fun:Set=nat.

Definition var:Set=nat.

Goal (x,x0:var)(<var>x=x0)\/(~<var>x=x0).
Intros;Elim (nat_eq_decP x x0);Auto.
Save var_eq_decP.

Goal (x,x0:var){<var>x=x0}+{~<var>x=x0}.
Intros;Elim (nat_eq_decS x x0);Auto.
Save var_eq_decS.

Goal (x,x0:fun)(<fun>x=x0)\/(~<fun>x=x0).
Intros;Elim (nat_eq_decP x x0);Auto.
Save fun_eq_decP.

Goal (x,x0:fun){<fun>x=x0}+{~<fun>x=x0}.
Intros;Elim (nat_eq_decS x x0);Auto.
Save fun_eq_decS.

 
Inductive Definition quasiterm:Set=
V:var->quasiterm                                 (*variable quasiterm*)
|C:fun->quasiterm                                (*constant quasiterm*)
|Root : fun->quasiterm->quasiterm                           (*rooting*)
|ConsArg  : quasiterm->quasiterm->quasiterm.
                                 (*building the pairs of quasiterms*)

(**********************************************************************)
(*********************** The arity function *****************************)
(**********************************************************************)

Hypothesis list_arity:list_nat.

Definition arity:fun->nat=(Constr_f (List_queue_nat list_arity)).

(**********************************************************************)
(********************** Length of a quasiterm ***************************)
(**********************************************************************)


Definition Length :quasiterm->nat=
[t:quasiterm](<nat>Match t with [x:var](S O) 
                 [f:fun](S O)
                 [f:fun][u:quasiterm][n:nat](S O)
                 [t1:quasiterm][m:nat][t2:quasiterm][n:nat]
                    (<nat>Match m with n [x:nat]S)).

(***********************************************************************)
(********************** Predicate SIMPLE *******************************)
(****** Simple term is constant term or variable term or rooted term. *******)
(***********************************************************************)

Definition SIMPLE:quasiterm->Prop=
[t:quasiterm](<Prop>Match t with
                 [x:var]True
                 [l:fun]True
                 [l:fun][u:quasiterm][p:Prop]True
                 [t1:quasiterm][p:Prop][t2:quasiterm][q:Prop]False).

(***********************************************************************)
(***************** Predicates l_term and L_TERM : ***********************)
(***********************************************************************)

Inductive Definition l_term:quasiterm->Prop=
l_term_initV:(x:var)(l_term (V x))
|l_term_initC:(f:fun)(<nat>O=(arity f))->(l_term (C f))
|l_term_Root:(f:fun)(t:quasiterm)(l_term t)
->(<nat>(arity f)=(Length t))->(l_term (Root f t))
|l_term_ConsArg:(t1,t2:quasiterm)(l_term t1)->(l_term t2)
->(SIMPLE t1)->(l_term (ConsArg t1 t2)).

Definition L_TERM : quasiterm->Prop
=[t:quasiterm](<Prop>Match t 
with [x:var]True 
     [c:fun](<nat>O=(arity c))
     [l:fun][t:quasiterm][p:Prop]p/\(<nat>(arity l)=(Length t))
     [t1:quasiterm][p1:Prop][t2:quasiterm][p2:Prop]
     (SIMPLE t1)/\p1/\p2).

(***********************************************************************)
(******************** L_TERM is equivalent to l_term : *******************)
(***********************************************************************)

Goal (t:quasiterm)(L_TERM t)->(l_term t).
Intro;Elim t;Simpl;Intros.
Apply l_term_initV;Auto.
Apply l_term_initC;Auto.
Apply l_term_Root.
Apply H;Elim H0;Auto.
Elim H0;Auto.
Apply l_term_ConsArg.
Apply H;Elim H1;Intros H2 H3;Elim H3;Intros;Auto.
Apply H0;Elim H1;Intros H2 H3;Elim H3;Intros;Auto.
Elim H1;Auto.
Save L_TERM_l_term.

Goal (t:quasiterm)(l_term t)->(L_TERM t).
Intros;Elim H;Simpl;Auto.
Save l_term_L_TERM.

(***********************************************************************)
(******************** Predicate term : ***********************************)
(***********************************************************************)

Inductive Definition term[t:quasiterm]:Prop=
term_init:(L_TERM t)->(SIMPLE t)->(term t).

(***********************************************************************)
(************ Decidability of L_TERM and decidability of term : ************)
(***********************************************************************)


Goal (t:quasiterm){(SIMPLE t)}+{~(SIMPLE t)}.
Intros;Elim t;Simpl;Auto.
Save SIMPLE_decS.

Goal (t:quasiterm){(L_TERM t)}+{~(L_TERM t)}.
Induction t;Simpl;Auto.
Intros f;Elim (nat_eq_decS O (arity f));Intros;Simpl;Auto.
Intros f y h0;Elim (nat_eq_decS (arity f) (Length y));Intros h.
Elim h;Elim h0;Intros;Auto.
Right;Unfold not;Intros h1;Elim h1;Intros;Elim b;Auto.
Right;Unfold not;Intros h1;Elim h1;Intros;Elim h;Auto.
Intros y h y0 h0;Elim h;Intros.
Elim h0;Intros.
Elim (SIMPLE_decS y);Intros h1.
Auto.
Right;Unfold not;Intros h2;Elim h2;Intros.
Elim h1;Unfold not;Intros;Auto.
Right;Unfold not;Intros h2;Elim h2;Intros.
Elim b;Elim H0;Intros;Auto.
Right;Unfold not;Intros h2;Elim h2;Intros.
Elim b;Elim H0;Intros;Auto.
Save L_TERM_decS.

Goal (t:quasiterm){(term t)}+{~(term t)}.
Intro;Elim (L_TERM_decS t);Intros h1;Elim (SIMPLE_decS t);Intros h2.
Left;Apply term_init;Auto.
Right;Unfold not;Intros h3;Elim h2;Elim h3;Intros;Auto.
Right;Unfold not;Intros h3;Elim h1;Elim h3;Intros;Auto.
Right;Unfold not;Intros h3;Elim h1;Elim h3;Intros;Auto.
Save term_decS.

(***********************************************************************)
(******************* Properties of  Length *******************************)
(***********************************************************************)

Goal (t:quasiterm){x:nat|<nat>(Length t)=(S x)}.
Induction t.
Exists O;Simpl;Auto.
Exists O;Simpl;Auto.
Exists O;Simpl;Auto.
Intros.
Elim H;Elim H0;Intros.
Simpl;Replace (Length y) with (S x0);
Replace (Length y0) with (S x).
Exists (<nat>Match x0 with (S x) [x:nat]S);Auto.
Save Length_n_O.
 
Goal (t,t0:quasiterm)~<nat>(S O)=(Length (ConsArg t t0)).
Intros;Elim (Length_n_O t);Elim (Length_n_O t0);Intros;Simpl.
Replace (Length t) with (S x0);
Replace (Length t0) with (S x).
Elim x0;Simpl.
Unfold not;Intros;Absurd <nat>O=(S x);Auto.
Intros;Unfold not;Intros;
Absurd <nat>O=(S (<nat>Match y with (S x) [x:nat]S));Auto.
Save n_SO_Length_ConsArg.

Goal (t:quasiterm)(SIMPLE t)-><nat>(S O)=(Length t).
Induction t;Simpl;Intros;Auto.
Absurd False;Auto.
Save SIMPLE_SO.

Goal (t:quasiterm)(L_TERM t)
->(<nat>(S O)=(Length t))->(term t).
Induction t;Intros;Apply term_init;Simpl;Auto.
Absurd <nat>(S O)=(Length (ConsArg y y0));Auto;
Apply n_SO_Length_ConsArg;Auto.
Save Length_SO_term.

Goal (t:quasiterm)
(term t)->((L_TERM t)/\(<nat>(S O)=(Length t))).
Induction t.
Intros;Simpl;Auto.
Intros;Simpl.
Split;Auto.
Elim H;Simpl;Intros h;Elim h;Auto.
Simpl;Intros.
Split;Auto.
Elim H0;Simpl;Intros h;Elim h;Intros;Split;Auto.
Simpl;Intros.
Split.
Elim H1;Simpl;Intros;Auto.
Elim H1;Simpl;Intros.
Absurd False;Auto.
Save term_L_TERM_Length.

End terms.

Section eq_quasiterm.

(***********************************************************************)
(************************ Structural predicates ***************************)
(***********************************************************************)

Definition BC:quasiterm->Prop=[t:quasiterm](<Prop>Match t 
with [x:var]False
     [x:fun]True [l:fun][p:quasiterm][P:Prop]False
     [t1:quasiterm][P1:Prop][t2:quasiterm][P2:Prop]False).

Definition BV:quasiterm->Prop=[t:quasiterm](<Prop>Match t
with [x:var]True
     [x:fun]False [l:fun][p:quasiterm][P:Prop]False
     [t1:quasiterm][P1:Prop][t2:quasiterm][P2:Prop]False).

Definition BRoot:quasiterm->Prop=[t:quasiterm](<Prop>Match t
with [x:var]False
     [x:fun]False [l:fun][p:quasiterm][P:Prop]True
     [t1:quasiterm][P1:Prop][t2:quasiterm][P2:Prop]False).

Definition BConsArg:quasiterm->Prop=[t:quasiterm](<Prop>Match t 
with [x:var]False
     [x:fun]False [l:fun][p:quasiterm][P:Prop]False
     [t1:quasiterm][P1:Prop][t2:quasiterm][P2:Prop]True).

(***********************************************************************)
(********************** Destroyers (Destructors): *************************)
(***********************************************************************)
Definition Destr1=[t:quasiterm](<quasiterm>Match t 
with V
     C
     [l:fun][p,q:quasiterm]p 
     [p1,p2,q1,q2:quasiterm]p1).

Definition Destr2=[t:quasiterm](<quasiterm>Match t
with V
     C
     [l:fun][p,q:quasiterm]p
     [p1,p2,q1,q2:quasiterm]q1).

Definition Destrvar=[X:var][t:quasiterm](<var>Match t
with [x:var]x
     [l:fun]X
     [l:fun][p:quasiterm][x:var]X
     [p1:quasiterm][x1:var][p2:quasiterm][x2:var]X).


Definition Destrfun=[F:fun][t:quasiterm](<fun>Match t
with [x:var]F
     [l:fun]l
     [l:fun][p:quasiterm][x:fun]l
     [p1:quasiterm][x1:fun][p2:quasiterm][x2:fun]F).

(**********************************************************************)
(******************** Equal in the Set of quasiterms : *******************)
(**********************************************************************)

Goal (l1,l2:fun)(<quasiterm>(C l1)=(C l2))->(<fun>l1=l2).
Intros;Replace l1 with (Destrfun l1 (C l1));Auto.
Replace l2 with (Destrfun l1 (C l2));Auto.
Elim H;Auto.
Save proj_C.

Goal (x1,x2:var)(<quasiterm>(V x1)=(V x2))->(<var>x1=x2).
Intros;Replace x1 with (Destrvar x1 (V x1));Auto.
Replace x2 with (Destrvar x1 (V x2));Auto.
Elim H;Auto.
Save proj_V.

Goal (t1,t2:quasiterm)(l1,l2:fun)
(<quasiterm>(Root l1 t1)=(Root l2 t2))->(<fun>l1=l2).
Intros;Replace l1 with (Destrfun l1 (Root l1 t1));Auto.
Replace l2 with (Destrfun l1 (Root l2 t2));Auto.
Elim H;Auto.
Save proj_Root1.

Goal (t1,t2:quasiterm)(l1,l2:fun)
(<quasiterm>(Root l1 t1)=(Root l2 t2))->(<quasiterm>t1=t2).
Intros;Replace t1 with (Destr1 (Root l1 t1));Auto.
Replace t2 with (Destr2 (Root l2 t2));Auto.
Elim H;Auto.
Save proj_Root2.

Goal (t1,t2,t3,t4:quasiterm)
(<quasiterm>(ConsArg t1 t2)=(ConsArg t3 t4))->(<quasiterm>t1=t3).
Intros;Replace t1 with (Destr1 (ConsArg t1 t2));Auto.
Replace t3 with (Destr1 (ConsArg t3 t4));Auto.
Elim H;Auto.
Save proj_ConsArg1.

Goal (t1,t2,t3,t4:quasiterm)
(<quasiterm>(ConsArg t1 t2)=(ConsArg t3 t4))->(<quasiterm>t2=t4).
Intros;Replace t2 with (Destr2 (ConsArg t1 t2));Auto.
Replace t4 with (Destr2 (ConsArg t3 t4));Auto.
Elim H;Auto.
Save proj_ConsArg2.

(**********************************************************************)
(****************** Not equal in the Set of the quasiterms : **************)
(**********************************************************************)
Goal  (l,l0:fun)(~<fun>l=l0)->(~(<quasiterm>(C l)=(C l0))).
Unfold not;Intros;Elim H;Apply proj_C;Auto.
Save C_diff_C.

Goal  (x,y:var)(~(<var>x=y))->(~(<quasiterm>(V x)=(V y))).
Unfold not;Intros;Elim H;Apply proj_V;Auto.
Save V_diff_V.

Goal  (t,t0,t1,t2:quasiterm)((~(<quasiterm>t=t1))\/(~(<quasiterm>t0=t2)))
->(~(<quasiterm>(ConsArg t t0)=(ConsArg t1 t2))).
Unfold not;Intros;Elim H;Intros;Elim H1.
Apply proj_ConsArg1 with t0 t2;Auto.
Apply proj_ConsArg2 with t t1;Auto.
Save ConsArg_diff_ConsArg.

Goal  (l,l0:fun)(t,t0:quasiterm)((~(<fun>l=l0))\/(~(<quasiterm>t=t0)))
->(~(<quasiterm>(Root l t)=(Root l0 t0))).
Unfold not;Intros;Elim H;Intros;Elim H1.
Apply proj_Root1 with t t0;Auto.
Apply proj_Root2 with l l0;Auto.
Save Root_diff_Root.

(**********************************************************************)
(*********** Decidability of the equality in the Set of quasiterms : *********)
(**********************************************************************)
Goal  (t,t0:quasiterm){<quasiterm>t=t0}+{~(<quasiterm>t=t0)}.
Induction t.
Induction t0.
(*t=(V ...)*)
Intros ;Elim (var_eq_decS v v0);Intros H.
Elim H;Auto.
Right;Apply V_diff_V;Auto.
Intros;Right;Apply (Diff quasiterm BV);Simpl;Auto.
Intros;Right;Apply (Diff quasiterm BV);Simpl;Auto.
Intros;Right;Apply (Diff quasiterm BV);Simpl;Auto.
(*t=(C ...)*)
Induction t0.
Intros;Right;Apply (Diff quasiterm BC);Simpl;Auto.
Intros;Elim (fun_eq_decS f f0);Intros H.
Elim H;Auto.
Right;Apply C_diff_C;Auto.
Intros;Right;Apply (Diff quasiterm BC);Simpl;Auto.
Intros;Right;Apply (Diff quasiterm BC);Simpl;Auto.
(*t=(Root...)*)
Induction t0.
Intros;Right;Apply (Diff quasiterm BRoot);Simpl;Auto.
Intros;Right;Apply (Diff quasiterm BRoot);Simpl;Auto.
Intros.
Elim (H y0);Intros.
Elim a;Elim (fun_eq_decS f f0);Intros.
Elim a0;Auto.
Right;Apply Root_diff_Root;Auto.
Right;Apply Root_diff_Root;Auto.
Intros;Right;Apply (Diff quasiterm BRoot);Simpl;Auto.
(*t=(ConsArg ...)*)
Induction t0.
Intros;Right;Apply (Diff quasiterm BConsArg);Simpl;Auto.
Intros;Right;Apply (Diff quasiterm BConsArg);Simpl;Auto.
Intros;Right;Apply (Diff quasiterm BConsArg);Simpl;Auto.
Intros.
Elim (H y1);Intros.
Elim (H0 y2);Intros.
Elim a;Elim a0;Auto.
Right;Apply ConsArg_diff_ConsArg;Auto.
Right;Apply ConsArg_diff_ConsArg;Auto.
Save quasiterm_eq_decS.

Goal  (t,t0:quasiterm)(<quasiterm>t=t0)\/(~<quasiterm>t=t0).
Intros;Elim (quasiterm_eq_decS t t0);Intros;Auto.
Save quasiterm_eq_decP.

(**********************************************************************)
(********************* End of equality in quasiterm **********************)
(**********************************************************************)


(***********************************************************************)
(************************) End eq_quasiterm. (***************************)
(***********************************************************************)

(*---------------------------------------------*)
(*------------ Verifications on the terms:end --------------*)
(*---------------------------------------------*)


Provide nat_term_eq_quasiterm.
