(*                               *)
(*  ENSEMBLES FINIS              *)
(*                               *)

(*  On definit 3 "types" mutuellement inductifs : Elt, Ensf et Word      *)
(*  On distingue elt et mot, car on a besoin du type mot plus tard.      *)
(*  Les constructeurs up et word permettent repectivement de considerer  *)
(*  un ensemble ou un mot en tant qu'element.				 *)

Inductive Set sortes =
  elt  : sortes
| ensf : sortes
| mot  : sortes.

Inductive Definition SE : sortes -> Set =
   SEempty   : (SE ensf)
 | SEadd     : (SE elt) -> (SE ensf) -> (SE ensf)

 | SEnatural : nat -> (SE elt)
 | SEcouple  : (SE elt) -> (SE elt) -> (SE elt)
 | SEup      : (SE ensf) -> (SE elt)
 | SEword    : (SE mot) -> (SE elt)

 | SEnil     : (SE mot)
 | SEcons    : (SE elt) -> (SE mot) -> (SE mot).

Definition Elt  = (SE elt).
Definition Ensf = (SE ensf).
Definition Word = (SE mot).

(*  Renommage des constructeurs  *)

Definition empty : Ensf = SEempty.
Definition add   : Elt -> Ensf -> Ensf = SEadd.

Definition natural : nat -> Elt = SEnatural.
Definition couple  : Elt -> Elt -> Elt = SEcouple.
Definition up      : Ensf -> Elt = SEup.
Definition word    : Word -> Elt = SEword.

Definition nil     : Word = SEnil.
Definition cons    : Elt -> Word -> Word = SEcons.  

(*  Inversion de quelques constructeurs  *)

Definition natural_inv : Elt -> nat =
  [e:Elt]
    (<[s:sortes]nat>Match e with
	O
        [x:Elt][Hx:nat][e:Ensf][He:nat]O

        [n:nat]n
        [a:Elt][Ha:nat][b:Elt][Hb:nat]O
	[e:Ensf][He:nat]O
	[w:Word][Hw:nat]O

	O
	[a:Elt][Ha:nat][w:Word][Hw:nat]O
	
    ).

Lemma nat_invol : (n:nat)<nat>(natural_inv (natural n))=n.
Goal.
Auto.
Save.

Definition word_inv : Elt -> Word =
  [e:Elt]
    (<[s:sortes]Word>Match e with
	nil
        [x:Elt][Hx:Word][e:Ensf][He:Word]nil

        [n:nat]nil
        [a:Elt][Ha:Word][b:Elt][Hb:Word]nil
	[e:Ensf][He:Word]nil
	[w:Word][Hw:Word]w

	nil
	[a:Elt][Ha:Word][w:Word][Hw:Word]nil
	
    ).


(*									*)
(*  Inductions : 							*)
(*     Pour pouvoir raisonner par induction sur la strucutre de Elt,	*)
(*     Ensf et Word, il faut montrer les trois lemmes ci-dessous, 	*)
(*     induction_ensf, induction_elt et induction_word.			*)
(*									*)
(*     Pour cela on commence par montrer un lemme d'induction pour	*)
(*     trois proprietes sur Ensf, Elt et Word simultanement, puis	*)
(*     on montre les trois lemmes voulus en fixant 2 des 3 proprietes	*)
(*     a True.								*)
(*									*)

Definition Pall = [Pensf:Ensf->Prop][Pelt:Elt->Prop][Pmot:Word->Prop]
  [s:sortes](<[t:sortes](SE t)->Prop>Match s with
                Pelt
		Pensf
		Pmot
	    ).

Lemma induction_all : (Pensf:Ensf->Prop)(Pelt:Elt->Prop)(Pmot:Word->Prop)
  (Pensf empty)
  -> ((x:Elt)(Pelt x)->(a:Ensf)(Pensf a)->(Pensf (add x a)))
  -> ((n:nat)(Pelt (natural n)))
  -> ((x:Elt)(Pelt x)->(y:Elt)(Pelt y)->(Pelt (couple x y)))
  -> ((e:Ensf)(Pensf e)->(Pelt (up e)))
  -> ((w:Word)(Pmot w)->(Pelt (word w)))
  -> (Pmot nil)
  -> ((x:Elt)(Pelt x)->(w:Word)(Pmot w)->(Pmot (cons x w)))
  -> (((x:Elt)(Pelt x)) /\ ((e:Ensf)(Pensf e)) /\ ((w:Word)(Pmot w))).
Goal.
Intros.
Cut (s:sortes)(x:(SE s))(Pall Pensf Pelt Pmot s x). 
Intro Hyp.
Split.
Exact (Hyp elt).
Split.
Exact (Hyp ensf).
Exact (Hyp mot).
Intros.
Elim x; Auto.
Save.

Lemma induction_ensf : (P:Ensf->Prop)
  (P empty)->((a:Elt)(b:Ensf)(P b)->(P (add a b)))->(e:Ensf)(P e).
Goal.
Intros.
Cut ( ((x:Elt)True) /\ ((e:Ensf)(P e)) /\ ((w:Word)True) ).
2:Apply (induction_all P [x:Elt]True [w:Word]True); Auto.
Intro Ht; Elim Ht; Clear Ht; Intros H1 Ht; Elim Ht; Clear Ht; Auto.
Save.

Lemma induction_elt : (P:Elt->Prop)
  ((n:nat)(P (natural n)))
  -> ((a,b:Elt)(P a)->(P b)->(P (couple a b)))
  -> ((e:Ensf)(P (up e)))
  -> ((w:Word)(P (word w)))
  -> ((x:Elt)(P x)).
Goal.
Intros.
Cut ( ((x:Elt)(P x)) /\ ((e:Ensf)True) /\ ((w:Word)True) ).
2:Apply (induction_all [e:Ensf]True P [w:Word]True); Auto.
Intro Ht; Elim Ht; Clear Ht; Auto.
Save.

Lemma induction_word : (P:Word->Prop)
  (P nil)
  -> ((x:Elt)(w:Word)(P w)->(P (cons x w)))
  -> ((w:Word)(P w)).
Goal.
Intros.
Cut ( ((x:Elt)True) /\ ((e:Ensf)True) /\ ((w:Word)(P w)) ).
2:Apply (induction_all [e:Ensf]True [x:Elt]True P); Auto.
Intro Ht; Elim Ht; Clear Ht; Intros H1 Ht; Elim Ht; Clear Ht; Auto.
Save.

(*									*)
(*  De meme pour (SE s)->Set.						*)
(*									*)

Axiom induction_word_Set : (P:Word->Set)
  (P nil)
  -> ((x:Elt)(w:Word)(P w)->(P (cons x w)))
  -> ((w:Word)(P w)).


(*								 	*)
(*  Les constructeurs sont distincts : 				 	*)
(*    On charge le fichier diff.v dans lequel on demontre les    	*)
(*    lemmes du type :						 	*)
(*    diff_natural_couple : ... ~(<Elt>(natural n)=(couple a b)) 	*)
(*									*)

Require diff.

(*  Quelques resultats triviaux sur les constructeurs...		*)

Lemma add_add : (a,b:Elt)(c,d:Ensf)
  (<Elt>a=b) -> (<Ensf>c=d)
     ->(<Ensf>(add a c)=(add b d)).
Goal.
Intros.
Rewrite H.
Rewrite H0.
Trivial.
Save.
Hint add_add.

Lemma couple_couple : (a,b,c,d:Elt)
  (<Elt>a=b) -> (<Elt>c=d)
     ->(<Elt>(couple a c)=(couple b d)).
Goal.
Intros.
Rewrite H.
Rewrite H0.
Trivial.
Save.

Lemma word_word : (a,b:Word)
  (<Word>a=b) -> (<Elt>(word a)=(word b)).
Goal.
Intros.
Apply (f_equal Word Elt); Auto.
Save.
Hint word_word.
 
Lemma word_word_inv : (a,b:Word) 
   (<Elt>(word a)=(word b)) -> (<Word>a=b).
Goal.
Intros.
Replace a with (word_inv (word a)); Auto.
Replace b with (word_inv (word b)); Auto.
Apply (f_equal Elt Word); Auto.
Save.


(*  Quelques simplifications  *)

Definition zero : Elt = (natural O).
Definition un   : Elt = (natural (S O)).
Definition singleton : Elt -> Ensf =
  [e:Elt] (add e empty).

(*  Quelques petits lemmes divers...  *)

Lemma False_imp_P : (P:Prop) False->P.
Goal.
Intros.
Absurd False; [ Red; Auto | Auto ].
Save.

Lemma equal_add : (a,b:Ensf)(e:Elt)
  <Ensf>a=b -> <Ensf>(add e a)=(add e b).
Goal.
Intros.
Apply (f_equal Ensf Ensf); Auto.
Save.

(*									*)
(*  APPARTENANCE :							*)
(*    On definit le predicat (dans x E) pour un element x et un		*)
(*    ensemble E.							*)
(*									*)

Inductive Definition dans : Elt -> Ensf -> Prop =
  dans_add1 : (x:Elt)(e:Ensf) (dans x (add x e))
| dans_add2 : (x,y:Elt)(e:Ensf) (dans x e) -> (dans x (add y e)).
Hint dans_add1 dans_add2.

Definition dans_inv : Elt -> Ensf -> Prop 
   = [G:Elt][F:Ensf]
     (<[s:sortes]Prop>Match F with
            False
            [x:Elt][Hx:Prop][e:Ensf][He:Prop]
                (<Elt>x=G \/ (dans G e))

            [n:nat]False
            [a:Elt][Ha:Prop][b:Elt][Hb:Prop]False
	    [e:Ensf][He:Prop]False
	    [w:Word][Hw:Prop]False

	    False
	    [a:Elt][Ha:Prop][w:Word][Hw:Prop]False
     ).

Lemma dans_INV : (E:Elt)(F:Ensf)(dans E F)->(dans_inv E F).
Goal.
Induction 1; Red; Intros; Auto.
Save.

(*  Quelques resultats concernant l'appartenance...  *)

Lemma dans_add : (x,y:Elt)(e:Ensf)
   (dans x (add y e)) -> ( (<Elt>y=x) \/ (dans x e) ).
Goal.
Intros x y e H.
Cut (dans_inv x (add y e)).
2:Apply dans_INV; Auto.
Unfold dans_inv.
Auto.
Save.

Lemma dans_add_contr : (x,y:Elt)(e:Ensf)
  ~(<Elt>y=x) -> ~(dans x e) -> ~(dans x (add y e)).
Goal.
Intros; Red; Intro.
Absurd ( <Elt>y=x \/ (dans x e)).
2:Apply dans_add; Auto.
Red.
Intro.
Elim H2; Auto.
Save.

Lemma empty_empty : (E:Elt)~(dans E empty).
Proof [E:Elt](dans_INV E empty).
Hint empty_empty.

Lemma dans_empty_imp_P : (x:Elt)(P:Prop) (dans x empty)->P.
Goal.
Intros.
Cut False.
Apply False_imp_P.
Cut (dans x empty); Auto.
Change ~(dans x empty); Auto.
Cut (dans x empty); Auto.
Change ~(dans x empty); Auto.
Save.

Lemma singl2 : (x:Elt) (dans x (singleton x)).
Goal.
Unfold singleton.
Auto.
Save.
Hint singl2.

Lemma singl2_inv : (x,e:Elt)
  (dans x (singleton e)) -> <Elt>x=e.
Goal.
Unfold singleton.
Intros x e H.
Cut ( <Elt>e=x \/ (dans x empty) ).
2:Apply dans_add; Auto.
Intro H0; Elim H0; Clear H0.
Auto.
Intro.
Apply (dans_empty_imp_P x); Auto.
Save.
Hint singl2_inv.

(*									*)
(*  UNION :								*)
(*    On definit ici l'union de 2 ensembles. On remarquera qu'un 	*)
(*    peut apparaitre plusieurs fois dans un ensemble, et que c'est     *)
(*    pour cela que l'on utilise 					*)
(*      union (add x E) F -> add x (union E F)				*)
(*    directement.							*)
(*									*)

Definition union : Ensf -> Ensf -> Ensf =
  [A,B:Ensf]
  (<[s:sortes]Ensf>Match A with
        B
        [x:Elt][Hx:Ensf][e:Ensf][He:Ensf] (add x He)

        [n:nat] empty
        [a:Elt][Ha:Ensf][b:Elt][Hb:Ensf] empty
	[e:Ensf][He:Ensf]empty
	[w:Word][Hw:Ensf]empty

	empty
	[a:Elt][Ha:Ensf][w:Word][Hw:Ensf]empty
  ).

Lemma union_a_empty : (a:Ensf) <Ensf>a=(union a empty).
Goal.
Intro a; Pattern a; Apply induction_ensf.
Apply refl_equal.
Intros a0 b H.
Replace (union (add a0 b) empty) with (add a0 (union b empty)).
2:Auto.
Rewrite <- H.
Auto.
Save.
Hint union_a_empty.

Lemma dans_union : (x:Elt)(a,b:Ensf)
  (dans x (union a b)) -> ( (dans x a) \/ (dans x b) ).
Goal.
Intros x a; Pattern a; Apply induction_ensf.
Intro b.
Replace (union empty b) with b; Auto.
Intros a0 b H b0.
Replace (union (add a0 b) b0) with (add a0 (union b b0)).
2:Auto.
Intro.
Cut ( <Elt>a0=x \/ (dans x (union b b0)) ).
2:Apply dans_add; Auto.
Intro H1; Elim H1.
Intro; Left.
Rewrite H2; Auto.
Intro.
Cut ((dans x b) \/ (dans x b0)); Auto.
Intro H3; Elim H3.
Intro; Left; Auto.
Intro; Right; Auto.
Save.
Hint dans_union.

Lemma union_g : (x:Elt)(a,b:Ensf)
  (dans x a) -> (dans x (union a b)).
Goal.
Intros x a; Pattern a; Apply induction_ensf.
Intros.
Apply (dans_empty_imp_P x); Auto.
Intros a0 b H b0.
Replace (union (add a0 b) b0) with (add a0 (union b b0)); Auto.
Intro.
Cut ( (<Elt>a0=x) \/ (dans x b) ).
2:Apply dans_add; Auto.
Intro H1; Elim H1; Clear H1.
Intro H1.
Rewrite H1; Auto.
Auto.
Save.
Hint union_g.

Lemma union_d : (x:Elt)(a,b:Ensf)
  (dans x b) -> (dans x (union a b)).
Goal.
Intros x a; Pattern a; Apply induction_ensf.
Intros.
Replace (union empty b) with b; Auto.
Intros a0 b H b0.
Replace (union (add a0 b) b0) with (add a0 (union b b0)); Auto.
Save.
Hint union_d.

Lemma dans_union_inv : (x:Elt)(a,b:Ensf)
  ( (dans x a) \/ (dans x b) ) -> (dans x (union a b)).
Goal.
Intros x a b H; Elim H; Clear H.
Intro; Apply union_g; Auto.
Intro; Apply union_d; Auto.
Save.

(*									*)
(*  first et second renvoient repsectivement le premier et le deuxieme 	*)
(*  element d'un couple.						*)
(*									*)
  
Definition first : Elt -> Elt =
  [x:Elt]
    (<[s:sortes]Elt>Match x with
	zero
	[x:Elt][Hx:Elt][e:Ensf][He:Elt]zero

	[n:nat]zero
	[a:Elt][Ha:Elt][b:Elt][Hb:Elt]a
	[e:Ensf][He:Elt]zero
	[w:Word][Hw:Elt]zero

	zero
	[a:Elt][Ha:Elt][w:Word][Hw:Elt]zero

    ).

Definition second : Elt -> Elt =
  [x:Elt]
    (<[s:sortes]Elt>Match x with
	zero
	[x:Elt][Hx:Elt][e:Ensf][He:Elt]zero

	[n:nat]zero
	[a:Elt][Ha:Elt][b:Elt][Hb:Elt]b
	[e:Ensf][He:Elt]zero
	[w:Word][Hw:Elt]zero

	zero
	[a:Elt][Ha:Elt][w:Word][Hw:Elt]zero

    ).

(* Grace a first et second on recupere facilement le lemme suivant : 	*)

Lemma equal_couple : (x,y,z,t:Elt)
  <Elt>(couple x y)=(couple z t) -> ( <Elt>x=z /\ <Elt>y=t ).
Goal.
Intros x y z t H.
Split.
Cut <Elt>(first (couple x y))=(first (couple z t)); Auto.
Apply (f_equal Elt Elt); Auto.
Cut <Elt>(second (couple x y))=(second (couple z t)); Auto.
Apply (f_equal Elt Elt); Auto.
Save.

Lemma couple_couple_inv1 : (a,b,c,d:Elt)
   (<Elt>(couple a c)=(couple b d)) -> (<Elt>a=b).
Goal.
Intros.
Cut ( <Elt>a=b /\ <Elt>c=d ).
Intro H0; Elim H0; Auto.
Apply equal_couple; Auto.
Save.
 
Lemma couple_couple_inv2 : (a,b,c,d:Elt)(<Elt>(couple a c)=(couple b d))->
(<Elt>c=d).
Goal.
Intros.
Cut ( <Elt>a=b /\ <Elt>c=d ).
Intro H0; Elim H0; Auto.
Apply equal_couple; Auto.
Save.

(*									*)
(*  PRODUIT CARTESIEN 							*)
(*    On definit ici le produit cartesien de 2 ensembles. Pour cela     *)
(*    on commence par definir (singleprod x E) qui est l'ensemble	*)
(*    des couples (x,y) pour y dans E, puis (prodcart E F) qui est	*)
(*    l'union des (singleprod x F) pour x dans E.			*)
(*									*)

Definition singleprod : Elt -> Ensf -> Ensf =
  [x:Elt][A:Ensf]
  (<[s:sortes]Ensf>Match A with
        empty
        [y:Elt][Hy:Ensf][e:Ensf][He:Ensf] (add (couple x y) He) 

        [n:nat] empty                            
        [a:Elt][Ha:Ensf][b:Elt][Hb:Ensf] empty   
	[e:Ensf][He:Ensf]empty
	[w:Word][Hw:Ensf]empty

	empty
	[a:Elt][Ha:Ensf][w:Word][Hw:Ensf]empty
  ).

Definition prodcart : Ensf -> Ensf -> Ensf =
  [A,B:Ensf]
  (<[s:sortes]Ensf>Match A with
	empty
        [x:Elt][Hx:Ensf][e:Ensf][He:Ensf] (union (singleprod x B) He)

        [n:nat] empty
        [a:Elt][Ha:Ensf][b:Elt][Hb:Ensf] empty
	[e:Ensf][He:Ensf]empty
	[w:Word][Hw:Ensf]empty

	empty
	[a:Elt][Ha:Ensf][w:Word][Hw:Ensf]empty
  ).

(*  On montre en premier que si (x,y) est dans (singleprod x0 b) alors	*)
(*  on a x=x0 et y dans b						*)

Lemma dans_singleprod : (x,y,x0:Elt)(b:Ensf)
  (dans (couple x y) (singleprod x0 b)) -> ( (<Elt>x=x0) /\ (dans y b) ).
Goal.
Intros x y x0 b; Pattern b; Apply induction_ensf.
Replace (singleprod x0 empty) with empty; Auto.
Intro.
Apply (dans_empty_imp_P (couple x y)); Auto.
Intros a b0 H.
Replace (singleprod x0 (add a b0)) with (add (couple x0 a) (singleprod x0 b0)); Auto.
Intro.
Cut ( <Elt>(couple x0 a)=(couple x y) \/ (dans (couple x y) (singleprod x0 b0)) ).
2:Apply dans_add; Auto.
Intro H1; Elim H1; Clear H1.
Intro.
Cut ( <Elt>x0=x /\ <Elt>a=y ).
2:Apply equal_couple; Auto.
Intro H2; Elim H2; Clear H2.
Intros.
Split; Auto.
Rewrite H3; Auto.
Intro.
Cut ( <Elt>x=x0 /\ (dans y b0) ); Auto.
Intro H2; Elim H2; Clear H2.
Intros.
Split; Auto.
Save.

(*  On peut ensuite en deduire que si (x,y) est dans AxB alors 		*)
(*  x est dans A et y est dans B.					*)
 
Lemma coupl2 : (x,y:Elt)(a,b:Ensf)
  (dans (couple x y) (prodcart a b)) -> ((dans x a) /\ (dans y b)).
Goal.
Intros x y a; Pattern a; Apply induction_ensf.
Intro b.
Replace (prodcart empty b) with empty; Auto.
Intro.
Apply (dans_empty_imp_P (couple x y)); Auto.
Intros a0 b H b0.
Replace (prodcart (add a0 b) b0) with (union (singleprod a0 b0) (prodcart b b0)); Auto.
Intro.
Cut ( (dans (couple x y) (singleprod a0 b0)) \/ (dans (couple x y) (prodcart b b0)) ).
2:Apply dans_union; Auto.
Intro H1; Elim H1; Clear H1.
Intro.
Cut ( <Elt>x=a0 /\ (dans y b0) ).
2:Apply dans_singleprod; Auto.
Intro H2; Elim H2; Clear H2.
Intros.
Rewrite H2.
Split; Auto.
Intro.
Cut ( (dans x b) /\ (dans y b0) ); Auto.
Intro H2; Elim H2; Clear H2.
Intros.
Split; Auto.
Save.

(*  Plus facile : l'inverse...						*)

Lemma dans_single : (x,y:Elt)(a:Ensf)
  (dans y a) -> (dans (couple x y) (singleprod x a)).
Goal.
Intros x y a; Pattern a; Apply induction_ensf.
Intro.
Apply (dans_empty_imp_P y); Auto.
Intros a0 b H H1.
Cut ( <Elt>a0=y \/ (dans y b) ).
2:Apply dans_add; Auto.
Intro H2; Elim H2; Clear H2.
Intro.
Replace (singleprod x (add a0 b)) with (add (couple x a0) (singleprod x b)); Auto.
Rewrite H0; Auto.
Replace (singleprod x (add a0 b)) with (add (couple x a0) (singleprod x b)); Auto; Auto.
Save.

Lemma coupl2_inv : (x,y:Elt)(a,b:Ensf)
  (dans x a) -> (dans y b) -> (dans (couple x y) (prodcart a b)).
Goal.
Intros x y a; Pattern a; Apply induction_ensf.
Intros b H.
Apply (dans_empty_imp_P x); Auto.
Intros a0 b H b0 H0.
Cut (<Elt>a0=x \/ (dans x b) ).
2:Apply dans_add; Auto.
Replace (prodcart (add a0 b) b0) with (union (singleprod a0 b0) (prodcart b b0)); Auto.
Intro H1; Elim H1; Clear H1.
Intros H1 H2.
Apply dans_union_inv.
Left.
Rewrite H1.
Apply dans_single; Auto.
Intros H1 H2.
Apply dans_union_inv.
Right.
Auto.
Save.
Hint coupl2_inv.

(*  De meme on commence ici par monter que si x est dans		*)
(*  (singleprod x0 b) alors x est de la forme (x0,y) avec y dans b	*)

Lemma dans_singleprod2 : (x,x0:Elt)(b:Ensf)
  (dans x (singleprod x0 b)) -> (<Elt>Ex ([y:Elt](
    <Elt>x=(couple x0 y) /\ (dans y b)
  ))).
Goal.
Intros x x0 b; Pattern b; Apply induction_ensf.
Intro.
Apply (dans_empty_imp_P x); Auto.
Intros a b0 H.
Replace (singleprod x0 (add a b0)) with (add (couple x0 a) (singleprod x0 b0)); Auto.
Intro.
Cut ( <Elt>(couple x0 a)=x \/ (dans x (singleprod x0 b0)) ).
2:Apply dans_add; Auto.
Intro H1; Elim H1; Clear H1.
Intro.
Exists a; Auto.
Intro.
Cut (<Elt>Ex ([y:Elt](<Elt>x=(couple x0 y) /\ (dans y b0)))); Auto.
Intro H2; Elim H2; Clear H2.
Intros.
Exists x1.
Elim H2; Clear H2.
Intros.
Split; Auto.
Save.

(*  On peut ensuite en deduire que si x est dans AxB alors x est de la	*)
(*  forme (x1,x2) avec x1 dans A et x2 dans B.				*)

Lemma coupl3 : (a,b:Ensf)(x:Elt) 
   (dans x (prodcart a b)) ->
     (<Elt>Ex ([x1:Elt] 
       (<Elt>Ex ([x2:Elt]
         (   (dans x1 a)
          /\ (dans x2 b)
          /\ (<Elt>x=(couple x1 x2))))))).
Goal.
Intro a ; Pattern a; Apply induction_ensf.
Intro b.
Replace (prodcart empty b) with empty; Auto.
Intros x H.
Apply (dans_empty_imp_P x); Auto.
Intros a0 b H b0 x.
Replace (prodcart (add a0 b) b0) with (union (singleprod a0 b0) (prodcart b b0)); Auto.
Intro.
Cut ( (dans x (singleprod a0 b0)) \/ (dans x (prodcart b b0)) ); Auto.
Intro H1; Elim H1; Clear H1.
Intro.
Cut (<Elt>Ex ([y:Elt](<Elt>x=(couple a0 y) /\ (dans y b0)))).
2:Apply dans_singleprod2; Auto.
Intro H2; Elim H2; Clear H2.
Intros x0 H2.
Exists a0.
Exists x0.
Elim H2; Clear H2.
Intros.
Split; Auto.
Intro.
Cut  (<Elt>Ex ([x1:Elt] (<Elt>Ex ([x2:Elt]
         (   (dans x1 b)
          /\ (dans x2 b0)
          /\ (<Elt>x=(couple x1 x2))))))); Auto.
Intro H2; Elim H2; Clear H2.
Intros x0 H2; Elim H2; Clear H2.
Intros x1 H2; Elim H2; Clear H2.
Intros H2 H3; Elim H3; Clear H3.
Intros H4 H5.
Exists x0.
Exists x1.
Split; Auto.
Save.

(*									*)
(*  INCLUSION  								*)
(*    On definit le predicat (inclus E F) par (dans x E)->(dans x F).	*)
(*    On montre ensuite facilement les resultats suivants :		*)
(*    -- inclus empty A							*)
(*    -- inclus A A							*)
(*    -- (inclus a b)->(inclus c d)					*)
(*		->(inclus (prodcart a c) (prodcart b d)			*)
(*									*)

Definition inclus : Ensf -> Ensf -> Prop =
  [A,B:Ensf]
    (x:Elt) (dans x A) -> (dans x B).

Lemma empty_inclus : (x : Ensf) (inclus empty x).
Goal.
Unfold inclus; Intros.
Absurd (dans x0 empty); Auto.
Save.
Hint empty_inclus.

Lemma refl_inclus : (x : Ensf)(inclus x x ).
Goal.
Unfold inclus; Auto.
Save.
Hint refl_inclus.

Lemma inclus_trans : (a,b,c:Ensf)
  (inclus a b) -> (inclus b c) -> (inclus a c).
Goal.
Unfold inclus.
Auto.
Save.

Lemma cart_inclus : (a,b,c,d : Ensf)
    (inclus a b) -> (inclus c d) -> (inclus (prodcart a c ) (prodcart b d)).
Goal.
Unfold inclus.
Intros.
Cut (<Elt>Ex ([x1:Elt](<Elt>Ex ([x2:Elt](
	   (dans x1 a)
	/\ (dans x2 c)
	/\ <Elt>x=(couple x1 x2)
    ))))).
2:Apply coupl3; Auto.
Intro H2; Elim H2; Clear H2.
Intros x1 H2; Elim H2; Clear H2.
Intros x2 H2; Elim H2; Clear H2.
Intros H2 H3; Elim H3; Clear H3.
Intros H3 H4.
Rewrite H4.
Auto.
Save.

Lemma inclus_add : (a,b:Ensf)(y:Elt)
  (inclus a b) -> (inclus a (add y b)).
Goal.
Unfold inclus.
Auto.
Save.
Hint inclus_add.

Lemma singl_inclus_add : (e:Elt)(a:Ensf)
  (inclus (singleton e) (add e a)).
Goal.
Unfold inclus.
Intros e a x H.
Cut <Elt>x=e; Auto.
Intro H0.
Rewrite H0; Auto.
Save.
Hint singl_inclus_add.

Lemma inclus_singl : (e:Elt)(a:Ensf)
  (inclus (singleton e) a) -> (dans e a).
Goal.
Unfold inclus.
Auto.
Save.

Lemma add_inclus : (x:Elt)(a,b:Ensf)
  (dans x b) -> (inclus a b) -> (inclus (add x a) b).
Goal.
Unfold inclus.
Intros.
Cut ( <Elt>x=x0 \/ (dans x0 a) ).
2:Apply dans_add; Auto.
Intro H2; Elim H2; Clear H2.
Intro H2; Rewrite <- H2; Auto.
Auto.
Save.
Hint add_inclus.

Lemma dans_trans : (x:Elt)(a,b:Ensf)
  (dans x a) -> (inclus a b) -> (dans x b).
Goal.
Unfold inclus.
Auto.
Save.

Lemma union_inclus : (a,b,c:Ensf)
  (inclus a c) -> (inclus b c) -> (inclus (union a b) c).
Goal.
Unfold inclus.
Intros.
Cut ( (dans x a) \/ (dans x b) ); Auto.
Intro H2; Elim H2; Auto.
Save.
Hint union_inclus.

Lemma inclus_g : (a,b:Ensf)
  (inclus a (union a b)).
Goal.
Intros.
Unfold inclus.
Intros.
Apply union_g; Auto.
Save.

Lemma inclus_d : (a,b:Ensf)
  (inclus b (union a b)).
Goal.
Intros.
Unfold inclus.
Intros.
Apply union_d; Auto.
Save.

Lemma inclus_g2 : (A,B,C:Ensf)
  (inclus A B) -> (inclus A (union B C)).
Goal.
Unfold inclus.
Intros.
Apply union_g; Auto.
Save.
Hint inclus_g2.

Lemma inclus_d2 : (A,B,C:Ensf)
  (inclus A C) -> (inclus A (union B C)).
Goal.
Unfold inclus.
Intros.
Apply union_d; Auto.
Save.
Hint inclus_d2.
 


(*									*)
(*  INTERSECTION 							*)
(*   L'intersection de 2 ensembles est definie comme un predicat sur    *)
(*   3 ensembles A B C en disant que C est l'intresection de A et B     *)
(*   si C est le plus grand ensemble inclus dans A et dans B		*)
(*									*)

Definition inter : Ensf -> Ensf -> Ensf -> Prop =
  [A,B:Ensf][C:Ensf]
    (inclus C A)
  /\(inclus C B)
  /\((x:Elt) (dans x A)->(dans x B)->(dans x C)).


Lemma union_inter : (a,b,c:Ensf)
  (inter a b empty) -> (inter a c empty)
   -> (inter a (union b c) empty).
Goal.
Unfold inter.
Intros.
Elim H0; Clear H0.
Intros H0 H1; Elim H1; Clear H1; Intros H1 H2.
Elim H; Clear H.
Intros H3 H4; Elim H4; Clear H4; Intros H4 H5.
Split; Auto.
Split.
Apply empty_inclus.
Intros.
Cut ( (dans x b) \/ (dans x c) ); Auto.
Intro H7; Elim H7; Auto.
Save.

Lemma inter_union : (A,B,C:Ensf)
  (inter A C empty) -> (inter B C empty)
   -> (inter (union A B) C empty).
Goal.
Unfold inter.
Intros.
Elim H0; Clear H0.
Intros H0 H1; Elim H1; Clear H1; Intros H1 H2.
Elim H; Clear H.
Intros H3 H4; Elim H4; Clear H4; Intros H4 H5.
Split; Auto.
Split; Auto.
Intros.
Cut ( (dans x A) \/ (dans x B) ); Auto.
Intro H7; Elim H7; Auto.
Save.

Lemma inter_dans : (A,B:Ensf)(x:Elt)
  (inter A B empty)
  -> (dans x A)
  -> ~(dans x B).
Goal.
Unfold inter.
Intros.
Elim H; Clear H; Intros H Ht; Elim Ht; Clear Ht; Intros H1 H2.
Red; Intro.
Cut (dans x empty); Auto.
Intro.
Apply dans_empty_imp_P with x; Auto.
Save.

Lemma sym_inter : (A,B,C:Ensf)
  (inter A B C)
  -> (inter B A C).
Goal.
Unfold inter.
Intros.
Elim H; Clear H; Intros H Ht; Elim Ht; Clear Ht; Intros H0 H1.
Auto.
Save.

(*									*)
(*  MAP "a la CAML"							*)
(*    On definit une fonction map qui applique une fonction a tous les  *)
(*    elements d'un ensemble et renvoie l'ensemble des resultats	*)
(*    Ceci permet, entre autres, de definir facilement l'union		*)
(*    disjointe (Voir ci-dessous)					*)
(*									*)

Definition map : (Elt->Elt) -> Ensf -> Ensf =
  [f:(Elt->Elt)][e:Ensf]
    (<[s:sortes]Ensf>Match e with
        empty
        [y:Elt][Hy:Ensf][e:Ensf][He:Ensf] (add (f y) He) 

        [n:nat] empty                            
        [a:Elt][Ha:Ensf][b:Elt][Hb:Ensf] empty   
	[e:Ensf][He:Ensf]empty
	[w:Word][Hw:Ensf]empty

	empty
	[a:Elt][Ha:Ensf][w:Word][Hw:Ensf]empty
    ).

(*									*)
(*  On montre ici le resultat general suivant :				*)
(*    (dans  x (map f A)) -> il existe y dans A tel que x=f y		*)
(*									*)

Lemma dans_map : (f:Elt->Elt)(a:Ensf)(x:Elt)
  (dans x (map f a)) -> (<Elt>Ex ([y:Elt] (
    (dans y a) /\ (<Elt>x=(f y))	
  ))).
Goal.
Intros f a; Pattern a; Apply induction_ensf.
Replace (map f empty) with empty; Auto.
Intros x H.
Cut False.
Apply False_imp_P.
Cut (dans x empty).
2:Auto.
Change ~(dans x empty).
Auto.
Cut (dans x empty).
2:Auto.
Change ~(dans x empty).
Auto.
Intros a0 b H x.
Replace (map f (add a0 b)) with (add (f a0) (map f b));Auto.
Intro.
Cut ( <Elt>(f a0)=x \/ (dans x (map f b)) ).
2:Apply dans_add; Auto.
Intro H1; Elim H1; Clear H1.
Intro; Exists a0; Auto.
Intro.
Cut (<Elt>Ex([y:Elt](dans y b)/\(<Elt>x=(f y)))).
Intro H2; Elim H2; Clear H2.
2:Auto.
Intros x0 H2; Elim H2; Clear H2.
Intros.
Exists x0.
Split; Auto.
Save.

Lemma dans_map_inv : (f:Elt->Elt)(x:Elt)(a:Ensf)
  (dans x a) -> (dans (f x) (map f a)).
Goal.
Intros f x a; Pattern a; Apply induction_ensf.
Intro.
Apply (dans_empty_imp_P x); Auto.
Intros a0 b H.
Replace (map f (add a0 b)) with (add (f a0) (map f b)); Auto.
Intro H1.
Cut  (<Elt>a0=x \/ (dans x b) ).
2:Apply dans_add; Auto.
Intro H2; Elim H2; Clear H2.
Intro.
Rewrite H0; Auto.
Auto.
Save.

Lemma map_union : (f:Elt->Elt)(a,b:Ensf)
  (<Ensf>(union (map f a) (map f b)) = (map f (union a b)) ).
Goal.
Intros f a; Pattern a; Apply induction_ensf.
Auto.
Intros.
Replace (map f (add a0 b)) with (add (f a0) (map f b));Auto.
Replace (union (add a0 b) b0) with (add a0 (union b b0)); Auto.
Replace (map f (add a0 (union b b0))) with (add (f a0) (map f (union b b0))); Auto.
Replace (union (add (f a0) (map f b)) (map f b0)) with (add (f a0) (union (map f b) (map f b0))); Auto.
Save.
Hint map_union.

Lemma dans_map_trans : (x:Elt)(f:Elt->Elt)(a,b:Ensf)
  (dans x (map f a))
  -> (inclus a b)
  -> (dans x (map f b)).
Goal.
Intros.
Cut (<Elt>Ex ([y:Elt]( (dans y a) /\ <Elt>x=(f y) ))).
2:Apply dans_map; Auto. 
Intro Ht; Elim Ht; Clear Ht.
Intros y Ht; Elim Ht; Clear Ht.
Intros.
Cut (dans y b).
2:Apply dans_trans with a; Auto.
Intro.
Rewrite H2.
Apply dans_map_inv; Auto.
Save.

Lemma map_egal : (f,g:Elt->Elt)(E:Ensf)
  ((x:Elt)(dans x E)->(<Elt>(f x)=(g x)))
     -> (<Ensf>(map f E)=(map g E)).
Goal.
Intros f g E; Pattern E; Apply induction_ensf.
Auto.
Intros.
Replace (map f (add a b)) with (add (f a) (map f b)); Auto.
Replace (map g (add a b)) with (add (g a) (map g b)); Auto.
Save.


Lemma map_inclus : (a,b:Ensf)(f:Elt->Elt)
  (inclus a b) -> (inclus (map f a) (map f b)).
Goal.
Unfold inclus.
Intros.
Cut (<Elt>Ex ([y:Elt]( (dans y a) /\ <Elt>x=(f y) ))).
2:Apply dans_map; Auto.
Intro Ht; Elim Ht; Clear Ht; Intros y Ht; Elim Ht; Clear Ht; Intros.
Cut (dans y b); Auto.
Intro.
Replace x with (f y); Auto.
Apply dans_map_inv; Auto.
Save.


(*									*)
(*  UNION DISJOINTE 							*)
(*    L'union disjointe de A et B est definie comme l'ensemble des 	*)
(*    des couples (x,zero) pour x dans A et (x,un) pour x dans B	*)
(*    Pour cela on definit 2 fonctions injgauche : x->(x,zero) et	*)
(*    injdroite : x->(x,un) et on fait l'union de (map injgauche A)	*)
(*    et de (map injdroite B).						*)
(*									*)

Definition injgauche : Elt -> Elt =
  [e:Elt](couple e zero).

Definition injdroite : Elt -> Elt =
  [e:Elt](couple e un).

Definition union_disj : Ensf -> Ensf -> Ensf =
  [e,f:Ensf](union (map injgauche e) (map injdroite f)).


Lemma  dans_map_injg : (e:Ensf)(x:Elt)
  (dans x (map injgauche e)) -> (dans (first x) e).
Goal.
Intros.
Cut (<Elt>Ex ([y:Elt]( (dans y e) /\ <Elt>x=(injgauche y) ))).
2:Apply dans_map; Auto.
Intro Ht; Elim Ht; Clear Ht.
Intros y Ht; Elim Ht; Clear Ht.
Intros.  
Unfold injgauche in H1.
Replace (first x) with y; Auto.
Symmetry.
Replace y with (first (couple y zero)); Auto.
Apply (f_equal Elt Elt); Auto.
Save.

Lemma dans_map_injd : (e:Ensf)(x:Elt)
  (dans x (map injdroite e)) -> (dans (first x) e).
Goal.
Intros.
Cut (<Elt>Ex ([y:Elt]( (dans y e) /\ <Elt>x=(injdroite y) ))).
2:Apply dans_map; Auto.
Intro Ht; Elim Ht; Clear Ht.
Intros y Ht; Elim Ht; Clear Ht.
Intros.  
Unfold injdroite in H1.
Replace (first x) with y; Auto.
Symmetry.
Replace y with (first (couple y un)); Auto.
Apply (f_equal Elt Elt); Auto.
Save.

Lemma absurd_injg_injd : (x:Elt)(e,f:Ensf)
  (dans x (map injgauche e)) -> ~(dans x (map injdroite f)).
Goal.
Intros.
Cut (<Elt>Ex ([y:Elt]( (dans y e) /\ <Elt>x=(injgauche y) ))).
2:Apply dans_map; Auto.
Intro Ht; Elim Ht; Clear Ht.
Intros y Ht; Elim Ht; Clear Ht.
Intros.  
Red.
Intro.
Cut (<Elt>Ex ([y':Elt]( (dans y' f) /\ <Elt>x=(injdroite y') ))).
2:Apply dans_map; Auto.
Intro Ht; Elim Ht; Clear Ht.
Intros y' Ht; Elim Ht; Clear Ht.
Intros.  
Absurd <Elt>zero=un.
Unfold zero.
Unfold un.
Red.
Intro.
Cut <nat>O=(S O).
2:Replace O with (natural_inv (natural O)); Auto.
2:Replace (S O) with (natural_inv (natural(S O))); Auto.
2:Apply (f_equal Elt nat); Auto.
Change ~(<nat>O=(S O)).
Apply O_S.

Unfold injdroite in H4.
Unfold injgauche in H1.
Replace zero with (second (couple y zero)); Auto.
Replace un with (second (couple y' un)); Auto.
Rewrite <- H4.
Rewrite <- H1.
Auto.
Save.

(*									*)
(*  On montre ici que si x est dans l'union disjointe de A et B alors	*)
(*  x est soit de la forme (injgauche y) avec y dans A, soit de la 	*)
(*  forme (injdroite y) avec y dans B					*)
(*									*)

Lemma union_disj1 : (x:Elt)(a,b:Ensf)
  (dans x (union_disj a b)) -> (
      (<Elt>Ex ([y:Elt]  ( (dans y a) /\ (<Elt>x=(injgauche y)) )))
   \/ (<Elt>Ex ([y:Elt]  ( (dans y b) /\ (<Elt>x=(injdroite y)) )))
  ).
Goal.
Unfold union_disj.
Intros.
Cut ( (dans x (map injgauche a)) \/ (dans x (map injdroite b)) ).
2:Auto.
Intro H0; Elim H0; Clear H0.
Intro; Left.
Apply dans_map; Auto.
Intro; Right.
Apply dans_map; Auto.
Save.

Lemma union_disj_d : (x:Elt)(a,b:Ensf)
  (dans x b) -> (dans (injdroite x) (union_disj a b)).
Goal.
Intros.
Unfold union_disj.
Apply union_d.
Apply dans_map_inv.
Auto.
Save.

Lemma union_disj_g : (x:Elt)(a,b:Ensf)
  (dans x a) -> (dans (injgauche x) (union_disj a b)).
Goal.
Intros.
Unfold union_disj.
Apply union_g.
Apply dans_map_inv.
Auto.
Save.

Lemma inclus_union_disj : (a,b,c,d:Ensf)
  (inclus a c) -> (inclus b d) 
    -> (inclus (union_disj a b) (union_disj c d)).
Goal.
Unfold inclus.
Intros.
Unfold union_disj in H1.
Cut ( (dans x (map injgauche a)) \/ (dans x (map injdroite b)) ); Auto.
Intro Ht; Elim Ht; Clear Ht.

Intro.
Cut (<Elt>Ex ([y:Elt]( (dans y a) /\ <Elt>x=(injgauche y) ))).
2:Apply dans_map; Auto.
Intro Ht; Elim Ht; Clear Ht.
Intros y Ht; Elim Ht; Clear Ht; Intros H3 H4.
Cut (dans y c); Auto.
Intro.
Unfold union_disj.
Apply union_g.
Rewrite H4.
Apply dans_map_inv; Auto.

Intro.
Cut (<Elt>Ex ([y:Elt]( (dans y b) /\ <Elt>x=(injdroite y) ))).
2:Apply dans_map; Auto.
Intro Ht; Elim Ht; Clear Ht.
Intros y Ht; Elim Ht; Clear Ht; Intros H3 H4.
Cut (dans y d); Auto.
Intro.
Unfold union_disj.
Apply union_d.
Rewrite H4.
Apply dans_map_inv; Auto.
Save.

(* 									*)
(*  Resultats n'ayant rien a voir avec les ensembles finis mais n'ayant	*)
(*  pas de place dans un fichier particulier.				*)
(*									*)

Lemma pair_equal : (A,B:Set)(x,x':A)(y,y':B)
  (<A>x=x') -> (<B>y=y') -> (<A*B>(<A,B>(x,y))=(<A,B>(x',y'))).
Goal.
Intros A B x x' y y' X Y.
Rewrite X.
Rewrite Y.
Apply refl_equal.
Save.
Hint pair_equal.

Provide Ensf.
