(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                               xgo.ml                                     *)
(****************************************************************************)

#open "std";;
#open "initial";;
#open "coqtoplevel";;
#open "coq";;
#open "machine";;
#open "compile";;
#open "fmlenv";;
#open "lexer";;
#open "interface";;
#open "inter_top";;
#open "version";;
#open "rt";;
#open "sys";;

pack_border:=0;;
pack_inter:=1;;
pack_band:=0;;
button_border:=1;;
button_font:="*-helvetica-bold-r-*--12-*";;
text_font:="7x14";; (* text_font:="*-courier-medium-r-*-12-*"; *)
title_font:="*-helvetica-bold-o-*-12-*";;

type start_state =
  Undefined
| SaveState of string
| RestoreState of string
;;
let start_state = ref Undefined;;

let usage() =
  prerr_string "Usage: "; prerr_string command_line.(0);
  prerr_endline " [options]";
  prerr_endline "  -x : boots version with X interface";
  prerr_endline "  -bw: black & white - only valid with -x";
  prerr_endline "  -gray: adapted gray - only valid with -x";
  prerr_endline "  -os file: load initial then save state into file";
  prerr_endline "  -is file: read initial state from file";
  prerr_endline " -I dir: add dir to loadpath";
  prerr_endline " -q : no autoload of ~/.coqrc";
  exit 1
;;

let get_arg i =
  if i >= vect_length command_line then usage()
  else command_line.(i)
;;

let load_rc = ref true;;

arg_loop 1
where rec arg_loop i =
  if i < vect_length command_line then (
    arg_loop (
      match command_line.(i) with
        "-bw" ->
          COLOR := false;
          i+1
      | "-gray" ->
          GRAYSCALE := true;
          i+1
      | "-os" ->
          start_state := SaveState(get_arg(i+1));
          i+2
      | "-is" ->
          start_state := RestoreState(get_arg(i+1));
          i+2
      | "-I" -> add_path (get_arg(i+1));i+2
      | "-q" -> (load_rc := false);i+1
      | x ->
          usage()
    )
  )
;;

try
  match !start_state with
    SaveState x ->
      load_initial();
      extern_state x
  | RestoreState x ->
      (intern_state x;tactics__restore_state "Initial")
  | _ ->
      ()
with
  Sys_error msg ->
    prerr_string "OS Error: "; prerr_endline msg; exit 2
| UserError s ->
    prerr_string "Error : "; prerr_endline s; exit 2
;;

type expr =
  EApp of expr * expr
| EIdent of string
| EUnit
| EString of string
| EFun of (expr -> expr)
| ENone
;;

exception IllTyped of string;;
let illt t = raise(IllTyped t);;

let valtab =
  let t = make_vect 11 [] in
  do_list (fun (x,v) -> hash_add_assoc (x,v) t) [
    "go",  EFun(fun EUnit   -> go(); EUnit    | _ -> illt "unit");
    "fml", EFun(fun EUnit   -> fml(); EUnit   | _ -> illt "unit");
    "X",   EFun(fun EUnit   -> X(); EUnit     | _ -> illt "unit");
    "goX", EFun(fun (EString x) -> goX x; EUnit   | _ -> illt "string");
    "pwd", EFun(fun EUnit   -> EString(getwd__getwd()) | _ -> illt "unit");
    "cd",  EFun(fun (EString x) -> (sys__chdir x;EString(getwd__getwd()))  | _ -> illt "string")
  ]; t
;;

let ml_keywords = ([| |] : (string * unit) list vect);;

let rec parse = function
  [<
    'Tident c;
    begin function
      [< expr e >] -> EApp(EIdent c, e)
    end r;
    'Tsemisemi
  >] -> r
| [< '_ >] -> raise Parse_error

and expr = function
  [< 'Tlparen; 'Trparen >] -> EUnit
| [< 'Tstring s >] -> EString s
;;

let printerr x =
  prerr_string ">>> "; prerr_endline x;
  prerr_endline "\
Available commands:
  go(), fml(), X(), goX \"...\", pwd(), cd \"...\", quit()
";
  ENone
;;

let rec eval = function
  EApp(f,x) ->
    begin match eval f with
      EFun v ->
        begin try v x with
          Sys_error m -> printerr m
        | IllTyped t -> printerr ("Parameter should be of type " ^ t)
        | Failure s -> printerr ("failed: " ^ s)
        | _ -> printerr "Uncaught exception"
        end
    | ENone -> ENone
    | _ -> printerr "Not a function"
    end
| EFun _ as x -> x
| EIdent x ->
    begin try hash_assoc x valtab
    with _ -> printerr ("Unbound variable: " ^ x)
    end
| ENone -> ENone
| _ -> printerr "Internal error"
;;

let print = function
  EUnit -> [< '"()" >]
| EString s -> [< '"\""; 's; '"\"" >]
| _ -> [< '"???" >]
;;

let print_eval x =
  match eval x with
    ENone -> ()
  | r ->
      print_string "- = ";
      do_stream print_string (print r);
      print_newline()
;;

let rec toplev_loop() =
  set_prompt "comm # ";
  match
    try parse (token_stream ml_keywords (stream_of_channel std_in))
    with
      Parse_failure ->
        EApp(EIdent "quit", EUnit)
    | Parse_error ->
        printerr "Syntax error";
        reset_lexer (stream_of_channel std_in); ENone
    | _ ->
        printerr "Uncaught exception in parsing"
  with
    EApp(EIdent "quit", EUnit) -> ()
  | x ->
      print_eval x;
      toplev_loop()
;;

make_silent true;;
extract_all();;
make_silent false;;
prerr_string "Welcome to Coq + X ";
prerr_endline version;;
if !load_rc then
    try load_vernacular "~/.coqrc"
    with _ -> message "Load of ~/.coqrc failed"
;;
try (go();ENone) with
Sys_error msg -> printerr msg
| IllTyped t -> printerr ("Parameter should be of type " ^ t)
| Failure s -> printerr ("failed: " ^ s)
| _ -> printerr "Uncaught exception"
;;
toplev_loop();;
prerr_endline "";;
