(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                             vernac.ml                                    *)
(****************************************************************************)

(* Concrete syntax of the mathematical vernacular MV V2.6 *)

#uninfix "THEN";; 
#uninfix "THENS";; 
#uninfix "ORELSE";;

#open "std";;
#open "initial";;
#open "extraction";;
#open "term";;
#open "machine";;
#open "tactics";;
#open "command";;
#open "pretty";;
#open "search";;
#open "lexer";;
#open "printer";;
#open "more_util";;
#open "compile";;
#open "sys";;
#open "program";;
#open "annote";;
#open "pp";;
#open "stdpp";;
#infix "o";;

let CURRENT_SUBGOAL = ref 0;;
let current_goal = function () -> !CURRENT_SUBGOAL;;
let set_current_goal n = CURRENT_SUBGOAL := n;;

let forward_load_vernacular = nref(forward : string -> unit);;
let load_vernacular = forward_load_vernacular;;

let forward_load_vernacular_from_loadpath = nref(forward : string -> unit);;
let load_vernacular_from_loadpath = forward_load_vernacular_from_loadpath;;

let forward_load_vernacular_noisily = nref(forward : string -> unit);;
let load_vernacular_noisily = forward_load_vernacular_noisily;;

let require_default id fname =
    let pl = search_packages () in
        if not (listset__memb pl id) then
        let start = timestamp() in
        (try (message ("Requiring " ^ fname);
              load_vernacular_from_loadpath.v fname;
              PPNL [< 'S"Finished requiring "; 'S fname ;
                      'S" in "; fmt_time_difference start (timestamp()) >])
         with
         Failure _ ->
         error ("Require of package " ^ id ^ " from file " ^ fname ^ " failed"));
        let pl' = search_packages () in
            if not(listset__memb pl' id) then
                error ("Require of package " ^ id ^ " failed")
;;

let REQUIRE_FN = ref [require_default];;

let require id fname = (hd !REQUIRE_FN) id fname;;
let set_require_fun fn = REQUIRE_FN := fn::(!REQUIRE_FN); ();;
let reset_require_fun () = REQUIRE_FN := tl(!REQUIRE_FN); ();;

let provide id =
    let pl = search_packages() in
        if listset__memb pl id then
            error ("Package " ^ id ^ " already provided")
        else declare_package id
;;

let print_loadpath () =
    let l = search_paths () in
        PP [< 'S"Load Path: " ;
            HOV 0 (prlist_with_sep (fun () -> [< 'FNL >]) prs l) >]
;;

let print_packages () =
    let l = search_packages() in
        PP [< 'S"Packages: ";
            HOV 0 (prlist_with_sep (fun () -> [< 'FNL >]) prs l) >]
;;

let forward_goX = nref(forward : string -> unit);;

let sup0 x = if x>=0 then x else 0;;

let TypeN = TypeC Null 
and TypeS = TypeC Pos
and PropN = PropC Null 
and PropD = PropC Data
and PropS = PropC Pos;;

let com_to_str = function RefC(s) -> s | _ -> error "Not an identifier";;

let safe_hd = function
    [] -> error "Empty tactic list"
  | x::_ -> x;;

(*
let apply_tactic e l = 
match caml_system_handler eval_syntax e with 
  dynamic (f: tactic) -> f
| dynamic (f: command -> tactic) -> f (safe_hd l)
| dynamic (f: command list -> tactic) -> f l
| dynamic (f: string -> tactic) -> f (com_to_str (safe_hd l))
| dynamic (f: string list -> tactic) ->f (map com_to_str l)
|  _ -> error "Ignored, not well-typed";;
*)
let apply_tactic e l =
  prerr_endline "<apply_tactic not written>";
  failwith "apply_tactic"
;;

(* The absurd tactic is defined in vernac because of the use of "not" *)

let absurd a = THEN (elim_type (RefC (id_of_string"False")))
                     (resolve_type (AppC(RefC(id_of_string"not"),a)));;

let contradict = THEN intros (THEN (elim_type (RefC (id_of_string "False"))) assumption);;

(**************************************************************)
(*    Introduction of conjonction, disjunction                *)
(*    existencial quantifier and equality tactics             *)
(**************************************************************)


let split_tac (_,_,cl as gl) = 
    match string_head cl with "and"  -> resolve (RefC (id_of_string"conj")) gl
                           |  "prod" -> resolve (RefC (id_of_string"pair")) gl
                           |   _     -> error "Not a predefined conjunction";;
let intros_split_tac = THEN intros split_tac;;

let right_tac (_,_,cl as gl) = 
    match string_head cl with 
         "sumbool" -> resolve (RefC (id_of_string"right")) gl
       | "sumor"   -> resolve (RefC (id_of_string"inright")) gl
       | "or"      -> resolve (RefC (id_of_string"or_intror")) gl
       | "sum"     -> resolve (RefC (id_of_string"inr")) gl
       |    _      -> error "Not a predefined disjunction";;

let intros_right_tac = THEN intros right_tac;;

let left_tac (_,_,cl as gl) = 
    match string_head cl with 
         "sumbool" -> resolve (RefC (id_of_string"left")) gl
       | "sumor"   -> resolve (RefC (id_of_string"inleft")) gl
       | "or"      -> resolve (RefC (id_of_string"or_introl")) gl
       | "sum"     -> resolve (RefC (id_of_string"inl")) gl
       |    _      -> error "Not a predefined disjunction";;
let intros_left_tac = THEN intros left_tac;;

let exists_tac c (_,_,cl as gl) = 
    match string_head cl with 
          "sig"  -> resolve_with [RefC (id_of_string"exist");c] gl
        | "sig2" -> resolve_with [RefC (id_of_string"exist2");c] gl
        | "sigS"  -> resolve_with [RefC (id_of_string"existS");c] gl
        | "sigS2" -> resolve_with [RefC (id_of_string"existS2");c] gl
        | "ex"   -> resolve_with [RefC (id_of_string"ex_intro");c] gl
        | "ex2"  -> resolve_with [RefC (id_of_string"ex_intro2");c] gl
        | "exT"  -> resolve_with [RefC (id_of_string"exT_intro");c] gl
        | "exT2" -> resolve_with [RefC (id_of_string"exT_intro2");c] gl
        |    _      -> error "Not a predefined existential";;
let intros_exists_tac c = THEN intros (exists_tac c);;

let replace_tac_vernac c1 c2 = 
   replace_tac (RefC (id_of_string"eq")) (RefC (id_of_string"eqT")) c1 c2 
   (RefC (id_of_string"sym_equal")) (RefC (id_of_string"sym_eqT"));;

let rewrite_tac_vernac c =
   equality_tac (RefC (id_of_string"eq")) (RefC (id_of_string"eqT")) c 
     (RefC (id_of_string"sym_equal")) (RefC (id_of_string"sym_eqT"));;

let reflexivity_tac (_,_,cl as gl) =
   match string_head cl with
     "eq" -> resolve (RefC (id_of_string"refl_equal")) gl
  |  "eqT" -> resolve (RefC (id_of_string"refl_eqT")) gl
  |    _   -> error "Not a predefined equality";;
let intros_reflexivity_tac = THEN intros reflexivity_tac;;

let symmetry_tac (_,_,cl as gl) =
   match string_head cl with
     "eq" -> resolve (RefC (id_of_string"sym_equal")) gl
  |  "eqT" -> resolve (RefC (id_of_string"sym_eqT")) gl
  |    _   -> error "Not a predefind eqpuality";;
let intros_symmetry_tac = THEN intros symmetry_tac;;

let transitivity_tac n (_,_,cl as gl) =
   match string_head cl with
     "eq" -> resolve_with [RefC (id_of_string"trans_equal");n] gl
  |  "eqT" -> resolve_with [RefC (id_of_string"trans_eqT");n] gl
  |    _   -> error "Not a predefined equality";;
let intros_transitivity_tac n = THEN intros (transitivity_tac n);;


type vernac_keyword =
  Kabort | Kabsurd | Kafter | Kall | Kallt | Kapply | Kassume
| Kassumes | Kassumption | Kauto | Kaxiom | Kbegin | Kbody | Kchange
| Kchapter | Kcheck | Kclear | Kclose | Kcompute | Kconstr
| Kcontradiction | Kcut | Kspecialize | Kdata | Kdefinition
| Kdiscriminate | Kdo | Kdrop | Kelim
| Kelimtype | Kelse | Kempty | Kend | Kerase | Keval | Kex | Kex2 | Kexact
| Kexists | Kext | Kextraction | Kext2 | Kfact | Kfst | Kgeneralize
| Kgoal | Khint | Khnf | Kgenerative
| Khypothesis | Kidtac | Kif | Kimmediate | Kin | Kind
| Kinduction | Kinductive | Kinfo | Kinhabits
| Kinspect | Kinstantiate | Kinterface | Kintro | Kintros | Kleave | Kleft
| Klemma | Klet | Kload | Klocal | Kmatch | Knoisy | Kopen
| Korelse | Kparameter | Kparameters | Kpattern | Kprint | Kproof | Kprop
| Kpostulate | Kquit | Kred
| Kreflexivity | Kremark | Krepeat | Kreplace | Kreset | Krestart
| Krewrite | Kright | Ksave | Ksearch | Ksection | Kset
| Kshow | Ksilent | Ksimpl | Ksnd | Ksplit
| Kstatement | Ksymmetry | Ksyntax
| Kverify | Ktranscript | Kstate
| Kthen | Ktheorem
| Ktransitivity | Ktrivial | Ktry | Ktype | Ktypes | Ktypeset | Kundo | Kunfold
| Kupon | Kuse
| Kuntil | Kvariable | Kvariables | Kverbose | Kfocus | Kunfocus | Kwith
| Kextract | Krec_wf | Kby | Kprogram_all |Kprogram | Ksee_program
| Kfml | Kcd | Kpwd
| Kstates | Kwrite | Kread | Krestore | Kremove
| Krequire | Kprovide | Kaddpath | Kdelpath | Kloadpath | Kpackages
;;

let (vernac_keywords,rev_vernac_keywords) =
  let t = make_vect 307 [] in
  let rev_t = make_vect 307 [] in
  do_list (fun (s,tok) -> hash_add_assoc (s,tok) t;
                          hash_add_assoc (tok,s) rev_t) [
    "Abort", Kabort; "Absurd", Kabsurd; "After", Kafter; "All", Kall; 
    "AllT", Kallt; "Apply", Kapply; "Assume", Kassume; "Assumes", Kassumes;
    "Assumption", Kassumption; "Auto", Kauto; "Axiom", Kaxiom;
    "Begin",Kbegin;
    "Body", Kbody; "Change", Kchange; "Chapter", Kchapter; "Check",
    Kcheck; "Clear", Kclear; "Close", Kclose; "Compute", Kcompute;
    "Constr", Kconstr; "Cut", Kcut; "Specialize", Kspecialize; "Contradiction", Kcontradiction;
    "Data", Kdata; "Definition", Kdefinition;
    "Discriminate",Kdiscriminate; "Do", Kdo;
    "Drop", Kdrop; "Elim", Kelim;
    "ElimType", Kelimtype; "else", Kelse; "Empty", Kempty;
    "End", Kend; "Erase", Kerase;
    "Eval", Keval; "Ex", Kex; "Ex2", Kex2; "Exact", Kexact; "Exists",
    Kexists; "ExT", Kext; "Extraction", Kextraction; "ExT2", Kext2;
    "Fact", Kfact; "Fst", Kfst; "Generalize", Kgeneralize;
    "Goal", Kgoal; "Hint", Khint; "Hnf",
    Khnf; "Generative",Kgenerative; "Hypothesis", Khypothesis; 
    "Idtac", Kidtac; "if", Kif; "Immediate", Kimmediate; "in", Kin; 
    "Ind", Kind; "Induction", Kinduction; "Inductive",
    Kinductive; "Info", Kinfo; "Inhabits", Kinhabits;
    "Inspect", Kinspect; "Instantiate", Kinstantiate;
    "Interface",Kinterface;
    "Intro", Kintro; "Intros", Kintros; "Leave", Kleave;
    "Left", Kleft; "Lemma",
    Klemma; "let", Klet; "Load", Kload; "Local", Klocal; "Match",
    Kmatch; "Noisy",Knoisy; "Open", Kopen; "Orelse", Korelse; "Parameter",
    Kparameter; "Parameters", Kparameters;
    "Pattern", Kpattern; "Print", Kprint; "Proof",
    Kproof; "Demonstration",Kgoal;
    "Prop", Kprop; "Postulate", Kpostulate; "Quit", Kquit;
    "Red", Kred;
    "Reflexivity", Kreflexivity; "Remark", Kremark; "Repeat", Krepeat;
    "Replace", Kreplace; "Reset", Kreset; "Restart", Krestart; "Rewrite",
    Krewrite; "Right", Kright; "Save", Ksave; "Qed",Ksave;
    "Search", Ksearch;
    "Section", Ksection; "Set", Kset; "Show", Kshow; "Silent",
    Ksilent; "Simpl", Ksimpl; "Snd", Ksnd;
    "Split",Ksplit; "Statement", Kstatement;
    "Symmetry", Ksymmetry;
    "Syntax", Ksyntax;
    "Verify",Kverify;
    "State",Kstate;
    "Transcript",Ktranscript;
    "then", Kthen; "Theorem", Ktheorem;
    "Transitivity", Ktransitivity;
    "Trivial", Ktrivial; "Try", Ktry; "Type", Ktype; "Types",
    Ktypes; "Type_Set", Ktypeset; "Undo", Kundo; "Unfold", Kunfold; 
    "until", Kuntil; "Upon",Kupon; "Use",Kuse; "Variable",
    Kvariable; "Variables", Kvariables; "Verbose", Kverbose; 
    "Focus", Kfocus; "Unfocus", Kunfocus; "with", Kwith;
    "Realizer", Kextract; "rec", Krec_wf; "By", Kby;
    "Program_all", Kprogram_all; "Program",Kprogram;
    "See_program",Ksee_program;
    "Fml", Kfml; "Cd", Kcd; "Pwd", Kpwd;
    "States",Kstates; "Write",Kwrite;
    "Read",Kread; "Restore",Krestore; "Remove",Kremove;
    "Require",Krequire; "Provide",Kprovide;
    "AddPath",Kaddpath; "DelPath",Kdelpath; "LoadPath",Kloadpath;
    "Packages",Kpackages
  ]; (t,rev_t)
;;

let plist elem = plist_rec
  where rec plist_rec = function
    [< elem e; plist_rec l >] -> e::l
  | [< >] -> []
;;

let ne_plist elem = function
  [< elem e; (plist elem) l >] -> (e,l)
;;

let ne_list_with_sep sep elem = do_rec
  where rec do_rec = function
    [< elem e; (function [< sep(); do_rec l >] -> l | [< >] -> []) l >] -> e::l
;;

let list_with_sep sep elem = function
    [< (ne_list_with_sep sep elem) l >] -> l
  | [< >] -> []
;;

let right_assoc op subexp =
  let rec do_loop x = function
    [< op f; subexp y; (do_loop y) z >] -> f x z
  | [< >] -> x in
  function [< subexp x; (do_loop x) y >] -> y
;;

let left_assoc op subexp =
  let rec do_loop x = function
    [< op f; subexp y; (do_loop (f x y)) z >] -> z
  | [< >] -> x in
  function [< subexp x; (do_loop x) y >] -> y
;;

let op_arrow x y = ArrowC(x,y);;
let op_apply s x y = AppC(AppC(RefC (id_of_string s),x),y);;

let op_iff x y = AppC(AppC(RefC (id_of_string "iff"),x),y);;

let inl_op_apply_inl = fun s (inl x) (inl y) -> inl(op_apply s x y);;
let inl_op_not_inl =
 fun (inl c) -> inl(AppC(RefC (id_of_string"not"),c))
   | (inr _) -> error "Cannot have a curly-braced term inside a negation";;


let inl_op_plus_inl = fun
    (inl c1) (inl c2) -> inl(op_apply "sum" c1 c2)
  | (inl c1) (inr c2) -> inl(op_apply "sumor" c1 c2)
  | (inr c1) (inr c2) -> inl(op_apply "sumbool" c1 c2)
  | (inr c1) (inl c2) -> error "Syntax error in sum parsing";;

let iterated_product bd c =
    it_list (fun cmd (s,ty) -> ProdC(s,ty,cmd)) c (rev bd);;
let iterated_lambda bd c =
    it_list (fun cmd (s,ty) -> LambdaC(s,ty,cmd)) c (rev bd);;

type 'a option = None | Some of 'a;;

let rec vernac = function
  [< 'Tkw Krequire ; 'Tident id ;
   (function
    [< 'Tstring fname ; 'Tdot >] -> fname
  | [< 'Tdot >] -> id) fname >] -> set_record Invisible; require id fname
| [< 'Tkw Kprovide ; 'Tident id ; 'Tdot >] -> set_record Invisible; provide id
| [< 'Tkw Kaddpath ; 'Tstring dir ; 'Tdot >] -> set_record Invisible; add_path dir
| [< 'Tkw Kdelpath ; 'Tstring dir ; 'Tdot >] -> set_record Invisible; del_path dir
| [< 'Tkw Kdrop; 'Tdot >]
                 -> set_record Invisible; drop ()
| [< 'Tkw Kquit; 'Tdot >]
                 -> set_record Invisible; sys__exit 0
| [< 'Tkw Kfml ; 'Tdot >] -> set_record Invisible; fml()
| [< 'Tkw Kinterface ;
   begin function
   [< 'Tstring s >] -> s
 | [< >] -> ""
   end s;
   'Tdot >] -> (set_record Invisible;
                try forward_goX.v s with
                Failure _ -> warning "X Interface Error")
| [< 'Tkw Kpwd ; 'Tdot >] -> set_record Invisible; print_string (getwd__getwd())
| [< 'Tkw Kcd ; begin function
                      [< 'Tident s >] -> s
                    | [< 'Tstring s >] -> s
                end s;
     'Tdot >] -> set_record Invisible;
                 (try sys__chdir (glob s) with
                  sys__Sys_error str ->
                  warning ("Cd failed: " ^ str));
                 print_string (getwd__getwd())
| [<
    'Tkw Kload;
    begin function
    [< 'Tkw Kverbose >] -> true
  | [< >] -> false
    end verbosely;
    begin function
      [< 'Tident s >] -> s
    | [< 'Tstring s >] -> s
    end s;
    'Tdot
  >]             -> set_record Invisible;
                    if verbosely then load_vernacular_noisily.v s
                    else load_vernacular.v s
| [<
    'Tkw Kopen;
    begin function
      [< 'Tident s >] -> s
    | [< 'Tstring s >] -> s
    end s;
    'Tdot
  >]             -> set_record Invisible; open_vernacular s
| [< 'Tkw Kclose; 'Tdot >]       -> close_vernacular ()
| [< 'Tkw Kgoal ;
     (function
      [< 'Tdot >] -> goal_mode ()
    | [< command c ; 'Tdot >] ->
      if not(refining()) then
          with_light_rollback (fun () -> (theorem "Unnamed_thm" 0; statement c; goal_mode())) ()
      else error "repeated Goal not permitted in refining mode") x >] -> x
| [< 'Tkw Kpostulate ; 'Tdot >] -> cast_to_postulate();()
| [< 'Tkw Kabort; 'Tdot >] -> set_record Invisible; abort_goals()
| [< 'Tkw Krestart; 'Tdot >] -> set_record Invisible; restart()
| [< 'Tkw Kinstantiate; command c; 'Tdot >] -> set_current_goal 0; instantiate 1 c
| [< tactic_com_list tcl; 'Tdot >]
                                     -> set_current_goal 1; by tcl
| [< 
    'Tint n; 'Tcolon; 
    begin function
      [< 'Tkw Kinstantiate; command c; 'Tdot >] -> set_current_goal 0;
                                                   instantiate n c
    | [< tactic_com_list tcl; 'Tdot >] -> set_current_goal n; solve n tcl
    end x
  >] -> x
| [< 'Tkw Kfocus; 'Tint n; 'Tdot >]
                                 -> set_record Invisible; make_focus(n)
| [< 'Tkw Kunfocus; 'Tdot >]
                                 -> set_record Invisible; make_focus(0)
| [<
    'Tkw Ksave;
    begin function
      [< 'Tdot >] -> with_light_rollback save_named ()
    | [< 'Tident s; 'Tdot >] -> with_light_rollback save_anonymous_thm s
    | [< 'Tkw Kremark; 'Tident s; 'Tdot >] -> with_light_rollback save_anonymous_remark s
    | [< 'Tkw Ktheorem; 'Tident s; 'Tdot >] -> with_light_rollback save_anonymous_thm s
    | [< 'Tkw Kstate; 'Tident name;
         (function
          [< 'Tstring desc >] -> desc
        | [< >] -> "") desc;
         'Tdot >] -> set_record Invisible;save_state name desc
    end x
  >] -> x

| [< 'Tkw Krestore; 'Tkw Kstate; 'Tident name ; 'Tdot >] -> set_record Invisible;restore_state name

| [< 'Tkw Kwrite; 'Tkw Kstates;
     (function
      [< 'Tident s >] -> s
    | [< 'Tstring s >] -> s) fname;
     'Tdot >] -> set_record Invisible;extern_state fname

| [< 'Tkw Kread ; 'Tkw Kstates;
     (function
      [< 'Tident s >] -> s
    | [< 'Tstring s >] -> s) fname;
     'Tdot >] -> set_record Invisible;intern_state fname

| [< 'Tkw Kremove ; 'Tkw State ; 'Tident name ; 'Tdot >] -> set_record Invisible; forget_state (is_silent()) name

| [<
    'Tkw Kundo;
    begin function
      [< 'Tint n ; 'Tdot >] -> n
    | [< 'Tdot >] -> 1
    end n
  >] -> set_record Invisible; Undo n
| [<
    'Tkw Kshow;
    begin function
      [< 'Tdot >]             -> set_record Invisible; show()
    | [< 'Tint n; 'Tdot >]   -> set_record Invisible; print_subgoal n
    | [< 'Tkw Kproof; 'Tdot >]   -> set_record Invisible; show_proof()
    | [< 'Tkw Kprogram; 
      begin function
          [< 'Tdot >] -> set_record Invisible; See_program()
        | [< 'Tint n; 'Tdot >] -> set_record Invisible; see_program n
      end x
      >] -> x
    end x
  >] -> x
| [<
    'Tkw Kprint;
    begin function
      [< 'Tkw Kall; 'Tdot >] ->
        set_record Invisible; PP(print_full_context_typ ())
    | [< 'Tdot >] ->
        set_record Invisible; PP(print_local_context()); PP(print_val())
    | [< 'Tident s; 'Tdot >] -> set_record Invisible; PP(print_name (id_of_string s))
    | [< 'Tkw Khint; 'Tdot >] ->
        set_record Invisible; print_search()
    | [< 'Tkw Ksection; 'Tident s; 'Tdot >] ->
        set_record Invisible; PP(print_sec_context_typ s)
    | [< 'Tkw Kstate ; 'Tdot >] ->
        set_record Invisible; verify__print_state()
    | [< 'Tkw Kstates ; 'Tdot >] ->
        set_record Invisible;
        PP (prlist (fun (n,desc) -> [< 'ID n; 'S" : "; 'S desc ; 'FNL >])
                   (list_saved_states()))
    | [< 'Tkw Kloadpath ; 'Tdot >] ->
      set_record Invisible; print_loadpath ()
    | [< 'Tkw Kpackages ; 'Tdot >] ->
      set_record Invisible; print_packages ()
    end x
  >] -> x
| [<
    'Tkw Khint;
    begin function
      [< 'Tkw Kunfold; lid l; 'Tdot >] ->
        set_record Context;
        with_heavy_rollback add_unfold l
    | [< 'Tkw Krewrite;  
        begin function 
           [< 'Tminusgreater; lid l; 'Tdot >] -> set_record Context; 
                                     with_heavy_rollback (add_rewrite false) l
         | [< 'Tlessminus; lid l; 'Tdot >] -> set_record Context; 
                                     with_heavy_rollback (add_rewrite true) l
         | [< lid l; 'Tdot >]              -> set_record Context; 
                                     with_heavy_rollback (add_rewrite false) l
        end x >] -> x
    | [< lid l; 'Tdot >] -> set_record Context;
                            with_heavy_rollback add_resolution l
    end x
  >] -> x
| [< 'Tkw Kimmediate; lid l; 'Tdot >] ->
            set_record Context;
            with_heavy_rollback add_trivial_list l
| [< 'Tkw Kerase; 
     begin function 
      [< 'Tkw Krewrite; 
           begin function 
           [< 'Tminusgreater; lid l; 'Tdot >] -> set_record Context; 
                                                 erase_tacs_rewrite false l
         | [< 'Tlessminus; lid l; 'Tdot >] -> set_record Context; 
                                              erase_tacs_rewrite true l
         | [< lid l; 'Tdot >]              -> set_record Context; 
                                              erase_tacs_rewrite false l
     end x >] -> x
    | [< lid l; 'Tdot >] -> set_record Context; erase_tacs l
     end x >] -> x
| [< 'Tsharp; sharp _ >]
            -> print_string "********* SHARP: line ignored"; print_newline();
               set_record Invisible; show()
| [< text t >]                       -> t
| [< '_ >] -> raise Parse_error

and sharp = function
  [< 'Tdot >] -> ()
| [< '_; sharp x >] -> x

and tactic_com_list =
  let rec do_loop x = function
    [< 'Tsemi; 
       begin function 
         [< 'Tlbracket; tactic_com_seq tcl;'Trbracket >] -> THENS x tcl
       | [< tactic_com_orelse y; (do_loop y) z >] -> THEN x z
       end t
    >] -> t
  | [< >] -> x in
  function [< tactic_com_orelse x; (do_loop x) y >] -> y

and tactic_com_seq x =
  ne_list_with_sep (function [< 'Tbar >] -> ()) tactic_com_list x

and tactic_com_orelse x =
    right_assoc (function [< 'Tkw Korelse >] -> ORELSE) tactic_com x

and tactic_com = function
  [< 'Tkw Kassumption >]       -> assumption 
| [< 'Tkw Kdiscriminate >]     -> discriminate__discriminate_const
| [< 'Tkw Kcontradiction >]    -> contradict
| [< 'Tkw Kexact; command c >] -> give_exact_com c
| [< 'Tkw Kintros;
     begin function
       [< 'Tkw Kuntil; 'Tident s >] -> intros_until (id_of_string s)
     | [< 'Tkw Kwith ; 'Tident s >] -> intros_with_id s
     | [< lid l >] -> (match l with [] -> intros | l -> intros_with l)
     end t
  >] -> t
| [< 'Tkw Kintro;
     begin function
       [< 'Tident s >]  -> intros_with [id_of_string s]
     | [< 'Tkw Kwith ; 'Tident s >] -> intro_with_id s
     | [< >] -> intro
     end t
  >] -> t
| [<
    'Tkw Kred;
    begin function
      [< 'Tkw Kin; 'Tident t >] -> red_hyp (id_of_string t)
    | [< >]                        -> red
    end x
  >] -> x
| [<
    'Tkw Ksimpl;
    begin function
      [< 'Tkw Kin; 'Tident t >] -> simpl_hyp (id_of_string t)
    | [< >]                        -> simpl_tac
    end x
  >] -> x
| [<
    'Tkw Kchange; command c;
    begin function
      [< 'Tkw Kin; 'Tident t >] -> change_hyp c (id_of_string t)
    | [< >]                        -> change c
    end x
  >] -> x
| [< 'Tkw Kpattern; (ne_plist pattern_occ) (l,ll) >] -> pattern (l::ll)
(*
(* | "Pattern"; ( + (parse ( * (parse INT n -> n)) ln; command c -> (ln,c)))
                 (c,l) -> pattern (c::l) *)
*)
| [< 'Tkw Kapply; command c;
     begin function
       [< 'Tkw Kwith; 
          begin function
            [< command c1;
               begin function
                 [< 'Tcolonequal; (must_be_ident c1) s; command c2; lassign l >] ->
                         resolve_with_name c (((Dep s),c2)::l)
               | [< lcom l >] -> resolve_with (c::c1::l)
               end x
            >] -> x
          | [< 'Tint n; 'Tcolonequal; command c2; lassign l >] ->
                    resolve_with_name c (((NoDep n),c2)::l)
          end x
       >] -> x
     | [< >] -> resolve c
     end c'
  >] -> c'
| [<
    'Tkw Kelim; command c;
    begin function
      [< 'Tkw Kwith; lcom l >] -> elim_with (c::l)
    | [< >] -> elim c
    end x
  >] -> x
| [< 'Tkw Kelimtype; command c >] -> elim_type c
| [<
    'Tkw Kinduction;
    begin function
      [< 'Tident s >] -> induct (id_of_string s)
    | [< 'Tint n >]   -> induct_nodep n
    end x
  >] -> x
| [< 'Tkw Kcut; command c >] -> cut c
| [< 'Tkw Kspecialize ;
     begin function
       [< 'Tint n ; lcommand c ; 'Tkw Kwith ; lassign l >] -> new_hyp_with_num n c l
     | [< lcommand c ;
          begin function
            [< 'Tkw Kwith ; lassign l >] -> new_hyp_with c l
          | [< >] -> new_hyp c
          end x >] -> x
     end x >] -> x
| [< 'Tkw Kgeneralize; lcom lc >] -> generalize lc
| [< 'Tkw Kuse ; command c >] -> imp_elim_tac c
| [< 'Tkw Kclear; lid l >] -> clear_hyp l
| [<
    'Tkw Kunfold; (ne_plist unfold_occ) (l, ll);
    begin function
      [< 'Tkw Kin; 'Tident t >] -> unfold_nth_hyp (l::ll) (id_of_string t)
    | [< >] -> unfold_nth (l::ll)
    end x
  >] -> x
| [< 'Tkw Ktrivial >] -> trivial
| [<
    'Tkw Kauto;
    begin function
      [< 'Tint n >] -> automatic n
    | [< >]          -> auto
    end x
  >] -> x
| [<
    'Tkw Khnf;
    begin function
      [< 'Tkw Kin; 'Tident t >] -> hnf_tac_hyp (id_of_string t)
    | [< >]                        -> hnf_tac
    end x
  >] -> x
| [< 'Tkw Kdo; 'Tint n; tactic_com tc >] -> DO n tc
| [< 'Tkw Ktry; tactic_com tc >] -> TRY tc
| [< 'Tkw Krepeat; tactic_com tc >] -> REPEAT tc
| [< 'Tkw Kprogram_all >] -> Program_all
| [< 'Tkw Kprogram >] -> Program
(*
| "By"; tactic t      -> t
*)
| [< 'Tkw Kextract ; annotation p >] -> prog_assoc p
| [< 'Tkw Kleft >]    -> intros_left_tac
| [< 'Tkw Kright >]   -> intros_right_tac
| [< 'Tkw Ksplit >]   -> intros_split_tac
| [< 'Tkw Kexists; command c >] -> intros_exists_tac c
| [< 'Tkw Kreplace; command c1; 'Tkw Kwith; command c2 >]
                       -> replace_tac_vernac c1 c2
| [<
    'Tkw Krewrite;
    begin function
      [< 'Tminusgreater; command c >] -> rewrite_tac_vernac c
    | [< 'Tlessminus; command c >] -> elim c
    | [< 'Tkw Kauto ; begin function [< 'Tint n >] -> rewrite_automatic n 
                                   | [< >]         -> rewrite_auto 
                      end x >] -> x
    | [< command c >] -> rewrite_tac_vernac c
    end x
  >] -> x
| [< 'Tkw Kreflexivity >] -> intros_reflexivity_tac
| [< 'Tkw Ksymmetry >]    -> intros_symmetry_tac
| [< 'Tkw Ktransitivity; command c >] -> intros_transitivity_tac c
| [< 'Tkw Kabsurd; command c >] -> absurd c
| [< 'Tkw Kidtac >] -> IDTAC
| [< 'Tlparen ; tactic_com_list tl ; 'Trparen >] -> tl
(*

and tactic = parse
    tactic1 t                         -> t
  | tactic1 t1; tactical T; tactic t2 -> T(t1,t2)

and tactic1 = parse
  {parse_caml_expr0 ()} e; lcom l -> apply_tactic e l
 
and tactical = parse 
    "THEN" -> THEN
  | "ORELSE" -> ORELSE
*)

and mode_selector = function
    [< 'Tkw Ksilent >] -> ((fun () -> make_silent(true)),
                           (fun () -> make_silent(false)))
  | [< 'Tkw Ktranscript >] -> ((fun () -> verify__set_transcript(true)),
                           (fun () -> verify__set_transcript(false)))
  | [< 'Tkw Knoisy >] -> ((fun () -> Noisy := true),
                           (fun () -> Noisy := false))

and text = function

(* sections *)
  [< chapter _; 'Tident id; 'Tdot >] -> set_record Context; open_section id
| [< 'Tkw Kleave; 'Tident id; 'Tdot >] -> set_record Context; close_section id false
| [< 'Tkw Kbegin ; mode_selector (f,_) ; 'Tdot >] ->
  set_record Invisible;f();()
| [< 'Tkw Kend;
   (function
    [<  mode_selector (_,f) ; 'Tdot >] ->
    set_record Invisible;f();()
  | [< 'Tident id; 'Tdot >] -> set_record Context;close_section id true) _ >] -> ()

(* theorem *)
| [<
    theorem_tok n; 'Tident s;
    begin function
      [< 'Tdot >] -> if not(refining()) then theorem s n
                     else error "Theorem not permitted in refining mode"
    | [< 'Tcolon ; command c ; 'Tdot >] ->
      if not(refining()) then
          with_light_rollback (fun () -> (theorem s n;statement c)) ()
      else error "Theorem not permitted in refining mode"
    | [< command c;
         (function
          [< 'Tkw Kproof; command c'; 'Tdot >] ->
          if not(refining()) then
              with_light_rollback (fun () -> (theorem s n; statement c; proof c')) ()
          else error "Theorem not permitted in refining mode"
        | [< 'Tdot >] ->
          if not(refining()) then
              with_light_rollback (fun () -> (theorem s n; statement c)) ()
              else error "Theorem not permitted in refining mode") x >] -> x
    end x
  >] -> x
| [< 'Tkw Kstatement; command c; 'Tdot >] ->
  if not(refining()) then statement c
  else error "Statement not permitted in refining mode"
| [< 'Tkw Kproof; command c; 'Tdot >] ->
      if not(refining()) then proof c
      else error "Proof not permitted in refining mode"

(* definition *)
| [<
    definition_tok n; 'Tident s;
    begin function
      [< 'Tdot >] -> definition s n
    | [<
        'Tequal; command c;
        begin function
          [< 'Tcolon; command c'; 'Tdot >]
                     -> with_light_rollback (fun () -> (definition s n; body_typ c c')) ()
        | [< 'Tdot >]
                     -> with_light_rollback (fun() -> (definition s n; body c)) ()
        end x
      >] -> x
    | [< 'Tcolon; command c'; 'Tequal; command c; 'Tdot >]
                     -> with_light_rollback (fun () -> (definition s n; body_typ c c')) ()
    | [<
        command c;
        begin function
          [< 'Tcolon; command c'; 'Tdot >]
                      -> with_light_rollback (fun () -> (definition s n; body_typ c c')) ()
        | [< 'Tdot >]
                     -> with_light_rollback (fun () -> (definition s n; body c)) ()
        end x
      >] -> x
    end x
  >] -> x
| [<
    'Tkw Kbody; command c;
    begin function
      [< 'Tdot >] -> body c
    | [< 'Tcolon; command c'; 'Tdot >] -> body_typ c c'
    end x
  >] -> x

(* variable definition *)
| [< hyp stre; binder b;
     begin function
       [< 'Tcolon; command c; 'Tdot >] ->
             with_light_rollback (fun () -> (do_list (fun s -> hypothesis (string_of_id s) stre; def_var c) b)) ()
     | [< 'Tdot >] ->
             begin match b with
               s::[] -> hypothesis (string_of_id s) stre
             | _ -> raise Parse_error
             end
     end x >] -> x
| [< assume _; command c; 'Tdot >] -> def_var c
| [< sortdef s ; binder b; 'Tdot >]
               -> let sec = read_sec ()
                  in with_light_rollback (fun () -> (do_list (fun v -> (hypothesis (string_of_id v) sec; def_var s)) b)) ()

(* Inductive definitions *)
| [<
    begin function
    [< 'Tkw Kgenerative ; 'Tkw Kinductive >] -> generative_elim
  | [< 'Tkw Kinductive ;
       begin function
           [< 'Tkw Kupon ; 'Tident s >] -> Name(id_of_string s)
         | [< >] -> Anonymous
       end stamp_tok >] -> (structural_elim stamp_tok)
    end elim_fun;
    begin function
      [<
        definition_tok n; 'Tident s; indpar la;
        'Tcolon; command c; 'Tequal; lidcom li; 'Tdot
      >] -> (set_record Context;
             elim_fun la (id_of_string s) c li (Strength n))
    | [< sortdef s; 'Tident n; indpar la; 'Tequal; lidcom li; 'Tdot >] -> 
            (set_record Context;
             elim_fun la (id_of_string n) s  li (Strength 0))
    end x
  >] -> x
(* Checking commands  commands *)
| [< checkcom f; command c; 'Tdot >] -> set_record Invisible; construct c; f()
| [< 'Tkw Ksyntax; 'Tident s; 'Tstring concrete; 'Tdot >]
                    -> let_syntax (id_of_string s) concrete;()
| [< 'Tkw Kverify ; sequent_list seql ; 'Tdot >] ->
  set_record Invisible;
  if not verify__verify_state seql then
      error "Verification failed."
  else ()
| [<
    'Tkw Kreset;
    begin function
      [< 'Tkw Kafter; 'Tident s; 'Tdot >] ->
        reset_keeping_name (id_of_string s)
    | [< 'Tkw Kall; 'Tdot >] -> reset_all()
    | [< 'Tident "Initial"; 'Tdot >] -> reset_initial()
    | [< 'Tident s; 'Tdot >] -> reset_name (id_of_string s)
    | [< 'Tkw Ksection; 'Tident s; 'Tdot >] -> reset_section s
    | [< 'Tkw Kempty; 'Tdot >] -> reset_prelude()
    end x
  >] -> x
(*
(* | "Reset"; "Prelude"   empeche l'utilisation de prelude comme section *)
*)
| [<
    'Tkw Kextraction;
    begin function
      [< 'Tident s; 'Tdot >] -> set_record Invisible; PP(print_extracted_name (id_of_string s))
    | [< 'Tdot >] -> set_record Invisible; PP(print_extraction ())
    end x
  >] -> x
| [< 'Tkw Kinfo; 'Tident s; 'Tdot >] ->
    set_record Invisible; PP(info_constr (id_of_string s))
| [< 'Tkw Kcheck; 'Tident s; 'Tdot >] ->
    set_record Invisible; PP(print_type (id_of_string s))
| [< 'Tkw Ksearch; 'Tident s; 'Tdot >] ->
    set_record Invisible; PP(print_crible (id_of_string s))
| [< 'Tkw Kinspect; 'Tint n; 'Tdot >] ->
    set_record Invisible; PP(inspect n)

and checkcom = function
  [< 'Tkw Keval >]    -> PP o print_evaluate
| [< 'Tkw Kcompute >] -> PP o print_compute
| [< 'Tkw Kcheck >]   -> PP o print_val

and theorem_tok = function
  [< 'Tkw Ktheorem >]  -> 0
| [< 'Tkw Klemma >]    -> sup0 (read_sec () -3)
| [< 'Tkw Kfact >]     -> sup0(read_sec() -1)
| [< 'Tkw Kremark >]   -> read_sec() 

and definition_tok = function
  [< 'Tkw Kdefinition >] -> 0
| [< 'Tkw Klet >]        -> sup0 (read_sec () -2)
| [< 'Tkw Klocal >]      -> read_sec() 

and chapter = function
(*
   "Book" -> ()
 | "Part" -> ()
*)
   [< 'Tkw Kchapter >] -> ()
 | [< 'Tkw Ksection >] -> ()

and hyp = function
  [< 'Tkw Kaxiom >] -> 0
| [< 'Tkw Kparameter >] -> 0
| [< 'Tkw Kparameters >] -> 0
| [< 'Tkw Khypothesis >] -> read_sec () 
| [< 'Tkw Kvariable >] -> read_sec ()
| [< 'Tkw Kvariables >] -> read_sec ()

and assume = function
  [< 'Tkw Kassumes >] -> ()
| [< 'Tkw Kinhabits >] -> ()
| [< 'Tkw Kassume >] -> ()

and sortdef = function
   [< 'Tkw Kdata >] -> PropD
 | [< 'Tkw Kset >] -> PropS
 | [< 'Tkw Ktype >] -> TypeN
 | [< 'Tkw Ktypes >] -> TypeN
 | [< 'Tkw Ktypeset >] -> TypeS

and constructor = function
   [< 'Tident s; 'Tcolon; command c >] -> (id_of_string s,c)

and lidcom = function
   [< constructor c;
      (plist (function [< 'Tbar; constructor c' >] -> c')) lc >] -> c::lc
 | [< >] -> []

and lidabs = function
   [< bdcom bd ; (plist (function [< 'Tsemi; bdcom bd' >] -> bd')) lbd >] ->
                  bd@(flat lbd)
and indpar = function
   [< 'Tlbracket; lidabs la; 'Trbracket >] -> la
 | [< >] -> []

and unfold_occ = function
   [< (ne_plist (function [< 'Tint n >] -> n)) (n,ln); 'Tident s >]
                -> (n::ln,id_of_string s)
 | [< 'Tident s >]
                -> ([],id_of_string s)

and pattern_occ = function
  [< (ne_plist (function [< 'Tint n >] -> n)) (n,ln); command s >]
                  -> (n::ln,s)
| [< command s >] -> ([],s)

and lassign x = plist assign x

and assign = function
    [< 'Tident s; 'Tcolonequal; command valu >] -> (Dep(id_of_string s),valu)
  | [< 'Tint n ; 'Tcolonequal ; command valu >] -> (NoDep(n),valu)

and lequal x = plist 
  (function
   [< 'Tident var; 'Tcolonequal; command val >] -> (id_of_string var,val)
  ) x

and lid x = plist (function [< 'Tident s >] -> (id_of_string s)) x

and lcom x = plist command x

(* parser for commands *)

and stamp = function
    [< 'Tlbracket ; 'Tident s ; 'Trbracket >] -> Name(id_of_string s)
  | [< >] -> Anonymous

and bdcom = function
   [< binder bd; 'Tcolon; command c >] -> map (fun x -> (x,c)) bd

and binder x =
  ne_list_with_sep
    (function [< 'Tcomma >] -> ())
    (function [< 'Tident v >] -> id_of_string v)
    x

and lcommand = function
    [< (ne_plist command) (h,t) >] -> it_list (fun a b -> AppC(a,b)) h t

and constructl = function
   [< 'Tlbrace;
      (list_with_sep (function [< 'Tbar >] -> ()) command) lc;
      'Trbrace >] -> lc

and lcommand_after c = function
    [< (plist command) l >] -> it_list (fun a b -> AppC(a,b)) c l

(* "<term,term>" + "<lterm>" *)
and angle_bracketed_terms = function
    [< 'Tless ; command c1 ;
       (function
        [< 'Tcomma ; command c2 ; 'Tgreater >] -> inl(c1,c2)
      | [< (lcommand_after c1) l ; 'Tgreater >] -> inr l) x >] -> x

and sig_body v = function
    [< command c1;
       begin function
             [< 'Tbar >] -> ("sig2", "sig")
           | [< 'Tampersand >] -> ("sigS2", "sigS")
       end (s1,s2);
      command c2;
      begin function
            [< 'Tampersand; command c3 >] ->
            AppC(AppC(AppC(RefC (id_of_string s1),c1),LambdaC(v,c1,c2))
                 ,LambdaC(v,c1,c3))
          | [< >] ->      AppC(AppC(RefC (id_of_string s2),c1),LambdaC(v,c1,c2))
      end c;
      'Trbrace
      >] -> c

(* "term" + "{term}" *)
and curly_braced_terms = function
    [< 'Tlbrace ;
       (function
        [< 'Tident id ;
         begin function
               [< 'Tcolon ; (sig_body (id_of_string id)) c >] -> inl c
             | [< (lcommand_after (RefC (id_of_string id))) lc ; 'Trbrace >] -> inr lc
         end x >] -> x
      | [< lcommand lc ; 'Trbrace >] -> inr lc) x >] -> x

and conj_cases = function (c1,c2) -> function
    [< 'Tlparen ; command c3 ; 'Tcomma ; command c4 ; 'Trparen >] ->
    AppC(AppC(AppC(AppC(RefC (id_of_string"pair"),c1),c2),c3),c4)
  | [< 'Tlbrace ; command c3 ; 'Tcomma ; command c4 ; 'Trbrace >] ->
    AppC(AppC(AppC(AppC(RefC (id_of_string"conj"),c1),c2),c3),c4)
  | [< 'Tkw Kfst;
        begin function
              [< 'Tlbrace; lcommand l; 'Trbrace >] ->
              AppC(AppC(AppC(RefC (id_of_string"proj1"),c1),c2),l)
            | [< 'Tlparen; lcommand l; 'Trparen >] ->
              AppC(AppC(AppC(RefC (id_of_string"fst"),c1),c2),l)
            end x >] -> x
  | [< 'Tkw Ksnd;
        begin function
              [< 'Tlbrace; lcommand l; 'Trbrace >] ->
              AppC(AppC(AppC(RefC (id_of_string"proj2"),c1),c2),l)
            | [< 'Tlparen; lcommand l; 'Trparen >] ->
              AppC(AppC(AppC(RefC (id_of_string"snd"),c1),c2),l)
            end x >] -> x

and bdcom_or_lcommand = function strm ->
    if (match fst(stream_get strm) with
            Tident _ -> true | _ -> false) then
        (function [< lcommand lc ;
                 (function
                  [< 'Tcomma ; (must_be_ident lc) id ;
                     binder bd ; 'Tcolon ; command c >] ->
                      inl(map (function x -> (x,c)) (id::bd))
                | [< 'Tcolon ; (must_be_ident lc) id ; command c >] ->
                      inl[(id,c)]
                | [< >] -> inr lc) rslt >] -> rslt) strm
    else
        inr(lcommand strm)

and syntax_cases l1 = function
    [< 'Tkw Kex; 'Tlparen; lcommand l2; 'Trparen >]
    -> AppC(AppC(RefC (id_of_string"ex"),l1),l2)
  | [< 'Tkw Kex2; 'Tlparen; command c1; 'Tcomma; command c2; 'Trparen >]
    -> AppC(AppC(AppC(RefC (id_of_string"ex2"),l1),c1),c2)
  | [< 'Tkw Kmatch; command c; 'Tkw Kwith >] ->
    ElimC(c,l1)
  | [< 'Tkw Kall; 'Tlparen; lcommand l2; 'Trparen >]
    -> AppC(AppC(RefC (id_of_string"all"),l1),l2)
  | [< 'Tkw Kallt; 'Tlparen; lcommand l2; 'Trparen >]
    -> AppC(AppC(RefC (id_of_string"allT"),l1),l2)
  | [< 'Tkw Kext; 'Tlparen; lcommand l2; 'Trparen >]
    -> AppC(AppC(RefC (id_of_string"exT"),l1),l2)
  | [< 'Tkw Kext2; 'Tlparen; command c1; 'Tcomma; command c2; 'Trparen >]
    -> AppC(AppC(AppC(RefC (id_of_string"exT2"),l1),c1),c2)
  | [< core_command c1;
       begin function
             [< 'Tequal >] -> "eq"
           | [< 'Tequalequal >] -> "eqT"
       end t;
       core_command c2 >]
    -> AppC(AppC(AppC(RefC (id_of_string t),l1),c1),c2)

and core_command = function
    [< simple_command d >] ->
    (match d with inl c -> c | inr _ -> error "Parsing error in eq")

and core_simple_command = function
    [< 'Tlbracket; bdcom bd; 'Trbracket; command c2 >] ->
    iterated_lambda bd c2

  | [< 'Tlparen ;
       bdcom_or_lcommand disj ;
       (match disj with
        inl b -> (function [< 'Trparen ; command c >] -> iterated_product b c)
      | inr lc -> (function [< 'Trparen >] -> lc)) x >] -> x
     
  | [< angle_bracketed_terms disj ;
       (match disj with
        inl p -> conj_cases p
      | inr l -> syntax_cases l) x >] -> x

| [< 'Tkw Kconstr; 'Tlparen; 'Tint n; 'Tcomma; command c; 'Trparen >] -> 
         ConstrC(n,c)
| [< 'Tkw Kind; stamp st ; 'Tlparen; 'Tident s; 'Tcolon; command c; 'Trparen; 
     constructl lc >] -> IndC(st,id_of_string s,c,lc)
| [< 'Tkw Kif; command c; 'Tkw Kthen; command t;
     'Tkw Kelse; command e >]
  -> AppC(AppC(AppC(RefC (id_of_string"IF"),c),t),e)
| [< 'Tident v >] -> RefC(id_of_string v)
| [< 'Tkw Kprop >] -> PropN
| [< 'Tkw Kset >] -> PropS
| [< 'Tkw Kdata >]  -> PropD
| [< 'Tkw Ktype >] -> TypeN
| [< 'Tkw Ktypeset >] -> TypeS

and simple_command = function
    [< core_simple_command x >] -> inl x
  | [< curly_braced_terms disj >] -> disj

and plus_command x =
   (left_assoc  (function [< 'Tplus >] -> inl_op_plus_inl)
   (right_assoc (function [< 'Tstar >] -> inl_op_apply_inl "prod")
   (function
    [< 'Ttilde ; simple_command c >] -> inl_op_not_inl c
  | [< simple_command c >] -> c)
    )) x

and strip_plus_command  =function
    [< plus_command (inl c) >] -> c

and command x =
    right_assoc (function [< 'Tminusgreater >] -> op_arrow
                        | [< 'Tlessminusgreater >] -> op_iff)
   (right_assoc (function [< 'Tbslashslash >] -> op_apply "or")
   (right_assoc (function [< 'Tslashbslash >] -> op_apply "and")
    strip_plus_command
    )) x

(*and command x =
    right_assoc (function [< 'Tminusgreater >] -> op_arrow)
   (right_assoc (function [< 'Tbslashslash >] -> op_apply "or")
   (right_assoc (function [< 'Tslashbslash >] -> op_apply "and")
    (function [< plus_command (inl c) >] -> c)
    )) x*)

and must_be_ident l _ =
  match l with
    RefC v -> v
  | _ -> raise Parse_error

and must_be_anno_ident l _ =
  match l with
    ApurC(RefC v) -> v
  | _ -> raise Parse_error

and must_be_none x _ =
  match x with
    None -> ()
  | _ -> raise Parse_error

and annotation = function
    [< simple_annotation_with_operators c1 ;
       (function [< 'Tlparencolon ; command c2 ; 'Tcolon; 'Trparen >] -> AnnotC(c1,c2)
               | [< >] -> c1) t >] -> t

and lannotation =
  let rec ann_loop c = function
    [< annotation c'; (ann_loop (AappC(c,c'))) c'' >] -> c''
  | [< >] -> c in
  function [< annotation c; (ann_loop c) c' >] -> c'

and all_opers = function
                [< 'Tslashbslash >] ->
                (fun c1 c2 -> AappC(AappC(ApurC(RefC (id_of_string"and")),c1),c2))
              | [< 'Tbslashslash >] ->
                (fun c1 c2 -> AappC(AappC(ApurC(RefC (id_of_string"or")),c1),c2))
              | [< 'Tstar >] ->
                (fun c1 c2 -> AappC(AappC(ApurC(RefC (id_of_string"prod")),c1),c2))
              | [< 'Tplus >] ->
                (fun c1 c2 -> AappC(AappC(ApurC(RefC (id_of_string"sum")),c1),c2))
              | [< 'Tminusgreater >] ->
                (fun c1 c2 -> ApurC(ArrowC(forget_annot c1, forget_annot c2)))

and simple_annotation_with_operators = function strm ->
    right_assoc all_opers simple_annotation strm

and simple_annotation = function
  [< 'Tlbracket; binder b; 'Tcolon; command c1; 'Trbracket;
     annotation c2 >] ->
                       it_list (fun annotation str -> AlambdaC(str,c1,annotation))
        c2 (rev b)
| [< 'Tlbracketlbrace; binder b; 'Tcolon; command c1; 'Trbrace ; 'Trbracket;
     annotation c2 >] ->
                       it_list (fun annotation str -> AlambdacomC(str,c1,annotation))
        c2 (rev b)
| [< 'Tlparen; lannotation l1;
     begin function
       [< 'Trparen >] -> l1
     | [< (must_be_anno_ident l1) v;
          begin function
            [< 'Tcolon; command c1; 'Trparen; command c2 >] ->
                   ApurC(ProdC(v,c1,c2))
          | [< 'Tcomma; binder b; 'Tcolon; command c1; 'Trparen;
               command c2 >] ->
                   ApurC(it_list (fun annotation str -> ProdC(str,c1,annotation))
                         c2 (rev (v::b)))
          end c
       >] -> c
     end c
  >] -> c
| [< 'Tkw Kconstr; 'Tlparen; 'Tint n; 'Tcomma; command c; 'Trparen >] -> 
         ApurC(ConstrC(n,c))
| [< 'Tkw Kind; stamp st ; 'Tlparen; 'Tident s; 'Tcolon; command c; 'Trparen; 
     constructl lc >] -> ApurC(IndC(st,id_of_string s,c,lc))
| [<
    'Tless; command c1;
    begin function
      [< lcommand l >] -> Some l
    | [< >] -> None
    end lcomm;
    begin function
      [<
        'Tgreater;
        begin let l1 = match lcomm with Some l -> AppC(c1,l) | None -> c1 in
        (function
         [< 'Tkw Kmatch; annotation c; 'Tkw Kwith >] -> ArecC(c,l1)
       | [< 'Tkw Kif ; annotation c1 ;
            'Tkw Kthen ; annotation c2 ;
            'Tkw Kelse ; annotation c3 >] -> AappC(AappC(ArecC(c1,l1),c2),c3)
       | [< 'Tkw Klet ; 'Tlparen ; lidabs lb ; 'Trparen ;
            'Tequal ; annotation c2 ; 'Tkw Kin ; annotation c3 >] ->
         AappC(ArecC(c2,l1),
               it_list (fun com (str,t) -> AlambdaC(str,t,com)) c3 (rev lb))
       | [< 'Tkw Krec_wf ; binder a ; 
            'Tlparencolon ; command c2 ; 'Tcolon ; 'Trparen ; 
            'Tlbracket; binder b; 'Tcolon; command c3; 'Trbracket; 
            annotation c4 >] -> 
              AappC(ArecursionC(RefC (id_of_string"well_founded_induction"),
                    c3,l1,c2),
                    (it_list (fun annotation str -> 
                                AlambdaC(str,c3,annotation)) 
                         (it_list (fun annotation str -> 
                                    (AlambdaC(str,ArrowC(c3,l1),annotation))) 
                                  c4 (rev a))
                         (rev b)))
         )
        end x
      >] -> x
    | [<
        (must_be_none lcomm) _;
        'Tcomma; command c2; 'Tgreater;
        begin function
          [<
            'Tlparen; annotation c3; 'Tcomma; annotation c4; 'Trparen
          >]              -> AappC(AappC(ApurC(AppC(AppC(RefC (id_of_string"pair"),c1),c2)),c3),c4)
        | [<
            'Tkw Kfst;
            begin function
               [< 'Tlparen; lannotation l; 'Trparen >]
                       -> AappC(ApurC(AppC(AppC(RefC (id_of_string"fst"),c1),c2)),l)
            end x
          >] -> x
        | [<
            'Tkw Ksnd;
            begin function
               [< 'Tlparen; lannotation l; 'Trparen >]
                       -> AappC(ApurC(AppC(AppC(RefC (id_of_string"snd"),c1),c2)),l)
            end x
          >] -> x
        end x
      >] -> x
    end x
  >] -> x
| [< 'Tident v >] -> ApurC(RefC(id_of_string v))
| [< 'Tkw Kdata >]  -> ApurC(PropD)

and decl = function
    [< 'Tident id ; 'Tcolon ; command v >] -> (Name(id_of_string id),v)

and assumptions strm = plist decl strm

and sequent = function
    [< command concl ; (plist (function [< 'Tequal >] -> ()
                                    | [< 'Tequalequal >] -> ())) _ ;
       (plist decl) hyps >] -> (hyps,concl)

and sequent_list strm =
    plist_with_sep (function [< 'Tbar ; 'Tbar >] -> ()) sequent strm
;;

let parse_vernac = vernac;;
