(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                              univ.ml                                     *)
(****************************************************************************)

(* Universes are stratified by a partial ordering >=.
   Let ~ be the associated equivalence. We also have a strict ordering
   > between equivalence classes, and we maintain that > is acyclic,
   and contained in >= in the sense that [U]>[V] implies U>=V.

   At every moment, we have a finite number of universes, and we
   maintain the ordering in the presence of assertions U>V and U>=V.

   The equivalence ~ is represented by a tree structure, as in the
   union-find algorithm. The assertions > and >= are represented by
   adjacency lists *)

#open "std";;
#open "initial";;

type universe = Universe of int;;
type relation = 
   Greater of bool * universe * relation (* if bool then > else >= *)
 | Equiv of universe
 | Terminal;;

type arc = Arc of universe * relation | Undef
and universes = Graph of arc list;;

(* in Arc(u,Greater(b,v,r))::arcs, we have u>v if b, and u>=v if not b, 
   and r is the next relation pertaining to u; this relation may be
   Greater or Terminal. *)

(* points to the first arc pertaining to u 
let arc_of u = arc_of_u where rec arc_of_u = function
    [] -> anomaly "Wrong universe structure"
  | (Arc(v,_) as arc)::l -> if eq(u,v) then arc else arc_of_u l;;
A completely applicative structure would search the graph structure
with (arc_of u graph). We prefer, for efficiency, give a direct access
as ARC.(Int i), where (Universe i) = u. *)

let initial_universes = (Graph []);;

(* The component of the state containing the structure of universes *)
let UNI = ref initial_universes;;

(* Fast access to first arc pertaining to given universe *)
let ARC = make_vect 1000 Undef;;

let enter_arc a = match a with
   Arc(Universe(i),_) -> let (Graph l) = !UNI 
                         in UNI:=Graph(a::l); ARC.(i) <- a
 | Undef -> anomaly "Undefined universe";;

(* Every universe has a unique canonical arc representative *)

(* repr : universe -> arc *)
(* canonical representative : we follow the Equiv links *)
let rec repr (Universe u) = 
  match ARC.(u) with 
    Arc(_,Equiv(v)) -> repr v
  | Undef -> anomaly "Undefined universe"
  | arc -> arc;;

let can = map repr;;

(* transitive closure : we follow the Greater links *)
(* close : relation -> universe list * universe list *)
let close = closerec ([],[]) 
where rec closerec ((U,V) as pair) = function
    Terminal           -> pair
  | Greater(true,v,r)  -> closerec (v::U,V) r
  | Greater(false,v,r) -> closerec (U,v::V) r
  | _ -> anomaly "Wrong universe structure";;

(* reprgeq : arc -> arc list *)
(* All canonical arcv such that arcv>=arcu with arcv#arcu *)
let reprgeq = function
    Undef -> anomaly "Undefined universe"
  | (Arc(_,ru) as arcu) -> let (_,V) = close ru
                 in searchrec [] V
where rec searchrec W = function
     [] -> W
   | v::V -> let arcv = repr v 
             in if memq arcv W or eq(arcu,arcv) then searchrec W V
                else searchrec (arcv::W) V;;

(* collect : arc -> arc list * arc list *)
(* collect u = (V,W) iff V={v canonical | u>v} W={w canonical | u>=w}-V *)
(* i.e. collect does the transitive closure of what is known about u *)
let collect u = coll_rec [] [] ([],[u])
where rec coll_rec V W = function
    [],[] -> (V,subtractq W V)
  | (Arc(_,rv) as arcv)::V',W' -> if memq arcv V then coll_rec V W (V',W')
                                  else let (gt,geq) = close rv
                             in coll_rec (arcv::V) W (can(gt@geq)@V',W')
  | [],(Arc(_,rw) as arcw)::W' -> 
          if (memq arcw V) or (memq arcw W) then coll_rec V W ([],W')
          else let (gt,geq) = close rw 
               in coll_rec V (arcw::W) (can(gt),can(geq)@W')
  | _ -> anomaly "Undefined universe";;

type order = EQ | GT | GE | NGE;;

(* compare : universe -> universe -> order *)
let compare(u,v) = 
    let arcu = repr(u) and arcv = repr(v)
    in if eq(arcu,arcv) then EQ
       else let (V,W) = collect(arcu)
            in if memq arcv V then GT
               else if memq arcv W then GE
               else NGE;;

(* Invariants : compare(u,v) = EQ <=> compare(v,u) = EQ
                compare(u,v) = GT or GE => compare(v,u) = NGE
                compare(u,v) = NGE => compare(v,u) = NGE or GE or GT

   Adding u>=v is consistent iff compare(v,u) # GT 
    and then it is redundant iff compare(u,v) # NGE
   Adding u>v is consistent iff compare(v,u) = NGE 
    and then it is redundant iff compare(u,v) = GT *)


(* between : universe -> arc -> arc list *)
(* we assume  compare(u,v) = GE with v canonical    *)
(* between u v = {w|u>=w>=v, w canonical}          *)     
let between u arcv = snd (explore ([(arcv,[arcv])],[]) (repr u))
    where rec explore (memo,l) arcu = 
       try (memo,unionq (assq arcu memo) l) (* when memq arcu memo *)
       with Not_found -> let W = reprgeq arcu
         in let (memo',sols)=it_list explore (memo,[]) W
            in let sols'=if sols=[] then l else arcu::(unionq sols l)
               in ((arcu,sols')::memo',sols');;
(* Note: hd(between u v) = repr u  *)
(* between is the most costly operation *)

(* setgt : universe -> universe -> unit *)
(* forces u > v *)
let setgt u v = match repr(u) with
     Arc(u',ru) -> enter_arc(Arc(u',Greater(true,v,ru))); ()
   | Undef -> anomaly "Undefined universe";;

(* checks that non-redondant *)
let setgt_if u v = match compare(u,v) with
    GT -> ()
  | _ -> setgt u v;;

(* setgeq : universe -> universe -> unit *)
(* forces u >= v *)
let setgeq u v = match repr(u) with
    Arc(u',ru) -> enter_arc(Arc(u',Greater(false,v,ru))); ()
  | Undef -> anomaly "Undefined universe";;

(* checks that non-redondant *)
let setgeq_if u v = match compare(u,v) with
    NGE -> setgeq u v
  | _ -> ();;

(* merge : universe -> universe -> unit *)
(* we assume  compare(u,v) = GE *)
(* merge u v  forces u ~ v with repr u as canonical repr *)
let merge u v = match between u (repr v) with
   Arc(u',_)::V -> let redirect (W,W') = function
        Arc(v',rv) -> let V,V' = close rv
                      in (enter_arc (Arc(v',Equiv(u'))); (unionq V W,V'@W'))
      | Undef -> anomaly "Undefined universe"
                   in let (W,W') = it_list redirect ([],[]) V
                      in (map (setgt_if u') W; map (setgeq_if u') W'; ())
 | Undef::_ -> anomaly "Undefined universe"
 | [] -> anomaly "between";;

(* merge_disc : universe -> universe -> unit *)
(* we assume  compare(u,v) = compare(v,u) = NGE *)
(* merge_disc u v  forces u ~ v with repr u as canonical repr *)
let merge_disc u v = match repr u with
        Arc(u',_) -> (match repr v with
          Arc(v',rv) -> let V,V' = close rv
                        in (enter_arc (Arc(v',Equiv(u')));
                            map (setgt_if u') V;
                            map (setgeq_if u') V'; ())
        | Undef -> anomaly "Undefined universe")
      | Undef -> anomaly "Undefined universe";;


(* exception INCONSISTENCY;; *)

(* enforcegeq : universe -> universe -> unit *)
(* enforcegeq u v will force u>=v if possible, will fail otherwise *)
let enforcegeq u v =
    match compare(u,v) with
        NGE -> (match compare(v,u) with
                  GT -> raise INCONSISTENCY
                | GE -> merge v u
                | NGE -> setgeq u v
                | EQ -> anomaly "compare")
      | _ -> ();;

(* enforceq : universe -> universe -> unit *)
(* enforceq u v will force u=v if possible, will fail otherwise *)
let enforceq u v =
    match compare(u,v) with
        EQ -> ()
      | GT -> raise INCONSISTENCY
      | GE -> merge u v
      | NGE -> match compare(v,u) with
                    GT -> raise INCONSISTENCY 
                  | GE -> merge v u
                  | NGE -> merge_disc u v
                  | EQ -> anomaly "compare";;

(* enforcegt : universe -> universe -> unit           UNUSED
(* enforcegt u v will force u>v if possible, will fail otherwise *)
let enforcegt u v =
    match compare(u,v) with
        GT -> ()
      | GE -> setgt u v
      | EQ -> raise INCONSISTENCY
      | NGE -> (match compare(v,u) with
                     NGE -> setgt u v
                   | _ -> raise INCONSISTENCY);;
*)

let Dummy_univ = Universe(0);; (* for prover terms *)

(* The latest universe *)
let U = ref 0;;

let next_univ () = let n = !U
                   in if n = 999 then error "Too many universes"
                      else (U:=succ_int !U; !U);;

let mk_univ () = Universe(next_univ());;

let New_univ () = 
    let v = mk_univ() in enter_arc(Arc(v,Terminal)); v
and Super u = 
    let v = mk_univ() in enter_arc(Arc(v,Greater(true,u,Terminal))); v;;

(* UNUSED 
let Supereq u = 
    let v = mk_univ() in enter_arc(Arc(v,Greater(false,u,Terminal))); v;;
*)

let sup(u,v) = match compare(u,v) with
   NGE -> (match compare(v,u) with
               NGE -> let w = New_univ() in (setgeq w u; setgeq w v; w)
             | _   -> v)
 | _   -> u;;

let reset_universes universes = 
  if eq(!UNI,universes) then ()  (* ARC and U are OK *)
  else (* We have to recompute UNI, ARC and U from universes *)
       (modify_vect (fun _ -> Undef) ARC;
        UNI:=universes;
        U:=0;
        let reset_arc = function
          Undef -> anomaly "Undefined universe"
        | (Arc(Universe(u),_) as a) -> match ARC.(u) with
                   Undef -> ARC.(u) <- a;
                            if gt(u,!U) then (U:=u; ())
                 | _ -> ()
        in let (Graph arcs) = universes in do_list reset_arc arcs);;

let read_uni () = !UNI;;

(* Debug *)
let print_universes () = (let (Graph arcs) = !UNI in map pr arcs; ())
    where pr = function 
      Arc(Universe(u),r) ->
          print_int u; 
          (match r with
               Greater(b,Universe(v),_) ->
                     print_string(if b then ">" else ">="); print_int v
             | Terminal -> print_string "."
             | Equiv(Universe(v)) ->  print_string "="; print_int v);
          print_newline()
    | Undef -> anomaly "UNI";;
