(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                              *)
(*                            Projet Coq                    *)
(*                                              *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                    Lyon                       *)
(*                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                 termfw.ml                        *)
(****************************************************************************)

(* The F-omega terms for extraction of programs *)

#infix "o";;
#open "std";;
#open "initial";;
#open "univ";;

(********************************************************************)

(*                Abstract syntax of F-omega                        *)

(********************************************************************)


(* type fterm =
    Fvar of name * fterm                  (* free variables *)
  | Fconst of name * fterm                (* constants *)
  | Frel of int                           (* variables *)
  | Fomega                                (* proposition types *)
  | Fapp of fterm * fterm                 (* application  (M N) *)
  | Flambda of name * fterm * fterm       (* abstraction  [x:T]M *)
  | Fprod of name * fterm * fterm         (* product      (x:T)M *)
  | Find of fterm * fterm list * fspecifr * fterm list
  | Fconstr of int * fterm
  | Frec of fterm list * fterm 
  | Fimplicit
and fspecifr = Fspecifr of int * (int * fterm) list
;;*)

(********************************************************************)

(*                   Substitution functions                         *)

(********************************************************************)

let fapplist = it_list (fun x y -> Fapp(x,y));;

(* Check if term is closed *)
exception FreeVar;;

let fclosed term = let rec closed_rec n = function
    Frel(m)       -> if m>n then raise FreeVar
  | Fapp(c,c')    -> closed_rec n c; closed_rec n c'
  | Flambda(_,c,c') -> closed_rec n c; closed_rec (n+1) c'
  | Fprod(_,c,c') -> closed_rec n c; closed_rec (n+1) c'
  | Find(_,c,l,_,_)   -> closed_rec n c; do_list (closed_rec (n+1)) l
  | Fconstr(_,c)  -> closed_rec n c
  | Frec(l,c)     -> closed_rec n c; do_list (closed_rec n) l
  | _             -> ()
in try (closed_rec 0 term; true) with FreeVar -> false;;

exception Occur;;
let foccurn n term = 
  let rec occur_rec n = function
    Frel(m)         -> if m=n then raise Occur
  | Fapp(c,c')      -> occur_rec n c; occur_rec n c'
  | Flambda(_,c,c') -> occur_rec n c; occur_rec (n+1) c'
  | Fprod(_,c,c')   -> occur_rec n c; occur_rec (n+1) c'
  | Find(stamp,c,l,_,_)   -> occur_rec n c; do_list (occur_rec (n+1)) l
  | Fconstr(_,c)    -> occur_rec n c
  | Frec(l,c)       -> occur_rec n c; do_list (occur_rec n) l
  | _               -> ()
  in try (occur_rec n term; true) 
     with Occur -> false;;


(* Relocate the free variables of c to depth n *)
let fliftn k n c = if k=0 then c
    else (liftrec n) c where rec
    liftrec n = function
     (Frel i as x)          -> if i<n then x else Frel (i+k)
   | Fapp(c1,c2)            -> Fapp(liftrec n c1,liftrec n c2)
   | Flambda(name,c1,c2)    -> Flambda(name,liftrec n c1,liftrec (n+1) c2)
   | Fprod(name,c1,c2)      -> Fprod(name,liftrec n c1,liftrec (n+1) c2)
   | Find(stamp,c,lc, Fspecifr(k,s),lP) ->
                     Find(stamp,liftrec n c, map (liftrec (n+1)) lc,
                           Fspecifr(k, map (fun (i,x)->(i,liftrec n x)) s),
                          map (liftrec (n+1)) lP)
   | Fconstr(i,c)      -> Fconstr(i,liftrec n c)
   | Frec(lf,c)        -> Frec(map (liftrec n) lf, liftrec n c)
   | x                 -> x;;

let flift k = fliftn k 1;;

(* Lifting the binding depth up 1 binding *)

let fpop  = flift (-1);;

(* substitute lam for Frel(1) in c *)

let fsubst_lift lam =  substrec 1
    where rec substrec n = function
    (Frel(k) as c) -> if k=n then flift (n-1) lam
                      else if k<n then c
                      else Frel (k-1)
   | Fapp(c1,c2)            -> Fapp(substrec n c1,substrec n c2)
   | Flambda(name,c1,c2)    -> Flambda(name,substrec n c1,substrec (n+1) c2)
   | Fprod(name,c1,c2)      -> Fprod(name,substrec n c1,substrec (n+1) c2)
   | Find(stamp,c,lc, Fspecifr(k,s),lP) ->
                     Find(stamp,substrec n c, map (substrec (n+1)) lc,
                           Fspecifr(k, map (fun (i,x)->(i,substrec n x)) s),
                          map (substrec (n+1)) lP)
   | Fconstr(i,c)      -> Fconstr(i,substrec n c)
   | Frec(lf,c)        -> Frec(map (substrec n) lf,substrec n c)
   | x                 -> x;;


(* 2nd : closed case *)

let fsubst_closed lam = substrec 1
    where rec substrec n = function
    (Frel(k) as c) -> if k=n then lam
                      else if k<n then c else Frel (k-1)
   | Fapp(c1,c2)            -> Fapp(substrec n c1,substrec n c2)
   | Flambda(name,c1,c2)    -> Flambda(name,substrec n c1,substrec (n+1) c2)
   | Fprod(name,c1,c2)      -> Fprod(name,substrec n c1,substrec (n+1) c2)
   | Find(stamp,c,lc, Fspecifr(k,s),lP) ->
                     Find(stamp,substrec n c, map (substrec (n+1)) lc,
                           Fspecifr(k, map (fun (i,x)->(i,substrec n x)) s),
                          map (substrec (n+1)) lP)
   | Fconstr(i,c)      -> Fconstr(i,substrec n c)
   | Frec(lf,c)        -> Frec(map (substrec n) lf,substrec n c)
   | x                 -> x;;

let fsubst1 lam = if fclosed lam then fsubst_closed lam
                  else fsubst_lift lam;;

(* substitute Frel(1) for Fvar(..name..) in closed term *)
let fsubst_var name = substrec 1
    where rec substrec n = function
     (Fvar(name',_) as fvar') -> if eq(name,name') then Frel(n)
                                  else fvar'
   | Fapp(c1,c2)            -> Fapp(substrec n c1,substrec n c2)
   | Flambda(name,c1,c2)    -> Flambda(name,substrec n c1,substrec (n+1) c2)
   | Fprod(name,c1,c2)      -> Fprod(name,substrec n c1,substrec (n+1) c2)
   | Find(stamp,c,lc, Fspecifr(k,s),lP) ->
                     Find(stamp,substrec n c, map (substrec (n+1)) lc,
                           Fspecifr(k, map (fun (i,x)->(i,substrec n x)) s),
                          map (substrec (n+1)) lP)
   | Fconstr(i,c)      -> Fconstr(i,substrec n c)
   | Frec(lf,c)        -> Frec(map (substrec n) lf,substrec n c)
   | x                 -> x;;

let fsubst_const name lam =  substrec 1
    where rec substrec n = function
    (Fconst(Name(name'),_) as c) -> 
            if eq(name,name') then flift (n-1) lam
                        else c
   | (Fvar(Name(name'),_) as c) ->
                        if eq(name,name') then flift (n-1) lam
                        else c
   | Fapp(c1,c2)            -> Fapp(substrec n c1,substrec n c2)
   | Flambda(name,c1,c2)    -> Flambda(name,substrec n c1,substrec (n+1) c2)
   | Fprod(name,c1,c2)      -> Fprod(name,substrec n c1,substrec (n+1) c2)
   | Find(stamp,c,lc, Fspecifr(k,s),lP) ->
                     Find(stamp,substrec n c, map (substrec (n+1)) lc,
                           Fspecifr(k, map (fun (i,x)->(i,substrec n x)) s),
                          map (substrec (n+1)) lP)
   | Fconstr(i,c)      -> Fconstr(i,substrec n c)
   | Frec(lf,c)        -> Frec(map (substrec n) lf,substrec n c)
   | x                 -> x;;


(* Tests whether a name (constant or variable) occurs in a fterm *)
let foccur_eq s = occur_rec where rec occur_rec = function
    Fvar(Name(s'),_)        -> eq(s,s')
  | Fconst(Name(s'),_)      -> eq(s,s')
  | Fapp(c,c')              -> (occur_rec c') or (occur_rec c)
  | Flambda(_,c,c')         -> (occur_rec c') or (occur_rec c)
  | Fprod(_,c,c')           -> (occur_rec c') or (occur_rec c)
  | Find(_,c,l,_,_)       -> (occur_rec c) or (exists occur_rec l)
  | Fconstr(_,c)        -> occur_rec c
  | Frec(l,c)           -> (exists occur_rec l) or (occur_rec c)
  | _                       -> false;;

let fglobals c = make_set(glob_rec [] c) where rec glob_rec globs = function
    Fvar(Name(s),_)  -> s::globs
  | Fconst(Name(s),_)-> s::globs
  | Fapp(c,c')       -> glob_rec (glob_rec globs c') c
  | Flambda(_,c,c')  -> glob_rec (glob_rec globs c') c
  | Fprod(_,c,c')    -> glob_rec (glob_rec globs c') c
  | Find(_,c,l,_,_)    -> it_list glob_rec (glob_rec globs c) l
  | Fconstr(_,c)     -> glob_rec globs c
  | Frec(l,c)        -> it_list glob_rec (glob_rec globs c) l
  | _                -> globs;;


(* Reduction for inductive elimination *)

let fabs_implicit c = Flambda(Anonymous,Fimplicit,c);;

let rec fredapplist c = function
    []   -> c
  | a::l -> match c with
               Flambda(_,_,c') -> fredapplist (fsubst1 a c') l
             | _               -> fapplist (Fapp(c,a)) l;;

let freduce_app c = redrec [] c
 where rec redrec l x = match x with
      Fconst(_,x)    -> redrec l x
    | Fapp(c1,c2)    -> redrec (c2::l) c1
    | Flambda(_,_,c) -> (match l with []     -> x,l
                                    | (a::m) -> redrec m (fsubst1 a c))
    | _              -> x,l;;

let find_finductype c = match freduce_app c with
                 (Find (stamp,x1,x2,x3,x4)),l -> (stamp,x1,x2,x3,x4),l
               | _          -> raise Finduc;;

let finductype c = try let (x,l) = find_finductype c
                       in if l=[] then x
                          else anomaly "Not an inductive type 1"
                   with Finduc -> anomaly "Not an inductive type 2";;

let fmake_rec F n = iterate fabs_implicit (n+1)
                           (Frec(map (flift (n+1)) F,Frel 1));;

exception Felim;;

let rec fred_elimination (lf,c) =
    let (i,ityp),l = fhnfconstruct c in
    let (_,_,_,Fspecifr(n,lt),_) = finductype ityp
    and fi = nth lf (i+1)
    in try let ti = assoc i lt
           and F = fmake_rec lf n 
           in fredapplist ti (fi::F::l)
    with Not_found -> fredapplist fi l
and fhnfconstruct c =  hnfstack [] c
    where rec hnfstack stack = 
    function (Fconst(_,c) as x) -> hnfstack stack c
           | Fapp(c1,c2)    -> hnfstack (c2::stack) c1
           | Flambda(_,_,c) ->
                   (match stack with 
                       []       -> anomaly "Cannot be a constructor"
                    |  c'::rest -> hnfstack rest (fsubst1 c' c))
           | Frec(x1,x2)    -> hnfstack stack (fred_elimination(x1,x2))
           | Fconstr(c1,c2) -> ((c1,c2),stack)
           | _              ->  raise Felim;;

(* Converts an expression to its normal form *)
let rec fnf c = fnf_app [] c
and fnf_app stack =
  let applist_stack c = fapplist c (map fnf stack) in function
    Flambda(name,c1,c2)   -> (match stack with
                                [] -> Flambda(name,fnf c1,fnf c2)
                  | a1::rest -> fnf_app rest (fsubst1 a1 c2))
  | Fprod(name,c1,c2)     -> Fprod(name,fnf c1,fnf c2)
  | Fapp(con1,con2)       -> fnf_app (con2::stack) con1
  | Find(stamp,c,lc, Fspecifr(k,s),lP) ->
           applist_stack (Find(stamp,fnf c,map fnf lc,
                                Fspecifr(k,map (fun (i,x)->(i,fnf x)) s),
                               map fnf lP))
  | Fconstr(i,c)      -> applist_stack (Fconstr(i,fnf c)) 
  | Frec(lf,x)            -> (try fnf_app stack (fred_elimination (lf,x))
                  with Felim ->
                                   applist_stack(Frec(map fnf lf,fnf x)))
  | x                     -> applist_stack x;;


(* substitute lam1 for name in closed term lam2 and normalize *)
let fsubst_norm name lam1 lam2 = fnf(fsubst_const name lam1 lam2);;

(********************************************************************)

(*                       Approximation                              *)

(********************************************************************)

(* the two kinds of variables *)
(*type freference =
     Flocal of int
   | Fglobal of name * fterm;;

type fapproximation =
     Fabstraction of fterm * fterm
   | Fproduct of fterm * fterm
   | Fvariable of freference * fterm list
   | Fconstant of (name * fterm) * fterm list
   | Finductive of name * fterm * fterm list * fterm list
   | Fconstructor of int * fterm * fterm list
   | Felimination of fterm list * fterm list
   | Fomegaconst
   | Fimplicitconst;;
*)
(* One step of approximation *)
(* fapprox : (fterm list) -> fterm -> fapproximation *)
let rec fapprox stack = hnf
where rec hnf = function
     Frel(n)         -> Fvariable(Flocal(n),stack)
   | Fvar(v1,v2)     -> Fvariable(Fglobal(v1,v2),stack)
   | Fconst(c1,c2)   -> Fconstant((c1,c2),stack)
   | Fomega          -> if stack=[] then Fomegaconst
                        else anomaly "Prop cannot be applied"
   | Find(stamp,c,l,_,_)   -> Finductive(stamp,c,l,stack)
   | Fconstr(i,c)    -> Fconstructor(i,c,stack)
   | Frec(l,c)       -> (try hnf (fred_elimination (l,c))
             with Felim -> Felimination(l,c::stack))
   | Fapp(c1,c2)     -> fapprox (c2::stack) c1
   | Flambda(_,c,c')   -> (match stack with
                          []      -> Fabstraction(c,c')
                        | c''::rest -> fapprox rest (fsubst1 c'' c'))
   | Fprod(_,c,c')   -> if stack=[] then Fproduct(c,c')
                        else anomaly "Product cannot be applied"
   | Fimplicit       ->  Fimplicitconst;;

(* approxim : fterm -> fapproximation *)
let fapproxim c = fapprox [] c;;

let fexpand = function
  Fconstant((_,c),stack) -> fapprox stack c
| _ -> anomaly "Trying to expand a non-constant";; 

(* equality of terms *)
let rec fconv term1 term2 = feqappr (fapproxim term1,fapproxim term2)
and feqappr = function
       (Fabstraction(c1,c2),Fabstraction(c'1,c'2)) -> fconv c1 c'1
                                                    & fconv c2 c'2
     | (Fproduct(c1,c2),Fproduct(c'1,c'2)) -> fconv c1 c'1
                                            & fconv c2 c'2
     | (Fvariable(n1,l1),Fvariable(n2,l2)) -> (n1=n2) &
                                              for_all2eq fconv l1 l2
     | (Fomegaconst,Fomegaconst) -> true
     | ((Fconstant(s1,l1) as appr1),(Fconstant(s2,l2) as appr2)) ->
            (* try first intensional equality *)
           ((s1=s2) & (for_all2eq fconv l1 l2))
           (* else expand the second occurrence (should it be the first?) *)
        or feqappr(appr1,fexpand appr2)
     | ((Fconstant(s1,l1) as appr1),appr2) -> feqappr(fexpand appr1,appr2)
     | (appr1,(Fconstant(s2,l2) as appr2)) -> feqappr(appr1,fexpand appr2)
     | (Finductive(stamp1,c1,l1,stack1),Finductive(stamp2,c2,l2,stack2)) ->
        stamp1 = stamp2 &
        (fconv c1 c2) & (for_all2eq fconv l1 l2) & 
        (for_all2eq fconv stack1 stack2)
     | (Fconstructor(i1,c1,l1),Fconstructor(i2,c2,l2)) ->
        (i1=i2) & (fconv c1 c2) & (for_all2eq fconv l1 l2)
     | (Felimination(l1,m1),Felimination(l2,m2)) ->
            (for_all2eq fconv l1 l2) & (for_all2eq fconv m1 m2)
     | Fimplicitconst,Fimplicitconst -> true
     | _ -> false;;     

let fhnftype = apprec []
where rec apprec stack = app_stack
where rec app_stack = function
   (Fprod(_,_,_)|Fomega as c) -> if stack=[] then c
                        else anomaly "Cannot be applied"
  | (Fconst(_,c))   -> app_stack c
  | Fapp(c1,c2)     -> apprec (c2::stack) c1
  | Flambda(_,_,c)  -> (match stack with
                           []       -> error "Typing error 3"
                         | c'::rest -> apprec rest (fsubst1 c' c))
  | _ -> error "fhnftype";;

(* Inductive types in Fw, building elimination *)

let freduce  = redrec []
where rec redrec largs x = match x with
        Flambda(n,t,c)  -> (match largs with
                                [] -> (x,largs)
                              | a::rest -> redrec rest (fsubst1 a c))
      | Fapp(c1,c2)    -> redrec (c2::largs) c1
      | Fconst(_,x)     -> redrec largs x
      | Frec(c1,c2)     -> (try  redrec largs (fred_elimination (c1,c2))
                           with Felim -> (x,largs))
      | _             -> (x,largs);;

let fhnf_constr c = let (x,l) = freduce c in fapplist x l;;

(* For manipulation of lists of terms *)

let rec fhdchar = function
     Fprod(_,_,c)     -> fhdchar c
   | Fapp(c,_)        -> fhdchar c
   | Flambda(_,_,c)   -> fhdchar c
   | Fconst(Name s,_) -> hd (explode_id s)
   | Fvar(Name s,_)   -> hd (explode_id s)
   | _                -> "y";;

let fnamed_hd A = function Anonymous -> Name(id_of_string(fhdchar A)) 
                         | x         -> x;;

let fprod_it = it_list (fun c (n,t) -> Fprod(n,t,c));;
let farrow c1 c2 = Fprod(Anonymous,c1,c2);;
let farrow_it = list_it farrow;;
let fprod_name (n,A,B) = Fprod(fnamed_hd A n,A,B)
and flambda_name (n,A,B) = Flambda(fnamed_hd A n,A,B)
and fprod_create (A,B) = Fprod(Name(id_of_string(fhdchar A)),A,B)
and flambda_create (A,B) =  Flambda (Name(id_of_string(fhdchar A)),A,B);;

let flambda_implicit A = Flambda(Name(id_of_string "y"),Fimplicit,A);;

(* fapp_rel n c m = (a Frel(n+m)...Frel(n+1)) *)

let frel_list n m = relrec [] 1 
 where rec relrec l k = if k > m then l 
                        else relrec (Frel(n+k)::l) (k+1);;

exception Fnorec;;

let rec fmake_pos k = function
    Fprod(n,A1,A2) -> if foccurn (k+1) A1 then Fprod(n,A1,fmake_pos (k+1) A2)
                      else error "Non strictly positive type 1"
  | x              -> let (hd,largs) = freduce x in
        (match hd with
              Fprod(_,_,_) -> fmake_pos k hd
            | Frel(k') -> if for_all (foccurn k) largs then
                                if k=k' then fapplist hd largs
                                else raise Fnorec
                          else error "Non strictly positive type 2"
            |  _       -> if for_all (foccurn k) largs then raise Fnorec
                          else error "Non strictly positive type 3");;

let rec ftransf_posF u = function
   Fprod(n,A,B) -> flambda_name(n,A,ftransf_posF(Fapp(flift 1 u,Frel 1)) B)
 | x            -> Fapp(x,u);;

type fconstructor =       
      Frecarg of name * fterm * fterm * fconstructor
    | Fconstarg of name * fterm * fconstructor
    | Fhead  of fterm ;;

let rec fconstant_constr = function
         Frecarg(_,_,_,_) -> false
       | Fconstarg(_,_,c) -> fconstant_constr(c)
       | Fhead(_)         -> true;;

let fmake_constr = mrec 1
 where rec mrec k = function
    Fprod(n,B,C)-> (try
          Frecarg(n,fliftn 1 k B, flift 1 (fmake_pos k B),mrec (k+1) C)
          with Forec ->
          Fconstarg(n,fliftn 1 k B,mrec (k+1) C))
  | x            -> let (hd,largs) = freduce x in
       (match hd with
              Fprod(_,_,_) -> mrec k hd
            | Frel(k') -> if for_all (foccurn k) largs then
                                if k=k' then Fhead (fapplist hd largs)
                                else error "Not a fconstructor 1"
                          else error "Not a fconstructor 2"
            |  _       -> error "Not a fconstructor 3");;

let rec ftransf_constrP = function
   Frecarg(n,BX,B,C)-> Fprod(n,BX,
                             farrow B (flift 1 (ftransf_constrP C)))
  |Fconstarg(n,B,C)  -> Fprod(n,B,ftransf_constrP C)
  |Fhead(c)          -> c;;

let farg = flambda_implicit o ftransf_constrP;;

let rec ftransf_constrF f = function
   Frecarg(n,B,_,C) ->
        flambda_implicit (ftransf_constrF
                   (Fapp (Fapp (flift 1 f,Frel 1),ftransf_posF (Frel 1) B)) C)
 | Fconstarg(n,_,C) ->
      flambda_implicit (ftransf_constrF (Fapp (flift 1 f,Frel 1)) C)
 | Fhead(_)         -> f;;


let freduce_arg ci = Flambda(Name(id_of_string"fi"),Fimplicit,
                             Flambda(Name(id_of_string"F"),Fimplicit,
                             (ftransf_constrF (Frel 2) ci)));;

let fnb_prod = nbrec 0 where rec nbrec k c = match fhnftype c with
         Fprod(_,_,c') -> nbrec (k+1) c'
       | _             -> k;;

let fmake_ind stamp t lc =
         let lC = map fmake_constr lc in
         let lf = mrec 1 [] lC where rec mrec k lfres =
            function C::lCrest ->
                     if fconstant_constr C then mrec (k+1) lfres lCrest
                     else mrec (k+1) ((k,freduce_arg C)::lfres) lCrest
                   | []        -> lfres
    in Find(stamp,t,lc,Fspecifr(fnb_prod t,lf),map farg lC);;

let ftype_elim_arg ind P lpn i =
    let fi = fsubst1 ind (try nth lpn i with Failure _-> anomaly "No such arg")
    in fredapplist fi [P];;

let ftype_elim_args ind P
        = map (function pn -> fredapplist(fsubst1 ind pn) [P]);;

let ftype_elim ct lpft P =
     try let ((stamp,t,x1,x2,lpn as ind),la) = find_finductype ct in
     let i = Find(stamp,t,x1,x2,lpn) in
        (match lpft with []     -> error "Ill-formed elimination 1"
                     | (Pt::lf) ->
         if (fconv t Pt) & (for_all2eq fconv (ftype_elim_args i P lpn) lf)
         then fapplist P la else error "Ill-typed elimination")
     with Induc -> error "Ill-formed elimination 2";;

let fmake_elim ind =
     let (_,t,_,Fspecifr(n,_),lpd) = finductype ind
     in flambda_create(t,mkfrec 1 lpd)
       where rec mkfrec k = function
        fti::lpdrest -> Flambda(Name(id_of_string"f"),fredapplist (fsubst1 ind fti)[Frel k],
                                mkfrec (k+1) lpdrest)
      | [] -> mkelimabs k t where mkelimabs k = mkelim
   where rec mkelim c = match fhnftype(c) with 
          Fprod(name,A,B) -> flambda_name(name,A,mkelim B)
        | _               -> flambda_create(fapplist ind (frel_list 0 n),
                                            Frec(frel_list (n+1) k,Frel 1));;

(* exception Omega;; *)

let  rec ftype_constr rel = function
    Fvar(_,t)->t
  | Fconst(_,v) -> ftype_constr rel v
  | Frel n       -> flift n (nth rel n)
  | Fomega       -> raise Omega
  | Fapp(v1,v2)  -> (match fhnftype (ftype_constr rel v1) with 
                           Fprod(_,_,t)->(fsubst1 v2 t)
                         | _           -> anomaly"ftype_constr")
  |Flambda(n,t,v)-> Fprod(n,t,ftype_constr (t::rel) v)
  |Fprod(_,t,t1)  -> ftype_constr (t::rel) t1
  |Find(_,t,_,_,_)  -> t
  |Fconstr(i,ind)-> let (_,_,lc,_,_) = finductype ind 
                    in fsubst1 ind (nth lc i)
  | Frec(P::_,c) -> let ct = ftype_constr rel c in
                    let (_,la) = find_finductype ct in fapplist P la
  | _            -> anomaly"ftype_constr 2";; 

let ftype_term = ftype_constr;;

let ftype_of t = try ftype_term [] t
                    with Omega   -> error "Fomega"
                       | reraise -> raise(reraise);;

let farity t l =
let rec far l n = function
     Frel(m) -> n-1
   | Fprod(_,t,T) -> (far (Fimplicit::l)
                          (n + (try (ftype_term l t;1)
                                   with Omega->0
                                 | reraise -> raise(reraise)))
                         T)
  | _ -> error "bad constructor type"
in far l 1 t;;

let is_ftype t l = try (ftype_term l t;true)
                    with Omega -> false
                       | reraise -> raise(reraise);;


let fmake_elimination c P = 
 let ct = ftype_of c in
 let ((stamp,t,lc,x,lpn as ind),la) = find_finductype ct in
     let i = Find(stamp,t,lc,x,lpn) in
 let lft = ftype_elim_args i P lpn in
     (list_it (fun ft e -> Flambda(Name(id_of_string"f"),ft,e)) lft 
              (Frec(P::(frel_list 0 (length lpn)),c)));;
