(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                              term.ml                                     *)
(****************************************************************************)

(* A type-checker of the full system with a predicative hierarchy of types *)
(* Three flavors of Prop and Type for Christine's extraction algorithm     *)
(* Inductive types                                                         *)

#infix "o";;
#open "std";;
#open "initial";;
#open "more_util";;
#open "univ";;
#open "extraction";;
#open "pp";;
#open "stdpp";;

(********************************************************************)

(*                       Abstract syntax                            *)

(********************************************************************)


(* type constr =
    Meta of int list                   (* meta-variables (for synthesis) *)
  | Var of variable                    (* free variables *)
  | Const of constant                  (* constants *)
  | Rel of int                         (* variables *)
  | Prop of contents                   (* proposition types *)
  | Type of contents * universe        (* universes *)
  | App of constr * constr             (* application  (M N) *)
  | Lambda of name * constr * constr   (* abstraction  [x:T]M *)
  | Prod of name * constr * constr     (* product      (x:T)M *)
  | Ind of constr * constr list * specifr * specift
                                       (* inductive type *)
  | Construct of int * constr          (* constructor *)
  | Rec of bool * constr list * constr (* elimination *)
  | Implicit
and specifr = Specifr of int * (int * constr) list
and specift = Specift of (constr list * constr list)
                       * (constr list * constr list)
and variable = Decl of name * judgement * information
                                       (* variable declarations *)
and constant = Def of name * judgement * information
                                       (* constant definitions *)
and judgement = Judge of constr * constr * level;; *)

let forward_prterm = nref(forward : constr -> std_ppcmds);;
let prterm = forward_prterm;;

(**************************************************************************)
(* Specification of inductive types gives
       pairs n,(i,phi i) for reduction of recursive constructors
       [p1;.;pn], [P1;..;Pp] of allowed types of elimination with
           corresponding types of argument for dependent elimination
       [p1;.;pn], [P1;..;Pp] of allowed types of elimination with
           corresponding types of argument for non-dependent elimination  *)
(**************************************************************************)

let prop = Prop(Null) 
and spec = Prop(Pos) 
and data = Prop(Data) 
and typep = Type(Null,Dummy_univ)
and types =  Type(Pos,Dummy_univ);;

(* Levels *)

(* Checks that its argument is a kind, and returns the level of terms
   whose types are of that kind *)

(********************************************************************)

(*                   Substitution functions                         *)

(********************************************************************)

let dependent =
 let rec deprec k = function
     Rel n -> n = k
   | App (c,c') -> deprec k c or deprec k c'
   | Lambda (_,c,c') -> deprec k c or deprec (k+1) c'
   | Prod (_,c,c') -> deprec k c or deprec (k+1) c'
   | Ind (_,c,l,_,_) -> deprec k c or exists (deprec (k+1)) l
   | Construct (i,c) -> deprec k c
   | Rec (_,lf,c) -> exists (deprec k) lf or deprec k c
   | _ -> false in
  deprec;;

(* Check if term is closed *)
exception FreeVar;;

let closed term = let rec closed_rec n = function
    Rel(m)      -> if m>n then raise FreeVar
  | App(c,c')       -> closed_rec n c; closed_rec n c'
  | Lambda(_,c,c')  -> closed_rec n c; closed_rec (n+1) c'
  | Prod(_,c,c')    -> closed_rec n c; closed_rec (n+1) c'
  | Ind(_,c,l,_,_)    -> closed_rec n c; do_list (closed_rec (n+1)) l
  | Construct(_,c)  -> closed_rec n c
  | Rec(_,f,c)      -> do_list (closed_rec n) f; closed_rec n c
  | _           -> ()
in try (closed_rec 0 term; true) with FreeVar -> false;;

(* Check if (Rel n) occurs in term  *)
exception Occur;;

let noccurn n term = let rec occur_rec n = function
    Rel(m)      -> if m=n then raise Occur
  | App(c,c')       -> occur_rec n c; occur_rec n c'
  | Lambda(_,c,c')  -> occur_rec n c; occur_rec (n+1) c'
  | Prod(_,c,c')    -> occur_rec n c; occur_rec (n+1) c'
  | Ind(_,c,l,_,_)    -> occur_rec n c; do_list (occur_rec (n+1)) l
  | Construct(_,c)  -> occur_rec n c
  | Rec(_,f,c)      -> do_list (occur_rec n) f; occur_rec n c
  | _           -> ()
in try (occur_rec n term; true) with Occur -> false;;

(* Lifting the binding depth across k bindings *)

let liftn k c n = if k=0 then c
    else liftrec n c where rec
    liftrec n = function
     (Rel i as c)   -> if i<n then c else Rel(i+k)
   | App(c1,c2)     -> App(liftrec n c1,liftrec n c2)
   | Lambda(name,c1,c2) -> Lambda(name,liftrec n c1, liftrec (n+1) c2)
   | Prod(name,c1,c2)   -> Prod(name,liftrec n c1, liftrec (n+1) c2)
   | Ind(stamp,c,l,Specifr(p,s),Specift((kd,lpd),(kn,lpn)))
                        -> Ind(stamp,liftrec n c, map (liftrec (n+1)) l,
                               Specifr(p,map (fun (i,x)->(i,liftrec n x)) s),
                               Specift((kd,map (liftrec (n+1)) lpd),
                                       (kn,map (liftrec (n+1)) lpn)))
   | Construct(i,c) -> Construct(i,liftrec n c)
   | Rec(b,lpf,c)   -> Rec(b,map (liftrec n) lpf,liftrec n c)
   | x              -> x;;

let lift k c= liftn k c 1;;


(* substitute lam for Rel(1) in c *)

(* 1st : general case *)

let subst_lift lam = substrec 1
    where rec substrec n = function 
     (Rel k as x)   -> if k=n then lift (n-1) lam 
                       else if k<n then x else Rel(k-1)
   | App(c1,c2)     -> App(substrec n c1,substrec n c2)
   | Lambda(name,c1,c2) -> Lambda(name,substrec n c1, substrec (n+1) c2)
   | Prod(name,c1,c2)   -> Prod(name,substrec n c1, substrec (n+1) c2)
   | Ind(stamp,c,l,Specifr(k,s),Specift((kd,lpd),(kn,lpn)))
                        -> Ind(stamp,substrec n c, map (substrec (n+1)) l,
                               Specifr(k,map (fun (i,x)->(i,substrec n x)) s),
                               Specift((kd,map (substrec (n+1)) lpd),
                                       (kn,map (substrec (n+1)) lpn)))
   | Construct(i,c) -> Construct(i,substrec n c)
   | Rec(b,lpf,c)   -> Rec(b,map (substrec n) lpf,substrec n c)
   | x              -> x;;

(* 2nd : closed case *)

let subst_closed lam = substrec 1
    where rec substrec n = function
     (Rel k as x)   -> if k=n then lam                       
                       else if k<n then x else Rel(k-1)
   | App(c1,c2)     -> App(substrec n c1,substrec n c2)
   | Lambda(name,c1,c2) -> Lambda(name,substrec n c1, substrec (n+1) c2)
   | Prod(name,c1,c2)   -> Prod(name,substrec n c1, substrec (n+1) c2)
   | Ind(stamp,c,l,Specifr(k,s),Specift((kd,lpd),(kn,lpn)))
                        -> Ind(stamp,substrec n c, map (substrec (n+1)) l,
                               Specifr(k,map (fun (i,x)->(i,substrec n x)) s),
                               Specift((kd,map (substrec (n+1)) lpd),
                                       (kn,map (substrec (n+1)) lpn)))
   | Construct(i,c) -> Construct(i,substrec n c)
   | Rec(b,lpf,c)   -> Rec(b,map (substrec n) lpf,substrec n c)
   | x              -> x;;


let subst1 lam = if closed lam then subst_closed lam else subst_lift lam;;


(* substitute Rel(1) for Var(..name..) in term *)
let subst_var = function
  Anonymous -> I
| Name(str) -> substrec 1
    where rec substrec n = function
     (Var(Decl(Name(str'),_,_)) as x) -> 
                       if eq(str,str') then Rel(n) else x 
   | App(c1,c2)     -> App(substrec n c1,substrec n c2)
   | Lambda(name,c1,c2) -> Lambda(name,substrec n c1, substrec (n+1) c2)
   | Prod(name,c1,c2)   -> Prod(name,substrec n c1, substrec (n+1) c2)
   | Ind(stamp,c,l,Specifr(k,s),Specift((kd,lpd),(kn,lpn)))
                        -> Ind(stamp,substrec n c, map (substrec (n+1)) l,
                               Specifr(k,map (fun (i,x)->(i,substrec n x)) s),
                               Specift((kd,map (substrec (n+1)) lpd),
                                       (kn,map (substrec (n+1)) lpn)))
   | Construct(i,c) -> Construct(i,substrec n c)
   | Rec(b,lpf,c)   -> Rec(b,map (substrec n) lpf,substrec n c)
   | x              -> x;;


(* for building validation from a construction in LCF *)
let applist = it_list (fun x y -> App(x,y));;

let lambda name typ c = Lambda(name,typ,subst_var name c)
and produit name typ c = Prod(name,typ,subst_var name c);;
                          
(* substitute lam for constant name in term *)
let subst_const name lam = substrec 1
    where rec substrec n = function
     (Const(Def(Name(name'),_,_)) as x) -> 
            if eq(name,name') then lift (n-1) lam else x
   | (Var(Decl(Name(name'),_,_)) as x) ->
            if eq(name,name') then lift (n-1) lam else x
   | App(c1,c2)     -> App(substrec n c1,substrec n c2)
   | Lambda(name,c1,c2) -> Lambda(name,substrec n c1, substrec (n+1) c2)
   | Prod(name,c1,c2)   -> Prod(name,substrec n c1, substrec (n+1) c2)
   | Ind(stamp,c,l,Specifr(k,s),Specift((kd,lpd),(kn,lpn)))
                        -> Ind(stamp,substrec n c, map (substrec (n+1)) l,
                               Specifr(k,map (fun (i,x)->(i,substrec n x)) s),
                               Specift((kd,map (substrec (n+1)) lpd),
                                       (kn,map (substrec (n+1)) lpn)))
   | Construct(i,c) -> Construct(i,substrec n c)
   | Rec(b,lpf,c)   -> Rec(b,map (substrec n) lpf,substrec n c)
   | x              -> x;;


(* Tests whether a name (constant or variable) occurs in a construction *)
let occur s = occur_rec where rec occur_rec = function
    Var(Decl(Name(s'),_,_))  -> s=s'
  | Const(Def(Name(s'),_,_)) -> s=s'
  | App(c,c')              -> (occur_rec c') or (occur_rec c)
  | Lambda(_,c,c')         -> (occur_rec c') or (occur_rec c)
  | Prod(_,c,c')           -> (occur_rec c') or (occur_rec c)
  | Ind(_,c,l,_,_)           -> (occur_rec c) or (exists occur_rec l)
  | Construct(_,c)         -> occur_rec c 
  | Rec(_,f,c')            -> (exists occur_rec f) or occur_rec c'
  | _                      -> false;;

(* Tests whether a name (constant or variable) occurs in a construction *)
let occur_eq s = occur_rec where rec occur_rec = function
    Var(Decl(Name(s'),_,_))  -> eq(s,s')
  | Const(Def(Name(s'),_,_)) -> eq(s,s')
  | App(c,c')              -> (occur_rec c') or (occur_rec c)
  | Lambda(_,c,c')         -> (occur_rec c') or (occur_rec c)
  | Prod(_,c,c')           -> (occur_rec c') or (occur_rec c)
  | Ind(_,c,l,_,_)           -> (occur_rec c) or (exists occur_rec l)
  | Construct(_,c)         -> occur_rec c 
  | Rec(_,f,c')            -> (exists occur_rec f) or (occur_rec c')
  | _                      -> false;;

let occur_name name c = match name with
     Name(str) -> occur str c
   | Anonymous -> false;;

(* All globals appearing in a term *)
let globals c = make_set(glob_rec [] c) where rec glob_rec globs = function
    Var(Decl(Name(s),_,_))  -> s::globs
  | Const(Def(Name(s),_,_)) -> s::globs
  | App(c,c')             -> glob_rec (glob_rec globs c') c
  | Lambda(_,c,c')        -> glob_rec (glob_rec globs c) c'
  | Prod(_,c,c')          -> glob_rec (glob_rec globs c) c'
  | Ind(_,c,l,_,_)          -> it_list glob_rec (glob_rec globs c) l
  | Construct(_,c)        -> glob_rec globs c
  | Rec(_,f,c)            -> it_list glob_rec (glob_rec globs c) f
  | _                     -> globs;;

(* Value of a constant *)

let value_of (Def(_,Judge(c,_,_),_)) =  c;;

let info_of  = function
   Const(Def(_,_,inf)) -> inf 
 | _                   -> error "Not a constant";;

(***************************************)
(* Reduction for inductive elimination *)
(***************************************)

let abs_implicit c = Lambda(Anonymous,Implicit,c);;

(* [Rel (n+m);...;Rel(n+1)] *)

let rel_list n m = reln [] 1
where rec reln l p = if p>m then l else reln (Rel(n+p)::l) (p+1);;

(****************************************************************************)
(* Only Beta-reduction                                                      *)
(****************************************************************************)

let rec redabs c = redapplist [] c
and redapplist stack =
      let app_stack c = applist c (map redabs stack)
in function
    Lambda(name,c1,c2) -> (match stack with
                [] -> Lambda(name,redabs c1,redabs c2)
                | a1::rest -> let c = subst1 a1 c2 in
                                          redapplist rest c)
  | Prod(name,c1,c2)    -> app_stack(Prod(name,redabs c1,redabs c2))
  | App(con1,con2)      -> redapplist (con2::stack) con1
  | Ind(stamp,t,lc,Specifr(k,lf),Specift((kd,ld),(kn,ln))) ->
                      app_stack (Ind(stamp,redabs t, map redabs lc,
                                Specifr(k,map (fun (i,x)->(i,redabs x)) lf),
                                 Specift((kd,map redabs ld),
                                         (kn,map redabs ln))))
  | Rec(b,lf,c)         ->  app_stack (Rec(b,map redabs lf, redabs c))
  | Construct(i,c)  -> app_stack (Construct(i,redabs c)) 
  | x               -> app_stack x;;

exception Redelimination;;
exception Induc;;

let make_rec b F n = iterate abs_implicit (n+1) 
                     (Rec(b,map (lift (n+1)) F,Rel 1));;

let specifr  = function (_,_,_,Specifr(l1,l2),_)->l1,l2;;

let rec red_elimination (b,lf,c) =
    let (i,ityp),l = hnfconstruct c in
    let n,lt = specifr (inductype_spec ityp)
    and fi = nth lf (i+1) 
    in try let ti = assoc i lt
           and F =  make_rec b lf n 
           in (redapplist (fi::F::l) ti)
    with Not_found -> (redapplist l fi)
and hnfconstruct c =  (hnfstack [] c
   where rec hnfstack stack = function
             Const(c)   -> hnfstack stack (value_of c)
           | App(c1,c2) -> hnfstack (c2::stack) c1
           | Lambda(_,_,c) ->
                (match stack with [] -> anomaly "Cannot be a constructor"
                              |  c'::rest -> hnfstack rest (subst1 c' c))
           | Rec(r1,r2,r3) -> hnfstack stack (red_elimination (r1,r2,r3))
           | Construct(c1,c2) -> (c1,c2),stack
           | _ ->  raise Redelimination)
and find_inductype c = (findrec [] c
   where rec findrec l = function
    Const(c)      -> findrec l (value_of c)
  | App(c1,c2)    -> findrec (c2::l) c1
  | Lambda(_,_,c) -> (match l with [] -> raise Induc
                                 | (a::m) -> findrec m (subst1 a c))
  | Rec(r1,r2,r3) -> (try findrec l (red_elimination (r1,r2,r3))
                      with Redelimination -> raise Induc)
  | Ind(stamp,x1,x2,x3,x4) -> (stamp,x1,x2,x3,x4),l
  | _             -> raise Induc)
and inductype_spec c = 
   try let (x,l) = find_inductype c
       in if l<>[] then anomaly "Not an inductive type 1"
          else x
   with Induc -> anomaly "Not an inductive type 2";;

let check_inductype_spec c =
       try let (x,l) = find_inductype c
           in if l<>[] then error "Not an inductive type 1" else x
       with Induc -> error "Not an inductive type 2";;

(* Check that F is equal to [x1]..[xn]Rec(a1 ac,(Rel ip))
with x ip of type (Ind() Rel(i1),..Rel(i(p-1))) distinct variables that do not
occur in a1..ac *)

exception Elimconst;;

let is_elim  = srec 0 [] []
  where rec srec n lapp labs =  function
     Const(c)       -> srec n lapp labs (value_of c)
   | Lambda(_,T,G)  -> (match lapp with [] -> srec (n+1) [] (T::labs) G
                                |(a::lrest)-> srec n lrest labs (subst1 a G))
   | App(c1,c2)     -> if (c2 = Rel 1) & (lapp = []) & (noccurn 1 c1)
                       then srec (n-1) lapp (tl labs) (lift (-1) c1)
                       else srec n (c2::lapp) labs c1
   | Rec(_,lf,Rel i) -> if (lapp = []) & (for_all (noccurn i) lf) then
                        let (_,la) = find_inductype (nth labs i) in
                        let li =
                        map (function Rel k -> let j = i+k in
                                   if for_all (noccurn j) lf
                                   then j else raise Elimconst
                                    | _ -> raise Elimconst)
                           la
                        in if (distinct li)  then li@[i],n
                           else raise Elimconst
                        else raise Elimconst
   | _            -> raise Elimconst;;

let make_elim_fun F largs =
  try let (lv,n) = is_elim F and ll = length largs
      in if ll < n then raise Redelimination else
      let labs,_ = chop_list n largs in
      let F' =  let p = length lv in
        let la' = map_i (fun q aq ->
                    try (Rel (p+1-(index (n+1-q) lv))) with Failure _ -> aq) 1
                  (map (lift p) labs) in
        it_list (fun c _ -> Lambda(Name(id_of_string"x"),Implicit,c)) (applist F la') lv
      in F'
  with Elimconst | Failure _ -> raise Redelimination;;

let rec reduce_app_list x l = match x with
    Const(c)      -> reduce_app_list (value_of c) l
  | App(c1,c2)    -> reduce_app_list c1 (c2::l)
  | Lambda(_,_,c) -> (match l with [] -> x,l
                                 | (a::m) -> reduce_app_list (subst1 a c) m)
  | x             -> x,l;;


let rec red_elim_const r largs =
    let F = make_elim_fun (Const r) largs in
    match reduce_app_list (value_of r) largs with
  (Rec(_,lf,c),lrest) ->
    let ((i,ityp),l) = construct_const c in
    let n,lt = specifr (inductype_spec ityp)
    and fi = nth lf (i+1)
    in let res = try let ti = assoc i lt in (redapplist (fi::F::l) ti)
                 with Not_found -> redapplist l fi
       in res,lrest
 | _ -> anomaly "reduce_app_list"
and construct_const c = hnfstack [] c
    where rec hnfstack stack = function
      Const(c)      -> (try let (c',lrest) = red_elim_const c stack
                            in hnfstack lrest c'
                        with Redelimination -> hnfstack stack (value_of c))
    | App(c1,c2)    -> hnfstack (c2::stack) c1
    | Lambda(_,_,c) -> (match stack with 
                           [] -> anomaly "Cannot be a constructor"
                        |  c'::rest -> hnfstack rest (subst1 c' c))
    | Rec(r1,r2,r3) -> hnfstack stack (red_elimination (r1,r2,r3))
    | Construct(c1,c2)  -> (c1,c2),stack
    | _             -> raise Redelimination;;


(* Converts an expression to its normal form for beta-conversion
    and elimination *)

let rec simplify c = nf_app [] c
where rec nf_app stack =
      let app_stack c = applist c (map simplify stack)
in function
    Lambda(name,c1,c2) -> (match stack with
                [] -> Lambda(name,simplify c1,simplify c2)
                | a1::rest -> let c = subst1 a1 c2 in
                            nf_app rest c)
  | Prod(name,c1,c2)    ->  app_stack (Prod(name,simplify c1,simplify c2))
  | App(con1,con2)      -> nf_app (con2::stack) con1
  | (Ind(stamp,t,lc,Specifr(k,lf),Specift((kd,ld),(kn,ln))) as x) ->
                      app_stack (Ind(stamp,simplify t, map simplify lc,
                                Specifr(k,map (fun (i,x)->(i,simplify x)) lf),
                                 Specift((kd,map simplify ld),
                                         (kn,map simplify ln))))
  | Rec(b,lf,c)         -> (try nf_app stack (red_elimination (b,lf,c)) 
                            with Redelimination ->
                                app_stack(Rec(b,map simplify lf, simplify c)))
  | Construct(i,c)  -> app_stack (Construct(i,simplify c)) 
  | x               -> app_stack x;;


(* substitute lam1 for name in term lam2 and normalize *)
let subst_norm name lam1 lam2 = simplify (subst_const name lam1 lam2);;

(* linear substitution (following pretty-printer) of the value of name in c *)
(* at the occurrences of occ_list *)
(* occ_list must be non empty *)

let substlin occ_list name c =
  let rec substrec n ol = function
   (Const(Def(Name(name'),j,_)) as co)
                   -> if name=name' then
                           (if (hd ol)=n then
                                 let (Judge(cs,_,_)) = j in ((n+1),(tl ol),cs)
                              else ((n+1),ol,co))
                        else (n,ol,co)
 | App(c1,c2)      -> let (n1,ol1,c1') = substrec n ol c1
                        in (match ol1 with 
                               [] -> (n1,[],App(c1',c2))
                             | _  -> let (n2,ol2,c2') = substrec n1 ol1 c2
                                       in (n2,ol2,App(c1',c2')))
 | Lambda(na,c1,c2) -> let (n1,ol1,c1') = substrec n ol c1
                        in (match ol1 with 
                               [] -> (n1,[],Lambda(na,c1',c2))
                             | _  -> let (n2,ol2,c2') = substrec n1 ol1 c2
                                       in (n2,ol2,Lambda(na,c1',c2')))
 | Prod(na,c1,c2)   -> let (n1,ol1,c1') = substrec n ol c1
                        in (match ol1 with 
                               [] -> (n1,[],Prod(na,c1',c2))
                             | _  -> let (n2,ol2,c2') = substrec n1 ol1 c2
                                       in (n2,ol2,Prod(na,c1',c2')))
 | Rec(b,P::lf,d)   -> let rec substlist nn oll = function
                                   [] -> (nn,oll,[])
                             | f::lfe -> (let (nn1,oll1,f') = substrec nn oll f
                                           in (match oll1 with
                                             [] -> (nn1,[],f'::lfe)
                                           | _  -> let (nn2,oll2,lfe') =
                                                      substlist nn1 oll1 lfe 
                                                     in (nn2,oll2,f'::lfe')))
                       in (match (substlist n ol (P::d::lf)) with
                            (n2,ol2,P'::d'::lf') -> (n2,ol2,Rec(b,P'::lf',d'))
                          | _ -> anomaly "substlist")
 | co               -> (n,ol,co)
  in
 let (nn,oll,co) = (substrec 1 (sort (fun (x,y)-> (x<y)) occ_list) c)
   in match oll with
       [] -> simplify co
     | l  -> if nn=1 then error ((string_of_id name)^" does not occur")
                     else error ("bad occurrence numbers of "^(string_of_id name));;

(* unfoldn : (int list -> string -> constr -> constr) *)
(* unfolds a constant in a term following a list of occurrences *)
let unfoldn = function
   []  -> error "Empty list of occurrences"
 | l   -> substlin l;;


(* Same as simplify, but also reduces elimination constants *)
let rec nf c = nf_app [] c
and nf_app stack =
      let app_stack c = applist c (map nf stack)
      in function
    Lambda(name,c1,c2)    -> (match stack with
                              [] -> Lambda(name,nf c1,nf c2)
                              | a1::rest -> let c = subst1 a1 c2 in
                                            nf_app rest c)
  | Prod(name,c1,c2)    -> app_stack(Prod(name,nf c1,nf c2))
  | App(con1,con2)      -> nf_app (con2::stack) con1
  | (Ind(stamp,t,lc,Specifr(k,lf),Specift((kd,ld),(kn,ln))) as x) ->
      app_stack(Ind(stamp,nf t, map nf lc, Specifr(k,map (fun (i,x)->(i,nf x)) lf),
                    Specift((kd,map nf ld),(kn,map nf ln))))
  | Rec(b,lf,c)         -> (try nf_app stack (red_elimination (b,lf,c))
                            with Redelimination ->
                            app_stack(Rec(b,map nf lf,nf c)))
  | Construct(i,c)      -> app_stack (Construct(i,nf c)) 
  | (Const(c) as x)     -> (try let (c',lrest) = red_elim_const c stack
                                in nf_app lrest c'
                            with Redelimination -> app_stack x)
  | x                   -> app_stack x;;


(********************************************************************)

(*                       Approximation                              *)

(********************************************************************)
(* One step of approximation *)

let rec apprec stack = app_stack
where rec app_stack = function
     App(c1,c2)             -> apprec (c2::stack) c1
   | (Lambda(_,_,c) as x)   -> (match stack with
                                  []       -> (x,[])
                                | c'::rest -> apprec rest (subst1 c' c))
   | (Rec(r1,r2,r3) as x)   -> (try app_stack(red_elimination (r1,r2,r3))
                                with Redelimination -> (x,stack))
   | x                      -> (x,stack);;

let hnf = apprec [];;


(********************************************************************)

(*                           Equality                               *)

(********************************************************************)


(* equality of terms with universe adjustment *)
let rec conv term1 term2 =
      eqappr (hnf term1,hnf term2)
and eqappr = function
       ((Var(Decl(s,_,_)),l1),(Var(Decl(s',_,_)),l2)) ->
           (eq(s,s') & (length(l1) = length(l2)) & (for_all2 conv l1 l2))
     | ((Rel(n),l1),((Rel(m),l2))) ->
           ((n=m) & (length(l1) = length(l2)) & (for_all2 conv l1 l2))
     | ((Meta(n),l1),(Meta(m),l2)) ->
           ((n=m) & (length(l1) = length(l2)) & (for_all2 conv l1 l2))
     | ((Prop c1,[]),(Prop c2,[])) -> c1=c2
     | ((Type (c1,u1),[]),(Type (c2,u2),[])) -> c1=c2 & (enforceq u1 u2;true)
     | ((Prod(_,c1,c2),[]),(Prod(_,c'1,c'2),[])) -> conv c1 c'1 & conv c2 c'2
     | ((Lambda(_,c1,c2),[]),(Lambda(_,c'1,c'2),[])) ->
                                          conv c1 c'1 & conv c2 c'2
     | (((Const(s1),l1) as appr1),(Const(s2),l2)) ->
           let universes = read_uni() (* protection for backtracking *) in
           (* try first intensional equality *)
           (eq(s1,s2) & (length(l1) = length(l2)) & (for_all2 conv l1 l2))
           (* else expand the second occurrence (arbitrary heuristic) *)
        or (reset_universes universes; eqappr(appr1, apprec l2 (value_of s2)))
     | ((Const(s1),l1),appr2)      ->  eqappr(apprec l1 (value_of s1), appr2)
     | (appr1,(Const(s2),l2))      -> eqappr(appr1, apprec l2 (value_of s2))
     | ((Ind(stamp1,c1,l1,_,_),l'1),(Ind(stamp2,c2,l2,_,_),l'2)) ->
           stamp1 = stamp2 &
           (conv c1 c2) & (for_all2eq conv l1 l2) & (for_all2eq conv l'1 l'2)
     | ((Construct(i1,c1),l1),(Construct(i2,c2),l2)) ->
        (i1=i2) & (conv c1 c2) & (for_all2eq conv l1 l2)
     | ((Rec(_,l1,c1),l'1),(Rec(_,l2,c2),l'2)) -> 
            (conv c1 c2) & (for_all2eq conv l1 l2) & (for_all2eq conv l'1 l'2)
     | (Implicit,[]),(Implicit,[]) -> true
     | _ -> false

(* we check that term1 <= term2 in the cumulative sense *)
and conv_leq term1 term2 =
      eqappr_leq (hnf term1,hnf term2)
and eqappr_leq = function
       ((Prod(_,c1,c2),[]),(Prod(_,c'1,c'2),[])) -> 
                       conv c1 c'1 & conv_leq c2 c'2  (* Luo's system *)
     | ((Type (c1,u1),[]),(Type (c2,u2),[])) -> c1=c2 & (enforcegeq u2 u1;true)
     | (((Const(s1),l1) as appr1),(Const(s2),l2)) ->
           let universes = read_uni() (* protection for backtracking *) in
           (* try first intensional equality *)
           (eq(s1,s2) & (length(l1) = length(l2)) & (for_all2 conv_leq l1 l2))
           (* else expand the second occurrence (arbitrary heuristic) *)
       or (reset_universes universes; eqappr_leq(appr1,apprec l2 (value_of s2)))
     | ((Const(s1),l1),appr2)         ->  eqappr_leq(apprec l1 (value_of s1),
                                                     appr2)
     | (appr1,(Const(s2),l2))         -> eqappr_leq(appr1,
                                                    apprec l2 (value_of s2))
     | ((Ind(stamp1,c1,l1,_,_),l'1),(Ind(stamp2,c2,l2,_,_),l'2)) ->
 stamp1 = stamp2 &
 (conv_leq c1 c2) & (for_all2eq conv_leq l1 l2) & (for_all2eq conv_leq l'1 l'2)
     | ((Construct(i1,c1),l1),(Construct(i2,c2),l2)) ->
         (i1=i2) & (conv_leq c1 c2) & (for_all2eq conv_leq l1 l2)
     | ((Rec(_,l1,c1),l'1),(Rec(_,l2,c2),l'2)) -> 
 (conv_leq c1 c2) & (for_all2eq conv_leq l1 l2) & (for_all2eq conv_leq l'1 l'2)
     | x -> eqappr x
;;

(* One step of head normal form leading to Prop, Type, or Prod  *)
(* Used for making  explicit the type of a construction *)
(* hnftype: constr -> constr *)
let msgpr s t = [< HOV 0 [< 'S s ; 'SPC ; HOV 0 [< prterm.v t >] ; 'SPC >] ;
                   'CUT >];;
#infix "msgpr";;
let hnftype_error t =
 PPNL[< "Illegal type :" msgpr t; 
             'S" is neither a sort nor a product" >]; error "Type error";;

let hnftype = apprec []
where rec apprec stack = app_stack
where rec app_stack t = match t with
     Rel(_)         -> hnftype_error t
   | Var(_)         -> hnftype_error t
   | Ind(_,_,_,_,_) -> applist t stack
   | Construct(_,_) -> hnftype_error t
   | Implicit       -> anomaly "Implicit1"
   | Const(c)       -> app_stack(value_of c)
   | App(c1,c2)     -> apprec (c2::stack) c1
   | Lambda(_,_,c)  -> (match stack with
                            []       -> hnftype_error t
                          | c'::rest -> apprec rest (subst1 c' c))
   | Rec(r1,r2,r3)  -> (try app_stack(red_elimination (r1,r2,r3))
                        with Redelimination -> hnftype_error t)
     (* Prod/Prop/Type *)
   | c              -> if stack=[] then c
                       else anomaly "Cannot be applied";;




let level_of_kind t = match hnftype t with 
   Type(_,_) -> Object
 | Prop(_) -> Proof
 | _       -> error "Not a proposition or a type";;


(* A few additional primitives for term manipulation *)

let rec is_indtype = function
    Lambda(_,_,c) -> is_indtype c
  | Ind(_,_,l,_,_)  -> true
  | App(f,_)        -> is_indtype f
  | _              -> false;;

let red_product = simplify o redrec
    where rec redrec = function
   App(c1,c2)      -> App(redrec c1,c2)
 | Prod(s,c1,c2)   -> Prod(s,c1,redrec c2)
 | Const(c)        -> let t = value_of c
                      in if is_indtype t then error "Term not reducible"
                         else t
 | _               -> error "Term not reducible";;


(********************************************************************)

(*           Additional functions for use in synthesis              *)

(********************************************************************)

(* head-normal form + delta *)

let reduce  = redrec [] 
where rec redrec largs x = match x with
        Lambda(n,t,c) -> (match largs with
                                []      -> (x,largs)
                              | a::rest -> redrec rest (subst1 a c))
      | App(c1,c2)    -> redrec (c2::largs) c1
      | Const(c)      -> (try let c',lrest = red_elim_const c largs
                              in redrec lrest c'
                          with Redelimination -> redrec largs (value_of c))
      | Rec(r1,r2,r3) -> (try redrec largs (red_elimination (r1,r2,r3))
              with Redelimination -> (x,largs) )
      | _         -> (x,largs);;

let hnf_constr c = let (x,l) = reduce c in applist x l;;

(* Type of a construction in a given context, and check its correction *)
(* in a system with Type:Type *)

(* equality of terms without universe adjustment for synthesis *)
let rec conv_x term1 term2 =
      eqappr_x (hnf term1,hnf term2)
and eqappr_x = function
       ((Rel(n),l1),(Rel(m),l2)) ->
           (n=m) & (length(l1) = length(l2)) & (for_all2 conv_x l1 l2)
     | ((Var(Decl(s,_,_)),l1),(Var(Decl(s',_,_)),l2)) ->
           (eq(s,s') & (length(l1) = length(l2)) & (for_all2 conv_x l1 l2))
     | ((Meta(n),l1),(Meta(m),l2)) ->
           ((n=m) & (length(l1) = length(l2)) & (for_all2 conv_x l1 l2))
     | ((Prop c1,[]),(Prop c2,[])) -> c1=c2
     | ((Type (c1,u1),[]),(Type (c2,u2),[])) -> c1=c2
     | ((Lambda(_,c1,c2),[]),(Lambda(_,c'1,c'2),[])) -> 
                                          conv_x c1 c'1 & conv_x c2 c'2
     | ((Prod(_,c1,c2),[]),(Prod(_,c'1,c'2),[])) -> 
                                          conv_x c1 c'1 & conv_x c2 c'2
     | ((Const(s1),l1),(Const(s2),l2)) ->
           (eq(s1,s2) & (length(l1) = length(l2)) & (for_all2 conv_x l1 l2))
        or eqappr_x ((Const(s1),l1), apprec l2 (value_of s2))
     | ((Const(s1),l1),p2) ->  
                   eqappr_x(apprec l1 (value_of s1),p2)
     | (p1,(Const(s2),l2))  -> 
                   eqappr_x(p1,apprec l2 (value_of s2))
     | ((Ind(stamp1,c1,l1,_,_),l'1),(Ind(stamp2,c2,l2,_,_),l'2)) ->
          stamp1 = stamp2 &
          (conv_x c1 c2) & (for_all2eq conv_x l1 l2) & (for_all2eq conv_x l'1 l'2)
     | ((Construct(i1,c1),l1),(Construct(i2,c2),l2)) ->
               (i1=i2) & (conv_x c1 c2) & (for_all2eq conv_x l1 l2)
     | ((Rec(_,l1,c1),l'1),(Rec(_,l2,c2),l'2)) -> 
         (conv_x c1 c2) & (for_all2eq conv_x l1 l2) & (for_all2eq conv_x l'1 l'2)
     | (Implicit,[]),(Implicit,[]) -> true
     | _ -> false;;

(* alpha conversion : ignore print names *)
let rec eq_constr pair = if eq(pair) then true else
    match pair with
     (Lambda(_,t1,c1),Lambda(_,t2,c2)) -> eq_constr(t1,t2) & eq_constr(c1,c2)
   | (Prod(_,t1,c1),Prod(_,t2,c2))     -> eq_constr(t1,t2) & eq_constr(c1,c2)
   | (App(t1,u1),App(t2,u2))           -> eq_constr(t1,t2) & eq_constr(u1,u2)
   | (Rel p1,Rel p2)                   -> p1=p2
   | (Meta p1,Meta p2)                 -> p1=p2
   | (Var u1,Var u2)                   -> eq(u1,u2)
   | (Const p1,Const p2)               -> eq(p1,p2)
   | (Ind(stamp1,c1,l1,_,_),Ind(stamp2,c2,l2,_,_)) -> stamp1 = stamp2 &
                                          eq_constr(c1,c2) &
                                          for_all2eq (curry eq_constr) l1 l2
   | (Construct(i,c1),Construct(j,c2)) -> (i=j) & eq_constr(c1,c2)
   | (Rec(_,l1,c1),Rec(_,l2,c2))       -> eq_constr(c1,c2) &
                                          for_all2eq (curry eq_constr) l1 l2
   | _                                 -> false;;

(* substitute (rel 1) for all occurrences of a term c in a closed term t *)

let subst_term  = substrec 1
where rec substrec k c t = 
   if eq_constr(t,c) then Rel(k) else match t with
     App(t1,t2)      -> App(substrec k c t1,substrec k c t2)
  |  Lambda(n,t1,t2) -> Lambda(n,substrec k c t1,substrec (k+1) (lift 1 c) t2)
  |  Prod(n,t1,t2)   -> Prod(n,substrec k c t1,substrec (k+1) (lift 1 c) t2)
  |  Ind(stamp,t,l,s1,s2)    -> Ind(stamp,substrec k c t, map (substrec (k+1)(lift 1 c)) l,
                              s1,s2)
  |  Construct(i,t)  -> Construct(i,substrec k c t)
  |  Rec(b,lf,d)     -> Rec(b,map (substrec k c) lf,substrec k c d)
  |  _               -> t;;

(* Substitute only a list of occurrence locc, the empty list is interpreted
   as substitute all, if 0 is in the list then no substitution is done*)
let subst_term_occ locc c t = if locc = [] then subst_term c t
                              else if mem 0 locc then t
                              else snd (substcheck 1 1 c t)
where rec substcheck k occ c t =
        if exists (function u -> u>=occ) locc then substrec k occ c t
        else (occ,t)
and substrec k occ c t =
   if eq_constr(t,c) then if mem occ locc then (occ+1,Rel(k))
                          else (occ+1,t)
   else match t with
     App(t1,t2)      -> let (nocc1,t1')=substrec k occ c t1 in
                        let (nocc2,t2')=substcheck k nocc1 c t2
                        in nocc2,(App (t1',t2'))
  |  Lambda(n,t1,t2) -> let (nocc1,t1')=substrec k occ c t1 in
                        let (nocc2,t2')=substcheck (k+1) nocc1 (lift 1 c) t2
                        in nocc2,Lambda(n,t1',t2')
  |  Prod(n,t1,t2)   -> let (nocc1,t1')=substrec k occ c t1 in
                        let (nocc2,t2')=substcheck (k+1) nocc1 (lift 1 c) t2
                        in nocc2,Prod(n,t1',t2')
  |  Construct(i,t)  -> let (nocc,t')= substrec k occ c t in
                        nocc,Construct(i,t')
  |  Rec(b,P::lf,d)  -> let (noccP,P')= substrec k occ c P in
                        let (noccd,d')= substcheck k noccP c d in
                        let (noccf,lf') =
                         it_list (fun (nocc',lfd) f ->
                                  let (nocc'',f') = substcheck k nocc' c f in
                                   (nocc'',f'::lfd)) (noccd,[]) lf
                        in noccf,Rec(b,P'::(rev lf'),d')
  |  _               -> occ,t
  ;;

(********************************************************************)

(*                      Inductive types                 *)

(********************************************************************)

let type_construct i ind = let (_,_,lc,_,_) = check_inductype_spec ind in
            subst1 ind
                (try nth lc i
                 with Failure _ -> error "Not such a constructor");;

let rec hdchar = 
let lower_case c = let n=ascii_code(c) in if n<97 then ascii (n+32) else c
in function
     Prod(_,_,c)           -> hdchar c
   | App(c,_)              -> hdchar c
   | Lambda(_,_,c)         -> hdchar c
   | Const(Def(Name(s),_,_)) -> lower_case(hd(explode_id s))
   | Var(Decl(Name(s),_,_))  -> lower_case(hd(explode_id s))
   | Prop(_)               -> "P"
   | Type(_,_)             -> "T"
   | _                     -> "y";;
   
let named_hd A = function Anonymous -> Name(id_of_string(hdchar A)) 
                        | x         -> x;;

(* For manipulation of lists of terms *)

let prod_it = it_list (fun c (n,t)  ->  Prod(n,t,c))
and arrow c1 c2 = Prod(Anonymous,c1,c2);;
let arrow_it = list_it arrow;;
let prod_name (n,A,B) = Prod(named_hd A n,A,B)
and lambda_name (n,A,B) = Lambda(named_hd A n,A,B)
and prod_create (A,B) = Prod(Name(id_of_string(hdchar A)),A,B)
and lambda_create (A,B) =  Lambda (Name(id_of_string(hdchar A)),A,B);;
let lambda_implicit A = Lambda(Name(id_of_string"y"),Implicit,A);;

(* compute the type of a well-formed expression *)

let rec type_wfconstr l = function
  Var(Decl(_,Judge(t,_,_),_))  -> t
| Const(Def(_,Judge(_,t,_),_)) -> t
| Rel(k)        -> lift k (nth l k)
| App(c1,c2)    -> (match hnftype (type_wfconstr l c1) with
          Prod(_,u1,u2) -> simplify (subst1 c2 u2)
        | _             -> anomaly "Not well-typed 1")
| Ind(_,c,_,_,_)  -> c
| Construct(i,c)-> type_construct i c
| Rec(b,P::_,c) -> let ct = type_wfconstr l c in
                       (try let (_,la) = find_inductype ct
                            in let p = applist P la 
                               in if b then App(p,c) else p
                        with Induc -> anomaly "Not Well typed 3")
| Prod(_,c1,c2) -> (let sort = type_wfconstr (c1::l) c2 
                    in match hnftype sort with
                         (Prop(_) | Type(_,_)) -> sort
                       | _ -> error "Not a proposition or a type")
| Lambda(name,c1,c2) -> Prod(name,c1,type_wfconstr (c1::l) c2)
| Prop(cst)      -> Type(cst,Dummy_univ)
| (Type(_,_) as t) -> t      (* Here we use Type:Type without danger *)
| _              -> error "Not a closed term";;

exception TypeRestr;;

let decomp_prod = decrec 0
  where rec decrec m c =
   match hnftype c with
     (Prop(_) | Type(_,_) as x) -> m,x
   | Prod(n,A,C)            -> decrec (m+1) C
   | _                      -> anomaly "hnftype";;

let kind_of = snd o decomp_prod;;

let lev_of_ind ind = let (_,x,_,_,_) = inductype_spec ind in
  match decomp_prod x with (_,Prop(_)) -> Proof | _ -> Object;;

let make_arity_dep k ar = mrec(lift 1 ar)
  where rec mrec c M = match hnftype c with 
    (Prop(_) | Type(_,_)) -> arrow M k
  | Prod(n,B,C)         -> prod_name(n,B,mrec C (App(lift 1 M,Rel 1)))
  | _                   -> anomaly "hnftype";;

let make_arity_nodep k = mrec 
 where rec mrec c = match hnftype c with 
       (Prop(_) | Type(_,_)) -> k
     | Prod(x,B,C)         -> Prod(x,B,mrec C)
     | _                   -> anomaly "hnftype";;


(*********************************************************************
* Let B be such that X,Delta(X)|-B(X)                    *
* let k = |Delta(X)|+1                                               *
* put B(X), if possible, as (x1:B1)...(xp:Bp)(X u1..uk)              *
* raise Norec if X does not occur in B,                              *
* error in the other cases.                                          *
*********************************************************************)

exception Norec;;

let rec make_pos k =  function  
   Prod(n,A1,A2) -> if noccurn k A1 then Prod(n,A1,make_pos (k+1) A2)
            else error "Non strictly positive type 1"
  | x            -> let (hd,largs) = reduce x in
          (match hd with 
              Prod(_,_,_) -> make_pos k hd
            | (Rel k') -> if for_all (noccurn k) largs then 
                    if k=k' then applist hd largs
                        else raise Norec
                          else error "Non strictly positive type 2"
            |  _       -> if for_all (noccurn k) (hd::largs) then raise Norec
              else error "Non strictly positive type 3");;

(***************************************************************************
* Let B be such that X,Delta(X)|-B(X) strictly positive, we assume that    *
*    B(X)=(x1:B1)...(xp:Bp)(X u1..uk)                      *
*    let u such that X,Delta(X)|-u:B(X),                       *
*     compute X,P,Delta(X)|-B(P,u)=(x1:B1)...(xp:Bp)(P u1..uk (u x1..xp))  *
*       and   X,F,Delta(X)|-B(F,u)=[x1:B1]...[xp:Bp](F u1..uk (u x1..xp))  *
*                                  :B(P,u)                                 *
***************************************************************************)

let rec transf_posP u = function 
    Prod(n,A1,A2) -> prod_name(n,A1,transf_posP (App(lift 1 u,Rel 1)) A2)
  | x             -> App(x,u);;

let rec transf_posF u = function
    Prod(n,A1,A2) -> lambda_name(n,A1,transf_posF (App(lift 1 u,Rel 1)) A2)
  | x             -> App(x,u);;


(***************************************************************************)
(* Extension of operations to constructors                                 *)
(***************************************************************************)

(***********************************************************************
* Let C be a constructor G,X|-C(X).                                    *
*                                                                      *
* C(X) reduces to (x1:C1(X))..(xn:Cn(X))(X u1..uk)                     *
* put C(X), if possible, as                                            *
* G,X,a,P |- {x1,B1(X),B1(P)+},..{xi,Bi},..{xn,Bn(X),Bn(P)+}(P u1..uk).*
************************************************************************)

type constructor =
      Recarg of name * constr * constr * constructor
    | Constarg of name * constr * constructor
    | Head of constr;;

let make_constr = mrec 1
 where rec mrec k = function
    Prod(n,B,C) ->
        (try (let B' = (make_pos k B) in
              Recarg(n,liftn 2 B k,lift 1 (liftn 2 B' (k+1)),mrec (k+1) C))
         with Norec -> Constarg(n,liftn 2 B k,mrec (k+1) C))
  | x           -> let (hd,largs) = reduce x in
       (match hd with
              Prod(_) -> mrec k hd
            | (Rel k') -> if for_all (noccurn k) largs then
                            if k=k' then
                 Head(applist hd (map (function c -> liftn 2 c k) largs))
                                else error "Not a constructor 1"
                          else error "Not a constructor 2"
            |  _       -> error "Not a constructor 3");;


let rec constant_constr = function
             Recarg(_,_,_,_) -> false
           | Constarg(_,_,c) -> constant_constr(c)
           | Head(_)         -> true;;

(*************************************************************************
* Let C be a constructor of X. X,Delta(X)|-C(X).
* C(X)=(x1:B1(X))..(xn:Bn(X))(X u1..uk).
*
* let u such that X,Delta(X)|-u:C(X)
*   compute
      X,a,P,Delta+(X)|-C(P,u)=(x1:B1(X)){B1(P,x1)}->..(P u1..uk (u x1...xn))
*     X,P,Delta+(X)  |-C(P)=(x1:B1(X)){B1(P)}->..(P u1..uk)
* and let f such that Delta(X)|-f:C(P,u)
*   compute F,Delta(X)|-C(F,u):(x1:B1(X))..(xn:Bn(X))(P u1..uk (u x1..xn))
***************************************************************************)

let rec transf_constrP_dep u = function
    Recarg(n,BX,B,C) ->
            prod_name(n,BX,
              arrow (transf_posP (Rel 1) B)
                    (lift 1 (transf_constrP_dep (App(lift 1 u,Rel 1)) C)))
  | Constarg(n,B,C)  -> prod_name (n,B,
                             transf_constrP_dep (App(lift 1 u,Rel 1)) C)
  | Head(c)          -> App(c,u);;

let rec transf_constrP_nodep = function
    Recarg(n,BX,B,C) -> Prod(n,BX,
                             arrow B (lift 1 (transf_constrP_nodep C)))
  | Constarg(n,B,C)  -> Prod(n,B,transf_constrP_nodep C)
  | Head(c)          -> c;;

let rec transf_constrF f = function
   Recarg(n,_,B,C) -> lambda_implicit (transf_constrF
                        (App (App (lift 1 f,Rel 1),transf_posF (Rel 1) B)) C)
 | Constarg(n,_,C) ->
      lambda_implicit (transf_constrF (App (lift 1 f,Rel 1)) C)
 | Head(_)         -> f;;
  

(************************************************************************
* let C be a constructor of X. X|-C(X).                 *
* C(X)=(x1:B1(X))..(xn:Bn(X))(X u1..uk).                *
* compute X|-[u:C(X)][P:C(X)]C(P,u) and X|-[P]C(P)                      *
*************************************************************************)

let check_constr l = crec (Implicit::Implicit::Implicit::l)
  where rec crec l = function
    Constarg(n,A,C) -> (match hnftype (type_wfconstr l A) 
                        with Prop(_) -> ()
                           | _       -> raise TypeRestr);
                       crec (A::l) C
  | Recarg(n,A,_,C) -> crec (A::l) C
  | _               -> ();;

let allowed_sorts lt lc = function
   Type(Pos,_)   -> [prop;spec;typep;types],[prop;spec;typep;types]
 | Type(Null,_)   -> [prop;typep],[prop;typep]
 | Type(Data,_)   -> error "Cannot build an inductive definition on Type_Data"
 | Prop(cts) -> let l = (try do_list (check_constr lt) lc;[typep]
                         with TypeRestr -> [])
                in (match cts with
                     Null -> [],[prop]
                   | Pos  -> prop::spec::l,prop::spec::l
                   | Data -> [],[data])
 | _         -> anomaly "Not a sort";;

let arg_dep C = lambda_implicit(lambda_implicit(transf_constrP_dep (Rel 2) C));;

let arg_nodep C = lambda_implicit (liftn (-1) (transf_constrP_nodep C) 2);;

let reduce_arg C = Lambda(Name(id_of_string"fi"),Implicit,
                          Lambda(Name(id_of_string"F"),Implicit,
                                 transf_constrF (Rel 2) C));;

let make_ind_rel stamp lt t lc =
    let lC = map make_constr lc and (n,k) = decomp_prod t in
    let kd,kn = allowed_sorts lt lC k 
    in let lpd = if kd = [] then [] else map arg_dep lC
       and lpn = if kn = [] then [] else map arg_nodep lC
       and lf = mrec 1 [] lC where rec mrec k lfres =
            function C::lCrest ->
                     if constant_constr C then mrec (k+1) lfres lCrest
                     else mrec (k+1) ((k,reduce_arg C)::lfres) lCrest
                   | []        -> lfres
    in Ind(stamp,t,lc,Specifr(n,lf),Specift((kd,lpd),(kn,lpn)));;

let make_ind st = make_ind_rel st [];;

(************************************************************************)
(*                                  *)
(* Type checking of non necessary closed inductive types                *)
(*                                  *)
(************************************************************************)

let wf_ind lt = function
 Ind(stamp,t1,l1,Specifr(n1,lf1),Specift((ad1,lpd1),(an1,lpn1))) ->
  (match (make_ind_rel stamp lt t1 l1) with
     Ind(_,_,_,Specifr(n2,lf2),Specift((ad2,lpd2),(an2,lpn2))) ->
         (n1=n2) & (for_all2eq (fun (_,c1) (_,c2) -> conv_x c1 c2) lf1 lf2)
       & (for_all2eq conv_x ad1 ad2) & (for_all2eq conv_x an1 an2)
       & (for_all2eq conv_x lpd1 lpd2) & (for_all2eq conv_x lpn1 lpn2)
   | _ -> anomaly "make_ind")
| _ -> anomaly "wf_ind";;

(************************************************************************)
(*                                                                      *)
(* Type checking of elimination                                         *)
(*                                                                      *)
(************************************************************************)

let type_elim_arg_dep ind P lpd i ci =
    let fi = subst1 ind (try nth lpd i with Failure _ -> anomaly "No such arg")
    in redapplist [ci;P] fi;;

let type_elim_arg_nodep ind P lpn i =
    let fi = subst1 ind (try nth lpn i with Failure _ -> anomaly "No such arg")
    in redapplist [P] fi;;

let type_elim_args_dep ind P =
    map2 (fun ci pdi -> redapplist [ci;P] (subst1 ind pdi));;

let ind_constructs ind n = crec n []
    where rec crec k l = if k = 0 then l
                         else  crec (k-1) (Construct(k,ind)::l);;

let type_elim_args_dep_constr ind P =
   map_i (fun i pdi -> redapplist [Construct(i,ind);P] (subst1 ind pdi)) 1;;

let type_elim_args_nodep ind P
        = map (fun pn -> redapplist [P] (subst1 ind pn));;


(***********************************************************************)
(* Type inference of elimination argument                              *)
(***********************************************************************)
(****************************************************)
(* Type of elimination                              *)
(****************************************************)


let is_correct_arity_dep kd = srec where rec srec ind (Pt,t) =
  match hnftype Pt,hnftype t with
    Prod(_,A1,A2),Prod(_,A1',A2') -> conv_x A1 A1' &
                                     srec (App(lift 1 ind,Rel 1)) (A2,A2')
  | Prod(_,A1,A2),_               -> conv_x A1 ind & exists (conv_x A2) kd
  | _                             -> false;;

let is_correct_arity_nodep kd = srec where rec srec (Pt,t) =
  match hnftype Pt,hnftype t with
    Prod(_,A1,A2),Prod(_,A1',A2') -> conv_x A1 A1' & srec (A2,A2')
  | Prod(_,A1,A2),_               -> false
  | (_,Prod(_,_,_))               -> false
  | (k,_)                         -> exists (conv_x k) kd;;

exception Arity;;
 
let is_correct_arity kd kn = srec 
 where rec srec ind (Pt,t) =
     match hnftype Pt,hnftype t with
    Prod(_,A1,A2),Prod(_,A1',A2') 
          -> if conv_x A1 A1' then srec (App(lift 1 ind,Rel 1)) (A2,A2') 
             else raise Arity
  | Prod(_,A1,A2),_              
          -> let k = hnftype A2 in 
             if (conv_x A1 ind)&(exists (conv_x k) kd)  then 
                                        true,level_of_kind k
             else raise Arity
  | k,Prod(_,_,_) -> raise Arity
  | k,_       -> if exists (conv_x k) kn then false,level_of_kind k
                 else raise Arity;;

let type_elim ct lpft b P c =
  try
    let ((stamp,t,lc,x,(Specift((kd,lpd),(kn,lpn)) as s) as ind),la) = find_inductype(ct)
    in let i = Ind(stamp,t,lc,x,s) in
        (match lpft with 
            []       -> error "Ill-formed elimination 1"
          | (Pt::lf) -> if b then
             (if (is_correct_arity_dep kd i (Pt,t)) &
                 (for_all2eq conv_x (type_elim_args_dep_constr i P lpd) lf)
              then App(applist P la,c) else error "Ill-typed elimination 1")
                        else
             (if (is_correct_arity_nodep kn (Pt,t)) &
                 (for_all2eq conv_x (type_elim_args_nodep i P lpn) lf)
              then applist P la else error "Ill-typed elimination 2"))
  with Induc -> error "Ill-formed elimination 2";;

(*****************************************************)
(* Type checking of constructions                    *)
(*****************************************************)

let rec type_constr l = function
  Var(Decl(_,Judge(t,_,_),_))  -> t
| Const(Def(_,Judge(_,t,_),_)) -> t
| Rel(k)    -> lift k (nth l k)
| App(c1,c2)    -> (match hnftype (type_constr l c1) with
          Prod(_,u1,u2) -> if conv_x (type_constr l c2) u1 (* Type : Type *)
                           then simplify (subst1 c2 u2)
                           else error "Ill-typed 1"
    | _     -> error "Ill-typed 2")
| (Ind(_,c,_,_,_) as ind) -> if wf_ind l ind then c
                           else error "Ill-formed induction"
| Construct(i,c)-> type_construct i c
(* Warning it is not a typechecking *)
| Rec(b,lpf,c) -> (match lpf with 
                     []   -> error "Ill-formed elimination"
                   | P::_ -> let ct = type_constr l c
                             and lpft = map (type_constr l) lpf 
                             in type_elim ct lpft b P c)
| Prod(_,c1,c2) -> (level_of_kind (type_constr l c1);
                    (let sort = type_constr (c1::l) c2 
                     in match hnftype sort with
                         (Prop(_) | Type(_,_)) -> sort
                       | _ -> error "Not a proposition or a type"))
| Lambda(name,c1,c2) -> (level_of_kind (type_constr l c1);
                         Prod(name,c1,type_constr (c1::l) c2))
| Prop(cst)  -> Type(cst,Dummy_univ)
| (Type(_,_) as t) -> t      (* Here we use Type:Type without danger *)
| _      -> error "Not a closed term";;

let type_of = type_constr [];;

let is_induc c = let (_,l) = find_inductype c in null(l);;


let one_step_reduce = redrec []
where rec redrec largs = function
   Lambda(n,t,c)  -> (match largs with
                      []      -> error "Not reducible 1"
                    | a::rest -> applist (subst1 a c) rest)
 | App(c1,c2)     -> redrec (c2::largs) c1
 | Const(c)       -> (try let (c',l) = red_elim_const c largs
                          in applist c' l
                      with Redelimination -> applist (value_of c) largs)
 | Rec(r1,r2,r3)  -> (try  applist (red_elimination (r1,r2,r3)) largs
                      with Redelimination -> error "Not reducible 2")
 | _              -> error "Not reducible 3";;


(************************************************************************)
(* find_induc_comb c = ind,l if c = applist ind l and ind is an Ind{}{} *)
(* try to reduce c as least as possible                                 *)
(************************************************************************)

let find_induc_comb c =
  let rec findrec l = (findl
               where rec findl c =
               if is_induc c then c,l
               else match c with
                      App(c1,c2) -> (try findrec (c2::l) c1
                                     with Induc -> findl(one_step_reduce c))
                    | _          -> findl(one_step_reduce c))
 in try findrec [] c with Induc -> error "Not an Inductive Definition"
 ;;


(*******************************************)
(* Building curryfied elimination          *)
(*******************************************)

(*********************************************)
(* lc is the list of the constructors of ind *)
(*********************************************)
(* rel_list n m = [Rel(n+m);...Rel(n+1)] *)

let rel_list n m = relrec [] 1
  where rec relrec l k = if k>m then l
                         else relrec (Rel (n+k)::l) (k+1);;

let make_elim_dep ind kind lc =
 let (_,t,_,Specifr(n,_),Specift((kd,lpd),_)) = inductype_spec ind
 in if (exists (conv_x kind) kd) then
       let ar = make_arity_dep kind t ind
       in Lambda(Name(id_of_string"P"),ar,mkfrec 1 (lpd,lc))
          where rec mkfrec k = function
             (fti::lpdrest,ci::lcrest)
                   -> Lambda(Name(id_of_string"f"),redapplist [ci;Rel k] (subst1 ind fti),
                             mkfrec (k+1) (lpdrest,lcrest))
           | [],[] -> (mkelimabs k ar 
                       where mkelimabs k = mkelim
                                           where rec mkelim = (function
                          Prod(name,A,B) -> lambda_name(name,A,mkelim B)
                        | _              -> Rec(true,rel_list (n+1) k,Rel 1)))
           | _ -> anomaly "make_elim_dep"
    else error "Dependent elimination not allowed on such kind";;


let make_elim_nodep ind k =
 let (_,t,_,Specifr(n,_),Specift(_,(kd,lpd))) = inductype_spec ind
 in if (exists (conv_x k) kd) then
       let ar = make_arity_nodep k t
       in lambda_create(ar,mkfrec 1 lpd)
          where rec mkfrec k = function
            fti::lpdrest -> Lambda(Name(id_of_string"f"),redapplist [Rel k] (subst1 ind fti),
                                   mkfrec (k+1) lpdrest)
          | [] -> mkelimabs k ar 
                  where mkelimabs k = mkelim
                                      where rec mkelim = function
          Prod(name,A,B) -> lambda_name(name,A,mkelim B)
        | _              -> lambda_create(applist ind (rel_list 0 n),
                                          Rec(false,rel_list (n+1) k,Rel 1))
     else error "Elimination not allowed on such kind";;

let correct_arities kd kn ind t =
    (map (function s -> make_arity_dep s t ind) kd)@
    (map (function s -> make_arity_nodep s t) kn);;
 
let arity_error kd kn ind P Pt t = 
 PPNL[< 'S "The elimination predicate"; 'SPC ; prterm.v P; 'SPC ;
        'S "has type" ; 'SPC; prterm.v Pt; 'S ";";
        'CUT; 'S "it should be ";
         'S (if (length kd)+(length kn) > 1 then "one of : " else ": ");
         HOV 0 [< prlist_with_sep (function () -> [< 'S ","; 'SPC >])
                  prterm.v (correct_arities kd kn ind t)>]; 'S ".";
        'CUT >];
        error "Not a correct arity";;
 
let make_elim c typc P typP =    
  let (ind,la) = find_induc_comb typc
  in let (_,t,_,_,Specift((kd,lpd),(kn,lpn))) = inductype_spec ind
     in let (b,lev) = 
        (try is_correct_arity kd kn ind (typP,t)
         with Arity -> arity_error kd kn ind P typP t)
        and Pl = applist P la 
        in let (n,lft,tye) =     
               if b then (length lpd,type_elim_args_dep_constr ind P lpd,
                          App(Pl,c))
               else (length lpn,type_elim_args_nodep ind P lpn,Pl)
           in list_it (fun fit (e,te) -> (Lambda(Name(id_of_string"f"),fit,e),arrow fit te))
                      lft (Rec(b,P::(rel_list 0 n),c),tye) , lev;;

let make_elim_val c P = 
    let ((v,_),_) = make_elim c (type_of c) P (type_of P) in v;;
 
let find_except f = frec []
 where rec frec deb = function 
     []   -> failwith "find_except"
   | a::m -> if f(a) then a,deb@m
             else frec (a::deb) m;;

let possible_elim c = let (_,_,_,_,Specift((kd,_),(kn,_))) = inductype_spec c
                      in kd,kn;;

(**************************************************************************)
(*            Types for synthesis                                         *)
(**************************************************************************)


(*type signature == (name * constr) list;;

type goal == (int list) * signature * constr;;

type prooftree =
    INTRO of name * constr * prooftree
 |  APP of prooftree list
 |  PF of constr
 |  INCOMPLET of goal;;

type tactic == goal -> prooftree;;*)
