(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                             tactics.ml                                   *)
(****************************************************************************)

#infix "o";;
#open "std";;
#open "initial";;
#open "univ";;
#open "extraction";;
#open "term";;
#open "search";;
#open "printer";;
#open "machine";;
#open "matching";;
#open "trad";;
#open "command";;
#open "more_util";;
#open "pp";;
#open "stdpp";;

let toplevel_message s = if not(is_silent()) then message s ;;

(***********************)
(* Partial proof trees *)
(***********************)

let top_thm_goal = function
   PF c -> c
 | _    -> error "Proof not complete";;

let init_goals() = INCOMPLET([],[],prop);;

let app_simpl_pf  = 
 let rec simplrec c = function
     (PF x)::rest -> simplrec (App(c,x)) rest
    | []          -> (PF c)
    | l           -> APP(PF(c)::l)
 in function PF(c)::l -> simplrec c l
           | l        -> APP(l);;

let intro_simpl_pf s c = function  PF c1 -> PF (lambda s c c1)
                                |  x     -> INTRO(s,c,x);;

let rec simpl_pf = function
   INTRO(s,c,pf) -> intro_simpl_pf s c (simpl_pf pf)
 | APP(l)        -> app_simpl_pf (map simpl_pf l)
 | p             -> p;;

(* Hypothesis: substitution s is valid in signature sign recursively *)
(* This must be true when called from exact *)
let inst_pf s pf = simpl_pf (irec pf)
 where rec irec = function
    INTRO(s',c,pf)      -> INTRO(s',c,irec pf)
  | APP(l)              -> APP(map irec l)
  | (PF _ as u)         -> u
  | INCOMPLET(k,sign,c) -> (try let c' = assoc k s in
               if not (closed c') then error "Instantiation not closed"
               else if conv_x (instance s c) (type_of c') then PF c'
               else error "Instantiation not well-typed"
                            with Not_found -> INCOMPLET(k,sign,instance s c));;

(* A system of routines to maintain goals and partial results 

Operations:     
   goal g           state the top-level goal 
   save_thm name    call save_thm (name, top of thmstack) 
   undo ()      undo last proof step (may be repeated)  *)

(* List all incomplete leaves *)

(* list_pf : prooftree -> goal list *)
let rec list_pf = function
   INCOMPLET(g1,g2,g3) -> [g1,g2,g3]
 | INTRO(_,_,pf) -> list_pf pf
 | APP(l)        -> flat (map list_pf l)
 | _             -> [];;

(* Apply the tactic tac to all incomplete leaves *)

(* apply_tac : tactic -> prooftree -> prooftree *)
let apply_tac tac pf = simpl_pf (arec pf)
 where rec arec = function
   INTRO(s,c,pf) -> INTRO(s,c,arec pf)
 | APP(l)        -> APP(map arec l)
 | INCOMPLET(g1,g2,g3) -> tac(g1,g2,g3)
 | pf            -> pf;;

let selective_apply_tac taclist pf = 
 let TL = ref taclist in 
 let rec arec = function
   INTRO(s,c,pf) -> INTRO(s,c,arec pf)
 | APP(l)        -> APP(map arec l)
 | INCOMPLET(k,s,cl) -> 
    let tac = try (hd !TL)
              with _ -> error "More remaining subgoals than provided tactics"
      in TL:=tl !TL; tac (k,s,cl)
 | pf            -> pf in
 let res = arec pf in
 if !TL=[] then simpl_pf res 
   else error "Less remaining subgoals than provided tactics";;

(* same as selective_apply_tac but does not require the same number of
subgoals and tactics, applies the last tactic to all remaining subgoals 
if any *)

let select_apply_tac taclist pf = 
 let TL = ref taclist in 
 if taclist = [] then pf else 
 let DEF = ref (hd taclist) in 
 let rec arec = function
   INTRO(s,c,pf) -> INTRO(s,c,arec pf)
 | APP(l)        -> APP(map arec l)
 | INCOMPLET(k,s,cl) -> 
    let tac = (try let t = hd !TL in DEF:=t; TL:=(tl !TL); t 
               with Failure _ -> !DEF)
               in tac (k,s,cl)
 | pf            -> pf 
 in simpl_pf (arec pf);;


let rec occur_meta = function
   Prod(_,t,c)   -> (occur_meta t) or (occur_meta c)
 | Lambda(_,t,c) -> (occur_meta t) or (occur_meta c)
 | App(c1,c2)    -> (occur_meta c1) or (occur_meta c2)
 | Meta(_)       -> true
 | _             -> false;;

(* Expand the kth subgoal using the tactic *)
(* Warning this is not correct for dependent types *)
(* inst_pf ought to be used (except that pf1 is pf and not a constr) *)
let subst_pf u pf1 pf = simpl_pf (srec pf)
 where rec srec = function
    INTRO(s,c,pf) -> INTRO(s,c,srec pf)
 |  APP(l) -> APP(map srec l)
 |  (PF _ as pf) -> pf
 |  (INCOMPLET(k,_,cl) as pf) ->
     if k=u then if occur_meta cl then error "Open subgoal : use Instantiate"
                 else pf1
     else pf;;

(* This renames bound variablew with fresh and distinct names *)
(* in such a way that the printer doe not generate new names  *)
(* and therefore that printed names are the intern names      *)
(* In this way, tactics such as Induction works well          *)

let rec rename_bound_var l = function
   Prod(Name(s),c1,c2)  -> if dependent 1 c2 then
               let s' = next_ident_away s (globals c2@l)
                in Prod(Name(s'),c1,rename_bound_var (s'::l) c2)
                     else Prod(Name(s),c1,rename_bound_var l c2)
 | Prod(Anonymous,c1,c2) -> Prod(Anonymous,c1,rename_bound_var l c2)
 |  x -> x;;

let rename_bound_var_pf =
  let names_of sign = it_list (fun l -> (fun (Name s,_) -> s::l
                               | _ -> anomaly "rename_bound_var_pf"))
                [] sign
   in ren_rec
  where rec ren_rec = function
    INTRO(s,c,pf) -> INTRO(s,c,ren_rec pf)
 |  APP(l) -> APP(map ren_rec l)
 |  (PF _ as pf) -> pf
 |  INCOMPLET(k,sign,cl) ->
      INCOMPLET(k,sign,rename_bound_var (names_of sign) cl);;


(* Apply a tactic to a subgoal, fails if the subgoal contains a Meta *)

let solve_pf k tac pf = 
let (u,sign,cl) = try nth (list_pf pf) k with Failure _ -> error "No such goal"
in  subst_pf u (rename_bound_var_pf (tac (u,sign,cl))) pf;;

let solve_last tac pf =
let (u,sign,cl) = try last (list_pf pf)
  with Failure _ ->  error "No more subgoals"
     in subst_pf u (rename_bound_var_pf (tac (u,sign,cl))) pf;;

(* General tactics and tactical *)

let IDTAC(g1,g2,g3) = INCOMPLET(g1,g2,g3);;

#infix "ORELSE";;
let prefix ORELSE (T1:tactic) (T2:tactic) = 
    function g -> try T1 g 
                  with UserError _ -> T2 g;;

#infix "NEXT";;
let prefix NEXT (T:tactic) (f:(prooftree -> prooftree)) =
    function g -> f (T g);;

(* apply T1 then T2 to all subgoals generated by T1 *)
#infix "THEN";;
let prefix THEN T1 T2 = T1 NEXT (apply_tac T2);;

(* apply T1 then T2 to the first subgoal generated by T1 *)
#infix "THENF";;
let prefix THENF T1 T2 = T1 NEXT (solve_pf 1 T2);;

(* apply T1 then T2 to the last subgoal generated by T1 *)
#infix "THENL";;
let prefix THENL T1 T2 = T1 NEXT (solve_last T2);;

(* apply T1 then T2 to the last subgoal generated by T1 *)
#infix "THENS";;
let prefix THENS T1 TL = T1 NEXT (selective_apply_tac TL);;

#infix "THENSS";;
let prefix THENSS T1 TL = T1 NEXT (select_apply_tac TL);;

(* Beware: call by need of CAML, g is needed *)

let rec REPEAT = fun T g ->
     ((T THEN (REPEAT T)) ORELSE IDTAC) g;;

(* Try the first tactic that does not fail in a list of tactics *)

let rec FIRST = fun tacl g ->
 match tacl with
    []      -> error "FIRST : no applicable tactic"
 |  T::rest -> (try T g with UserError _ -> FIRST rest g);;

let TRY T = T ORELSE IDTAC;;

let COMPLETE (T:tactic) g =
match simpl_pf (T g) with 
   (PF _ as u) -> u
 | _           -> error "COMPLETE expects a proof";;
    
let AT_LEAST_ONE T = T THEN REPEAT T;;

let (FAILTAC:tactic) = fun _ -> error "FAILTAC";;

(* Iteration tactical *)
let DO n t = dorec n 
   where rec dorec k = 
          if k < 1 then error "DO"
          else if k = 1 then t else t THEN (dorec (k-1));;

(****************************************************************)
(* General functions on the current prooftree                   *)
(****************************************************************)

let GOALS = ref(init_goals():prooftree) ;;
let get_goals() = !GOALS;;
let get_goal() = hd (list_pf (get_goals()));;

let UNDO = ref([] : prooftree list);;
let undo_limit = ref(12);;

type refine_status = NO_GOAL
                   | PROVING of goal;;

let REFINE = ref(NO_GOAL);; (* top-down proof-search status *)
let refining () = match !REFINE with NO_GOAL -> false | _ -> true;;

type record_status = NO_RECORD 
                   | RECORDING of out_channel;;

(* History mechanism : we keep lists of commands of the user (in the form
   of strings of his concrete syntax, in order to record them on file.
   We maintain two lists :
    HIST_CONT contains global contexts additions during a proof development
   and HIST_PROOF contains proof development steps *)

(* will be recording current developments *)
let RECORD = ref(NO_RECORD);; 

let HIST_CONT  = ref([] : string list)
and HIST_PROOF = ref([] : string list);;

let open_vernacular filename = 
    match !RECORD with
         RECORDING _  -> error "Already recording"
       | NO_RECORD    -> 
(*   if refining() then error "Must save or abort current goal first"; *)
         RECORD := RECORDING(open_trapping_failure open_append filename ".v");
             ();;

(* type record_mode = Context | Delayed | Flush | Invisible;; *)

(* Delayed: standard option : we record in history squence 
   Flush: we record and flush the history 
   Invisible: no record, interactive command
   Context: context record; immediate if not refining *)

let RECORDING_CODE = ref(Delayed);;

let set_record code = RECORDING_CODE := code; ();;

let record_line str len = 
    match !RECORD with 
      NO_RECORD -> ()
    | RECORDING chan -> let str = sub_string str 0 len in
            match !RECORDING_CODE with
              Invisible -> ()
            | Context   -> if refining() then (HIST_CONT := str::!HIST_CONT;())
                                         else (
                                          output_string chan str; flush chan)
            | Delayed   -> HIST_PROOF := str::!HIST_PROOF; ()
            | Flush     -> HIST_PROOF := str::!HIST_PROOF; 
                           do_list (output_string chan) 
                                 ((rev !HIST_CONT)@(rev !HIST_PROOF)); 
                           HIST_CONT := []; 
                           HIST_PROOF := []; 
                           flush chan;;

let close_vernacular () = 
    match !RECORD with
         NO_RECORD      -> error "No pending recording"
       | RECORDING chan -> set_record Flush;
                           record_line "\n" 1;
                           close_out chan;
                           RECORD := NO_RECORD;
                           ();;

let change_state newgoals = 
let newundo = try fst (chop_list (!undo_limit) (!UNDO))
              with _ -> !UNDO (* dangerous *)
in UNDO := !GOALS::newundo; GOALS := newgoals; ();;

let reset_goals () = 
   GOALS := init_goals(); 
   UNDO := []; 
   REFINE := NO_GOAL;
   ();;

let set_goal g = change_state(IDTAC g); UNDO := []; ();;

let reset_to_goal () = 
    match !REFINE with NO_GOAL -> error "No current goal"
                     | PROVING(g) -> set_goal g;
    () ;;

let top_thm () = top_thm_goal (!GOALS) ;;

(* Restore the previous proof states;  discard current state. *)

let undo n =
  let rec undorec p =
    if p>0 then
    ((match !RECORD with
            NO_RECORD -> ()
                  | _ -> match !HIST_PROOF with
                     []   -> ()
                  | _::h  -> HIST_PROOF := h;());
    match !UNDO with
        []            -> if p=n then error "Empty undo list"
                         else warning "Undo list has now become empty"
      | newg::newundo -> GOALS := newg; UNDO := newundo;undorec (p-1))
  in undorec n;;


(*********************************************************************)
(*      Printing functions                                   *)
(*********************************************************************)

let print_goal_default n = if refining() 
               then PPNL(pr_subgoal n (list_pf !GOALS))
               else message "No current goal";;

let PRINT_SUBGOAL = ref [print_goal_default];;

let print_subgoal n = (hd !PRINT_SUBGOAL) n;;

let set_print_goal_fun f = PRINT_SUBGOAL := f::(!PRINT_SUBGOAL); ();;

let reset_print_goal_fun () = PRINT_SUBGOAL := tl (!PRINT_SUBGOAL); ();;

let print_state_verbose () =
    PP (pr_subgoals (list_pf !GOALS)) ;;

let print_state1 () = if not(is_silent()) then
    let n = focus() in 
       if n=0 then PPNL(pr_subgoals (list_pf !GOALS))
       else try (print_subgoal n)
            with UserError "No such goal" -> 
    (message "Focused goal solved";
     if null(list_pf !GOALS) then message "Goal proved!");;

let PRINT_FN = ref [print_state1];;

let set_print_fun f = PRINT_FN := f::(!PRINT_FN); ();;
let reset_print_fun () = PRINT_FN := tl (!PRINT_FN); ();;

let print_state () = (hd !PRINT_FN)() ;;

let show_default () = if refining() then print_state_verbose ()
                            else message "No current goal";;

let SHOW_FN = ref [show_default];;

let set_show_fun f = SHOW_FN := f::(!SHOW_FN); ();;

let reset_show_fun () = SHOW_FN := tl(!SHOW_FN); ();;

let show () = (hd !SHOW_FN)();;

let Undo n = if refining() then (undo n; print_state())
                            else error "No current goal";;

let show_proof () = if refining() then PP(pr(top_thm()))
                            else error "No current goal";;

(*********************************************************************)
(*      Saving  functions                                    *)
(*********************************************************************)

let save_named () =
   let proof_ident = current_section () in 
            execute(top_thm());
            close_section' proof_ident false;
   let ident = current_section () and (Strength n) = pop_scope () in 
            verify(); 
            declare (id_of_string ident) n;
            reset_goals();
            set_record Flush;  (* flush *)
            message(ident ^ " is defined");
            close_section ident false;;
    
let save_anonymous save_ident n =
   let proof_ident = current_section () in 
            execute(top_thm());
            close_section' proof_ident false;
   let ident = current_section () in 
            pop_scope();
            verify();
            if ident = "Unnamed_thm" then
                declare (id_of_string save_ident) n
            else begin
                message("Overriding name " ^ ident ^ " and using " ^ save_ident);
                declare (id_of_string save_ident) n
            end;
            reset_goals();
            set_record Flush;  (* flush *)
            message(save_ident ^ " is defined");
            close_section ident false;;

let save_anonymous_thm id = save_anonymous id 0;;
let save_anonymous_remark id = save_anonymous id (max((read_sec())-2) 0);;

(*********************************************************************)
(*              Abort   functions                                    *)
(*********************************************************************)
 
let abort_goals () = if refining() then
                        ((try raw_reset_section "Unnamed_thm"
                          with (UserError _) -> ());
                        reset_goals();
                         
                         HIST_PROOF := []; 
                         HIST_CONT := [];  (* optional *)
                     message "Current goal aborted")
                     else error "No current goal";;

let restart () = if refining() then
                    (reset_keeping_cast(); reset_to_goal();
                     match !RECORD with
                       NO_RECORD -> ()
                     | _ -> if !HIST_PROOF <> [] 
                                then (HIST_PROOF := [last !HIST_PROOF]; ());
                            HIST_CONT := []; (* optional *)
                     message "Current goal restarted")
                 else error "No current goal";;

let abort_refine f x = 
    if refining() then (abort_goals(); f(x))
       (* used to be: error "Must save or abort current goal first" *)
    else f(x);;

let reset_name = abort_refine raw_reset_name
and reset_keeping_name = abort_refine raw_reset_keeping_name
and save_state = abort_refine raw_save_state
and restore_state = abort_refine raw_restore_state
and reset_all = abort_refine raw_reset_all
and reset_section = abort_refine raw_reset_section;;

let reset_prelude () = restore_state "Prelude"
and reset_initial () = restore_state "Initial";;

    
(*********************************************************************)
(*      Postulate  functions                                    *)
(*********************************************************************)

let cast_to_postulate () =
   let proof_ident = current_section () in 
       if "Proof_of_Unnamed_thm" = proof_ident then
           error "Cannot convert unnamed goal into postulate";
       reset_section proof_ident;
   let ident = current_section () and (Strength n) = pop_scope () in 
   let varname = match ident with "Unnamed_thm" -> Anonymous
                                    | s -> Name (id_of_string s) in
            assume_cast varname n;
            close_section ident true;
            toplevel_message (ident ^ " is assumed");;


(*********************************************************************)
(*              Modifying the current prooftree                      *)
(*********************************************************************)

let goal_default () =
  if refining() then error "Must save or abort current goal first";
  let (Judge(top_val,_,_),_,_) = search_cast() in
  let top_goal = [],[],top_val
  in (set_goal(top_goal); REFINE:=PROVING(top_goal); ());;

let GOAL_FUN = ref [goal_default];;

let set_goal_fun f = GOAL_FUN := f::(!GOAL_FUN);();;
let reset_goal_fun () = GOAL_FUN := tl (!GOAL_FUN);();;
let goal_mode () = (hd !GOAL_FUN) ();;

let solve k tac = 
  if refining() then
         (change_state(solve_pf k tac (get_goals()));print_state())
       else error "No current goal";;
let by = solve 1;;

(*********************************************************************)
(*                             Tactics                   *)
(*********************************************************************)

(*************************************************************)
(*   Auxiliary functions for fonctional abstraction          *)
(*   should be in term or matching ?                 *)
(*************************************************************)

(* takes a substitution s, an open term op and a closed term cl
   try to find a subterm of cl which matches op, if op is just a Meta
   do nothing because of the lack of type checking of meta variables *)

let match_subterm s =
    function (Meta(_) as op),_ -> s,op
            | op,cl            -> matchrec cl
    where rec matchrec cl =
    try if closed cl then (matching s op cl,cl) 
                     else error "Bound 1"
    with UserError(_)->
        match cl with App(c1,c2) ->
            (try matchrec c1 with UserError(_) -> matchrec c2)
                    | Prod(_,t,c) ->
            (try matchrec t with UserError(_) -> matchrec c)
                    | Lambda(_,t,c) ->
            (try matchrec t with UserError(_) -> matchrec c)
                    | _ -> error "Match_subterm";;

let match_subterm_list si oplist t = 
    list_it (fun op (s,l) -> 
                if occur_meta op then 
                    let (s',cl) = match_subterm s (op,t) in (s',cl::l)
                    else (s,op::l)) oplist (si,[]);;


let decomp_app c = decomprec (c,[])
where rec decomprec = function
             App(t1,t2),l -> decomprec (t1,t2::l)
           | x,l          -> (x,l);;

let args_typ = decrec []
    where rec decrec l c = match hnftype c with
             Prod(n,A,B) -> decrec ((named_hd A n,A)::l) B
           | x           -> l;;

(* if lname_type typ is [xn,An;..;x1,A1] and lists is a list of terms,
gives [x1:A1]..[xn:An]c' such that c converts to ([x1:A1]..[xn:An]c' l) *)

let abstract_scheme c l lname_typ =
    it_list2 (fun t (locc,a) (na,ta) ->
    if occur_meta ta then error "abstract_list"
    else if occur_meta a then Lambda(na,ta,t)
         else Lambda(na,ta,subst_term_occ locc a t)) c (rev l) lname_typ;;

(* non-used 
let abstract_list typ c l =  
  let P = abstract_scheme c l (args_typ typ) in
  if conv_x (type_of P) typ then P else 
     error "Cannot find a well typed generalisation of the goal";; 
*)

let abstract_list_all typ c l =
  let P = abstract_scheme c (map (function a -> [],a) l) (args_typ typ)
  in try if conv_x (type_of P) typ then P else 
     error "cannot find a generalisation of the goal with the appropriate type"
     with UserError(_)->
     error "Cannot find a well-typed generalisation of the goal";; 

(****************************************)
(* General functions                    *)
(****************************************)

let (tactic_com : (constr -> tactic) -> command -> tactic) = fun tac t 
    ((_,sign,_) as x) -> tac (constr_of_com sign t) x;;


(**********************************************************************)
(* apply a tactic to the nth element of the signature             *)
(**********************************************************************)

let NTH m (tac:constr->tactic) (_,sign,_ as gl) =
  tac (snd (try nth sign m with Failure _ -> error "No such assumption")) gl;;

let LAST = NTH 1;;

(*******************************************)
(*         Introduction tactics        *)
(*******************************************)

(* Seems complicated with the gensym *)

let exists_sign str sign =
  try assoc str sign; true with Not_found -> false;;

let intro_base name c1 c2 sign k =
        let v = creation_var name c1 in
    INTRO(name,c1,INCOMPLET((-1)::k,(name,v)::sign,subst1 v c2));;

let intro_new str c1 c2 sign =
    let name = Name(str) in
        if (exists_sign name sign) (* or  (exists_var str) *)
        then error ((string_of_id str) ^ " is already a variable")
        else intro_base name c1 c2 sign;;

let rec idents_from = function
   [] -> []
 | (Name s,_)::l -> s::(idents_from l)
 | _ -> anomaly "Il faut nettoyer ce code";;

let new_string_signl l name sign =
                     next_ident_away name ((idents_from sign)@l);;
let new_string_sign = new_string_signl [];;

let intro_use s1 name c1 c2 sign u =
 let identifier = match name with
     Anonymous -> s1
   | Name s2   -> s2
 in intro_base (Name(new_string_signl (globals c1@globals c2) identifier sign))
               c1 c2 sign u;;

(* Create a name if needed *)
let (intro:tactic) = fun (u,sign,c) ->
  match c with
   Prod(name,c1,c2) -> (match hnftype(type_of c1) with
          Prop Data -> intro_use (id_of_string(hdchar c1)) name c1 c2 sign u
        | Prop _    -> intro_use (id_of_string "H") name c1 c2 sign u
        | Type(_,_)    -> intro_use (id_of_string "X") name c1 c2 sign u
        | _         -> anomaly "Wrong sort") 
 | _ -> error "Intro needs a product";;

let intros = REPEAT intro;;

(******************************************************)
(* Introduce names not used in the global environment *)
(******************************************************)

let new_global_string name sign = new_namel name (idents_from sign) ;;


let intro_global_use s1 name c1 c2 sign u =
 let identifier = match name with
     Anonymous -> s1
   | Name s2   -> s2
 in intro_base (Name (new_global_string identifier sign))
               c1 c2 sign u;;

let (intro_global:tactic) = function (u,sign,c) ->
  match c with
   Prod(name,c1,c2) -> let name' = if dependent 1 c2 then Anonymous else name
                       in (match hnftype(type_of c1) with
                  Prop Data -> intro_global_use (id_of_string(hdchar c1)) name c1 c2 sign u
                | Prop _    -> intro_global_use (id_of_string "H") name c1 c2 sign u
                | Type(_,_) -> intro_global_use (id_of_string "X") name c1 c2 sign u
                | _         -> anomaly "Wrong sort")
 | _ -> error "Intro needs a product";;

let intros_global = REPEAT intro_global;;

let intro_with_id str = function (u,sign,c) ->
  match c with
   Prod(name,c1,c2) ->
   let id = id_of_string str
   in intro_base (Name (new_global_string id sign)) c1 c2 sign u
 | _ -> error "Intro needs a product"
;;
let intros_with_id str = REPEAT(intro_with_id str);;

(**********************************)
(* Intros with explicit names     *)
(**********************************)

let intro_with str (u,sign,c) = match c with
   Prod(_,c1,c2) -> intro_new str c1 c2 sign u
 | _             -> error "Intro needs a product";;

let rec intros_with  =
    function []      -> IDTAC
       | str::l -> (intro_with str) THEN intros_with l;;


let intros_until s g = 
  let depth = let (_,_,c) = g in lookup 1 c
    where rec lookup n = function
           Prod(name,_,c') -> (match name with
                  Name(s')  -> if s'=s then n
                               else lookup (n+1) c'
                | Anonymous -> lookup (n+1) c')
         | _ -> error "No such hypothesis in current goal"
  in DO depth intro g;;

let intros_do n g = 
  let depth = let (_,_,c) = g in lookup 1 1 c
    where rec lookup all nodep = function
           Prod(name,_,c')  -> (match name with
                  Name(s')  -> if dependent 1 c' then lookup (all+1) nodep c'
                               else if nodep=n then all
                               else lookup (all+1) (nodep+1) c'
                | Anonymous -> if nodep=n then all
                               else lookup (all+1) (nodep+1) c')
         | _ -> error "No such hypothesis in current goal"
  in DO depth intro g;;

(********************************************************)
(*  Resolution tactics                              *)
(********************************************************)

(* Refinement tactic: unification against the head of the head normal form
   of the type of a term *)

let forme_clausale u = 
 let rec clrec k = function
     Prod(_,c1,c2) -> let (c,l) = clrec (k+1) (subst1 (Meta(k::u)) c2) 
                  in (c,(k::u,c1)::l)
   | c             -> (c,[]) 
 in clrec 1;;

let map_sign sign = map (fun (k,c) -> INCOMPLET(k,sign,c));;

let refine c t (u,sign,c1) =
   let (cl,hl) = forme_clausale u t in
   inst_pf (matching [] cl c1) (APP(PF(c)::(map_sign sign hl)));;

(* to replace with let res_pf c = refine c (type_of c)
  without implicit reduction *)

let res_pf c = refine c (hnf_constr (type_of c));;

let resolve = tactic_com res_pf;;

(* a scheme trying first-order matching and then doing an abstraction *)

let refinew_scheme pt (cl,hypl) sb (_,sign,c1) = 
    match decomp_app (instance sb cl) with
        (Meta(p),oplist)
            -> (* try first first-order matching *)
               (try inst_pf (matching sb cl c1) (APP(pt::(map_sign sign hypl)))
        with UserError(_) -> 
                let (s,cllist) = match_subterm_list sb oplist c1 in
                    let typp = instance s (assoc p hypl)
                in inst_pf
                   (add_matching s (p,abstract_list_all typp c1 cllist))
                   (APP(pt::map_sign sign hypl)))
       | _ -> inst_pf (matching sb cl c1) (APP(pt::(map_sign sign hypl)));;

(* Should not be used anymore, replaced by Pattern+Resolve * *)

let hnf_type_of = hnftype o type_of;;

let resolvew_tac c ((u,_,_) as gl) =
   let t = simplify (hnf_type_of c)
   in refinew_scheme (PF c) (forme_clausale u t) [] gl;;

let resolvew = tactic_com resolvew_tac;;


(* Apply a resolution using a given type which becomes a subgoal *)

let refine_type t (u,sign,c1) =
    (* check that t is well formed *)
    match hnf_type_of t with
    (Prop(_)|Type(_,_)) ->
        let (cl,hypl) = forme_clausale u (hnftype t) in
         inst_pf (matching [] cl c1)
                 (APP(INCOMPLET(0::u,sign,t)::(map_sign sign hypl)))
  | _ -> error "Not a type" ;;

let resolve_type = tactic_com refine_type;;

(* Cut tactics *)

let cut_tac c (n,sign,cl) =
  match hnf_type_of c with
    (Prop(_) | Type(_,_)) -> APP [INCOMPLET(0::n,sign,Prod(Anonymous,c,cl)); 
                                INCOMPLET(1::n,sign,c)]
  | _                   -> error "Not a proposition or a type";;

let cut = tactic_com cut_tac;;

let generalize_tac lconstr (n,sign,cl) = 
    let newcl = it_list 
      (fun cl' ci -> let t = type_of ci in
            match ci with (Var(Decl(na,_,_))) -> produit na t cl'
                        | _                   -> Prod(Anonymous,t,cl'))
      cl (rev lconstr)
    in APP (INCOMPLET(0::n,sign,newcl)::
           (map (function pf -> PF(pf)) lconstr));;

let generalize lcom (_,sign,_ as gl) =
    let translate = constr_of_com sign in
     generalize_tac (map translate lcom) gl;;

(*********************************************************)
(*  Resolve with missing arguments                   *)
(*********************************************************)

(* find the list of numbers of dependent products in a type
   not dependent in the goal *)

let conclusion = crec 1
   where rec crec p = function
      Prod(_,_,c) -> crec (p+1) c
    |      cl     -> (cl,p);;

let find_missing u t =
   let (cl,p) = conclusion t
   in findrec 1 [] t
   where rec findrec k l = function
      Prod(_,_,c) -> if dependent (p-k) cl then findrec (k+1) l c
                     else if dependent 1 c then findrec (k+1) ((k::u)::l) c
                     else findrec (k+1) l c
    |     _        -> (rev l);;

let resolve_with_tac chd mlist (u,_,_ as gl) =
   let t = type_of chd
   in let occlist = find_missing u t
   in let s0 =
         if list_length occlist = list_length mlist
         then combine (occlist,mlist)
         else error "Not the right number of missing arguments"
   in refinew_scheme (PF chd) (forme_clausale u t) s0 gl;;

let resolve_with = fun
  (c::lc) ((_,sig,_) as gl) ->
    let translate = constr_of_com sig
    in resolve_with_tac (translate c) (map translate lc) gl
| _ _ -> anomaly "resolve_with";;

let nodep_depth n c = lookup 1 1 c
    where rec lookup all nodep = function
           Prod(name,_,c')  -> (match name with
                  Name(s')  -> if dependent 1 c' then lookup (all+1) nodep c'
                               else if nodep=n then all
                               else lookup (all+1) (nodep+1) c'
                | Anonymous -> if nodep=n then all
                               else lookup (all+1) (nodep+1) c')
         | _ -> error "No such hypothesis";;

let dep_depth s c = lookup 1 c
    where rec lookup n = function
           Prod(name,_,c') -> (match name with
                  Name(s')  -> if s'=s then n
                               else lookup (n+1) c'
                | Anonymous -> lookup (n+1) c')
         | _ -> error "No such hypothesis";;

let match_args s hypl =
  let rec match_args_rec s = function
    [] -> s
  | ((k,c)::cl) ->
          match_args_rec (matching s (hnf_constr (type_of c)) (hnf_constr (assoc k hypl))) cl
  in match_args_rec s;;

let resolve_with_name_tac chd blist (u,_,_ as gl) =
   let t = rename_bound_var [] (type_of chd)
   in let (cl,hypl) = forme_clausale u t
   in let index_constr = fun
         (Dep(s),c) -> ((dep_depth s t)::u,c)
       | (NoDep(n),c) -> ((nodep_depth n t)::u,c)
   in let s0 = map index_constr blist
   in let s1 = match_args s0 hypl s0
   in refinew_scheme (PF chd) (cl,hypl) s1 gl;;

let resolve_with_name c lc ((_,sig,_) as gl) =
   let translate = constr_of_com sig
   in let translate_list = fun (n,val) -> (n,translate val)
   in resolve_with_name_tac (translate c)
     (map translate_list lc) gl;;

(********************************************************************)
(* A more atomic imp_elim tactic (Yves Bertot)                      *)
(********************************************************************)

let imp_elim c t (u,sign,goal_constr) =
  match t with
    Prod(_,c1,c2) ->
        if dependent 1 c2 then
         error "Imp_elim needs a non-dependent product"
        else
        (APP[INCOMPLET(0::u,sign,Prod(Anonymous,c2,goal_constr));
             APP[PF(c); INCOMPLET(1::1::u,sign,c1)]])
  | _ -> error "Imp_elim needs a non-dependent product";;

let imp_elim_tac =
   tactic_com (function c -> imp_elim c (hnf_constr (type_of c)));;


let (prod_elim: constr -> constr -> constr -> tactic) = fun
    c t v (u,sign,goal_constr) ->
(* no check is done to know whether the value actually has the right type *)
    match t with
      Prod(Name(_),c1,c2) -> if conv_x c1 (type_of v) then
         (APP[INCOMPLET(0::u,sign,Prod(Anonymous,subst1 v c2,goal_constr));
             APP[PF(c);PF(v)]])
       else error "The type of the argument does not match"
      | _ -> error "prod_elim needs a dependent product";;

let (prod_elim_tac: command list -> tactic) = fun
com_list ((u,sign,c) as x) ->
    match com_list with
      [prod_command;val_command] -> let prod = (constr_of_com sign prod_command)
 in
          prod_elim prod (hnf_constr (type_of prod))
              (constr_of_com sign val_command)
              x
    | _ -> error "prod_elim_tac needs two arguments";;

(********************************************************************)
(*               Exact tactics                                      *)
(********************************************************************)

let (assumption:tactic) = fun (_,sign,cl) ->
let rec arec = function
   []          -> error "No such assumption"
 | (s,v)::rest -> if conv_x cl (type_of v) then v else arec rest
in PF (arec sign);;

(* Solves the nth subgoal with the given solution *)
(* instantiate is the only global tactic, transforming globally a proof tree *)
(* This is the only prover function which does correctly global *)
(* substitutions to meta-variables *)
(* instantiate_pf : int -> command -> prooftree -> prooftree *)
let instantiate_pf n com pf = 
   let (u,sign,cl) = nth (list_pf pf) n
   in let c = constr_of_com sign com
      in let t = type_of c
         in if conv_x cl t then inst_pf [u,c] pf  (* redondant ? *)
        else let s = matching [u,c] cl t
                 in inst_pf s pf;;

let instantiate n t =
    change_state (instantiate_pf n t (get_goals())); print_state();;


(* Give exact proof term *)
let give_exact c (_,_,cl) =
    if conv_x cl (type_of c) then PF(c) else error "Not an exact proof";;

let give_exact_com = tactic_com give_exact;;

(**************************************************************)
(*             Adding new hypotheses                          *)
(**************************************************************)

let new_hyp_tac c = (cut_tac (type_of c)) THENL (give_exact c);;

let new_hyp = tactic_com new_hyp_tac;;

let new_hyp_with_tac c blist m hypl (u,sign,_ as g) =
    let t = rename_bound_var [] (type_of c)
    in let index_constr = fun
         (Dep(s),com) -> let h = (constr_of_com sign com)
                         in ((dep_depth s t)::u,h)
       | (NoDep(n),com) ->  let h = (constr_of_com sign com)
                            in ((nodep_depth n t)::u,h)
    in let clist = map index_constr blist
    in let rec make_appl ac n =
       if n > m
       then ac
       else let k = n::u
            in make_appl (App(ac,(try (assoc k clist) with Not_found -> Meta(k)))) (n+1)
    in let cut_pf =
       instance (match_args [] hypl clist) (make_appl c 1)
    in ((cut_tac (type_of cut_pf)) THENL (give_exact cut_pf)) g;;

let new_hyp_with com blist (u,sign,_ as g) =
    let c = (constr_of_com sign com)
    in let (_,hypl) = forme_clausale u (type_of c)
    in new_hyp_with_tac c blist (length hypl) hypl g;;

let new_hyp_with_num m com blist (u,sign,_ as g) =
    let c = (constr_of_com sign com)
    in let (_,hypl) = forme_clausale u (type_of c)
    in new_hyp_with_tac c blist m hypl g;;


(**************************************************************)
(*             Conversion tactics                             *)
(**************************************************************)

let change_pf c (u,sign,cl) = 
(try type_of c with UserError s -> error ("Not well-typed expression :"^s));
if conv_x c cl then INCOMPLET(u,sign,c) else error "Not convertible";;

let change  = tactic_com change_pf;;

let red (u,sign,cl) = INCOMPLET(u,sign,red_product cl);;

let hnf_tac (u,sign,cl) = INCOMPLET(u,sign,hnf_constr cl);;

let simpl_tac (u,sign,cl) = INCOMPLET(u,sign,nf cl);;

let pattern_list l (_,_,cl as gl) =
    let lname_typ = it_list (fun ln (_,c) -> let t = type_of c in
                         ((named_hd t Anonymous,t)::ln)) [] l in
    change_pf (applist (abstract_scheme cl l lname_typ) (map snd l)) gl ;;

let pattern l (_,sign,_ as gl) =
    let loc_co = map (function (n,com) -> (n, constr_of_com sign com)) l
    in pattern_list loc_co gl;;

let expand_const name = substrec where rec substrec = function
  (Const(Def(Name(name'),Judge(c,_,_),_)) as x) ->
                                 if name=name' then c else x (* = not eq *)
 | App(c1,c2)      -> App(substrec c1,substrec c2)
 | Lambda(n,c1,c2) -> Lambda(n,substrec c1,substrec c2)
 | Prod(n,c1,c2)   -> Prod(n,substrec c1,substrec c2)
 | Rec(b,lf,d)     -> Rec(b,map substrec lf, substrec d)
 | x               -> x;;

let unfold_nth loccname (u,sign,cl) = 
  INCOMPLET(u,sign, simplify
   (it_list 
   (fun c (lo,name) -> match lo with
      [] -> expand_const name c
    | _  -> unfoldn lo name c)  cl loccname));;

(**********************************************************************)
(*  Simplication of local context                                     *)
(**********************************************************************)

(* function clear_hyp: command list -> tactic *)
(* This tactic enables the user to remove hypotheses from the signature *)
(* so that these hypotheses do not appear in the assumption list.  Some *)
(* care is taken to prevent him from removing variables that are        *)
(* subsequently used in other hypotheses or in the conclusion of the    *)
(* goal.  The same result could be achieved in a safer way if the       *)
(* signature used to perform type checking and the actual list of       *)
(* hypotheses displayed on the screen were distinct data structures.    *)

let clear_hyp str_list (u,sign,c) =
   (* actually modify the signature for one argument at a time *)
   let rec remove_pair s sign =
     match sign with
       (((Name(s'), Var(Decl(_,Judge(C,_,_),_))) as pair)::T) -> 
           if (s = s') then T
           else if (occur s C) then
                error ((string_of_id s) ^ " is used in the hypothesis " ^ (string_of_id s'))
                else pair::(remove_pair s T)
     | (pair::_) -> anomaly "clear_hyp: signature not well-formed" 
     | [] -> error ((string_of_id s) ^ " is not among the assumptions.")
   (* For each argument, perform the checking and trigger the           *)
   (* signature modification *)
   and clear_hyp_aux sign s =
      if (occur s c) 
      then error ((string_of_id s) ^ " is used in the conclusion.")
      else remove_pair s sign
   in
     INCOMPLET (u, it_list clear_hyp_aux sign str_list, c);;

(**********************************************************************)
(*  Conversion tactics in an hypothesis                               *)
(**********************************************************************)

let apply_in_hyp reductor namehyp (u,sign,cl) =
   let rec do_rec = fun
     (((name',Var(Decl(na,Judge(c,r1,r2),inf))) as s)::rest) ->
             if (Name namehyp) = name' then
      ((name',Var(Decl(na,Judge(reductor c,r1,r2),inf)))::rest)
             else (s::(do_rec rest))
      | [] -> failwith ""
   in try INCOMPLET(u,do_rec sign,cl)
      with Failure _ -> error "bad hypothesis name";;

let red_hyp = apply_in_hyp red_product;;

let hnf_tac_hyp  = apply_in_hyp hnf_constr;;

let change_hyp_constr c = apply_in_hyp (fun ch -> if conv_x c ch then c else
         error "Not convertible");;

let change_hyp c nh ((u,sign,cl) as gl) =
  change_hyp_constr (constr_of_com sign c) nh gl;;

let simpl_hyp = apply_in_hyp nf;;

let unfold_nth_hyp loccname =
  apply_in_hyp (fun ch -> simplify
   (it_list 
    (fun c (lo,name) -> match lo with
        [] -> expand_const name c
      | _  -> unfoldn lo name c) 
   ch loccname));;

(**********************************************************************)
(*  Elimination tactics                           *)
(**********************************************************************)

let rec is_induc = function
    Ind(_,_,_,_,_)        -> true
  | Lambda(_,_,c) -> is_induc c
  | _             -> false;;

(* elimination on the inductive definition t, generate a new subgoal t
(the last one), enough for non-dependent elimination *)

let elim_scheme_type elim t (n,_,_ as gl) =
    let te = type_of elim in
    let (cl,hypl) = forme_clausale n te in
    let s = matching [] (snd (last hypl)) t 
    in refinew_scheme (PF elim) (cl,hypl) s gl;;

(* put t as t'=(x1:A1)..(xn:An)B with B an inductive definition of name name
 return name, B and t' *)

let prod_it = it_list (fun c (n,t)  ->  Prod(n,t,c));;

let reduce_to_ind t = elimrec t []
    where rec elimrec t l = match decomp_app(t) with
              (Const(Def(Name(m),Judge(v,_,_),_)),_) ->
                   if is_induc v then (m,t,prod_it t l)
                   else elimrec (simplify (one_step_reduce t)) l
             | (Prod(n,ty,t'),[]) -> elimrec t' ((n,ty)::l)
             | _ -> error "Not an inductive product";;

(* Find the right elimination suffix corresponding to the sort of the goal *)
 let suff cl = match hnf_type_of cl with
                  Type(Null,_) -> "_rect"
                | Type(Pos,_) -> "_recs"
                | Type(_,_)    -> error "elimination not allowed on this sort"
                | Prop(Null)   -> "_ind"
                | Prop(Data)   -> "_recd"
                | Prop(Pos)    -> "_rec"
                | _            -> anomaly "goal should be a type";;

(* find the list of numbers of dependent products in a type *)

let find_dep = findrec 1 []
   where rec findrec p l = function
       Prod(_,_,c)-> if dependent 1 c then findrec (p+1) (p::l) c
                     else findrec (p+1) l c
     | c -> (rev l);;

(* c should be of type A1->.. An->B with B an inductive definition *)

let compose sb = map (function (n,c) -> n,instance sb c) sb;;

let elimination largs c (n,sign,cl) =
  let (name,_,t) = reduce_to_ind (type_of c) in
  let name_elim = id_of_string((string_of_id name)^(suff cl)) in
  let elim = global(name_elim) in
  let te = type_of elim in
  let (cli,lsg) = forme_clausale n te in
  let ((np,lind),lsgf) = sep_last lsg in
  let (glc,lsgc) = forme_clausale np t in
  let sa = if largs=[] then [] else
           let intdep = find_dep t in
           if list_length intdep = list_length largs then
             map2 (fun u v -> u::np,v) intdep largs
           else error "some dependent arguments are missing" in
  let sb =  matching (if lsgc=[] then (np,c)::sa else sa) lind glc in
  let pfres s =
  inst_pf (compose s)
     (APP(PF(elim)::(map_sign sign lsgf)@
                    [inst_pf s (APP(PF(c)::(map_sign sign lsgc)))]))
  in try pfres (matching sb cli cl)
     with UserError(_) ->
        (match decomp_app (instance sb cli) with
             (Meta(p),oplist) ->
                let (s,cllist) = match_subterm_list sb oplist cl in
                let typp = instance (compose s) (assoc p lsg)
                in let P = abstract_list_all typp cl cllist
                in pfres (add_matching s (p,P))
            | _ -> error ((string_of_id name_elim)^" not an elimination combinator"));;

let elimination_type t (_,_,cl as gl) =
  let (name,tind,t) = reduce_to_ind t in
  let elim = global(id_of_string((string_of_id name)^(suff cl)))
  in match t with Prod(_,_,_) -> error "Not an inductive definition"
                | _       -> elim_scheme_type elim tind gl;;

let elim = tactic_com (elimination []);;

let elim_with = fun
  (c::lc) ((_,sig,_) as gl) ->
    let translate = constr_of_com sig
     in elimination (map translate lc) (translate c) gl
| _ _ -> anomaly "elim_with";;

let elim_last  = LAST (elimination []);;

let elim_type = tactic_com elimination_type;;

let pattern_elim_last =
       (function (_,sign,_ as gl) -> pattern_list [[],(snd (hd sign))] gl)
       THEN elim_last;;

(* Induction tactics *)
let induct s = (intros_until s) THEN pattern_elim_last;;

let induct_nodep n = (intros_do n) THEN elim_last;;

(* Tactics "Macro" for treatment of equality : Need the definition of *)
(* Leibniz equality on Set and on Type as an inductive type .        *)
(* exact names of the constants are in vernac.ml *)

(* eq=RefC("eq") eqt=RefC("eqt") *)
(* sym_equal=RefC("sym_equal") sym_eqt=RefC("sym_eqT") *)

let replace_tac eq eqt c2 c1 sym_equal sym_eqt (_,sign,_ as gl) =
  let cc1 = constr_of_com sign c1 in
  let cc2 = constr_of_com sign c2 in
  let t = type_of cc1 in
  let (e,sym) = match hnf_type_of t with 
     Prop(Pos) -> (eq,sym_equal)
   | Type(_,_) -> (eqt,sym_eqt)
   | _         -> error "replace_tac"
 in (elimination_type (applist (constr_of_com [] e) [t;cc1;cc2])
  THENL (assumption 
     ORELSE TRY (res_pf (constr_of_com [] sym) THEN assumption))) gl;;

(* This applies Elim on a term of conclusion an equality but *)
(* with inverting the sense of the equality *)

let equality_tac eq eqt c sym_equal sym_eqt (_,sign,_ as gl) =
  let cc = (constr_of_com sign c) in
    let rec decomp_prodrec k l = fun
      (Prod(n,t,c))    -> lambda n t (decomp_prodrec (k+1) (Rel(k)::l) c)
   |  (App(App((App(e,t) as et),cc2),cc1))    -> 
         let sym = if (e=(constr_of_com [] eq)) then sym_equal              else if (e=(constr_of_com [] eqt)) then sym_eqt
              else error "No equality here"
          in applist (constr_of_com [] sym) [t;cc2;cc1;applist cc l]
   |  _ -> error "No equality here"
   in let ccc = decomp_prodrec 1 [] (type_of cc)
    in elimination [] ccc gl;;

(*********************************************)
(* Automatic tactics                         *)
(*********************************************)

(* Auxiliary functions *)
exception BOUND;;

let rec string_head_bound = function 
     Prod(_,_,c2)            -> string_head_bound c2
  |  App(c1,_)               -> string_head_bound c1
  |  Const(Def(Name(str),_,_)) -> string_of_id str
  |  Var(Decl(Name(str),_,_)) -> string_of_id str
  |  _                       -> raise BOUND;;

let string_head c = try string_head_bound c
             with BOUND -> error "Bound head variable";;

let rec nb_hyp = function
     Prod(_,_,c2) -> if dependent 1 c2 then nb_hyp c2 else 1+(nb_hyp c2)
   | _            -> 0 ;;

let applyname = "Apply "
and revrewname = "Rewrite <- "
and rewname = "Rewrite "
and exactname = "Exact "
and unfoldname = "Unfold ";;

(* adding and removing tactics in the search table *)

let map_error_succeed f = map_f where rec map_f =
 function [] -> []
        (* Note: map_f never raises exception UserError *)
 |  h::t -> try (f h :: map_f t) with UserError _ -> map_f t;;

let make_apply_entry (name_tac,c,t) =
    if null (find_missing [] t) then
  let nt = string_head t
     in match t with
          Prod(_,_,c2) -> (nt,(applyname^name_tac,(nb_hyp t,Res_pf c)))
        | _            -> error "caught"
  else error "caught"
;;

let make_exact_entry (name_tac,c,t) =
    if null (find_missing [] t) then
  let nt = string_head t
     in match t with
          Prod(_,_,c2) -> error "caught"
        | _            -> (nt,(exactname^name_tac,(0,Give_exact c)))
  else error "caught"
;;

let save_tac_constr top (name_tac,c) =
  let t = type_of c in
  let hnft = hnf_constr t
  and add_search = if top then (fun (str,nt) -> add_tac_search str nt)
                   else fun (str, nt) -> (insert_search str nt;()) in
  let ents = match t with
      Prod _ ->
           map_error_succeed (fun f -> f())
           [(fun () -> make_apply_entry(name_tac,c,t))]
   | _ -> map_error_succeed (fun f -> f())
     [(fun () -> make_apply_entry(name_tac,c,hnft));
      (fun () -> make_exact_entry(name_tac,c,t))]
  in if ents = [] then
      error (name_tac^"cannot be used as a hint")
     else app add_search ents
;;

let erase_tac_constr (name_tac,c) =
  let t = type_of c in
  let hnft = hnf_constr t in
  let ents = match t with
      Prod _ ->
           map_error_succeed (fun f -> f())
           [(fun () -> make_apply_entry(name_tac,c,t))]
   | _ -> map_error_succeed (fun f -> f())
     [(fun () -> make_apply_entry(name_tac,c,hnft));
      (fun () -> make_exact_entry(name_tac,c,t))]
  in app (fun (strh,(text,_)) -> erase_search strh text) (rev ents)
;;

let add_resolve str = save_tac_constr true (string_of_id str,global str);;

let add_unfold_default = do_list
   (function str -> (global str;
                     add_tac_search (string_of_id str)
                     (unfoldname^(string_of_id str),(4,(Unfold_nth str)))));;

let ADD_UNFOLD_FN = ref [add_unfold_default];;

let add_unfold l = (hd !ADD_UNFOLD_FN) l;;

let set_add_unfold_fun f = ADD_UNFOLD_FN := f::(!ADD_UNFOLD_FN); ();;
let reset_add_unfold_fun f = ADD_UNFOLD_FN := tl (!ADD_UNFOLD_FN); ();;


let erase_tac str = erase_tac_constr (string_of_id str,global str);;

let add_resolution_default = do_list 
    (function n -> (global n;add_resolve n));;

let ADD_RESOLUTION_FN = ref [add_resolution_default];;

let add_resolution arg = (hd !ADD_RESOLUTION_FN) arg;;

let set_add_resolution_fun f = ADD_RESOLUTION_FN := f::(!ADD_RESOLUTION_FN); ();;
let reset_add_resolution_fun () = ADD_RESOLUTION_FN := tl (!ADD_RESOLUTION_FN); ();;

let erase_tacs_default = do_list 
    (function n -> try erase_tac n 
                   with UserError(_) -> ());;

let ERASE_TACS_FN = ref [erase_tacs_default];;

let erase_tacs arg = (hd (!ERASE_TACS_FN)) arg;;

let set_erase_tacs_fun f = ERASE_TACS_FN := f::(!ERASE_TACS_FN); ();;

let reset_erase_tacs_fun() = ERASE_TACS_FN := tl(!ERASE_TACS_FN);();;

let save_sign = do_list
    (function (Name(str),t) -> (try save_tac_constr false (string_of_id str,t);() 
                      with UserError(_) -> ())
            | _ -> failwith "save_sign");;

let erase_sign = do_list
    (function (Name(str),t) -> (try erase_tac_constr (string_of_id str,t);()
                                with UserError(_) -> ())
            | _ -> failwith "erase_sign");;

(* tactics with a trace mechanism for automatic search *)

(* A concrete notion of tactics *)

type concretetac = 
     Capply of constr 
   | Cexact of constr 
   | Celim of constr 
   | Cassumption 
   | Cintro
   | Cidtac
   | Cunfold of identifier
   | Ccompose of concretetac * concretetac list;;

type ntactic == goal -> concretetac * prooftree;;

let name_ntactic (n,tac) g = (n,tac g);;

let nres_pf c =  name_ntactic (Capply c,res_pf c);;

(* A special tactic for rewriting which fails if the goal will be unchanged *)

let elim_rewrite c (n,sign,cl) =
  let (name,_,t) = reduce_to_ind (type_of c) in
  let name_elim = id_of_string((string_of_id name)^(suff cl)) in
  let elim = global(name_elim) in
  let te = type_of elim in
  let (cli,lsg) = forme_clausale n te in
  let ((np,lind),lsgf) = sep_last lsg in
  let (glc,lsgc) = forme_clausale np t in
  let sb =  matching (if lsgc=[] then [(np,c)] else []) lind glc in
  let pfres s =
  inst_pf (compose s)
     (APP(PF(elim)::(map_sign sign lsgf)@
                    [inst_pf s (APP(PF(c)::(map_sign sign lsgc)))]))
  in try pfres (matching sb cli cl)
     with UserError(_) ->
        (match decomp_app (instance sb cli) with
             (Meta(p),oplist) ->
                let (s,cllist) = match_subterm_list sb oplist cl in
                let typp = instance (compose s) (assoc p lsg)
                in let P = abstract_list_all typp cl cllist
                and CP = it_list (fun t (na,ta) -> Lambda(na,ta,t)) 
                         cl (args_typ typp)
                in if eq_constr(P,CP) then 
                error "Rewrite does not make any progress"
                else pfres (add_matching s (p,P))
            | _ -> error ((string_of_id name_elim)^" not an elimination combinator"));;


let nelim_rew c = name_ntactic (Celim c,elim_rewrite c);;

let nunfold id = name_ntactic (Cunfold id, unfold_nth [([],id)]);;

let ngive_exact c = name_ntactic (Cexact c,give_exact c);;

let nintro = name_ntactic (Cintro,intro)
and nassumption = name_ntactic (Cassumption,assumption);;

let rec prconc = function
  Capply c          -> [< 'S "Apply "; prterm c >]
| Cexact c          -> [< 'S "Exact "; prterm c >]
| Celim c           -> [< 'S "Rewrite <- "; prterm c >]
| Cassumption       -> [< 'S "Assumption" >]
| Cintro            -> [< 'S "Intro" >]
| Cidtac            -> [< 'S "Idtac" >]
| Cunfold id        -> [< 'S "Unfold "; print_id id >]
| Ccompose (t1,[])  -> prconc t1
| Ccompose (t1,[t2]) -> HOV 0 [< prconc t1; 'SPC; 'S ";"; 'SPC ;
                                 prconc t2 >] 
| Ccompose (t1,lt)  -> HOV 0 [< prconc t1; 'SPC; 'S ";"; 'SPC ; 
                        HOV 0 [< 'S "["; 
                          prlist_with_sep 
                          (function _ -> [<'SPC;'S"|"; 'SPC >]) prconc lt;
                          'S "]" >] >]
;;

let ntactic_tactic (ntac:ntactic) g = let (n,pf) = ntac g in
    if not (is_silent()) then
        PPNL (HOV 1 [< 'S "Use :"; 'SPC; (prconc n) >]);
    pf;;

let npriority = prrec []
    where rec prrec m = function
        [] -> (rev m)
      | (n,(pr,t))::l -> if pr=0 then prrec (t::m) l else rev m;;

let rec NFIRST = fun tacl g ->
 match tacl with
    []      -> error "NFIRST : no applicable tactic"
 |  T::rest -> (try (T g) with UserError _ -> NFIRST rest g);;

let NCOMPLETE (T: ntactic) g =
 let (n,pft) = (T g) in
match simpl_pf pft with
   (PF _ as u) -> n,u
 | _           -> error "NCOMPLETE expects a proof";;

(* apply a tactic to all goals in a prooftree, gives as answer the list
   of concrete tactics *)

let napply_tac (tac : ntactic) pf =
    let rec arec = function
   INTRO(s,c,pf) -> let (l,pf') = arec pf in l,INTRO(s,c,pf')
 | APP(lpf)      -> let (ln,lpff)= it_list
                    (fun (lc,lpfc) pf1 -> let (l1,pf1') = arec pf1 in
                                         (l1@lc,pf1'::lpfc)) ([],[]) lpf
                    in (ln,APP(rev lpff))
 | INCOMPLET(g1,g2,g3) -> let (n,pf) = tac(g1,g2,g3) in [n],pf
 | pf            -> [],pf
in let (l,pf') = arec pf in (rev l), simpl_pf pf';;


(* same as selective_apply_tac but does not fail if the number of 
tactic is different from the number of subgoals, if more remaining subgoals 
then it applies the last given tactic  to all the remaining subgoals *)

let nselect_apply_tac taclist pf = 
 let TL = ref taclist in if taclist = [] then ([],pf) else 
 let DEF = ref (hd taclist) in 
 let rec arec = function
   INTRO(s,c,pf) -> let (l,pf')=arec pf in l,INTRO(s,c,pf')
 | APP(lpf)        -> let (ln,lpff)= it_list
                    (fun (lc,lpfc) pf1 -> let (l1,pf1') = arec pf1 in
                                         (l1@lc,pf1'::lpfc)) ([],[]) lpf
                    in (ln,APP(rev lpff))
 | INCOMPLET(k,s,cl)   -> 
    let tac = (try let t = hd !TL in DEF:=t; TL:=(tl !TL); t 
               with Failure _ -> !DEF)
      in let (n,pf)= tac (k,s,cl) in [n],pf
 | pf            -> [],pf in
 let (l,res) = arec pf in  (rev l), simpl_pf res ;;

let compose_concrete t1 = function 
               []       -> t1 
             | [Cidtac] -> t1
             | lt       -> Ccompose(t1,lt);;

#infix "NTHEN";;
let prefix NTHEN (T1:ntactic) (T2:ntactic) = function g ->
    let (n1,pf1) = T1 g in
    let (lt,pf2) = (napply_tac T2 pf1)
    in compose_concrete n1 lt,pf2;;

#infix "NTHENSS";;
let prefix NTHENSS (T1:ntactic) (LT:ntactic list) = function g ->
    let (n1,pf1) = T1 g in
    let (lt,pf2) = (nselect_apply_tac LT pf1)
    in  compose_concrete n1 lt,pf2;;


let NIDTAC (g1,g2,g3 as g) = (Cidtac,INCOMPLET g);;

let NTRY (T:ntactic) g = try (T g)
  with UserError _ -> NIDTAC g;;

let NREPEAT n T = reprec n
  where rec reprec k g = 
    if k = 0 then NIDTAC g
    else try (T NTHEN (reprec (k-1))) g 
         with UserError _ -> NIDTAC g;;

let rec ntrivial_fail add_sign ((_,sign,cl) as goal) =
  save_sign add_sign;
  try let res = NFIRST (map NCOMPLETE
               (nassumption::
               (nintro
                NTHEN (function (_,sign',_ as t')->
                                     (ntrivial_fail [hd sign'] t')))::
               (ntrivial_resolve cl))) goal
  in erase_sign add_sign;res
  with reraise -> (erase_sign add_sign); raise(reraise)

and npossible_resolve gl =
    try map (function (_,(_,t))-> t)
            (my_find_search (string_head_bound gl))
    with BOUND | Not_found -> []

and nsearch n add_sign ((_,sign,cl) as t) =
  if n=0 then error "BOUND 2"
  else save_sign add_sign;
       try
       let res = NFIRST
                 (nassumption::
                  (nintro NTHEN (function (_,sign',_ as t')
                                    -> nsearch n [hd sign'] t'))
                  ::(map (function ntac -> ntac NTHEN nsearch (n-1) [])
                          (npossible_resolve cl))) t
       in erase_sign add_sign; res
       with reraise -> erase_sign add_sign; raise(reraise)

and nsearch_auto n (_,sign,_ as t) = nsearch n sign t

and ncomplete_auto n = NCOMPLETE (nsearch_auto n)

and my_find_search str =
  map (fun (a,(b,t)) -> (a,(b,
    match t with
      Res_pf c -> nres_pf c
    | Give_exact c -> ngive_exact c
    | Res_pf_THEN_trivial_fail c -> (nres_pf c) NTHEN (ntrivial_fail [])
    | Unfold_nth str -> nunfold str
    | Rewrite_complete c -> (nelim_rew c) NTHENSS [NIDTAC;ncomplete_auto 3]
  )))
 (find_search str)

and ntrivial_resolve gl =
   try npriority (my_find_search (string_head_bound gl))
   with BOUND | Not_found -> []
;;


let automatic n = ntactic_tactic (NTRY (NCOMPLETE (nsearch_auto n)));;

let SEARCH_DEPTH = ref(5);;
let auto = automatic (!SEARCH_DEPTH);;

let trivial  (_,sign,_ as t) = ntactic_tactic (NTRY (ntrivial_fail sign)) t;;

let add_trivial str =
   let c = global str
   in let t = type_of c
      in let name_hd = string_head t
         in add_tac_search name_hd
            ((applyname^(string_of_id str)^" ; Trivial"),(1,Res_pf_THEN_trivial_fail c));;

let add_trivial_list_default = do_list add_trivial;;

let ADD_TRIVIAL_LIST_FN = ref [add_trivial_list_default];;

let add_trivial_list l = (hd !ADD_TRIVIAL_LIST_FN) l;;

let set_add_trivial_list_fun f 
      = ADD_TRIVIAL_LIST_FN := f::!ADD_TRIVIAL_LIST_FN;();;

let reset_add_trivial_list_fun ()
      = ADD_TRIVIAL_LIST_FN := tl !ADD_TRIVIAL_LIST_FN;();;

(* A tactic for automatic rewriting *)

(* assume commands for eq, eqt, symeq, symeqt are predefined *)

type eqtable = {mutable Eq : command; mutable Eqt : command; 
                mutable Symeq : command; mutable Symeqt : command};;

let refC_str s = RefC (id_of_string s);;

let eqnames = 
    {Eq=refC_str "eq";Eqt=refC_str "eqT";Symeq =refC_str "sym_equal";
    Symeqt=refC_str "sym_eqT"};;

let eq = eqnames.Eq and eqt = eqnames.Eqt and symeq = eqnames.Symeq
and symeqt = eqnames.Symeqt;;

let eq_constr_com e c = 
   try let e' = constr_of_com [] c in eq_constr(e,e')
   with UserError _ -> false;;

let rec find_eq_rew_rev = function
      (Prod(_,_,c))    -> find_eq_rew_rev c
   |  (App(App(App(e,_),_),cc1))   -> 
         if (eq_constr_com e eq) or (eq_constr_com e eqt)
         then try string_head_bound cc1 with BOUND -> ""
         else error "No equality here"
   |  _ -> error "No equality here";;

let find_eq_rew c t = 
      let rec decomp_prodrec k l = fun
      (Prod(n,t',c')) -> let (str,cc') = decomp_prodrec (k+1) (Rel(k)::l) c'
                        in (str,lambda n t' cc')
   |  (App(App((App(e,t') as et),cc2),cc1))    -> 
         let sym = if eq_constr_com e eq then symeq
              else if eq_constr_com e eqt then symeqt
              else error "No equality here"
         and str = (try string_head_bound cc2 with BOUND -> "")
         in (str,applist (constr_of_com [] sym) [t';cc2;cc1;applist c l])
   |  _ -> error "No equality here"
   in decomp_prodrec 1 [] t;;

let save_tac_rewrite top rev (name_tac,c) =
  let t = type_of c
  and add_search = if top then add_tac_search
                   else fun str nt -> (insert_search str nt;())
  in if null (find_missing [] t) then   
  let ((nt,c'),rewn) = if rev then (find_eq_rew_rev t,c),revrewname
                else find_eq_rew c t,rewname
  in add_search (rewname^nt) (rewn^name_tac,(nb_hyp t,
                               Rewrite_complete c'))
  else error
 (rewname^name_tac^" generates open subgoals, cannot be used automatically");;

let add_tac_rewrite rev id = save_tac_rewrite true rev
     (string_of_id id, global id);;

let erase_tac_rewrite rev (name,c) =
  let t = type_of c in 
  let (nt,rewn) = if rev then (find_eq_rew_rev t,revrewname)
                  else (fst(find_eq_rew c t), rewname)
  in erase_search (rewname^nt) (rewn^name);;

let erase_tac_rewrite_id rev name = 
     erase_tac_rewrite rev (string_of_id name,global name);;

 let erase_tacs_rewrite_default rev =  do_list (erase_tac_rewrite_id rev);;

let ERASE_TACS_REWRITE_FN = ref [erase_tacs_rewrite_default];;

let erase_tacs_rewrite arg = (hd (!ERASE_TACS_REWRITE_FN)) arg;;
     
let set_erase_tacs_rewrite_fun f =
         ERASE_TACS_REWRITE_FN := f::(!ERASE_TACS_REWRITE_FN); ();;

let reset_erase_tacs_rewrite_fun() =
         ERASE_TACS_REWRITE_FN := tl(!ERASE_TACS_REWRITE_FN);();;

    
let save_sign_rewrite = do_list
    (function (Name(str),t) -> 
        (try save_tac_rewrite false true (string_of_id str,t);() 
                      with UserError(_) -> ())
       | _ -> failwith "save_sign_rewrite");;

let erase_sign_rewrite = do_list
    (function (Name(str),t) -> 
           (try erase_tac_rewrite true (string_of_id str,t);()
                                with UserError(_) -> ())
            | _ -> failwith "erase_sign_rewrite");;

let add_rewrite_default rev  =  do_list (add_tac_rewrite rev);;

let ADD_REWRITE_FN = ref [add_rewrite_default];;

let add_rewrite arg = (hd !ADD_REWRITE_FN) arg;;

let set_add_rewrite_fun f = ADD_REWRITE_FN := f::(!ADD_REWRITE_FN); ();;
let reset_add_rewrite_fun () = ADD_REWRITE_FN := tl (!ADD_REWRITE_FN); ();;

let collect_rewrite_tac = srec [] 
 where rec srec ltac =
   function [] -> (try (my_find_search revrewname)@ltac with Not_found -> ltac)
       | id::lid -> try let lts = my_find_search (rewname^(string_of_id id)) 
                      in srec (lts@ltac) lid
                  with Not_found -> (srec ltac lid);;

let REWRITE_DEPTH = ref (10);;

let nrewrite_auto n (_,sign,cl as gl) = 
   save_sign_rewrite sign; 
   let lrew = map (function (_,(_,t))-> t) 
                  (collect_rewrite_tac (globals cl))
   in let pft =  NREPEAT n (NFIRST lrew) gl
   in (erase_sign_rewrite sign; pft);;

let rewrite_automatic n = ntactic_tactic (nrewrite_auto n);;

let rewrite_auto = rewrite_automatic !REWRITE_DEPTH;;
