(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                               std.ml                                     *)
(****************************************************************************)
(*      Compatibility file with caml V3.1                                   *)
(****************************************************************************)

#open "hashtbl";;
#open "pp";;
#open "stdpp";;
#open "unix";;
#open "sys";;

(* type 'a nref = {mutable v : 'a};; *)
let nref x = {v = x};;
let forward _ = failwith "forward";;

let message_default x = PPNL [< 'S x >];;
let MESSAGE_DEFAULT_FN = ref [message_default];;
let message x = (hd !MESSAGE_DEFAULT_FN) x;;
let set_message_fun f = MESSAGE_DEFAULT_FN :=f::(!MESSAGE_DEFAULT_FN); ();;
let reset_message_fun () = MESSAGE_DEFAULT_FN :=tl (!MESSAGE_DEFAULT_FN); ();;

(* *)

#infix "o";;
let prefix o f g x = f(g x);;
let length = list_length;;
let string_of_num = string_of_int
and num_of_string = int_of_string;;
let date (s:string) = prerr_endline "<pas de date>"; "01"^s^"01"^"01";;

(* dml.ml *)

let succ_int = (succ : int -> int);;
let modify_vect f v = modify_rec 0
  where rec modify_rec i =
    if i < vect_length v then (
      v.(i) <- f(v.(i));
      modify_rec (succ i)
    )
;;
let explode s = explode_rec 0
  where rec explode_rec n =
    if n >= string_length s
    then []
    else make_string 1 (nth_char s n) :: explode_rec (succ n)
;;

let implode sl =
  let len = it_list (function a -> function b -> a+(string_length b)) 0 sl in
  let dest = create_string len in
      (it_list (function start -> function src ->
                let src_len = string_length src in
                    (blit_string src 0 dest start src_len;
                     start + src_len))
               0 sl;
       dest)
;;

let eq_string(a,b) = (a:string) = (b:string);;
let explode_ascii s = explode_rec 0
  where rec explode_rec n =
    if n >= string_length s
    then []
    else int_of_char (nth_char s n) :: explode_rec (succ n)
;;

(* glob.ml *)

let item l n = if n<0 then failwith "item" else
 (item_op (l,n) where rec item_op = 
   fun (hd::_,0) -> hd
   |   (_::tl,n) -> item_op (tl,pred n)
   |   ([],_) -> failwith "item");;
let null = function [] -> true | _ -> false
and hd = function [] -> failwith "hd" | (x::_) -> x
and tl = function [] -> failwith "tl" | (_::l) -> l
;;
let except_assoc e = except_e where rec except_e = function
    [] -> []
 | (x,_ as y)::l -> if x=e then l else y::except_e l
;;
let append = prefix @;;

let union l1 l2 = urec l1 where rec urec =
    fun []->l2 | (a::l) -> if mem a l2 then urec l else a :: (urec l);;

(* prel.ml *)

let index x l = index_x (1,l)
 where rec index_x = function
     n,y::l -> if x = y then n else index_x (succ n,l)
   | _ -> failwith "index"
;;

let curry f x y = f (x,y)
;;

let for_all2 rel = for_all2_rel
 where rec for_all2_rel =
  fun [] -> (function
           [] -> true | _ -> failwith "for_all2")
    | (x1::l1) -> function
           x2::l2 -> rel x1 x2 & for_all2_rel l1 l2
         | _ -> failwith "for_all2"
;;

let eq(x,y) = x == y
and gt(x,y) = x > y
;;

(* exception Identity;; *)
let share f x = try f x with Identity -> x
;;
let filter p = (share filter_aux
 where rec filter_aux = function
     [] -> raise Identity
   | x::l ->
       if p x then x::filter_aux l else share filter_aux l)
;;
let neg predicate x = not (predicate x)
;;
let C f x y = f y x
and I x = x
;;
let make_set l = share make_aux l
 where rec make_aux = function
     [] -> raise Identity
   | x::l ->
       if mem x l then share make_aux l else x::make_aux l
;;
let subtractq l1 l2 = filter (neg (C memq l2)) l1;;
let rec distinct = function
    h::t -> not (mem h t) & distinct t | _ -> true
;;
let chop_list = (fun n l -> chop_aux n ([],l))
 where rec chop_aux =
  fun 0 (l1,l2) -> rev l1,l2
    | _ (_,[]) -> failwith "chop_list"
    | n (l1,h::t) -> chop_aux (pred n) (h::l1,t)
;;
let unionq l1 l2 = urec l1
 where rec urec = function
     [] -> l2
   | a::l -> if memq a l2 then urec l else a::urec l
;;

let iterate f = iterate_f
 where rec iterate_f n x =
  if n <= 0 then x else iterate_f (pred n) (f x)
;;
let nth l n = item l (pred n)
;;
let map f = function
    [] -> []
  | [a] -> [f a]
  | [a1; a2] -> let v = f a1 in [v; f a2]
  | l -> map_f l
      where rec map_f = function
          [] -> [] | a::l -> let v = f a in v::map_f l
;;
let map_i f = map_i_rec
 where rec map_i_rec i = function
     [] -> [] | x::l -> let v = f i x in v::map_i_rec (i+1) l
;;
let sort gt l =
 let rec merge = function
     l1,[] -> l1
   | [],l2 -> l2
   | (h1::t1 as l1),(h2::t2 as l2) ->
       if gt(h1, h2) then h1::merge (t1,l2)
        else h2::merge (l1,t2)
 and merge_sort_step = function
     [] -> []
   | [l] -> [l]
   | l1::l2::t -> merge (l1,l2)::merge_sort_step t
 and merge_sort_aux = function
     [] -> []
   | [l] -> l
   | L -> merge_sort_aux (merge_sort_step L) in
  merge_sort_aux (map (fun x -> [x]) l)
;;
let ascii_code s = int_of_char(nth_char s 0);;
let ascii i = make_string 1 (char_of_int i);;

let rec rev_append =
 fun [] x -> x | (x::l') l -> rev_append l' (x::l)
;;
let it_vect f a v = it_vect_f a 0
 where rec it_vect_f a n =
  if n >= vect_length v then a
   else it_vect_f (f a v.(n)) (succ n)
;;
let K x y = x;;
let rec first_n =
 fun 0 -> K []
   | n -> function
          [] -> failwith "first_n"
        | x::l -> x::first_n (pred n) l
;;

let flat ll = list_it append ll []
;;
let rec last = function
    [] -> failwith "last"
  | x::[] -> x
  | x::l -> last l
;;
let rec sep_last = function
    [] -> failwith "sep_last"
  | hd::[] -> (hd,[])
  | hd::tl ->
      let (l,tl) = sep_last tl in (l,hd::tl)
;;

let words s = rev(words_rec [] s "" 0)
  where rec words_rec l s w i =
    if i >= string_length s then (if w = "" then l else w::l)
    else
      match nth_char s i with
        ` ` | `\n` | `\t` ->
          words_rec (if w = "" then l else w::l) s "" (i+1)
      | c   ->
          words_rec l s (w^(make_string 1 c)) (i+1)
;;
let scan_string s1 s2 = scan_rec
  where rec scan_rec pos =
    if pos >= string_length s1 then -1
    else
      let c = nth_char s1 pos in mem_rec 0
      where rec mem_rec i =
        if i >= string_length s2 then scan_rec(pos+1)
        else if c == nth_char s2 i then pos
        else mem_rec(i+1)
;;

let interval n m = interval_n ([],m)
 where rec interval_n (l,m) =
  if n > m then l else interval_n (m::l,pred m)
;;

let range = interval 1;;
let add_set x s = if mem x s then s else x::s;;
let filter_pos = filter;;
let intersect l1 l2 = filter_pos (C mem l2) l1;;

(* hash.ml *)

let hash_clear v = modify_vect (fun _ -> []) v;;
let hash_add_assoc (key,val as pair) v =
  let i = (hash key) mod (vect_length v) in
  v.(i) <- pair::v.(i); ()
and hash_remove_assoc (key,val as pair) v =
  let i = (hash key) mod (vect_length v) in
  v.(i) <- except pair v.(i); ()
and hash_assoc key v =
  assoc key v.((hash key) mod (vect_length v))
;;

(* devraient etre dans unix.ml *)

let mode_append = 16;;
let l_xtnd = 2;;

(* *)

let comline c =
  let child = fork () in
  if child < 0 then
    prerr_endline "fork failed"
  else if child > 0 then (wait ();())
  else (
    execv "/bin/sh" [| "sh"; "-c"; c |];
    prerr_endline "execv failed";
    exit 1
  )
;;

let Noisy = ref false;;
let cocorico() = 
try (if (getenv "ARCH") = "sun4" & !Noisy
    then comline ("cp " ^ (filename__concat (getenv "COQTOP") (filename__concat "RELEASED" (filename__concat (getenv "ARCH") "rooster.au"))) ^ " /dev/audio")
    else ())
with _ -> ();;

let old_vect_item(x,y) = x.(y);;
let old_vect_assign(x,y,z) = x.(y) <- z; z;;

let print_flush() = flush std_out;;

let length_string = string_length;;

let (set_prompt, input_char,get_prompt) =
  let prompt = nref ""
  and bol = nref true in

(* set_prompt *)
  (function x ->
    prompt.v <- x; ()
  ),
(* input_char *)
  (function ic ->
    if bol.v & ic == std_in then (
      prerr_string prompt.v; flush std_err
    );
    let c = io__input_char ic in
    if ic == std_in then (
      bol.v <- c == `\n`;
      ()
    );
    c
  ),
(* get_prompt *)
  (function () -> prompt.v)
;;

(* exception break;; *)
sys__catch_break true;;

let open_append name = 
    let fd = open name
      [O_WRONLY; O_CREAT; O_APPEND]
        (s_irall + s_iwall)
    in
        open_descriptor_out fd
;;

let min x y = if x < y then x else y;;
let max x y = if x > y then x else y;;

let repeat n action arg = repeat_action n
 where rec repeat_action n =
  if n <= 0 then () else (action arg;repeat_action (pred n))
;;

let break_string sep string =
 fst (it_list scan ([],[]) (rev (sep::explode string)))
 where scan (strs,chars) char =
  if char = sep then implode chars::strs,[]
   else strs,char::chars
;;

let skip_string skip s = skip_rec 0
  where rec skip_rec i =
    if i >= string_length s then ""
    else if mem (nth_char s i) skip then skip_rec (i+1)
    else sub_string s i (string_length s - i)
;;

let skip_space_return = skip_string [` `; `\n`; `\t`];;

let last_n_string n s =
 let l = string_length s in
 if l < n then failwith "last_n_string"
   else sub_string s (l-n) n

and first_n_string n s =
 if string_length s < n then failwith "first_n_string"
  else sub_string s 0 n
;;

let caml_assoc e l = try assoc e l with Not_found -> raise (Failure "assoc");;

let fst3 (a,b,c) = a;;
  
let map_succeed f = map_f where rec map_f =
 function [] -> []
        (* Note: map_f never raises exception failure *)
 |  h::t -> try (f h :: map_f t) with Failure _ -> map_f t;;

let make_set l = share make_aux l where rec make_aux =
    fun [] -> raise Identity
    |   (x::l) -> if mem x l then share make_aux l else x::make_aux l;;
