(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                             program.ml                                   *)
(****************************************************************************)
#infix "THEN";;
#infix "o";;
#open "std";;
#open "initial";;
#open "extraction";;
#open "term";;
#open "termfw";;
#open "command";;
#open "trad";;
#open "tactics";;
#open "machine";;
#open "matching";;

let caml_assoc_sign str l =
    try assoc_sign str l with
    Not_found -> raise (Failure "assoc_sign");;

(*************************************)
(* Part of program using only fterms *)
(*************************************)


(* Contents and level of an arity *)
let rec contents_level_of_arity A = 
  match (hnftype A) with 
    Prop(c)     ->  c , Proof
  | Type(c,_)   ->  c , Object
  | Prod(n,B,C) ->  (contents_level_of_arity C)
  | _           ->  anomaly "shouldn't appear(contents_level_of_arity)" ;;


(********************************************************)
(* Information and level extraction from a construction *)
(********************************************************)
let rec extract lrel lmeta A = 
  let extract_inf lrel lmeta A = 
    let (infA,_) = extract lrel lmeta A in
      (match infA with
        Inf(n,A') ->  A'
      | _         ->  anomaly "shouldn't appear(extract 1)")
  in
    match A with
      Var(Decl(n,Judge(c1,c2,lev),info))  ->  info , lev
    | Const(Def(n,Judge(c1,c2,lev),info)) ->  info , lev
    | Rel(n)    ->  let (_,cn,ln) = nth lrel n in
                      (match cn with
                        Pos   ->  Inf(Extracted,(Frel n))
                      | Null  ->  Logic
                      | Data  ->  Inf(Fw,(Frel n))) , ln
    | Prop(c)   ->  inf_kind c , Object
    | Type(c,_) ->  inf_kind c , Object
    | Lambda(Name(x),A,B) ->  let (iA,_) = extract lrel lmeta A
                              and (cA,lA) = contents_level_of_arity 
                                              (type_wfconstr (map fst3 lrel) A) in
                                let (iB,lB) = extract ((A,cA,lA)::lrel) lmeta B in
                                  (match iB with 
                                    Logic     ->  Logic
                                  | Inf(n,cB) ->  (match (compat lB n lA iA) with
                                                    Comp(cA)  ->  Inf(n,Flambda(Name(x),cA,cB))
                                                  | Invar     ->  Inf(n,fpop cB)
                                                  | Error     ->  anomaly "shouldn't appear(extract 2)"))
                                , lB
    | App(t1,t2)         -> let (i1,l1) = extract lrel lmeta t1 in
                              let (i2,l2) = extract lrel lmeta t2 in
                                inf_apply l1 l2 i1 i2 , l1
    | Prod(nom,A,B)  ->  let (iA,_) = extract lrel lmeta A
                         and (cA,lA) = contents_level_of_arity 
                                             (type_wfconstr (map fst3 lrel) A)
                         and (_,lB) =  contents_level_of_arity 
                                             (type_wfconstr (A::(map fst3 lrel)) B)
                         in
                         let (iB,_) = extract ((A,cA,lA)::lrel) lmeta B in
                               (match iB with 
                                 Logic     ->  Logic
                               | Inf(n,cB) ->  (match (compat lB n lA iA) with
                                                 Comp(cA)  ->  Inf(n,Fprod(nom,cA,cB))
                                               | Invar     ->  Inf(n,fpop cB)
                                               | Error     ->  anomaly "shouldn't appear(extract 3)"))
                               , lB
    | Construct(i,ind)  ->  let (_,lev) = contents_level_of_arity
                                            (type_wfconstr (map fst3 lrel) ind)
                            and (infind,_) = extract lrel lmeta ind in
                              (match infind with
                                Inf(n,ind') ->  Inf(n,Fconstr(i,ind'))
                              | _           ->  Logic) , lev
    | Ind(stamp,A,lc,_,_) ->  let (infA,_) = extract lrel lmeta A in
                              (match infA with
                                Inf(n,A') ->  Inf(n,(fmake_ind stamp A' 
                                           (map (extract_inf ((A,Pos,Object)::lrel) lmeta) lc)))
                              | Logic     ->  Logic) , Object
    | Rec(b,P::lf,c)    ->  let (cP,lP) = contents_level_of_arity 
                                            (type_wfconstr (map fst3 lrel) P) in
                              if (cP = Null) then Logic , lP
                              else let frec' = 
                                  (match (map (extract_inf lrel lmeta) (c::P::lf)) with
                                    (c'::lfP') -> Frec(lfP',c') 
                                  | _          -> anomaly"shoudn't appear(extract 5)") in
                                Inf((if (cP = Pos) then Extracted else Fw),frec') , lP
    | Meta(n)           ->  caml_assoc n lmeta, Object
    | _                 ->  anomaly "shouldn't appear(extract 4)" ;;


(* Transformation Var or Const to Fterm *) 
let fterm_of_ident = function
    Var(Decl(_,_,Inf(_,t)))   ->  t
  | Const(Def(_,_,Inf(_,t)))  ->  t
  | _   -> error "should be an informative variable or constant" ;;


(* Transformation of an element of the signature to (name*fterm) *)
let fterm_of_sign = function
    n,Var(Decl(_,_,Inf(_,t)))   -> n, t
  | _   -> failwith "should be an informative variable" ;;


(* Transformation of a signature to (name*fterm) list *)
let fsign_of_sign sign = map_succeed fterm_of_sign sign ;;


(*******************************************)
(* Transformation of a command to an fterm *)
(*******************************************)
let flambda name typ c = Flambda(name,typ,(fsubst_var name c)) ;;

let fprod name typ c = Fprod(name,typ,(fsubst_var name c)) ;;

let fterm_of_com = crec where rec crec vl = crecvl where rec
  crecvl = function
    RefC(str)         -> (try caml_assoc_sign str vl with Failure _ ->
      (try (fterm_of_ident (val_of str)) with Undeclared -> error((string_of_id str)^"  not declared(1)")))
  | PropC(c)          -> Fomega
  | TypeC(c)          -> Fomega
  | AppC(c1,c2)       -> Fapp(crecvl c1,crecvl c2)
  | LambdaC(s,c1,c2)  -> (let u1 = crecvl c1 and name = Name(s) in
               flambda name u1 (crec ((name,Fvar(name,u1))::vl) c2))
  | ProdC(s,c1,c2)    -> (let u1 = crecvl c1 and name = Name(s) in
               fprod name u1 (crec ((name,Fvar(name,u1))::vl) c2))
  | ArrowC(c1,c2)     -> Fprod(Anonymous,crecvl c1,flift 1 (crecvl c2))
  | ConstrC(i,c)      -> Fconstr(i,crecvl c)
  | IndC(stamp,s,c,l)       -> let t = crecvl c and name = Name(s) in
                          let v = (name,Fvar(name,t)) in 
                            fmake_ind stamp t (map (((fun a -> fun b -> (Fvar(a,b))) name) o (crec (v::vl))) l)
  | ElimC(c,P)        -> fmake_elimination (crecvl c) (crecvl P)
  | _                 -> error "command not implemented" ;;


(* Information extraction from a construction *)
let info_of_constr cl = fst(extract [] [] cl) ;;

let info_of_constr_rel lrel cl = fst(extract lrel [] cl) ;;

let info_of_constr_meta lmeta cl = fst(extract [] lmeta cl) ;;


(* If an information is logic or not *)
let is_logic inf = match inf with
                      Logic     ->  true 
                   |  Inf(n,t)  ->  false ;;


(* If an information is informative or not *)
let is_info inf = match inf with
                      Logic     ->  false 
                   |  Inf(n,t)  ->  true ;;


(************************************************************************)
(* Transformation of a command to an fterm using a well typed signature *)
(************************************************************************)
let fterm_of_com_use sign c = fterm_of_com (fsign_of_sign sign) c ;;


(* Number of reductions and introductions *) 
let rec nb_red_intro nr ni gl = function
    Prod(x,A,B) ->  if is_logic(info_of_constr A) then
                      let pf = intro_global gl in
                        (match (list_pf pf) with
                          [(u',(n,c)::sign,_) as gl'] ->   
                      nb_red_intro nr (ni+1) gl' (subst1 c B)
                        | _  -> anomaly"shoudn't appear(nb_red_intro)")
                    else
                      (nr,ni+1)
 |  t           -> nb_red_intro (nr+1) ni gl (red_product t) ;;


let DOTHEN n tac1 tac2 = 
  if (n>0) then
    DO n tac1 THEN tac2
  else
    tac2 ;;




(********************************************************)
(* Extraction of the proof term associated to a program *)
(********************************************************)
let rec find_ind = function
    Prod(s,c1,c2) ->  find_ind c2
  | t             ->  let (x,l) = reduce t in
                          match x with
                            Prod(_,_,_)   ->  find_ind x
                          | Ind(_,_,_,_,_)  ->  applist x l
                          | _             ->  error "shouldn't appear(find_ind)" ;;

let constr_of_fterm (_,sign,cl as gl) = crec where rec crec = function
    Fvar(Name(n),c)   ->  (try caml_assoc_sign n sign with Failure _ -> 
                  (try (val_of n) with Undeclared -> error((string_of_id n)^" not declared(2)")))
  | Fconst(Name(n),c) ->  global n
  | Frel(n)           ->  Rel(n)
  | Fomega            ->  Prop(Pos)  
  | Fapp(t1,t2)       ->  App(crec t1,crec t2)
  | Flambda(x,A,B)    ->  Lambda(x,crec A,crec B)
  | Fprod(x,A,B)      ->  Prod(x,crec A,crec B)
  | Find(stamp,A,lc,_,_)    ->  let A' = crec A
                          and lc' = map crec lc in
                            make_ind stamp A' lc'
  | Fconstr(i,ind)    ->  Construct(i,find_ind cl)
  | Frec(lf,c)        ->  Rec(false,map crec lf,crec c)
  | _                 ->  anomaly "shouldn't appear(constr_of_fterm)" ;;

let constr_of_pure_fterm sign = crec [] where rec crec l = function
    Fvar(Name(n),c)   ->  let c' = crec l c
                          and c'' = try caml_assoc_sign n sign with Failure _ ->
                  (try val_of n with Undeclared -> error((string_of_id n)^" not declared(3)"))
                          in if (conv_x c' (type_of c'')) then
                              c''
                            else
                              raise (NONPUR "not a pure program")
  | Fconst(Name(n),c) ->  let c' = crec l c
                          and c'' = global n in
                            if (conv_x c' c'') then
                              c''
                            else
                              raise (NONPUR "not a pure program")
  | Frel(n)           ->  Rel(n)
  | Fomega            ->  Prop(Pos)
  | Fapp(t1,t2)       ->  App(crec l t1,crec l t2)
  | Flambda(x,A,B)    ->  let A' = crec l A in
                            Lambda(x,A',crec (A'::l) B)
  | Fprod(x,A,B)      ->  let A' = crec l A in
                            Prod(x,A',crec (A'::l) B)
  | Find(stamp,A,lc,_,_)    ->  let A' = crec l A in
                          let lc' = map (crec (A'::l)) lc in
                            make_ind_rel stamp l A' lc'
  | Fconstr(i,ind)    ->  Construct(i,crec l ind)
  | Frec(lf,c)        ->  Rec(false,map (crec l) lf,crec l c)
  | Fimplicit         ->  Implicit 
  | _                 ->  anomaly "shouldn't appear(constr_of_pure_fterm)" ;;

let constr_of_pure_fterml l sign = crec l where rec crec l = function
    Fvar(Name(n),c)   ->  let c' = crec l c
                          and c'' = try caml_assoc_sign n sign with Failure _ ->
                  (try val_of n with Undeclared -> error((string_of_id n)^" not declared(3)"))
                          in if (conv_x c' (type_of c'')) then
                              c''
                            else
                              raise (NONPUR "not a pure program")
  | Fconst(Name(n),c) ->  let c' = crec l c
                          and c'' = global n in
                            if (conv_x c' c'') then
                              c''
                            else
                              raise (NONPUR "not a pure program")
  | Frel(n)           ->  Rel(n)
  | Fomega            ->  Prop(Pos)
  | Fapp(t1,t2)       ->  App(crec l t1,crec l t2)
  | Flambda(x,A,B)    ->  let A' = crec l A in
                            Lambda(x,A',crec (A'::l) B)
  | Fprod(x,A,B)      ->  let A' = crec l A in
                            Prod(x,A',crec (A'::l) B)
  | Find(stamp,A,lc,_,_)    ->  let A' = crec l A in
                          let lc' = map (crec (A'::l)) lc in
                            make_ind_rel stamp l A' lc'
  | Fconstr(i,ind)    ->  Construct(i,crec l ind)
  | Frec(lf,c)        ->  Rec(false,map (crec l) lf,crec l c)
  | Fimplicit         ->  Implicit 
  | _                 ->  anomaly "shouldn't appear(constr_of_pure_fterm)" ;;


(* Extraction of the fterm of an information  *)
let fterm_of = function
    Inf(_,t)  ->  t
  | Logic     ->  anomaly "shouldn't appear(fterm_of)" ;;


(* Arguments list of an application *)
let list_arg = 
  let rec lrec la = function
    Fapp(c1,c2) ->  lrec (c2::la) c1
  | x           ->  (x,la) 
  in lrec [] ;;


(* Total reduction of products of an fterm *)
let rec fred_all = function
    Fprod(s,c1,c2)  ->  Fprod(s,c1,fred_all c2)
  | t               ->  let (x,l) = freduce t in
                          match x with
                            Fprod(_,_,_)  ->  fred_all x
                          | _             ->  fapplist x l ;;

(* Is a term logic or not *)
let is_logical hl P = isrec [] P where rec isrec lv P =
  match P with
    Lambda(x,A,B)       ->  isrec ((isrec lv A)::lv) B
  | Prod(x,A,B)         ->  isrec ((isrec lv A)::lv) B
  | App(c1,c2)          ->  isrec lv c1
  | Var(Decl(_,_,info)) ->  (info=Logic)
  | Const(Def(_,_,info))->  (info=Logic)
  | Rel(n)              ->  nth lv n
  | Meta(m)             ->  isrec lv (caml_assoc m hl)
  | Prop(c)             ->  (c=Null)
  | Type(c,_)           ->  (c=Null)
  | Ind(_,C,_,_,_)      ->  isrec lv C
  | Rec(_,c::l,_)       ->  isrec lv c
  | _                   ->  error "shouldn't appear(is_logical)" ;;

(* Total reduction of products of a construction + bound or not *)
let rec red_all_lie = function
    Prod(s,c1,c2) ->  let (b,c2') = red_all_lie c2 in b,Prod(s,c1,c2')
  | t             ->  let (x,l) = reduce t in
                          match x with
                            Prod(_,_,_) ->  red_all_lie x
                          | Rel _       ->  true,applist x l
                          | _           ->  false,applist x l ;;


(* Total reduction of products of a construction *)
let red_all = snd o red_all_lie ;;


(* Try a matching a first level and next if fails *)
let rec try_matching subst t1 t2 = 
  try matching subst t1 t2 with
  UserError _ -> if not(occur_meta(t1)) & not(occur_meta(t2)) then 
                  try let t1' = one_step_reduce t1 
                      and t2' = one_step_reduce t2 in
                        try_matching subst t1' t2'
                  with
                    UserError _ -> raise NONSPECIF
                 else if not(occur_meta(t1)) then
                        try let t1' = one_step_reduce t1 in 
                          try_matching subst t1' t2
                        with
                          UserError _ -> raise NONSPECIF
                      else if not(occur_meta(t2)) then
                        try let t2' = one_step_reduce t2 in 
                          try_matching subst t1 t2'
                       with
                          UserError _ -> raise NONSPECIF 
                      else raise NONSPECIF ;;


(* Go to the first informative term *)
let go_to_info n l = gorec n l where rec gorec n l = function
  Prod(x,A,B) ->  if (is_logic(info_of_constr_meta l A)) then
                      gorec (n+1) (([n],Logic)::l) (subst1 (Meta([n])) B)
                    else
                      Prod(x,A,B),n,l
| x           -> try let x' = one_step_reduce x in
                        gorec n l x'
                  with
                    UserError _ -> anomaly "shouldn't appear (go_to_info)" ;;


(* Test if a term is dependent or not of an other *)
let rec dependant t gl =
   if eq_constr(gl,t) then true else match gl with
     App(t1,t2)      -> dependant t t1 or dependant t t2
  |  Lambda(n,t1,t2) -> dependant t t1 or dependant t t2
  |  Prod(n,t1,t2)   -> dependant t t1 or dependant t t2
  |  Ind(_,t',l,s,_)     -> dependant t t' or exists (dependant t) l
  |  Construct(i,t') -> dependant t t'
  |  Rec(b,lf,d)     -> (exists (dependant t) lf) or dependant t d
  |  _               -> false ;;


(* Number of non informative dependent introductions on a term *)
let nb_intros_liees_list lc = 
  let rec nrec ni = function
    Prod(x,A,B) ->  if not (is_logic(info_of_constr A)) 
                    or exists (fun t -> dependant t A) lc 
                    then ni
                    else nrec (ni+1) B
  | _           ->  ni 
in nrec 0 ;;

let nb_intros_liees c = nb_intros_liees_list [c] ;;


(* Number of non informative introductions *)
let nb_intros = 
  let rec nrec ni = function
    Prod(x,A,B) ->  if is_logic(info_of_constr A) then
                      nrec (ni+1) B
                    else
                      ni
  | _           ->  ni 
in nrec 0 ;;


(**********************************************************)
(* New type annotation and all new fonctions on this type *)
(**********************************************************)

(* type annote =
    Avar of name * fterm                  (* free variables *)
  | Aconst of name * fterm                (* constants *)
  | Arel of int                           (* variables *)
  | Aomega                                (* proposition types *)
  | Aapp of annote * annote               (* application  (M N) *)
  | Alambda of name * fterm * annote      (* abstraction  [x:T]M *)
  | Alambdacom of name * constr * annote
  | Aprod of name * fterm * fterm         (* product      (x:T)M *)
  | Aind of stamp * fterm * fterm list * fspecifr * fterm list
  | Aconstr of int * fterm
  | Arec of fterm * annote list * annote
  | Arecursion of fterm * fterm * fterm * constr
  | Annot of annote * constr
  | Aimplicit ;;

type annoted_command =
    AappC of annoted_command * annoted_command                 
  | AlambdaC of identifier * command * annoted_command  
  | AlambdacomC of identifier * command * annoted_command     
  | ArecC of annoted_command * command 
  | ArecursionC of command * command * command * command
  | AnnotC of annoted_command * command
  | ApurC of command
;;
*)
let rec forget_annot = function
    AappC(t1,t2)       ->  AppC(forget_annot t1,forget_annot t2)
 |  AlambdaC(x,A,B)    ->  LambdaC(x,A,forget_annot B)
 |  AlambdacomC(x,A,B) ->  forget_annot B
 |  ArecC(l,c)         ->  ElimC(forget_annot l,c)
 |  AnnotC(c,P)        ->  forget_annot c
 |  ApurC(c)           ->  c 
 | _                   ->  anomaly "shouldn't appear(forget_annot)" ;;


let rec annot_of_fterm = function
    Fvar(n,c)       ->  Avar(n,c)
 |  Fconst(n,c)     ->  Aconst(n,c)
 |  Frel(n)         ->  Arel(n)
 |  Fomega          ->  Aomega
 |  Fapp(t1,t2)     ->  Aapp(annot_of_fterm t1,annot_of_fterm t2)
 |  Flambda(x,A,B)  ->  Alambda(x,A,annot_of_fterm B)
 |  Fprod(x,A,B)    ->  Aprod(x,A,B)
 |  Find(stamp,i,l1,l2,l3)->  Aind(stamp,i,l1,l2,l3)
 |  Fconstr(i,c)    ->  Aconstr(i,c)
 |  Frec(lf,c)      ->  Arec(hd lf,map annot_of_fterm (tl lf),annot_of_fterm c)
 |  Fimplicit       ->  Aimplicit ;;

let rec fterm_of_annot  = function
    Avar(n,c)         ->  Fvar(n,c)
 |  Aconst(n,c)       ->  Fconst(n,c)
 |  Arel(n)           ->  Frel(n)
 |  Aomega            ->  Fomega
 |  Aapp(t1,t2)       ->  Fapp(fterm_of_annot t1,fterm_of_annot t2)
 |  Alambda(x,A,B)    ->  Flambda(x,A,fterm_of_annot B)
 |  Alambdacom(x,A,B) ->  flift (-1) (fterm_of_annot B)
 |  Aprod(x,A,B)      ->  Fprod(x,A,B)
 |  Aind(stamp,i,l1,l2,l3)  ->  Find(stamp,i,l1,l2,l3)
 |  Aconstr(i,c)      ->  Fconstr(i,c)
 |  Arec(P,lf,c)      ->  Frec((P::map fterm_of_annot lf),fterm_of_annot c)
 |  Arecursion(wf,A,P,R)-> Fapp(Fapp(wf,A),P)
 |  Annot(c,P)        ->  fterm_of_annot c
 |  Aimplicit         ->  Fimplicit ;;

let subst_varn n = function
  Anonymous -> I
| Name(str) -> substrec n
    where rec substrec n = function
     (Var(Decl(Name(str'),_,_)) as x) -> 
                       if eq(str,str') then Rel(n) else x 
   | App(c1,c2)     -> App(substrec n c1,substrec n c2)
   | Lambda(name,c1,c2) -> Lambda(name,substrec n c1, substrec (n+1) c2)
   | Prod(name,c1,c2)   -> Prod(name,substrec n c1, substrec (n+1) c2)
   | Ind(stamp,c,l,Specifr(k,s),Specift((kd,lpd),(kn,lpn)))
                        -> Ind(stamp,substrec n c, map (substrec (n+1)) l,
                               Specifr(k,map (fun (i,x)->(i,substrec n x)) s),
                               Specift((kd,map (substrec (n+1)) lpd),
                                       (kn,map (substrec (n+1)) lpn)))
   | Construct(i,c) -> Construct(i,substrec n c)
   | Rec(b,lpf,c)   -> Rec(b,map (substrec n) lpf,substrec n c)
   | x              -> x;;


let fsubst_varn name n = substrec n
  where rec substrec n = function
    (Fvar(name',_) as fvar') -> if eq(name,name') then Frel(n)
                                else fvar'
  | Fapp(c1,c2)              -> Fapp(substrec n c1,substrec n c2)
  | Flambda(name',c1,c2)     -> Flambda(name',substrec n c1,substrec (n+1) c2)
  | Fprod(name',c1,c2)       -> Fprod(name',substrec n c1,substrec (n+1) c2)
  | Find(stamp,c,lc, Fspecifr(k,s),lP) ->
                      Find(stamp,substrec n c, map (substrec (n+1)) lc,
                           Fspecifr(k, map (fun (i,x)->(i,substrec n x)) s),map (substrec (n+1)) lP)
  | Fconstr(i,c)            -> Fconstr(i,substrec n c)
  | Frec(lf,c)              -> Frec(map (substrec n) lf,substrec n c)
  | x                       -> x;;

let asubst_var name = substrec 1
  where rec substrec n = function
    (Avar(name',_) as avar')    -> if eq(name,name') then Arel(n)
                                    else avar'
  | Aapp(c1,c2)                 -> Aapp(substrec n c1,substrec n c2)
  | Alambda(name',c1,c2)        -> Alambda(name',fsubst_varn name n c1,substrec (n+1) c2)
  | Alambdacom(name',c1,c2)     -> Alambdacom(name',subst_varn n name c1,substrec (n+1) c2)
  | Aprod(name',c1,c2)          -> Aprod(name',fsubst_varn name n c1,fsubst_varn name (n+1) c2)
  | Aind(stamp,c,lc,Fspecifr(k,s),lP) -> 
                Aind(stamp,fsubst_varn name n c, map (fsubst_varn name (n+1)) lc,
                      Fspecifr(k,map (fun (i,x)->(i,fsubst_varn name n x)) s),
                      map (fsubst_varn name (n+1)) lP)
  | Aconstr(i,c)                -> Aconstr(i,fsubst_varn name n c)
  | Arec(P,lf,c)                -> Arec(fsubst_varn name n P,map (substrec n) lf,substrec n c)
  | Arecursion(wf,A,P,R)        -> Arecursion(wf,A,P,R)
  | Annot(p,t)                  -> Annot(substrec n p,subst_varn n name t)
  | x                           -> x ;;

let aapplist = it_list (fun a -> fun b -> (Aapp(a,b))) ;;

exception FreeVar ;;

let fclosedn n term = let rec closed_rec n = function
    Frel(m)       -> if m>n then raise FreeVar
  | Fapp(c,c')    -> closed_rec n c; closed_rec n c'
  | Flambda(_,c,c') -> closed_rec n c; closed_rec (n+1) c'
  | Fprod(_,c,c') -> closed_rec n c; closed_rec (n+1) c'
  | Find(_,c,l,_,_)     -> closed_rec n c; do_list (closed_rec (n+1)) l
  | Fconstr(_,c)  -> closed_rec n c
  | Frec(l,c)     -> closed_rec n c; do_list (closed_rec n) l
  | _             -> ()
in (closed_rec n term; true) ;;

let aclosed term = let rec closed_rec n = function
    Arel(m)             -> if m>n then raise FreeVar
  | Aapp(c,c')          -> closed_rec n c; closed_rec n c'
  | Alambda(_,c,c')     -> fclosedn n c; closed_rec (n+1) c'
  | Alambdacom(_,_,c)   -> closed_rec (n+1) c
  | Aprod(_,c,c')       -> fclosedn n c; fclosedn (n+1) c' ; ()
  | Aind(_,c,l,_,_)     -> fclosedn n c; do_list (fclosedn (n+1)) l ; ()
  | Aconstr(_,c)        -> fclosedn n c; ()
  | Arec(P,l,c)         -> closed_rec n c; fclosedn n P; do_list (closed_rec n) l ; ()
  | Arecursion(wf,A,P,R)-> ()
  | Annot(c,P)          -> closed_rec n c
  | _                   -> ()
in try (closed_rec 0 term; true) with FreeVar -> false ;;

let aliftn k n c = if k=0 then c
  else (liftrec n) c where rec
    liftrec n = function
    (Arel(i) as x)              ->  if i<n then x else Arel(i+k)
  | Aapp(c1,c2)                 -> Aapp(liftrec n c1,liftrec n c2)
  | Alambda(name,c1,c2)         -> Alambda(name,fliftn k n c1,liftrec (n+1) c2)
  | Alambdacom(name,c1,c2)      -> Alambdacom(name,liftn k c1 n,liftrec (n+1) c2)
  | Aprod(name,c1,c2)           -> Aprod(name,fliftn k n c1,fliftn k (n+1) c2)
  | Aind(stamp,c,lc,Fspecifr(k,s),lP) ->
                  Aind(stamp,fliftn k n c,map (fliftn k (n+1)) lc,
                        Fspecifr(k,map (fun (i,x)->(i,fliftn k n x)) s),
                        map (fliftn k (n+1)) lP)
  | Aconstr(i,c)                -> Aconstr(i,fliftn k n c)
  | Arec(P,lf,c)                -> Arec(fliftn k n P,map (liftrec n) lf, liftrec n c)
  | Arecursion(wf,A,P,R)        -> Arecursion(wf,A,P,R)
  | Annot(c,P)                  -> Annot(liftrec n c,liftn k P n)
  | x                           -> x ;;

let alift k = aliftn k 1 ;;

exception ANNOT ;;

let is_annot term = let rec purec = function
    Alambda(n,t,c)    ->  purec c
  | Alambdacom(n,t,c) ->  raise ANNOT
  | Aapp(c1,c2)       ->  purec c1 ; purec c2
  | Arec(P,l,c)       ->  purec c ; do_list purec l
  | Arecursion(wf,A,P,R)->  ()
  | Annot(c,P)        ->  raise ANNOT
  | x                 ->  ()
in try (purec term ; true) with ANNOT -> false ;;

let fsubst_liftn lam n =  substrec n
  where rec substrec n = function
    (Frel(k) as c) -> if k=n then flift (n-1) lam
                      else if k<n then c
                      else Frel (k-1)
  | Fapp(c1,c2)            -> Fapp(substrec n c1,substrec n c2)
  | Flambda(name,c1,c2)    -> Flambda(name,substrec n c1,substrec (n+1) c2)
  | Fprod(name,c1,c2)      -> Fprod(name,substrec n c1,substrec (n+1) c2)
  | Find(stamp,c,lc, Fspecifr(k,s),lP) ->
                     Find(stamp,substrec n c,map (substrec (n+1)) lc,
                           Fspecifr(k, map (fun (i,x)->(i,substrec n x)) s),map (substrec (n+1)) lP)
  | Fconstr(i,c)      -> Fconstr(i,substrec n c)
  | Frec(lf,c)        -> Frec(map (substrec n) lf,substrec n c)
  | x                 -> x ;;

let subst_liftn n lam = substrec n
    where rec substrec n = function 
     (Rel k as x)   -> if k=n then lift (n-1) lam 
                       else if k<n then x else Rel(k-1)
   | App(c1,c2)     -> App(substrec n c1,substrec n c2)
   | Lambda(name,c1,c2) -> Lambda(name,substrec n c1, substrec (n+1) c2)
   | Prod(name,c1,c2)   -> Prod(name,substrec n c1, substrec (n+1) c2)
   | Ind(stamp,c,l,Specifr(k,s),Specift((kd,lpd),(kn,lpn)))
                        -> Ind(stamp,substrec n c, map (substrec (n+1)) l,
                               Specifr(k,map (fun (i,x)->(i,substrec n x)) s),
                               Specift((kd,map (substrec (n+1)) lpd),
                                       (kn,map (substrec (n+1)) lpn)))
   | Construct(i,c) -> Construct(i,substrec n c)
   | Rec(b,lpf,c)   -> Rec(b,map (substrec n) lpf,substrec n c)
   | x              -> x ;;

let subst_liftfn n lam = substrec n
    where rec substrec n = function 
     (Rel k as x)   -> if k=n then lift (n-1) (constr_of_pure_fterm [] lam) 
                       else if k<n then x else Rel(k-1)
   | App(c1,c2)     -> App(substrec n c1,substrec n c2)
   | Lambda(name,c1,c2) -> Lambda(name,substrec n c1, substrec (n+1) c2)
   | Prod(name,c1,c2)   -> Prod(name,substrec n c1, substrec (n+1) c2)
   | Ind(stamp,c,l,Specifr(k,s),Specift((kd,lpd),(kn,lpn)))
                        -> Ind(stamp,substrec n c, map (substrec (n+1)) l,
                               Specifr(k,map (fun (i,x)->(i,substrec n x)) s),
                               Specift((kd,map (substrec (n+1)) lpd),
                                       (kn,map (substrec (n+1)) lpn)))
   | Construct(i,c) -> Construct(i,substrec n c)
   | Rec(b,lpf,c)   -> Rec(b,map (substrec n) lpf,substrec n c)
   | x              -> x ;;

let asubst_liftn lam n =  substrec n
  where rec substrec n = function
    (Frel(k) as c) -> if k=n then alift (n-1) lam
                      else if k<n then (annot_of_fterm c)
                      else annot_of_fterm(Frel (k-1))
  | Fapp(c1,c2)            -> Aapp(substrec n c1,substrec n c2)
  | Flambda(name,c1,c2)    -> Alambda(name,fterm_of_annot(substrec n c1),substrec (n+1) c2)
  | Fprod(name,c1,c2)      -> annot_of_fterm(Fprod(name,fterm_of_annot(substrec n c1),fterm_of_annot(substrec (n+1) c2)))
  | Find(stamp,c,lc, Fspecifr(k,s),lP) ->
                     annot_of_fterm(Find(stamp,fterm_of_annot(substrec n c),map fterm_of_annot( map (substrec (n+1)) lc),
                           Fspecifr(k, map (fun (i,x)->(i,fterm_of_annot(substrec n x))) s),map fterm_of_annot(map (substrec (n+1)) lP)))
  | Fconstr(i,c)      -> annot_of_fterm(Fconstr(i,fterm_of_annot(substrec n c)))
  | Frec(P::lf,c)        -> Arec(fterm_of_annot(substrec n P),map (substrec n) lf,substrec n c)
  | x                 -> annot_of_fterm(x) ;;

let asubst_lift lam =  substrec 1
  where rec substrec n = function
    (Arel(k) as c)              -> if k=n then alift (n-1) lam
                                   else if k<n then c
                                   else Arel(k-1)
  | Aapp(c1,c2)                 -> Aapp(substrec n c1,substrec n c2)
  | Alambda(name,c1,c2)         -> Alambda(name,fsubst_liftn (fterm_of_annot lam) n c1,substrec (n+1) c2)
  | Alambdacom(name,c1,c2)      -> Alambdacom(name,subst_liftfn n (fterm_of_annot lam) c1,substrec (n+1) c2)
  | Aprod(name,c1,c2)           -> Aprod(name,fsubst_liftn (fterm_of_annot lam) n c1,fsubst_liftn (fterm_of_annot lam) (n+1) c2)
  | Aind(stamp,c,lc,Fspecifr(k,s),lP) ->
                 Aind(stamp,fsubst_liftn (fterm_of_annot lam) n c, map (fsubst_liftn (fterm_of_annot lam) (n+1)) lc,
                      Fspecifr(k,map (fun (i,x)->(i,fsubst_liftn (fterm_of_annot lam) n x)) s),
                      map (fsubst_liftn (fterm_of_annot lam) (n+1)) lP)
  | Aconstr(i,c)                -> Aconstr(i,fsubst_liftn (fterm_of_annot lam) n c)
  | Arec(P,lf,c)                -> Arec(fsubst_liftn (fterm_of_annot lam) n P,map (substrec n) lf,substrec n c)
  | Arecursion(wf,A,P,R)        -> Arecursion(wf,A,P,R)
  | Annot(c,P)              -> Annot(substrec n c,subst_liftfn n (fterm_of_annot lam) P)
  | x                           -> x ;;

let asubst_lift2 lam lam' =  substrec 1
  where rec substrec n = function
    (Arel(k) as c)              -> if k=n then alift (n-1) lam
                                   else if k<n then c
                                   else Arel(k-1)
  | Aapp(c1,c2)                 -> Aapp(substrec n c1,substrec n c2)
  | Alambda(name,c1,c2)         -> Alambda(name,fsubst_liftn (fterm_of_annot lam) n c1,substrec (n+1) c2)
  | Alambdacom(name,c1,c2)      -> Alambdacom(name,subst_liftn n lam' c1,substrec (n+1) c2)
  | Aprod(name,c1,c2)           -> Aprod(name,fsubst_liftn (fterm_of_annot lam) n c1,fsubst_liftn (fterm_of_annot lam) (n+1) c2)
  | Aind(stamp,c,lc,Fspecifr(k,s),lP) ->
                 Aind(stamp,fsubst_liftn (fterm_of_annot lam) n c, map (fsubst_liftn (fterm_of_annot lam) (n+1)) lc,
                      Fspecifr(k,map (fun (i,x)->(i,fsubst_liftn (fterm_of_annot lam) n x)) s),
                      map (fsubst_liftn (fterm_of_annot lam) (n+1)) lP)
  | Aconstr(i,c)                -> Aconstr(i,fsubst_liftn (fterm_of_annot lam) n c)
  | Arec(P,lf,c)                -> Arec(fsubst_liftn (fterm_of_annot lam) n P,map (substrec n) lf,substrec n c)
  | Arecursion(wf,A,P,R)        -> Arecursion(wf,A,P,R)
  | Annot(c,P)              -> Annot(substrec n c,subst_liftn n lam' P)
  | x                           -> x ;;

let fsubst_closedn lam n = substrec n
  where rec substrec n = function
    (Frel(k) as c) -> if k=n then lam
                      else if k<n then c else Frel (k-1)
  | Fapp(c1,c2)         -> Fapp(substrec n c1,substrec n c2)
  | Flambda(name,c1,c2) -> Flambda(name,substrec n c1,substrec (n+1) c2)
  | Fprod(name,c1,c2) -> Fprod(name,substrec n c1,substrec (n+1) c2)
  | Find(stamp,c,lc, Fspecifr(k,s),lP) ->
                     Find(stamp,substrec n c, map (substrec (n+1)) lc,
                           Fspecifr(k, map (fun (i,x)->(i,substrec n x)) s),map (substrec (n+1)) lP)
  | Fconstr(i,c)      -> Fconstr(i,substrec n c)
  | Frec(lf,c)        -> Frec(map (substrec n) lf,substrec n c)
  | x                 -> x ;;

let subst_closedn n lam = substrec n
    where rec substrec n = function
     (Rel k as x)   -> if k=n then lam                       
                       else if k<n then x else Rel(k-1)
   | App(c1,c2)     -> App(substrec n c1,substrec n c2)
   | Lambda(name,c1,c2) -> Lambda(name,substrec n c1, substrec (n+1) c2)
   | Prod(name,c1,c2)   -> Prod(name,substrec n c1, substrec (n+1) c2)
   | Ind(stamp,c,l,Specifr(k,s),Specift((kd,lpd),(kn,lpn)))
                        -> Ind(stamp,substrec n c, map (substrec (n+1)) l,
                               Specifr(k,map (fun (i,x)->(i,substrec n x)) s),
                               Specift((kd,map (substrec (n+1)) lpd),
                                       (kn,map (substrec (n+1)) lpn)))
   | Construct(i,c) -> Construct(i,substrec n c)
   | Rec(b,lpf,c)   -> Rec(b,map (substrec n) lpf,substrec n c)
   | x              -> x ;;

let subst_closedfn n lam = substrec n 
    where rec substrec n = function
     (Rel k as x)   -> if k=n then constr_of_pure_fterm [] lam
                       else if k<n then x else Rel(k-1)
   | App(c1,c2)     -> App(substrec n c1,substrec n c2)
   | Lambda(name,c1,c2) -> Lambda(name,substrec n c1, substrec (n+1) c2)
   | Prod(name,c1,c2)   -> Prod(name,substrec n c1, substrec (n+1) c2)
   | Ind(stamp,c,l,Specifr(k,s),Specift((kd,lpd),(kn,lpn)))
                        -> Ind(stamp,substrec n c, map (substrec (n+1)) l,
                               Specifr(k,map (fun (i,x)->(i,substrec n x)) s),
                               Specift((kd,map (substrec (n+1)) lpd),
                                       (kn,map (substrec (n+1)) lpn)))
   | Construct(i,c) -> Construct(i,substrec n c)
   | Rec(b,lpf,c)   -> Rec(b,map (substrec n) lpf,substrec n c)
   | x              -> x ;;

let asubst_closedn lam n = substrec n
  where rec substrec n = function
    (Frel(k) as c) -> if k=n then lam
                      else if k<n then annot_of_fterm(c) else annot_of_fterm(Frel (k-1))
  | Fapp(c1,c2)         -> Aapp(substrec n c1,substrec n c2)
  | Flambda(name,c1,c2) -> Alambda(name,fterm_of_annot(substrec n c1),substrec (n+1) c2)
  | Fprod(name,c1,c2) -> annot_of_fterm(Fprod(name,fterm_of_annot(substrec n c1),fterm_of_annot(substrec (n+1) c2)))
  | Find(stamp,c,lc, Fspecifr(k,s),lP) ->
                     annot_of_fterm(Find(stamp,fterm_of_annot(substrec n c), map fterm_of_annot (map (substrec (n+1)) lc),
                           Fspecifr(k, map (fun (i,x)->(i,fterm_of_annot(substrec n x))) s),map fterm_of_annot (map (substrec (n+1)) lP)))
  | Fconstr(i,c)      -> annot_of_fterm(Fconstr(i,fterm_of_annot(substrec n c)))
  | Frec(P::lf,c)     -> Arec(fterm_of_annot(substrec n P),map (substrec n) lf,substrec n c)
  | x                 -> annot_of_fterm(x) ;;

let asubst_closed lam = substrec 1
  where rec substrec n = function
    (Arel(k) as c)              -> if k=n then lam
                                   else if k<n then c else Arel(k-1)
  | Aapp(c1,c2)                 -> Aapp(substrec n c1,substrec n c2)

  | Alambda(name,c1,c2)         -> Alambda(name,fsubst_closedn (fterm_of_annot lam) n c1,substrec (n+1) c2)
  | Alambdacom(name,c1,c2)      -> Alambdacom(name,subst_closedfn n (fterm_of_annot lam) c1,substrec (n+1) c2)
  | Aprod(name,c1,c2)           -> Aprod(name,fsubst_closedn (fterm_of_annot lam) n c1,fsubst_closedn (fterm_of_annot lam) (n+1) c2)
  | Aind(stamp,c,lc,Fspecifr(k,s),lP) ->
                  Aind(stamp,fsubst_closedn (fterm_of_annot lam) n c,
                        map (fsubst_closedn (fterm_of_annot lam) (n+1)) lc,
                        Fspecifr(k,map (fun (i,x)->(i,fsubst_closedn (fterm_of_annot lam) n x)) s),
                        map (fsubst_closedn (fterm_of_annot lam) (n+1)) lP)
  | Aconstr(i,c)                -> Aconstr(i,fsubst_closedn (fterm_of_annot lam) n c)
  | Arec(P,lf,c)                -> Arec(fsubst_closedn (fterm_of_annot lam) n P,map (substrec n) lf,substrec n c)
  | Arecursion(wf,A,P,R)        -> Arecursion(wf,A,P,R)
  | Annot(c,P)                  -> Annot(substrec n c,subst_closedfn n (fterm_of_annot lam) P)
  | x                           -> x ;;

let asubst_closed2 lam lam' = substrec 1
  where rec substrec n = function
    (Arel(k) as c)              -> if k=n then lam
                                   else if k<n then c else Arel(k-1)
  | Aapp(c1,c2)                 -> Aapp(substrec n c1,substrec n c2)

  | Alambda(name,c1,c2)         -> Alambda(name,fsubst_closedn (fterm_of_annot lam) n c1,substrec (n+1) c2)
  | Alambdacom(name,c1,c2)      -> Alambdacom(name,subst_closedn n lam' c1,substrec (n+1) c2)
  | Aprod(name,c1,c2)           -> Aprod(name,fsubst_closedn (fterm_of_annot lam) n c1,fsubst_closedn (fterm_of_annot lam) (n+1) c2)
  | Aind(stamp,c,lc,Fspecifr(k,s),lP) ->
                  Aind(stamp,fsubst_closedn (fterm_of_annot lam) n c,
                        map (fsubst_closedn (fterm_of_annot lam) (n+1)) lc,
                        Fspecifr(k,map (fun (i,x)->(i,fsubst_closedn (fterm_of_annot lam) n x)) s),
                        map (fsubst_closedn (fterm_of_annot lam) (n+1)) lP)
  | Aconstr(i,c)                -> Aconstr(i,fsubst_closedn (fterm_of_annot lam) n c)
  | Arec(P,lf,c)                -> Arec(fsubst_closedn (fterm_of_annot lam) n P,map (substrec n) lf,substrec n c)
  | Arecursion(wf,A,P,R)        -> Arecursion(wf,A,P,R)
  | Annot(c,P)                  -> Annot(substrec n c,subst_closedn n lam' P)
  | x                           -> x ;;

let asubst1 lam = if aclosed lam then asubst_closed lam
                  else asubst_lift lam ;;

let asubst2 lam lam' = if aclosed lam then asubst_closed2 lam lam'
                                      else asubst_lift2 lam lam' ;;

let asubstc_lift lam =  substrec 1
  where rec substrec n = function
    Aapp(c1,c2)            -> Aapp(substrec n c1,substrec n c2)
  | Alambda(name,c1,c2)    -> Alambda(name,c1,substrec (n+1) c2)
  | Alambdacom(name,c1,c2) -> Alambdacom(name,subst_liftn n lam c1,substrec (n+1) c2)
  | Arec(P,lf,c)           -> Arec(P,map (substrec n) lf,substrec n c)
  | Arecursion(wf,A,P,R)   -> Arecursion(wf,A,P,R)
  | Annot(c,P)             -> Annot(substrec n c,subst_liftn n lam P)
  | x                      -> x ;;

let asubstc_closed lam = substrec 1
  where rec substrec n = function
    Aapp(c1,c2)             -> Aapp(substrec n c1,substrec n c2)
  | Alambda(name,c1,c2)     -> Alambda(name,c1,substrec (n+1) c2)
  | Alambdacom(name,c1,c2)  -> Alambdacom(name,subst_closedn n lam c1,substrec (n+1) c2)
  | Arec(P,lf,c)            -> Arec(P,map (substrec n) lf,substrec n c)
  | Arecursion(wf,A,P,R)    -> Arecursion(wf,A,P,R)
  | Annot(c,P)              -> Annot(substrec n c,subst_closedn n lam P)
  | x                       -> x ;;                

let asubstconstr1 lam = if closed lam then asubstc_closed lam
                        else asubstc_lift lam ;;

let ahnftype = apprec []
where rec apprec stack = app_stack
where rec app_stack = function
   (Aprod(_)|Aomega as c) -> if stack=[] then c
                        else anomaly "Cannot be applied"
  | Aconst(_,c)           -> app_stack (annot_of_fterm c)
  | Aapp(c1,c2)           -> apprec (c2::stack) c1
  | Arecursion(wf,A,P,R)  -> Arecursion(wf,A,P,R)
  | Alambda(_,_,c)        -> (match stack with
                                []       -> error "Typing error 3"
                              | c'::rest -> apprec rest (asubst1 c' c))
  | Alambdacom(x,c1,c2)   -> Alambdacom(x,c1,apprec stack c2)
  | Annot(c,P)            -> Annot(apprec stack c,P)
  | _ -> error "ahnftype" ;;

let areduce_app c = redrec [] c
 where rec redrec l x = match x with
    Aconst(_,x)       -> redrec l (annot_of_fterm x)
  | Aapp(c1,c2)       -> redrec (c2::l) c1
  | Alambda(_,_,c)    -> (match l with []     -> x,l
                                    | (a::m)  -> redrec m (asubst1 a c))
  | Alambdacom(_,_,c) -> redrec l c
  | Annot(c,P)        -> redrec l c
  | _                  -> x,l ;;

exception Ainduc ;;

let find_ainductype c = match areduce_app c with
                 (Aind x),l -> x,l
               | _          -> raise Ainduc ;;

let ainductype c = try let (x,l) = find_ainductype c
                       in if l=[] then x
                          else anomaly "Not an inductive type 1"
                   with Ainduc -> anomaly "Not an inductive type 2" ;;

let aconv t1 t2 = fconv (fterm_of_annot t1) (fterm_of_annot t2) ;;

let rec aredapplist c = function
    []   -> c
  | a::l -> match c with
               Alambda(_,_,c') -> aredapplist (asubst1 a c') l
             | _               -> aapplist (Aapp(c,a)) l ;;

let atype_elim_args ind P
        = map (function pn -> aredapplist(asubst1 ind pn) [annot_of_fterm P]);;

let atype_elim ct lft Pt P =
     try let ((stamp,t,x1,x2,lpn as ind),la) = find_ainductype ct in
     let i = Find(stamp,t,x1,x2,lpn) in
         if (fconv t (fterm_of_annot Pt)) & (for_all2eq fconv (ftype_elim_args i P lpn) (map fterm_of_annot lft))
         then aapplist (annot_of_fterm P) la 
         else error "Ill-typed elimination"
     with Ainduc -> error "Ill-formed elimination 4";;

let rec atype_constr rel = function
    Avar(_,t)             -> annot_of_fterm t
  | Aconst(_,v)           -> atype_constr rel (annot_of_fterm v)
  | Arel(n)               -> alift n (nth rel n)
  | Aomega                -> raise Omega
  | Aapp(v1,v2)         -> (match ahnftype (atype_constr rel v1) with 
          Aprod(_,u1,u2)-> if (aconv (atype_constr rel v2) (annot_of_fterm u1))
                           then (asubst1 v2 (annot_of_fterm u2))
                           else error "Ill_typed 1"
        | _             -> anomaly "atype_constr 1")
  | Alambda(n,t,v)      -> Aprod(n,t,fterm_of_annot(atype_constr ((annot_of_fterm t)::rel) v))
  | Alambdacom(n,t,v)   -> atype_constr rel (alift (-1) v)
  | Aprod(_,t,t1)       -> atype_constr ((annot_of_fterm t)::rel) (annot_of_fterm t1)
  | Aind(_,t,_,_,_)       -> annot_of_fterm t
  | Aconstr(i,ind)      -> let (_,_,lc,_,_) = ainductype (annot_of_fterm ind)
                              in annot_of_fterm(fsubst1 ind (nth lc i))
  | Arec(P,lpf,c)       -> (match lpf with
                              [] -> error "ill-formed elimination"
                            | _  -> let ct = atype_constr rel c
                                    and lpft = map (atype_constr rel) lpf
                                    and Pt = atype_constr rel (annot_of_fterm P)
                                      in atype_elim ct lpft Pt P)
  | Arecursion(wf,A,P,R)  -> atype_constr rel (annot_of_fterm (Fapp(Fapp(wf,A),P)))
  | Annot(c,P)            -> atype_constr rel c
  | _                     -> anomaly "atype_constr 2" ;;

let atype_term = atype_constr ;;

let atype_of t = try atype_term [] t
                    with Omega   -> error "Fomega"
                       | reraise -> raise(reraise) ;;

let amake_elimination c P = 
 let ct = atype_of c in
 let ((stamp,t,lc,_,lpn as ind),la) = find_ainductype ct in
     let i = Aind(ind) in
 let lft = ftype_elim_args (fterm_of_annot i) (fterm_of_annot P) lpn in
     (list_it (fun ft e -> Alambda(Name(id_of_string"f"),ft,e)) lft 
              (Arec(fterm_of_annot P,map annot_of_fterm (frel_list 0 (length lpn)),c))) ;;

let amake_elimination_rel rel c P = 
 let ct = atype_constr rel c in
 let ((stamp,t,lc,_,lpn as ind),la) = find_ainductype ct in
     let i = Aind(ind) in
 let lft = ftype_elim_args (fterm_of_annot i) (fterm_of_annot P) lpn in
     (list_it (fun ft e -> Alambda(Name(id_of_string"f"),ft,e)) lft 
              (Arec(fterm_of_annot P,map annot_of_fterm (frel_list 0 (length lpn)),c))) ;;

let aabs_implicit c = Alambda(Anonymous,Fimplicit,c) ;;

let amake_rec P F n = iterate aabs_implicit (n+1)
                           (Arec(P,map (alift (n+1)) F,Arel 1)) ;;

exception Aelim ;;

let rec ared_elimination (P,lf,c) =
 let ahnfconstruct c = hnfstack [] c
    where rec hnfstack stack = 
    function (Aconst(_,c) as x)   -> hnfstack stack (annot_of_fterm c)
           | Aapp(c1,c2)          -> hnfstack (c2::stack) c1
       | Alambda(_,_,c)       ->
                   (match stack with 
                       []       -> anomaly "Cannot be a constructor"
                    |  c'::rest -> hnfstack rest (asubst1 c' c))
       | Alambdacom(_,_,c)    -> hnfstack stack c
       | Arec(x)              -> hnfstack stack (ared_elimination x)
       | Arecursion(wf,A,P,R) -> hnfstack stack (annot_of_fterm (Fapp(Fapp(wf,A),P)))
       | Annot(c,P)           -> hnfstack stack c 
           | Aconstr(c)           -> (c,stack)
       | _                    -> raise Aelim
in
    let (i,ityp),l = ahnfconstruct c in
    let (_,_,_,Fspecifr(n,lt),_) = finductype ityp
    and fi = nth lf i
    in try let ti = caml_assoc i lt
           and F = amake_rec P lf n 
           in aredapplist (annot_of_fterm ti) (fi::F::l)
    with Failure _ -> aredapplist fi l ;;

let rec anf c = anf_app [] c
and anf_app stack =
  let applist_stack c = aapplist c (map anf stack) in function
    Alambda(name,c1,c2)           -> (match stack with
                          []       -> Alambda(name,fnf c1,anf c2)
                        | a1::rest -> (match c2 with
                                        Annot(c,P) -> Aapp(Alambda(name,fnf c1,anf c2),a1)
                                      | _  -> anf_app rest (asubst1 a1 c2)))
  | Alambdacom(name,c1,c2)        -> applist_stack (Alambdacom(name,c1,anf c2))
  | Aprod(name,c1,c2)             -> Aprod(name,fnf c1,fnf c2)
  | Aapp(con1,con2)               -> anf_app (con2::stack) con1
  | Aind(stamp,c,lc,Fspecifr(k,s),lP)   ->
                applist_stack (Aind(stamp,fnf c,map fnf lc,
                                    Fspecifr(k,map (fun (i,x)->(i,fnf x)) s),
                                    map fnf lP))
  | Aconstr(i,c)                  -> applist_stack (Aconstr(i,fnf c))
  | Arec((P,lf,x) as c)           -> (try anf_app stack (ared_elimination c)
                          with Aelim ->
                                   applist_stack(Arec(fnf P,map anf lf,anf x)))
  | Arecursion(wf,A,P,R)          -> applist_stack (Arecursion(fnf wf,A,P,R))
  | Annot(c,P)                    -> applist_stack (Annot(anf c,P))
  | x                             -> applist_stack x ;;

let areduce  = redrec []
  where rec redrec largs x = match x with
    Aconst(_,x)       -> redrec largs (annot_of_fterm x)
  | Alambda(n,t,c)    -> (match largs with
                            []      -> (x,largs)
                          | a::rest -> redrec rest (asubst1 a c))
  | Alambdacom(n,t,c) -> redrec largs c
  | Aapp(c1,c2)       -> redrec (c2::largs) c1
  | Arec(c)           -> (try  redrec largs (ared_elimination c)
                          with Aelim -> (x,largs))
  | Annot(c,P)        -> redrec largs c
  | _                 -> (x,largs) ;;


let rec fconv2 term1 term2 = feqappr2 (fapproxim term1,fapproxim term2)
and feqappr2 = function
       (Fabstraction(c1,c2),Fabstraction(c'1,c'2)) -> fconv2 c1 c'1
                                                    & fconv2 c2 c'2
     | (Fproduct(c1,c2),Fproduct(c'1,c'2)) -> fconv2 c1 c'1
                                            & fconv2 c2 c'2
     | (Fvariable(n1,l1),Fvariable(n2,l2)) -> (n1=n2) &
                                              for_all2eq fconv2 l1 l2
     | (Fomegaconst,Fomegaconst) -> true
     | ((Fconstant(s1,l1) as appr1),(Fconstant(s2,l2) as appr2)) ->
            (* try first intensional equality *)
           ((s1=s2) & (for_all2eq fconv2 l1 l2))
           (* else expand the second occurrence (should it be the first?) *)
        or feqappr2(appr1,fexpand appr2)
     | ((Fconstant(s1,l1) as appr1),appr2) -> feqappr2(fexpand appr1,appr2)
     | (appr1,(Fconstant(s2,l2) as appr2)) -> feqappr2(appr1,fexpand appr2)
     | Finductive(_,_,[Fprod(_,c,Frel(_))],_),x -> feqappr2(fapproxim c,x)
     | x,Finductive(_,_,[Fprod(_,c,Frel(_))],_) -> feqappr2(fapproxim c,x)
     | (Finductive(stamp1,c1,l1,stack1),Finductive(stamp2,c2,l2,stack2)) ->
        stamp1 = stamp2 &
        (fconv2 c1 c2) & (for_all2eq fconv2 l1 l2) & 
        (for_all2eq fconv2 stack1 stack2)
     | (Fconstructor(i1,c1,l1),Fconstructor(i2,c2,l2)) ->
        (i1=i2) & (fconv2 c1 c2) & (for_all2eq fconv2 l1 l2)
     | (Felimination(l1,m1),Felimination(l2,m2)) ->
            (for_all2eq fconv2 l1 l2) & (for_all2eq fconv2 m1 m2)
     | Fimplicitconst,Fimplicitconst -> true
     | _ -> false;; 

let aconv2 t1 t2 = fconv2 (fterm_of_annot t1) (fterm_of_annot t2) ;;

let atype_elim2 ct lft Pt P =
     try let ((stamp,t,x1,x2,lpn as ind),la) = find_ainductype ct in
     let i = Find(stamp,t,x1,x2,lpn) in
         if (fconv2 t (fterm_of_annot Pt)) & (for_all2eq fconv2 (ftype_elim_args i P lpn) (map fterm_of_annot lft))
         then aapplist (annot_of_fterm P) la 
         else error "Ill-typed elimination"
     with Ainduc -> error "Ill-formed elimination 4";;

let rec atype_constr2 rel = function
    Avar(_,t)             -> annot_of_fterm t
  | Aconst(_,v)           -> atype_constr2 rel (annot_of_fterm v)
  | Arel(n)               -> alift n (nth rel n)
  | Aomega                -> raise Omega
  | Aapp(v1,v2)           -> (match ahnftype (atype_constr2 rel v1) with 
          Aprod(_,u1,u2) -> if (aconv2 (atype_constr2 rel v2) (annot_of_fterm u1))
                           then (asubst1 v2 (annot_of_fterm u2))
                           else error "Ill_typed 1"
        | _              -> anomaly "atype_constr 1")
  | Alambda(n,t,v)        -> Aprod(n,t,fterm_of_annot(atype_constr2 ((annot_of_fterm t)::rel) v))
  | Alambdacom(n,t,v)     -> atype_constr2 rel (alift (-1) v)
  | Aprod(_,t,t1)         -> atype_constr2 ((annot_of_fterm t)::rel) (annot_of_fterm t1)
  | Aind(_,t,_,_,_)       -> annot_of_fterm t
  | Aconstr(i,ind)        -> let (_,_,lc,_,_) = ainductype (annot_of_fterm ind)
                              in annot_of_fterm(fsubst1 ind (nth lc i))
  | Arec(P,lpf,c)         -> (match lpf with
                              [] -> error "ill-formed elimination"
                            | _  -> let ct = atype_constr2 rel c
                                    and lpft = map (atype_constr2 rel) lpf
                                    and Pt = atype_constr2 rel (annot_of_fterm P)
                                      in atype_elim2 ct lpft Pt P)
  | Arecursion(wf,A,P,R)  -> atype_constr2 rel (annot_of_fterm (Fapp(Fapp(wf,A),P)))
  | Annot(c,P)            -> atype_constr2 rel c
  | _                     -> anomaly "atype_constr 2" ;;

let atype_term2 = atype_constr2 ;;

let atype_of2 t = try atype_term2 [] t
                    with Omega   -> error "Fomega"
                       | reraise -> raise(reraise) ;;

let remove_annote = function
    Annot(c,P) -> c
  | c          -> c ;;

let amake_elimination2 c P = 
 let ct = atype_of2 c in
 let ((stamp,t,lc,_,lpn as ind),la) = find_ainductype ct in
     let i = Aind(ind) in
 let lft = ftype_elim_args (fterm_of_annot i) (fterm_of_annot P) lpn in
     (list_it (fun ft e -> Alambda(Name(id_of_string"f"),ft,e)) lft 
              (Arec(fterm_of_annot P,map annot_of_fterm (frel_list 0 (length lpn)),c))) ;;
