(* System functions for interactive use. *)

#open "modules";;
#open "unix";;
#open "types";;
#open "globals";;
#open "symtable";;
#open "meta";;
#open "misc";;
#open "obj";;

let zero x = (x := 0);;

#open "float";;
#open "sys";;
let utime () = (times()).tms_utime;;
type counter = {mutable cumul : float;
                mutable from : float;
                mutable started : bool;
                calls : int ref};;

let make_ctr () = {cumul = 0.0; calls = ref 0 ; from = 0.0; started = false};;

let init ctr = (ctr.cumul <- 0.0;
                zero ctr.calls;
                ctr.started <- false);;

let start ctr = if not ctr.started then
    (ctr.from <- utime();
     ctr.started <- true);;

let stop ctr =
    (ctr.cumul <- ctr.cumul + (utime() - ctr.from);
     incr ctr.calls;
     ctr.from <- utime();
     ctr.started <- false);;

#close "float";;

(* Monitoring *)

let profile_env = ref ([] : (int * (obj * string * counter)) list);;

let rec profile_instr obj ty ctr =
  match (type_repr ty).typ_desc with
    Tarrow(t1,t2) ->
      repr(fun arg ->
        start ctr;
        try
          let res = (magic_obj obj : obj -> obj) arg in
            stop ctr;
            profile_instr res t2 ctr
        with exc ->
          stop ctr;
          raise exc)
  | _ -> obj
;;

let profile name =
  try
    let val_desc = find_value_desc (GRname name) in
    let pos = get_slot_for_variable val_desc.qualid in
      if mem_assoc pos !profile_env then begin
        prerr_string ">> "; prerr_string name;
        prerr_endline " is already profiled."
      end else begin
      let ctr = make_ctr() in
        profile_env := (pos, (global_data.(pos),name,ctr)) :: !profile_env;
        global_data.(pos) <-
          profile_instr global_data.(pos) val_desc.info.val_typ ctr;
        prerr_string ">> "; prerr_string name;
        prerr_endline " is now profiled."
      end
  with Desc_not_found ->
    prerr_string ">> "; prerr_string name; prerr_endline " is undefined."
;;

let graph_of_hashtab htab =
let graph = ref [] in
    hashtbl__do_table (fun key val -> graph := (key,val)::!graph) htab;
    !graph
;;

let funs_of_module modname =
let mtab = (hashtbl__find module_table modname).mod_values in
let mgraph = graph_of_hashtab mtab
in map fst mgraph
;;

let unprofile name =
  try
    let val_desc = find_value_desc (GRname name) in
    let pos = get_slot_for_variable val_desc.qualid in
    let rec except = function
      [] -> prerr_string ">> "; prerr_string name;
            prerr_endline " was not profiled.";
            []
    | (pos',(obj,_,ctr) as pair)::rest ->
        if pos == pos' then begin
          global_data.(pos) <- obj;
          prerr_string ">> "; prerr_string name;
          prerr_endline " is no more profiled.";
          rest
        end else
          pair :: except rest
    in
      profile_env := except !profile_env;
      ()
  with Desc_not_found ->
    prerr_string ">> "; prerr_string name; prerr_endline " is undefined."
;;

let init_profiling() =
    do_list (fun (n,(obj,_,ctr)) -> init ctr) !profile_env
;;

let report_profiling () =
    map (fun (n,(_,name,ctr)) -> (name,(ctr.cumul,!(ctr.calls)))) !profile_env
;;

let trace_modules l = do_list (fun s -> do_list trace (funs_of_module s)) l;;
let untrace_modules l = do_list (fun s -> do_list untrace (funs_of_module s)) l;;
