(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                             pretty.ml                                    *)
(****************************************************************************)

(* A Pretty-Printer for the Calculus of Inductive Constructions *)

#open "std";;
#open "univ";;
#open "initial";;
#open "term";;
#open "termfw";;
#open "extraction";;
#open "search";;
#open "machine";;
#open "printer";;
#open "pp_control";;
#open "pp";;
#open "stdpp";;

let print_typed_value (val,typ) =
  [< pr val; prs "     : "; pr typ >];;

(* To be improved; the type should be used to provide the types in the
   abstractions. This should be done recursively inside prterm, so that
   the pretty-print of a proposition (P:(nat->nat)->Prop)(P [u]u)
   synthesizes the type nat of the abstraction on u *)

let string_of = function
     Name(str) -> string_of_id str
   | Anonymous -> "Anonymous";;

let print_var name typ =
    [< prs "*** [" ; 'S name ; prs " :";
     prterm typ; 
     prs "]";line() >];;

let print_context with_values = 
    let sep = if with_values then " = " else " : " in print_rec 
  where rec print_rec  = function
    (Vardecl(Decl(name,Judge(typ,_,_),_),_,_),_)::rest ->
    [< print_rec rest;
       print_var (string_of name) typ >]
  | (Constdecl(Def(name,Judge(val,typ,_),_),_,_),_)::rest ->
    [< print_rec rest;
       'S(string_of name) ; 'S sep;
       if with_values then print_typed_value (val,typ) else pr typ >]
  | (Value(Judge(val,typ,_),_),_)::rest ->
    [< print_rec rest;
       prs(" Stacked value" ^ sep);
       if with_values then print_typed_value (val,typ) else pr typ >]
  | (Loaded _,_)::rest -> [< print_rec rest ; 'S" LoadPath Marker" ; 'FNL >]
  | (Cast(Judge(val,typ,_),_),_)::rest ->
    [< print_rec rest;
       prs(" Stacked cast" ^ sep);
       if with_values then print_typed_value (val,typ) else pr typ >]
  | (Scope(strength),_)::rest -> 
    [< print_rec rest;
       prs " >>>>>>> Scope " ; 'FNL >]
  | (Section(str),_)::rest -> 
        [< print_rec rest;
         prs(" >>>>>>> Section " ^ str) ; 'FNL >]
  | [] -> line();;

let print_full_context () = print_context true (read_context());;

let print_full_context_typ () = print_context false (read_context());;

(* For printing an inductive definition with
   its constructors and elimination,
   assume that the declaration of constructors and eliminations
   follows the definition of the inductive type *)

exception Notinduc;;

let rec is_induc_def = function
    Ind(_,_,lc,_,Specift((kd,_),(kn,_)))
                      -> (length lc, max (length kd) (length kn))
  | Lambda(_,_,c)   -> is_induc_def c
  | _               -> raise Notinduc;;

let info_constr name =
let deb,m = chop_context name (read_context()) in
match m with [] -> error ((string_of_id name) ^ " not declared")
          | (Vardecl(Decl(Name(s),Judge(typ,_,_),_),_,_),_)::_      ->
              print_var (string_of_id s) typ
          | (Constdecl(Def(Name(s),Judge(val,typ,_),_),_,_),_)::_ ->
            (try let  nc,nelim =  is_induc_def val in
                 let (constrs,rest) = chop_list nc deb in
                 let elims = first_n nelim rest
                 in
                     [< prs(" >>>>>>> Inductive Definition");line();
                      print_id s ; prs " = "; print_typed_value (val,typ);
                      prs(" >> with constructors :");
                      print_context true (rev constrs);
                      prs(" >> and elimination combinators :");
                      print_context false (rev elims) >]
             with Notinduc ->
                  [< print_id s ; prs" = "; print_typed_value (val,typ) >])
          | _ -> [< >];;

let print_crible_default name =
  let const = global name
  in let rec head_const = function
      Prod(_,_,c) -> head_const c
    | App(c,_)    -> head_const c
    | def         -> def
  in let rec print_rec  = function
    (Vardecl(Decl(namec,Judge(typ,_,_),_),_,_),_)::rest ->
    [< print_rec rest;
       if (head_const typ)=const then
           [< 'S(string_of namec) ; 'S":";pr typ >]
           else [< >] >]
  | (Constdecl(Def(namec,Judge(val,typ,_),_),_,_),_)::rest ->
    [< print_rec rest;
       if (head_const typ)=const then
           [< 'S(string_of namec); 'S":";pr typ >]
           else [< >] >]
  |   _::rest -> print_rec rest
  |   []      -> [< >]
  in try print_rec (read_context ())
     with Undeclared -> error ((string_of_id name) ^ " not declared");;

let print_crible_extend name =
  let const = global name
  and and_const = global (id_of_string "and")
  and or_const = global (id_of_string "or")
  in let rec head_const = function
      Prod(_,_,c) -> head_const c
    | App(c,_)    -> head_const c
    | def         -> def
  in let rec const_in = function
      Prod(_,_,c) -> const_in c
    | App(c,d)    -> (const_in c) or ((mem (head_const c) [and_const;or_const]) &
                                        (const_in d))
    | def         -> def = const
  in let rec print_rec  = function
    (Vardecl(Decl(namec,Judge(typ,_,_),_),_,_),_)::rest ->
     [< print_rec rest;
        if (const_in typ) then [< 'S(string_of namec) ; 'S":";pr typ>] else [< >]>]
  | (Constdecl(Def(namec,Judge(val,typ,_),_),_,_),_)::rest ->
     [< print_rec rest;
        if (const_in typ) then [< 'S(string_of namec); 'S":";pr typ>] else [< >]>]
  |   _::rest -> print_rec rest
  |   []      -> [< >]
  in try print_rec (read_context ())
     with Undeclared -> error ((string_of_id name) ^ " not declared");;

let PRINT_CRIBLE = ref [print_crible_default] ;;

let set_print_crible_fun f  = PRINT_CRIBLE := f::(!PRINT_CRIBLE); ();;
let reset_print_crible_fun() = PRINT_CRIBLE := tl(!PRINT_CRIBLE); ();;

let print_crible s= (hd !PRINT_CRIBLE) s;;

let read_sec_context sec =
    let rec get_cxt in_cxt = function
        ((Section(str),_) as hd)::rest -> if str = sec then (hd::in_cxt)
                                          else get_cxt (hd::in_cxt) rest
      | [] -> []
      | hd::rest -> get_cxt (hd::in_cxt) rest in
    let cxt = (read_context()) in
    rev (get_cxt [] cxt);;

let print_sec_context sec = print_context true (read_sec_context sec);;

let print_sec_context_typ sec = print_context false (read_sec_context sec);;

let print_val_default () =
     let (_,val,typ,_,_) = read_state() in print_typed_value(val,typ);;

let PRINT_VAL = ref [print_val_default];;

let set_print_val_fun f = PRINT_VAL := f::(!PRINT_VAL); ();;

let reset_print_val_fun() = PRINT_VAL := tl(!PRINT_VAL); ();;

let print_val() = (hd !PRINT_VAL)();;

let print_name_default name = 
    try (match (global name) with
        Const(Def(_,Judge(c,t,_),_)) -> print_typed_value(c,t)
      | Var(Decl(_,Judge(t,_,_),_))    -> print_var (string_of_id name) t
      | _ -> failwith "print_name")
    with Undeclared -> error ((string_of_id name) ^ " not declared");;

let PRINT_NAME = ref [print_name_default];;

let set_print_name_fun f = PRINT_NAME := f::(!PRINT_NAME); ();;

let reset_print_name_fun() = PRINT_NAME := tl(!PRINT_NAME); ();;

let print_name s = (hd !PRINT_NAME) s;;

let print_type name =
    try (match (global name) with
        Const(Def(_,Judge(c,t,_),_)) -> [< print_id name ; 'S" : ";pr t >]
      | Var(Decl(_,Judge(t,_,_),_))    -> print_var (string_of_id name) t
      | _ -> failwith "print_type")
    with Undeclared -> error ((string_of_id name) ^ " not declared");;

let print_evaluate () =
    let (_,val,typ,_,_) = read_state() in
        [< prs "     = ";print_typed_value (simplify val,typ) >];;

let print_compute () = 
    let (_,val,typ,_,_) = read_state() in
        [< prs "     = ";print_typed_value (nf val,typ) >];;

let print_status () = 
    [< print_full_context();line();
     print_val() >];;

let print_local_context () =
let env = read_context() in
   [< print_var_rec env;  print_last_const env >]
where rec print_var_rec = function 
    [] -> line()
  | ((Vardecl (Decl(name,Judge(typ,_,_),_),_,_)),_)::rest ->
    [< print_var_rec rest;
       line ();
       print_var (string_of name) typ >]
 | _::rest -> print_var_rec rest
and print_last_const = function
   ((Constdecl(Def(name,Judge(val,typ,_),_),_,_)),_)::rest ->
   [< print_last_const rest;
      'S(string_of name) ;'S" = ";
      print_typed_value (val,typ) >]
 | ((Value(Judge(val,typ,_),_)),_)::rest ->
   [< print_last_const rest;
      prs " Stacked value = ";
      print_typed_value (val,typ) >]
 | (Loaded _,_)::rest -> print_last_const rest
 | ((Cast(Judge(val,typ,_),_)),_)::rest ->
   [< print_last_const rest;
      prs " Stacked cast = ";
      print_typed_value (val,typ) >]
 | (Vardecl(_,_,_),_)::_ -> [< >]
 | _ -> [< >];;

let fprint_var name typ =
    [< prs ("*** [" ^ name ^ " :");
       fprterm typ; 
       prs "]";line() >];;

let print_extracted_name name =
    try (match (global name) with
        Const(Def(_,_,cont)) ->
           (match cont with
               Inf(n,Fconst(_,d)) ->
               [< if n = Fw then prs ((string_of_id name) ^" = ")
                  else prs ((string_of_id name) ^" ==> ");
                   fprterm d; line() >]
             | _                  -> error "Non informative term")
      | Var(Decl(_,_,cont))  ->
           (match cont with
               Inf(_,Fvar(name,t)) -> fprint_var (string_of name) t
             | _                   -> error "Non informative term")
      | _ -> anomaly "should be a variable or constant")
    with Undeclared -> error ((string_of_id name) ^ " not declared");;

let print_extraction () = [< print_rec (read_context()); line () >]
 where rec print_rec = function
    ((Constdecl(Def(_,_,cont),_,_)),_)::rest ->
    [< print_rec rest;
      (match cont with
            Inf(n,Fconst(name,d))->
            let s = string_of name in
                [< (if n = Fw then prs (s ^" = ") else prs (s ^" ==> "));
                    fprterm d; line() >]
          | _      -> [< >]) >]
  | ((Value(_,cont)),_)::rest ->
      [< print_rec rest;
       let s="Stacked value"
      in (match cont with Inf(n,d) ->
        [< if n = Fw then prs (s ^" = ") else prs (s ^" ==> ");
            fprterm d; line() >]
                         | _      -> [< >]) >]
  | (Loaded _,_)::rest -> print_rec rest
  | ((Cast(_,cont)),_)::rest ->
      [< print_rec rest;
      let s="Stacked cast"
      in (match cont with Inf(n,d) ->
      [< if n = Fw then prs (s ^" = ") else prs (s ^" ==> ");
          fprterm d; line() >]
                        | _      -> [< >]) >]
  | ((Vardecl(Decl(_,_,cont),_,_)),_)::rest ->
        [< print_rec rest;
        (match cont with Inf(_,Fvar(name,t)) ->
                            fprint_var (string_of name) t
                       | _      -> [< >]) >]
  | _::rest -> print_rec rest
     
  | [] -> line();;

let print_contents () = let (_,val,typ,_,inf) = read_state() in
    [< prs "INF = ";
       match inf with 
       Inf(n,c) ->
       [< if n = Fw then prs "Programme : " else prs "Extraction : ";
        fprterm c;line() >]
       | Logic  -> [< prs "Logical term"; line() >] >];;

let print_extracted_context () =
let env = read_context() in [< print_var_rec env; print_last_constants env >]
where rec print_var_rec = function 
    ((Vardecl(Decl(name,Judge(typ,_,_),cont),_,_)),_)::rest ->
        [< print_var_rec rest ; line();
           match cont with
             Inf(_,t) -> fprint_var (string_of name) t
            | _      -> [< >] >]
  |  _::rest -> print_var_rec rest
  | [] -> line()
and print_last_constants = function 
    ((Constdecl(Def(name,Judge(c,typ,_),cont),_,_)),_)::rest ->
         [< print_last_constants rest;
         let s=string_of name
         in (match cont with Inf(n,d) -> 
            [< if n = Fw then  prs (s ^" = ") else prs (s ^" ==> ");
                fprterm d; line() >]
                           | _      -> [< >]) >]
 |  ((Value(Judge(c,typ,_),cont)),_)::rest ->
         [< print_last_constants rest;
         let s="Stacked value"
         in (match cont with Inf(n,d) ->
             [< if n = Fw then  prs (s ^" = ") else prs (s ^" ==> ");
                 fprterm d; line() >]
                           | _      -> [< >]) >]
 |  (Loaded _,_)::rest -> print_last_constants rest
 |  ((Cast(Judge(c,typ,_),cont)),_)::rest ->
         [< print_last_constants rest;
         let s="Stacked cast"
         in (match cont with Inf(n,d)  ->
             if n = Fw then  prs (s ^" = ") else prs (s ^" ==> ") 
                           | _      -> [< >]) >]
 | _ -> [< >];;

dflt_gp.ellipsis := " ... ";;
dflt_gp.format_ellipsis := false;; (* Pour ne pas avoir une bardee de .... *)
dflt_gp.limit_depth := 1000;; (* So that the printer does not abort *)
dflt_gp.max_depth := 80;;



let print_extracted_vars () =
let env = read_context() in print_var_rec env
where rec print_var_rec = function
    ((Vardecl(Decl(_,_,cont),_,_)),_)::rest ->
        [< print_var_rec rest ; line();
           match cont with
             Inf(_,Fvar(name,t)) -> fprint_var (string_of name) t
            | _      -> [< >] >]
  |  _::rest -> print_var_rec rest
  | [] -> line()
;;

(* for debug *)
let inspect depth = 
    let rec inspectrec n res env = if n=0 or env=[] then res
                                   else inspectrec (n-1) (hd env::res) (tl env)
    in let items = inspectrec depth [] (read_context())
       in print_context false items;;
