(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                            more_util.ml                                  *)
(****************************************************************************)
#open "std";;
#open "initial";;
#open "sys";;
#open "unix";;
#infix "o";;

type 'a option = NONE | SOME of 'a
;;

let rec in_dom a = function
    [] -> false
  | ((k,_)::t) -> (k = a) or (in_dom a t)
;;

let rec nth_from_zero = function
    (0,h::t) -> h
  | (n,h::t) -> nth_from_zero(n-1,t)
;;

let rec nth_from_one = function
    (1,h::t) -> h
  | (n,h::t) -> nth_from_one(n-1,t)
;;

let maxList l = it_list max 0 l
;;

let maxNEList l = it_list max (hd l) l
;;

let in_range (l , h) = function (c:char) ->
    (int_of_char l)<=(int_of_char c) & (int_of_char c)<=(int_of_char h)
;;

let islower = in_range(`a`,`z`);;

let isupper = in_range(`A`,`Z`);;

let isalpha c = islower c or isupper c;;

let isdigit = in_range(`0`,`9`);;

let isxdigit c = in_range(`0`,`9`) c or in_range(`a`,`f`) c or in_range(`A`,`F`) c;;

let isalnum c = isdigit c or isalpha c;;

let is_ident_start c = isalpha c or mem c [`_`;`$`];;

let is_ident_rest c = isalnum c or mem c [`_`;`$`;`'`;`-`];;

let app f = apprec where rec apprec = function
    [] -> ()
  | (h::t) -> (f h;apprec t)
;;

let appLR = app;;

let implode_chars cl =
  let len = list_length cl in
  let dest = create_string len in
      (it_list (function start -> function src ->
                    (set_nth_char dest start src;
                     start + 1))
               0 cl;
       dest)
;;

let explode_chars s =
    let slen = string_length s in
    let rec aux n = if n < slen then (nth_char s n)::(aux (n+1)) else [] in
        aux 0
;;

let plist elem = plist_rec
  where rec plist_rec = function
    [< elem e; plist_rec l >] -> e::l
  | [< >] -> []
;;

let cons(a,b) = a::b;;

let maybe_empty_list_with_sep = fun nil_elem cons_fun sep elem ->
let rec do_rec = function
    [< sep() ; elem e ; do_rec l >] -> cons_fun(e,l)
  | [< >] -> nil_elem
in
    function
    [< elem e ; do_rec l >] -> cons_fun(e,l)
  | [< >] -> nil_elem
;;

let plist_with_sep sep elem =
    maybe_empty_list_with_sep [] cons sep elem
;;

let rec prlist = fun
    elem [] -> [< >]
  | elem (h::t) -> [< elem h  ; prlist elem t >]
;;

let rec prlist_with_sep sep = fun
    elem [] -> [< >]
  | elem [h] -> [< elem h >]
  | elem (h::t) -> [< elem h  ; sep() ; prlist_with_sep sep elem t >]
;;

let string_to_char_stream str = str_rec 0
  where rec str_rec i =
    if i >= string_length str then [< >]
    else [< 'nth_char str i; str_rec(i+1) >]
;;

let ensure_eof f = function
    [< f rslt ; (function [< 'x >] -> raise Parse_error | [< >] -> ()) () >] -> rslt
;;

let char_to_string c = make_string 1 c;;

let rec stream_to_list = function
    [< 'x ; stream_to_list l >] -> x::l
  | [< >] -> []
;;

let rec stream_of_list = function
    [] -> [< >]
  | (h::t) -> [< 'h ; stream_of_list t >]
;;

let string_stream_to_string ss = implode (stream_to_list ss)
;;

let rec string_stream_to_char_stream = function
    [< 'str ; s >] -> [< string_to_char_stream str ;
                         string_stream_to_char_stream s >]
  | [< >] -> [< >]
;;

let atoi_base n =
    fun s -> (it_list (fun r a -> int_of_char(a)-int_of_char(`0`)+n*r) 0 (explode_chars s))
;;

let load_string s = fun () -> [< 's >]
;;

let rec output_string_stream = function
    [< 's ; strm' >] -> (print_string s ; output_string_stream strm')
  | [< >] -> ()
;;

let rec revc = function
    ([],l) -> l
  | (h::t,l) -> revc(t,h::l)
;;

let pure_index(a,l) =
    try index a l with _ -> -1
;;

let rec first = function
    (0,_) -> []
  | (n,h::t) -> h::(first(n-1,t))
;;

let rec uniquize = function
    [] -> []
  | (h::t) -> if mem h t then uniquize t else h::(uniquize t)
;;

let p_atom = function
    [< (stream_check is_ident_start) a ; (plist (stream_check is_ident_rest)) l >] -> implode_chars(a::l)
;;

let number =
    function
    [< '`0`..`9` as d ; s >] -> num_rec [d] s
    where rec num_rec l = function
    [< '`0`..`9` as d; s >] -> num_rec (d::l) s
  | [< >] -> int_of_string(implode_chars(rev l))
;;

let string_for_read_octal s =
  let n = ref 0 in
    for i = 0 to string_length s - 1 do
      n := !n +
        (match nth_char s i with
           `"` | `\\` | `\n` | `\t` -> 2
          | c -> if int_of_char c >= 32 & int_of_char c < 128 then 1 else 4)
    done;
    if !n == string_length s then s else begin
      let s' = create_string !n in
        n := 0;
        for i = 0 to string_length s - 1 do
          begin
            match nth_char s i with
              `"` -> set_nth_char s' !n `\\`; incr n; set_nth_char s' !n `"`
            | `\\` -> set_nth_char s' !n `\\`; incr n; set_nth_char s' !n `\\`
            | `\n` -> set_nth_char s' !n `\\`; incr n; set_nth_char s' !n `n`
            | `\t` -> set_nth_char s' !n `\\`; incr n; set_nth_char s' !n `t`
            | c ->
              let a = int_of_char c in
                if a >= 32 & a < 128 then
                  set_nth_char s' !n c
                else begin
                  set_nth_char s' !n `\\`;
                  incr n;
                  set_nth_char s' !n (char_of_int (48 + a / 64));
                  incr n;
                  set_nth_char s' !n (char_of_int (48 + (a / 8) mod 8));
                  incr n;
                  set_nth_char s' !n (char_of_int (48 + a mod 8))
                end
          end;
          incr n
        done;
        s'
      end
;;

let pair(a,b) = (a,b);;

let rec upto = fun  (from:int) -> fun (bound:int) ->
    if (bound < from) then []
    else (from::(upto (from+1) bound))
;;

let map_trapping_failure f = map_f
    where rec map_f = function
    [] -> []
  | (h::t) -> try (f h)::(map_f t) with _ -> (map_f t)
;;

let string_of_char c = fstring__make_string 1 c
;;

let map_stream f = map_f where rec map_f = function
    [< >] -> [< >]
  | [< 'a ; s >] -> [< f a ; map_f s >]
;;

let map_stream_tokens f = 
map_f where rec map_f = function
    [< 'a ; s >] -> [< '(f a) ; map_f s >]
  | [< >] -> [< >]
;;

let safe_getenv n =
    try getenv n with Not_found -> "$" ^ n
;;

let safe_getlogin () =
    try getlogin()
    with _ -> (getpwuid(getuid())).pw_name
;;

let glob fname =
    let get_revexp_env n =
        rev(explode_chars(safe_getenv n)) in
    let get_revexp_pwdir n =
        rev(explode_chars(getpwnam n).pw_dir) in
    let rec aux sofar = function
        [< '`/` ;
         (function
          [< '`~` ;
           (function
            [< p_atom n >] -> n
          | [< >] -> safe_getlogin()) uname ;
           s >] -> aux (get_revexp_pwdir uname) s
        | [< s >] -> aux (`/`::sofar) s) rslt >] -> rslt
      | [< '`$` ; p_atom a ; s >] ->
        aux ((get_revexp_env a)@sofar) s
      | [< 'c ; s >] -> aux (c::sofar) s
      | [< >] -> sofar
    in
        (implode_chars o rev)
        ((function
          [< '`~` ; s >] -> aux [] [< '`/` ; '`~` ; s >]
        | [< s >] -> aux [] s)
         (string_to_char_stream fname))
;;

let open_trapping_failure open_fun name suffix =
    try open_fun (glob (name^suffix))
    with _ -> error("Can't open " ^ name)
;;

let open_with_suffix open_fun name suffix =
    try open_fun (glob(name ^ suffix))
    with _ -> try open_fun (glob name)
              with _ -> error("Can't open " ^ name)
;;

(* Rendu portable entre Unix/Mac/PC par emploi judicieux
   du module filename. -XL. *)

let open_with_suffix_from_path open_fun path name suffix =
    (if filename__is_absolute (glob name) then
        open_with_suffix open_fun name suffix
    else open_rec path)
where rec open_rec = function
(dir :: rest) -> (try open_with_suffix open_fun (filename__concat dir name) suffix
                  with _ -> open_rec rest)
| [] -> error ("Can't find file " ^ name ^ " on loadpath")
;;

type ('a,'b) union = inl of 'a | inr of 'b;;

#open "stdpp";;
#open "pp";;
let timestamp () = (time(),times());;
let fmt_time_difference (startreal,start) (stopreal,stop) =
    [< 'INT(stopreal - startreal); 'S" secs ";
       'S"(";
       'REAL(sub_float stop.tms_utime start.tms_utime); 'S"u";
       'S",";
       'REAL(sub_float stop.tms_stime start.tms_stime); 'S"s";
       'S")" >]
;;
