(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                             matching.ml                                  *)
(****************************************************************************)

#infix "o";;
#open "std";;
#open "initial";;
#open "univ";;
#open "extraction";;
#open "term";;
#open "printer";;
#open "pp";;
#open "stdpp";;

(* from tactics.ml *)
let rec occur_meta = function
   Prod(_,t,c)   -> (occur_meta t) or (occur_meta c)
 | Lambda(_,t,c) -> (occur_meta t) or (occur_meta c)
 | App(c1,c2)    -> (occur_meta c1) or (occur_meta c2)
 | Meta(_)       -> true
 | _             -> false;;

(* Holds a non-matching pair for error reporting in toplevel *)
let CONFLICT = ref(x where x = prop,prop);;

let report() = let (m,n) = !CONFLICT in
              PPNL [< 'S"Impossible to convert "; pr m;
                      'S"With "; pr n >];;

let simpl M N = if not(occur_meta M) & not(occur_meta N) & conv_x M N then [] else simplrec [(M,N)]
   where rec simplrec = function
  []               -> []
| ((m,n) as pc)::l -> match pc with
      Meta(_),_                       -> pc::(simplrec l)
    | _,Meta(_)                       -> (n,m)::(simplrec l)
    | Lambda(_,t1,c1),Lambda(_,t2,c2) -> simplrec((t1,t2)::((c1,c2)::l))
    | Prod(_,t1,c1),Prod(_,t2,c2)     -> simplrec((t1,t2)::((c1,c2)::l))
    | App(m1,m2),App(n1,n2)           -> simplrec((m1,n1)::((m2,n2)::l))
    | Rec(b1,lpf1,c1),Rec(b2,lpf2,c2) -> if b1=b2 then simplrec((c1,c2)::
                                          (try combine (lpf1,lpf2)
                                           with Invalid_argument _ ->
                                             (CONFLICT:=pc;error "simpl"))
                                          @l)
                                         else (CONFLICT:=pc;error "simpl")
    | Construct(i,c1),Construct(j,c2) -> if i=j then simplrec ((c1,c2)::l)
                                         else (CONFLICT:=pc;error "simpl")
    | Ind(stamp1,t1,lC1,_,_),Ind(stamp2,t2,lC2,_,_) ->
      if stamp1 = stamp2 then simplrec ((t1,t2)::
                                        (try combine (lC1,lC2)
                                         with Invalid_argument _ ->
                                             (CONFLICT:=pc;error "simpl"))
                                        @l)
      else (CONFLICT:=pc;error "simpl")
    | _ -> if not(occur_meta m) & not(occur_meta n) & conv_x m n then simplrec l else (CONFLICT:=pc ;error "simpl");;

(* lexicographic ordering *)

let rec less (l1,l2) = match (l1,l2) with
      (_,[])  -> false
    | ([],_)  -> true
    | (x1::rest1,x2::rest2) -> x1<x2 or (x1=x2 & less(rest1,rest2)) ;;

type subst == (string * constr) list;;

let add_matching s (p,c) = addrec s
where rec addrec = function
    []                 -> [(p,c)]
  | ((p1,c1)::s1 as s) -> 
      if p=p1 then if conv_x c c1 then s 
              else error "Two variable occurrences conflict"
      else if less(p,p1) then (p,c)::s 
      else (p1,c1)::(addrec s1);;

let instance s c = if s=[] then c else (simplify o irec) c
where rec irec = function
   ((Meta p) as u)     -> (try assoc p s with Not_found -> u)
  | App(u1,u2)         -> App(irec u1,irec u2)
  | Lambda(s,c1,c2)    -> Lambda(s,irec c1,irec c2)
  | Prod(s,c1,c2)      -> Prod(s,irec c1,irec c2)
  | Rec(b,lf,c)        -> Rec(b,map irec lf, irec c)
  | Construct(i,c)     -> Construct(i,irec c)
  | Ind(stamp,c,l,Specifr(k,s),Specift((kd,lpd),(kn,lpn)))
                        -> Ind(stamp,irec c, map irec l,
                               Specifr(k,map (fun (i,x)->(i,irec x)) s),
                               Specift((kd,map irec lpd),(kn,map irec lpn)))
  | x                  -> x;;

let matching s M N = matchrec s (simpl (instance s M) N)
where rec matchrec s = function
    []            -> s
  | (Meta p,c)::l -> matchrec (add_matching s (p,c)) l
  | _             -> error "matching";;
