(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                             machine.ml                                   *)
(****************************************************************************)

(********************************************************************)
(*                     The Constructive Engine                      *)
(*                                                                  *)
(*                    _________________________                     *)
(*      ENV          |_________________________|                    *)
(*                                                                  *)
(*      Contains constant definitions, variable declarations,       *)
(*      values waiting to be applied as arguments,                  *)
(*      types waiting to be used for type casting,                  *)
(*      and section headings.                                       *)
(*                                                                  *)
(*      VAR          list of current variable declarations          *)
(*      UNI          universe structure                             *)
(*                            ___     ___     ___     ___           *)
(*      JUDGE                |___|   |___|   |___|   |___|          *)
(*                            VAL     TYP     LEV     INF           *)
(*                                                                  *)
(*      SEC          depth of opened sections                       *)
(*      UPD          tactics added in search since the last var     *)
(*                   or const declaration                           *)
(*                                                                  *)
(********************************************************************)

(* The state of the Constructive Machine consists of:
1. an environment of variables and constants !ENV
2. a current judgement !JUDGE with fields:
- the current construction  !VAL
- its type !TYP
- its level !LEV
- its information contents !INF
3.  The list of variables !VAR is just an optimized access to ENV
    !SEC is an optimized access to the number of opened sections in ENV
    !UNI is the structure of current universes
    !UPD is a temporary storage of auto-tactics
*)

#infix "o";;
#open "std";;
#open "initial";;
#open "univ";;
#open "extraction";;
#open "term";;
#open "search";;
#open "printer";;
#open "pp";;
#open "stdpp";;
#open "more_util";;

(* just to print terms more prettily in error messages *)

let msgpr s t = [< HOV 0 [< 'S s ; 'SPC ; HOV 0 [< prterm t >] ; 'SPC >] ;
                   'CUT >];;

#infix "msgpr";;

(* Structure of the environment *)

(* Strength of constants *)

(* type strength = Strength of int ;; *)

(* Declarations *)
(* type declaration =  Vardecl of variable * strength * updates
                  | Constdecl of constant * strength * updates
                  | Value of judgement * information
                  | Cast of judgement * information
                  | Scope of strength
                  | Section of string
                  | Loaded of string list;; *)

(* type context == (declaration * universes) list;; *)

type state == context * (declaration list) * int *
              constr * constr * level * universes * information * 
              (string * namedtac list) list * updates *
              syntax_table;;

(* A state without saving the SEARCH table for use in command *)
type state_light == context * (declaration list) * int *
              constr * constr * level * universes * information;;

(* All occurrences of Prop share the same universe to avoid proliferation *)
let prop_univ = New_univ();;
let initial_universes = read_uni();;

(* The initial state *)
let LOADPATH = ref ([]:string list);;
let ENV = ref ([]:context)           (* global environment *)
and VAR = ref ([]:declaration list)  (* list of variables (optimisation) *)
and SEC = ref 0                      (* depth of opened sections *)
and VAL = ref (Prop(Null))           (* The current construction *)
and TYP = ref (Type(Null,prop_univ)) (* Its type *)
and LEV = ref (Object)               (* Its level *)
and INF = ref (logic ())             (* Its information contents *)
and UPD = ref ([]:updates)           (* List of added tactics in SEARCH *)
                    (* name,!ENV,!VAR,!VAL,!TYP,!LEV,read_uni(),!INF *)
and STATES = ref ([]:(string * (string * state)) list)
and SAVE = ref ([]: state_light list);;

(* Environment construction *)

(* The only place where ENV is updated *)
let push_env_uni uni decl = ENV:=(decl,uni)::!ENV;;

let push_env decl = push_env_uni (read_uni()) decl;;


(********************************************************************)

(*                 Read the state of the machine                    *)

(********************************************************************)

let read_state   () = (!ENV,!VAL,!TYP,!LEV,!INF)
and read_judge   () = (!VAL,!TYP,!LEV,!INF)
and read_context () = !ENV
and read_var     () = !VAR
and read_sec     () = !SEC
and read_val     () = !VAL
and read_typ     () = !TYP
and read_lev     () = !LEV
and read_inf     () = !INF
and read_upd     () = !UPD
;;


(********************************************************************)

(*                Service routines of the machine                   *)

(********************************************************************)

(* Adding one tactic in search *)
let add_tac_search name_head named_tac =
    let upd = insert_search_upd name_head named_tac
    in UPD:=upd::!UPD;();;

(* Updating part of the environment in the reverse order *)

(* -- UNUSED
let update_tac_env (l:context) =
  UPD:=it_list update_decl (!UPD) (rev l);()
  where update_decl upd = function
           (Vardecl(_,_,lt),_)   -> update_search upd; lt
         | (Constdecl(_,_,lt),_) -> update_search upd; lt
         | _                     -> upd;; *)

let update_tac_decl l =
  if l = [] then () 
  else (UPD:=it_list update_decl (!UPD) (rev l);())
       where update_decl upd = function 
           Vardecl(_,_,lt)   -> update_search upd; lt
        |  Constdecl(_,_,lt) -> update_search upd; lt
        |  _                 -> upd;;

(* exception Undeclared;; *)

let pop_var () = match !VAR with
  _::rest -> VAR:= rest
| []      -> anomaly "VAR empty";;

let search_in_context str = search_rec
where rec search_rec = function 
  [] -> raise Undeclared
| (decl,_)::rest -> match decl with
       Vardecl(Decl(Name(s),_,_),_,_)  -> if s=str then decl else search_rec rest
     | Constdecl(Def(Name(s),_,_),_,_) -> if s=str then decl else search_rec rest
     | _                           -> search_rec rest;;

let search_in_vars str = search_rec
where rec search_rec = function 
  [] -> raise Undeclared
| ((Vardecl(Decl(Name(s),_,_),_,_)) as decl)::rest  ->
                                 if s=str then decl else search_rec rest
| (Vardecl(Decl(Anonymous,_,_),_,_))::rest  -> search_rec rest
| _  -> anomaly "VAR anomalous";;

let search_new_name str = search_in_vars str (read_var ());;

let exists_var str = try search_new_name str; true
                     with Undeclared -> false;;

(* Searching for name of global variable or constant *)
(* Beware : we search variables before constants, so that axioms have
   fast access. Maybe this should be replaced by a hash table of globals *)

(* search : string -> declaration *)
let search str =
   try search_new_name str
   with Undeclared -> search_in_context str (read_context());;

(* returns name if unused, a gensym of it otherwise *)
(* absurd : fights again itself *)
let new_name name =
    let s = stringpart_of_id name in
    let rec next_name n =
        let name' = IDENT(s,n)
        in try search_new_name name';next_name (n+1)
            with Undeclared -> name'
    in
        (try search_new_name name; next_name (-1)
         with Undeclared -> name)
;;

let new_namel name l =
    let s = stringpart_of_id name in
    let rec next_name n =
        let name' = IDENT(s,n)
        in try search_new_name name';next_name (n+1)
            with Undeclared -> if (mem name' l) then next_name (n+1)
                               else name'
    in
        (try search_new_name name; next_name (-1)
         with Undeclared -> if (mem name l) then next_name (-1)
                            else name)
;;

(********************************************************************)

(*                        Initializations                           *)

(********************************************************************)


let reset_update () = update_search (!UPD); UPD:=[]; ();;

let reset_current () =
VAL:=Prop(Null);
TYP:=Type(Null,prop_univ);
LEV:=Object;
INF:=logic ();
();;

(* Initializing the environment *)
let reset_env () = ENV:=[]; VAR:=[]; reset_search();;

(********************************************************************)

(*                The elementary machine operations                 *)

(********************************************************************)

(*   There are two basic kinds of operations:
        A| building up the environment,
     or B| building up the current construction. *)

(* A| Building up the environment. We use the current construction as
new hypothesis, or as new constant. *)

let constr_of_decl = function 
  Vardecl(variable,_,_)   -> Var(variable)
| Constdecl(constant,_,_) -> Const(constant)
| _                     -> anomaly "constr_of_decl";;


(* A1| Introducing a new hypothesis *)

(* Used in assume_name below, and in create_var for synthesis *)
let mk_vardecl name stre =
 let typ = hnftype !TYP
 in let lev = level_of_kind typ
    in Vardecl(Decl(name,Judge(!VAL,typ,lev),inf_var name !INF),
               Strength stre,!UPD);;

(* Used in assume_name below, and in synthesis *)
let enter_variable declaration =
   push_env declaration; 
   VAR:= declaration::!VAR;UPD:=[];
   ();;

(* Var intro, Naming the hypothesis *)
let assume_name name stre =
    let declaration = mk_vardecl name stre
    in enter_variable declaration; declaration;;

let assume str stre = assume_name (Name str) stre; ();;

(* Anonymous hypothesis *)
let postulate stre = assume_name Anonymous stre; ();;

let cast_to_vardecl name stre =
    try
    let (Cast(Judge(valu,typ,lev),info),_) = hd (!ENV) in
        ENV := tl(!ENV);
        Vardecl(Decl(name,Judge(valu,typ,lev),inf_var name info),
                Strength stre,!UPD)
    with Match_failure _ ->
        error "Did not find a cast at the top of the environment - postulate conversion failed"
;;

let assume_cast name stre =
    let declaration = cast_to_vardecl name stre
        in enter_variable declaration; ();;

(* used only in unstack_n_var and inductype_spec *)
let unstack_var () = 
    match !ENV with
      []                 -> error "No hypothesis in current context"
    | ((Vardecl(_,_,lt) as d),_)::rest
                         -> ENV:=rest; update_search !UPD; UPD:=lt; pop_var();d
    | _                  -> error "Last item not a variable";;

(* A2| Declaring a constant *)

let declare_str str strength = 
   try search_new_name str;
       error("Clash with global variable " ^ (string_of_id str))
   (* we forbid to hide a global variable with a constant *)
   with Undeclared ->
            let name = Name(str) in
            let decl =
             Constdecl(Def(name,Judge(!VAL,!TYP,!LEV),inf_const name !INF)
                      ,strength,!UPD)
               in push_env decl; UPD:=[]; decl;;

let declare_decl str n = constr_of_decl (declare_str str (Strength n));;

let declare str n = declare_str str (Strength n); ();;

(* A3| Value for future application *)

let stack_value () = let decl = Value(Judge(!VAL,!TYP,!LEV),!INF)
                     in push_env decl; ();;


(* A4| Type for future casting *)

let stack_cast () = 
 let typ = hnftype(!TYP) 
 in let lev = level_of_kind typ
 in let decl = Cast(Judge(!VAL,typ,lev),!INF)
    in push_env decl; ();;

(* A5| Scope *)

let push_scope strength = push_env (Scope(strength)); ();;

let pop_scope () = pop_scope_rec [] !ENV
where rec pop_scope_rec already = function 
  ((Scope s),_)::rest -> ENV:= rev_append already rest;s
| i::rest            -> pop_scope_rec (i::already) rest
| []                 -> error "No scope in current environment";;

(* B| Building up the current judgement *)

(* Remark: There is no Meta intro, meta-variables are only temporarily
existing during the search for proofs in synthesis mode.
The Rel intro rule is only used by execute, in command execution mode.
The de Bruijn representation is built up
by the subst_var operation during Lambda and Prod intro *)


(* B1| Global (Var/Const) intro *)

let global str = 
 try let decl = search str in
    (match decl with
         Constdecl((Def(n,Judge(_,typ,lev),inf) as val),_,_) -> 
            (TYP:=typ;
             LEV:=lev;
             VAL:=Const(val);
             INF:=inf)
       | Vardecl((Decl(n,Judge(typ,_,lev),inf) as val),_,_) -> 
            (TYP:=typ;
             LEV:=lev;
             VAL:=Var(val);
             INF:=inf)
       | _ -> anomaly "search"
    ); !VAL
 with Undeclared -> error((string_of_id str) ^ " not declared");;

(* consider is used by function construct in command.ml *)
let consider str = global str; ();;

let valof x = match global x with
       Const(c) -> value_of c
     | _ -> error((string_of_id x) ^ " is a variable, not a defined constant")
and typof = type_of o global;;

(* -- UNUSED
(* Popping the value of a constant *)
let consider_value str = 
 try let decl = search str in
    (match decl with
         Constdecl(Def(_,Judge(def,typ,lev),inf),_,_) -> 
            TYP:=typ;
            LEV:=lev;
            VAL:=def;
            INF:=inf_value inf
       | Vardecl(_,_,_) -> error (str ^ " is a variable, not a constant")
    ); ()
 with Undeclared -> error(str ^ " not declared");;

*)

(* B2| Prop intro *)

let proposition cts =
    LEV:=Object;
    VAL:=Prop cts;
    TYP:=Type(cts,prop_univ);
    INF:=inf_kind cts;
    ();;


(* B3| Type intro *)

let type_with_universe (cts,u) =
   let v = Super u in
     (LEV:=Object;
      TYP:=Type(cts,v);
      VAL:=Type(cts,u);
      INF:=inf_kind cts;
      ());;

let new_type cts = type_with_universe(cts,New_univ());;

(* B4| App intro *)

(* Extracts the last pushed argument in environment *)
let unstack_value () =
    match !ENV with
          []                 -> error "No value in current context"
        | (Value(j1,j2),_)::rest -> ENV:=rest; j1,j2
        | _                  -> error "Last item not a value";;

let pop_apply () =
  match hnftype !TYP with
     Prod(_,c1,c2) -> 
       let (Judge(val,typ,lev),inf) = unstack_value()
       and universes = read_uni() (* protection *)
       in if conv_leq typ c1 then (
            VAL:=App(!VAL,val);
            TYP:=subst1 val c2;
            INF:=inf_apply !LEV lev !INF inf; ())
          else (reset_universes universes;
                PPNL
        [< "Illegal application :" msgpr !VAL; 
           "cannot be applied to :" msgpr val; 
           "Since the formal type :" msgpr c1; 
           "does not match the actual type :" msgpr typ >];
        error "Application would violate typings")
   | _ -> PPNL
          [< "Illegal application :" msgpr !VAL; 
             "cannot be applied, since it has a non-functional type :" msgpr !TYP >];
          error "Non-functional construction";;

(* -- UNUSED
let search_value () = search_rec (!ENV,[])
where rec search_rec (before,after) = match before with
    (Value(v),_)::rest -> (v,rest,after)
  | item::rest         -> search_rec (rest,item::after)
  | []                 -> error "No value in current environment";;

let last_value () = let ((Judge(val,_),_),_) = search_value() 
                    in val;;

(* Extracts the last pushed value in environment *)
let pop_value () = let (c,lists) = search_value() in  
in (ENV:=rev_append c lists);;

*)


(* B5| Discharging constants *)

let replace_const (name,val,inf) =
    VAL:=subst_norm name val !VAL;
    TYP:=subst_norm name val !TYP;
    INF:=inf_replace name inf !INF;
    ();;

(* Erase last constant *)
let pop_const () =
     match !ENV with
        [] -> error "No constant"
      | (Constdecl(Def(Name(str),Judge(val,_,_),inf),_,lt),_)::rest ->
                      replace_const(str,val,inf_value inf); ENV:=rest;
                      update_search (!UPD); UPD:=lt;()
      | _ -> error "Last item not a constant";;

  
(* B6| Lambda intro *)
(* used in [x:A]B no new tactic added after x has been put in the environment,
   !UPD should be [] *)
let abs_var () =
  match !ENV with
    [] -> error "No hypothesis in current context"
  | (Vardecl(Decl(name,Judge(typ,_,lev),inf),_,lt),_)::rest -> 
          ENV:=rest; pop_var();
          VAL:=Lambda(name,typ,subst_var name !VAL);
          TYP:=Prod(name,typ,subst_var name !TYP);
      INF:=inf_abs_var !LEV lev !INF inf;
          if !UPD=[] then UPD:=lt
          else anomaly "A new tactic should not be defined here";
          ()
  | _ -> error "Last item not a variable (abs)";;


(* B7| Prod intro and quantification *)

(* Cumulativity: slide type t up if product with second type imposes it *)
let adjust t = function
    Prop(_)    -> t
  | Type(_,u') -> (match t with 
         Type(c,u) -> let v = sup(u,u') in if eq(u,v) then t else Type(c,v)
       | _ -> anomaly "adjust")
  | _          -> anomaly "level_of_kind/adjust";;

let gen_var () = 
  if !LEV = Proof then error "Proof objects can only be abstracted";
  match !ENV with
    [] -> error "No hypothesis in current context"
  | (Vardecl(Decl(name,Judge(hyptyp,kind,lev),inf),_,lt),_)::rest -> 
      (match hnftype !TYP with 
         Prop(_) -> (* quantification : TYP stays Prop *)
             INF:=inf_generalize Proof lev !INF inf; ()
       | (Type(_,_) as typ) -> (* product *)
          INF:=inf_generalize Object lev !INF inf;
          TYP:=adjust typ kind (* predicativity *); ()
       | _ -> error "Should be typed by Prop or Type");
      ENV:=rest; pop_var();
      VAL:=Prod(name,hyptyp,subst_var name !VAL);
      if !UPD=[] then UPD:=lt
      else anomaly "A new tactic should not be defined here";
      ()
  | _ -> error "Last item not a variable (prod)";;


(* B8| Type casting *)

(* Assumes typ is a valid construction at level lev in current context *)
(* safe only when used by cast and verify below *)
(* should not be exported *)
let coerce_to (typ,_,lev) =
   if lev<>!LEV then error "Wrong level";
   let universes = read_uni() in
       if conv_leq !TYP typ then (TYP:=typ; ())
       else (reset_universes universes;
             PPNL
         [< 'S"Cannot coerce to intended type"; 'FNL;
            "Since the intended type :" msgpr typ; 
            "does not match the current type :" msgpr !TYP >];
          error "Illegal type conversion");;

(* Casting *)
(* Use: ... build type;; stack_cast();; construct term;; cast();;  *)

(* Extracts the last item in environment, which must be a cast *)
let unstack_cast () =
    match !ENV with
          []                  -> error "No cast in current context"
        | (Cast(j,_),_)::rest -> ENV:=rest;j
        | _                   -> error "Last item not a cast";;

(* assigns the intended type *)
let cast () = let (Judge(j1,j2,j3)) = unstack_cast() in coerce_to(j1,j2,j3);;

let search_cast () = search_rec (!ENV,[])
where rec search_rec (before,after) = match before with
    (Cast(j,_),_)::rest -> (j,rest,after)
  | item::rest          -> search_rec(rest,item::after)
  | []                  -> error "No cast in current environment";;

let search_before_cast () = search_rec (!ENV,[])
where rec search_rec (env,after) = match env with
    (Cast(j,_),_)::_ -> env,after
  | item::rest       -> search_rec(rest,item::after)
  | []               -> error "No cast in current environment";;

(* Extracts the last cast in environment *)
let pop_cast () =
    let (j,rest,after) = search_cast()
    in (ENV:=rev_append after rest; j);;

let reset_cast () = 
    let (_,rest,after) = search_cast()
    in ENV:=rev_append after rest; ();;

let reset_keeping_cast () = 
    let up_to_cast,after = search_before_cast()
    in ENV:=rev_append after up_to_cast; ();;

(* Proof checking *)
(* Use: ... build type;; stack_cast();; ... build proof ... verify();;  *)

(* assigns the intended type *)
let verify () = let (Judge(j1,j2,j3)) = pop_cast() in coerce_to(j1,j2,j3);;

(* C| Sections *)

(* C1| Section opening *)

let open_section name = push_env(Section(name)); incr SEC; ();;


(* C2| Name of current section *)

let current_section () = search_current_section !ENV
where rec search_current_section = function
  ((Section s),_)::rest -> s
| _::rest               -> search_current_section rest
| []                    -> error "No section opened";;


(* C3| Section closing *)

let scan_env id = scan_env_rec ([],0) !ENV 
where rec scan_env_rec (already,vars) = function
  (((Section(s),_)::rest) as env)  -> 
      if s=id then (ENV:=rest; repeat_action vars pop_var; already)
      else error("Section " ^s^ " ought to be closed first")
| (((Vardecl(_,_,_)) as item),_)::rest -> scan_env_rec (item::already,vars+1) rest
| (item,_)::rest -> scan_env_rec (item::already,vars) rest
| [] -> error "No section opened";;

(* substitute substitutes the item str2 by its value c2 in definition
   of item1 *)

let sub1 (Judge(c1,t1,lev1)) inf1 = function
  (Name(str2),_,c2,inf2) ->
   let subc2 c = if (occur_eq str2 c) then (subst_norm str2 c2 c) else c
   in let val1' = subc2 c1
      and typ1' = subc2 t1
      and inf1' = inf_replace str2 inf2 inf1
      in Judge(val1',typ1',lev1), inf1'
 | _ -> anomaly "sub1";;

let substitute item1 item2 = match item1 with
  Constdecl(Def(name,jud,inf),strength,lt1) ->
        let (jud',inf')  = sub1 jud (inf_value inf) item2
        in Constdecl(Def(name,jud',inf_const name inf'),strength,lt1)
| Cast(jud,inf) ->
        let (jud',inf')  = sub1 jud inf item2
        in Cast(jud',inf')
| Vardecl(Decl(name,jud,inf),strength,lt1) ->
        let (jud',inf') = sub1 jud (inf_vartype inf) item2
        in Vardecl(Decl(name,jud',inf_var name inf'),strength,lt1)
| _ -> item1;;   (* Values ? *)

(* functionalise takes a item and a variable. (and a list)
   if item is a constant and that var occurs in it, functionalise abstracts
   var in item; if item is a variable or a cast and var occurs in it,
   functionalise generalises var in item.
   If the item is modified it is added to the list.
   functionalise returns the item and the list. *)

let functionalise clever (item,modifiedlist)  
   (Decl(varname,Judge(vartyp,_,varlev),varinf) as var) =  match item with

 Constdecl((Def(name,Judge(val,typ,lev),inf) as def),n,lt) ->

if (not clever) or (occur_name varname val or occur_name varname typ)
then (* this constant depends on the variable *)
      let val' = Lambda(varname,vartyp,subst_var varname val)
      and typ' = Prod(varname,vartyp,subst_var varname typ)
      and inf',modifinf = inf_abs_apply lev varlev inf varinf
      in let c' = Def(name,Judge(val',typ',lev),inf')
      in let decl = Constdecl(c',n,lt) in
                     (* Replace local value of name by (name varname) *)
      let modif = (name,typ',App(Const(c'),Var var),modifinf)
      in decl,modif::modifiedlist
else  (item, modifiedlist)

| Cast(Judge(val,typ,lev),inf)  ->

   let val' = Prod (varname,vartyp,subst_var varname val)
   and inf' = inf_generalize lev varlev inf varinf
   in let decl = Cast(Judge(val',typ,lev),inf')
      in decl, modifiedlist

| Vardecl(Decl(name,Judge(val,typ,lev),inf),n,lt) ->

if (not clever) or (occur_name varname val or occur_name varname typ)
then (* this variable depends on the variable *)
    let val' = Prod (varname,vartyp,subst_var varname val)
    and inf',modifinf = inf_gen_apply lev varlev inf varinf
    in let c' = Decl(name,Judge(val',typ,lev),inf')
    in let decl = Vardecl(c',n,lt)
    in let modif =  (name,val',App(Var c',Var var) ,modifinf)
    in decl,modif::modifiedlist
else (item,modifiedlist)

| item -> item,modifiedlist;;

(* If item is not kept then clean1 eliminates it and prepares the future *)
(* modifications of other items; if item is kept then clean1 updates it  *)
(* (i.e. modifies it according to the former transformation), puts it    *)
(* back in the env and prepares the modification that have to be made to *)
(* the others item according to these transformations.                   *)
(* functlist is the list of eliminated variable before item (excluded),  *)
(* substlist is the list of substitution to be done,  and modifiedlist   *)
(* is the list of comments to be printed.                                *)
(* Returns the same four lists updated                                   *)

let modify_update item =
    let newupd = !UPD in
    match item with (Vardecl(v,n,upd)) -> if upd=newupd then item
                                          else Vardecl(v,n,newupd)
                  | (Constdecl(d,n,upd))-> if upd=newupd then item
                                          else Constdecl(d,n,newupd)
                  | _                  -> item;;

let update_kept_item item functlist substlist modifiedlist clever =
   let item0 = modify_update item in
   let item' = it_list substitute item0 substlist
in let (item'',justmodifiedlist) =
                 it_list (functionalise clever) (item',[]) functlist 
in let substlist' = rev_append justmodifiedlist substlist
   and modifiedlist' = rev_append justmodifiedlist modifiedlist
in (item'',functlist,substlist',modifiedlist');;

let update1 clever (functlist,substlist,modifiedlist) item =
match item with
  (Constdecl(Def(((Name str) as name),Judge(val,typ,_),inf),Strength p,_)) ->
    if p>=!SEC then (* item is a non kept constant, we add it to substlist *)
       functlist,((name,typ,val,inf_value inf)::substlist),modifiedlist
    else (* item is a kept constant *)
    let (item',list1',list2',list3') =
                update_kept_item item functlist substlist modifiedlist clever
    in push_env item'; UPD:=[];(list1',list2',list3')

| (Vardecl(_,Strength (p),_)) ->
    if p>=!SEC then (* item is a variable not kept *)
        match (it_list substitute item substlist) with
          Vardecl(hyp,_,_) -> (hyp::functlist,substlist,modifiedlist)
        | _ -> anomaly "update1"
    else (* item is a kept variable *)
        let (item',list1',list2',list3') =
            update_kept_item item functlist substlist modifiedlist clever
        in push_env item'; UPD:=[]; VAR := item'::!VAR; list1',list2',list3'

| _ -> let (item',list1',list2',list3') =
            update_kept_item item functlist substlist modifiedlist clever
       in push_env item'; list1',list2',list3';;

(* update does the same thing as update1, but for
   all the elements of litems *)

let update litems clever = it_list (update1 clever) ([],[],[]) litems;;

(* modified extracts the name of functionalised items from the value returned
   by eliminate_variable *)

let modified (_,_,modifiedlist) = it_list sift [] modifiedlist
    where sift decls = function
          (Name(name),typ,_,_) -> (name,typ)::except_assoc name decls
        | _ -> anomaly "Modified";;

(* display_modified takes the value returned by update variable and writes
   warnings for the user. *)

let display_modified l = do_list display_modified_one (modified l)
      where display_modified_one (s,typ) = PP [< print_id s ; 'S " : ";'FNL ; pr typ >];;

(* we now combine everything into a close_section function *)
(* eliminates all variables and constants local to the section *)

let close_section ident clever =
    let debenv = scan_env ident in
    update_tac_decl debenv;
    display_modified (update debenv clever);
    decr SEC;
    reset_current();;

(* closes the section while saving VAL temporarily as a named constant ident *)
let close_section' ident clever  =
    declare (id_of_string ident) 0; 
    close_section ident clever;
    match !ENV with 
       (Constdecl(Def(_,Judge(val,typ,lev),inf),_,lt),_)::rest ->
            ENV:=rest; VAL:=val; TYP:=typ; LEV:=lev; INF:=inf_value inf;
            UPD:=lt; ()
     | _ -> anomaly "Last item is not a constant";;

(* D| Miscellaneous commands *)

(* D1 | Resetting *)

let restore_universes () =
  let olduni = match !ENV with
       [] -> initial_universes
     | (_,uni)::_ -> uni
  in reset_universes(olduni);;

(* keep is a boolean flag *)
let reset_glob keep str = (reset_rec !SEC !VAR [!UPD] !ENV;
                          restore_universes ())
where rec reset_rec sec var ltac = function
   [] -> error ((string_of_id str) ^ " not declared")
 | (((Vardecl(Decl(Name(str'),_,_),_,lt),_)::rest) as env)   -> 
       if str=str' then 
         (if keep then (ENV:=env; VAR:=var; UPD:=update_list_search ltac)
          else (ENV:=rest; VAR:=tl var; UPD:=update_list_search (lt::ltac));
                SEC:=sec)
       else reset_rec sec (tl var) (lt::ltac) rest 
 | (Vardecl(Decl(Anonymous,_,_),_,lt),_)::rest               -> 
       reset_rec sec (tl var) (lt::ltac) rest 
 | (((Constdecl(Def(Name (str'),_,_),_,lt),_)::rest) as env) ->
       if str=str' then
           (if keep then (ENV:=env; UPD:=update_list_search ltac)
            else (ENV:=rest; UPD:=update_list_search (lt::ltac));
            VAR:=var;SEC:=sec)
       else reset_rec sec var (lt::ltac) rest 
 | (Constdecl (Def(Anonymous,_,_),_,lt),_)::rest             -> 
       reset_rec sec var (lt::ltac) rest 
 | (Cast(_,_),_)::rest                                    -> 
       reset_rec sec var ltac rest  
 | (Loaded _,_)::rest                                    -> 
       reset_rec sec var ltac rest  
 | (Value (_,_),_)::rest                                  -> 
       reset_rec sec var ltac rest 
 | (Scope (_),_)::rest                                  ->
       reset_rec sec var ltac rest
 | (Section (_),_)::rest                                ->
       reset_rec (sec-1) var ltac rest;;

let raw_reset_name = reset_glob false
and raw_reset_keeping_name = reset_glob true;;

let raw_reset_section str = (reset_rec !SEC !VAR [!UPD] !ENV;
                            restore_universes ())
where rec reset_rec sec var ltac = function
   [] -> error (str ^ ": section not declared")
 | (Section(str'),_)::rest  ->  
      if str=str' then (ENV:=rest; VAR:=var; UPD:=update_list_search ltac;
                        SEC:=sec-1)
      else reset_rec (sec-1) var ltac rest
 | (Vardecl(_,_,lt),_)::rest   -> reset_rec sec (tl var) (lt::ltac) rest
 | (Constdecl(_,_,lt),_)::rest -> reset_rec sec var (lt::ltac) rest
 | (Cast(_,_),_)::rest      -> reset_rec sec var ltac rest
 | (Loaded _,_)::rest      -> reset_rec sec var ltac rest
 | (Value (_,_),_)::rest    -> reset_rec sec var ltac rest
 | (Scope (_),_)::rest    -> reset_rec sec var ltac rest;;

(* D2| State saving/restoring *)

let raw_save_state name desc = 
    if listmap__in_dom !STATES name then error(name ^ " already exists")
    else STATES:=
        (listmap__add (!STATES)
                     (name,(desc,(!ENV,!VAR,!SEC,!VAL,!TYP,!LEV,read_uni(),!INF,
                                  save_search(),!UPD,read_syntax()))));
    ();;

let reset_state (env,var,sec,val,typ,lev,uni,inf,sea,upd,syn) =
    ENV:= env; VAR:= var; SEC:= sec; VAL:= val; TYP:= typ; LEV:= lev; 
    reset_universes uni;
    INF:= inf; restore_search sea; UPD:=upd; reset_syntax syn; ();;

let raw_restore_state name = 
    try reset_state(snd(listmap__map (!STATES) name))
    with Not_found -> error(name ^ ": unknown state");;

let forget_state verbose name =
    if name = "Prelude" then error "Cannot forget Prelude" 
    else if name = "Initial" then error "Cannot forget Initial" 
    else if listmap__in_dom (!STATES) name then
        (STATES := listmap__rmv (!STATES) name;
         if verbose then warning(name ^ " forgotten"))
    else warning ("There is no state named " ^ name)
;;

let list_saved_states () =
    map (fun (n,(desc,_)) -> (n,desc)) (!STATES);;

let save_light () = 
     SAVE:=(!ENV,!VAR,!SEC,!VAL,!TYP,!LEV,read_uni(),!INF)
                 ::!SAVE;();;

let restore_light () = 
 match !SAVE with 
    [] -> anomaly "restore_light"
 | (env,var,sec,val,typ,lev,uni,inf)::tl ->
    ENV:= env; VAR:= var; SEC:= sec; VAL:= val; TYP:= typ; LEV:= lev; 
    reset_universes uni; INF:= inf;SAVE:=tl; ();;

let forget_light () = SAVE:=(tl !SAVE);();;

(* Total resetting *)
let raw_reset_all () = reset_current(); reset_env(); SEC:=0; reset_search ();
    reset_universes(initial_universes); ();;

let MAGIC = 19755;;

(* create persistent state *)
let extern_state name = 
  try
  let channel = open_trapping_failure open_out name ".coq" in
   output_binary_int channel MAGIC;
   output_value channel (!STATES);
      close_out channel
  with sys__Sys_error s -> error("System error: " ^ s)
;;

(* load persistent state *)
let intern_state name = 
  try
  let channel = open_with_suffix_from_path open_in (!LOADPATH) name ".coq" in
  begin try
    if input_binary_int channel <> MAGIC then error "Bad magic number";
    match input_value(channel) with
     (saved_states : (string * (string * state)) list) ->
     let repeats = listset__intersect (listmap__dom saved_states) (listmap__dom (!STATES)) in
         (if not(repeats=[]) then
              do_list (fun n -> warning ("Removing repeated state " ^ n)) repeats;
          STATES := ((!STATES) @ (it_list listmap__rmv saved_states repeats)))
  with sys__Sys_error s -> error("System error: " ^ s)
  end;
  close_in channel
  with sys__Sys_error s -> error("System error: " ^ s)
;;

let chop_context name = iterec []
  where rec
  iterec deb = function 
      ([]:context) -> deb,[]
  |  ((Vardecl(Decl(Name(s),_,_),_,_),_ as c)::l as m) -> if name=s then deb,m
                                                      else iterec (c::deb) l
  |  ((Constdecl(Def(Name(s),_,_),_,_),_ as c)::l as m) -> if name=s then deb,m
                                                      else iterec (c::deb) l
  |  c :: l -> iterec (c::deb) l;;

   
(* E| Declaration of an inductive type *)


(* Building the inductive type, 
   its constructors and the rules of elimination *)

let indtype stamp typ inftyp lconstr linfconstr = 
    VAL:=make_ind stamp typ lconstr;TYP:=typ;LEV:=Object;
    INF:=inf_indtype stamp inftyp linfconstr; ();;

let unstack_n_var = unstack []
where rec unstack l =
     function 0 -> l 
            | n -> match unstack_var() with
        Vardecl(Decl(name,Judge(typ,_,_),inftyp),_,_) ->
                unstack ((name,typ,inftyp)::l) (n-1)
      | _ -> anomaly "unstack_var";;

let inductype_spec stamp n =
      match unstack_n_var (n+1) with
        [] -> anomaly "unstack_n_var"
      | (rname,typ,inftyp)::lci ->
    let lname,lconstr,linfconstr = decomp lci
          where rec decomp = function 
         [] -> [],[],[]
       | (name,c,ic)::lrest -> let ln,lc,lic = decomp lrest
                               in name::ln,(subst_var rname c)::lc,
                               (abstract_inf rname (inf_vartype ic))::lic
    in indtype stamp typ (inf_vartype inftyp) lconstr linfconstr;
    rname,lname,lconstr;;

let inductype stamp n = inductype_spec stamp n; ();;
   
let indconstr i ind infind typ =
    let newtyp = subst1 ind typ in 
    VAL:=Construct(i,ind);TYP:=newtyp;LEV:=lev_of_ind ind;
    INF:=inf_indconstr infind i; ();;

let constructor i =
    let ind = !VAL in 
    VAL:=Construct(i,ind);TYP:=type_construct i ind;
    LEV:=lev_of_ind ind;INF:=inf_indconstr !INF i; ();;

let contents_of_kind = function 
    Prop(n)   -> n 
  | Type(n,_) -> n
  | _         -> error "Not a proposition or a type";;


(* F| Execution of a construction *)

let gen_rel var = 
  if !LEV=Proof then error "Proof objects can only be abstracted" 
  else let (name,typ,kind,lev,inf) = var
       in (match hnftype !TYP with 
                 Prop(_) -> (* quantification : TYP stays Prop *)
                  INF:=inf_gen_rel Proof lev !INF inf; ()
               | (Type(_,_) as typ) -> (* product *)
                  INF:=inf_gen_rel Object lev !INF inf;
                  TYP:=adjust typ kind (* predicativity *); ()
               | _ -> error "Should be typed by Prop or Type");
          VAL:=Prod(name,typ,!VAL);;

let abs_rel var =
 let (name,typ,_,lev,inf) = var in
         INF:=inf_abs_rel !LEV lev !INF inf;
         VAL:=Lambda(name,typ,!VAL); 
         TYP:=Prod(name,typ,!TYP);;

let (nth_var,add_var) =  
   let VARIABLES = make_vect 400
      (Anonymous,Prop(Null),Type(Null,prop_univ),Object,logic ())
   in ((fun n -> VARIABLES.(n)),(fun v top -> VARIABLES.(top) <- v));;
 
let relative n nbvariables = 
    if n>nbvariables then  error "Non closed term"
    else let (_,typ,_,lev,inf) = nth_var(nbvariables-n)
         in VAL:=Rel(n);
             (* We cannot assume, as in the normal machine operation, that the 
            types of variables are closed terms, and thus we must lift them *)
            TYP:=lift n typ;
            LEV:=lev;
            INF:=inf_rel n inf; ();;

let apply_rel (val,typ,lev,inf) =
  match hnftype !TYP with
     Prod(_,c1,c2) -> let universes = read_uni()
                      in if conv_leq typ c1 then (
            VAL:=App(!VAL,val);
            TYP:=subst1 val c2;
        INF:=inf_apply !LEV lev !INF inf; ())
         else (reset_universes universes;
               PPNL
                [< "Illegal application :" msgpr !VAL; 
                   "cannot be applied to :" msgpr val; 
                   "Since the formal type :" msgpr c1; 
                  "does not match the actual type :" msgpr typ >];
                error "Application would violate typings")
   | _ -> error "Non-functional construction";;

(* Execution of a construction *)
(* After successful execution of (execute c), we get !VAL=c *)
(* Used by Save tactic *)

let rec level_of_arity c =
     match hnftype c with Prop(_)      -> Proof
                        | Type(_,_)      -> Object
                        | Prod(_,_,c2) -> level_of_arity c2
                        | _            -> anomaly "hnftype";;

let execute constr = 
    (save_light ();
     try execute_rec 0 constr; forget_light ()
     with reraise -> restore_light(); raise(reraise))
where rec execute_rec nbvariables =  function
  Meta(_)                -> error "Incomplete construction cannot be executed"
| Rel(n)                 -> relative n nbvariables
| (Var(Decl((Name(str) as name),Judge(typ,_,lev),inf)) as val) -> 
            (VAL:=val;
             TYP:=typ;
             LEV:=lev;
             INF:=inf;
             ())
| (Const(Def((Name(str) as name),Judge(_,typ,lev),inf)) as val) -> 
            (VAL:=val;
             TYP:=typ;
             LEV:=lev;
             INF:=inf;
             ())
| Prop(c)                 -> proposition c; ()
| Type(c,u)               -> type_with_universe (c,u); ()
| App(c1,c2)              -> 
            (execute_rec nbvariables c2;
             let (val,typ,lev,inf) = (!VAL,!TYP,!LEV,!INF)
             in execute_rec nbvariables c1;apply_rel (val,typ,lev,inf))
| Lambda(name,c1,c2)      -> 
            (execute_rec nbvariables c1; 
             let var = let typ = hnftype !TYP 
                       in let lev = level_of_kind typ
                          in (name,!VAL,typ,lev,inf_var name !INF)
             in (execute_rec (add_var var nbvariables; nbvariables + 1) c2; 
                 abs_rel var; ()))
| Prod(name,c1,c2)        ->
            (execute_rec nbvariables c1; 
             let var = let typ = hnftype !TYP 
                       in let lev = level_of_kind typ
                          in (name,!VAL,typ,lev,inf_var name !INF )
             in (execute_rec (add_var var nbvariables; nbvariables + 1) c2;
                 gen_rel var; ()))
| Ind(stamp,c,lc,_,_)       -> (execute_rec nbvariables c;
                          let inftyp = !INF 
                          and (_,typ,_,_,_ as var) =
                                (Anonymous,!VAL,!TYP,!LEV,
                                  inf_var Anonymous !INF )
                          in add_var var nbvariables;
               let linfc = map (fun c -> execute_rec (nbvariables + 1) c;!INF)
                           lc
               in indtype stamp typ inftyp lc linfc; ())
| Construct(i,c)          -> (execute_rec nbvariables c;
                              constructor i; ())
| (Rec(b,lpf,c) as val) -> (let ct,cinf = execute_rec nbvariables c;!TYP,!INF
  in let lpft,linf = split (map (fun c -> execute_rec nbvariables c;!TYP,!INF)
                            lpf) in
  VAL:=val; let P = (try hd lpf with Failure _ -> error "Elimination")
            in TYP:=type_elim ct lpft b P c;
  LEV:= level_of_arity (hd lpft);INF:=inf_inst_elim linf cinf; ())
| _ -> error "Cannot be executed";;

(* Vardecl creation for the introduction tactics *)
let create_var name c stre = execute c; mk_vardecl name stre;;

let creation_var name c = 
 match create_var name c 0 with
       Vardecl(x,_,_) -> Var x 
     | _ -> anomaly "create_var";;

(* used in operations *)

let val_of name = match search(name) with
   Vardecl(v,_,_)   -> Var(v)
 | Constdecl(v,_,_) -> Const(v)
 | _              -> error "Not a variable or constant";;


(*******************************************************************)
(* Building elimination combinators                                *)
(*******************************************************************)

let indelim_dep ind kind lconstr =
    execute (make_elim_dep ind kind lconstr);;

let indelim_nodep ind kind  =
    execute (make_elim_nodep ind kind);;

let dep_elimination kind lconstr =
    let elim = make_elim_dep (!VAL) kind lconstr
    in execute elim;;

let nodep_elimination kind  =
    let elim = make_elim_nodep (!VAL) kind
    in execute elim;;

let elimination () = let
    (Judge(P,typP,_),infP) = unstack_value() in
    let (e,typ),lev = make_elim !VAL !TYP P typP
    in VAL:=e;TYP:=typ;LEV:=lev;
    INF:=inf_make_elimination !INF infP; ();;

(* Construction of the inductive type *)

let recursive stamp elim nbc (Strength n) =
  match inductype_spec stamp nbc with
    (Name(indid),lname,ltypconstr) ->
      let indstr = string_of_id indid in
      let ind = declare_decl indid n in
      let infind = inf_make_const indid (info_of ind) in
      let lconstr = build 1 (lname,ltypconstr)
          where rec build i = function
              ((Name str)::ln,ty::lt)-> indconstr i ind infind ty;
                                  let d = declare_decl str n in
                                  d::build (i+1) (ln,lt)
             | [],[]           -> []
             | _ -> anomaly "recursive"
      in if elim then
      let (kd,kn) = possible_elim(ind) in
      if mem prop kd then
        (indelim_dep ind prop lconstr; declare (id_of_string(indstr^"_ind")) n)
      else if mem prop kn then
        (indelim_nodep ind prop; declare (id_of_string(indstr^"_ind")) n);
      if mem spec kd then
        (indelim_dep ind spec lconstr; declare (id_of_string(indstr^"_rec")) n)
      else if mem spec kn then
        (indelim_nodep ind spec; declare (id_of_string(indstr^"_rec")) n);
      if mem data kn then
        (indelim_nodep ind data; declare (id_of_string(indstr^"_recd")) n);
      if mem typep kd then
        (indelim_dep ind (Type(Null,New_univ())) lconstr; 
                 declare (id_of_string(indstr^"_rect")) n)
      else if mem typep kn then
         (indelim_nodep ind (Type(Null,New_univ())); 
                 declare (id_of_string(indstr^"_rect")) n);
      if mem types kd then
         (indelim_dep ind (Type(Pos,New_univ())) lconstr; 
                 declare (id_of_string(indstr^"_recs")) n)
      else if mem types kn then
         (indelim_nodep ind (Type(Pos,New_univ())); 
                 declare (id_of_string(indstr^"_recs")) n)
  | _ -> anomaly "recursive"
;;

(* Current Package Loading/Search *)

let search_packages () = search_rec (!ENV)
where rec search_rec = function
(Loaded l,_)::_ -> l
  | _::rest -> search_rec rest
  | [] -> []
;;

let declare_package n =
    let pl = search_packages () in
        push_env (Loaded (n::pl))
;;

let add_path dir =
    if listset__memb (!LOADPATH) dir then
        warning (dir ^ " already on loadpath")
    else LOADPATH := listset__add (!LOADPATH) dir
;;

let del_path dir =
    if not(listset__memb (!LOADPATH) dir) then
        error (dir ^ " not on loadpath")
    else LOADPATH := listset__rmv (!LOADPATH) dir
;;

let search_paths () = !LOADPATH
;;
