(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                               lazy.ml                                    *)
(****************************************************************************)
(*                                                                          *)
(*                        Translators from FML to ML                        *)
(*                                                                          *)
(****************************************************************************)

#open "std";;
#open "fmlenv";;
#open "fmlterm";;
#open "pp";;
#open "stdpp";;
#open "more_util";;

let CurLang = ref LML;;

let constrsep() = match !CurLang with 
          LML -> "+"
        | GAML -> "|";;

let constrid() = match !CurLang with
          LML -> "*"
        | GAML -> "'";;
let typdef() = match !CurLang with
          LML -> "let type "
        | GAML -> "type ";;

let lazyTypVar s = match !CurLang with
              LML -> "*"^s
            | GAML -> "'"^s ;;
let lazyLamb1 s = match !CurLang with
          LML -> "\\"^s^"."
        | GAML -> "fun "^s^" ";;

let lazyLamb2 s = match !CurLang with
         LML -> "\\"^s^"."
       | GAML -> s^" ";;

let lazyLamb3 () = match !CurLang with
          LML -> ""
        | GAML -> "->";;

let LML_name = ref "extract";;

let rem_quote s =
 let s = ref s in
 let n = ref ((string_length !s)-1) in
 let m = ref 0 in
 let rec aux () =
 (if (nth_char !s !n)=`'` 
    then (set_nth_char !s !n `_`;m:=!m+1)
    else ();
  if !n=0 then () else (n:=!n-1;aux()))
 in 
  (aux();
  if !m=0 then !s 
         else !s^"_q"^(string_of_int !m));;


let LMLreserved = ref ["cons";"nil";"true";"false";
                       "and";"Max";"Min";"Or";"Sum";"Times";"abs";"all";
                      "difference";"exists";"for";"from";"fst";"hd";"head";
                      "last";"odd";"sort";"split";"tl";"tail"];;

let lmlsafe s = match !CurLang with
              LML -> if (mem s !LMLreserved) then (s^"_ren") else s
            | GAML -> let s = (rem_quote s) in
                       if (mem s !LMLreserved) then (s^"_ren") else s;;

exception except_in_lazy;;
exception unEXT;;
exception unterm;;

let rec atomised = function
          (Fmlambda(s,t)) -> atomised t
        | t -> t;;


let rec LML_term = function
    t -> HOV 0 [< LML_infix t >]

and LML_infix = function
    (Fmapp (Fmapp(Fmext E,t1),t2)) as t->
    if (mem E (map fst !LMLextinfix))
        then
            HOV 0 [< 'S "("; LML_term t1;
                   'S (assoc E !LMLextinfix);
                   LML_term t2; 
                   'S ")" >]
    else
        HOV 0 [< LML_term1 t >]

  | (Fmapp(Fmapp(Fmapp(Fmext If,b),t1),t2))  ->
    HOV 0 [< 'S "if";'SPC; LML_term b;'SPC; 'S "then";'SPC;
           LML_term t1;'SPC; 'S "else";'SPC; LML_term t2 >]
  | t -> HOV 0 [< LML_term1 t >]

and LML_term1 = function
    (Fmapp (t1,t2)) ->
    HOV 0 [< 'S"(";
           HOV 1 [< LML_app t1; 'SPC; 
                  LML_term t2 >];
           'CUT; 'S")" >]

  | (Fmlambda (s,t)) ->
    HOV 0 [< 'S"("; HOV 0 [< 'S(lazyLamb1 (lmlsafe s));
                                 'BRK(0,1);
                                 LML_lambda t >];
           'BRK(0,2);
           HOV 0 [< LML_term (atomised t) >];
           'S")" >]

  | (Fmvar s) -> HOV 0 [< 'S (lmlsafe s) >]

  | (Fmconst s) -> HOV 0 [< 'S (lmlsafe s) >]

  | (Fmlocal(s,(FmRec t1),t2)) ->
    HOV 0 [< 'S"(";
           HOV 1 [< 'S "let rec "; 'S(lmlsafe s); 
                  'S " = " >]; 'BRK(0,1); HOV 0 [< LML_term t1 >];
           'BRK(1,2); 'S "in";
           'BRK(1,2); HOV 1 [< LML_term t2 >] ;'S")" >]

  | (Fmlocal(s,t1,t2)) ->
    HOV 0 [< 'S "("; HOV 0 [< 'S "let "; 'S(lmlsafe s);
                                  'S " = " >];
           'BRK(0,1); HOV 0 [< LML_term t1 >];
           'BRK(1,2); 'S "in";
           'BRK(1,2); HOV 0 [< LML_term t2 >]; 'S")" >]

  | (Fmext E) -> lazy_ext E

  | (Fmconstruct (n,s,[])) ->
    HOV 0 [< 'S((fst(nth(assoc s !Fmindenv)n))^"_C") >]

  | (Fmconstruct (n,s,tl)) ->
   HOV 0 [< 'S "(";'S ((fst(nth(assoc s !Fmindenv)n))^"_C");
          prlist LML_term_sp tl; 'S")" >]

  | Fmerror -> HOV 0  [< 'S "(fail \"Exception\") " >]

  | (Fmmatch(t,s,pl)) -> HOV 0 [< Lazy_match (t,s,pl) >]

  | _ -> raise unterm

and Lazy_match = function
    (t,s,pl) ->
    (match !CurLang with
     LML->LML_match(t,s,pl)
   | GAML-> Gaml_match(t,s,pl))

and LML_match = function
    (t,s,pl) ->
    HOV 0
    [< 'S"(";
     HOV 0
     [< 'S"case"; 'SPC; LML_term t; 'SPC; 'S"in";'SPC;
         HOV 0 [< LML_pat_list (s,pl) >];
         'S"end" >];
     'SPC; 'S")" >]

and Gaml_match = function
    (t,s,pl) ->
    HOV 0
    [< 'S"(";
     HOV 0
     [< 'S"match"; 'SPC; LML_term t; 'SPC; 'S"with"; 'SPC;
      Gaml_pat_list (s,pl) >];
     'SPC; 'S")" >]

and Gaml_pat_list = function
    (s,[]) -> HOV 0 [< 'S"" >]
  | (s,(p::pl)) -> HOV 0 [< HOV 0 [<'S " ";'S " ";Gaml_pat (s,p);
                                   'FNL >];
                          Gaml_pat_list1 (s,pl) >]

and Gaml_pat_list1 = function
    (s,[]) -> HOV 0 [< 'S "" >]
  | (s,(p::pl)) -> HOV 0 [< HOV 0 [< 'S"| ";Gaml_pat (s,p); 'FNL >];
                          Gaml_pat_list1 (s,pl) >]

and Gaml_pat = function
    (s,(n,[],t)) ->
    HOV 0 [< HOV 0 [< 'S((fst(nth(assoc s !Fmindenv) n))^"_C");
                    'S"->" >];
                     'SPC;
                     HOV 0 [< LML_term t >] >]

  | (s,(n,vl,t)) ->
    HOV 0 [< HOV 0 [< 'S"(";
                    'S((fst(nth(assoc s !Fmindenv) n))^"_C");
                    LML_varlist vl; 'S")-> " >]; 'SPC;
           HOV 0 [< LML_term t >] >]

and LML_pat_list = function
    (s,[]) -> HOV 0 [< 'S"" >]
  | (s,(p::pl)) ->
    HOV 0 [< HOV 0 [< 'S " ";'S " ";LML_pat (s,p);
                  'FNL >];
           LML_pat_list1 (s,pl) >]

and LML_pat_list1 = function
    (s,[]) -> HOV 0 [< 'S"" >]
  | (s,(p::pl)) -> HOV 0 [< HOV 0 [< 'S"||";LML_pat (s,p); 'FNL >];
                          LML_pat_list1 (s,pl) >]

and LML_pat = function
    (s,(n,[],t)) ->
    HOV 0 [< HOV 0 [< 'S((fst(nth(assoc s !Fmindenv) n))^"_C"); 
                    'S" : " >] ;
           'SPC;
           HOV 0 [< LML_term t >] >]

  | (s,(n,vl,t)) ->
    HOV 0 [< HOV 0 [< 'S"(";
                    'S((fst(nth(assoc s !Fmindenv) n))^"_C"); 
                    LML_varlist vl; 'S") : " >];
           'SPC;
           HOV 0 [< LML_term t >] >]

and LML_varlist = function
    [] -> HOV 0 [< 'S"" >]
  | (s::l) ->
    HOV 0 [< 'S " "; 'S(lmlsafe s); LML_varlist l; 'CUT >]

and LML_lambda = function
    (Fmlambda(s,t)) -> HOV 0  [< 'S (lazyLamb2 (lmlsafe s));
                               'BRK(0,1); LML_lambda t >]
  | t -> HOV 0 [< 'S(lazyLamb3()) >]

and LML_app = function 
    (Fmapp(t1,t2)) -> HOV 2 [< LML_app t1; 'SPC ; LML_term t2 >]
  | t -> LML_term t

and lazy_ext = function
    E -> HOV 0 [< (match !CurLang with
                   LML -> LML_ext E
                 | GAML -> GAML_ext E) >]

and GAML_ext = function
    (Num n) -> HOV 0 [< 'S(string_of_int n) >]
  | Plus -> HOV 0 [< 'S "fun x y->x+y" >]
  | Minus -> HOV 0 [< 'S "fun x y->x-y" >]
  | Times -> HOV 0 [< 'S "fun x y->x*y" >]
  | If -> HOV 0 [< 'S "fun b t1 t2->(if b then t1 else t2)" >]
  | Lesser -> HOV 0 [< 'S "fun x y->x<y" >]
  | Equal -> HOV 0 [< 'S "fun x y->x=y" >]
  | True -> HOV 0 [< 'S "true" >]
  | False -> HOV 0 [< 'S "false" >]
  | _ -> raise unEXT

and LML_ext = function
    (Num n) -> HOV 0 [< 'S(string_of_int n) >]
  | Plus -> HOV 0 [< 'S "(fun x y ->(x+y))" >]
  | Minus -> HOV 0 [< 'S "(fun x y ->(x-y))" >]
  | Times -> HOV 0 [< 'S "(fun x y ->(x*y))" >]
  | If -> HOV 0 [< 'S "(fun b t1 t2 -> (if b then t1 else t2))" >]
  | Lesser -> HOV 0 [< 'S "(fun x y ->(x<y))" >]
  | Equal -> HOV 0 [< 'S "(fun x y ->(x=y))" >]
  | True -> HOV 0 [< 'S "true" >]
  | False -> HOV 0 [< 'S "false" >]
  | _ -> raise unEXT

and LML_term_sp = function 
    t -> HOV 0 [< 'S" "; LML_term t >];;

let rec is_indt = function
      (FmInd(_,_))->true
    | (FmLambda(s,t))-> is_indt t
    |_ -> false
;;

let exp_typ = ref true;;



let rec LML_type = function
    (FmConst s ) ->
    [< 'S(if not(mem_assoc s !Fmtypenv) then
                    (if (!CurLang=GAML & s="Int") then "int"
                     else s)
                else s) >]

  | (FmVar s) -> HOV 0 [< 'S((constrid())^s) >]

  | (FmArr(t1,t2)) ->
    HOV 0  [< 'S"(";
            HOV 1 [< LML_type t1; 'S "->"; 'CUT;
                   LML_arrow t2 >];
            'S")" >]

  | (FmApp(t1,t2)) -> HOV 0  [< 'S"("; HOV 1 [< LML_App t1; 'BRK(1,3);
                                                    LML_type t2 >];
                              'CUT; 'S")" >]

  | (FmLambda(s,t)) -> HOV 0 [< LML_type t >]

  | (FmInd(s,cl)) -> HOV 0 [< LML_constr cl >]

and LML_arrow = function
    (FmArr(t1,t2)) ->
    HOV 0 [< LML_type t1; 'S"->"; 'CUT; LML_arrow t2 >]
  | t -> HOV 0 [< LML_type t >]

and LML_App = function
    (FmApp(t1,t2)) -> HOV 0 [< LML_App t1; 'BRK(1,3); LML_type t2 >]
  | t -> HOV 0 [< LML_type t >]

and LML_constr = function
    [] -> HOV 0 [< 'CUT >]
  | ((s,tl)::l) -> HOV 1 [< 'S(s^"_C"); 'BRK(1,3); prlist LML_type_sp tl;
                          'S(if l<>[] then constrsep() else "");
                          'BRK(1,0);
                          LML_constr l >]

and LML_type_sp = function
    t -> HOV 0 [< LML_type t; 'S" " >];;


let rec LML_term_def = function
    (s,(FmRec t)) ->
    HOV 0
    [< HOV 0
     [< 'S"let rec "; 'S(lmlsafe s); LML_term_param t;
      'S" =" >];
     'BRK(1,1); HOV 0 [< LML_term_body t >]; 'FNL >]

  | (s,t) ->
    HOV 0
    [< HOV 0 [< 'S"let "; 'S(lmlsafe s); LML_term_param t;
              'S" =" >];
     'BRK(1,1); HOV 0 [< LML_term_body t >]; 'FNL >]

and LML_term_param = function 
    (Fmlambda(s,t)) ->
    HOV 0 [< 'S" "; 'S(lmlsafe s); LML_term_param t >]
  | _ -> HOV 0  [< 'S"" >]

and LML_term_body = function
    (Fmlambda(s,t)) -> LML_term_body t
  | t -> HOV 0  [< LML_term t >];;


let rec getvar = function
      (FmLambda(s,t)) -> add_set s (getvar t)
    | (FmApp(t1,t2)) -> union(getvar t1)(getvar t2)
    | (FmArr(t1,t2)) -> union(getvar t1)(getvar t2)
    | (FmVar s) -> [s]
    | (FmConst s) -> []
    | (FmError _) -> []
    | (FmInd(s,cl)) -> list_it (fun t->union
                                        (list_it (fun g ->union
                                                            (getvar g))
                                                  t
                                                  []))
                                (map snd cl)
                                [];;


let rec LML_type_def = function
    (s,t) ->
    HOV 0 [< 'S(typdef()); 'ID s; 'S " ";
           HOV 0 [< LML_typ_var (getvar t) >];
           'CUT;
           'S(if (is_indt t) then "=" else "=="); 'CUT;
           LML_type t >]

and LML_typ_var = function
    [] -> HOV 0  [< 'S"" >]
  | (s::l) -> HOV 0 [< 'S((constrid())^s); 'SPC; LML_typ_var l >];;
