(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                            interface.ml                                  *)
(****************************************************************************)

#open "std";;
(* rt *)
#open "rt";;
(* coq *)
#open "initial";;
#open "term";;
#open "tactics";;
#open "printer";;
#open "pp_control";;
#open "pp";;
#open "stdpp";;

(**************************************************************************)
(* Generation of goal number from int list (first compenent of type goal) *)
(**************************************************************************)

let rec make_name path = match path with
    [] -> "1"
  | ((-1)::tl) -> make_name tl
  | (n::tl) -> ((make_name tl) ^ "." ^ (string_of_int n));;

(**********)
(* Colors *)
(**********)

let COLOR = ref true;;

let GRAYSCALE = ref false;;

let make_color r g b xd =
 ColorBg(try rt_create_color(xd,r,g,b) 
         with Failure "rt_create_color" -> rt_white_color xd);;

let make_white xd = ColorBg(rt_white_color xd);;

let make_ltblue xd = 
    if !GRAYSCALE then make_color 135 206 255 xd
   else if !COLOR then make_color 135 206 255 xd
                  else make_white xd;;

let make_ltyellow xd = 
    if !GRAYSCALE then make_color 255 255 135 xd
   else if !COLOR then make_color 255 255 135 xd
                  else make_white xd;;

let make_orangered xd =
    if !GRAYSCALE then make_color 255 200 200 xd
   else if !COLOR then make_color 255 80 80 xd
                  else make_white xd;;

(* Change rt_create_color into make_color as above if using below
let make_purple xd = ColorBg(if !COLOR then
                       rt_create_color(xd, 150, 0, 255) else
                       rt_white_color xd);;

let make_ltpurple xd = ColorBg(if !COLOR then
                    rt_create_color(xd, 150, 110, 255) else
                    rt_white_color xd);;

let make_salmon xd = ColorBg(if !COLOR then
                    rt_create_color(xd, 255, 128, 122) else
                    rt_white_color xd);;

let make_green xd = ColorBg(if !COLOR then
                    rt_create_color(xd, 34, 139, 24) else
                    rt_white_color xd);;

let make_btgreen xd = ColorBg(if !COLOR then
                    rt_create_color(xd, 60, 255, 60) else
                    rt_white_color xd);;

let make_ltgreen xd = ColorBg(if !COLOR then
                    rt_create_color(xd, 80, 255, 80) else
                    rt_white_color xd);;

let make_ltorange xd = ColorBg(if !COLOR then
                    rt_create_color(xd, 255, 143, 52) else
                    rt_white_color xd);;

let make_rose xd = ColorBg(if !COLOR then
                    rt_create_color(xd, 255, 115, 165) else
                    rt_white_color xd);; *)

(*********************************************************************)
(* XGOALS stores information associated to each goal,                *)
(* The type xgoal contains:                                          *)
(*  1) a string recording the name of the subgoal                    *)
(*  2) a boolean which is true for the current goal, false otherwise *)
(*  3) the goal                                                      *)
(*********************************************************************)

type xgoal = Subgoal of string * bool * goal;;

let XGOALS = ref([]:xgoal list);;
let XUNDO = ref([]:xgoal list list);;
let XNEW = ref([]:string list);;

type windowinfo = {GSav:int; GVis:int;
                   HSav:int; HVis:int;
                   SSav:int; SVis:int;
                   SubHeight:int; SubWidth:int;
                   TWidth:int;
                   Main:string; Cxt:string;
                   InWind:string; InText:string;
                   TacTop:string; GoalTac:string; TacFill:string;
                   Goal:string; Hyps:string;
                   NewSubG:string; SubGText:string;
                   NewSubGPre:string; SubGPre:string;                   
                   SubGoals:string; SubGSel:string;
                   Bravo:string;
                   BgCol:attribute; ButCol:attribute; QuitCol:attribute};;

let rec list_xgoals = function
    [] -> []
  | (Subgoal(_,_,g)::gls) -> (g::list_xgoals gls);;

let goal_number () =
  let rec nth_goal n = function
      [] -> error "No such goal"
    | (Subgoal(_,current,_)::xgs) ->
            if current then n else nth_goal (n+1) xgs in
  nth_goal 1 !XGOALS;;

let rec up_to_date_xgoals glst wlst =
   match glst with
       [] -> (wlst = [])
     | (g::tl) -> if (mem g wlst) then up_to_date_xgoals tl (except g wlst)
                  else false;;

let choose_current xgoals =
  let rec exists_current = function
      [] -> false
    | (Subgoal(_,current,(_,sign,c))::gls) ->
         if current then true else exists_current gls in
  let make_current = function
      Subgoal(name,_,g) -> Subgoal(name,true,g) in
  if (exists_current xgoals) or (null xgoals)
    then xgoals
    else (make_current (hd xgoals)::tl xgoals);; 

let init_xgoals () =
  let glst = list_pf (get_goals()) in
  let wlst = list_xgoals !XGOALS in
  if not (up_to_date_xgoals glst wlst)
    then (let make_xgoal g =
            let (k,_,_) = g in Subgoal(make_name k,false,g) in
          let init_xgoals = function
              [] -> []
            | (g::gls) -> let (k,_,_) = g in
                          (Subgoal(make_name k,true,g)::map make_xgoal gls) in
          XUNDO := map init_xgoals (map list_pf !UNDO);
          XGOALS := init_xgoals glst)
    else XGOALS := choose_current !XGOALS;
  !XGOALS;;

(************************)
(* Global Output Window *)
(************************)

let OUT_WIND = ref "";;

let set_output_window str = OUT_WIND := str;;

(*****************************************************************************)
(* Functions for treating text moved by scroll bar, for treating text        *)
(* input from keyboard, for locating pop_ups                                 *)
(* my_scroll_set and my_text_button taken from ddr's rt-toolkit test program *)
(*****************************************************************************)

let my_scroll_set vmin vmax shift act (wid, but, val) =
  let oval = (scroll_val wid)-shift in
  let val =
    if but = 1 then oval-1
    else if but = 3 then oval+1
    else val in
  let val = max vmin (min vmax val) in
  if val <> oval then (
    scroll_set (wid, val+shift);
    act val);;

let scroll_shift wid_acc Twid val = text_shift(wid_acc Twid, val);;

let scroll_fun sav vis wid_acc Twid =
  my_scroll_set 0 (sav-vis) vis (scroll_shift wid_acc Twid);;

let my_text_button xd treat_text_input (wid, but, lin, col) =
  match but with
    1 ->
      text_set_mark (wid, lin, col)
  | -1 | 3 ->
      let txt = text_get_text (wid, lin, col) in
      rt_set_cut_buffer (xd, txt)
  | -2 ->
      let txt = rt_get_cut_buffer xd in
      treat_text_input (wid, txt)
  | _ ->
      ()
;;

let xll xd wid = xevent_x_root xd - xevent_x xd - widget_border wid
and yll xd wid = xevent_y_root xd - xevent_y xd + widget_height wid
;;

let locate_popup xd popup_wid bwid =
    let xll = xll xd bwid
    and yll = yll xd bwid in
    rt_map_popup_widget(popup_wid, xll, yll, 0);;

(**************************************)
(* Basic functions used in call_backs *)
(**************************************)

let TEXT_INPUT = ref "";;
let TEXT_OUTPUT = ref "";;
let VERNAC = ref ([]:string list);;
let BYVERNAC = ref false;; (* To know if a command is type by means of *)
                           (* the Vernac button *)
let OPEN = ref false;;

let no_keyp_output (wid, c) = ();;

let keyp_output (wid, c) =
    text_send_string (wid, " \b");
    text_send_string (wid, c);
    text_send_string (wid, "_\b");;

let end_vernac_input InWid acc_wid wid =
    TEXT_INPUT := ""; OPEN := false;
    rt_unmap_widget (acc_wid InWid);;

let send_and_end_input InWid acc_wid wid =
    TEXT_OUTPUT := !TEXT_INPUT;
    end_vernac_input InWid acc_wid wid;;
 
let keyp_input InWid acc_wid (wid, c) =
    keyp_output(wid, c); 
    if c = "\b"
      then TEXT_INPUT := sub_string !TEXT_INPUT 0
                                    ((length_string !TEXT_INPUT) - 1)
      else TEXT_INPUT := !TEXT_INPUT ^ c;
();;(*
    if c = "\n" then 
      send_and_end_input InWid acc_wid wid;;
*)

let text_clear_home wid_acc TWid nvis nsav =
  let twid = (wid_acc TWid) in
  let nlines = (nsav - nvis) in
    text_clear twid;
    text_shift (twid, nlines);
    text_home twid;;

let text_clear_end wid_acc TWid nvis nsav =
  let twid = (wid_acc TWid) in
  let nlines = (nsav - nvis) in
    text_clear twid;
    text_shift (twid, nlines);
    text_goto (twid, nsav, 0);;

let add_to_input str = TEXT_INPUT := !TEXT_INPUT ^ str;;

let first_input () = !TEXT_INPUT = "";;

let vernac_command str OutWid wid =
  TEXT_OUTPUT := str;
  VERNAC := (str::!VERNAC); set_output_window OutWid; ();;

let transient_input wid_acc InWid Twid str x y =
  let twid = (wid_acc Twid) in
  if not !OPEN
    then (rt_map_transient_widget(wid_acc InWid,x,y);
          OPEN := true;
          text_clear twid;
          keyp_output(twid,"\n"));
  keyp_output(twid,str);;

let new_input xd InWid TWid OutWid wid_acc init_str wid =
  add_to_input init_str;
  set_output_window OutWid;
  transient_input wid_acc InWid TWid init_str (xll xd wid) (yll xd wid);;

let vernac_input xd InWid TWid OutWid wid_acc init_str wid =
  VERNAC := (init_str::!VERNAC);
  new_input xd InWid TWid OutWid wid_acc init_str wid;;

let full_tac xd InWid TWid OutWid wid_acc str wid =
  let n = ((string_of_int (goal_number ())) ^ ":") in
  if !OPEN
    then let pfx = if first_input() then n else " " in
         new_input xd InWid TWid OutWid wid_acc (pfx ^ str) wid
    else vernac_command (n ^ str ^ ".") OutWid wid;;

let partial_tac xd InWid TWid OutWid wid_acc str wid =
  let pfx = if first_input()
            then ((string_of_int (goal_number())) ^ ":")
            else " " in
  new_input xd InWid TWid OutWid wid_acc (pfx ^ str) wid;;

let redisplay_context wid_acc CWid nvis nsav str wid =
  text_clear_home wid_acc CWid nvis nsav;
  vernac_command str CWid wid;;

let window_exists wid_acc name =
  try (wid_acc name); true with _ -> false;;

(***********************************)
(* Gestion des canaux d'impression *)
(***********************************)

let PIPE_BUFF = nref " ";;
let PIPE_BUFF_LEN = nref 0;;

let open_interface_pipe () =
let output = (fun s0 start slen ->
    let s = sub_string s0 start slen in
    while PIPE_BUFF_LEN.v + slen >= string_length PIPE_BUFF.v do
      PIPE_BUFF.v <- PIPE_BUFF.v ^ PIPE_BUFF.v; ()
    done;
    replace_string PIPE_BUFF.v s PIPE_BUFF_LEN.v;
    PIPE_BUFF_LEN.v <- PIPE_BUFF_LEN.v + slen; ()
  ) in
    std_fp.output := output;
    std_fp.flush_out := (function () -> ())
;;

let close_interface_pipe () =
    (std_fp.output := (output std_out);
     std_fp.flush_out := (function () ->  flush std_out))
;;


let pipe_to_window wid =
  keyp_output(wid, sub_string PIPE_BUFF.v 0 PIPE_BUFF_LEN.v);
  PIPE_BUFF_LEN.v <- 0; ()
;;

let pipe_to_out_win wid_acc = pipe_to_window (wid_acc !OUT_WIND);;

(***********************************)
(* l'impression de l'environnement *)
(***********************************)

let pr_hyps sign =
 let pr_sign = function
     [] -> [< >]
      |  l  -> (pr_rec l)
 where rec pr_rec = function
     []      -> [< >]
      |  [u]     -> print_decl u
      |  u::rest -> [< print_decl u; 'FNL ; pr_rec rest >]
 in pr_sign sign;;

let print_g c wid_acc GoalW Gvis Gsav =
  PP(prterm c);
  text_clear_end wid_acc GoalW Gvis Gsav;
  pipe_to_window (wid_acc GoalW);;

let print_h sign wid_acc HypsW Hvis Hsav =
  PP(pr_hyps sign);
  text_clear_home wid_acc HypsW Hvis Hsav;
  pipe_to_window (wid_acc HypsW);;

let print_goal Xinfo wid_acc c =
  print_g c wid_acc Xinfo.Goal Xinfo.GVis Xinfo.GSav;;

let print_hyps Xinfo wid_acc sign =
  print_h sign wid_acc Xinfo.Hyps Xinfo.HVis Xinfo.HSav;;

let print_subgoalX Xinfo wid_acc name c =
  print_g c wid_acc (name ^ Xinfo.SubGText) Xinfo.SVis Xinfo.SSav;;

let print_to_tac_window Xinfo wid_acc g =
  if is_mapped (wid_acc Xinfo.TacTop)
    then (let (_,sign,c) = g in
          print_goal Xinfo wid_acc c;
          print_hyps Xinfo wid_acc sign);;

let print_current Xinfo wid_acc =
  let rec find_current = function
    [] -> ()
  | (Subgoal(_,current,g)::gls) ->
       if current
         then print_to_tac_window Xinfo wid_acc g
         else find_current gls in
  find_current !XGOALS;;

(*********************)
(* SubGoal SubWidget *)
(*********************)

let subgoal_wid xd Xinfo wid line goalname xname button_fn =
  let SubGText = xname ^ Xinfo.SubGText
  and SubSel = xname ^ Xinfo.SubGSel
  and SScroll = xname ^ "SScroll"
  and Svis = Xinfo.SVis
  and Ssav = Xinfo.SSav
  and TWidth = Xinfo.TWidth
  and bgCol = Xinfo.BgCol
  and buttonCol = Xinfo.ButCol
  and currentCol = Xinfo.QuitCol in
  let wid_acc = widget_named xd in
  let nul_fn wid = () in
  let subgoal_window = rt_create_subwidget(wid,0,line,
      PackA [bgCol; NameAtt xname] (Horizontal, [
        SelectA [NameAtt SubSel; WidthAtt 55] [
          ButtonA [buttonCol] (goalname, button_fn);
          ButtonA [currentCol] ("X", nul_fn)
          ];
        ScrollA [bgCol; NameAtt SScroll] (Vertical, 0, Ssav, Svis,
              scroll_fun Ssav Svis wid_acc SubGText);
        TextA [NameAtt SubGText] (Svis, TWidth, Ssav, no_keyp_output,
                my_text_button xd no_keyp_output)
        ])) in
  scroll_set (wid_acc SScroll, Ssav);
  text_clear_end wid_acc SubGText Svis Ssav;
  subgoal_window;; 

(***************************************)
(* Updating Tactic and SubGoal Windows *)
(***************************************)

let select_refining wid_acc Xinfo =
  select_raise (wid_acc Xinfo.GoalTac,0);
  select_raise (wid_acc Xinfo.TacFill,0);;

let select_not_refining wid_acc Xinfo =
  select_raise (wid_acc Xinfo.GoalTac,2);
  select_raise (wid_acc Xinfo.SubGoals,0);;

let select_goal_not_saved wid_acc Xinfo =
  select_raise (wid_acc Xinfo.GoalTac,0);
  select_raise (wid_acc Xinfo.TacFill,2);
  select_raise (wid_acc Xinfo.SubGoals,0);;

let select_mult_goals wid_acc Xinfo =
  select_raise (wid_acc Xinfo.GoalTac,1);;

let select_subgoal_proved wid_acc Xinfo =
  select_raise (wid_acc Xinfo.GoalTac,0);
  select_raise (wid_acc Xinfo.TacFill,1);;

let destroy_subgoals wid_acc lst =
  let destroy_subgoal name =
    if (window_exists wid_acc name)
    then rt_destroy_widget (wid_acc name) in
  map destroy_subgoal lst;();;

let destroy_old_subgoals wid_acc Pre old new =
  let rec get_names Pre = function
      [] -> []
    | (Subgoal(name,_,_)::xgoals) ->
            ((Pre ^ name)::get_names Pre xgoals) in
  let old_names = subtract (get_names Pre old) (get_names Pre new) in
  destroy_subgoals wid_acc old_names;;

let reset_tac_window Xinfo wid_acc =
  destroy_subgoals wid_acc !XNEW;
  XNEW := [];
  select_refining wid_acc Xinfo;;

let new_current_goal Xinfo wid_acc name =
  reset_tac_window Xinfo wid_acc;
  let Pre = Xinfo.SubGPre in
  let Sel = Xinfo.SubGSel in
  let rec change_current = function
      [] -> []
    | (Subgoal(gname,current,g)::gls) ->
         let selname = (Pre ^ gname ^ Sel) in
         if eq_string(name,gname)
           then (select_raise (wid_acc selname, 1);
                 print_to_tac_window Xinfo wid_acc g;
                 (Subgoal(gname,true,g)::change_current gls))
           else (select_raise (wid_acc selname, 0);
                 (Subgoal(gname,false,g)::change_current gls)) in
  XGOALS := change_current !XGOALS;();;

let update_subgoal_window xd Xinfo newxgoals =
  let wid_acc = (widget_named xd) in
  let SubG = Xinfo.SubGoals in
  if is_mapped (wid_acc SubG)
    then (let Pre = Xinfo.SubGPre in
          let Sel = Xinfo.SubGSel in
          let Width = Xinfo.SubWidth in
          let Height = Xinfo.SubHeight in
          let put_window n = function
            Subgoal(name,current,((_,_,c) as g)) ->
              let y = n * Xinfo.SubHeight in
              let xname = Pre ^ name in
              let SubSel = xname ^ Xinfo.SubGSel in
              if (window_exists wid_acc xname)
                then rt_move_widget(wid_acc xname, 0, y)
                else (let button_fn wid =
                          new_current_goal Xinfo wid_acc name in
                      rt_map_widget
                        (subgoal_wid xd Xinfo (wid_acc SubG) y name xname
                                     button_fn));
              if current
                then select_raise (wid_acc SubSel, 1)
                else select_raise (wid_acc SubSel, 0);
              print_subgoalX Xinfo wid_acc xname c in
          let rec put_windows n = function
              [] -> let m = if (n = 0) then 1 else n in
                    rt_resize_widget(wid_acc SubG, Width, m * Height)
            | (xg::xgls) -> put_window n xg; put_windows (n+1) xgls in
          put_windows 0 newxgoals);;

let new_subgoal_windows xd Xinfo sgl wid pre_name =
  let get_name (k,_,_) = make_name k in
  let names = map get_name sgl in
  let get_xname name = (pre_name ^ name) in
  let xnames = map get_xname names in
  XNEW := xnames;
  let x = Xinfo.SubHeight in
  let wid_acc = widget_named xd in
  let rec map_subwindows n nlst xnlst = fun
      [] -> ()
    | (((path,_,c) as g)::lst) ->
      let goalname = (hd nlst) in
      let xname = (hd xnlst) in
      let button_fn wid = new_current_goal Xinfo wid_acc goalname in
      if not (window_exists wid_acc xname)
        then rt_map_widget
          (subgoal_wid xd Xinfo wid (x*n) goalname xname button_fn);
      print_subgoalX Xinfo wid_acc xname c;
      map_subwindows (n+1) (tl nlst) (tl xnlst) lst
  in map_subwindows 0 names xnames sgl;;

let restart_tacprover xd Xinfo =
  let xgoals = init_xgoals () in
  let wid_acc = widget_named xd in
  rt_map_widget (wid_acc Xinfo.TacTop);
  rt_map_widget (wid_acc Xinfo.SubGoals);
  reset_tac_window Xinfo wid_acc;
  update_subgoal_window xd Xinfo xgoals;
  print_current Xinfo wid_acc;;

let update_no_goal xd Xinfo =
  let wid_acc = widget_named xd in
  rt_map_widget (wid_acc Xinfo.TacTop);
  rt_map_widget (wid_acc Xinfo.SubGoals);
  if refining() then select_goal_not_saved wid_acc Xinfo
                else (text_clear_home wid_acc Xinfo.Hyps Xinfo.HVis Xinfo.HSav;
                      select_not_refining wid_acc Xinfo);;

(**********************************************)
(* mise a jour de la structure d'arbre XGOALS *)
(**********************************************)

let forward_state new_xgoals = 
  let new_xundo = try fst (chop_list !undo_limit !XUNDO)
                    with _ -> !XUNDO in
  XUNDO := !XGOALS::new_xundo; XGOALS := new_xgoals; ();;

let backward_state () =
  match !XUNDO with
    []            -> error "Empty undo list"
  | newx::new_xundo -> XGOALS := newx; XUNDO := new_xundo; ();;

let is_subgoal parent child =
  ((length_string child) >= (length_string parent))
  & (eq_string (sub_string child 0 (length_string parent), parent));;

let rec subgoals name = function
     [] -> []
   | (((k,_, _) as p)::l) -> if is_subgoal name (make_name k)
                             then (p::(subgoals name l))
                             else subgoals name l;;

let update_xgoals xd Xinfo newgoals oldxgoals =
  let wid_acc = (widget_named xd) in
  let Pre = Xinfo.NewSubGPre in
  let SubG = Xinfo.NewSubG in
  let update_current name g =
    let sub_glst = (subgoals name newgoals) in
    match sub_glst with
        [] -> select_subgoal_proved wid_acc Xinfo; []
      | [sg] -> if sg <> g then
                   print_to_tac_window Xinfo wid_acc sg;
                [Subgoal(name,true,sg)]
      | ((_,sign,_)::_) ->
                select_mult_goals wid_acc Xinfo;
                new_subgoal_windows xd Xinfo sub_glst (wid_acc SubG) Pre;
                print_hyps Xinfo wid_acc sign;
                let make_xgoal g =
                  let (k,_,_) = g in Subgoal(make_name k,false,g) in
                map make_xgoal sub_glst in
  let update_noncurrent name g =
    let sub_glst = (subgoals name newgoals) in
    match sub_glst with
        [sg] -> if sg <> g then
                   print_to_tac_window Xinfo wid_acc sg;
                Subgoal(name,false,sg)
      | _ -> error "Error in goal structure" in
  let rec replace_current = function
      [] -> []
    | (Subgoal(name,current,g)::lst) ->
            if current
              then append (update_current name g) (replace_current lst)
              else (update_noncurrent name g::replace_current lst) in
  replace_current oldxgoals;;

let update_forward xd Xinfo =
  let old_xgoals = !XGOALS in
  let new_goals = list_pf (get_goals()) in
  let new_xgoals = update_xgoals xd Xinfo new_goals old_xgoals in
  update_subgoal_window xd Xinfo new_xgoals;
  forward_state new_xgoals;
  let wid_acc = widget_named xd in
  print_current Xinfo wid_acc;
  if new_xgoals = [] then (select_goal_not_saved wid_acc Xinfo; cocorico());;

let update_backward xd Xinfo =
  repeat_action (((length !XUNDO) - (length !UNDO))) backward_state;
  let new_xgoals = !XGOALS in
  update_subgoal_window xd Xinfo new_xgoals;
  let wid_acc = widget_named xd in
  print_current Xinfo wid_acc;
  reset_tac_window Xinfo wid_acc;;

let new_goal () = (mem "Restart. " !VERNAC) or (mem "Goal " !VERNAC);;

let forward_step () =
  let goals = list_pf (get_goals())
  and oldgoals = list_xgoals !XGOALS
  and undo = !UNDO
  and undoX = !XUNDO in
  let rec same_list lst = function
      [] -> null lst
    | g::gs -> if (mem g lst) then (same_list (except g lst) gs)
                              else false in
  let greater_undo = (length undo) > (length undoX)
  and equal_undo = (length undo) = (length undoX) 
  and not_same_goals = not (same_list goals oldgoals) in
  ((!XGOALS <> []) & (not mem "Goal " !VERNAC) &
   (greater_undo or (equal_undo & not_same_goals)));;

let backward_step () = (length !UNDO) < (length !XUNDO);;

let update xd Xinfo =
  let wid_acc = (widget_named xd) in
  let nul_fn xd Xinfo = () in
  let restart_if_open xd Xinfo =
    if is_mapped (wid_acc Xinfo.TacTop) then update_no_goal xd Xinfo in
  let update_fn = if not(refining()) then restart_if_open
                  else if (new_goal() or !BYVERNAC) then restart_tacprover
                  else if forward_step() then update_forward
                  else if backward_step() then update_backward
                  else nul_fn in
  let old_xgoals = !XGOALS in
  update_fn xd Xinfo;
  let new_xgoals = !XGOALS in
  destroy_old_subgoals wid_acc Xinfo.SubGPre old_xgoals new_xgoals;;

(**********************)
(* The Subgoal Window *)
(**********************)

let create_subgoal_window xd Xinfo =
  let width = Xinfo.SubWidth
  and height = (Xinfo.SubHeight * 1)
  and name = Xinfo.SubGoals
  and color = Xinfo.ButCol in
  rt_create_widget(xd, "Subgoals", "Coq Subgoals",
    SelectA [NameAtt name] [
      TitleA [color] "No Current Subgoals";
      RawA [color] (width, height, 3, [])
      ]);;

(******************************************************************)
(* La fenetre standard avec tactiques, goals, environnement local *)
(******************************************************************)

let create_goal_window xd Xinfo =
  let Hvis = Xinfo.HVis and Hsav = Xinfo.HSav
  and Gvis = Xinfo.GVis and Gsav = Xinfo.GSav
  and TWidth = Xinfo.TWidth in
  let Main = Xinfo.Main in
  let InputW = Xinfo.InWind in
  let InputT = Xinfo.InText in
  let Tac = Xinfo.TacTop in
  let GoalTac = Xinfo.GoalTac in
  let TacFill = Xinfo.TacFill in
  let Goal = Xinfo.Goal in
  let Filler = Xinfo.NewSubG in
  let Hyps = Xinfo.Hyps in
  let HScroll = "HScroll" in
  let GScroll = "GScroll" in
  let bgCol = Xinfo.BgCol in
  let buttonCol = Xinfo.ButCol in
  let quitCol = Xinfo.QuitCol in
  let wid_acc = widget_named xd in
  let intro_wid =
    rt_create_popup_widget(xd, PackD(Vertical, [
      CommA [buttonCol] ("Intro            ",
        full_tac xd InputW InputT Main wid_acc "Intro");
      CommA [buttonCol] ("Intro _         ",
        partial_tac xd InputW InputT Main wid_acc "Intro ");
      CommA [buttonCol] ("Intros           ",
        full_tac xd InputW InputT Main wid_acc "Intros");
      CommA [buttonCol] ("Intros _         ",
        partial_tac xd InputW InputT Main wid_acc "Intros ");
      CommA [buttonCol] ("Intros until _   ",
        partial_tac xd InputW InputT Main wid_acc "Intros until ")
    ])) in
  let resolution_wid =
    rt_create_popup_widget(xd, PackD(Vertical, [
      CommA [buttonCol] ("Apply _          ",
        partial_tac xd InputW InputT Main wid_acc "Apply ");
      CommA [buttonCol] ("Cut _            ",
        partial_tac xd InputW InputT Main wid_acc "Cut ");
      CommA [buttonCol] ("Specialize _       ",
        partial_tac xd InputW InputT Main wid_acc "Specialize ");
      CommA [buttonCol] ("Generalize _     ",
        partial_tac xd InputW InputT Main wid_acc "Generalize ");
      CommA [buttonCol] ("Left             ",
        full_tac xd InputW InputT Main wid_acc "Left");
      CommA [buttonCol] ("Right            ",
        full_tac xd InputW InputT Main wid_acc "Right");
      CommA [buttonCol] ("Split            ",
        full_tac xd InputW InputT Main wid_acc "Split");
      CommA [buttonCol] ("Exists _         ",
        partial_tac xd InputW InputT Main wid_acc "Exists ");
      CommA [buttonCol] ("Reflexivity      ",
        full_tac xd InputW InputT Main wid_acc "Reflexivity");
      CommA [buttonCol] ("Symmetry         ",
        full_tac xd InputW InputT Main wid_acc "Symmetry");
      CommA [buttonCol] ("Transitivity _   ",
        partial_tac xd InputW InputT Main wid_acc "Transitivity ")
    ])) in
  let auto_wid =
    rt_create_popup_widget(xd, PackD(Vertical, [
      CommA [buttonCol] ("Auto             ",
        full_tac xd InputW InputT Main wid_acc "Auto");
      CommA [buttonCol] ("Auto _           ",
        partial_tac xd InputW InputT Main wid_acc "Auto ");
      CommA [buttonCol] ("Trivial          ",
        full_tac xd InputW InputT Main wid_acc "Trivial");
      CommA [buttonCol] ("Exact _          ",
        partial_tac xd InputW InputT Main wid_acc "Exact ");
      CommA [buttonCol] ("Assumption       ",
        full_tac xd InputW InputT Main wid_acc "Assumption");
      CommA [buttonCol] ("Instantiate _    ",
        partial_tac xd InputW InputT Main wid_acc "Instantiate ")
    ])) in
  let equality_wid =
    rt_create_popup_widget(xd, PackD(Vertical, [
      CommA [buttonCol] ("Unfold _         ",
        partial_tac xd InputW InputT Main wid_acc "Unfold ");
      CommA [buttonCol] ("Change _         ",
        partial_tac xd InputW InputT Main wid_acc "Change ");
      CommA [buttonCol] ("Red              ",
        full_tac xd InputW InputT Main wid_acc "Red");
      CommA [buttonCol] ("Simpl            ",
        full_tac xd InputW InputT Main wid_acc "Simpl");
      CommA [buttonCol] ("Hnf              ",
        full_tac xd InputW InputT Main wid_acc "Hnf");
      CommA [buttonCol] ("Pattern _        ",
        partial_tac xd InputW InputT Main wid_acc "Pattern ")
    ])) in
  let induct_wid =
    rt_create_popup_widget(xd, PackD(Vertical, [
      CommA [buttonCol] ("Elim _           ",
        partial_tac xd InputW InputT Main wid_acc "Elim ");
      CommA [buttonCol] ("ElimType _       ",
        partial_tac xd InputW InputT Main wid_acc "ElimType ");
      CommA [buttonCol] ("Induction _      ",
        partial_tac xd InputW InputT Main wid_acc "Induction ");
      CommA [buttonCol] ("Rewrite -> _     ",
        partial_tac xd InputW InputT Main wid_acc "Rewrite -> ");
      CommA [buttonCol] ("Rewrite <- _     ",
        partial_tac xd InputW InputT Main wid_acc "Rewrite <- ");
      CommA [buttonCol] ("Replace _        ",
        partial_tac xd InputW InputT Main wid_acc "Replace ");
      CommA [buttonCol] ("Absurd _         ",
        partial_tac xd InputW InputT Main wid_acc "Absurd ")
    ])) in
  let exp_wid =
    rt_create_popup_widget(xd, PackD(Vertical, [
      CommA [buttonCol] ("_ Orelse _       ",
        partial_tac xd InputW InputT Main wid_acc "Orelse");
      CommA [buttonCol] ("Try _            ",
        partial_tac xd InputW InputT Main wid_acc "Try");
      CommA [buttonCol] ("Repeat _         ",
        partial_tac xd InputW InputT Main wid_acc "Repeat");
      CommA [buttonCol] ("Do _             ",
        partial_tac xd InputW InputT Main wid_acc "Do ");
      CommA [buttonCol] ("Other            ",
        new_input xd InputW InputT Main wid_acc "")
    ])) in
  let extract_wid =
    rt_create_popup_widget(xd, PackD(Vertical, [
      CommA [buttonCol] ("Realizer _        ",
        partial_tac xd InputW InputT Main wid_acc "Realizer ")
    ])) in
  let program_wid =
    rt_create_popup_widget(xd, PackD(Vertical, [
      CommA [buttonCol] ("Program          ",
        full_tac xd InputW InputT Main wid_acc "Program");
      CommA [buttonCol] ("Program_all      ",
        full_tac xd InputW InputT Main wid_acc "Program_all");
      CommA [buttonCol] ("Show Program      ", vernac_command "Show Program. " Main)
    ])) in
  let tacpack = PackA [bgCol] (Horizontal, [
    PackA [bgCol] (Vertical, [
      PopupA [buttonCol] ("Introduction", locate_popup xd intro_wid);
      PopupA [buttonCol] ("Convertibility", locate_popup xd equality_wid)
      ]);
    PackA [bgCol] (Vertical, [
      PopupA [buttonCol] ("Resolution", locate_popup xd resolution_wid);
      PopupA [buttonCol] ("Induction", locate_popup xd induct_wid)
      ]);
    PackA [bgCol] (Vertical, [
      PopupA [buttonCol] ("Automatic/Exact", locate_popup xd auto_wid);
      PopupA [buttonCol] ("Tactic Expression", locate_popup xd exp_wid)
      ]);
    PackA [bgCol] (Vertical, [
      PopupA [buttonCol] ("Realizer", locate_popup xd extract_wid);
      PopupA [buttonCol] ("Program", locate_popup xd program_wid)
      ])]) in
  let savepack = PackA [buttonCol] (Horizontal, [
     RawA [buttonCol; FillerAtt] (1, 1, 0, []);
       PackA [buttonCol] (Vertical, [
         TitleA [quitCol] "Goal Proved!";
         PackA [buttonCol] (Horizontal, [
           ButtonA [buttonCol] ("Save Theorem",
             vernac_input xd InputW InputT Main wid_acc "Save ");
           ButtonA [buttonCol] ("Save Remark",
             vernac_input xd InputW InputT Main wid_acc "Save Remark ")
           ])
         ]);
       RawA [buttonCol; FillerAtt] (1, 1, 0, [])]) in
  let goal_window = rt_create_widget(xd,"Goal Directed Proof", "Coq Tactics",
     PackA [bgCol;NameAtt Tac] (Vertical, [
      SelectA [FillerAtt; NameAtt GoalTac] [
       PackA [bgCol] (Vertical, [
         PackA [bgCol] (Horizontal, [
             TitleA [bgCol; FillerAtt] ("Tactics");
             ButtonA [quitCol] ("Undo", vernac_command "Undo. " Main);
             ButtonA [quitCol] ("Restart", vernac_command "Restart. " Main)
             ]);
         SelectA [NameAtt TacFill] [
           tacpack;
           TitleA [buttonCol] "SubGoal Proved!";
           savepack
           ];
         TitleA [bgCol] ("Current Subgoal");
         PackA [bgCol;FillerAtt] (Horizontal, [
           ScrollA [bgCol; NameAtt GScroll] (Vertical, 0, Gsav, Gvis,
                 scroll_fun Gsav Gvis wid_acc Goal);
           TextA [FillerAtt; NameAtt Goal] (Gvis, TWidth, Gsav, no_keyp_output,
                   my_text_button xd no_keyp_output)
           ])
         ]);
       PackA [bgCol] (Vertical, [
        TitleA [bgCol] ("Generated Subgoals");
        TextA [FillerAtt; NameAtt Filler] (1, (TWidth + 5), 0, no_keyp_output,
                my_text_button xd no_keyp_output)
        ]);
       TitleA [buttonCol] "No Current Goal"
       ];
      TitleA [bgCol] ("Hypotheses");
      PackA [bgCol;FillerAtt] (Horizontal, [
         ScrollA [bgCol; NameAtt HScroll] (Vertical, 0, Hsav, Hvis,
               scroll_fun Hsav Hvis wid_acc Hyps);
         TextA [FillerAtt; NameAtt Hyps] (Hvis, TWidth, Hsav, no_keyp_output,
                 my_text_button xd no_keyp_output)
         ])
       ])) in
  scroll_set (wid_acc HScroll, Hsav);
  scroll_set (wid_acc GScroll, Gsav);
  text_clear_home wid_acc Hyps Hvis Hsav;
  text_clear_end wid_acc Goal Gvis Gsav;
  goal_window;;

(**********************)
(* The Context Window *)
(**********************)

let open_context_window xd Xinfo top_wid =
  let Cxt = Xinfo.Cxt
  and InputW = Xinfo.InWind
  and InputT = Xinfo.InText
  and TWidth = Xinfo.TWidth
  and CScroll = "CScroll" in
  let NVIS = 55 and NSAV = 300 in
  let bgCol = Xinfo.BgCol
  and buttonCol = Xinfo.ButCol in
  let wid_acc = widget_named xd in
  let x = (widget_width top_wid) + (2 * (widget_border top_wid)) + 20 in
  let print_wid =
    rt_create_popup_widget(xd, PackD(Vertical, [
      CommA [buttonCol] ("Print            ",
        vernac_command "Print. " Cxt);
      CommA [buttonCol] ("Print _          ",
        vernac_input xd InputW InputT Cxt wid_acc "Print ");
      CommA [buttonCol] ("Print Hint       ",
        vernac_command "Print Hint. " Cxt);
      CommA [buttonCol] ("Print All        ",
        redisplay_context wid_acc Cxt NVIS NSAV "Print All. ");
      CommA [buttonCol] ("Print Section _  ",
        vernac_input xd InputW InputT Cxt wid_acc "Print Section ");
      CommA [buttonCol] ("Check _          ",
        vernac_input xd InputW InputT Cxt wid_acc "Check ");
      CommA [buttonCol] ("Search _         ",
        vernac_input xd InputW InputT Cxt wid_acc "Search ");
      CommA [buttonCol] ("Info _           ",
        vernac_input xd InputW InputT Cxt wid_acc "Info ")
    ])) in
  let inspect_wid =
    rt_create_popup_widget(xd, PackD(Vertical, [
      CommA [buttonCol] ("Inspect 5        ",
        vernac_command "Inspect 5. " Cxt);
      CommA [buttonCol] ("Inspect 10       ",
        vernac_command "Inspect 10. " Cxt);
      CommA [buttonCol] ("Inspect 15       ",
        vernac_command "Inspect 15. " Cxt);
      CommA [buttonCol] ("Inspect 25       ",
        vernac_command "Inspect 25. " Cxt);
      CommA [buttonCol] ("Inspect _          ",
        vernac_input xd InputW InputT Cxt wid_acc "Inspect ")
    ])) in
  let reset_wid =
    rt_create_popup_widget(xd, PackD(Vertical, [
      CommA [buttonCol] ("Reset _          ",
        vernac_input xd InputW InputT Cxt wid_acc "Reset ");
      CommA [buttonCol] ("Reset After _    ",
        vernac_input xd InputW InputT Cxt wid_acc "Reset After ");
      CommA [buttonCol] ("Reset Section _  ",
        vernac_input xd InputW InputT Cxt wid_acc "Reset Section ");
      CommA [buttonCol] ("Reset Empty    ",
        vernac_command "Reset Empty. " Cxt);
      CommA [buttonCol] ("Reset Initial    ",
        vernac_command "Reset Initial. " Cxt)
    ])) in
  let context_wid =
    rt_create_located_widget(xd, "Context", "Coq Context", UserPosition(x,0),
       PackA [bgCol] (Vertical,[
         PackA [bgCol] (Horizontal, [
           PopupA [buttonCol] ("Print", locate_popup xd print_wid);
           PopupA [buttonCol] ("Inspect", locate_popup xd inspect_wid);
           PopupA [buttonCol] ("Reset", locate_popup xd reset_wid)
           ]);
         PackA [bgCol;FillerAtt] (Horizontal,[
          ScrollA [bgCol; NameAtt CScroll] (Vertical, 0, NSAV, NVIS,
            my_scroll_set 0 (NSAV-NVIS) NVIS (scroll_shift wid_acc Cxt));
          TextA [FillerAtt; NameAtt Cxt] (NVIS, TWidth, NSAV, no_keyp_output,
            my_text_button xd no_keyp_output)
          ])
         ])) in
  scroll_set (wid_acc CScroll, NSAV);
  text_clear_home wid_acc Cxt NVIS NSAV;
  rt_map_widget context_wid;;

(***************************************************)
(* La fenetre principale avec vernaculaire de base *)
(***************************************************)

(* exception QuitX;; *)

let open_top_window xd Xinfo =
  let Main = Xinfo.Main
  and InputW = Xinfo.InWind
  and InputT = Xinfo.InText
  and MScroll = "MScroll" in
  let bgCol = Xinfo.BgCol
  and buttonCol = Xinfo.ButCol
  and quitCol = Xinfo.QuitCol in
  let wid_acc = widget_named xd in
  let tac_fn wid = if (refining() & (not (null !XGOALS)))
                   then restart_tacprover xd Xinfo
                   else update_no_goal xd Xinfo in
  let quit_fun wid = raise QuitX in
  let NVIS = 15 and NSAV = 100 in
  let hint_wid =
    rt_create_popup_widget(xd, PackD(Vertical, [
      CommA [buttonCol] ("Hint _           ",
        vernac_input xd InputW InputT Main wid_acc "Hint ");
      CommA [buttonCol] ("Immediate _      ",
        vernac_input xd InputW InputT Main wid_acc "Immediate ");
      CommA [buttonCol] ("Hint Unfold _    ",
        vernac_input xd InputW InputT Main wid_acc "Hint Unfold ");
      CommA [buttonCol] ("Erase _          ",
        vernac_input xd InputW InputT Main wid_acc "Erase ");
      CommA [buttonCol] ("Print Hint       ",
        vernac_command "Print Hint. " Main)
    ])) in
  let top_window =
  rt_create_located_widget(xd,"Coq: Calculus of Inductive Constructions",
   "Coq", UserPosition(0, 0),
   PackA [bgCol] (Vertical,[
     PackA [bgCol] (Horizontal, [
        TitleA [bgCol] ("Main Window");
        ButtonA [buttonCol] ("Goal",
          vernac_input xd InputW InputT Main wid_acc "Goal ");
        ButtonA [buttonCol] ("TacProver", tac_fn);
        ButtonA [buttonCol] ("Open",
          vernac_input xd InputW InputT Main wid_acc "Open ");
        ButtonA [buttonCol] ("Close", vernac_command "Close. " Main);
        PopupA [buttonCol] ("Hints", locate_popup xd hint_wid);
        ButtonA [buttonCol] ("Vernac",(BYVERNAC:=true;
          vernac_input xd InputW InputT Main wid_acc ""));
        ButtonA [quitCol] ("Abort", vernac_command "Abort. " Main);
        ButtonA [quitCol] ("Quit", quit_fun)
        ]);
      PackA [bgCol;FillerAtt] (Horizontal, [
        ScrollA [bgCol; NameAtt MScroll] (Vertical, 0, NSAV, NVIS,
          my_scroll_set 0 (NSAV-NVIS) NVIS (scroll_shift wid_acc Main));
        TextA [FillerAtt; NameAtt Main] (NVIS, 80, NSAV, no_keyp_output,
                 my_text_button xd no_keyp_output)
        ])
      ])) in
  scroll_set (wid_acc MScroll, NSAV);
  text_clear_home wid_acc Main NVIS NSAV;
  rt_map_widget top_window; top_window;;

(*******************************)
(* The Vernacular Input Window *)
(*******************************)

let create_input_window xd Mainwid Xinfo =
  let InputW = Xinfo.InWind
  and InputT = Xinfo.InText
  and Main = Xinfo.Main in
  let wid_acc = widget_named xd in
  let treat_text_input = (keyp_input InputW wid_acc) in
  let ok_fn wid = send_and_end_input InputW wid_acc wid in
  let abort_fn wid = VERNAC := []; end_vernac_input InputW wid_acc wid in
    rt_create_transient_widget(Mainwid, "Vernacular Input",
      PackA [Xinfo.BgCol; NameAtt InputW] (Horizontal,[
        TextA [NameAtt InputT] (2, 80, 0, treat_text_input,
               my_text_button xd treat_text_input);
        ButtonA [Xinfo.ButCol] (" ; ", 
               new_input xd InputW InputT Main wid_acc " ;");
        ButtonA [Xinfo.ButCol] ("OK", ok_fn);
        ButtonA [Xinfo.QuitCol] ("Abort", abort_fn)
        ]));;

(***************************)
(* La fonction d'interface *)
(***************************)

let open_interface dname =
  open_interface_pipe ();
  let stdin_fun1 () = let c = input_char std_in in () in
  let stdin_fun2 () = () in
  let print_fun = fun () -> () in set_print_fun print_fun;
  let end_fun1() = reset_print_fun(); close_interface_pipe();
                   OPEN := false; VERNAC := [] in
  begin try
    (let xd = rt_initialize dname in
     COLOR := !COLOR & is_colored xd;
     let print_fun = fun () -> () in set_print_fun print_fun;
     let end_fun() = end_fun1(); rt_end xd in
     let Xinfo = {GSav = 20; GVis = 4;
                  HSav = 50; HVis = 10;
                  SSav = 40; SVis = 3;
                  SubHeight = 44; SubWidth = 525;
                  TWidth = 75;
                  Cxt = "Cxt"; Main = "Main";
                  InWind = "InWind"; InText = "InText";
                  GoalTac = "GoalTac"; TacFill = "TacFill";
                  Goal = "Goal"; Hyps = "Hyps"; NewSubG = "Filler";
                  NewSubGPre = "New"; SubGPre = "";
                  SubGText = "S"; SubGSel = "SubGSel";
                  SubGoals = "SubGoals"; TacTop = "Tac"; Bravo = "Bravo";
                  BgCol = (BackgroundAtt (make_ltblue xd));
                  ButCol = (BackgroundAtt (make_ltyellow xd));
                  QuitCol = (BackgroundAtt (make_orangered xd))} in
     let wid_acc = widget_named xd in
     let args = rt_args[xd] in
     rt_select_file(args, 0, stdin_fun2);
     begin try
      (let top_window = open_top_window xd Xinfo in
       open_context_window xd Xinfo top_window;
       create_input_window xd top_window Xinfo;
       create_goal_window xd Xinfo;
       create_subgoal_window xd Xinfo;
       let read_fn() =
         while !TEXT_OUTPUT = "" do
           rt_treat_one_event args
         done;
         let out_str = !TEXT_OUTPUT in
           TEXT_OUTPUT := "";
           out_str
       and write_fun() =
           pipe_to_out_win wid_acc;
           update xd Xinfo;
           VERNAC := []; BYVERNAC := false; ()
       in
         (read_fn, write_fun, end_fun))
     with reraise -> end_fun(); raise(reraise)
     end)
  with reraise -> end_fun1(); raise(reraise)
  end;;

