(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                             initial.ml                                   *)
(****************************************************************************)

#open "std";;
#open "pp";;
#open "stdpp";;

let repeat_action n action = repeat_action_rec n
    where rec repeat_action_rec n =
    if n<=0 then () else (action(); repeat_action_rec (pred n));;

(********************************************************************)

(*                 Error and Warning management                     *)

(********************************************************************)

(* exception UserError of string;; (* User errors *) *)
(* exception Anomaly of string;;  (* System errors *) *)

(* Warnings *)
let warning string = message ("Warning: " ^ string);;

(* User errors *)
let error string = raise (UserError string);;

(* System errors *)
let anomaly string = raise (Anomaly string);;

let SILENT = ref false;;

let make_silent flag = SILENT:=flag; ();;

let is_silent () = !SILENT;;

let FOCUS = ref 0;;

let make_focus n = FOCUS:=n; ();;

let focus () = !FOCUS;;

(********************************************************************)
(*                             Names                                *)
(********************************************************************)

(* type identifier == string * int;; *)
(* type name = Name of identifier | Anonymous;; *)

(********************************************************************)
(*                             Levels                                *)
(********************************************************************)

(* The level of a construction indicates what is the current kind of 
   judgement M : T as follows: 
 If M is a Proof, then T:Prop(_) is a proposition, and its level is Proof
 If M is an Object, then T:Type(_,u) is a type, and its level is Object
Thus Prop, Type, propositions, proposition schemas, etc. are objects *)

(* type level = Object | Proof;; *)

(* level comparisons *)
let le_level l m = (l = Proof) or (m = Object);;


let for_all2eq f l1 l2 = try for_all2 f l1 l2 with Failure _ -> false;;

let explode_id (IDENT(s,n)) = (explode s)
                      @(if n = (-1) then []
                        else explode(string_of_int n))
;;

let print_id (IDENT(s,n)) =
    [< 'S s ; (if n = (-1) then [< >] else [< 'INT n >]) >]
;;

let string_of_id (IDENT(s,n)) =
    s ^ (if n = (-1) then "" else (string_of_int n))
;;

let id_of_string s =
    let slen = string_length s in
    let rec numpart n =
        if n = 0 then failwith("identifier " ^ s ^ " cannot be split")
        else let c = int_of_char(nth_char s (n-1)) in
            if (int_of_char `0`) <= c & c <= (int_of_char `9`) then
                numpart (n-1)
            else
                n
    in
    let numstart = numpart slen in
    if numstart = slen then IDENT(s,(-1)) else
        IDENT(sub_string s 0 numstart,
              int_of_string (sub_string s numstart (slen - numstart)))
;;

let stringpart_of_id (IDENT(s,n)) = s;;
let index_of_id(IDENT(s,n)) = n;;

let next_ident_away (IDENT(str,_) as id) l = name_rec (-1)
where rec name_rec i =
    let new = if i = (-1) then id else IDENT(str, i)
    in if mem new l then name_rec (i+1) else new;;

let next_name_away name l = match name with
   Name(str) ->  next_ident_away str l
 | Anonymous -> id_of_string "Anonymous";;
