(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                               goml.ml                                    *)
(****************************************************************************)

#open "std";;
#open "initial";;
#open "coqtoplevel";;
#open "coq";;
#open "centaur";;
#open "machine";;
#open "compile";;
#open "fmlenv";;
#open "lexer";;
#open "sys";;
#open "version";;
#open "vernac";;
#open "tactics";;

type start_state =
  Undefined
| SaveState of string
| RestoreState of string
;;
let start_state = ref Undefined;;

let usage() =
  prerr_string "Usage: "; prerr_string command_line.(0);
  prerr_endline " [options]";
  prerr_endline "  -x : boots version with X interface";
  prerr_endline "  -bw: black & white - only valid with -x";
  prerr_endline "  -gray: adapted gray - only valid with -x";
  prerr_endline "  -os file: load initial then save state into file";
  prerr_endline "  -is file: read initial state from file";
  prerr_endline "  -centaur: start a toplevel loop for the Centaur interface";
  prerr_endline " -I dir: add dir to loadpath";
  prerr_endline " -q : no autoload of ~/.coqrc";
  exit 1
;;

let get_arg i =
  if i >= vect_length command_line then usage()
  else command_line.(i)
;;

let load_rc = ref true;;

let starting_loop = ref go;;

arg_loop 1
where rec arg_loop i =
  if i < vect_length command_line then (
    arg_loop (
      match command_line.(i) with
        "-os" ->
          start_state := SaveState(get_arg(i+1));
          i+2
      | "-is" ->
          start_state := RestoreState(get_arg(i+1));
          i+2
      | "-I" -> add_path (get_arg(i+1));i+2
      | "-q" -> (load_rc := false);i+1
      | "-centaur" -> starting_loop := Centaur_go;i+1
      | x ->
          usage()
    )
  )
;;

try
  match !start_state with
    SaveState x ->
      load_initial();
      extern_state x
  | RestoreState x ->
      intern_state x;tactics__restore_state "Initial"
  | _ ->
      ()
with
  Sys_error msg ->
    prerr_string "OS Error: "; prerr_endline msg; exit 2
| UserError s ->
    prerr_string "Error : "; prerr_endline s; exit 2
;;

type expr =
  EApp of string * expr
| EUnit
| EString of string
| EFun of (expr -> expr)
| ENone
;;

exception IllTyped of string;;

let valtab =
  let illt t = raise(IllTyped t) in
  let t = make_vect 11 [] in
  do_list (fun (x,v) -> hash_add_assoc (x,v) t) [
    "Centaur_go",   EFun(fun EUnit   -> Centaur_go(); EUnit     | _ -> illt "unit");
    "go",  EFun(fun EUnit   -> go(); EUnit    | _ -> illt "unit");
    "fml", EFun(fun EUnit   -> fml(); EUnit   | _ -> illt "unit");
    "pwd", EFun(fun EUnit   -> EString(getwd__getwd()) | _ -> illt "unit");
    "cd",  EFun(fun (EString x) -> (sys__chdir x;EString(getwd__getwd()))  | _ -> illt "string")
  ]; t
;;

let ml_keywords = ([| [] |] : (string * unit) list vect);;

let parse = function
  [<
    'Tident c;
    begin function
      [< 'Tlparen; 'Trparen >] -> EUnit
    | [< 'Tstring s >] -> EString s
    end a;
    'Tsemisemi
  >] -> EApp(c,a)
| [< '_ >] -> raise Parse_error
;;

let print = function
  EUnit -> [< '"()" >]
| EString s -> [< '"\""; 's; '"\"" >]
| _ -> [< '"???" >]
;;

let printerr x =
  prerr_string ">>> "; prerr_endline x;
  prerr_endline "Available commands: Centaur_go(), go(), fml(), pwd(), cd \"...\", quit()";
  ENone
;;

let rec toplev_loop() =
  set_prompt "comm # ";
  match
    try parse (token_stream ml_keywords (stream_of_channel std_in))
    with
      Parse_failure ->
        EApp("quit", EUnit)
    | Parse_error ->
        printerr "Syntax error";
        reset_lexer (stream_of_channel std_in); ENone
    | _ ->
        printerr "Uncaught exception in parsing"
  with
    EApp("quit", EUnit) -> ()
  | x ->
      begin match
        match x with
          EApp(f,x) ->
            begin match try hash_assoc f valtab with _ -> ENone
            with
              EFun v ->
                begin try v x with
                  Sys_error msg -> printerr msg
                | IllTyped t -> printerr ("Parameter should be of type " ^ t)
                | Failure s -> printerr ("failed: " ^ s)
                | _ -> printerr "Uncaught exception"
                end
            | ENone -> printerr ("Unbound variable: " ^ f)
            | _ -> printerr "Not a function"
            end
        | ENone -> ENone
        | _ -> printerr "Internal error"
      with
        ENone -> ()
      | r ->
          print_string "- = ";
          do_stream print_string (print r);
          print_newline()
      end;
      toplev_loop()
;;

make_silent true;;
extract_all();;
make_silent false;;
prerr_string "Welcome to Coq ";
prerr_endline version;;
if !load_rc then
    try load_vernacular "~/.coqrc"
    with _ -> message "Load of ~/.coqrc failed"
;;
begin try ((!starting_loop)();ENone) with
      Sys_error msg -> printerr msg
    | IllTyped t -> printerr ("Parameter should be of type " ^ t)
    | Failure s -> printerr ("failed: " ^ s)
    | _ -> printerr "Uncaught exception"
end;;
toplev_loop();;
prerr_endline "";;
