(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                            gentermio.ml                                  *)
(****************************************************************************)
#open "initial";;
#open "genterm";;
#open "more_util";;
#open "extraction";;

let p_char c = stream_check (fun c' -> (c=c'))
;;

let rec p_white =
    function
    [< '` ` ; p_white _ >] -> ()
  | [< >] -> ()
;;

let PWA f =
    function
    [< f rslt ; p_white _ >] -> rslt
;;


let rec parse_list_exactly = function
    [] ->
    (function [< >] -> ())
  | (h::t) ->
    (function [< (p_char h) _ ;
             (parse_list_exactly t) _ >] -> ())
;;

let p_lit w =
    PWA
    (function
     [< (parse_list_exactly (explode_chars w)) _ >] -> ())
;;

let p_option itp =
    PWA
    (function
     [< (p_lit "<>") _ >] -> NONE
   | [< (p_lit "[") _ ; itp v ; (p_lit "]") _ >] -> (SOME v))
;;

let pr_option itpr = function
    NONE -> [< '"<>" >]
  | (SOME v) -> [< '"[" ; itpr v ; '"]" >]
;;

let p_ident =
    PWA
    (function
     [< p_atom a ;
      (function
       [< (p_lit "@") _ ; number n >] -> n
     | [< >] -> 0) n >] -> IDENT(a,n))
;;

let pr_ident = function
    IDENT(a,-1) -> [< 'a >]
  | IDENT(a,n) -> [< 'a ; '"@" ; 'string_of_int n >]
;;

let braced_list = fun lend rend sep elem ->
    PWA
    (function
    [< lend _;
       (maybe_empty_list_with_sep [] cons sep elem) vl;
       rend _ >] -> vl)
;;

let p_term parse_param = p_term where rec
    p_param_list =
    fun strm ->
        braced_list (p_lit "[") (p_lit "]") (p_lit ";") parse_param strm

and p_operator param_parser = 
    fun strm ->
        PWA
        (function
         [< p_atom opn ; p_param_list pl >] -> (opn,pl)) strm

and p_term_list =
    fun strm ->
        braced_list (p_lit "(") (p_lit ")") (p_lit ";") p_term strm

and p_ident_option_list =
    fun strm ->
        (braced_list (p_lit "[") (p_lit "]") (p_lit ",") (p_option p_ident))
        strm

and p_ident_or_op = 
    fun strm ->
    (PWA
     (function
      [< p_atom a ;
       (function
        [< (p_lit "@") _ ; number n >] -> let id = IDENT(a,n) in VAR id
      | [< p_param_list pl;
         (PWA
          (function
           [< p_term_list tl >] -> tl
         | [< >] -> [])) tl >] -> (OP(a,pl,tl))
      | [< p_term_list tl >] -> (OP(a,[],tl))
      | [< >] -> VAR(IDENT(a,0))) rslt >] -> rslt))
    strm

and p_term =
    fun strm ->
    (PWA
     (function
      [< p_ident_or_op rslt >] -> rslt
    | [< p_ident_option_list idl; p_term t >] -> (SLAM(idl,t)))) strm
;;

let pr_maybe_empty_list_with_sep =
fun sep elempr -> aux where rec aux = function
    [] -> [< >]
  | [h] -> [< elempr h >]
  | (h::t) -> [< elempr h ; sep() ; aux t >]
;;

let pr_braced_list =
fun lend rend sep elem vl ->
    [< lend();
       (pr_maybe_empty_list_with_sep sep elem) vl;
       rend() >]
;;

let pr_term print_param =
    prterm where rec prterm = function
    (VAR id) -> pr_ident id
  | (OP(opn,pl,tl)) ->
    [< 'opn ;
       pr_braced_list (load_string "[") 
                      (load_string "]")
                      (load_string ";")
                      print_param pl;
       pr_braced_list (load_string "(") 
                      (load_string ")")
                      (load_string ";")
                      prterm tl >]
  | (SLAM(ids,body)) ->
    [< pr_braced_list (load_string "[") 
                      (load_string "]")
                      (load_string ",")
                      (pr_option pr_ident) ids;
       prterm body >]
;;


let p_genterm = p_term parse_param
;;
let pr_genterm = pr_term print_param
;;

let coqpr_genterm = prterm where rec prterm = function
    (VAR id) ->
    (match id with IDENT(s,-1) -> [< 's >]
                 | IDENT(s,n) -> [< 's ; 'string_of_int n >])
  | (OP("META",[],[])) -> [< '"<Meta(n)>" >]
  | (OP("PROP",[CONTENTS Null],[])) -> [< '"Prop" >]
  | (OP("PROP",[CONTENTS Pos],[])) -> [< '"Set" >]
  | (OP("PROP",[CONTENTS Data],[])) -> [< '"<Prop(Data)>" >]
  | (OP("TYPE",[CONTENTS Null],[])) -> [< '"Type" >]
  | (OP("TYPE",[CONTENTS Pos],[])) -> [< '"Type(Pos,_)" >]
  | (OP("TYPE",[CONTENTS Data],[])) -> [< '"Type(Data,_)" >]
  | (OP("LAMBDA",[],[Dom;SLAM([SOME id],Body)])) ->
    [< '"[" ; prterm (VAR id) ; '":(" ; prterm Dom ; '")]" ;
       '"(" ; prterm Body ; '")" >]
  | (OP("LAMBDA",[],[Dom;SLAM([NONE],Body)])) ->
    [< '"[Anon:(" ; prterm Dom ; '")]" ;
       '"(" ; prterm Body ; '")" >]
  | (OP("APP",[],[l;r])) ->
    [< paren_prterm l ; '" " ; paren_prterm r >]
  | (OP("PROD",[],[Dom;SLAM([SOME id],Rng)])) ->
    [< '"(" ; prterm (VAR id) ; '":(" ; prterm Dom ; '"))" ;
       '"(" ; prterm Rng ; '")" >]
  | (OP("PROD",[],[Dom;SLAM([NONE],Rng)])) ->
    [< '"(" ; prterm Dom ; '") -> " ;
       '"(" ; prterm Rng ; '")" >]
  | (OP("IND",[],[Arity;
                  SLAM([SOME bid],
                       OP("CONSTRUCTORS",[],l))])) ->
    [< '"Ind(" ; prterm (VAR bid) ; '":" ; prterm Arity ; '")(" ;
     prlist_with_sep (fun () -> [< '"|" >]) prterm l ;
     '")" >]
  | (OP("ELIM",[],c::P::cl)) ->
    [< '"(<" ; prterm P ; '">" ;
       '"Match (" ; prterm c ; '") with" ;
       prlist (fun t -> [< '" " ; prterm t >]) cl ;
       '")" >]
  | (OP("CONSTR",[INT n],[c])) ->
    [< '"Constr(" ; 'string_of_int n ; '"," ; prterm c ; '")" >]
  | (OP(opn,[],[])) -> [< 'opn >]
  | _ -> error "coqpr_genterm was fed a non-coq-legal genterm"

and paren_prterm t = [< '"(" ; prterm t ; '")" >]
;;

let coqprint_genterm gt = coqpr_genterm gt;;
